/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <pml/PhysicalModel.h>

#include "PMManagerDC.h"
#include "PMManagerDCPopup.h"
#include "StructuralComponentDC.h"
#include "CellDC.h"
#include "AtomDC.h"
#include "MultiComponentDC.h"
#include "LoadsManager.h"
#include "AtomDCWidget.h"

#include <Application.h>
#include <InteractiveViewer.h>

#include <vtkUnstructuredGrid.h>
#include <vtkHexahedron.h>
#include <vtkWedge.h>
#include <vtkQuad.h>

#include <QFileInfo>

using namespace camitk;

// -------------------- default constructor --------------------
PMManagerDC::PMManagerDC(const QString &fileName) throw(AbortException) : MeshComponent(fileName) {
    myPM = NULL;
    initialBoundingRadius = 0.0;
    myLoadsManager = NULL;
    myAtomDCWidget = NULL;

    // create the Physical model DCs
    try {

        if (!QFileInfo(myFileName).exists()) {
            throw AbortException("Cannot Open File:" + myFileName.toStdString() + ": file does not exist!");
        }

        //-- create the PhysicalModel
        // directly instantiate physical model
        myPM = new PhysicalModel(myFileName.toStdString().c_str());

        if (!myPM || myPM->getAtoms() == NULL) {
            throw AbortException("Cannot Open PML, either not a valid PML document or no atoms in this PML (for file:" + myFileName.toStdString() + ")");
        }

        computeBoundingRadius();

        // -- initialize fields
        myPopupMenu = NULL;

        setName(myPM->getName().c_str());

        // build all the sub DCs
        buildPhysicalModelDCs();

        // add it in the InteractiveViewer
        setVisibility(InteractiveViewer::get3DViewer(), true);

        // -- create the Loads Manager
        myLoadsManager = new LoadsManager(this);

        //-- init fake representation
        initRepresentation();

        //--reset modification flag
        setModified(false);
    }
    catch (AbortException & e) {
        // has to do that, because the PMManagerDC constructor
        // inherits from Component constructor, which register the object
        // as a member of the DataManager, i.e. myDCs.size() is equal to 1
        // although the DC is not completely initialized!
        // delete all the DCs (the Manager DC included)
        while (getChildren().size() > 0) {
            Component *dc = getChildren().last();
            removeChild(dc);
        }

        childrenComponent.clear();

        // re-throw the exception
        throw e;
    }
}

// --------------------- constructor ----------------------------
PMManagerDC::PMManagerDC(PhysicalModel *pm, const QString &fileName) : MeshComponent(fileName) {
    myPM = pm;
    initialBoundingRadius = 0.0;
    myLoadsManager = NULL;
    myAtomDCWidget = NULL;

    computeBoundingRadius();

    // -- initialize fields
    myPopupMenu = NULL;

    setName(myPM->getName().c_str());

    // build all the sub DCs
    buildPhysicalModelDCs();

    // add it in the InteractiveViewer
    setVisibility(InteractiveViewer::get3DViewer(), true);

    // -- create the Loads Manager
    myLoadsManager = new LoadsManager(this);

    //-- init fake representation
    initRepresentation();

    //--reset modification flag
    setModified(false);
}


// ---------------------- destructor ----------------------------
PMManagerDC::~PMManagerDC() {
    delete myPopupMenu;
    myPopupMenu = NULL;

    delete myAtomDCWidget;
    myAtomDCWidget = NULL;

    delete myLoadsManager;
    myLoadsManager = NULL;

    // remove all the adc myself
    for (unsigned int i = 0;i < myPM->getAtoms()->getNumberOfStructures(); i++) {
        AtomDC * adc = getDC(dynamic_cast<Atom*>(myPM->getAtoms()->getStructure(i)));
        delete adc;
    }

    delete myPM;
    myPM = NULL;
}

// -------------------- initRepresentation --------------------
void PMManagerDC::initRepresentation() {
    myGeometry = new Geometry(getName(), vtkSmartPointer<vtkUnstructuredGrid>::New(), InterfaceGeometry::None);
}

// -------------------- progressOneStep --------------------
void PMManagerDC::progressOneStep() {
    nrOfDoneSteps++;
    float done = (100.0 * ((float)nrOfDoneSteps) / (float)nrOfSteps);

    if (done > 99.0)
        nrOfDoneSteps--;

    Application::setProgressBarValue(50.0 + done*0.5);
}


// -------------------- buildPhysicalModelDCs --------------------
void PMManagerDC::buildPhysicalModelDCs() {
    nrOfDoneSteps = 0;
    nrOfSteps =  1 + myPM->getNumberOfCells();

    // create and add the atoms SC
    StructuralComponent *sc = myPM->getAtoms();

    if (sc)
        new StructuralComponentDC(this, this, sc);

    // create and add the exclusive components
    if (myPM->getNumberOfExclusiveComponents() > 0)
        new MultiComponentDC(this, this, myPM->getExclusiveComponents());

    // create and add the informative component
    if (myPM->getNumberOfInformativeComponents() > 0)
        new MultiComponentDC(this, this, myPM->getInformativeComponents());
}

// -------------------- createPointData --------------------
void PMManagerDC::createPointData() {
    // ask the exclusive and informative component to create the point data
    foreach(Component *child, getChildren()) {
        if (child->isInstanceOf("MultiComponentDC")) {
            dynamic_cast<MultiComponentDC *>(child)->createPointData();
        }
    }
}

// -------------------- destroyPointData --------------------
void PMManagerDC::destroyPointData() {
    // destroy all point data
    foreach(Component *child, getChildren()) {
        if (child->isInstanceOf("MultiComponentDC")) {
            dynamic_cast<MultiComponentDC *>(child)->destroyPointData();
        }
    }
}

// -------------------- setName --------------------
void PMManagerDC::setName(const QString & n) {
    myPM->setName(n.toStdString());
    Component::setName(n);
}

// -------------------- getPopupMenu --------------------
QMenu * PMManagerDC::getPopupMenu(QWidget* parent) {
    if (!myPopupMenu) {
        myPopupMenu = new PMManagerDCPopup(this, parent);
    }

    dynamic_cast<PMManagerDCPopup *>(myPopupMenu)->updateMenuActions();

    return myPopupMenu;
}

//------------------------ getPixmap ---------------------
#include "physicalmodel_20x20.xpm"
QPixmap * PMManagerDC::myPixmap = NULL;
QPixmap PMManagerDC::getIcon() {
    if (!myPixmap) {
        myPixmap = new QPixmap(physicalmodel_20x20);
    }

    return (*myPixmap);
}

// ---------------------- getDC ----------------------------
MultiComponentDC * PMManagerDC::getDC(MultiComponent *mc) {
    if (!mc)
        return NULL;

    std::ComponentDCMapIterator result = myMCDCMap.find(dynamic_cast< ::Component *>(mc));

    return (result == myMCDCMap.end()) ? NULL : dynamic_cast<MultiComponentDC *>(result->second);
}

StructuralComponentDC * PMManagerDC::getDC(StructuralComponent *sc) {
    if (!sc)
        return NULL;

    std::ComponentDCMapIterator result = mySCDCMap.find(dynamic_cast< ::Component *>(sc));

    return (result == mySCDCMap.end()) ? NULL : dynamic_cast<StructuralComponentDC *>(result->second);
}

// ---------------------- static getModifiedFlag ----------------------------
bool PMManagerDC::getModified() const {
    return (modifiedFlag || myLoadsManager->isModified() || myPM->isModified());
}

// -------------------- Cell / DC Map : addCellDCPair --------------------
void PMManagerDC::addCellDCPair(std::ComponentDCPair p) {
    // cout << "add a new pair <" << p.first << "," << p.second << ">" << endl;
    myCDCMap.insert(p);
}

// -------------------- Cell / DC Map : : getDC --------------------
CellDC * PMManagerDC::getDC(Cell *sc) {
    /* Component *c;
     c = (StructuralComponent *) sc;
     cout << "search for " << sc << ": " << (Component *) sc << " / " << c << endl;
     */
    std::ComponentDCMapIterator result = myCDCMap.find(dynamic_cast< ::Component *>(sc));
    return (result == myCDCMap.end()) ? NULL : dynamic_cast<CellDC *>(result->second);
}

// -------------------- getAtomDCWidget --------------------
QWidget * PMManagerDC::getAtomDCWidget(AtomDC *adc, QWidget *parent) {

    // create if needed
    if (!myAtomDCWidget && adc) {
        myAtomDCWidget = new AtomDCWidget(adc, parent);
    }

    // update the properties
    if (myAtomDCWidget) {
        myAtomDCWidget->updateProperties(adc);
    }

    // return the ptr
    return myAtomDCWidget;
}

// -------------------- toPMRenderingMode --------------------
::RenderingMode::Mode PMManagerDC::toPMRenderingMode(InterfaceGeometry::RenderingModes renderingModes) {
    ::RenderingMode converted;

    if (renderingModes & InterfaceGeometry::Surface) {
        converted.setVisible(::RenderingMode::SURFACE, true);
    }

    if (renderingModes & InterfaceGeometry::Wireframe) {
        converted.setVisible(::RenderingMode::WIREFRAME, true);
    }

    if (renderingModes & InterfaceGeometry::Points) {
        converted.setVisible(::RenderingMode::POINTS, true);
    }

    return converted.getMode();
}

// -------------------- toDCRenderingMode --------------------
InterfaceGeometry::RenderingModes PMManagerDC::toDCRenderingMode(::RenderingMode::Mode renderingMode) {
    InterfaceGeometry::RenderingModes converted;

    switch (renderingMode)  {
    case ::RenderingMode::NONE:
        converted = InterfaceGeometry::None;
        break;
    case ::RenderingMode::POINTS:
        converted = InterfaceGeometry::Points;
        break;
    case ::RenderingMode::POINTS_AND_SURFACE:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Surface;
        break;
    case ::RenderingMode::SURFACE:
        converted = InterfaceGeometry::Surface;
        break;
    case ::RenderingMode::WIREFRAME_AND_SURFACE_AND_POINTS:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Surface | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME_AND_SURFACE:
        converted = InterfaceGeometry::Surface | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME_AND_POINTS:
        converted = InterfaceGeometry::Points | InterfaceGeometry::Wireframe;
        break;
    case ::RenderingMode::WIREFRAME:
        converted = InterfaceGeometry::Wireframe;
        break;
    }
    return converted;
}

// --------------- getBounds -------------------
void PMManagerDC::getBounds(double bounds[6]) {
    // compute the bounding radius
    // bounds = [xmin,xmax, ymin,ymax, zmin,zmax]
    bounds[0] = bounds[2] = bounds[4] = 999999999.9f; //MAXFLOAT;
    bounds[3] = bounds[1] = bounds[5] = -999999999.9f; //MINFLOAT;
    Atom *a;
    double pos[3];

    StructuralComponent *theAtoms;
    theAtoms = myPM->getAtoms();

    for (unsigned int i = 0;i < theAtoms->getNumberOfStructures();i++) {
        a = (Atom *) theAtoms->getStructure(i);
        a->getPosition(pos);

        if (pos[0] < bounds[0])
            bounds[0] = pos[0];

        if (pos[0] > bounds[1])
            bounds[1] = pos[0];

        if (pos[1] < bounds[2])
            bounds[2] = pos[1];

        if (pos[1] > bounds[3])
            bounds[3] = pos[1];

        if (pos[2] < bounds[4])
            bounds[4] = pos[2];

        if (pos[2] > bounds[5])
            bounds[5] = pos[2];
    }

}

// --------------- computeBoundingRadius -------------------
void PMManagerDC::computeBoundingRadius() {
    double bounds[6];
    getBounds(bounds);
    double xLength = fabs(bounds[1] - bounds[0]);
    double yLength = fabs(bounds[3] - bounds[2]);
    double zLength = fabs(bounds[5] - bounds[4]);
    initialBoundingRadius = sqrt(xLength * xLength + yLength * yLength + zLength * zLength) / 2.0;
}

