/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include "RendererWidget.h"
#include "GeometricObject.h"
#include "Log.h"

// -- QT stuff
#include <QApplication>
#include <QKeyEvent>
#include <QFileInfo>
#include <QCursor>

// -- VTK stuff
// ---- SmartPointer
#include <vtkSmartPointer.h>
// ---- Interactor
#include <vtkInteractorStyleTrackballCamera.h>
#include <vtkInteractorStyleJoystickCamera.h>
#include <vtkInteractorStyleImage.h>
#include <vtkCallbackCommand.h>
#include <vtkEventQtSlotConnect.h>
// ---- Renderer
#include <vtkRenderWindow.h>
#include <vtkRendererCollection.h>
#include <vtkCamera.h>
// ---- Various vtk stuff
#include <vtkProperty.h>
#include <vtkMatrix4x4.h>
#include <vtkTransform.h>
#include <vtkActor2D.h>
#include <vtkFollower.h>
#include <vtkAssemblyPath.h>
#include <vtkPoints.h>
#include <vtkMath.h>
// ---- Screenshots
#include <vtkWindowToImageFilter.h>
#include <vtkBMPWriter.h>
#include <vtkJPEGWriter.h>
#include <vtkPNGWriter.h>
#include <vtkPostScriptWriter.h>
#include <vtkGL2PSExporter.h>
#include <vtkOBJExporter.h>
#include <vtkVRMLExporter.h>
#include <vtkRIBExporter.h>
// ---- Text in the background
#include <vtkTextSource.h>
#include <vtkVectorText.h>
#include <vtkPolyDataMapper.h>
#include <vtkProperty.h>
#include <vtkTransformPolyDataFilter.h>
#include <vtkTextMapper.h>
#include <vtkTextProperty.h>
// ---- Picking
#include <vtkPicker.h>
#include <vtkProp3DCollection.h>
// ---- Axes
#include <vtkAxesActor.h>
#include <vtkCaptionActor2D.h>
#include <vtkTextActor.h>
#include <vtkProperty2D.h>
// ---- Annotated cube
#include <vtkAnnotatedCubeActor.h>
#include <vtkAssembly.h>
// ---- Color scale
#include <vtkScalarBarWidget.h>
#include <vtkScalarBarActor.h>
#include <vtkWindowLevelLookupTable.h>
#include <vtkScalarBarRepresentation.h>

namespace camitk {
// static instanciation (global variable, global only for this file)
QMap <RendererWidget::ScreenshotFormat , RendererWidget::ScreenshotFormatInfo*> screenshotMap;

//---------------------- Constructor ------------------------
RendererWidget::RendererWidget(QWidget* parent, ControlMode mode) : QVTKWidget(parent) {
    setObjectName( "RendererWidget" );

    //-- initialize screenshotMap only at the first invocation of constructor
    static bool initScreenshotMap = false;
    if (!initScreenshotMap) { //true only for the first instance of LoadCase
        initScreenshotMap = true;
        //populate the map
        buildScreenshotMap();
    }

    //-- Qt stuff
    setMinimumSize(150, 150);
    QSizePolicy policy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    policy.setHeightForWidth(false);
    setSizePolicy(policy);

    //-- display options (this is not a state manage by the renderer, but a property of the actors
    backfaceCulling = false;
    displayCopyright = true;
    rendering3DRedBlue = false;
    displayColorScale = false;
    pointSize = 4.0;

    //-- render window
    vtkSmartPointer<vtkRenderWindow> renderWindow = vtkRenderWindow::New();

    //-- renderer
    renderer = vtkRenderer::New();
    renderWindow->AddRenderer( renderer );
    SetRenderWindow( renderWindow );

    //-- interactor
    interactor = vtkSmartPointer<QVTKInteractor>::New();
    interactor->SetRenderWindow( renderWindow );
    renderWindow->SetInteractor(interactor);

    //-- camera control mode
    controlMode = NONE;
    cameraOrientation = RIGHT_UP;
    interactorStyle = NULL;
    setControlMode(mode);
    setCameraOrientation( cameraOrientation );
    interactor->Initialize();

    //-- connection for picking and other interaction
    connector = vtkSmartPointer<vtkEventQtSlotConnect>::New();
    connector->Connect(interactor, vtkCommand::LeftButtonPressEvent, this, SLOT(leftClick()));
    connector->Connect(interactor, vtkCommand::RightButtonPressEvent, this, SLOT(rightClick()));

    pickingButtonDiverter = vtkSmartPointer<vtkCallbackCommand>::New();
    pickingButtonDiverter->SetClientData(interactorStyle);
    pickingButtonDiverter->SetCallback(RendererWidget::divertionCallback);
    pickingDiverter = false;

    //-- init extra actors

    //-- axes
    axes = vtkSmartPointer<vtkAxesActor>::New();
    axes->SetShaftTypeToCylinder();
    axes->SetXAxisLabelText("x");
    axes->SetYAxisLabelText("y");
    axes->SetZAxisLabelText("z");
    axes->SetTotalLength(0.1, 0.1, 0.1); // 10% of unit length
    vtkSmartPointer<vtkTextProperty> axeXTextProp = vtkSmartPointer<vtkTextProperty>::New();
    axeXTextProp->SetFontSize(20);
    axeXTextProp->BoldOn();
    axeXTextProp->ItalicOn();
    axeXTextProp->ShadowOff();
    axeXTextProp->SetFontFamilyToArial();
    axes->GetXAxisCaptionActor2D()->SetCaptionTextProperty(axeXTextProp);
    // remove the autoscaling so that the font size is smaller than default autoscale
    axes->GetXAxisCaptionActor2D()->GetTextActor()->SetTextScaleModeToNone();
    // set the color
    axes->GetXAxisCaptionActor2D()->GetCaptionTextProperty()->SetColor(1, 0.3, 0.3);
    // make sure the label can be hidden by any geometry, like the axes
    axes->GetXAxisCaptionActor2D()->GetProperty()->SetDisplayLocationToBackground();

    vtkSmartPointer<vtkTextProperty> axeYTextProp = vtkSmartPointer<vtkTextProperty>::New();
    axeYTextProp->ShallowCopy(axeXTextProp);
    axes->GetYAxisCaptionActor2D()->SetCaptionTextProperty(axeYTextProp);
    axes->GetYAxisCaptionActor2D()->GetTextActor()->SetTextScaleModeToNone();
    axes->GetYAxisCaptionActor2D()->GetCaptionTextProperty()->SetColor(0.3, 1, 0.3);
    axes->GetYAxisCaptionActor2D()->GetProperty()->SetDisplayLocationToBackground();

    vtkSmartPointer<vtkTextProperty> axeZTextProp = vtkSmartPointer<vtkTextProperty>::New();
    axeZTextProp->ShallowCopy(axeXTextProp);
    axes->GetZAxisCaptionActor2D()->SetCaptionTextProperty(axeZTextProp);
    axes->GetZAxisCaptionActor2D()->GetTextActor()->SetTextScaleModeToNone();
    axes->GetZAxisCaptionActor2D()->GetCaptionTextProperty()->SetColor(0.3, 0.3, 1.0);
    axes->GetZAxisCaptionActor2D()->GetProperty()->SetDisplayLocationToBackground();
    // add the axes (not visible)
    axes->VisibilityOff();
    addProp(axes);

    //-- annotated cube (right, left, anterior, posterior, inferior, superior)
    annotatedCube = vtkSmartPointer<vtkAnnotatedCubeActor>::New();
    annotatedCube->SetXPlusFaceText("R"); // right
    annotatedCube->SetXMinusFaceText("L"); // left
    annotatedCube->SetYPlusFaceText("A"); // anterior
    annotatedCube->SetYMinusFaceText("P"); // posterior
    annotatedCube->SetZPlusFaceText("I"); // inferior
    annotatedCube->SetZMinusFaceText("S"); // superior
    annotatedCube->SetXFaceTextRotation(-90);
    annotatedCube->SetYFaceTextRotation(180);
    annotatedCube->SetZFaceTextRotation(90);
    annotatedCube->SetFaceTextScale(0.65);
    vtkSmartPointer<vtkTransform> transform = vtkTransform::New();
    transform->Identity();
    transform->Scale(0.05, 0.05, 0.05); // 5% of unit length
    annotatedCube->GetAssembly()->SetPosition(0.0, 0.0, 0.0);
    annotatedCube->GetAssembly()->SetUserTransform( transform );
    vtkSmartPointer<vtkProperty> acProp = annotatedCube->GetCubeProperty();
    acProp->SetColor(0.5, 1, 1);
    acProp = annotatedCube->GetTextEdgesProperty();
    acProp->SetLineWidth(1);
    acProp->SetDiffuse(0);
    acProp->SetAmbient(1);
    acProp->SetColor(0.18, 0.28, 0.23);
    acProp = annotatedCube->GetXPlusFaceProperty();
    acProp->SetColor(1, 0, 0);
    acProp->SetInterpolationToFlat();
    acProp = annotatedCube->GetXMinusFaceProperty();
    acProp->SetColor(1, 0, 0);
    acProp->SetInterpolationToFlat();
    acProp = annotatedCube->GetYPlusFaceProperty();
    acProp->SetColor(0, 1, 0);
    acProp->SetInterpolationToFlat();
    acProp = annotatedCube->GetYMinusFaceProperty();
    acProp->SetColor(0, 1, 0);
    acProp->SetInterpolationToFlat();
    acProp = annotatedCube->GetZPlusFaceProperty();
    acProp->SetColor(0, 0, 1);
    acProp->SetInterpolationToFlat();
    acProp = annotatedCube->GetZMinusFaceProperty();
    acProp->SetColor(0, 0, 1);
    acProp->SetInterpolationToFlat();
    // add the annotatedCube
    annotatedCube->VisibilityOff();
    addProp(annotatedCube);

    // -- text in background
    vtkSmartPointer<vtkTextProperty> copyrightTextProp = vtkSmartPointer<vtkTextProperty>::New();
    copyrightTextProp->ShallowCopy(axeXTextProp);
    copyrightTextProp->SetFontSize(12);
    copyrightTextProp->ShadowOn();

    vtkSmartPointer<vtkTextMapper> copyrightTextMapper = vtkSmartPointer<vtkTextMapper>::New();
    copyrightTextMapper->SetInput(" CamiTK\n(c) TIMC - IMAG");
    copyrightTextMapper->GetTextProperty()->ShallowCopy(copyrightTextProp);
    //  copyrightTextMapper->SetConstrainedFontSize(viewport, targetWidth, targetHeight);

    copyrightTextActor = vtkSmartPointer<vtkActor2D>::New();
    copyrightTextActor->SetMapper(copyrightTextMapper);
    copyrightTextActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();

    //      y ^
    //        |
    //        |
    //  (0,0) +-----------> x
    copyrightTextActor->GetPositionCoordinate()->SetValue(0.85, 0.1);

    // by default the copyright actor is there!
    addProp(copyrightTextActor);

    //-- color scale widget
    // create an inversed the lut
    vtkSmartPointer<vtkWindowLevelLookupTable> vtklup = vtkSmartPointer<vtkWindowLevelLookupTable>::New();
    // inverse hue (from blue to red)
    // other default ( saturation=[1,1] and value=[1,1]) are ok
    vtklup->SetHueRange(2.0 / 3.0, 0.0);
    vtklup->ForceBuild();

    // the widget it self
    colorBarWidget = vtkSmartPointer<vtkScalarBarWidget>::New();
    colorBarWidget->SetInteractor(interactor);
    colorBarWidget->RepositionableOn();
    colorBarWidget->SelectableOn();

    // the color bar parameters
    vtkScalarBarActor *sbActor = colorBarWidget->GetScalarBarActor();
    sbActor->SetLookupTable(vtklup);
    vtkScalarBarRepresentation *rep =  vtkScalarBarRepresentation::SafeDownCast(colorBarWidget->GetRepresentation());
    //sbActor->SetOrientationToHorizontal();
    sbActor->SetNumberOfLabels(9);
    // set the representation and the way the color bar is displayed
    rep->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
    // default = bottom, horizontal
    rep->SetOrientation(VTK_ORIENT_HORIZONTAL);
    rep->SetPosition(0.1, 0.01);
    rep->SetPosition2(0.8, 0.08);
    vtkSmartPointer<vtkTextProperty> sbtProp = vtkSmartPointer<vtkTextProperty>::New();
    sbtProp->SetColor(0.0, 0.0, 0.0);
    sbActor->SetTitleTextProperty(sbtProp);
    sbActor->SetLabelTextProperty(sbtProp);

    // default background
    displayGradient = false;

    // set the background color
    setBackgroundColor(0.0, 0.0, 0.0);
}


//----------------------   Destructor   ------------------------
RendererWidget::~RendererWidget() {
}

//---------------------- buildScreenshotMap ------------------------
void RendererWidget::buildScreenshotMap() {
    // NB: if you add something here, add something in the switch statement of the screenshot method
    screenshotMap[PNG] = new ScreenshotFormatInfo(PNG, "png", "Portable Network Graphics");
    screenshotMap[JPG] = new ScreenshotFormatInfo(JPG, "jpg", "JPEG");
    screenshotMap[BMP] = new ScreenshotFormatInfo(BMP, "bmp", "Bitmap");
    screenshotMap[PS] = new ScreenshotFormatInfo(PS, "ps", "PostScript");
    screenshotMap[EPS] = new ScreenshotFormatInfo(EPS, "eps", "Encapsulated PostScript");
    screenshotMap[PDF] = new ScreenshotFormatInfo(PDF, "pdf", "Portable Document Format");
    screenshotMap[TEX] = new ScreenshotFormatInfo(TEX, "TeX", "LaTeX(text only)");
    screenshotMap[SVG] = new ScreenshotFormatInfo(SVG, "svg", "Scalable Vector Graphics");
    screenshotMap[OBJ] = new ScreenshotFormatInfo(OBJ, "obj", "Alias Wavefront .OBJ ");
    screenshotMap[RIB] = new ScreenshotFormatInfo(RIB, "rib", "RenderMan/BMRT .RIB");
    screenshotMap[VRML] = new ScreenshotFormatInfo(VRML, "VRML", "VRML 2.0");
    screenshotMap[NOT_SUPPORTED] = new ScreenshotFormatInfo();
}

//---------------------- getScreenshotFormatInfo ------------------------
const RendererWidget::ScreenshotFormatInfo * RendererWidget::getScreenshotFormatInfo(unsigned int id) {
    if (id > NOT_SUPPORTED)
        return RendererWidget::getScreenshotFormatInfo(NOT_SUPPORTED);
    else
        return RendererWidget::getScreenshotFormatInfo((ScreenshotFormat)id);
}

const RendererWidget::ScreenshotFormatInfo * RendererWidget::getScreenshotFormatInfo(QString ext)  {
    unsigned int i = 0;
    bool found = false;

    while (i != NOT_SUPPORTED && !found) {
        found = (RendererWidget::getScreenshotFormatInfo(i)->extension == ext);
        i++;
    }

    if (found) {
        i--;
        return RendererWidget::getScreenshotFormatInfo(i);
    }
    else
        return RendererWidget::getScreenshotFormatInfo(NOT_SUPPORTED);
}

const RendererWidget::ScreenshotFormatInfo * RendererWidget::getScreenshotFormatInfo(ScreenshotFormat f)  {
    return screenshotMap.value(f);
}

//---------------------- screenshot ------------------------
void RendererWidget::screenshot(QString filename) {
    const char * fileprefix = QString(QFileInfo(filename).absolutePath() + "/" + QFileInfo(filename).baseName()).toStdString().c_str();

    // declare exporter/writer that can be used for more than one format
    vtkSmartPointer<vtkImageWriter> imageWriter = NULL;
    vtkSmartPointer<vtkExporter> exporter = NULL;

    // check the extension
    QString extension = QFileInfo(filename).suffix().toLower();

    // default is PNG
    if (extension.length() == 0) {
        extension = screenshotMap.value(PNG)->extension;
        filename += "." + extension;
    }

    // it is the correct format
    switch (getScreenshotFormatInfo(extension)->type) {
    case BMP:
        imageWriter = vtkSmartPointer<vtkBMPWriter>::New();
        break;
    case JPG:
        imageWriter = vtkSmartPointer<vtkJPEGWriter>::New();
        break;
    case PS:
        imageWriter = vtkSmartPointer<vtkPostScriptWriter>::New();
        break;
    case EPS:
        exporter = vtkSmartPointer<vtkGL2PSExporter>::New();
        vtkGL2PSExporter::SafeDownCast(exporter)->CompressOff();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        vtkGL2PSExporter::SafeDownCast(exporter)->DrawBackgroundOn();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFileFormatToEPS();
        break;
    case PDF:
        exporter = vtkSmartPointer<vtkGL2PSExporter>::New();
        vtkGL2PSExporter::SafeDownCast(exporter)->CompressOff();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        vtkGL2PSExporter::SafeDownCast(exporter)->DrawBackgroundOn();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFileFormatToPDF();
        break;
    case TEX:
        exporter = vtkSmartPointer<vtkGL2PSExporter>::New();
        vtkGL2PSExporter::SafeDownCast(exporter)->CompressOff();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        vtkGL2PSExporter::SafeDownCast(exporter)->DrawBackgroundOn();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFileFormatToTeX();
        break;
        /* TODO export to POV and export to X3D using corresponding writer
        vtkPOVExporter *povexp = vtkPOVExporter::New();
        povexp->SetRenderWindow(renWin);
        povexp->SetFileName("TestPOVExporter.pov");
        povexp->Write();
          */
    case SVG:
        exporter = vtkGL2PSExporter::New();
        vtkGL2PSExporter::SafeDownCast(exporter)->CompressOff();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        vtkGL2PSExporter::SafeDownCast(exporter)->DrawBackgroundOn();
        vtkGL2PSExporter::SafeDownCast(exporter)->SetFileFormatToSVG();
        break;
    case OBJ:
        exporter = vtkSmartPointer<vtkOBJExporter>::New();
        vtkOBJExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        break;
    case RIB:
        exporter = vtkSmartPointer<vtkRIBExporter>::New();
        vtkRIBExporter::SafeDownCast(exporter)->SetFilePrefix(fileprefix);
        break;
    case VRML:
        exporter = vtkSmartPointer<vtkVRMLExporter>::New();
        vtkVRMLExporter::SafeDownCast(exporter)->SetFileName(filename.toStdString().c_str());
        break;
    case NOT_SUPPORTED:
        break;
    case PNG: // default
    default:
        imageWriter = vtkSmartPointer<vtkPNGWriter>::New();
        break;
    }

    if (exporter) {
        exporter->SetInput(GetRenderWindow());
        exporter->Write();
    }
    else {
        if (imageWriter) {
            // create the proper vtk filter for export
            vtkSmartPointer<vtkWindowToImageFilter> imageFilter = vtkSmartPointer<vtkWindowToImageFilter>::New();
            imageFilter->SetInput(GetRenderWindow());

            // write image
            imageWriter->SetInputConnection(imageFilter->GetOutputPort());
            imageWriter->SetFileName(filename.toStdString().c_str());
            imageWriter->Write();
        }
    }
}

//---------------------- refresh ------------------------
void RendererWidget::refresh() {
    update();
}


//---------------------- setBackgroundColor ------------------------
void RendererWidget::setBackgroundColor(double r, double g, double b) {
    renderer->SetBackground(r, g, b);
    // update copyright text color
    vtkTextMapper::SafeDownCast(copyrightTextActor->GetMapper())->GetTextProperty()->SetColor(1.0 - r, 1.0 - g, 1.0 - b);
}

//---------------------- getBackgroundColor ------------------------
void RendererWidget::getBackgroundColor(double &r, double &g, double &b) {
    renderer->GetBackground(r, g, b);
}

//--------------------- setGradientBackground --------------
bool RendererWidget::getGradientBackground() {
    return displayGradient;
}

//--------------------- setGradientBackground --------------
void RendererWidget::setGradientBackground(bool gb) {
    removeProp(copyrightTextActor);

    displayGradient = gb;

    if (displayGradient) {
        // Setup the background gradient
        renderer->GradientBackgroundOn();
        renderer->SetBackground(0.823, 0.823, 0.921); // light blue in the bottom
        renderer->SetBackground2(1, 1, 1); // white on top
    }

    // have to do that to force the display of (c) on top
    if (displayCopyright)
        addProp(copyrightTextActor);
}

//--------------------- toggleCopyright --------------
void RendererWidget::toogle3DRedBlue() {
    rendering3DRedBlue = !rendering3DRedBlue;
    if (rendering3DRedBlue) {
        // 3D red/blue
        GetRenderWindow()->StereoCapableWindowOn();
        GetRenderWindow()->SetStereoTypeToRedBlue();
        GetRenderWindow()->StereoRenderOn(); //On();
        GetRenderWindow()->StereoUpdate();
    }
    else {
        GetRenderWindow()->StereoRenderOff(); //On();
        GetRenderWindow()->StereoUpdate();
    }
}

//--------------------- toggleCopyright --------------
void RendererWidget::toggleCopyright(bool c) {
    displayCopyright = c;
    removeProp(copyrightTextActor);

    if (displayCopyright) {
        addProp(copyrightTextActor);
    }
}

// --------------------- getControlMode ----------------------------
RendererWidget::ControlMode RendererWidget::getControlMode() const {
    return controlMode;
}

// --------------------- setControlMode ----------------------------
void RendererWidget::setControlMode(ControlMode mode) {
    if (mode != controlMode) {
        controlMode = mode;

        switch (controlMode) {
        case JOYSTICK:
            interactorStyle = vtkSmartPointer<vtkInteractorStyleJoystickCamera>::New();
            break;

        case TRACKBALL:
            interactorStyle = vtkSmartPointer<vtkInteractorStyleTrackballCamera>::New();
            break;
        case TRACKBALL_2D:
            interactorStyle = vtkSmartPointer<vtkInteractorStyleImage>::New();
            break;

        default : //NONE
            interactorStyle = NULL;
            break;
        }

        interactor->SetInteractorStyle(interactorStyle);

        update();
    }
}


// --------------------- pick -----------------------------------
void RendererWidget::pick() {
    // Screen coordinates
    QPoint mousePointerPos = QCursor::pos();

	// Widget coordinates
    QPoint inWidgetCoord = mapFromGlobal(mousePointerPos);

    pickActor(inWidgetCoord.x(), inWidgetCoord.y());
//	pickActor(mousePointerPos.x(), mousePointerPos.y());
}

// --------------------- pickActor -----------------------------------
void RendererWidget::pickActor(int x, int y) {
    //-- if there is no picker or a picker that we can't use with (x,y), then picking was not initialized, so do nothing!
    vtkSmartPointer<vtkPicker> picker = vtkPicker::SafeDownCast(GetInteractor()->GetPicker());
    if (!picker)
        return;

    // get the current renderer for picking
    renderer = GetInteractor()->FindPokedRenderer(GetInteractor()->GetEventPosition()[0], GetInteractor()->GetEventPosition()[1]);

    // perform picking operation
    int picked = picker->Pick(GetInteractor()->GetEventPosition()[0],
                 GetInteractor()->GetEventPosition()[1],
                 0,  // always zero.
                 GetInteractor()->GetRenderWindow()->GetRenderers()->GetFirstRenderer()); //x, height() - y - 1, 0.0, renderer);
    
    if (picked) {
        // if something was picked, cry it out loud!
        if (picker->GetProp3Ds()->GetLastProp() != NULL)
            emit actorPicked(picker);
    }

}

// ----------------------- setBackfaceCulling -----------------------------
void RendererWidget::setBackfaceCulling(bool culling) {
    backfaceCulling = culling;

    // update all currently managed actors
    vtkSmartPointer<vtkActorCollection> ac = renderer->GetActors();
    ac->InitTraversal();
    vtkSmartPointer<vtkActor> actor;
    while (actor = ac->GetNextActor())
        actor->GetProperty()->SetBackfaceCulling(backfaceCulling);

    update();
}

// ----------------------- getBackfaceCulling -----------------------
bool RendererWidget::getBackfaceCulling() const {
    return backfaceCulling;
}

// --------------- setCameraOrientation ------------------------------------------------
void RendererWidget::setCameraOrientation( RendererWidget::CameraOrientation a) {
    cameraOrientation = a;
}

// --------------- getCameraOrientation ------------------------------------------------
RendererWidget::CameraOrientation RendererWidget::getCameraOrientation() const {
    return cameraOrientation;
}


// ----------------------- setLightFollowCamera -----------------------
void RendererWidget::setLightFollowCamera(bool lightFollow) {
    lightFollowCamera = lightFollow;
    interactor->LightFollowCameraOff(); // legacy
    renderer->SetLightFollowCamera(lightFollowCamera);
}

// ----------------------- getLightFollowCamera -----------------------
bool RendererWidget::getLightFollowCamera() const {
    return lightFollowCamera;
}

// ----------------------- setPointSize -----------------------
void RendererWidget::setPointSize(double size) {
    pointSize = size;

    // update all currently managed actors
    vtkSmartPointer<vtkActorCollection> ac = renderer->GetActors();
    ac->InitTraversal();
    vtkSmartPointer<vtkActor> actor;
    while (actor = ac->GetNextActor())
        actor->GetProperty()->SetPointSize(pointSize);

    update();
}

// ----------------------- getPointSize -----------------------
double RendererWidget::getPointSize() const {
    return pointSize;
}

//----------------------- toggleAxes -------------------
void RendererWidget::toggleAxes(bool f) {
    axes->SetVisibility(f);
    annotatedCube->SetVisibility(f);
    // remove the axes and annotated cube from the computed bounds
    // (otherwise if they are the biggest object, they are going to set the bounds!)
    axes->SetUseBounds(false);
    annotatedCube->SetUseBounds(false);
    // update sizes
    updateAxes();
}

// ----------------------- updateAxes -----------------------
void RendererWidget::updateAxes() {
    // set the size depending on the bounding box
    // bounds[] = xmin, xmax, ymin, ymax, zmin, zmax
    double bounds[6];
    computeVisiblePropBounds(bounds);
    double xLength, yLength, zLength;
    xLength = fabs(bounds[1] - bounds[0]);
    yLength = fabs(bounds[3] - bounds[2]);
    zLength = fabs(bounds[5] - bounds[4]);
    double maxLength = (xLength > yLength) ?
                       ((xLength > zLength) ? xLength : zLength) :
                               ((yLength > zLength) ? yLength : zLength);

    // the size of the axe is about 10% the size of the bound
    maxLength /= 10.0;
    axes->SetTotalLength(maxLength, maxLength, maxLength);

    // the size of the annotated cube is about 5% of the size of the bound
    maxLength /= 2.0;
    vtkSmartPointer<vtkTransform> transform = vtkTransform::New();
    transform->Identity();
    transform->Scale(maxLength, maxLength, maxLength);
    annotatedCube->GetAssembly()->SetUserTransform(transform);
}

// ----------------------- setColorScale -----------------------
void RendererWidget::setColorScale(bool state) {
    displayColorScale = state;
    if (displayColorScale) {
        colorBarWidget->EnabledOn();
    }
    else {
        colorBarWidget->EnabledOff();
    }
}

// ----------------------- getColorScale -----------------------
bool RendererWidget::getColorScale() const {
    return displayColorScale;
}

//------------------------- setColorScaleMinMax ----------------------------
void RendererWidget::setColorScaleMinMax(double min, double max) {
    if (!displayColorScale) {
        setColorScale(true);
    }

    vtkLookupTable::SafeDownCast(colorBarWidget->GetScalarBarActor()->GetLookupTable())->SetTableRange(min, max);
}

//------------------------- setColorScaleTitle ----------------------------
void RendererWidget::setColorScaleTitle(QString title) {
    if (!displayColorScale) {
        setColorScale(true);
    }

    colorBarWidget->GetScalarBarActor()->SetTitle(title.toStdString().c_str());
}

// ----------------------- addProp -----------------------
void RendererWidget::addProp(vtkSmartPointer<vtkProp> prop, bool refresh) {
    if (prop != NULL && !containsProp(prop)) {
        //-- "intelligence" is defined here
        // NOTE: please add a document any addition/modification in the header autodocumentation!

        //-- for actor set backface culling, point size
        vtkSmartPointer<vtkActor> aPart = vtkActor::SafeDownCast(prop);
        if (aPart) {
            aPart->InitPathTraversal();
            vtkSmartPointer<vtkAssemblyPath> path;
            while ((path = aPart->GetNextPath()) != NULL) {
                vtkSmartPointer<vtkAssemblyNode> node = path->GetLastNode();
                vtkSmartPointer<vtkActor> a;
                if (node && (a = vtkActor::SafeDownCast(node->GetViewProp()))) {
                    a->GetProperty()->SetBackfaceCulling(backfaceCulling);

                    // check if the point representation is active, and if it is, set the proper size
                    if (a->GetProperty()->GetRepresentation() == VTK_POINTS)
                        a->GetProperty()->SetPointSize(pointSize);
                }
            }
        }

        // now add it!
        renderer->AddViewProp(prop);

        if (refresh)
            // update axes size
            updateAxes();
    }

}

// ----------------------- containsProp -----------------------
bool RendererWidget::containsProp(vtkSmartPointer<vtkProp> prop) {
    return (renderer->GetViewProps()->IsItemPresent(prop) != 0);
}

// ----------------------- removeProp -----------------------
void RendererWidget::removeProp(vtkSmartPointer<vtkProp> prop, bool refresh) {
    if (prop != NULL && containsProp(prop)) {
        renderer->RemoveViewProp(prop);
        if (refresh)
            // update axes size
            updateAxes();
    }
}

// --------------------- actorTransform --------------------------------
void RendererWidget::actorTransform(vtkSmartPointer<vtkActor> actor, double *boxCenter, int numRotation,
                                    double **rotate, double *translate, double *scale) {
    vtkSmartPointer<vtkMatrix4x4> oldMatrix = vtkSmartPointer<vtkMatrix4x4>::New();
    actor->GetMatrix(oldMatrix);

    double orig[3];
    actor->GetOrigin(orig);

    vtkSmartPointer<vtkTransform> newTransform = vtkSmartPointer<vtkTransform>::New();
    newTransform->PostMultiply();

    if (actor->GetUserMatrix() != NULL) {
        newTransform->SetMatrix(actor->GetUserMatrix());
    }
    else {
        newTransform->SetMatrix(oldMatrix);
    }

    if ((rotate != NULL) || (scale != NULL)) {
        newTransform->Translate(-(boxCenter[0]), -(boxCenter[1]),
                                -(boxCenter[2]));

        if (rotate != NULL) {
            for (int i = 0; i < numRotation; i++) {
                newTransform->RotateWXYZ(rotate[i][0], rotate[i][1],
                                         rotate[i][2], rotate[i][3]);
            }
        }

        if (scale != NULL && (scale[0] * scale[1] * scale[2]) != 0.0) {
            newTransform->Scale(scale[0], scale[1], scale[2]);
        }

        newTransform->Translate(boxCenter[0], boxCenter[1], boxCenter[2]);

        // now try to get the composit of rotate, and scale
        newTransform->Translate(-(orig[0]), -(orig[1]), -(orig[2]));
        newTransform->PreMultiply();
        newTransform->Translate(orig[0], orig[1], orig[2]);
    }

    // Last - Do the TRANSLATION
    if (translate != NULL) {
        newTransform->PostMultiply();
        newTransform->Translate(translate[0], translate[1],
                                translate[2]);
    }

    // set the new Actor - Matrix
    if (actor->GetUserMatrix() != NULL) {
        newTransform->GetMatrix(actor->GetUserMatrix());
    }
    else {
        actor->SetPosition(newTransform->GetPosition());
        actor->SetScale(newTransform->GetScale());
        actor->SetOrientation(newTransform->GetOrientation());
    }

}

// ----------------------- computeVisiblePropBounds -----------------------
void RendererWidget::computeVisiblePropBounds(double bounds[6]) {
    renderer->ComputeVisiblePropBounds(bounds);
}

// --------------------- RotateCamera---------------------
void RendererWidget::rotateCamera(double angle, int axe) {
    vtkSmartPointer<vtkCamera> cam = renderer->GetActiveCamera();

    switch (axe) {
    case 0 : //axe X
        cam->Azimuth(0.0);
        cam->Elevation(angle);
        cam->OrthogonalizeViewUp();
        break;

    case 1: //axe y
        cam->Azimuth(angle);
        cam->Elevation(0.0);
        cam->OrthogonalizeViewUp();
        break;

    default :
        break;
    }

    renderer->ResetCameraClippingRange();

    update();
}


// --------------------- keyPressEvent --------------------------------
void RendererWidget::keyPressEvent(QKeyEvent* e) {
    // NOTE: nothing but the following line should be here
    // All interaction should be defined directly in the class that use this instance
    // for example, see InteractiveViewer::keyPressEvent method
    e->ignore();
}


// --------------------- mouseReleaseEvent ----------------------------------
void RendererWidget::mouseReleaseEvent(QMouseEvent* event) {
    interactorStyle->RemoveObserver(pickingButtonDiverter);
    pickingDiverter = false;
    update();
    QVTKWidget::mouseReleaseEvent(event);
    vtkSmartPointer<vtkPicker> emptyPicker;
    emit actorPicked(emptyPicker);
}

//---------------------------- slotLeftClick ---------------------------------
void RendererWidget::leftClick() {
    // Ctrl+Left click
    if (QApplication::keyboardModifiers() == Qt::ControlModifier) {
        // divert the left button press event, so that it won't
        // have any effect on the scene while continuously allow picking
        if (!pickingDiverter) {
            pickingDiverter = true;
            interactorStyle->AddObserver(vtkCommand::MouseMoveEvent, pickingButtonDiverter);
        }

        // do the picking action
        pick();
    }
}

//---------------------------- slotRightClick ---------------------------------
void RendererWidget::rightClick() {
    emit rightButtonPressed();
}

// --------------------- mouseMoveEvent ----------------------------------
void RendererWidget::mouseMoveEvent(QMouseEvent* event) {
    if (QApplication::keyboardModifiers() == Qt::ControlModifier && QApplication::mouseButtons() == Qt::LeftButton) {
        // continuous picking mode:
        // you don't have to release the button to do another picking
        // just keep the button pressed
        pick();
    }
    QVTKWidget::mouseMoveEvent(event);
}


// --------------------- resetCamera ----------------------
void RendererWidget::resetCamera() {
    resetCameraSettings();

    // auto calculate/update the bounds
    renderer->ResetCamera();
    renderer->ResetCameraClippingRange();

    update();
}

void RendererWidget::resetCamera(double bounds[6]) {
    resetCameraSettings();

    // set the proper bound
    renderer->ResetCamera(bounds);
    renderer->ResetCameraClippingRange(bounds);

    update();
}

// --------------------- resetCameraSettings ----------------------
void RendererWidget::resetCameraSettings() {
    vtkSmartPointer<vtkCamera> cam = renderer->GetActiveCamera();

    switch (cameraOrientation) {
    case LEFT_UP:
        // default position
        cam->SetPosition(0, 0, -1);
        cam->SetFocalPoint(0, 0, 0);
        cam->SetViewUp(0, 1, 0);
        cam->OrthogonalizeViewUp();
        break;
    case RIGHT_DOWN:
        // default position
        cam->SetPosition(0, 0, -1);
        cam->SetFocalPoint(0, 0, 0);
        cam->SetViewUp(0, -1, 0);
        cam->OrthogonalizeViewUp();
        break;
    case RIGHT_UP:
    default:
        cam->SetPosition(0, 0, 1);
        cam->SetFocalPoint(0, 0, 0);
        cam->SetViewUp(0, 1, 0);
        cam->OrthogonalizeViewUp();
        break;
    }
}

// --------------------- getCameraSettings ----------------------
void RendererWidget::getCameraSettings(double position[3], double focalPoint[3], double viewUp[3]) {
    vtkSmartPointer<vtkCamera> cam = renderer->GetActiveCamera();

    cam->GetPosition(position);
    cam->GetFocalPoint(focalPoint);
    cam->GetViewUp(viewUp);
}

// --------------------- setActiveCamera --------------------------
void RendererWidget::setActiveCamera( vtkCamera * cam ) {
    renderer->SetActiveCamera( cam );
}

// --------------------- setActiveCamera --------------------------
vtkCamera *RendererWidget::getActiveCamera() {
    return renderer->GetActiveCamera();
}

// --------------------- getMouse3DCoordinates ----------------------
void RendererWidget::getMouse3DCoordinates(double & x, double & y, double & z) {
    QPoint cursor_pos = mapFromGlobal(QCursor::pos());
    renderer = GetInteractor()->FindPokedRenderer(cursor_pos.x(), cursor_pos.y());

    vtkSmartPointer<vtkCoordinate> c1 = vtkSmartPointer<vtkCoordinate>::New();
    c1->SetCoordinateSystemToDisplay();
    c1->SetValue(double(cursor_pos.x()), double(cursor_pos.y()));
    double * position = c1->GetComputedWorldValue(renderer);

    x = position[0];
    y = position[1];
    z = position[2];
}

// --------------------- setPicker ----------------------
void RendererWidget::setPicker(vtkSmartPointer<vtkPicker> woodyWood) {
    if (GetInteractor()) {
        // I know, it is really Woody Wood Pecker, but you know, with a french accent...
        GetInteractor()->SetPicker(woodyWood);
    }
}


}
