/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
// CamiTK includes
#include "DicomComponent.h"
#include <ImageOrientationHelper.h>
#include <Log.h>
#include "DicomParser.h"

// Qt includes
#include <QDir>

// VTK includes
#include <vtkStringArray.h>
#include <vtkPointData.h>

// GDCM includes
#include <vtkGDCMImageReader.h>
#include <gdcmReader.h>
#include <gdcmScanner.h>
#include <gdcmIPPSorter.h>



using namespace camitk;

// --------------- Constructor -------------------
DicomComponent::DicomComponent(DicomSerie* dicomSerie) throw(AbortException) : ImageComponent("") {
    std::vector<std::string> stdFileNames = dicomSerie->getStdFileNames();


    // scan files for serie description
    gdcm::Scanner scanner;
    gdcm::Tag serieDescriptionTag = gdcm::Tag(0x0008, 0x103e);
    scanner.AddTag(serieDescriptionTag);
    scanner.Scan(stdFileNames);
    setName(QString(scanner.GetValue(stdFileNames.at(0).c_str(), serieDescriptionTag)));
//    std::cout << "SERIE name = " << getName().toStdString() << std::endl;

    // sort the files according to Patient position ect ...
    gdcm::IPPSorter ippSorter;
    ippSorter.SetComputeZSpacing(true);
    ippSorter.SetZSpacingTolerance(0.1);
    if(!ippSorter.Sort(stdFileNames))
        CAMITK_ERROR("DicomSerieComponent", "DicomSerieComponent", "IPPSorter sorting failed. Try to adjust Z spacing tolerance.")
        vtkSmartPointer<vtkStringArray> fileNamesSorted = vtkSmartPointer<vtkStringArray>::New();
    std::vector< std::string > files = ippSorter.GetFilenames();
//    std::cout << "IPP Sorter has found " << files.size() << " files" << std::endl;
    foreach(std::string file, files)
    fileNamesSorted->InsertNextValue(file.c_str());

    // get image orientation information
    // we need to get the rotation matrix from tag "Direct cos angle"
    ImageOrientationHelper::PossibleImageOrientations orientation = readDirectCosinesAngle(files);


    // get the image position orientation
    // TODO read the tag (0018, 5100) Patient position

    // create image data corresponding to the component
    imageReader = vtkGDCMImageReader::New();
    imageReader->SetFileNames(fileNamesSorted);
    imageReader->Update();
    vtkSmartPointer<vtkImageData> rawData = imageReader->GetOutput();

    // TODO Store Image position patient as a CamiTK property?

    // Update Z-spacing
    // vtkGDCMImageReader missses this information, see: http://gdcm.sourceforge.net/2.4/html/classvtkGDCMImageReader.html#details
    double *spacing = imageReader->GetDataSpacing();
    double ippZSpacing = ippSorter.GetZSpacing();
    if (ippZSpacing == 0) {
        // if IPPSorter returned 0, it means that its input parameters were wrong, see: http://gdcm.sourceforge.net/2.4/html/classgdcm_1_1IPPSorter.html#aaef7af132da723719c52be5f8d3cb21d
        CAMITK_WARNING("DicomSerieComponent", "DicomSerieComponent", "IPPSorter Z spacing computation failed, try to adjust the tolerance. Setting it back to 1")
        ippZSpacing = 1;
    }
    vtkImageChangeInformation *imageInfoChanger = vtkImageChangeInformation::New();
    imageInfoChanger->SetInput(rawData); //translatedData);
//    std::cout << "spacing found = (" << spacing[0] << ", " << spacing[1] << ", " << ippZSpacing << ")" << std::endl;
    imageInfoChanger->SetOutputSpacing(spacing[0], spacing[1], ippZSpacing);
    imageInfoChanger->Update();

    // Flip all actors in the Y axis
    // Explanation:
    // DICOM stores the upper left pixel as the first pixel in an image.
    // However, VTK stores the lower left pixel as the first pixel in an image
    // As we based our image frame on RAI coordinate (DICOM LPS located at the bottom left hand corner)
    // with a Radiologist point of view (Axial, Coronal, Sagittal)
    // To ensure that VTK frame is in RAI coordinates, we flip the image in the Y axis
    vtkSmartPointer<vtkImageFlip> flipYFilter = vtkSmartPointer<vtkImageFlip>::New();
    flipYFilter->SetFilteredAxis(1); // flip y axis
    flipYFilter->SetInputConnection(imageInfoChanger->GetOutputPort());
    flipYFilter->Update();

    // Retrieve image position patient information


    // put back to the origin
    vtkSmartPointer<vtkImageData> image = flipYFilter->GetOutput();
    setImageData(image, false, orientation);

    // Wait for the LUT update in CamiTK and / or support for color image
    // updateLUT();


}

// --------------- destructor -------------------
DicomComponent::~DicomComponent() {

}

// --------------- updateLUT -------------------
void DicomComponent::updateLUT() {
    // Update LUT
    // Initialize our lut with vtkGDCMImageReader information found, as our LUT needs repair ...
    std::cout << "DicomSerieComponent::updateLUT >> ";
    double range[2] = {0.0, 0.0};
    imageReader->GetOutput()->GetScalarRange(range);
    getLut()->SetRange(range);  // we need to set up range and table values
    getLut()->SetNumberOfTableValues(abs(range[0]) + abs(range[1]));
    std::cout << "UPDATE DONE" << std::endl;
}

// --------------- readDirectCosinesAngle -------------------
camitk::ImageOrientationHelper::PossibleImageOrientations DicomComponent::readDirectCosinesAngle(const std::vector< std::string > & fileNames) const {

    // scan files Image Orientation Patient
    gdcm::Scanner scanner;
    gdcm::Tag iopTag = gdcm::Tag(0x0020, 0x0037);
    scanner.AddTag(iopTag);
    scanner.Scan(fileNames);
    std::string value = scanner.GetValue(fileNames[0].c_str(), iopTag);

    // convert the string into the appropriate couple of cosine vectors
    double x[3] = {0.0, 0.0, 0.0};
    double y[3] = {0.0, 0.0, 0.0};
    std::sscanf(value.c_str(), "%lf\\%lf\\%lf\\%lf\\%lf\\%lf", &x[0], &x[1], &x[2], &y[0], &y[1], &y[2]);

//    std::cout << "DicomSerieComponent::DicomSerieComponent value = " << value << std::endl;
//    std::cout << "DicomSerieComponent::DicomSerieComponent x = (" << x[0] << ", " << x[1] << ", " << x[2] << ")" << std::endl;
//    std::cout << "DicomSerieComponent::DicomSerieComponent y = (" << y[0] << ", " << y[1] << ", " << y[2] << ")" << std::endl;

    // get the 90 degrees closest cosines for each vector
    x[0] = roundCosine(x[0]);
    x[1] = roundCosine(x[1]);
    x[2] = roundCosine(x[2]);
    y[0] = roundCosine(y[0]);
    y[1] = roundCosine(y[1]);
    y[2] = roundCosine(y[2]);

    // return the appropriate ImageOrientation corresponding to this cosines
    // we only need to check one component of vector X and Y as they are colinear (other components are 0) to one of the frame vectors

    if((x[0] == 1.0) && (y[1] == 1.0)) // identity matrix => Image frame is the scanner frame
        return ImageOrientationHelper::RAI;

    // other cases Image frame is different
    if(x[0] == 1.0) {
        if(y[1] == -1.0)
            return ImageOrientationHelper::RPS;
        if(y[2] == 1.0)
            return ImageOrientationHelper::RIP;
        if(y[2] == -1.0)
            return ImageOrientationHelper::RSA;
    }
    if(x[0] == -1.0) {
        if(y[1] == 1.0)
            return ImageOrientationHelper::LAS;
        if(y[2] == -1.0)
            return ImageOrientationHelper::LPI;
        if(y[2] == 1.0)
            return ImageOrientationHelper::LIA;
    }
    if(x[1] == 1.0) {
        if(y[0] == 1.0)
            return ImageOrientationHelper::ARS;
        if(y[0] == -1.0)
            return ImageOrientationHelper::ALI;
        if(y[2] == 1.0)
            return ImageOrientationHelper::AIR;
        if(y[2] == -1.0)
            return ImageOrientationHelper::ASL;
    }
    if(x[1] == -1.0) {
        if(y[0] == 1.0)
            return ImageOrientationHelper::PRI;
        if(y[0] == -1.0)
            return ImageOrientationHelper::PLS;
        if(y[2] == 1.0)
            return ImageOrientationHelper::PIL;
        if(y[2] == -1.0)
            return ImageOrientationHelper::PSR;
    }
    if(x[2] == 1.0) {
        if(y[0] == 1.0)
            return ImageOrientationHelper::IRA;
        if(y[0] == -1.0)
            return ImageOrientationHelper::ILP;
        if(y[1] == 1.0)
            return ImageOrientationHelper::IAL;
        if(y[1] == -1.0)
            return ImageOrientationHelper::IPR;
    }
    if(x[2] == -1.0) {
        if(y[0] == 1.0)
            return ImageOrientationHelper::SRP;
        if(y[0] == -1.0)
            return ImageOrientationHelper::SLA;
        if(y[1] == 1.0)
            return ImageOrientationHelper::SAR;
        if(y[1] == -1.0)
            return ImageOrientationHelper::SPL;
    }

    // should never return UNKNOW
    CAMITK_WARNING("DicomSerieComponent", "readDirectCosinesAngle", "No orientation found for this image (direct cosines).")
    return ImageOrientationHelper::UNKNOWN;

}

// --------------- roundCosine -------------------
double DicomComponent::roundCosine(const double & value) const {
    if(value < -0.5)
        return -1.0;
    if((value >= -0.5) && (value <= 0.5))
        return 0.0;
    else
        return 1.0;
}
