/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef DICOMPARSER_H
#define DICOMPARSER_H

// CamiTK includes
#include <DicomSerie.h>

// Qt includes
#include <QMap>
#include <QList>
#include <QString>

// C++ includes
#include <stdlib.h>
#include <vector>


/**
 * @ingroup group_cepimaging_components_dicomimage
 *
 * @brief DicomParser allows one to parse a directory DICOM files looking for studies and series
 *
 * @note This class is part of the model (in the MVP pattern) of the DICOM COMPONENT
 *
 */
class DicomParser {

public:

    DicomParser() {};

    /// Default Destructor
    virtual ~DicomParser() {};

    /**
     * @brief Parse the input directory for DICOM series.
     * @param directory The input directory to parse DICOM image files.
     * @return A list of DicomSerie, high level CamiTK object which store information of a SERIE
     */
    static QList<DicomSerie*> parseDirectory(const QString & directory);

private:
    /**
     * @brief Convert a simple std list of string into a QList of QString.
     * @param inputList std list of string to convert.
     * @return QList of QString version of the input std list.
     */
    static QList<QString> stdListOfStringToQt(const std::vector< std::string > & inputList);

    /**
     * @brief Convert a QList of QString into a std list of string.
     * @param inputList QList of QString to convert.
     * @return std list of string version of the input Qt list.
     */
    static std::vector< std::string > qtListOfStringToStd(const QList<QString> & inputList);

    /**
     * @brief Retrieve the SERIE acquisition date.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the acquisition date of this SERIE.
     */
    static QDate getAcquisitionDate(const std::vector<std::string> & serieFileNames);

    /**
     * @brief Retrieve the SERIE acquisition time.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the acquisition time of this SERIE.
     */
    static QTime getAcquisitionTime(const std::vector<std::string> & serieFileNames);

    /**
     * @brief Retrieve the patient name information from the input files.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the patient name.
     */
    static QString getPatientName(const std::vector<std::string> & serieFileNames);

    /**
     * @brief Retrieve the serie name.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the serie name.
     */
    static QString getSerieName(const std::vector<std::string> & serieFileNames);

    /**
     * @brief Retrieve the serie description.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the serie description.
     */
    static QString getSerieDescription(const std::vector<std::string> & serieFileNames);

    /**
     * @brief Retrieve the study name.
     * @param serieFileNames The filenames belonging to a single serie.
     * @return the study name.
     */
    static QString getStudyName(const std::vector<std::string> & serieFileNames);



};

#endif // DICOMPARSER_H

