/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include <Application.h>
#include <MainWindow.h>
#include <InteractiveViewer.h>
#include <Core.h>
#include <ExtensionManager.h>
#include <Action.h>


using namespace camitk;

// CLI stuff
#include "CommandLineOptions.hxx"

// Description:
//This is a test application, it is not intended to be used as a CamiTK application.
//-a <file> action to load (.dll file)
//-i <file> is the file to open - will not be modified

// usage for this application
void usage(char *appName) {
    std::cerr << appName << std::endl;
    std::cerr << std::endl;
    std::cerr << "Usage: " << appName << " [options]" << std::endl;
    std::cerr << "Build using " << Core::version << std::endl;
    std::cerr << std::endl;
    std::cerr << "Options:" << endl;
    options::print_usage(std::cerr);
}

int main(int argc, char *argv[]) {
    try {

        int end; // End of options.
        options o(argc, argv, end);

        // if specific help or no options provided
        if (   o.help()
                || (o.input() == "" && o.action() == "")
           ) {
            std::cout << "Help or no arguments: " << std::endl;
            usage(argv[0]);
            return EXIT_SUCCESS;
        } else if( o.input() == "" || o.action() == "" ) {
            std::cout << "Bad arguments: " << std::endl;
            usage(argv[0]);
            return EXIT_FAILURE;
        } else {
            std::cout << "Arguments OK: " << std::endl;
            bool extOpen = false;
            // init the camitk application contextEXIT_SUCCESS
            Application a( "testactions", argc, argv, false, false );// No autoload + registerFileExtension false to avoid the execution of this testapp hampered by PopUp message

            // avoid all redirection to console
            MainWindow *defaultMainWindow = dynamic_cast<MainWindow*>(a.getMainWindow());
            defaultMainWindow->redirectToConsole(false);

            // Remember, autoload search for extension dynamic libraries in the lib/actions and lib/components subdirectories of
            // the following trees:
            // * current working directory (build)
            // * local installation
            // * global installation
            // If it finds duplicates, it keeps the first found.
            ExtensionManager::autoload( ExtensionManager::COMPONENT );

            std::cout << "Open data file " << o.input() << std::endl;
            Component* comp = Application::open( o.input().data() );

            if( comp ) {
                std::cout << "Component loaded: " << o.input() << std::endl;
                Action::ApplyStatus status;
                comp->setSelected( true, true );
                // Find the actions directory, according to this application's location
                // If the application directory is in the build/bin directory, then the action directory will point on the build/lib/camitk-version/actions directory
                // Note : do not use Core::getActionDirectories as it uses Core::getCurrentWorkingDir which is clearly less accurate
                QString actionsDirectory = Application::instance()->applicationDirPath() + "/../lib/" + Core::shortVersion + "/actions/";

                //Load the action extension defined in the entry parameters
                if (!(actionsDirectory.isEmpty())
                        && (ExtensionManager::loadExtension(ExtensionManager::ACTION, actionsDirectory + o.action().data()))) {
                    std::cout << "Loaded action extension "<< o.action().data() << std::endl;
                    // ActionExtension* actionExtension = ExtensionManager::getActionExtensions().value(o.action().data());
                    ActionList actl =  Application::getActions( comp );
                    if( !( actl.isEmpty() )) {
                        std::cout << "Action extension contains " << actl.size() << "actions." << std::endl;

                        foreach( Action *action , actl ) {
                            std::cout << "Test action " << action->getName().toStdString() << std::endl;
                            action->setInputComponent( comp );
                            std::cout << action->getName().toStdString() << "Call apply() ...";
                            status = action->apply();
                            std::cout << "ok." << std::endl;

                            if( status != Action::SUCCESS ) {
                                std::cout << "Apply action " <<  action->getName().toStdString() << " on file " << comp->getFileName().toStdString() << " FAILED" << std::endl;
                                return EXIT_FAILURE;
                            } else {
                                std::cout << "Apply action " <<  action->getName().toStdString() << " on file " << comp->getFileName().toStdString() << " SUCCESS" << std::endl;
                                return EXIT_SUCCESS;
                            }
                        }
                    } else {
                        //you cannot apply this action extension on this comp
                        std::cout << "this action EXTENSION is not compatible with this component type - " << o.action() << std::endl;
                        return EXIT_SUCCESS;
                    }
                } else {
                    std::cout << "Loading action extension :" <<  o.action() << " FAILED" << std::endl;
                    return EXIT_FAILURE;
                }
            } else {
                std::cout << "Loading data file :" <<  o.input() << " FAILED" << std::endl;
                return EXIT_FAILURE;
            }
        }
    } catch (camitk::AbortException& e) {
        std::cout << argv[0] << " aborted..." << std::endl << "camitk AbortException:" << std::endl << e.what() << std::endl;
        return EXIT_FAILURE;
    } catch (std::exception& e) {
        std::cout << argv[0] << " aborted..." << std::endl << "std AbortException:" << std::endl << e.what() << std::endl;
        return EXIT_FAILURE;
    } catch (...) {
        std::cout << argv[0] << " aborted..." << std::endl << "Unknown Exception" << std::endl;
        return EXIT_FAILURE;
    }
}
