#if !defined(NDEBUG)

#include <qglobal.h>

#if defined(Q_OS_WIN32) || defined(_OS_WIN32_)
#include <Windows.h>
#endif
#if defined(Q_OS_UNIX) || defined(_OS_UNIX_)
#include <stdio.h>
#endif

#include <stdlib.h>

#define NUMTRACEDLINES 10000
#define POPBLOCKSIZE	1000

#include <qaccel.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qdir.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qsplitter.h>
#include <qtextview.h>
#include <qvaluelist.h>

#include "TraceManager.h"

static CTraceManager *s_pTraceManager = 0;

class CTraceListViewItem: public QListViewItem
{

public:
   CTraceListViewItem(QListView *parent, const CTraceModule &module)
      : QListViewItem(parent)
   {
      setText(0, module.GetName());
      setText(1, QString::number(module.GetLevel()));
   };
};

/***************************************************************************/

CTraceManager::CTraceManager() : m_FullText("")
{
   m_pWindow = 0;
   m_Indent = 0;
   m_pSelectedModule = 0;
   m_Visible = false;
   m_Mute = false;
   m_Output = true;
   m_CatchQDebug = true;
   m_CatchQWarning = true;
   m_AppendToFile = false;
   m_ResetText = false;
}

CTraceManager::~CTraceManager()
{
   qInstallMsgHandler(0); // We no longer want qDebug and qWarning
}

// private

void CTraceManager::OpenLog()
{
}

void CTraceManager::CloseLog()
{

}

void CTraceManager::MessageHandler(QtMsgType type, const char *msg)
{
   if (s_pTraceManager == 0)
     return;

   switch(type)
   {
     case QtDebugMsg:
       if (s_pTraceManager->m_CatchQDebug)
         s_pTraceManager->Log(QString("D: ") + msg);
       break;
     case QtWarningMsg:
       if (s_pTraceManager->m_CatchQWarning)
         s_pTraceManager->Log(QString("W: ") + msg);
       break;
     case QtFatalMsg:
       s_pTraceManager->Log(QString("F: ") + msg);
       QMessageBox::critical(0, QString("Fatal Error"), QString(msg), QMessageBox::Ok, QMessageBox::NoButton);
       abort();
       break;
     default: // should not happen.
       s_pTraceManager->Log(QString("?: ") + msg);
       break;
   }
}


// protected

void CTraceManager::IncIndent()
{
   m_Indent++;
   m_IndentString.fill(' ', m_Indent * 2);
}

void CTraceManager::DecIndent()
{
   if (m_Indent > 0)
   {
     m_Indent--;
     m_IndentString.fill(' ', m_Indent * 2);
   }
}


// protected slots

void CTraceManager::ClickedListItem(QListViewItem *item)
{
   if (item == 0)
     m_pSelectedModule = 0;
   else
     m_pSelectedModule = m_Modules[item->text(0)];
}

void CTraceManager::IncLogging()
{
   if (m_pSelectedModule == 0)
     return;
   m_pSelectedModule->IncLevel();
   m_pWindow->m_ModuleList->currentItem()->setText(1, QString::number(m_pSelectedModule->GetLevel()));
}

void CTraceManager::DecLogging()
{
   if (m_pSelectedModule == 0)
     return;
   m_pSelectedModule->DecLevel();
   m_pWindow->m_ModuleList->currentItem()->setText(1, QString::number(m_pSelectedModule->GetLevel()));
}


void CTraceManager::ToggledCatchQDebug(bool on)
{
   m_CatchQDebug = on;
}

void CTraceManager::ToggledCatchQWarning(bool on)
{
   m_CatchQWarning = on;
}

void CTraceManager::ToggledOutput(bool on)
{
   m_Output = on;
}

void CTraceManager::ToggledAppendToFile(bool on)
{
   m_AppendToFile = on;
   if (on && !m_LogFileName.isEmpty())
     OpenLog();
}

void CTraceManager::SetLogFileName()
{
   if (!m_LogFileName.isEmpty())
     CloseLog();
   m_LogFileName = m_pWindow->m_LogFileName->text();
   if (m_AppendToFile)
     OpenLog();
}



// public

CTraceManager *CTraceManager::Instance()
{
   if (s_pTraceManager == 0)
     s_pTraceManager = new CTraceManager();
   if (s_pTraceManager != 0)
     qInstallMsgHandler(CTraceManager::MessageHandler);
   return s_pTraceManager;
}

CTraceModule *CTraceManager::RegisterModule(const QString &module_name, int initial_value)
{
   CTraceModule *mod;

   mod = m_Modules.find(module_name);
   if (mod == 0) {
     mod = new CTraceModule(this, module_name, initial_value);
     CHECK_PTR(mod);
     m_Modules.insert(module_name, mod);

#if 0
     // Search <modules> config node for a matching entry; read level and set
     // in module trace object
     QDomNode search = m_ModulesNode.firstChild();
     QDomElement el;
     while (!search.isNull())
     {
        if (search.isElement())
        {
          el = search.toElement();
          if (el.hasAttribute("name") &&
              el.attribute("name") == module_name)
          {
            int StoredLevel;
            bool Okay;

            StoredLevel = el.text().toInt(&Okay);
            if (Okay)
            {
              mod->SetLevel(StoredLevel);
              break;
            }
          }
        }

        search = search.nextSibling();
     }
#endif
  }
  return mod;
}

void CTraceManager::Log(const QString &log_string)
{
   static int numline = 0;
   QString ns;

   if (m_Mute)
     return;

   ns = m_IndentString + log_string.latin1() + "\n";	// don't remove the latin1(). If the program traces stuff that resembles rich text
   							// you will have Aborts using m_FullText.join() or segmentation faults using iterators over m_FullText
   m_FullText += ns;
   numline++;
   if (numline > NUMTRACEDLINES) {
     for (int i=0; i < POPBLOCKSIZE; i++) {
        m_FullText.remove(m_FullText.first());
	numline--;
     }
     m_ResetText = true;
   }

   if (m_pWindow != 0 && m_Visible) {
     if (m_ResetText) {
       m_pWindow->m_LogEdit->setText(m_FullText.join(""));
       m_ResetText = false;
     }
     else {
       m_pWindow->m_LogEdit->append(ns);
     }
//     m_pWindow->m_LogEdit->scrollToBottom();	// this statement makes the trace window very slow
   }
   if (m_Output) {
#if defined(Q_OS_WIN32) || defined(_OS_WIN32_)
     // use windows debug function
     OutputDebugString((const char *)ns);
#endif
#if defined(Q_OS_UNIX) || defined(_OS_UNIX_)
     // Good ol' stderr on Unix
     fprintf(stderr, "%s", (const char *)ns);
#endif
   }
}


/**
  \brief Create subtree in XML format with current configuration
  \param node [out] The XML subnode

  This will return the current configuration by replacing or adding
  values in subnode <tracer> in \b node. In other words, pass the
  parent of the <tracer> node, not the node itself (this saves you the
  hassle of checking and creating the node itself).
*/
void CTraceManager::GetConfiguration(QDomNode &node)
{
   QDictIterator<CTraceModule> dit(m_Modules);
   CTraceModule *mmod;
   QDomElement el;
   QDomNode root;

   // Create new, clean node.
   root = node.namedItem("tracer");
   if (!root.isNull())
     node.removeChild(root);
   root = node.ownerDocument().createElement("tracer");
   node.appendChild(root);

   // Walk all modules, add trace level as text value
   QDomElement modules = node.ownerDocument().createElement("modules");
   for (; (mmod = dit.current()); ++dit) {
      if (mmod == 0)
        continue;
      QDomElement module = node.ownerDocument().createElement("module");

      module.setAttribute("name", mmod->GetName());
      module.appendChild(node.ownerDocument().createTextNode(QString::number(mmod->GetLevel())));
      modules.appendChild(module);
   }
   root.appendChild(modules);

   // Add some global values as nodes
   el = node.ownerDocument().createElement("qdebug");
   el.appendChild(node.ownerDocument().createTextNode(m_CatchQDebug ? "true" : "false"));
   root.appendChild(el);

   el = node.ownerDocument().createElement("qwarning");
   el.appendChild(node.ownerDocument().createTextNode(m_CatchQWarning ? "true" : "false"));
   root.appendChild(el);

   el = node.ownerDocument().createElement("output");
   el.appendChild(node.ownerDocument().createTextNode(m_Output ? "true" : "false"));
   root.appendChild(el);

   el = node.ownerDocument().createElement("logfilename");
   el.appendChild(node.ownerDocument().createTextNode(m_LogFileName));
   root.appendChild(el);

   el = node.ownerDocument().createElement("appendtofile");
   el.appendChild(node.ownerDocument().createTextNode(m_AppendToFile ? "true" : "false"));
   root.appendChild(el);
}


/**
  \brief Restore configuration from XML node
  \param node [in] XML node that should contain our configuration node
  \sa GetConfiguration

  This function will restore the configuration of the tracemanager by
  reading the information from the <tracer> XML subnode. So, you pass the
  parent of this node, not the actual <tracer> node.

*/
void CTraceManager::SetConfiguration(const QDomNode &node)
{
   QDomNode root, v;
   QDomElement el;
   QString s;

   // Sanity check
   if (node.isNull())
     return;
   root = node.namedItem("tracer");
   if (root.isNull())
     return;

   // Walk the elements in the <modules> section, find existing modules
   // and set level
   v = root.namedItem("modules").firstChild();
   while (!v.isNull())
   {
      if (v.isElement() && v.nodeName() == "module")
      {
        el = v.toElement();
        if (el.hasAttribute("name"))
        {
          CTraceModule *pTraceModule;
          int StoredLevel;
          bool NumOkay;

          // Get name and stored value
          s = el.attribute("name");
          StoredLevel = el.text().toInt(&NumOkay);
          // see if we have a tracer module by that name
          pTraceModule = m_Modules.find(s);
          if (pTraceModule != 0)
          {
            if (NumOkay)
            {
              pTraceModule->SetLevel(StoredLevel);
            }
          }
          else
          {
            // Nope. Create module; this in case a module does not get initialized
            // during this run of the program
            if (!NumOkay)
            {
              StoredLevel = 1; // default
            }
            pTraceModule  = new CTraceModule(this, s, StoredLevel);
            CHECK_PTR(pTraceModule);
            m_Modules.insert(s, pTraceModule);
          }
        }
      }

      v = v.nextSibling();
   }

   // Finally, fetch some global settings
   el = root.namedItem("qdebug").toElement();
   if (!el.isNull())
   {
     s = el.text();
     ToggledCatchQDebug(s == "true");
   }

   el = root.namedItem("qwarning").toElement();
   if (!el.isNull())
   {
     s = el.text();
     ToggledCatchQWarning(s == "true");
   }

   el = root.namedItem("output").toElement();
   if (!el.isNull())
   {
     s = el.text();
     ToggledOutput(s == "true");
   }

   el = root.namedItem("logfilename").toElement();
   if (!el.isNull())
   {
     m_LogFileName = el.text();
   }

   el = root.namedItem("appendtofile").toElement();
   if (!el.isNull())
   {
     s = el.text();
     ToggledAppendToFile(s == "true");
   }
}



// public slots

void CTraceManager::Show()
{
   /* Creation of the window is delayed until here because it is possible
      this class is created before the main window loop is setup, when there
      isn't a graphical context yet, causing all kind of problems.
    */
   if (m_pWindow == 0) {
     QDictIterator<CTraceModule> it(m_Modules);
     QValueList<int> Stretch;
     QFont FixedFont;
     QAccel *ac;

     m_pWindow = new CTraceWindowDlg(0, "trace window", FALSE, Qt::WType_TopLevel | Qt::WStyle_Dialog);
     Stretch.append(100);
     Stretch.append(100);
//     m_pWindow->m_Splitter->setSizes(Stretch);
     // Create list of modules
     for (; it.current(); ++it)
        new CTraceListViewItem(m_pWindow->m_ModuleList, *it.current());

     FixedFont.setFamily("Courier");
     FixedFont.setFixedPitch(TRUE);

     //m_pWindow->m_LogEdit->setWordWrap(QTextEdit::NoWrap);
     //m_pWindow->m_LogEdit->setCurrentFont(FixedFont);
     m_pWindow->m_LogEdit->setTextFormat(QTextView::PlainText);

     // Pre-set first, then do connect()
     m_pWindow->m_CatchQDebugBox->setChecked(m_CatchQDebug);
     m_pWindow->m_CatchQWarningBox->setChecked(m_CatchQWarning);
     m_pWindow->m_OutputBox->setChecked(m_Output);
     m_pWindow->m_AppendToFileBox->setChecked(m_AppendToFile);
     m_pWindow->m_LogFileName->setText(m_LogFileName);

     connect(m_pWindow->m_CatchQDebugBox, SIGNAL(toggled(bool)), this, SLOT(ToggledCatchQDebug(bool)));
     connect(m_pWindow->m_CatchQWarningBox, SIGNAL(toggled(bool)), this, SLOT(ToggledCatchQWarning(bool)));

     connect(m_pWindow->m_ModuleList, SIGNAL(clicked(QListViewItem *)), this, SLOT(ClickedListItem(QListViewItem *)));

     connect(m_pWindow->m_OutputBox, SIGNAL(toggled(bool)), this, SLOT(ToggledOutput(bool)));
     connect(m_pWindow->m_AppendToFileBox, SIGNAL(toggled(bool)), this, SLOT(ToggledAppendToFile(bool)));
     connect(m_pWindow->m_FileNameSetButton, SIGNAL(clicked()), this, SLOT(SetLogFileName()));

     connect(m_pWindow->m_ClearButton, SIGNAL(clicked()),    this, SLOT(ClearText()));
     connect(m_pWindow->m_MarkButton, SIGNAL(clicked()),     this, SLOT(Mark()));
     connect(m_pWindow->m_MuteButton, SIGNAL(toggled(bool)), this, SLOT(SetMute(bool)));

     connect(m_pWindow->m_CloseButton, SIGNAL(clicked()), this, SLOT(Hide()));

     // Create +/- accelerators
     ac = new QAccel(m_pWindow, "trace level accelerators");
     ac->connectItem(ac->insertItem(Key_Plus),  this, SLOT(IncLogging()));
     ac->connectItem(ac->insertItem(Key_Minus), this, SLOT(DecLogging()));
   }
//   m_pWindow->m_LogEdit->scrollToBottom();
   m_pWindow->show();
   m_ResetText=true;
   m_Visible = true;
}

void CTraceManager::Hide()
{
   if (m_pWindow)
     m_pWindow->hide();
   m_Visible = false;
}

void CTraceManager::ClearText()
{
   m_FullText.clear();
   if (m_Visible)
     m_pWindow->m_LogEdit->setText("");
}

void CTraceManager::Mark()
{
   Log("\n-------------- MARK ---------------\n");
}

void CTraceManager::SetMute(bool mute)
{
   m_Mute = mute;
}

void CTraceManager::UseOutput(bool on)
{
   m_Output = on;
}


#endif
