/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#include "interface.h"
#include "support.h"

#include "shared.h"
#include "memory.h"
#include "custom_find.h"
#include "gui_custom_view.h"
#include "mp3info.h"
#include "lib_mpgheader.h"
#include "lib_id3v1.h"
#include "lib_id3v2.3.h"


/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// TRUE if directory, FALSE if not.
static int
is_dir(gchar *filepathname)
{
	struct stat pstat;
	stat(filepathname, &pstat);
	return(S_ISDIR(pstat.st_mode));
}

// Return the file size of the given file *fully named*
static gint
get_file_size (char *filepathname, char *origin)
{
	struct stat pstat;
	gchar buf[5000];
	if ((lstat (filepathname, &pstat)) < 0)
	{
		sprintf (buf, "%s/%s", origin, filepathname);
		if ((lstat (buf, &pstat)) < 0)
			return (gint) 0;
	}
	return (gint)pstat.st_size;
}

// gets a list of all files matching the wildcard-pattern "must_match" in "dir"
// either recursive (TRUE) or - not!
static GList *
get_matching_filenames(gchar *dir, gchar *must_match, gboolean recursive)
{
	extern GtkWidget *wait;
	GList *foundfiles = NULL;
	Mp3Info *mp3info = NULL;
	
	struct dirent *pdirent = NULL;
	DIR *pdir = NULL;
	gchar current[2048];
	gchar subdir[2048];
	gchar waittext[2048];

// If the widget is not visible, then the user has closed it. break.
	if(!GTK_WIDGET_VISIBLE(wait))
		return NULL;
	
// Open directory
	if( (pdir = opendir(dir)) == NULL ) 
	{
		printf("200 - Error opening requested path ('%s')!\n", dir);
		closedir(pdir);
		return NULL;
	}
	
	snprintf(waittext, 2047, "Reading %s", dir);
	gtk_label_set_text(GTK_LABEL(lookup_widget(wait, "label_wait_text")), waittext);
	while(g_main_iteration(FALSE));
	
// get entry from directory
	while( (pdirent = readdir(pdir)) != NULL )
	{
// If the widget is not visible, then the user has closed it. break.
		if(!GTK_WIDGET_VISIBLE(wait))
		{
			closedir(pdir);
			return NULL;
		}
		
		sprintf (current, "%s%s", dir, pdirent->d_name);
		if ( !is_dir(current) )
		{
			if( strrchr(current, '.')
				&& ((strncmp(strrchr(current, '.'), ".mp3", 4)==0)
					|| (strncmp(strrchr(current, '.'), ".flac", 5)==0)
#ifdef HAVE_OGG_H
					|| (strncmp(strrchr(current, '.'), ".ogg", 4)==0)
#endif
					)
				&& (access(current, R_OK)==0)
				&& strwccmp(must_match, pdirent->d_name, TRUE) )
			{
				mp3info = malloc(sizeof(Mp3Info));
				memset(mp3info, 0, sizeof(Mp3Info));
				
				mp3file_get_info_1(dir, pdirent->d_name, mp3info);
				strncpy(mp3info->directory, dir, 2047); 
				strncpy(mp3info->filename, pdirent->d_name, 1023);

				foundfiles = g_list_append(foundfiles, mp3info);
			}
		}
		else if( strcmp(strrchr(current, '/') + 1, ".")
			&& strcmp(strrchr(current, '/') + 1, "..") )
		{
			sprintf(subdir, "%s/", current);
			
			if( recursive==TRUE
				&& access(subdir, R_OK)==0)
				foundfiles = g_list_concat(foundfiles, g_list_first(get_matching_filenames(subdir, must_match, TRUE)));
		}
	}
	closedir(pdir);
	
	return foundfiles;
}

// true if file has a v1 tag
static gboolean
has_v1tag(gchar *filename)
{
	id3Tag *tag = calloc(1,sizeof(id3Tag));
	if(get_id3v1_tag(tag, filename)==0)
	{
		free(tag);
		return TRUE;
	}
	free(tag);
	
	return FALSE;
}

// true if the file has a v2 tag
static gboolean
has_v2tag(gchar *filename)
{
	id3Tag *tag = calloc(1,sizeof(id3Tag));
	if(get_id3v2_tag(tag, filename)==0)
	{
		free(tag);
		return TRUE;
	}
	free(tag);
	
	return FALSE;
}

// true if the v1 tag is empty
static gboolean
has_empty_v1tag(gchar *filename)
{
	id3Tag *tag = calloc(1,sizeof(id3Tag));
	
	memset(tag->artist, '\0', 1024);
	memset(tag->title, '\0', 1024);
	memset(tag->album, '\0', 1024);
	memset(tag->year, '\0', 1024);
	memset(tag->comment, '\0', 1024);
	
	if( (get_id3v1_tag(tag, filename)==0)
		&& (*tag->artist!='\0'
		|| *tag->title!='\0'
		|| *tag->album!='\0'
		|| *tag->year!='\0'
		|| *tag->comment!='\0') )
	{
		free(tag);
		return TRUE;
	}
	free(tag);
	
	return FALSE;
}

// true if the v2 tag is empty
static gboolean
has_empty_v2tag(gchar *filename)
{
	id3Tag *tag = calloc(1,sizeof(id3Tag));
	
	memset(tag->artist, '\0', 1024);
	memset(tag->title, '\0', 1024);
	memset(tag->album, '\0', 1024);
	memset(tag->year, '\0', 1024);
	memset(tag->comment, '\0', 1024);
	
	if( (get_id3v2_tag(tag, filename)==0)
		&& (*tag->artist!='\0'
		|| *tag->title!='\0'
		|| *tag->album!='\0'
		|| *tag->year!='\0'
		|| *tag->comment!='\0') )
	{
		free(tag);
		return TRUE;
	}
	free(tag);
	
	return FALSE;
}




/*
 * Similarity check.
 * (c) 2002 by Samuel Abels.
 *
 * Determines if string1 is similar to string2
 * string1, string2 are the strings to be compared
 * tolerance is the tolerance (1 to 100)
 * returns: TRUE if similar, FALSE if not.
 *
static gboolean
is_similar(gchar *string1, gchar *string2, gint tolerance)
{
	gchar *pstring1 = string1;
	gchar *pstring2 = string2;
	gint matches = 0;
	gchar current[2];
	
// the smaller string has to come first
	if(strlen(string1) > strlen(string2))
	{
		pstring1=string1;
		string1=string2;
		string2=pstring1;
	}

// length too different? (the "+2" is to make this less sensible, try to experiment with)
	if( (strlen(string2) - strlen(string1))
		> ((gfloat)strlen(string1)/100*tolerance)+2 )
		return FALSE;

// now to the "real" similarity check.
// i will get the first two chars of string 1, and look if this combination is found anywhere in string2, too.
// if yes -> increase matches by one.
// Then go get two chars from string 1 again, beginning with the second character of string 1, seek this in string2, and so on...
// When were done, matches should be max. len of string1.
	pstring1 = string1;
	while(*pstring1!='\0')
	{
		current[0]=*pstring1;
		current[1]=*(pstring1+1);
		
		pstring2 = string2;
		while(*pstring2!='\0')
		{
			if( (current[0]==*pstring2)
				&& (current[1]==*(pstring2+1)) )
			{
				++matches;
				break;
			}
			++pstring2;
		}
		++pstring1;
	}

// if: matches > the size of (string1 - some tolerance) return TRUE
	if( matches >= (gfloat)strlen(string1) - (gfloat)strlen(string1)/100*tolerance )
		return TRUE;
	
	return FALSE;
}
*/
/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/






void
custom_find_find(void)
{
	extern GtkWidget *custom_find;
	extern GtkWidget *custom_view;
	extern GtkWidget *wait;
	GList *foundfiles = NULL;
	GList *item = NULL;
	gchar *wildcardstring = NULL;
	gchar *tempstring = NULL;
	gchar fullfilename[2048];
	Mp3Info *mp3info = NULL;
	gboolean removefile = FALSE;
	GtkProgressBar *progressbar =NULL;
	gint numrows = 0;
	gint currentrow = 0;
	gchar directory[2048];

// Init progressbar/show progresswindow.
	progressbar = GTK_PROGRESS_BAR(lookup_widget(wait, "progressbar_wait"));
	gtk_progress_bar_update(progressbar, 0);
	gtk_widget_show(wait);
	while(g_main_iteration(FALSE));

// Get the strings from the GUI
	wildcardstring = g_strconcat( gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_must_match"))),
		".mp3", '\0' );
	strncpy(directory, gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_directory"))), 2047);
	if( directory[strlen(directory)-1] != '/' )
		directory[strlen(directory)] = '/';
	
// get list of the filenames matching all criterias
	foundfiles = get_matching_filenames(
		directory,
		wildcardstring,
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "check_custom_find_filename_recursive"))) );
	free(wildcardstring);
	
// the same for .flac
	wildcardstring = g_strconcat( gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_must_match"))),
		".flac", '\0' );
	strncpy(directory, gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_directory"))), 2047);
	if( directory[strlen(directory)-1] != '/' )
		directory[strlen(directory)] = '/';
	
// get list of the filenames matching all criterias
	foundfiles = g_list_concat( foundfiles,
		get_matching_filenames(	directory,	wildcardstring,	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "check_custom_find_filename_recursive"))) ) );
	free(wildcardstring);
  
#ifdef HAVE_OGG_H
// the same for .ogg
	wildcardstring = g_strconcat( gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_must_match"))),
		".ogg", '\0' );
	strncpy(directory, gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_directory"))), 2047);
	if( directory[strlen(directory)-1] != '/' )
		directory[strlen(directory)] = '/';
	
// get list of the filenames matching all criterias
	foundfiles = g_list_concat( foundfiles,
		get_matching_filenames(	directory,	wildcardstring,	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "check_custom_find_filename_recursive"))) ) );
	free(wildcardstring);
#endif

	if(!foundfiles)
	{
		gtk_widget_hide(wait);
		return;
	}

// If the widget is not visible, then the user has closed it. break.
	if(!GTK_WIDGET_VISIBLE(wait))
	{
		mem_free(foundfiles);
		return;
	}
	
// Count rows
	item = g_list_first(foundfiles);
	while((item = item->next)) ++numrows;

	gtk_label_set_text(GTK_LABEL(lookup_widget(wait, "label_wait_text")), "Applying filters...");

	item = g_list_first(foundfiles);
	while(item)
	{
		mp3info = (Mp3Info *)item->data;
		item = item->next;
		removefile = FALSE;
		
		gtk_progress_bar_update(progressbar, (gfloat)100/numrows*(currentrow++)*0.01);
		while(g_main_iteration(FALSE));
// If the widget is not visible, then the user has closed it. break.
		if(!GTK_WIDGET_VISIBLE(wait))
		{
			mem_free(foundfiles);
			return;
		}
		
// Filter: NOT MATCH Filename
		wildcardstring = g_strconcat( gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_must_not_match"))),
			".*", '\0' );
		if( strcmp(wildcardstring, ".*")!=0
			&& strwccmp(wildcardstring, mp3info->filename, TRUE) )
			removefile = TRUE;
		free(wildcardstring);

// Filter: NOT MATCH Directory
		tempstring = gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_filename_directory_not")));
		if( *tempstring!='\0'
			&& strncmp(tempstring, mp3info->directory, strlen(tempstring)) == 0 )
			removefile = TRUE;

		snprintf(fullfilename, 2047, "%s%s", mp3info->directory, mp3info->filename);
// Filter: V1 tag MUST be present
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "radio_custom_find_id3v1_present")))
			&& !has_v1tag(fullfilename))
			removefile = TRUE;
		
// Filter: V1 tag MUST NOT be present
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "radio_custom_find_id3v1_notag")))
			&& has_v1tag(fullfilename))
			removefile = TRUE;
		
// Filter: V1 tag MUST NOT be empty
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "check_custom_find_id3v1_notempty")))
			&& !has_empty_v1tag(fullfilename))
			removefile = TRUE;
		
// Filter: V2 tag MUST be present
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "radio_custom_find_id3v2_present")))
			&& !has_v2tag(fullfilename))
			removefile = TRUE;
		
// Filter: V2 tag MUST NOT be present
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "radio_custom_find_id3v2_notag")))
			&& has_v2tag(fullfilename))
			removefile = TRUE;
		
// Filter: V2 tag MUST NOT be empty
		if(gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON(lookup_widget(custom_find, "check_custom_find_id3v2_notempty")))
			&& !has_empty_v2tag(fullfilename))
			removefile = TRUE;

// Filter: Bitrate
		tempstring = gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_bitrate_mode")));
		if( strcmp(tempstring, "greater then")==0
			&& mp3info->bitrate <= atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_bitrate_bitrate")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "equal to")==0
			&& mp3info->bitrate != atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_bitrate_bitrate")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "less then")==0
			&& mp3info->bitrate >= atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_bitrate_bitrate")))) )
					removefile = TRUE;
		
// Filter: Samplerate
		tempstring = gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_samplerate_mode")));
		if( strcmp(tempstring, "greater then")==0
			&& mp3info->frequency <= atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_samplerate_samplerate")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "equal to")==0
			&& mp3info->frequency != atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_samplerate_samplerate")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "less then")==0
			&& mp3info->frequency >= atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_mpeg_samplerate_samplerate")))) )
					removefile = TRUE;
		
// Filter: Filesize
		tempstring = gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_other_filesize_mode")));
		if( strcmp(tempstring, "greater then")==0
			&& get_file_size(mp3info->filename, mp3info->directory) <= 1024*atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_other_filesize_kb")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "equal to")==0
			&& get_file_size(mp3info->filename, mp3info->directory) != 1024*atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_other_filesize_kb")))) )
					removefile = TRUE;
		if( strcmp(tempstring, "less then")==0
			&& get_file_size(mp3info->filename, mp3info->directory) >= 1024*atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(custom_find, "entry_custom_find_other_filesize_kb")))) )
					removefile = TRUE;
		
// If one of the filters matched, remove the file from the list.
		if(removefile)
			foundfiles = mem_free_1(foundfiles, (gpointer)mp3info);
	}

// show the custom_view widget and insert found data.
	custom_view = create_custom_view();
	gtk_widget_hide(wait);
	gtk_widget_show(custom_view);
	while(g_main_iteration(FALSE));
	clist_custom_view_add_files(foundfiles);
}
