/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */  

#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "lib_freedb.h"
#include "plugs.h"

//#define _DEBUG_
//#define _DEBUG_EXCESSIVE_

/***************************************************************************************
 * STATICS
 ***************************************************************************************/
/* receive one line from a host */
static int get_line (int socket, char * line, int maxlen)
{
  int i = 0;
  int errno = 0;
  
  memset (line, 0, maxlen);
  
  /* read characters into the buffer, until the first occurence of a newline */
  errno = read (socket, line, 1);
  while ( *(line + i) != '\n' ) {
    ++i;
    if ( i >= maxlen )
      return errno;
    
    if ( errno <= 0 )
      return errno;
    
    errno = read (socket, line + i, 1);
  }
  
  /* Remove the newline chars. */
  while ( *(line + i) == '\r'
    || *(line + i) == '\n' )
    *(line + i--) = '\0';
  
  return errno;
}


/* Change all spaces in the searchstring to "+" (URLsafe) */
static void
urlencode (char *pstring)
{
  while ( strchr (pstring, ' ') != NULL )
    *strchr (pstring, ' ') = '+';
}


/*
 * Create a request string out of a cddb id that can be sent to a freedb host.
 */
static void
freedb_create_id_request (char *request, char *hostname, unsigned int host_port, char *cgi_path, char *progname, char *version, CDInfo *cd, int maxlen)
{
  char *myusername = getenv("USER");
  char myhostname[2048] = "";
  
  /* Get my own hostname */
  gethostname (myhostname, sizeof(myhostname) - 1);
  
  snprintf (
    request,
    maxlen,
    "GET http://%s%s?cmd=cddb+read+%s+%08lx&hello=%s@%s+%s+%s+%s&proto=1 HTTP/1.1\r\n"
    "Connection: close\r\n"
    "Host: %s:%i\r\n"
    "\r\n",
    hostname,
    cgi_path,
    cd->genre,
    cd->id,
    myusername,
    myhostname,
    myhostname,
    progname,
    version,
    hostname,
    host_port);
}


/*
 * Create a request string that can be sent to a freedb host to request a list of albums.
 */
static void
freedb_create_searchstring_request (char *request, char *searchstring, int maxlen)
{
  char *hostname = NULL;
  int host_port = 80;
  
  hostname = strdup ("www.freedb.org");
  
  /* Prepare request */
  snprintf(
    request,
    maxlen,
    "GET http://%s/freedb_search.php?words=%s&allfields=YES&fields=artist&fields=title&allcats=YES&grouping=none HTTP/1.1\r\n"
    "Connection: close\r\n"
    "Host: %s:%i\r\n"
    "\r\n",
    hostname,
    searchstring,
    hostname,
    host_port);
}


/*
 * Grab album informations from a freedb string into a new cd struct and append it to a cd list.
 */
static int
freedb_add_album_to_cdlist(char *line, GList **cdlist)
{
  CDInfo *cd = NULL;
  char genre[512];
  char id[20];
  char album[1024];
  
  /* The cd genre should begin after char no. 65. */
  /* It ends at the first "&" char.               */
  line += 65;
  strncpy (genre, line, sizeof (genre) - 1);
  if ( strchr (genre, '&') )
    *strchr (genre, '&') = '\0';
  
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_add_album_to_cdlist: genre: %s\n", genre);
#endif
  
  /* The freedb id begins after the first occurence of '='. */
  if ( !(line = strchr (line, '=')) )
    return -1;
  /* The id ends with the '">". */
  if ( !strstr (line, "\">") )
    return -2;
  
  strncpy (id, line + 1, sizeof (id) - 1);
  
  /* remove the closing quote */
  if ( !strchr (id, '\"') )
    return -3;
  *strchr (id, '\"') = '\0';
  
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_add_album_to_cdlist: id: %s\n", id);
#endif
  
  /* The title begins after the first occurence of the ">" char. */
  if ( !(line = strchr(line, '>')) )
    return -4;
  strncpy (album, line + 1, 1023);
  
  /* The title should end at the first occurence of "</a><br><". */
  /* So we strip all unneeded chars off the string end here.     */
  if ( strstr (album, "</a><br><") != NULL )
    *strstr (album, "</a><br><") = '\0';
  else
    return -5;
  
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_add_album_to_cdlist: album: %s\n", album);
#endif
  
  /* Init a new CD struct */
  cd = malloc (sizeof (CDInfo));
  cd->track[0] = NULL;
  
  /* Copy the data into the struct. */
  cd->id = strtoul (id, NULL, 16);
  memset (cd->year, 0, 20);
  strncpy (cd->genre, genre, sizeof (cd->genre));
  strncpy (cd->cdtitle, album, sizeof (cd->cdtitle));
  
  *cdlist = g_list_append (*cdlist, cd);
  
  return 0;
}


/*
 * Grab a track number from a freedb string into an already existing track struct.
 */
static int
freedb_line_to_track(char *line, CDTrack *track)
{
  /* The line has the format: "TTITLEnn=<title>"                           *
  * Thus, we save anything between "TTITLE" and the "=" as track number.  */
  strncpy (track->number, line + 6, sizeof(track->number));
  
  /* The track must now contain a "=" character */
  if ( strchr (track->number, '=') == NULL )
    return -1;
  
  /* Strip all characters beginning with the "=" char. */
  *strchr (track->number, '=') = '\0';
  
  /* Make the number twodigit */
  if ( atoi(track->number) + 1 > 9 )
    snprintf (track->number, sizeof(track->number), "%i", atoi(track->number) + 1);
  else
    snprintf (track->number, sizeof(track->number), "0%i", atoi(track->number) + 1);
  
  return 0;
}


/*
 * Grab a track title from a freedb string into an already existing track struct.
 */
static void
freedb_line_to_title(char *line, CDTrack *track)
{
  strncpy (track->title, strchr(line, '=') + 1, sizeof(track->title));
}


/*
 * Find a track in a cd.
 * Returns a pointer to the track.
 */
static CDTrack * freedb_find_track (char *number, CDInfo *cd)
{
  CDTrack * track = NULL;
  int i = -1;
  
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_find_track: Searching\n");
#endif
  
  while ( cd->track[++i] != NULL ) {
    track = cd->track[i];
    
    if ( strcmp (track->number, number) == 0 )
      return (track);
  }
  
  return (NULL);
}


/*
 * Append a new track to a cd.
 */
static void
freedb_new_track (CDTrack *track, CDInfo *cd)
{
  int i = -1;
  
  /* Find the last track */
  while ( cd->track[++i] != NULL );
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_new_track: lasttrack: %i\n", i);
#endif
  
  /* append a new track */
  cd->track[i] = track;
  cd->track[i + 1] = NULL;
  
  return;
}


/*
 * Append a track title string to an already existing title.
 */
static void
freedb_increase_track (CDTrack *newtrack, CDInfo *cd)
{
  CDTrack *havetrack = NULL;
  char newtitle[2048];
  
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_increase_track: newtrack->title: %s\n", newtrack->title);
#endif
  
  havetrack = freedb_find_track (newtrack->number, cd);
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_increase_track: havetrack->title: %s\n", havetrack->title);
#endif
  
  /* Merge the titles */
  snprintf (newtitle, 2048, "%s%s", havetrack->title, newtrack->title);
  strncpy (havetrack->title, newtitle, sizeof (havetrack->title));
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_increase_track: havetrack->title new: %s\n", havetrack->title);
#endif
}


/*
 * Parse a freedb output line and append it as track to a cd.
 * Returns <0 on an error, otherwise 0.
 */
static int
freedb_add_track (char *line, CDInfo *cd)
{
  CDTrack *track = calloc (1, sizeof (CDTrack));
  
  /* The line must contain a "=" character */
  if ( strchr (line, '=') == NULL )
    return -1;
  
  /* Grab the track number */
  if ( freedb_line_to_track (line, track) < 0 )
    goto error;
  
  /* Grab the title */
  freedb_line_to_title (line, track);
#ifdef _DEBUG_EXCESSIVE_
  fprintf (stderr, "freedb_add_track: track->title: %s\n", track->title);
#endif
  
  /* it is possible to split a track name via multiple freedb lines. *
  * thus, the same track may have occured before. In that case, we  *
  * just append the current line to the older one.                  */
  if ( freedb_find_track (track->number, cd) != NULL )
    freedb_increase_track (track, cd);
  else
    freedb_new_track (track, cd);
  
  return 0;
  
error:
  free (track);
  return -1;
}


static void
freedb_add_year(char *line, CDInfo *cd)
{
  /* the line has the format: "DYEAR=<year>" */
  if ( strchr (line, '=') != NULL )
    strncpy (cd->year, strchr (line, '=') + 1, 4);
#ifdef _DEBUG_
    printf ("cd->year: %s\n", cd->year);
#endif
}
/***************************************************************************************
 * STATICS END
 ***************************************************************************************/


/*
 * Connect to a host, return the socket number.
 */
int
freedb_connect_host (char *hostname, unsigned int host_port, char *proxy, unsigned int proxy_port) 
{
  struct sockaddr_in socket_address;
  struct hostent *hostent;
  int socket_number;

  /* Get a Socket number */
  if ( (socket_number = socket (AF_INET, SOCK_STREAM, 0)) == -1 ) {
    perror("Error: Can't open INET socket!");
    return -1;
  }
  
  /* No proxy? Then the host is proxy, also. */
  if ( !proxy || *proxy == '\0' ) {
    proxy = hostname;
    proxy_port = host_port;
  }
  
  /* Server address lookup */
  if ( !(hostent = gethostbyname(proxy)) ) {
    perror("Failed to lookup the server!");
    return -2;
  }
  
  socket_address.sin_family = AF_INET;
  memcpy(&socket_address.sin_addr, hostent->h_addr,	hostent->h_length);
  socket_address.sin_port = htons(proxy_port);
	
  /* Connect to the server... */
  if ( connect(socket_number, (struct sockaddr *) &socket_address, sizeof(socket_address)) == -1 ) {
    perror("Can't connect to FreeDB host!");
    return -3;
  }
  
  return (socket_number);
}


int freedb_get_album_by_id_and_category (int socket_number, char *hostname, unsigned int host_port, char *cgi_path, char *progname, char *version, CDInfo **pcd) 
{
  CDInfo *cd = *pcd;
  char bin[4096] = "";	    /* Binary in data  */
  char request[4096];
  
  /* A socket > 0 is mandatory. */
  if (socket_number <= 0)		
    return (-1);
  
  /* Create request */
  freedb_create_id_request (request, hostname, host_port, cgi_path, progname, version, cd, sizeof (request) - 1);
  
#ifdef _DEBUG_
  fprintf (stderr, "------------\nSending request: %s\n------------\n", request);
#endif
  
  /* Send request */
  if ( write(socket_number, request, strlen(request)) <= 0 ) {
    fprintf (stderr, "Error while sending request to the server!");
    return (-2);
  }
  
#ifdef _DEBUG_
  fprintf (stderr, "Request sent.\n");
#endif
  
  /* We get the answer of the query                      *
   * the first line does NOT contain a title (and there  *
   * may be more lines like that), but some header       *
   * informations we dont care about.                   */
  if ( get_line (socket_number, bin, sizeof(bin)) <= 0 ) {
    fprintf (stderr, "Error while receiving request from the server!");
    return (-3);
  }
  
  if ( strncmp (bin, "HTTP/1.1 200 ", 13) != 0
    && strncmp (bin, "HTTP/1.0 200 ", 13) != 0 ) {
    fprintf (stderr, "Error while receiving track information. (%s)\n", bin);
    return (-4);
  }
#ifdef _DEBUG_
  fprintf (stderr, ">> %s\n", bin);
#endif
  
  /* Receive the complete header. */
  while ( strcmp (bin, ".\r\n") != 0 ) {
    /* Get one line */
    if ( get_line (socket_number, bin, sizeof(bin)) <= 0 ) {
      fprintf (stderr, "Error while receiving the header!");
      return (-3);
    }
#ifdef _DEBUG_
  fprintf (stderr, ">> %s\n", bin);
#endif
    
    /* Empty line = end of the header. */
    if ( bin[0] == '\0' )
      break;
  }
  
  /* so the server sent a valid reply, we can get all titles now. *
   * we only care about lines beginning with "TTITLE", that will  *
   * skip all header lines and stuff. A line containing nothing   *
   * but a dot (plus a linefeed) means end of data.               */
  cd->track[0] = NULL;
  
  while ( strcmp (bin, ".\r\n") != 0 ) {
    /* Get one line */
    get_line (socket_number, bin, sizeof (bin));
#ifdef _DEBUG_
  fprintf (stderr, ">> %s\n", bin);
#endif
    
    /* If we received an empty line before we receive a *
    * track, return with an error.                     */
    if ( bin[0] == '\0' && cd->track[0] == NULL ) {
      fprintf (stderr, "Error while receiving track information!");
      return (-5);
    }
    
    /* Empty line = end of data */
    if ( bin[0] == '\0' )
      break;
    
    /* Received a title? Then add it to our track list */
    if ( strncmp (bin, "TTITLE", 6) == 0 )
      freedb_add_track (bin, cd);
    
    if ( strncmp (bin, "DYEAR", 5) == 0 )
      freedb_add_year (bin, cd);
  }
    
// Close the socket connection with my host
  return 0;
}



int
freedb_get_albumlist_by_searchstring(int socket_number, char *pstring, GList **cdlist) 
{
  char bin[4096] = "";     // In buffer
  char request[4096] = "";
  char searchstring[4096] = "";
  int i = 0;
  
  /* A searchstring is mandtory. */
  if ( !pstring || *pstring == '\0' )
    return 0;
  
  /* A socket > 0 is mandatory. */
  if ( socket_number <= 0 )
    return (-1);

  /* Change all spaces in the searchstring to "+" (URLsafe) */
  strncpy (searchstring, pstring, sizeof(searchstring) - 1);
  urlencode (searchstring);

  /* prepare request */
  freedb_create_searchstring_request (request, searchstring, sizeof (request));
  
#ifdef _DEBUG_
  fprintf (stderr, "------------\nSending request: %s\n------------\n", request);
#endif
  
  /* Send request */
  if ( write(socket_number, request, strlen(request)) < 0 ) {
    fprintf (stderr, "Error while sending request to the server!\n");
    return -4;
  }
  
#ifdef _DEBUG_
  fprintf (stderr, "Request sent.\n");
#endif
  
  /* Read the answer. */
  i = 0;
  while ( strncmp (bin, "</body>", 7) != 0 ) {
    /* Get one line. */
    if ( get_line (socket_number, bin, sizeof(bin)) <= 0 ) {
      fprintf (stderr, "Error while receiving cd information!\n");
      return -7;
    }
#ifdef _DEBUG_
  fprintf (stderr, ">> %s\n", bin);
#endif
    
    /* Have we found a line that looks like a cd title in the html output? */
    if ( strncmp (bin, "<tr><td><a href=\"http://www.freedb.org/freedb_search_fmt.php?cat=", 65) == 0 ) {
      /* Add the new cd to our list. */
      if ( freedb_add_album_to_cdlist (bin, cdlist) < 0 ) {
        perror ("Wrong format of the freedb output! Contact this programs author if this happens often.");
        return -9;
      }
    }
  }
  
  /* The user must close the socket connection with the freedb host. */
  return 0;
}
