/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#include "support.h"

#include "mp3info.h"
#include "lib_mpgheader.h"
#include "lib_id3v1.h"
#include "lib_id3v2.3.h"
#include "lib_vorbis.h"

/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// Return the file size of the given file *fully named*
static gint
get_file_size(char *filepathname, char *origin)
{
	struct stat pstat;
	gchar buf[5000];
	if ((lstat (filepathname, &pstat)) < 0)
	{
		sprintf (buf, "%s/%s", origin, filepathname);
		if ((lstat (buf, &pstat)) < 0)
			return (gint) 0;
	}
	return (gint)pstat.st_size;
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/




/**********************************************************************
 * get all available information of the mp3file "filename" in dir "dir"
 * and put them into the struct "Mp3Info"
 **********************************************************************/
void
mp3file_get_info_1 (gchar *pdir, gchar *pfilename, Mp3Info *mp3info)
{
	gchar filepathname[4096] = "";
	gchar filename[2048] = "";
	gchar dir[2048] = "";
	struct stat filestatus;
	
	Header *mp3header = malloc (sizeof(Header));
	id3Tag *tag = malloc (sizeof(id3Tag));

// i have to copy these, because the user may have given a pointer to the
// mp3info itselves, and when i clear it, it would be gone...
	strncpy (dir, pdir, 2047);
	strncpy (filename, pfilename, 2047);
	
// User must provide a pointer to memory for the infos!
	sprintf (filepathname, "%s%s", dir, filename);
	
	if( !mp3info
		|| access (filepathname, R_OK) != 0 )
	{
		free (mp3header);
		free (tag);
		return;
	}

// wipe out mp3info
	memset (mp3info, 0, sizeof(Mp3Info));
	
// directory and filename.
	strncpy (mp3info->filename, filename, 2047);
	strncpy (mp3info->directory, dir, 2047);
// get filesize	
	mp3info->filesize = (gulong)get_file_size (filename, dir);
	
	if ( stat (filepathname, &filestatus) != 0 )
	{
		free (mp3header);
		free (tag);
		return;
	}
	
// get owner
	if( getpwuid (filestatus.st_uid) )
		strncpy (mp3info->owner, getpwuid(filestatus.st_uid)->pw_name, 511);
	else
		strncpy (mp3info->owner, "?", 511);
// group
	if( getgrgid (filestatus.st_gid) )
		strncpy (mp3info->group, getgrgid(filestatus.st_gid)->gr_name, 511);
	else
		strncpy (mp3info->group, "?", 511);
// mode
	sprintf (mp3info->mode, "%u%u%u",
		(gint)(filestatus.st_mode & 0700) >> 6,
		(gint)(filestatus.st_mode & 0070) >> 3,
		(gint)(filestatus.st_mode & 0007) );

// prepare: get old Tag
	memset (tag->artist, '\0', 1024);
	memset (tag->title, '\0', 1024);
	memset (tag->album, '\0', 1024);
	memset (tag->year, '\0', 1024);
	memset (tag->comment, '\0', 1024);
	memset (tag->track, '\0', 3);
	memset (tag->genre, '\0', 512);
	
// get the v1 tag to a struct.
	if ( get_id3v1_tag (tag, filepathname) == 0 )
	{
// Copy the v1 taginfo into the fileinfo
		strncpy (mp3info->v1_artist, tag->artist, 1023);
		strncpy (mp3info->v1_title, tag->title, 1023);
		strncpy (mp3info->v1_album, tag->album, 1023);
		strncpy (mp3info->v1_year, tag->year, 4);
		strncpy (mp3info->v1_comment, tag->comment, 1023);
		strncpy (mp3info->v1_track, tag->track, 3);
		strncpy (mp3info->v1_genre, tag->genre, 511);
	}
// Copy the v2 taginfo into the fileinfo
	if( get_id3v2_tag (tag, filepathname) == 0 )
	{
		strncpy (mp3info->v2_artist, tag->artist, 1023);
		strncpy (mp3info->v2_title, tag->title, 1023);
		strncpy (mp3info->v2_album, tag->album, 1023);
		strncpy (mp3info->v2_year, tag->year, 4);
		strncpy (mp3info->v2_comment, tag->comment, 1023);
		strncpy (mp3info->v2_track, tag->track, 3);
		strncpy (mp3info->v2_genre, tag->genre, 511);
	}
#ifdef HAVE_OGG_H
	if ( get_vorbis_tag (tag, filepathname) == 0 )
	{
		strncpy (mp3info->ogg_artist, tag->artist, 1023);
		strncpy (mp3info->ogg_title, tag->title, 1023);
		strncpy (mp3info->ogg_album, tag->album, 1023);
		strncpy (mp3info->ogg_year, tag->year, 4);
		strncpy (mp3info->ogg_comment, tag->comment, 1023);
		strncpy (mp3info->ogg_track, tag->track, 3);
		strncpy (mp3info->ogg_genre, tag->genre, 511);
	}
#endif
	
// get mpg header infos
	if ( get_mpgheader(mp3header, filepathname) == 0 )
	{
		mp3info->seconds = mp3header->seconds;
		mp3info->bitrate = mp3header->bitrate;
		mp3info->frequency = mp3header->frequency;
		mp3info->channelmode = mp3header->mode;
	}

	if ( *mp3info->status == '\0' )
		strncpy (mp3info->status, "Todo", 255);
	
// cleanup
	free (mp3header);
	free (tag);
}





/**********************************************************************
 * get all available information of all mp3files in the given list
 * and return them in a new list
 **********************************************************************/
GList *
mp3file_get_infos(GList *fullfilenames)
{
	extern GtkWidget *cantus;
	GList *item = NULL;
	GList *mp3infolist = NULL;
	Mp3Info *mp3info = NULL;
	gchar directory[2048];
	gchar filename[2048];
	gint numrows = 0, currentrow = 0;
	gchar status[2048];
	
// Count rows (needed for the statusbar)
	item = g_list_first(fullfilenames);
	while(item)
	{
		item = item->next;
		++numrows;
	}
	
	item = g_list_first(fullfilenames);
	while(item)
	{
// Set status/progressbar.
		snprintf(status, 2047, "Examining: %s", (gchar *)item->data);
		gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), status);
		gnome_appbar_set_progress(GNOME_APPBAR (lookup_widget (cantus, "appbar")), (gfloat)100/numrows*currentrow++*0.01);
		while ( g_main_iteration(FALSE) );
		
		strncpy(directory, item->data, 2047);
		*(strrchr(directory, '/') + 1) = '\0';
		strncpy(filename, strrchr(item->data, '/') + 1, 2047);
		
		mp3info = malloc(sizeof(Mp3Info));
		mp3file_get_info_1(directory, filename, mp3info);
		mp3infolist = g_list_append(mp3infolist, mp3info);
		
		item = item->next;
	}
	
// Reset status
	gnome_appbar_set_status(GNOME_APPBAR (lookup_widget (cantus, "appbar")), "Ready.");
	gnome_appbar_set_progress(GNOME_APPBAR (lookup_widget (cantus, "appbar")), 0);
	return mp3infolist;
}
