/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <ctype.h>
#include <sys/stat.h>

#include "plugs.h"
#include "support.h"
#include "lib_id3v1.h"
#include "lib_id3v2.3.h"
#include "lib_vorbis.h"
#include "configfile.h"
#include "shared.h"

/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// determine the correct order for the queue columns
static gboolean
is_spaceonly(gchar *name)
{
	if( *name == '\0' )
		return FALSE;
	
	while( *name == ' ' )
		++name;
	
	if( *name == '\0' )
		return TRUE;
	
	return FALSE;
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/



/**********************************************************************
 * will strip all characters from the beginning of the string to the first occurence of a digit
 **********************************************************************/
void
apply_rule_strip_to_digit(gchar *source)
{
	gchar *psource;
	gchar *destination = calloc(2000, sizeof(gchar));
	gchar *temp = destination;

	if( source != NULL )
	{
		for(psource=source;(!isdigit(*psource) && *psource!='\0'); psource++);
		while(*psource!='\0')
			*destination++=*psource++;
		
		if( strlen(temp) > 0 )
			strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}












/**********************************************************************
 * this will remove parentheses. either all (all==TRUE) or only the first two.
 **********************************************************************/
void
apply_rule_remove_parentheses(gchar *source, gboolean all)
{
	gchar *psource;
	gchar *destination = calloc(2000, sizeof(gchar));
	gchar *temp = destination;

	gboolean firstopendone = FALSE;
	gboolean firstclosedone = FALSE;

	if(source != NULL)
	{
		for(psource = source; *psource != '\0'; psource++)
		{
			if( (*psource!='(')
			&& (*psource!=')') )
			{
				*destination++ = *psource;
			}
			else {
				if(all==FALSE)
				{
					if( (*psource=='(') && firstopendone )
						*destination++ = *psource;
					if( (*psource==')') && firstclosedone )
						*destination++ = *psource;

					if( (*psource=='(') && !firstopendone )
						firstopendone=TRUE;
					if( (*psource==')') && !firstclosedone )
						firstclosedone=TRUE;
				}
			}
		}
		
		strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}











/**********************************************************************
 * this will replace all multiple spaces by one space.
 * spaces are:
 * single space, tab, vertical tab, form feed, carriage return, or newline
 **********************************************************************/
void
apply_rule_strip_multiple_spaces(gchar *source)
{
	gchar *psource;
	gchar *destination = calloc(2000, sizeof(gchar));
	gchar *temp = destination;

	if( source != NULL )
	{
		for(psource = source; *psource != '\0'; psource++)
		{
			if( !isspace(*psource)
			|| (isspace(*psource) && !isspace(*(psource+1))) )
			{
				*destination++ = *psource;
			}
		}
		
		strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}











/****************************************************************************
 * this will set the first single(!!!) digit to twodigit.
 ****************************************************************************/
void
apply_rule_digit_to_twodigit(gchar *source)
{
	gchar *psource;
	gchar *destination = calloc(2000, sizeof(gchar));
	gchar *temp = destination;

	if(source != NULL)
	{
		for(psource = source; *psource != '\0'; psource++)
		{
// found twodigit? then return without a change.
			if(isdigit(*psource) && isdigit(*(psource+1)))
			{
				free(temp);
				return;
			}

			if(isdigit(*psource))
			{
				*destination++ = '0';
				while(*psource != '\0')
					*destination++ = *psource++;
				break;
			}
			else {
				*destination++ = *psource;
			}
		}
		
		strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}












/****************************************************************************
 * this will insert a char before each uppercase character exept the first.
 ****************************************************************************/
void
apply_rule_insert_char_before_uppercase(gchar *source, gchar *insert)
{
	gchar *psource;
	gchar *destination = calloc(2000,sizeof(gchar));
	gchar *temp=destination;

	if(source != NULL)
	{
		for(psource = source; *psource != '\0'; psource++)
		{
			if( isupper(*psource)
			&& (psource!=source))
			{
				*destination++ = *insert;
				while(*psource != '\0')
					*destination++ = *psource++;
				break;
			}
			else {
				*destination++ = *psource;
			}
		}
		
		strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}











/****************************************************************************
 * should be self-explanatory
 ****************************************************************************/
void
apply_rule_replace(gchar *source, gchar *from, gchar *to)
{
	gchar *psource;
	gchar *pto;
	gchar *destination = calloc(2048,sizeof(gchar));
	gchar *temp = destination;

	if( source != NULL
		&& from != NULL
		&& *from != '\0' )
	{
		for(psource = source; *psource != '\0'; psource++)
		{
			if( strncmp(psource, from, strlen(from)) == 0 )
			{
				pto = to;
				while(*pto != '\0')
					*destination++ = *pto++;
				psource += strlen(from) - 1;
			}
			else
			{
				*destination++ = *psource;
			}
		}
		strncpy(source, temp, 2047);
	}
	
	free(temp);
	return;
}












/****************************************************************************
 * this will delete all chars in "source" from position "from" to "to".
 ****************************************************************************/
void
apply_rule_delete_char(gchar *source, gint from, gint to)
{
	gchar *psource = NULL;
	gchar *destination = calloc(2000,sizeof(gchar));
	gchar *temp=destination;
	gint i=0;

	from--;
	to--;
	
	if(source != NULL)
	{
		for(psource = source; *(psource+i) != '\0'; i++)
		{
			if((i<from) || (i>to))
			{
				*destination++ = *(psource+i);
			}
		}
		
		strncpy(source, temp, 2047);
	}

	free(temp);
	return;
}





/****************************************************************************
 * this will delete all chars in "source" befor the first/last occurence
 * of a string.
 ****************************************************************************/
void
apply_rule_strip_to_str(gchar *source, gchar *to, gchar *string)
{
	gchar *psource = source;
	gint i = 1;

	if( source == NULL
		|| to == NULL
		|| string == NULL
		|| *string == '\0' )
		return;
	
	if(strcmp(to, "first occurence")==0)
	{
		while( *psource != '\0'
			&& strncmp(psource, string, strlen(string)) != 0 )
			++psource;
		
		if(*psource != '\0')
			strncpy(source, psource, 2048);
	}
	else if(strcmp(to, "last occurence")==0)
	{
		while( strncmp(psource+strlen(source)-i, string, strlen(string)) != 0
			&& i < strlen(source) )
			++i;
		
		if(i < strlen(source))
			strncpy(source, psource+strlen(source)-i, 2048);
	}
	
	return;
}














/****************************************************************************
 * this will increase the first digit in source by number.
 ****************************************************************************/
void
apply_rule_number_increase(gchar *source, gint number)
{
	gchar *psource;
	gchar *destination = calloc(2000,sizeof(gchar));
	gchar *temp=destination;
	gchar *digits = calloc(20,sizeof(gchar));
	gchar *digitspsource=digits;
	gchar *foo = calloc(20,sizeof(gchar));
	gchar *footemp=foo;
	gint i;

	if(source != NULL)
	{
		psource=source;
		while(!isdigit(*psource) && *psource!='\0')
		{
			*destination++ = *psource++;
		}
		if(*psource=='\0')
		{
			free(footemp);
			free(temp);
			free(digitspsource);
			return;
		}
		while(isdigit(*psource))
		{
			*digits++ = *psource++;
		}
		digits=digitspsource;
		i=(gint)atoi(digits);
		sprintf(foo, "%i", i+number);
		if(strlen(foo) < strlen(digits))
		{
			while(*foo!='\0') {
				digits++;
				*digits = *foo;
				foo++;
			}
		}
		else sprintf(digits, "%s", foo);
		digits = digitspsource;
		while(*digits!='\0')
		{
			*destination++=*digits++;
		}
		while(*psource!='\0')
		{
			*destination++=*psource++;
		}
		
		strncpy(source, temp, 2047);
	}

	free(footemp);
	free(temp);
	free(digitspsource);
	return;
}














/****************************************************************************
 * this will set the characters case:
 * mode=1:first word uppercase;
 * mode=2:every word uppercase;
 * mode=3:anything to lower;
 * mode=4:every character to upper.
 ****************************************************************************/
void
apply_rule_forcecc(gchar *source, gint mode)
{
	gchar *psource;
	gchar *destination = calloc(2000,sizeof(gchar));
	gchar *temp=destination;
	gboolean done=FALSE;

	if( source != NULL )
	{
		for(psource = source; *psource != '\0'; psource++)
		{
			if ( mode == 1 )
			{
				if( isalpha(*psource) && (done==FALSE) )
				{
					*destination++=toupper(*psource);
					done=TRUE;
				}
				else
					*destination++=tolower(*psource);
			}

			if ( mode == 2 )
			{
				if( psource == source
					|| ( (!isalpha(*(psource-1)) && (*(psource-1)!='\'') && (*(psource-1)!=''))
						&& (isalpha(*psource)) ) )
					*destination++ = toupper(*psource);
				else
					*destination++ = *psource;
			}

			if ( mode == 3 )
			{
				if(isalpha(*psource))
					*destination++ = tolower(*psource);
				else
					*destination++ = *psource;
			}

			if ( mode == 4 )
			{
				if(isalpha(*psource))
					*destination++ = toupper(*psource);
				else
					*destination++ = *psource;
			}
		}
		
		strncpy(source, temp, 2047);
	}

	free(temp);
	return;
}










/****************************************************************************
 * this will get data from an id3tag (v1, v2 or ogg) and store formatted in filename.
 ****************************************************************************/
gboolean
apply_rule_gettag(gchar *source, gchar *fullfilename, gchar *format, gint whichtag)
{
	id3Tag *tag = calloc(2000,sizeof(id3Tag));
	gboolean error=TRUE;
	gchar newname[4096];
	gchar onedigittrack[16];
	gchar twodigittrack[16];

// get the tag to a struct.
	if(whichtag==1 && get_id3v1_tag(tag, fullfilename)==0)
		error = FALSE;
	if(whichtag==2 && get_id3v2_tag(tag, fullfilename)==0)
		error = FALSE;
#ifdef HAVE_OGG_H
	if(whichtag==3 && get_vorbis_tag(tag, fullfilename)==0)
		error = FALSE;
#endif
	
// If one of the requested options is not present, go back without a change.
	if( strstr(format, "%a")!=NULL
		&& (*tag->artist=='\0' || is_spaceonly(tag->artist)) )
		error=TRUE;
	if( strstr(format, "%s")!=NULL
		&& (*tag->title=='\0' || is_spaceonly(tag->title)) )
		error=TRUE;
	if( strstr(format, "%b")!=NULL
		&& (*tag->album=='\0' || is_spaceonly(tag->album)) )
		error=TRUE;
	if( strstr(format, "%y")!=NULL
		&& (*tag->year=='\0' || is_spaceonly(tag->year)) )
		error=TRUE;
	if( strstr(format, "%t")!=NULL
		&& (*tag->track=='\0' || is_spaceonly(tag->track)) )
		error=TRUE;
	if( strstr(format, "%T")!=NULL
		&& (*tag->track=='\0' || is_spaceonly(tag->track)) )
		error=TRUE;
	
// set all data.
	if(!error)
	{
// The track can be eigther twodigit (e.g. "01"), or onedigit.
		snprintf(onedigittrack, 15, "%i", atoi(tag->track));
		if(atoi(tag->track) < 10)
			snprintf(twodigittrack, 15, "0%i", atoi(tag->track));
		else
			snprintf(twodigittrack, 15, "%i", atoi(tag->track));

		sprintf(newname, "%s", format);

		apply_rule_replace(newname, "%a", tag->artist);
		apply_rule_replace(newname, "%s", tag->title);
		apply_rule_replace(newname, "%b", tag->album);
		apply_rule_replace(newname, "%y", tag->year);
		apply_rule_replace(newname, "%t", onedigittrack);
		apply_rule_replace(newname, "%T", twodigittrack);
		apply_rule_replace(newname, "%f", source);
		
		if(*newname != '\0')
			strncpy(source, newname, 2047);
	}
	free(tag);
	
	if(error)
		return FALSE;
	else
		return TRUE;
}







/****************************************************************************
 * this will kill spaces at the start/end/both of source.
 * spaces are:
 * single space, tab, vertical tab, form feed, carriage return, or newline
 * loc: 0-start 1-end 2-both
 ****************************************************************************/
void
apply_rule_wipe_spaces(gchar *source, gint loc)
{
	gchar *psource = NULL;
	gchar *destination = calloc(2000,sizeof(gchar));
	gchar *temp=destination;

	if(source != NULL)
	{
		if( loc == 0
			|| loc == 2 )
		{
			strcpy(destination, source);
			while(*destination == ' ')
				destination++;
			sprintf(source, "%s", destination);
		}
		if( loc == 1
			|| loc == 2 )
		{
			for(psource = source + strlen(source) - 1; *psource != '\0'; psource--)
			{
				if( isspace(*psource) )
					*psource='\0';
				else
					break;
			}
		}
	}
	free(temp);
	return;
}






/****************************************************************************
 * this will extract infos from the current filename and put
 * and put it together again with a new format.
 ****************************************************************************/
void
apply_rule_format_to_format(gchar *filename, gchar *sformat, gchar *dformat)
{
	gint i=0, u=-1, f=0;
	
	gchar *trenner[8];
	gchar *ptrenner[8];
	gchar *option[8][2];
	gchar *psformat = sformat;
	gchar *pdformat = dformat;
	gchar *pfilename = filename;
	gchar foo[1000];
	gchar *pfoo = foo;
	gchar *destination = calloc(3000, sizeof(gchar));
	gchar *pdestination=destination;
	
	
	trenner[0] = calloc(2000, sizeof(gchar));
	ptrenner[0] = trenner[0];
	option[0][0] = calloc(2000, sizeof(gchar));
	option[0][1] = calloc(2000, sizeof(gchar));
	
/* Save the parts between %-flags in "trenner". Save the switches in "options" */
	if(sformat != NULL)
	{
		for(psformat=sformat; (*psformat!='\0'); psformat++)
		{
			if(*psformat!='%')
			{
				*ptrenner[i]++=*psformat;
			}
			else
			{
				if(*(++psformat)=='%')
				{
					*ptrenner[i]++=*psformat;
				}
				else
				{
					*option[i][0] = *psformat;
					if(i+1>=7) break;
					trenner[++i] = calloc(2000, sizeof(gchar));
					ptrenner[i] = trenner[i];
					option[i][0] = calloc(2000, sizeof(gchar));
					option[i][1] = calloc(2000, sizeof(gchar));
				}
			}
		}
	
		pfilename=filename;
		while(++u<=i)
		{
			if(*option[u][0]=='\0') break;
			ptrenner[u] = trenner[u];
			for(; (*ptrenner[u]!='\0'); pfilename++)
			{
				if(*pfilename!=*ptrenner[u]++)
				{
					for(f=0; f<=i; f++)
					{
						free(trenner[f]);
						free(option[f][0]);
						free(option[f][1]);
					}
					free(destination);
					return;
				}
			}
			
			memset(foo, 0, 1000);
 			pfoo=foo;
			for(; (*pfilename!='\0'); pfilename++)
			{
				if(*trenner[u+1]=='\0')
				{
					strcpy(foo, pfilename);
					break;
				}
				if(strncmp(pfilename, trenner[u+1], strlen(trenner[u+1]))==0)
					break;
				else
					*pfoo++=*pfilename;
			}
			strcpy(option[u][1], foo);
		}
	}

	for(f=0; f<=i; f++)
		free(trenner[f]);
		
// So we got the source-parts nice formatted in 'options', we now figure out, how to fit them together again.
	u=-1;
	pdformat=dformat;
	while(++u<=i)
	{
		if(*option[u][0]=='\0') break;
/* browse through the format given by the user as desired destination */
		for(; (*pdformat!='\0'); )
		{
			// no '%' means, copy the char to the destination string
			if(*pdformat!='%')
				*(pdestination++)=*pdformat++;

			// This signs a valid option. if found, quit this for-loop!
			if(*pdformat=='%'
				&& *(pdformat+1)!='%')
				break;

			// a double '%' means just print a single '%' out to the destination string.
			if(*pdformat=='%'
				&& *(pdformat+1)=='%')
			{
				*(pdestination++)=*pdformat;
				pdformat+=2;
			}
		}

/* we arrive here if either a valid option marker (e.g. '%a') was found, or if the format-string ended
 * so we will look for which option was requested and append this part to the destination string */
		if(*(++pdformat)!='\0')
		{
			for(f=0;f<=i;f++)
			{
				if(*option[f][0]==*pdformat)
				{
					strncpy(pdestination, option[f][1], strlen(option[f][1]));
					pdestination+=strlen(option[f][1]);
					break;
				}
			}
			++pdformat;
		}
	}
	memset(filename, 0, strlen(destination)+1);
	strncpy(filename, destination, strlen(destination));
	for(u=0; u<=i; u++)
	{
		free(option[u][0]);
		free(option[u][1]);
	}
	free(destination);
}





/**********************************************************************
 * this will change padding spaces in an ID3tag to NULL
 **********************************************************************/
gboolean
apply_rule_tagcleanup(gchar *fullfilename)
{
	id3Tag *tag = malloc(sizeof(id3Tag));
	gboolean noerror = TRUE;
	
// Get old Tag
	memset(tag->artist, '\0', 1024);
	memset(tag->title, '\0', 1024);
	memset(tag->album, '\0', 1024);
	memset(tag->year, '\0', 5);
	memset(tag->comment, '\0', 1024);
	memset(tag->genre, '\0', 512);
	if( get_id3v1_tag(tag, fullfilename)==0 )
	{
		apply_rule_wipe_spaces(tag->artist, 1);
		apply_rule_wipe_spaces(tag->title, 1);
		apply_rule_wipe_spaces(tag->album, 1);
		apply_rule_wipe_spaces(tag->year, 1);
		apply_rule_wipe_spaces(tag->comment, 1);
	
	// And put the changed tag to the file (if allowed)
		if(access(fullfilename, W_OK)==0)
		{
			if(set_id3v1_tag(tag, fullfilename)!=0)
				noerror = FALSE;
		}
		else
			noerror = FALSE;
	}
	
	memset(tag->artist, '\0', 1024);
	memset(tag->track, '\0', 3);
	memset(tag->title, '\0', 1024);
	memset(tag->album, '\0', 1024);
	memset(tag->year, '\0', 5);
	memset(tag->comment, '\0', 1024);
	memset(tag->genre, '\0', 512);
	if( get_id3v2_tag(tag, fullfilename)==0 )
	{
		apply_rule_wipe_spaces(tag->artist, 1);
		apply_rule_wipe_spaces(tag->title, 1);
		apply_rule_wipe_spaces(tag->album, 1);
		apply_rule_wipe_spaces(tag->year, 1);
		apply_rule_wipe_spaces(tag->comment, 1);
		apply_rule_wipe_spaces(tag->genre, 1);
	
	// And put the changed tag to the file (if allowed)
		if(access(fullfilename, W_OK)==0)
		{
			if(set_id3v2_tag(tag, fullfilename)!=0)
				noerror = FALSE;
		}
		else
			noerror = FALSE;
	}
	
	free(tag);
	return noerror;
}




/**********************************************************************
 * this will copy the contents of an ID3V1tag to the V2 tag.
 **********************************************************************/
gboolean
apply_rule_v1tov2(gchar *fullfilename)
{
	id3Tag *tag = calloc (1, sizeof(id3Tag));
	gboolean noerror = TRUE;
	
// Get old Tag
	memset (tag->artist, '\0', 1024);
	memset (tag->title, '\0', 1024);
	memset (tag->album, '\0', 1024);
	memset (tag->year, '\0', 5);
	memset (tag->comment, '\0', 1024);
	memset (tag->genre, '\0', 512);
	if ( get_id3v1_tag(tag, fullfilename) == 0 )
	{
		apply_rule_wipe_spaces (tag->artist, 1);
		apply_rule_wipe_spaces (tag->title, 1);
		apply_rule_wipe_spaces (tag->album, 1);
		apply_rule_wipe_spaces (tag->year, 1);
		apply_rule_wipe_spaces (tag->comment, 1);

// And put the changed tag to the file (if allowed)
		if( access (fullfilename, W_OK) == 0 )
		{
			if ( set_id3v2_tag(tag, fullfilename) != 0 )
				noerror = FALSE;
		}
		else
			noerror = FALSE;
	}
	
	free(tag);
	
	return noerror;
}





/**********************************************************************
 * this will copy the contents of an ID3V2tag to an V1 tag.
 **********************************************************************/
gboolean
apply_rule_v2tov1(gchar *fullfilename)
{
	id3Tag *tag = calloc(1,sizeof(id3Tag));
	gboolean noerror = TRUE;
	
// And put the changed tag to the file (if allowed)
	if( (get_id3v2_tag(tag, fullfilename)==0)
		&& (access(fullfilename, W_OK)==0) )
	{
		if(set_id3v1_tag(tag, fullfilename)!=0)
			noerror = FALSE;
	}
	else
		noerror = FALSE;
	
	free(tag);
	
	return noerror;
}




/**********************************************************************
 * this will set the permissions for a file.
 **********************************************************************/
gboolean
apply_rule_setperms(gchar *fullfilename, gchar *mode, gchar *rw)
{
	struct stat filestatus;
	gint oldmode = 0;
	gint newmode = 0;

// Get old file mode
	lstat(fullfilename, &filestatus);
	if( (gint)(filestatus.st_mode & 0400) >> 6 )
		oldmode = (oldmode | S_IRUSR);
	if( (gint)(filestatus.st_mode & 0200) >> 6 )
		oldmode = (oldmode | S_IWUSR);
	if( (gint)(filestatus.st_mode & 0100) >> 6 )
		oldmode = (oldmode | S_IXUSR);
	
	if( (gint)(filestatus.st_mode & 0040) >> 3)
		oldmode = (oldmode | S_IRGRP);
	if( (gint)(filestatus.st_mode & 0020) >> 3)
		oldmode = (oldmode | S_IWGRP);
	if( (gint)(filestatus.st_mode & 0010) >> 3)
		oldmode = (oldmode | S_IXGRP);
	
	if( (gint)(filestatus.st_mode & 0004) )
		oldmode = (oldmode | S_IROTH);
	if( (gint)(filestatus.st_mode & 0002) )
		oldmode = (oldmode | S_IWOTH);
	if( (gint)(filestatus.st_mode & 0001) )
		oldmode = (oldmode | S_IXOTH);

// Set the perms.
	if(strcmp(rw, "read")==0)
	{
		if( strcmp(mode, "permit (all)")==0 )
			newmode = (oldmode | S_IRUSR | S_IRGRP | S_IROTH);
		if( strcmp(mode, "deny (all)")==0 )
			newmode = (oldmode & ~(S_IRUSR | S_IRGRP | S_IROTH));
		if( strcmp(mode, "permit (user)")==0 )
			newmode = (oldmode | S_IRUSR);
		if( strcmp(mode, "deny (user)")==0 )
			newmode = (oldmode & (~S_IRUSR));
		if( strcmp(mode, "permit (group)")==0 )
			newmode = (oldmode | S_IRGRP);
		if( strcmp(mode, "deny (group)")==0 )
			newmode = (oldmode & (~S_IRGRP));
		if( strcmp(mode, "permit (others)")==0 )
			newmode = (oldmode | S_IROTH);
		if( strcmp(mode, "deny (others)")==0 )
			newmode = (oldmode & (~S_IROTH));
	}
	else if(strcmp(rw, "write")==0)
	{
		if( strcmp(mode, "permit (all)")==0 )
			newmode = (oldmode | S_IWUSR | S_IWGRP | S_IWOTH);
		if( strcmp(mode, "deny (all)")==0 )
			newmode = (oldmode & ~(S_IWUSR | S_IWGRP | S_IWOTH));
		if( strcmp(mode, "permit (user)")==0 )
			newmode = (oldmode | S_IWUSR);
		if( strcmp(mode, "deny (user)")==0 )
			newmode = (oldmode & (~S_IWUSR));
		if( strcmp(mode, "permit (group)")==0 )
			newmode = (oldmode | S_IWGRP);
		if( strcmp(mode, "deny (group)")==0 )
			newmode = (oldmode & (~S_IWGRP));
		if( strcmp(mode, "permit (others)")==0 )
			newmode = (oldmode | S_IWOTH);
		if( strcmp(mode, "deny (others)")==0 )
			newmode = (oldmode & (~S_IWOTH));
	}
	else if(strcmp(rw, "execute")==0)
	{
		if( strcmp(mode, "permit (all)")==0 )
			newmode = (oldmode | S_IXUSR | S_IXGRP | S_IXOTH);
		if( strcmp(mode, "deny (all)")==0 )
			newmode = (oldmode & ~(S_IXUSR | S_IXGRP | S_IXOTH));
		if( strcmp(mode, "permit (user)")==0 )
			newmode = (oldmode | S_IXUSR);
		if( strcmp(mode, "deny (user)")==0 )
			newmode = (oldmode & (~S_IXUSR));
		if( strcmp(mode, "permit (group)")==0 )
			newmode = (oldmode | S_IXGRP);
		if( strcmp(mode, "deny (group)")==0 )
			newmode = (oldmode & (~S_IXGRP));
		if( strcmp(mode, "permit (others)")==0 )
			newmode = (oldmode | S_IXOTH);
		if( strcmp(mode, "deny (others)")==0 )
			newmode = (oldmode & (~S_IXOTH));
	}
	if(!chmod(fullfilename, newmode))
		return TRUE;
	
	return FALSE;
}







/****************************************************************************
 * this will get the first two digit number from the filename,
 * and store it in the tags as track.
 ****************************************************************************/
gint
apply_rule_twodigit_to_tag(gchar *fullfilename, gchar *source)
{
	gchar *psource = source;
	id3Tag *tag = malloc(sizeof(id3Tag));

	if(source != NULL)
	{
		for(psource = source; *psource != '\0'; psource++)
		{
// found twodigits, beginning from the pointer?
			if( isdigit(*psource)
				&& isdigit(*(psource+1))
				&& !isdigit(*(psource+2)) )
			{
// make sure there is no digit BEFORE those two digits.
				if( psource==source
					|| (psource!=source	&& !isdigit(*(psource-1))) )
				{
// store the number in both tags as track.
					if(get_id3v1_tag(tag, fullfilename)==0)
					{
						strncpy(tag->track, psource, 3);
						if(set_id3v1_tag(tag, fullfilename)!=0)
						{
							free(tag);
							return 1;
						}
					}
					if(get_id3v2_tag(tag, fullfilename)==0)
					{
						strncpy(tag->track, psource, 3);
						if(set_id3v2_tag(tag, fullfilename)!=0)
						{
							free(tag);
							return 2;
						}
					}
#ifdef HAVE_OGG_H
					if(get_vorbis_tag(tag, fullfilename)==0)
					{
						strncpy(tag->track, psource, 3);
						if(set_vorbis_tag(tag, fullfilename)!=0)
						{
							free(tag);
							return 3;
						}
					}
#endif
					free(tag);
					return 0;
				}
			}
		}
	}
	free(tag);
	return 3;
}







void
apply_rule_replace_invalid_characters(gchar *source)
{
	gchar *psource = source;
	gchar config[10000];
	gchar destination[2048] = "";
	gchar value[2048] = "";
	gchar option[30] = "";
	gboolean found = FALSE;
	gint i = 0;
	
// Read configfile
	config_read(config, 10000);

	if(source != NULL)
	{
		memset(destination, 0, 2048);
		for(psource = source; *psource != '\0'; psource++)
		{
			i = 0;
			found = FALSE;
			snprintf(option, 30, "char%i#1", i);
			while ( configfile_option_get(config, "special characters", option, value) )
			{
				option[0] = *psource;
				option[1] = '\0';
				
				if ( strcmp (value, option) == 0 )
				{
					snprintf(option, 29, "char%i#2", i);
					if ( configfile_option_get(config, "special characters", option, value) )
						strcpy (destination + strlen(destination), value);
					found = TRUE;
					break;
				}
				
				snprintf(option, 30, "char%i#1", ++i);
			}
			
			if( !found && is_valid_filename_char(*psource) )
			{
				destination[strlen(destination) + 1] = '\0';
				destination[strlen(destination)] = *psource;
			}
		}
	}
	
	strncpy(source, destination, strlen(source));
}



void
apply_rule_romandigit_toupper(gchar *source)
{
	gchar *psource = source;
	gchar curword[2048];
	gint cur = 0, i = 0;
	
	if(!source || *source == '\0')
		return;
	
	cur = -1;
	i = 0;
	while(1)
	{
		if( *(psource+i) == 'i'
			|| *(psource+i) == 'I'
			|| *(psource+i) == 'v'
			|| *(psource+i) == 'V'
			|| *(psource+i) == 'x'
			|| *(psource+i) == 'X'
			|| *(psource+i) == 'm'
			|| *(psource+i) == 'M' )
		{
			curword[++cur] = *(psource+i);
			++i;
		}
		else
		{
			if( cur != -1
				&& curword[0] != '\0' )
			{
				if( *(psource+i) == '\0'
					|| *(psource+i) == ' ' )
				{
					for(cur = 0; cur < strlen(curword); cur++)
					{
						*(psource+i+cur-strlen(curword)) = toupper(curword[cur]);
					}
				}
			}
			
			if(*(psource+i) == '\0')
				break;
			
			cur = -1;
			++i;
		}
		
		curword[cur+1] = '\0';
	}
}



gint
apply_rule_move_file_to(gchar *fullfilename, gchar *curdir, gchar *destdir)
{
	gchar fullfilename_new[2048];
	
	if( !strrchr(fullfilename, '/') )
		return 2;
	
// append a slash, if there is none yet.
	if(*(destdir+strlen(destdir)-1) != '/')
		*(destdir+strlen(destdir)) = '/';
	
	snprintf(fullfilename_new, 2047, "%s%s", destdir, strrchr(fullfilename, '/')+1);
	
	if(access(fullfilename_new, F_OK)!=0)
	{
		if( rename(fullfilename, fullfilename_new) == 0 )
		{
			strncpy(curdir, destdir, 2047);
			strncpy(fullfilename, fullfilename_new, 2047);
			return 0;
		}
		else
			return 2;
	}
	else
		return 1;
}
