/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gnome.h>
#include <dirent.h>
#include <sys/stat.h>
#include <ctype.h>
#include "shared.h"
#include "memory.h"
#include "configfile.h"



/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

// TRUE if directory, FALSE if not.
static gboolean
is_dir(gchar *filepathname)
{
	struct stat pstat;
	stat(filepathname, &pstat);
	return(S_ISDIR(pstat.st_mode));
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/








/**********************************************************************
 * This function will compare a wildcardstring with a string.
 * TRUE if match. Otherwise FALSE.
 **********************************************************************/
gboolean
strwccmp(gchar *wcstring, gchar *string, gboolean casess)
{
	gchar *srcwcstring = strdup(wcstring);
	gchar *srcstring = strdup(string);
	gchar *pwcstring = NULL;
	gchar *pstring = NULL;
	gint cur = -1, i = -1;
	gchar *parts[100];
	gboolean match = TRUE;
	
// Only "*" will return TRUE
	if(*wcstring=='*' && *(wcstring+1)=='\0')
	{
		free(srcstring);
		return TRUE;
	}

// if comparison should be case insensitive, we uppercase both strings.
	if(!casess)
	{
		pwcstring = srcwcstring;
		while(*pwcstring!='\0')
		{
			if(isalnum(*pwcstring))
				*pwcstring = toupper(*pwcstring);
			pwcstring++;
		}
		pwcstring = srcstring;
		while(*pwcstring!='\0')
		{
			if(isalnum(*pwcstring))
				*pwcstring = toupper(*pwcstring);
			pwcstring++;
		}
	}
	
	pwcstring = srcwcstring;
// store everything between "*"-wildcards in "parts"
	while( (*pwcstring!='\0') && (cur<100) )
	{
		if(*pwcstring!='*')
			parts[++cur] = calloc(1, 2048);
		while( (*pwcstring!='*') && (*pwcstring!='\0') )
			*( parts[cur]+strlen(parts[cur]) ) = *pwcstring++;
		while( (*pwcstring=='*') && (*pwcstring!='\0') )
			pwcstring++;
	}
	
// the found parts have to occur in "string" in the same order as they were found
	pstring = srcstring;
	
// if the wildcardstring doesn`t start with "*", then the first part has to start exactly at the beginning of "string"
	if( *srcwcstring!='*' && strncmp(parts[0], srcstring, strlen(parts[0]))!=0 )
		return FALSE;
	
	while(++i<=cur)
	{
		if( pstring
			&& ((pstring = strstr(pstring, parts[i])) == NULL) )
			match = FALSE;
	}
// if there is something following after the last part, and the last char in wcstring is not a star: NO MATCH.
	if( pstring
		&& *(strstr(pstring, parts[cur])+strlen(parts[cur]))!='\0'
		&& *(srcwcstring+strlen(srcwcstring)-1)!='*' )
		match = FALSE;
		
	i = -1;
	while(++i<=cur)
		free(parts[i]);
	free(srcwcstring);
	free(srcstring);
	return match;
}





/**********************************************************************
 * returns a list of all files matching the pattern, searching recursive from "dir"
 **********************************************************************/
GList *
get_files_recursive(gchar *dir, gchar *pattern, gboolean casess)
{
	GList *foundfiles = NULL;
	gchar *rowdata = NULL;
	
	struct dirent *pdirent = NULL;
	DIR *pdir = NULL;
	gchar fullfilename[2048];
	
// Open directory
	if((pdir = opendir(dir)) == NULL) 
	{
		printf("200 - Error opening requested path ('%s')!\n", dir);
		exit(1);
	}
	
	while((pdirent = readdir(pdir)) != NULL)
	{
		snprintf(fullfilename, 2047, "%s%s", dir, pdirent->d_name);
		if( is_dir(fullfilename)
		    && strcmp(pdirent->d_name, "..")
		    && strcmp(pdirent->d_name, ".") )
		{
			fullfilename[strlen(fullfilename) + 1] = '\0';
			fullfilename[strlen(fullfilename)] = '/';
			
			if(access(fullfilename, R_OK) == 0)
			{
				GList *subfoundfiles = NULL;
				
				if( (subfoundfiles = get_files_recursive(fullfilename, pattern, casess)) != NULL )
					foundfiles = g_list_concat(foundfiles, subfoundfiles);
			}
		}
		else {
			if( strrchr(fullfilename, '.')
				&& strwccmp(pattern, fullfilename, casess) )
			{
				rowdata = malloc(2048);
				strncpy(rowdata, fullfilename, 2047);
				foundfiles = g_list_append(foundfiles, rowdata);
			}
		}
	}
	closedir(pdir);
	return foundfiles;
}







/**********************************************************************
 * Removes all items from haystack that have a datapointer that is also in needle.
 **********************************************************************/
GList *
g_list_remove_matching(GList *haystack, GList *needle)
{
	GList *haystack_item = NULL;
	GList *needle_item = NULL;
	gpointer haystack_data = NULL;
	gpointer needle_data = NULL;

	haystack_item = g_list_first(haystack);
	while(haystack_item)
	{
		haystack_data = haystack_item->data;
		haystack_item = g_list_next(haystack_item);
		
		needle_item = g_list_first(needle);
		while(needle_item)
		{
			needle_data = needle_item->data;
			needle_item = g_list_next(needle_item);
			if(needle_data == haystack_data)
			{
				haystack = mem_free_1(haystack, haystack_data);
				break;
			}
		}
	}
	return g_list_first(haystack);
}




gboolean
is_cursorkey(gint mychar)
{
	if( mychar == 65288
		|| mychar == 65289
		|| mychar == 65361
		|| mychar == 65363
		|| mychar == 65360
		|| mychar == 65367
		|| mychar == 65293
		|| mychar == 65535
		|| mychar == 65513 )
		return TRUE;
	return FALSE;
}

static gboolean
is_numblockdigit(gint mychar)
{
	if( mychar > 65456
		&& mychar < 65465 )
		return TRUE;
	return FALSE;
}


gboolean
is_valid_filename_char (gint mychar)
{
	gchar allowedchars[] = " !#&\'()+,-.?[]_~";
	gchar *pallowedchars = allowedchars;
	
// The 11041 character used to crash the program, even though i don't know why.
// Well, to be sure, i mask it out (as well as cr and lf)
	if( mychar == 13
		|| mychar == 10
		|| mychar == 11041 )
		return FALSE;
	
	if( isalnum(mychar) )
		return TRUE;
	
	while(*pallowedchars != '\0')
	{
		if( mychar == *pallowedchars )
			return TRUE;
		pallowedchars++;
	}
//	printf("%c: %i\n", mychar, mychar);
	return FALSE;
}



gboolean
entry_allow_filechar_only(GtkObject *entry, gint mychar)
{
	if( is_valid_filename_char(mychar)
		|| is_cursorkey(mychar) )
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}



gboolean
entry_allow_filechar_pattern_only(GtkObject *entry, gint mychar)
{
//	printf("CHAR: %i\n", mychar);
	if( is_valid_filename_char(mychar)
		|| is_cursorkey(mychar)
		|| mychar == '%' )
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}



gboolean
entry_allow_dirchar_only(GtkObject *entry, gint mychar)
{
	if( is_valid_filename_char(mychar)
		|| is_cursorkey(mychar)
		|| mychar == '/' )
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}



gboolean
entry_allow_dirchar_pattern_only(GtkObject *entry, gint mychar)
{
	if( is_valid_filename_char(mychar)
		|| is_cursorkey(mychar)
		|| mychar == '/'
		|| mychar == '%' )
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}



gboolean
entry_allow_digit_only(GtkObject *entry, gint mychar)
{
	if( isdigit(mychar)
		|| is_numblockdigit(mychar)
		|| is_cursorkey(mychar) )
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}



gboolean
entry_allow_all_but_lf(GtkObject *entry, gint mychar)
{
	if(mychar != 65379)
		return FALSE;
// abort event
	gtk_signal_emit_stop_by_name (GTK_OBJECT(entry), "key_press_event");
	return TRUE;
}







// Compares two strings case insensitive
gint
strtoucmp(gchar *string1, gchar *string2)
{
	while( *string1 != '\0' || *string2 != '\0' )
	{
		if( toupper(*string1) != toupper(*string2) )
			return -1;
		string1++;
		string2++;
	}
	return 0;
}


// Compares two strings case insensitive with maxlen len
gint
strntoucmp(gchar *string1, gchar *string2, gint len)
{
	gint i = -1;
	
	while( *string1 != '\0' || *string2 != '\0' )
	{
		if(++i >= len)
			return 0;
		
		if( toupper(*string1) != toupper(*string2) )
			return -1;
		
		string1++;
		string2++;
	}
	return 0;
}
