/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "pluginhandler.h"
using namespace std;

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
/* Constructor.
 */
PluginHandler::PluginHandler(void)
{
}


/* Destructor.
 */
PluginHandler::~PluginHandler(void)
{
#ifdef _DEBUG_
  cout << "PluginHandler destructor: Called.\n";
#endif
  std::map<std::string, Plugin*>::iterator iter;
  for (iter = plugins_str.begin(); iter != plugins_str.end(); iter++) {
    Plugin *plugin = iter->second;
    plugin->unref();
  }
}


/******************************************************************************
 * Public
 ******************************************************************************/
/* Register a new plugin.
 * Returns 0 on success, an errorcode < 0 otherwise.
 */
gint PluginHandler::reg(const gchar *filename, gint priority)
{
  // Allocate space for the plugin.
  Plugin *plugin    = new Plugin;
  
  if (plugin->load(filename) < 0) {
    plugin->unref();
    return -1;
  }
  
  // Register the plugin and emit a signal.
  plugin->set_priority(priority);
  plugins_str[filename] = plugin;
  plugins_int[priority] = plugin;
  
#ifdef _DEBUG_
  cout << "Successfully registered plugin '" << plugin->get_name() << "'\n"
       << "  from '" << filename << "'\n";
#endif
  
  plugin->signal_plugin_deleted.connect(signal_plugin_deleted);
  signal_plugin_loaded.emit(plugin);
  
  return 0;
}


/* Unregister a plugin.
 */
void PluginHandler::unreg(const gchar *filename)
{
  Plugin *plugin = (*plugins_str.find(filename)).second;
  if (!plugin) {
    cerr << "InputPlugin::unreg(): No such plugin (" << filename << ").\n";
    return;
  }
  
  // Unregister the plugin and emit a signal.
  plugins_str.erase(filename);
  plugins_int.erase(plugin->get_priority());
  signal_plugin_removed.emit(plugin);
  plugin->unref();
  
#ifdef _DEBUG_
  cout << "Successfully unregistered plugin '" << plugin->get_name() << "'\n"
       << "  from '" << filename << "'.\n";
#endif
}


/* Return a std::list of all plugins responsible for the given file type.
 * Don't forget to delete the list AND UNREF EVERY PLUGIN!!
 */
std::list<Plugin*> *PluginHandler::get_responsible_plugins(
                                                          const gchar *filename)
{
  std::map<int, Plugin*>::iterator iter;
  std::list<Plugin*> *pluginlist = new std::list<Plugin*>;
  for (iter = plugins_int.begin(); iter != plugins_int.end(); iter++) {
    if (!iter->second->handles(filename))
      continue;
    iter->second->ref();
    pluginlist->push_back(iter->second);
  }
  return pluginlist;
}


/* Given a GList of filenames, this function returns a GList of all responsible
 * plugins for the given filetypes.
 * Don't forget to delete the list AND UNREF EVERY PLUGIN!!
 */
std::list<Plugin*> *PluginHandler::get_responsible_plugins_from_list(
                                                               GList *filenames)
{
  std::map<int, Plugin*>::iterator iter;
  std::map<int, Plugin*> resp_hash;
  std::list<Plugin*> *responsible = new std::list<Plugin*>;
  // Grab a hash_map of all responsible plugins.
  while (filenames) {
    const gchar *filename = (const gchar*)filenames->data;
    for (iter = plugins_int.begin(); iter != plugins_int.end(); iter++) {
      if (resp_hash[iter->first])           // Known to be responsible?
        continue;
      if (!iter->second->handles(filename)) // Responsible?
        continue;
#ifdef _DEBUG_
      std::cout << "PluginHandler::get_responsible_plugins_from_list(): "
                << "Found: ", iter->second->get_name(), ".\n";
#endif
      iter->second->ref();
#ifdef _DEBUG_
      printf("PluginHandler::get_responsible_plugins_from_list(): Ref'd.\n");
#endif
      resp_hash[iter->first] = iter->second;
      responsible->push_back(iter->second);
    }
    filenames = filenames->next;
  }
  
#ifdef _DEBUG_
  printf("PluginHandler::get_responsible_plugins_from_list(): Finished.\n");
#endif
  return(responsible);
}


/* This function returns a list of all plugins. Don't forget to delete the list
 * AND UNREF EVERY PLUGIN!!
 */
std::list<Plugin*> *PluginHandler::get_plugins(void)
{
#ifdef _DEBUG_
  cout << "PluginHandler::get_plugins(): Called.\n";
#endif
  std::map<int, Plugin*>::iterator iter;
  std::list<Plugin*> *pluginlist = new std::list<Plugin*>;
  for (iter = plugins_int.begin(); iter != plugins_int.end(); iter++) {
    iter->second->ref();
    pluginlist->push_back(iter->second);
  }
  return pluginlist;
}


/******************************************************************************
 * Protected
 ******************************************************************************/
