/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "cantushashvalue.h"

//#define _DEBUG_


/******************************************************************************
 * Common for all containers.
 ******************************************************************************/
CantusHashValue *new_value(void)
{
  CantusHashValue *value = (CantusHashValue*)malloc(sizeof(CantusHashValue));
  return value;
}


CantusHashValue *value_duplicate(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  CantusHashValue *copy = NULL;
  switch (hvalue->type) {
  case G_TYPE_INT:
    copy = value_duplicate_int(hvalue);
    break;
   
  case G_TYPE_POINTER:
    copy = value_duplicate_pointer(hvalue);
    break;
   
  case G_TYPE_CHAR:
    copy = value_duplicate_char(hvalue);
    break;
   
  case G_TYPE_BOOLEAN:
    copy = value_duplicate_bool(hvalue);
    break;
  }
}


void value_delete(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type == G_TYPE_INT
    || hvalue->type == G_TYPE_CHAR)
    g_free(hvalue->value);
  g_free(hvalue);
}


int value_get_type(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  return hvalue->type;
}


void value_set(CantusHashValue *hvalue, int type, void *value)
{
  g_assert(hvalue != NULL);
  switch (type) {
  case G_TYPE_INT:
    value_set_int(hvalue, (gint)value);
    break;
  
  case G_TYPE_CHAR:
    g_assert(value != NULL);
    value_set_char(hvalue, (const gchar*)value);
    break;
  
  case G_TYPE_BOOLEAN:
    value_set_bool(hvalue, (gboolean)value);
    break;
  
  case G_TYPE_POINTER:
    value_set_pointer(hvalue, value);
    break;
  
  default:
    g_assert_not_reached();
    break;
  }
}


/******************************************************************************
 * Integer container.
 ******************************************************************************/
void value_set_int(CantusHashValue *hvalue, int value)
{
  g_assert(hvalue != NULL);
  hvalue->type  = G_TYPE_INT;
  hvalue->value = malloc(sizeof(int));
  *(int*)(hvalue->value) = value;
}


int value_get_int(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_INT)
    g_warning("G_TYPE_INT value requested from non-G_TYPE_INT container.\n");
  g_assert(hvalue->value != NULL);
  return *(int*)(hvalue->value);
}


CantusHashValue *value_duplicate_int(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_INT)
    g_warning("G_TYPE_INT value requested from non-G_TYPE_INT container.\n");
  g_assert(hvalue->value != NULL);
  CantusHashValue *copy = new_value();
  copy->type  = hvalue->type;
  copy->value = malloc(sizeof(int));
  *(int*)(copy->value) = *(int*)hvalue->value;
  return copy;
}


/******************************************************************************
 * Bool container.
 ******************************************************************************/
void value_set_bool(CantusHashValue *hvalue, gboolean value)
{
  g_assert(hvalue != NULL);
  hvalue->type  = G_TYPE_BOOLEAN;
  hvalue->value = (void*)value;
}


gboolean value_get_bool(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_BOOLEAN)
    g_warning("G_TYPE_BOOLEAN value requested from non-G_TYPE_BOOLEAN"
              " container.\n");
  return (gboolean)hvalue->value;
}


CantusHashValue *value_duplicate_bool(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_BOOLEAN)
    g_warning("G_TYPE_BOOLEAN value requested from non-G_TYPE_BOOLEAN"
              " container.\n");
  CantusHashValue *copy = new_value();
  copy->type  = hvalue->type;
  copy->value = hvalue->value;
  return copy;
}


/******************************************************************************
 * Pointer container.
 ******************************************************************************/
void value_set_pointer(CantusHashValue *hvalue, void *value)
{
  g_assert(hvalue != NULL);
  hvalue->type  = G_TYPE_POINTER;
  hvalue->value = value;
}


void *value_get_pointer(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_POINTER)
    g_warning("G_TYPE_POINTER value requested from non-G_TYPE_POINTER"
              " container.\n");
  return hvalue->value;
}


CantusHashValue *value_duplicate_pointer(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_POINTER)
    g_warning("G_TYPE_POINTER value requested from non-G_TYPE_POINTER"
              " container.\n");
  g_assert(hvalue->value != NULL);
  CantusHashValue *copy = new_value();
  copy->type  = hvalue->type;
  copy->value = hvalue->value;
  return copy;
}


/******************************************************************************
 * Char container.
 ******************************************************************************/
void value_set_char(CantusHashValue *hvalue, const gchar *value)
{
  g_assert(hvalue != NULL);
  hvalue->type  = G_TYPE_CHAR;
  hvalue->value = value ? (void*)g_strdup(value) : NULL;
}


const gchar *value_get_char(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
  if (hvalue->type != G_TYPE_CHAR)
    g_warning("G_TYPE_CHAR value requested from non-G_TYPE_CHAR container.\n");
  return hvalue->value ? (const gchar*)hvalue->value : "";
}


CantusHashValue *value_duplicate_char(CantusHashValue *hvalue)
{
  g_assert(hvalue != NULL);
#ifdef _DEBUG_
  printf("value_duplicate_char(): Source: %s\n", hvalue->value);
#endif
  if (hvalue->type != G_TYPE_CHAR)
    g_warning("G_TYPE_CHAR value requested from non-G_TYPE_CHAR container.\n");
  CantusHashValue *copy = new_value();
  copy->type  = hvalue->type;
  copy->value = hvalue->value ? (void*)g_strdup((gchar *)hvalue->value) : NULL;
#ifdef _DEBUG_
  printf("value_duplicate_char(): Copy: %s\n", copy->value);
#endif
  return copy;
}
