/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "displayarea.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
Displayarea::Displayarea(CantusHash *pplugindata)
{
  plugindata = pplugindata;
  CantusAddListenerSigCFunc addlistener = (CantusAddListenerSigCFunc)
                  cantushash_get_pointer(plugindata, "Cantus:AddListenerSigC");
  g_return_if_fail(addlistener != NULL);
  listenerids.push_back(
    addlistener("Filelist:Read:Start",
              SigC::slot(*this, &Displayarea::on_selection_changed_event)));
  listenerids.push_back(
    addlistener("File:Read:Finished",
              SigC::slot(*this, &Displayarea::on_file_read_finished_event)));
  listenerids.push_back(
    addlistener("GUI:PluginWidget:Destroyed",
              SigC::slot(*this, &Displayarea::on_uiwidget_destroyed_event)));
}


Displayarea::~Displayarea(void)
{
#ifdef _DEBUG_
  printf("mpegheader(): Displayarea::~Displayarea(): Called.\n");
#endif
  CantusRemoveListenerFunc removelistener = (CantusRemoveListenerFunc)
                  cantushash_get_pointer(plugindata, "Cantus:RemoveListener");
  g_return_if_fail(removelistener != NULL);
  std::list<long>::iterator iter = listenerids.begin();
  while (iter != listenerids.end()) {
    removelistener(*iter);
    iter++;
  }
}


/******************************************************************************
 * Public
 ******************************************************************************/
GtkWidget *Displayarea::build(void)
{
  Gtk::Table *table = new Gtk::Table(1, 1);
  table->set_border_width(5);
  
  label = new Gtk::Label(_("None selected."), 0, 0);
  //label->set_padding(5, 5);
  table->attach(*label, 0, 1, 0, 1, Gtk::FILL, Gtk::FILL, 0, 0);
  
  table->show_all();
  
  return (GtkWidget*)Glib::unwrap(table);
}


/******************************************************************************
 * Private
 ******************************************************************************/
/* The selection changed, but the selected files have not yet been read. */
void Displayarea::on_selection_changed_event(void *pselected)
{
#ifdef _DEBUG_
  std::cout << "mpegheader(): Displayarea::on_selection_changed_event(): "
            << "Called.\n";
#endif
  gint     numitems = pselected ? g_list_length((GList*)pselected) : 0;
  GString *summary  = g_string_sized_new(1024);
  gchar   *text     = NULL;
  
  switch (numitems) {
  case 0:
    label->set_text(_("None selected."));  // Update the GUI.
    break;
  
  case 1:
    update = TRUE;
    break;
  
  default:
    g_string_sprintf(summary, ngettext("%i File selected.",
                                       "%i Files selected.",
                                       numitems), numitems);
    text = summary->str;                   // Remove the last newline char.
    *(text + strlen(text) - 1) = '\0';
    label->set_text(text);                 // Update the GUI.
    update = FALSE;
    break;
  }
  
  g_string_free(summary, TRUE);
}


/* A file has been read successfully. */
void Displayarea::on_file_read_finished_event(void *pinfo)
{
#ifdef _DEBUG_
  std::cout << "mpegheader(): Displayarea::on_file_read_finished_event(): "
            << "Called.\n";
#endif
  if (!update)
    return;
  CantusHash  *info    = (CantusHash*)pinfo;
  GString     *summary = g_string_sized_new(1024);
  gchar       *text    = NULL;
  const gchar *mode    = NULL;
  gint         seconds, mins, secs;
  gint         mode_int;
  
  g_string_sprintf(summary,
                   _("MPEG V%s, Layer %i\n"
                     "<b>Sample rate:</b> %i\n"
                     "<b>Bitrate:</b> %i\n"),
                   cantushash_get_char(info, "MPEGHeader:Version"),
                   cantushash_get_int(info, "MPEGHeader:Layer"),
                   cantushash_get_int(info, "MPEGHeader:Samplerate"),
                   cantushash_get_int(info, "MPEGHeader:Bitrate"));
  
  // Mode:
  mode_int = cantushash_get_int(info, "MPEGHeader:Mode");
  switch (mode_int) {
  case 0:
    mode = _("Stereo");
    break;
  case 1:
    mode = _("Joint Stereo");
    break;
  case 2:
    mode = _("2 Channel");
    break;
  case 3:
    mode = _("Mono");
    break;
  default:
    g_assert_not_reached();
  }
  
  g_string_sprintfa(summary, _("<b>Mode:</b> %s\n"), mode);
  
  // Time:
  seconds = cantushash_get_int(info, "MPEGHeader:Seconds");
  mins    = seconds / 60;
  secs    = seconds % 60;
  
  g_string_sprintfa(summary, _("<b>Time:</b> %i:%02i\n"), mins, secs);
  
  text = summary->str;                // Remove the last newline char.
  *(text + strlen(text) - 1) = '\0';
  label->set_markup(text);              // Update the GUI.
  g_string_free(summary, TRUE);
}


void Displayarea::on_uiwidget_destroyed_event(void *name)
{
#ifdef _DEBUG_
  printf("id3v2(): Displayarea::on_uiwidget_destroyed_event(): Called.\n");
#endif
  if (strcmp((gchar*)name, cantushash_get_char(plugindata, "Plugin:Name")) == 0)
    delete this;
}
