#!/usr/bin/env python

# Copyright (C) 2006 Christian Dywan <software at twotoasts dot de>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys

try:
    import pygtk
    pygtk.require('2.0')
    import gtk
except:
    print "Error: It appears that pyGTK2 is missing."
    print "Please install it to run this program."
    sys.exit()

try:
    import gtk.glade
except:
    print "Error: It appears that Glade-support is missing."
    print "Please install a version of Gtk+ with Glade-support."
    sys.exit()

try:
    import xdg.Mime
except:
    print "Warning: The module xdg was not found. So file icons will be disabled."
    print "It is recommended to install python-xdg."

import os
import os.path
from optparse import OptionParser
import stat
import time
import gobject
import md5
import subprocess
import fnmatch

app_version = '0.1'

class catfish:
    def __init__(self):
        """Create the main window."""

        # Guess default fileman
        if os.environ.get('DESKTOP_SESSION', os.environ.get('GDMSESSION', ''))[:4] == 'xfce':
            default_fileman = 'Thunar'
            self.open_wrapper = 'exo-open'
        else:
            default_fileman = 'Nautilus'
            # Guess suitable file open wrapper
            for path in os.environ.get('PATH', '/usr/bin').split(os.pathsep):
                for wrapper in ['gnome-open', 'exo-open', 'xdg-open']:
                    if os.path.exists(os.path.join(path, wrapper)):
                        self.open_wrapper = wrapper
                        break

        # Parse command line options
        parser = OptionParser(usage='usage: catfish [options] keywords',
            version='catfish v' + app_version)
        parser.add_option('', '--large-icons', action='store_true', dest='icons_large', help='Use large icons')
        parser.add_option('', '--thumbnails', action='store_true', dest='thumbnails', help='Use thumbnails')
        parser.add_option('', '--iso-time', action='store_true', dest='time_iso', help='Display time in iso format')
        parser.add_option('', '--limit', type='int', metavar='LIMIT', dest='limit_results',
            help='Limit number of results')
        parser.add_option('', '--path', help='Search in folder PATH')
        parser.add_option('', '--fileman', help='Use FILEMAN as filemanager')
        parser.add_option('', '--wrapper', metavar='WRAPPER', dest='open_wrapper', help='Use WRAPPER to open files')
        parser.set_defaults(icons_large=0, thumbnails=0, time_iso=0,
            limit_results=0, path=os.path.expanduser('~'), fileman=default_fileman)
        self.options, args = parser.parse_args()
        keywords = ''
        for arg in args:
            keywords = keywords + arg
            if arg <> args[len(args) - 1]:
                keywords = keywords + ' '

        # Guess location of glade file
        glade_file = 'catfish.glade'
        glade_path = os.getcwd()
        if not os.path.exists(os.path.join(glade_path, glade_file)):
            glade_path = os.path.dirname(sys.argv[0])
            if not os.path.exists(os.path.join(glade_path, glade_file)):
                for path in os.environ.get('XDG_DATA_DIRS').split(os.pathsep):
                    if os.path.exists(os.path.join(path, glade_file)):
                        glade_path = path
                        break

        # Load interface from glade file
        try:
            self.xml = gtk.glade.XML(os.path.join(glade_path, glade_file))
            self.xml.signal_autoconnect(self)
        except:
            print "Error:"
            print "The glade file could not be found."
            sys.exit()

        # Prepare significant widgets
        self.window_search = self.xml.get_widget('window_search')
        self.entry_find_text = self.xml.get_widget('entry_find_text')
        self.checkbox_find_exact = self.xml.get_widget('checkbox_find_exact')
        self.checkbox_find_hidden = self.xml.get_widget('checkbox_find_hidden')
        self.checkbox_find_limit = self.xml.get_widget('checkbox_find_limit')
        self.spin_find_limit = self.xml.get_widget('spin_find_limit')
        self.combobox_find_method = self.xml.get_widget('combobox_find_method')
        self.button_find_folder = self.xml.get_widget('button_find_folder')
        self.button_find = self.xml.get_widget('button_find')
        self.button_findbar_hide = self.xml.get_widget('button_findbar_hide')
        self.scrolled_findbar = self.xml.get_widget('scrolled_findbar')
        self.treeview_files = self.xml.get_widget('treeview_files')
        self.menu_file = self.xml.get_widget('menu_file')
        self.statusbar = self.xml.get_widget('statusbar')

        # Retrieve available search methods
        listmodel = gtk.ListStore(gobject.TYPE_STRING)
        methods = ('find', 'locate', 'slocate', 'tracker-search', 'beagle-query')
        bin_dirs = os.environ.get('PATH', '/usr/bin').split(os.pathsep)
        for method in methods:
            for path in bin_dirs:
                if os.path.exists(os.path.join(path, method)):
                    if not os.path.islink(os.path.join(path, method)):
                        listmodel.append([method])
                        break
        self.combobox_find_method.set_model(listmodel)
        self.combobox_find_method.set_text_column(0)
        self.combobox_find_method.child.set_text(listmodel[0][0])
        self.combobox_find_method.child.set_editable(0)

        # Set some initial values
        self.icon_cache = {}
        self.icon_theme = gtk.icon_theme_get_default()
        if self.options.limit_results:
            self.checkbox_find_limit.set_active(1)
            self.checkbox_find_limit.toggled()
            self.spin_find_limit.set_value(self.options.limit_results)
        self.folder_thumbnails = os.path.expanduser('~/.thumbnails/normal/')
        self.button_find_folder.set_filename(self.options.path)

        # Initialize treeview
        self.treeview_files.append_column(gtk.TreeViewColumn('Icon', gtk.CellRendererPixbuf(), pixbuf=0))
        self.treeview_files.append_column(self.new_column('Filename', 1))
        if not self.options.icons_large and not self.options.thumbnails:
            self.treeview_files.append_column(self.new_column('Size', 2, 'filesize'))
            self.treeview_files.append_column(self.new_column('Last modified', 3))
            self.treeview_files.append_column(self.new_column('Location', 4))

        self.entry_find_text.set_text(keywords)
        self.button_find.activate()

# -- helper functions --

    def new_column(self, label, id, special=None):
        cell_renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(label, cell_renderer, text=id)
        if special == 'filesize':
            column.set_cell_data_func(cell_renderer, self.cell_data_func_filesize, id)
        column.set_sort_column_id(id)
        column.set_resizable(1)
        return column

    def cell_data_func_filesize(self, column, cell_renderer, tree_model, iter, id):
        filesize = self.format_size(int(tree_model.get_value(iter, id)))
        cell_renderer.set_property('text', filesize)
        return

    def format_size(self, size):
        if size > 2 ** 30:
            return "%s GB" % (size / 2 ** 30)
        elif size > 2 ** 20:
            return "%s MB" % (size / 2 ** 20)
        elif size > 2 ** 10:
            return "%s kB" % (size / 2 ** 10)
        elif size > -1:
            return "%s B" % size
        else:
            return ""

    def get_selected_filename(self):
        model = self.treeview_files.get_model()
        path = self.treeview_files.get_cursor()[0]
        if not self.options.icons_large and not self.options.thumbnails:
            return model.get_value(model.get_iter(path), 4), model.get_value(model.get_iter(path), 1)
        return model.get_value(model.get_iter(path), 4), model.get_value(model.get_iter(path), 3)

    def open_file(self, filename):
        """Open the file with its default app or the file manager"""

        if stat.S_ISDIR(os.stat(filename).st_mode):
            command = '%s "%s"' % (self.options.fileman, filename)
        else:
            command = '%s "%s"' % (self.open_wrapper, filename)
        try:
            subprocess.Popen(command, shell=True)
        except:
            print 'Error: Could not open ' + filename + '.'
            print 'Note:  The wrapper was ' + self.open_wrapper + '.'

    def string_wild_match(self, string, keyword, exact):
        return fnmatch.fnmatch(string.lower(), '*' + keyword.lower() + '*')

    def find(self):
        """Do the actual search."""

        self.button_find.set_label('gtk-cancel')
        self.window_search.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        self.window_search.set_title('Searching for "%s"' % self.entry_find_text.get_text())
        self.statusbar.push(self.statusbar.get_context_id('results'), 'Searching...')
        while gtk.events_pending(): gtk.main_iteration()

        # Reset treeview
        listmodel = gtk.ListStore(gtk.gdk.Pixbuf, gobject.TYPE_STRING,
            gobject.TYPE_INT, gobject.TYPE_STRING, gobject.TYPE_STRING)
        self.treeview_files.set_model(listmodel)
        self.treeview_files.columns_autosize()

        # Retrieve search parameters
        keyword = self.entry_find_text.get_text().replace(' ', '*')
        method = self.combobox_find_method.child.get_text()
        folder = self.button_find_folder.get_filename()
        exact = self.checkbox_find_exact.get_active()
        hidden = self.checkbox_find_hidden.get_active()
        if self.checkbox_find_limit.get_active():
            limit = self.spin_find_limit.get_value()
        else:
            limit = -1

        # Generate search command
        if keyword != '':
            command = method
            path_check = 1
            if method == 'find':
                command = '%s %s -ignore_readdir_race -noleaf' % (command, folder)
                if exact:
                    command = '%s -iwholename' % command
                else:
                    command = '%s -iwholename' % command
                command = '%s "*%s*"' % (command, keyword)
                path_check = 0
            elif method in ('locate', 'slocate'):
                if not exact:
                    command += ' -i'
                if limit and hidden:
                    command += ' -n %s' % int(limit)
                command = '%s "%s"' % (command, keyword)
            elif method == 'tracker-search' and limit and hidden:
                command = '%s -l %s "%s"' % (command, limit, keyword)
            elif method == 'beagle-query' and limit and hidden:
                command = '%s --max-hits=%s "%s"' % (command, limit, keyword)
            else:
                command = '%s "%s"' % (command, keyword)

            # Set display options
            if not self.options.icons_large and not self.options.thumbnails:
                icon_size = 24
            else:
                icon_size = 0
            if not self.options.time_iso:
                time_format = '%x %X'
            else:
                time_format = '%Y-%m-%w %H:%M %Z'

            # Run search command and capture the results
            search_process = subprocess.Popen(command, stdout=subprocess.PIPE, bufsize=1, shell=True)
            for filename in search_process.stdout:
                while gtk.events_pending(): gtk.main_iteration()
                if self.abort_find or len(listmodel) == limit:
                    break
                filename = filename.split(os.linesep)[0]
                if filename[:7] == 'file://':
                    filename = filename[7:]
                path, name = os.path.split(filename)
                try:
                    if path_check:
                        path_valid = self.string_wild_match(name, keyword, exact)
                    else:
                        path_valid = 1
                    if path_valid  and (not self.file_is_hidden(filename) or hidden):
                        if self.options.thumbnails:
                            icon = self.get_thumbnail(filename, icon_size)
                        else:
                            icon = self.get_file_icon(filename, icon_size)
                        filestat = os.stat(filename)
                        size = filestat.st_size
                        modified = time.strftime(time_format, time.gmtime(filestat.st_mtime))
                        if not self.options.icons_large and not self.options.thumbnails:
                            listmodel.append([icon, name, size, modified, path])
                        else:
                            listmodel.append([icon, '%s (%s) %s%s%s%s'
                                % (name, size, os.linesep, modified, os.linesep, path), -1, name, path])
                except:
                    pass # ignore inaccessible or outdated files
                self.treeview_files.set_model(listmodel)
                yield True
            if len(listmodel) == 0:
                if search_process.poll() <> 0 and method != 'find':
                    message = 'Fatal error, search was aborted.'
                else:
                    message = 'No files were found.'
                listmodel.append([None, message, -1, '', ''])
                status = 'No files found.'
            else:
                status = '%s files found.' % len(listmodel)
            self.statusbar.push(self.statusbar.get_context_id('results'), status)
        self.treeview_files.set_model(listmodel)

        self.window_search.window.set_cursor(None)
        self.button_find.set_label('gtk-find')
        yield False

    def file_is_hidden(self, filename):
        """Determine if a file is hidden or in a hidden folder"""

        if filename == '': return 0
        path, name = os.path.split(filename)
        if name[0] == '.':
            return 1
        for folder in path.split(os.path.sep):
            if len(folder):
                if folder[0] == '.':
                    return 1
        return 0

    def get_icon_pixbuf(self, name, icon_size=0):
        try:
            return self.icon_cache[name]
        except KeyError:
            if icon_size == 0:
                icon_size = 48
            icon = self.icon_theme.load_icon(name, icon_size, 0)
            self.icon_cache[name] = icon
            return icon

    def get_thumbnail(self, path, icon_size=0, use_mime=1):
        """Try to fetch a small thumbnail."""

        filename = '%s%s.png' % (self.folder_thumbnails, md5.new('file://' + path).hexdigest())
        try:
            return gtk.gdk.pixbuf_new_from_file(filename)
        except:
            return self.get_file_icon(path, icon_size, use_mime)

    def get_file_icon(self, path, icon_size=0 ,use_mime=1):
        """Retrieve the file icon."""

        try:
            is_folder = stat.S_ISDIR(os.stat(path).st_mode)
        except:
            is_folder = 0
        if is_folder:
            icon_name='folder'
        else:
            if use_mime:
                try:
                    # Get icon from mimetype
                    mime = xdg.Mime.get_type(path)
                    icon_name = 'gnome-mime-%s-%s' % (mime.media, mime.subtype)
                    return self.get_icon_pixbuf(icon_name, icon_size)
                except:
                    try:
                        # Then try generic icon
                        icon_name = '%s-x-generic' % mime.media
                        return self.get_icon_pixbuf(icon_name, icon_size)
                    except:
                        # Use default icon
                        icon_name = 'gnome-fs-regular'
            else:
                icon_name = 'gnome-fs-regular'
        return self.get_icon_pixbuf(icon_name, icon_size)

# -- events --
    def on_window_search_destroy(self, window):
        gtk.main_quit()

    def on_button_close_clicked(self, button):
        self.window_search.destroy()

    def on_checkbox_find_limit_toggled(self, checkbox):
        self.spin_find_limit.set_sensitive(checkbox.get_active())

    def on_combobox_find_method_changed(self, combobox):
        self.checkbox_find_exact.set_sensitive(
            self.combobox_find_method.child.get_text() in ('find', 'locate', 'slocate'))
        self.button_find_folder.set_sensitive(
            self.combobox_find_method.child.get_text() == 'find')

    def on_button_find_clicked(self, button):
        """Initiate the search thread."""

        if self.button_find.get_label() == 'gtk-find' and self.entry_find_text.get_text() <> '':
            self.abort_find = 0
            task = self.find()
            gobject.idle_add(task.next)
        else:
            self.abort_find = 1

    def on_treeview_files_row_activated(self, treeview, path, view_column):
        folder, filename = self.get_selected_filename()
        filename = os.path.join(folder, filename)
        self.open_file(filename)

    def on_treeview_files_button_pressed(self, treeview, event):        
        if event.button == 3:
            self.menu_file.popup(None, None, None, event.button, event.time)

    def on_menu_open_activate(self, menu):
        folder, filename = self.get_selected_filename()
        self.open_file(os.path.join(folder, filename))

    def on_menu_goto_activate(self, menu):
        folder, filename = self.get_selected_filename()
        self.open_file(folder)

catfish()
gtk.main()