/*
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include 	<stdio.h>
#include 	<unistd.h>
#include 	<stdlib.h>
#include 	<stdarg.h>
#include	<string.h>

#include 	<defs.h>
#include 	<vc.h>
#include        <vga.h>
#include 	<pinyin.h>
#include 	<hzinput.h>
#include	<term.h>
#include 	<font.h>

static hz_input_table* LoadInputMethod(char *filename);
static void  DisplaySelection(void);

/***************************************************************************
 *                           variable defines                              *
 ***************************************************************************/

#define VERSION_STRING    "CCEĻ" CCE_VER ""
unsigned char	*half_full_str[] = { "ǡ" , "ȫǡ" };
u_char *CurCoding[] = { "GBK " , "Big5" };
//unsigned char  *history_normal_str[] = {"롿","ʷ"};

unsigned char	fullchar[]	 =
	"磥"
	"£ãģţƣǣȣɣʣˣ̣ͣΣϣУѣңӣԣգ֣ףأ٣ڡܡޡ"
	"  ";

 int IsHanziInput;   /* 0 for ascii, 1 for hanzi input */
 int IsFullChar;     /* 0 for half char, 1 for full char */
 int current_method;
 hz_input_table *input_table[NR_INPUTMETHOD],*cur_table = NULL;
 char seltab[16][MAX_PHRASE_LENGTH];

 int CurSelNum=0;   /* Current Total Selection Number */
 unsigned long InpKey[MAX_INPUT_LENGTH],save_InpKey[MAX_INPUT_LENGTH];
   /* Input key buffer */

 int InputCount,InputMatch, StartKey,EndKey;
 int save_StartKey,save_EndKey, save_MultiPageMode, 
           save_NextPageIndex, save_CurrentPageIndex;

 int NextPageIndex,CurrentPageIndex,MultiPageMode;
/* When enter MultiPageMode:
   StartKey .... CurrentPageIndex .... NextPageIndex .... EndKey 
*/

 unsigned long val1, val2,key1,key2;
 int IsAssociateMode;
 int CharIndex[15];   // starting index of matched char/phrases

char *tabfname[10] = { NULL, "pinyin",/* "shuangpin",*/ 
                       "wubi" /*,"ziranma" */};
int UseAssociateMode = 0;

/* 6 bit a key mask */
 unsigned long mask[]=
{
  0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
  0x3F000000, 0x3FFC0000, 0x3FFFF000, 0x3FFFFFC0, 0x3FFFFFFF, 0x3FFFFFFF,
  0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF
};


static int ConfigInputMethod(const char *str)
{
        char *p, fn[128];
        int i = 1;

        message("Load input method 0 & 9: PinYin & Internal Code\r\n");
        p = strtok((char *)str, "\n");
        while(p && i < NR_INPUTMETHOD) 
       {
                while (*p == ' ' || *p == '\t') p++;

                strcpy(fn,p);
                if (access(fn, R_OK) == 0)
                {
                    message("Load input method %d: %s\r\n",i,p);
                    input_table[i++] = LoadInputMethod(fn);
                }
                else
                    input_table[i++] = NULL;
                p = strtok((char *)NULL, "\n");
        }
        return SUCCESS;
}

/***************************************************************************
 *                              public function                            *
 ***************************************************************************/

static hz_input_table* LoadInputMethod(char *filename)
{
  int  nread;
  FILE *fd;
  char phrase_filename[100],assoc_filename[100];
  hz_input_table *table;
  
  table = malloc(sizeof(hz_input_table));
  if (table == NULL)
    error("Out of memory in LoadInputMethod");
  fd = fopen(filename, "r");
  if (fd == NULL)
  {
    error("Cannot open input method %s", filename); 
    fclose(fd);
    free(table);
    return NULL;
  }

  nread = fread(table, sizeof(hz_input_table),1,fd);
  if (nread != 1)
  {
    error("Cannot read file header %s", filename);
    return NULL;
  }

  if( strcmp(MAGIC_NUMBER, table->magic_number) )
  {
        printf("is not a valid tab file\n\n");
        return NULL;
  }

  table->item = (ITEM *)malloc(sizeof(ITEM ) * table->TotalChar); 
  //warn("Totalchar=%d\n",table->TotalChar);

  if ( table->item == NULL ) 
  {
        error("Gosh, cannot malloc enough memory");
        return NULL;
  }
  
  fread(table->item, sizeof(ITEM) , table->TotalChar,fd );
  fclose( fd );
 
  if (table->PhraseNum > 0)
  {
     strcpy( phrase_filename, filename );
     strcat( phrase_filename, ".phr" );
     strcpy( assoc_filename, filename );
     strcat( assoc_filename, ".lx");
     
     table->PhraseFile = fopen( phrase_filename, "r" );
     table->AssocFile = fopen( assoc_filename, "r");

     if (table->PhraseFile == NULL || table->AssocFile == NULL) 
     {
          printf("Load Phrase/Assoc File error!\n");
          free(table);
          return NULL;
     }
  }
  else
  {
      table->PhraseFile = table->AssocFile = NULL;
  }

  return table;
}

void UnloadInputMethod(int i)
{
   if (input_table[i] == NULL) return;

   if (i == current_method)
   {
      if (IsHanziInput)
        ToggleInputMethod();
      current_method = 0;
    }

    if (input_table[i]->PhraseFile)
        fclose(input_table[i]->PhraseFile);
    if (input_table[i]->AssocFile)
        fclose(input_table[i]->AssocFile);

     //free(input_table[i]->item);
     free(input_table[i]);
     input_table[i] = NULL;

}

 void ResetInput()
{
  bzero( InpKey,   sizeof( InpKey   ) );
  bzero( seltab, sizeof( seltab ) );

  MultiPageMode = NextPageIndex = CurrentPageIndex = 0;
  CurSelNum = InputCount = InputMatch = 0;
  IsAssociateMode = 0;   /* lian xiang */
}

 void DisplayInput(void)
{
  int i,pos = InputAreaX, len = (current_method == 9? 4 : MAX_INPUT_LENGTH);
  char c;

  for( i = 0; i <= len ; i++)
  {
      if (i < InputCount) c = input_table[current_method]->KeyName[InpKey[i]];
      else c=' ';
      if (i == InputMatch && InputCount > InputMatch && i != 0) 
           pVideoInfo->input_sput(pos++, '-', INPUT_FGCOLOR,INPUT_BGCOLOR);
      
      pVideoInfo->input_sput(pos++, c, INPUT_FGCOLOR,INPUT_BGCOLOR);
  }
}

inline void ClearSelectionArea()
{
    int i, pos = (current_method == 9? IntCode_SelectionX: SelectionX);
    
    for(i = pos; i <= 79 /* SelectionXMax */; i++)
           // hard code here!!!!
	pVideoInfo->input_sput(i,' ',INPUT_BGCOLOR,INPUT_BGCOLOR);
}

inline void ClearInputArea()
{
   int i,pos = (current_method == 9? IntCode_SelectionX: SelectionX);

   for(i = InputAreaX; i < pos; i++)
      pVideoInfo->input_sput(i,' ',INPUT_BGCOLOR,INPUT_BGCOLOR);
}

void FindAssociateKey(int index)
{
    FILE *fp = cur_table->AssocFile;
    int ofs[2],offset;
   
    if (index < 0xB0A1) 
    {
       StartKey = EndKey = 0;
       return;  /* no match */
    }  
   
    offset = (index / 256 - 0xB0) * 94 + index % 256 - 0xA1; 
    fseek(fp, offset * sizeof(int), SEEK_SET);
    fread(ofs,sizeof(int),2,fp);
    StartKey = 72 * 94 + 1 + ofs[0];
    EndKey = 72 * 94 + 1 + ofs[1];
}

/* phrno: Phrase Number, return in tt */
 void LoadPhrase( int phrno, char *tt )
{
  FILE *fp = cur_table->PhraseFile;
  int ofs[2], len;

  fseek( fp, ( phrno + 1 ) * 4, SEEK_SET );
  fread( ofs, 4, 2, fp );
  len = ofs[1] - ofs[0];

  if ( len > 128 || len <= 0 ) {
     error( "phrase error %d\n" , len );
     strcpy( tt, "error" );
     return;
  }

  ofs[0] += ( cur_table->PhraseNum + 1 ) * 4;
   /* Add the index area length */
  fseek( fp, ofs[0], SEEK_SET );
  fread( tt, 1, len, fp );
  tt[len] = 0;
}

void putstr(int tty_fd, unsigned char *p )
{
    int index,len = strlen(p);
    write(tty_fd, p, len);

    ClearSelectionArea();
    ClearInputArea();

    if (InputCount <= InputMatch)  /* All Match */
    {
        index = (int)p[len-2] * 256 + p[len-1]; 
        ResetInput();
        if (UseAssociateMode)
        {
             FindAssociateKey(index); 
             CurrentPageIndex = StartKey;
             MultiPageMode = 0;
             FillAssociateChars(StartKey);  
             if (CurSelNum > 0)
             {
                IsAssociateMode = 1;
                DisplaySelection();
             }
        }
    }
    else
    {
      int nCount = InputCount - InputMatch,nMatch = InputMatch,i;
      MultiPageMode = NextPageIndex = CurrentPageIndex = 0;
      InputCount = InputMatch = 0;

      for(i = 0; i < nCount; i++)
        save_InpKey[i] = InpKey[nMatch+i];

      bzero(InpKey, sizeof(InpKey));
      for(i = 1; i <= nCount; i++)  /* feed the additional keys */
      {
         InpKey[InputCount] = save_InpKey[InputCount++];
         if (InputCount <= InputMatch+1)
         {
             FindMatchKey();
             MultiPageMode = 0;
             CurrentPageIndex = StartKey;
             FillMatchChars(StartKey);
         } 
      }
      if (InputMatch == 0)    /* left key has no match, delete */
      {
         ResetInput(); 
         return;
      }

      DisplaySelection();
      DisplayInput();
    }
}

void outchar(int tty_fd,unsigned char c)
{
  int key;
  if (IsFullChar && c >= ' ' && c <= 127)
  {
       key = (c - ' ') << 1;
       write(tty_fd, fullchar+key, 1);
       write(tty_fd, fullchar+key+1,1);
  }
  else write(tty_fd,&c,1);

}

/* After add/delete a char, search the matched char/phrase, update the
   StartKey/EndKey key,  save the related keys at first, if no match
   is found, we may restore its original value
*/
void FindMatchKey(void)
{
  save_StartKey = StartKey;
  save_EndKey = EndKey;
  save_MultiPageMode = MultiPageMode;
  save_NextPageIndex = NextPageIndex;
  save_CurrentPageIndex = CurrentPageIndex;

  val1 = InpKey[4] | (InpKey[3]<<6) | (InpKey[2]<<12) | (InpKey[1]<<18) | 
        (InpKey[0]<<24);
  val2 = InpKey[9] | (InpKey[8]<<6) | (InpKey[7]<<12) | (InpKey[6]<<18) | 
        (InpKey[5]<<24);

  if (InputCount == 1)
     StartKey = cur_table->KeyIndex[InpKey[0]];
  else
     StartKey = CharIndex[InputCount-1];

  EndKey = cur_table->KeyIndex[InpKey[0]+1];

  for (; StartKey < EndKey; StartKey++)
  {
      key1 = (cur_table->item[StartKey].key1 & mask[InputCount+5]);
      key2 = (cur_table->item[StartKey].key2 & mask[InputCount]);

      if (key1 > val1) break;
      if (key1 < val1) continue;
      if (key2 < val2) continue;
      break;
  }
  CharIndex[InputCount] = StartKey;
}

/*  Find the matched chars/phrases and fill it into SelTab
    The starting checked index is j 
 
    The Selection Line 1xxx 2xxx,  80-20=60 60/2=30 chinese chars only
    0-9 Selection can contain only 30 chinese chars

    SelectionXMax-SelectionX-20
*/

#define MAX_SEL_LENGTH (SelectionXMax-SelectionX-18)

void FillAssociateChars(int index)
{
    unsigned char str[25];
    int PhraseNo, CurLen = 0;
   
    CurSelNum = 0; 
    while( CurSelNum < cur_table->MaxDupSel && index < EndKey &&
              CurLen < MAX_SEL_LENGTH)
    {
         fseek( cur_table->AssocFile, index << 2, SEEK_SET );
         fread( &PhraseNo, sizeof(int), 1, cur_table->AssocFile );
         LoadPhrase( PhraseNo, str );
         strcpy(seltab[CurSelNum],str+2);
         CurLen += strlen(seltab[CurSelNum++]);
         index++;
    }

    /* check if more than one page */
    if ( index < EndKey && CurSelNum == cur_table->MaxDupSel )
    {
      /* has another matched key, so enter MultiPageMode, has more pages */
        NextPageIndex = index;
        MultiPageMode = 1;
    }
    else if (MultiPageMode)
    {
        NextPageIndex = StartKey; /* rotate selection */
    }
    else MultiPageMode = 0;
}

void FillMatchChars(int j)
{
    int SelNum = 0, CurLen = 0;
    //bzero( seltab, sizeof( seltab ) );

    while( ( cur_table->item[j].key1 & mask[InputCount+5] ) == val1 &&
            ( cur_table->item[j].key2 & mask[InputCount] ) == val2 &&
              SelNum < cur_table->MaxDupSel && j < EndKey && 
              CurLen < MAX_SEL_LENGTH)
    {
          if ( cur_table->item[j].ch < 0xA1A1 )
              LoadPhrase( cur_table->item[j].ch, seltab[SelNum] );
          else
          {
              memcpy( &seltab[SelNum], &(cur_table->item[j].ch),2);
              seltab[SelNum][2] = '\0';
          }
          CurLen += strlen(seltab[SelNum++]);
          j++;
    }
    
    if (SelNum == 0)  /* some match found */
    {
        StartKey = save_StartKey;
        EndKey = save_EndKey;
        MultiPageMode = save_MultiPageMode;
        NextPageIndex = save_NextPageIndex;
        CurrentPageIndex = save_CurrentPageIndex;
        return;    /* keep the original selection */
    }

    CurSelNum = SelNum;

    for(SelNum = CurSelNum; SelNum < 16; SelNum++)
       seltab[SelNum][0] = '\0';  /* zero out the unused area */
    InputMatch = InputCount; /* until now we have some matches */

    /* check if more than one page */
    if ( j < EndKey && (cur_table->item[j].key1 & mask[InputCount+5]) == val1 &&
         ( cur_table->item[j].key2 & mask[InputCount] ) == val2 &&
          CurSelNum == cur_table->MaxDupSel )
    {
      /* has another matched key, so enter MultiPageMode, has more pages */
        NextPageIndex = j;
        MultiPageMode = 1;
    }  
    else if (MultiPageMode)
    {
        NextPageIndex = StartKey; /* rotate selection */
    }
    else MultiPageMode = 0;
}

/* return value: characters output to tty_fd */
void HZFilter(int tty_fd, unsigned char key)
{
  int inkey = 0,vv;
  char *is_sel_key = (char*)0;
  
  if (!IsHanziInput) return outchar(tty_fd, key);
  if (current_method == 9) return Intcode_HZFilter(tty_fd,key);
  if (current_method == 0) return Pinyin_HZFilter(tty_fd,key);

  switch ( key )
  {
     case '\010':  /* BackSpace Ctrl+H */
     case '\177':  /* BackSpace */
        if ( InputCount > 0 ) 
        {
           InpKey[--InputCount]=0;
           if (InputCount == 0)
           {
               ResetInput();
               ClearSelectionArea();
           }
           else if (InputCount < InputMatch)
           {
               FindMatchKey();
               MultiPageMode = 0;
               CurrentPageIndex = StartKey;
               FillMatchChars(StartKey); 
               DisplaySelection();
           }
           DisplayInput();
        }
        else outchar(tty_fd,key);
        break;

   case '\033':  /* ESCAPE */
        if (InputCount > 0) 
        {
           ClearSelectionArea();
           ClearInputArea();
           ResetInput();
           DisplayInput();
        }
        else outchar(tty_fd,key);
        break;

     case '<':
     case '-':
     case ',':
     case '[':
        if ( MultiPageMode )
        {
           if ( CurrentPageIndex > StartKey) 
                CurrentPageIndex = CurrentPageIndex - cur_table->MaxDupSel;
           else CurrentPageIndex = StartKey;
           if (IsAssociateMode)
                FillAssociateChars(CurrentPageIndex);
           else FillMatchChars(CurrentPageIndex);
           DisplaySelection();
        }
        else outchar(tty_fd, key);
        break;

     case '>':
     case ']':
     case '.':
     case '=':
        if ( MultiPageMode )
        {
           CurrentPageIndex = NextPageIndex;
           if (IsAssociateMode)
               FillAssociateChars(CurrentPageIndex);
           else FillMatchChars(CurrentPageIndex);
           DisplaySelection();
        }
        else outchar(tty_fd, key);
        break;

     case ' ':
        if ( CurSelNum == 0 )
           outchar(tty_fd, key);
        if ( seltab[0][0] )
           putstr(tty_fd, seltab[0]);
        break;

     default:

        inkey   = cur_table->KeyMap[key];
        is_sel_key = strchr( cur_table->selkey, key);
        vv = is_sel_key - cur_table->selkey; 
          /* selkey index, strchr may return NULL */
 
      /* if a key is simultaneously inkey & is_sel_key, then selkey first?*/
        if ( (!inkey && !is_sel_key) || 
             (!inkey && is_sel_key && (CurSelNum == 0 || seltab[vv][0] == 0)) )
        {
           IsAssociateMode = 0;
           ResetInput();
           ClearSelectionArea();
           ClearInputArea();
           outchar(tty_fd, key);
           return;
        }

        if (is_sel_key && CurSelNum > 0 && seltab[vv][0])
        {
            putstr(tty_fd, seltab[vv]);
            return;
        }
        
        /* now it must be inkey? */
        IsAssociateMode = 0; 
        if ( inkey >= 1 && InputCount < MAX_INPUT_LENGTH )
            InpKey[InputCount++] = inkey;
             
        if (InputCount <= InputMatch+1) 
        {
             FindMatchKey();
             CurrentPageIndex = StartKey;
             MultiPageMode = 0;
             FillMatchChars(StartKey);
             if (InputCount >= cur_table->MaxPress && CurSelNum == 1 &&
                 cur_table->last_full)
             {  // left only one selection 
                 return putstr(tty_fd, seltab[0]);        
             }
             DisplaySelection();
        }
        DisplayInput();
        break;

  } /* switch */
}


static void  DisplaySelection(void)
{
  int i, pos = (current_method == 9? IntCode_SelectionX: SelectionX),len;

  ClearSelectionArea();

  if (MultiPageMode && CurrentPageIndex != StartKey) /* not first page */
  {
      InputAreaOutput(pos, "< ", INPUT_FGCOLOR,INPUT_BGCOLOR);    
      pos += 2;
  }

  for( i = 0; i < CurSelNum; i++ )
  {
     if ( !seltab[i][0] ) 
     {
        if (current_method == 0 && i == 0) continue; else break;
     }

     len = strlen(seltab[i]);
     if (pos+len+1 > SelectionXMax) break;

     pVideoInfo->input_sput(pos++, input_table[current_method]->selkey[i],
                INPUT_FGCOLOR,INPUT_BGCOLOR);
     InputAreaOutput(pos,seltab[i],INPUT_FGCOLOR,INPUT_BGCOLOR);
     pos += len + 1;
  }

  if ( MultiPageMode && NextPageIndex != StartKey)  /* not last page */
  {
       InputAreaOutput(pos, "> ", INPUT_FGCOLOR,INPUT_BGCOLOR);
  }
}

/*************************************************************************
 *                        public function                                *
 *************************************************************************/

void ToggleInputMethod(void)
{
  if (input_table[current_method] || current_method == 0)
  {
    IsHanziInput ^= 1;
    cur_table = input_table[current_method];
    ResetInput();
    RefreshInputArea();
  }  
}

void ToggleHalfFull(void)
{
   // temporarily disabled
   /*
   IsFullChar = 1 - IsFullChar;
   ResetInput();
   RefreshInputArea();
   */
}

void InputInit(void)
{
  IsHanziInput = 0;	// default is ascii input 
  IsFullChar = 0;
  current_method = 0;
  input_table[9] = IntCode_Init();

  DefineCap("InputMethod", ConfigInputMethod, NULL);

  InitPinyinInput(&Pinyin_Module);
}

void InputCleanup(void)
{
  int i;
  for (i = 1; i < NR_INPUTMETHOD; i++)
    if (input_table[i])
      UnloadInputMethod(i);

  PinyinInputCleanup(&Pinyin_Module);
}

void SetCurrentInputMethod(int curMethod)
{
  if (input_table[curMethod] || curMethod == 0)
  {
    current_method = curMethod;
    cur_table = input_table[curMethod];
    ResetInput();
    if (current_method != 0)
       UseAssociateMode = 0;  // default no associate
    else UseAssociateMode = 0;

    if (!IsHanziInput)
	ToggleInputMethod();
    else
    	RefreshInputArea();
  }
}

/* before call this function, must lock the console */
void RefreshInputArea(void)
{
  char buf[30];

  pVideoInfo->clear_input(INPUT_BGCOLOR);  // 1 blue

  if (!IsHanziInput)
  {
    sprintf(buf, "Ӣ%s", CurCoding[0]);
    InputAreaOutput(0, buf, INPUT_FGCOLOR,INPUT_BGCOLOR);
    InputAreaOutput((dispInfo.tx_avail-sizeof(VERSION_STRING))/2,
             VERSION_STRING,INPUT_FGCOLOR,INPUT_BGCOLOR);
    sprintf(buf, "ն%d%d",VtNum, curCon->window);
    InputAreaOutput(dispInfo.tx_avail-18, buf, INPUT_FGCOLOR, INPUT_BGCOLOR);
  }
  else if (current_method != 0)
  {
    InputAreaOutput(0, input_table[current_method]->cname,
          INPUT_FGCOLOR,INPUT_BGCOLOR);
    DisplayInput();
    DisplaySelection();
    /* ʾѡִ */

  }
  else RefreshPYInputArea(&Pinyin_Module);
}

void InputAreaOutput(int x, u_char *string, int fg, int bg)
{
  while (*string)
  {
     if (*string >= 0xA0)  // a valid Chinese Char
     {
         pVideoInfo->input_wput(x,*(string+1),*string,fg,bg);
         x += 2;
         string += 2;
     }
     else pVideoInfo->input_sput(x++,*string++,fg,bg);
  }
}


/***************************************************************************
 *                     Internal Code Input Method                          *
 ***************************************************************************/

hz_input_table *IntCode_Init(void)
{
  hz_input_table *table;
  int i,index;

  UseAssociateMode = 0;  /* force to no associate */
  table = malloc(sizeof(hz_input_table));
  if (table == NULL)
    error("Out of memory in loading internal code input method");

   strcpy(table->magic_number,MAGIC_NUMBER);
   strcpy(table->ename, "IntCode");
   strcpy(table->cname, "롿");
   strcpy(table->selkey, "0123456789abcdef");

   table->last_full = 1;
   for(i = 0; i < 128; i++)
   {
      table->KeyMap[i] = 0;
      if ((i >= '0' && i <= '9') || (i >= 'a' && i <= 'f'))
      {
         if (i >= '0' && i <= '9')
            index = i - '0';
         else index = i -'a' + 10;
         table->KeyMap[i] = index;
         table->KeyName[index] = toupper(i);
      }
   }    
   return table;
}

/* StartKey <= index < EndKey is valid */
void IntCode_FindMatchKey(void)
{
   unsigned long Key = (InpKey[0] << 12) | (InpKey[1] << 8);
   switch(InputCount)
   {
      case 0:
      case 1:
         StartKey = EndKey = 0;    /* not display selection */
         break;
      case 2:
         StartKey = Key + 0xA1; 
         EndKey = Key + 0xFF;  /* A1-A9,B0-F7 A1-FE */
         break;
      case 3:
         StartKey = Key + (InpKey[2] << 4);
         if (InpKey[2] == 10) StartKey++; /* A1 */
         EndKey = StartKey + 0x10;  
         if (InpKey[2] == 15) EndKey--;  /* FE */ 
         break;
    }
}

/* ABCD  AB then C=a..f, Sel=0-9 begin display selection
        ABC then D=0-f,  Sel=0-f 
*/
void IntCode_FillMatchChars(int index)
{
    int MaxSel,i;
   
    CurSelNum = 0;
    if (InputCount < 2) return;
    if (InputCount == 2) MaxSel = 10; else MaxSel = 16;
    
    if (index % 256 == 0xA1 && InputCount == 3) 
    {
       seltab[0][0] = '\0';
       CurSelNum++;
    }

    while( CurSelNum < MaxSel && index < EndKey)
    {
         seltab[CurSelNum][0] = index / 256; 
         seltab[CurSelNum][1] = index % 256;
         seltab[CurSelNum][2] = '\0';
         CurSelNum++;
         index++;
    }
    
    for(i = CurSelNum; i < 16; i++)
       seltab[i][0] = '\0';  /* zero out the unused area */

    InputMatch = InputCount;

    /* check if more than one page */
    if ( index < EndKey && CurSelNum == MaxSel && MaxSel == 10)
    {
      /* has another matched key, so enter MultiPageMode, has more pages */
        NextPageIndex = index;
        MultiPageMode = 1;
    }  
    else if (MultiPageMode)
    {
        NextPageIndex = StartKey; /* rotate selection */
    }
    else MultiPageMode = 0;
}

/* return value: characters output to tty_fd */
void Intcode_HZFilter(int tty_fd, unsigned char key)
{
  int inkey = 0;

  switch ( key )
  {
     case '\010':  /* BackSpace Ctrl+H */
     case '\177':  /* BackSpace */
        if ( InputCount > 0 ) 
        {
           InpKey[--InputCount]=0;
           if (InputCount == 0)
           {
               ResetInput();
               ClearSelectionArea();
           }
           else if (InputCount >= 2)
           {
               IntCode_FindMatchKey();
               MultiPageMode = 0;
               CurrentPageIndex = StartKey;
               IntCode_FillMatchChars(StartKey); 
               DisplaySelection();
           }
           DisplayInput();
        }
        else outchar(tty_fd,key);
        break;

   case '\033':  /* ESCAPE */
        if (InputCount > 0) 
        {
           ClearSelectionArea();
           ClearInputArea();
           ResetInput();
           DisplayInput();
        }
        else outchar(tty_fd,key);
        break;

     case '<':
     case '-':
     case ',':
     case '[':
        if ( MultiPageMode )
        {
           if ( CurrentPageIndex > StartKey) 
                CurrentPageIndex = CurrentPageIndex - 10;
           else CurrentPageIndex = StartKey;
           IntCode_FillMatchChars(CurrentPageIndex);
           DisplaySelection();
        }
        else outchar(tty_fd, key);
        break;

     case '>':
     case ']':
     case '.':
     case '=':
        if ( MultiPageMode )
        {
           CurrentPageIndex = NextPageIndex;
           IntCode_FillMatchChars(CurrentPageIndex);
           DisplaySelection();
        }
        else outchar(tty_fd, key);
        break;

     case ' ':
        if ( CurSelNum == 0 )
           outchar(tty_fd, key);
        if ( seltab[0][0] )
           putstr(tty_fd, seltab[0]);
        break;

     default:
        inkey   = cur_table->KeyMap[key];
        switch(InputCount)
        {
           case 0:  /* A-F */
           case 1:  /* 0-F */
           case 2:  /* A-F */
              if (inkey >=0 && !(InputCount != 1 && inkey < 10) &&
                   !(InputCount == 1 && 
                      ( (InpKey[0]==10 && inkey>9) ||       //A1-A9 
                        (InpKey[0]==10 && inkey==0) ||
                        (InpKey[0]==15 && inkey>7)) ) )      //B0-F7  
              {
                  InpKey[InputCount++] = inkey;
                  DisplayInput();
                  if (InputCount >=2)
                  {
                      IntCode_FindMatchKey();
                      MultiPageMode = 0;
                      CurrentPageIndex = StartKey;
                      IntCode_FillMatchChars(StartKey);
                      DisplaySelection();
                  }
                  DisplayInput();
              }
              else if (InputCount == 2 && inkey >= 0 && inkey < CurSelNum &&
                       seltab[inkey][0])
              {
                   putstr(tty_fd, seltab[inkey]);
              }
              else /* not select key or input key */
              {
                  ResetInput();
                  ClearSelectionArea();
                  ClearInputArea();
                  outchar(tty_fd, key);
                  return;
              }
              break;
           case 3:   /* output char */
                      /* A1A1-FEFE */
              if (inkey >=0 && !(InpKey[2] == 10 && inkey == 0) &&
                  !(InpKey[2] == 15 && inkey == 15))
              {
                  seltab[0][0] = (InpKey[0] << 4) | InpKey[1];
                  seltab[0][1] = (InpKey[2] << 4) | inkey ;
                  seltab[0][2] = '\0';
                  putstr(tty_fd, seltab[0]);
              }
              else
              {
                  ResetInput();
                  ClearSelectionArea();
                  ClearInputArea();
                  outchar(tty_fd, key);
                  return;
              }
              break;
        }
  } /* switch */
}

