/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#define NALLOC_INC 1024

void *
Duplicate(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  if (operatorArgc() > 1) cdoAbort("Too many arguments!");

  const int ndup = (operatorArgc() == 1) ? parameter2int(operatorArgv()[0]) : 2;
  if (Options::cdoVerbose) cdoPrint("ndup = %d", ndup);

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const int nvars = vlistNvars(vlistID1);
  int ntsteps = vlistNtsteps(vlistID1);

  if (ntsteps == 1)
    {
      for (varID = 0; varID < nvars; ++varID)
        if (vlistInqVarTimetype(vlistID1, varID) != TIME_CONSTANT) break;

      if (varID == nvars) ntsteps = 0;
    }

  if (ntsteps == 0)
    {
      for (varID = 0; varID < nvars; ++varID) vlistDefVarTimetype(vlistID2, varID, TIME_VARYING);
    }

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList;
  varListInit(varList, vlistID1);

  FieldVector3D vars;
  std::vector<int64_t> vdate;
  std::vector<int> vtime;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      if ((size_t)tsID >= vdate.size()) vdate.resize(vdate.size() + NALLOC_INC);
      if ((size_t)tsID >= vtime.size()) vtime.resize(vtime.size() + NALLOC_INC);
      if ((size_t)tsID >= vars.size()) vars.resize(vars.size() + NALLOC_INC);

      vdate[tsID] = taxisInqVdate(taxisID1);
      vtime[tsID] = taxisInqVtime(taxisID1);

      fieldsFromVlist(vlistID1, vars[tsID], FIELD_NONE);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          const size_t gridsize = varList[varID].gridsize;
          vars[tsID][varID][levelID].resize(gridsize);
          cdoReadRecord(streamID1, vars[tsID][varID][levelID].vec.data(), &nmiss);
          vars[tsID][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  int nts = tsID;

  for (int idup = 0; idup < ndup; idup++)
    {
      for (tsID = 0; tsID < nts; tsID++)
        {
          taxisDefVdate(taxisID2, vdate[tsID]);
          taxisDefVtime(taxisID2, vtime[tsID]);
          cdoDefTimestep(streamID2, idup * nts + tsID);

          for (varID = 0; varID < nvars; varID++)
            {
              const int nlevel = varList[varID].nlevels;
              for (levelID = 0; levelID < nlevel; levelID++)
                {
                  if (!vars[tsID][varID][levelID].vec.empty())
                    {
                      nmiss = vars[tsID][varID][levelID].nmiss;
                      cdoDefRecord(streamID2, varID, levelID);
                      cdoWriteRecord(streamID2, vars[tsID][varID][levelID].vec.data(), nmiss);
                    }
                }
            }
        }
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
