/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Merge      merge           Merge datasets with different fields
*/

#include <cdi.h>

#include "cdo_int.h"

#include "util_files.h"

static void
checkDupEntry(int vlistID1, int vlistID2, const char *filename)
{
  char vname1[CDI_MAX_NAME], vname2[CDI_MAX_NAME];
  std::vector<double> lev1, lev2;

  const int nvars1 = vlistNvars(vlistID1);
  const int nvars2 = vlistNvars(vlistID2);

  for (int varID1 = 0; varID1 < nvars1; ++varID1)
    {
      vlistInqVarName(vlistID1, varID1, vname1);
      const int param1 = vlistInqVarParam(vlistID1, varID1);
      const int gridID1 = vlistInqVarGrid(vlistID1, varID1);
      const int zaxisID1 = vlistInqVarZaxis(vlistID1, varID1);
      const int gtype1 = gridInqType(gridID1);
      const size_t gsize1 = gridInqSize(gridID1);
      const int ztype1 = zaxisInqType(zaxisID1);
      const size_t nlev1 = zaxisInqSize(zaxisID1);
      if (nlev1 > lev1.size()) lev1.resize(nlev1);
      cdoZaxisInqLevels(zaxisID1, lev1.data());

      for (int varID2 = 0; varID2 < nvars2; ++varID2)
        {
          vlistInqVarName(vlistID2, varID2, vname2);
          const int param2 = vlistInqVarParam(vlistID2, varID2);
          const int gridID2 = vlistInqVarGrid(vlistID2, varID2);
          const int zaxisID2 = vlistInqVarZaxis(vlistID2, varID2);
          const int gtype2 = gridInqType(gridID2);
          const size_t gsize2 = gridInqSize(gridID2);
          const int ztype2 = zaxisInqType(zaxisID2);
          const size_t nlev2 = zaxisInqSize(zaxisID2);
          if (gtype1 == gtype2 && gsize1 == gsize2 && ztype1 == ztype2 && nlev1 == nlev2)
            {
              if (nlev2 > lev2.size()) lev2.resize(nlev2);
              cdoZaxisInqLevels(zaxisID2, lev2.data());

              if (zaxisInqLevels(zaxisID1, nullptr) && zaxisInqLevels(zaxisID2, nullptr))
                {
                  for (size_t k = 0; k < nlev2; ++k)
                    if (!IS_EQUAL(lev1[k], lev2[k])) return;
                }

              if ((param1 < 0 || param2 < 0) && strcmp(vname1, vname2) == 0)
                {
                  cdoWarning("Duplicate entry of parameter name %s in %s!", vname2, filename);
                }
              else if (param1 == param2)
                {
                  char paramstr[32];
                  cdiParamToString(param2, paramstr, sizeof(paramstr));
                  cdoWarning("Duplicate entry of parameter %s in %s!", paramstr, filename);
                }
              else if (param1 != param2 && strcmp(vname1, vname2) == 0)
                {
                  cdoWarning("Duplicate entry of parameter name %s with different IDs in %s!", vname2, filename);
                }
            }
        }
    }
}
/*
static
int vlistConstVars(int vlistID)
{
  int nvars = vlistNvars(vlistID);

  for ( int varID = 0; varID < nvars; ++varID )
    if ( vlistInqVarTimetype(vlistID, varID) != TIME_CONSTANT ) return 0;

  return 1;
}
*/

static int
getTaxisindex(const std::vector<int> &vlistIDs)
{
  if (vlistNtsteps(vlistIDs[0]) == 0)
    {
      const int nmerge = vlistIDs.size();
      for (int index = 1; index < nmerge; index++)
        {
          if (vlistNtsteps(vlistIDs[index]) != 0) return index;
        }
    }

  return 0;
}

void *
Merge(void *process)
{
  int varID, varID2;
  int nrecs = 0;
  int levelID, levelID2;
  int index;
  size_t nmiss;

  cdoInitialize(process);

  const bool lcopy = unchangedRecord();

  const int streamCnt = cdoStreamCnt();
  const int nmerge = streamCnt - 1;

  const char *ofilename = cdoGetStreamName(streamCnt - 1);

  if (!Options::cdoOverwriteMode && fileExists(ofilename) && !userFileOverwrite(ofilename))
    cdoAbort("Outputfile %s already exists!", ofilename);

  std::vector<CdoStreamID> streamIDs(nmerge);
  std::vector<int> vlistIDs(nmerge);
  std::vector<int> numrecs(nmerge);
  std::vector<int> numsteps(nmerge);

  for (index = 0; index < nmerge; index++)
    {
      streamIDs[index] = cdoOpenRead((index));
      vlistIDs[index] = cdoStreamInqVlist(streamIDs[index]);
    }

  const int taxisindex = getTaxisindex(vlistIDs);

  const int taxisID1 = vlistInqTaxis(vlistIDs[taxisindex]);
  const int taxisID2 = taxisDuplicate(taxisID1);

  const int vlistID2 = vlistCreate();
  vlistCopy(vlistID2, vlistIDs[0]);
  for (index = 1; index < nmerge; index++)
    {
      checkDupEntry(vlistID2, vlistIDs[index], cdoGetCommandFromInStream(index));
      vlistMerge(vlistID2, vlistIDs[index]);
    }

  int numconst = 0;
  for (index = 0; index < nmerge; index++)
    {
      numsteps[index] = vlistNtsteps(vlistIDs[index]);
      if (numsteps[index] == 0) numsteps[index] = 1;
      if (numsteps[index] == 1) numconst++;
    }

  if (numconst > 0 && numconst < nmerge)
    for (index = 0; index < nmerge; index++)
      {
        if (numsteps[index] == 1)
          {
            const int vlistID1 = vlistIDs[index];
            const int nvars = vlistNvars(vlistID1);
            for (int varID = 0; varID < nvars; ++varID)
              {
                varID2 = vlistMergedVar(vlistID1, varID);
                vlistDefVarTimetype(vlistID2, varID2, TIME_CONSTANT);
              }
          }
      }

  if (Options::cdoVerbose)
    {
      for (index = 0; index < nmerge; index++) vlistPrint(vlistIDs[index]);
      vlistPrint(vlistID2);
    }

  CdoStreamID streamID2 = cdoOpenWrite((streamCnt - 1));

  vlistDefTaxis(vlistID2, taxisID2);
  cdoDefVlist(streamID2, vlistID2);

  std::vector<double> array;
  if (!lcopy)
    {
      const size_t gridsizemax = vlistGridsizeMax(vlistID2);
      array.resize(gridsizemax);
    }

  int tsID = 0;
  while (tsID >= 0)
    {
      for (index = 0; index < nmerge; index++)
        {
          if (vlistIDs[index] == -1) continue;
          numrecs[index] = cdoStreamInqTimestep(streamIDs[index], tsID);
        }

      for (index = 0; index < nmerge; index++)
        if (numrecs[index] != 0) break;
      if (index == nmerge) break;  // EOF on all input streams

      if (tsID == 1)
        {
          for (index = 0; index < nmerge; index++)
            if (numrecs[index] == 0 && numsteps[index] == 1) vlistIDs[index] = -1;
          /*
          for ( index = 0; index < nmerge; index++ )
            if ( vlistIDs[index] != -1 )
              {
                firstindex = index;
                break;
              }
          */
        }
      /*
      for ( index = 0; index < nmerge; index++ )
        printf("tsID %d   %d sID %d vID %d nrecs %d\n", tsID, index,
      streamIDs[index], vlistIDs[index], numrecs[index]);
      */
      if (numrecs[taxisindex] == 0)
        {
          for (index = 1; index < nmerge; index++)
            if (vlistIDs[index] != -1 && numrecs[index] != 0)
              cdoWarning("Input stream %d has %d timestep%s. Stream %d has more timesteps, skipped!", taxisindex + 1, tsID,
                         tsID == 1 ? "" : "s", index + 1);
          break;
        }
      else
        {
          for (index = 1; index < nmerge; index++)
            if (vlistIDs[index] != -1 && numrecs[index] == 0)
              {
                cdoWarning("Input stream %d has %d timestep%s. Stream %d has more timesteps, skipped!", index + 1, tsID,
                           tsID == 1 ? "" : "s", taxisindex + 1);
                break;
              }
          if (index < nmerge) break;
        }

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (index = 0; index < nmerge; index++)
        {
          CdoStreamID streamID1 = streamIDs[index];
          const int vlistID1 = vlistIDs[index];
          nrecs = numrecs[index];

          if (vlistID1 == -1) continue;

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);

              varID2 = vlistMergedVar(vlistID1, varID);
              levelID2 = vlistMergedLevel(vlistID1, varID, levelID);

              if (Options::cdoVerbose) cdoPrint("var %d %d %d %d", varID, levelID, varID2, levelID2);

              cdoDefRecord(streamID2, varID2, levelID2);
              if (lcopy)
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array.data(), &nmiss);
                  cdoWriteRecord(streamID2, array.data(), nmiss);
                }
            }
        }

      tsID++;
    }

  for (index = 0; index < nmerge; index++) cdoStreamClose(streamIDs[index]);

  cdoStreamClose(streamID2);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
