/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "cdo_wtime.h"
#include "process_int.h"
#include "functs.h"
#include "griddes.h"
#include "gridreference.h"
#include "grid_point_search.h"
#include <mpim_grid.h>
#include "cimdOmp.h"


constexpr double PI2 = M_PI * 2.0;

static
void read_coordinates(int gridID, Varray<double> &xvals, Varray<double> &yvals)
{
  const auto gridID0 = gridID;

  auto gridtype = gridInqType(gridID);

  if (gridtype == GRID_GME) gridID = gridToUnstructured(gridID, 0);

  if (gridtype != GRID_UNSTRUCTURED && gridtype != GRID_CURVILINEAR) gridID = gridToCurvilinear(gridID, 0);

  if (gridtype == GRID_UNSTRUCTURED && !gridHasCoordinates(gridID))
    {
      const auto reference = dereferenceGrid(gridID);
      if (reference.isValid) gridID = reference.gridID;
      if (reference.notFound) cdoAbort("Reference to source grid not found!");
    }

  if (!gridHasCoordinates(gridID)) cdoAbort("Cell center coordinates missing!");

  gridInqXvals(gridID, xvals.data());
  gridInqYvals(gridID, yvals.data());

  auto gridsize = gridInqSize(gridID);
  // Convert lat/lon units if required
  cdo_grid_to_radian(gridID, CDI_XAXIS, gridsize, xvals.data(), "grid center lon");
  cdo_grid_to_radian(gridID, CDI_YAXIS, gridsize, yvals.data(), "grid center lat");

  for (size_t i = 0; i < gridsize; ++i)
    {
      if (xvals[i] > PI2) xvals[i] -= PI2;
      if (xvals[i] < 0.0) xvals[i] += PI2;
    }

  if (gridID0 != gridID) gridDestroy(gridID);
}

static
Varray2D<size_t> gen_mapdata(int gridID1, int gridID2)
{
  auto gridsize1 = gridInqSize(gridID1);
  auto gridsize2 = gridInqSize(gridID2);

  Varray2D<size_t> mapdata(gridsize2);

  Varray<double> xvals1(gridsize1), yvals1(gridsize1);
  read_coordinates(gridID1, xvals1, yvals1);

  Varray<double> xvals2(gridsize2), yvals2(gridsize2);
  read_coordinates(gridID2, xvals2, yvals2);

  const auto nlon = gridInqXsize(gridID2);
  const auto nlat = gridInqYsize(gridID2);
  Varray<double> xbounds(nlon*2), ybounds(nlat*2);

  if (gridInqXbounds(gridID2, nullptr))
    {
      gridInqXbounds(gridID2, xbounds.data());
    }
  else
    {
      Varray<double> xvals(nlon);
      gridInqXvals(gridID2, xvals.data());
      grid_gen_bounds(nlon, xvals, xbounds);
    }

  if (gridInqYbounds(gridID2, nullptr))
    {
      gridInqYbounds(gridID2, ybounds.data());
    }
  else
    {
      Varray<double> yvals(nlat);
      gridInqYvals(gridID2, yvals.data());
      grid_gen_bounds(nlat, yvals, ybounds);
      grid_check_lat_borders(2 * nlat, ybounds.data());
    }

  for (size_t i = 0; i < 2*nlon; ++i) xbounds[i] *= DEG2RAD;
  for (size_t i = 0; i < 2*nlat; ++i) ybounds[i] *= DEG2RAD;

  for (size_t i = 0; i < nlon; ++i)
    {
      if (xbounds[2*i+1] > PI2)
        {
          xbounds[2*i]   -= PI2;
          xbounds[2*i+1] -= PI2;
        }
      if (xbounds[2*i+1] < 0.0)
        {
          xbounds[2*i]   += PI2;
          xbounds[2*i+1] += PI2;
        }
    }
  // for (size_t i = 0; i < nlon; ++i) printf("%zu %g %g\n", i+1, xbounds[2*i], xbounds[2*i+1]);
  // for (size_t i = 0; i < nlat; ++i) printf("%zu %g %g\n", i+1, ybounds[2*i], ybounds[2*i+1]);

  auto start = Options::cdoVerbose ? cdo_get_wtime() : 0.0;

  bool xIsCyclic = false;
  size_t dims[2] = { gridsize1, 0 };
  GridPointSearch gps;
  gridPointSearchCreate(gps, xIsCyclic, dims, gridsize1, xvals1, yvals1);

  if (Options::cdoVerbose) cdoPrint("Point search created: %.2f seconds", cdo_get_wtime() - start);

  start = Options::cdoVerbose ? cdo_get_wtime() : 0;

  size_t ndist = gridsize1;
  std::vector<double> values(gridsize1);
  std::vector<size_t> adds(gridsize1);
  std::vector<double> dist(gridsize1);  

  for (size_t i = 0; i < gridsize2; ++i)
    {
      const auto iy = i / nlon;
      const auto ix = i - iy * nlon;

      const auto lon2 = xvals2[i];
      const auto lat2 = yvals2[i];
      double p1[3], p2[3];
      gcLLtoXYZ(lon2, lat2, p1);
      double lons[4], lats[4];
      lons[0] = xbounds[2*ix];
      lons[1] = xbounds[2*ix];
      lons[2] = xbounds[2*ix+1];
      lons[3] = xbounds[2*ix+1];
      lats[0] = ybounds[2*iy+1];
      lats[1] = ybounds[2*iy];
      lats[2] = ybounds[2*iy];
      lats[3] = ybounds[2*iy+1];
      double maxdist = 0.0;
      for (size_t k = 0; k < 4; ++k)
        {
          gcLLtoXYZ(lons[k], lats[k], p2);
          const auto sdist = squareDistance(p1, p2);
          if (sdist > maxdist) maxdist = sdist;
        }
      maxdist = 1.01 * std::sqrt(maxdist);

      auto nadds = gridPointSearchDistanceQnearest(gps, maxdist, lon2, lat2, ndist, adds.data(), dist.data());
      // printf("%zu nadds %zu\n", i+1, nadds);

      size_t nvalues = 0;
      for (size_t k = 0; k < nadds; ++k)
        {
          const auto x = xvals1[adds[k]];
          const auto y = yvals1[adds[k]];
          if (y >= ybounds[2*iy] && y < ybounds[2*iy + 1])
            if ((x >= xbounds[2*ix] && x < xbounds[2*ix + 1]) ||
                ((x-PI2) >= xbounds[2*ix] && (x-PI2) < xbounds[2*ix + 1]))
              {
                adds[nvalues] = adds[k];
                nvalues++;
              }
        }
      //if (Options::cdoVerbose) printf("%zu nadds %zu nvalues %zu  maxdist %g\n", i+1, nadds, nvalues, maxdist);

      if (nvalues)
        {
          mapdata[i].resize(nvalues);
          for (size_t k = 0; k < nvalues; ++k) mapdata[i][k] = adds[k];
        }
    }

  if (Options::cdoVerbose) cdoPrint("Point search qnearest: %.2f seconds", cdo_get_wtime() - start);

  gridPointSearchDelete(gps);
  
  return mapdata;
}


template <typename T>
static
T remap_kernel(int operfunc, const Varray<size_t> &adds, size_t &nmiss2, Field &field, Varray<T> &fieldvec, const Varray<T> &vec1, const T missval)
{
  T value;
  const auto nvalues = adds.size();
  if (nvalues)
    {
      field.nmiss = 0;
      for (size_t k = 0; k < nvalues; ++k)
        {
          const auto v1 = vec1[adds[k]];
          fieldvec[k] = v1;
          if (DBL_IS_EQUAL(v1, missval)) field.nmiss++;
        }

      field.size = nvalues;
      field.missval = missval;
      value = fieldFunction(field, operfunc);
      if (DBL_IS_EQUAL(value, missval)) nmiss2++;
    }
  else
    {
      value = missval;
      nmiss2++;
    }

  return value;
}

static
void remap_field(const Varray2D<size_t> &mapdata, const Field &field1, Field &field2, int operfunc)
{
  std::vector<Field> fields(Threading::ompNumThreads);

  auto gridsize2 = gridInqSize(field2.grid);
  auto start = Options::cdoVerbose ? cdo_get_wtime() : 0.0;

  size_t nmiss2 = 0;
#ifdef _OPENMP
#pragma omp parallel for default(none) schedule(static) reduction(+ : nmiss2) shared(operfunc, gridsize2, mapdata, fields, field1, field2)
#endif
  for (size_t i = 0; i < gridsize2; ++i)
    {
      const auto &adds = mapdata[i];
      const auto nvalues = adds.size();
      const auto missval = field1.missval;

      const auto ompthID = cdo_omp_get_thread_num();
      auto &field = fields[ompthID];
      field.memType = field1.memType;
      if (field.memType == MemType::Float)
        field.vec_f.resize(nvalues);
      else
        field.vec_d.resize(nvalues);

      if (field.memType == MemType::Float)
        field2.vec_f[i] = remap_kernel(operfunc, adds, nmiss2, field, field.vec_f, field1.vec_f, (float)missval);
      else
        field2.vec_d[i] = remap_kernel(operfunc, adds, nmiss2, field, field.vec_d, field1.vec_d, missval);
    }

  field2.nmiss = nmiss2;

  if (Options::cdoVerbose) cdoPrint("Remap: %.3f seconds", cdo_get_wtime() - start);
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("remaprange", func_range,  0, nullptr);
  cdoOperatorAdd("remapmin",   func_min,    0, nullptr);
  cdoOperatorAdd("remapmax",   func_max,    0, nullptr);
  cdoOperatorAdd("remapsum",   func_sum,    0, nullptr);
  cdoOperatorAdd("remapmean",  func_mean,   0, nullptr);
  cdoOperatorAdd("remapavg",   func_avg,    0, nullptr);
  cdoOperatorAdd("remapstd",   func_std,    0, nullptr);
  cdoOperatorAdd("remapstd1",  func_std1,   0, nullptr);
  cdoOperatorAdd("remapvar",   func_var,    0, nullptr);
  cdoOperatorAdd("remapvar1",  func_var1,   0, nullptr);
  cdoOperatorAdd("remapskew",  func_skew,   0, nullptr);
  cdoOperatorAdd("remapkurt",  func_kurt,   0, nullptr);
  // clang-format on
}

void *
Remapstat(void *process)
{
  int nrecs;
  int varID, levelID;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  operatorInputArg("grid description file or name");
  const auto gridID2 = cdoDefineGrid(cdoOperatorArgv(0));
  auto gridtype = gridInqType(gridID2);
  if (gridtype != GRID_LONLAT && gridtype != GRID_GAUSSIAN)
    cdoAbort("Remaping to %s data unsupported!", gridNamePtr(gridtype));

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto ngrids = vlistNgrids(vlistID1);
  const auto gridID1 = vlistGrid(vlistID1, 0);
  for (int index = 0; index < ngrids; index++)
    {
      if (index > 0) cdoAbort("Too many different grids!");

      // const auto gridID1 = vlistGrid(vlistID1, index);
      gridtype = gridInqType(gridID1);

      const bool lprojparams = (gridtype == GRID_PROJECTION) && grid_has_proj_params(gridID1);
      if (!gridProjIsSupported(gridID1) && !lprojparams && gridtype != GRID_LONLAT && gridtype != GRID_GAUSSIAN
          && gridtype != GRID_GME && gridtype != GRID_CURVILINEAR && gridtype != GRID_UNSTRUCTURED)
        cdoAbort("Interpolation of %s data unsupported!", gridNamePtr(gridtype));

      vlistChangeGridIndex(vlistID2, index, gridID2);
    }

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  Varray2D<size_t> mapdata = gen_mapdata(gridID1, gridID2);

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  Field field1, field2;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          field1.init(varList1[varID]);
          cdoReadRecord(streamID1, field1);

          field2.init(varList2[varID]);
 
          remap_field(mapdata, field1, field2, operfunc);

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, field2);
        }
      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
