/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>

#include "cdw_thread.h"
#include "cdw_regex_dispatch.h"
#include "cdw_md5sum.h"
#include "cdw_md5sum_regex.h"
#include "cdw_debug.h"



extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
/* extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1]; */


extern int digest_index;
extern char md5sum_digest_disc[];
extern char md5sum_digest_image[];



/* functions used for stdout output handling */
static void handle_md5sum_output(regex_t *regex, regmatch_t *matches);


static regex_t regex1;
static regmatch_t *matches1 = NULL;




/**
 * \brief Allocate memory for regexp data, compile regular expressions (stdout version)
 *
 * Allocate memory for all variables of type regex_t and regmatch_t that
 * are used in processing of data in stdout_pipe_buffer[].
 *
 * Compile all regular expressions that are used in processing of data in
 * stdout_pipe_buffer[].
 *
 * Call regcomp_error_handler() on errors.
 */
void md5sum_stdout_regexp_prepare(void)
{
	/* dea059d48b26fc29602e4f130d27614c  file_name */
	int rv = regcomp(&regex1, "^([a-zA-Z0-9]+)", REG_EXTENDED);
	matches1 = (regmatch_t *) calloc(1, (regex1.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &regex1, 1);
	}

	return;

}





/**
 * \brief Match data in stdout_pipe_buffer[] agains regular expressions, call handler function
 *
 * This function calls regexec() for all regular expressions to compare them
 * against data in stdout_pipe_buffer[]. If there will be any match, the
 * function will call proper haldler function to further process data from
 * stdout_pipe_buffer[]
 */
void md5sum_stdout_regexp_execute(void)
{
	stdout_pipe_buffer[PIPE_BUFFER_SIZE - 1] = '\0';

	/* dea059d48b26fc29602e4f130d27614c  file_name */
	int rv = regexec(&regex1, stdout_pipe_buffer, regex1.re_nsub + 1, matches1, 0);
	if (rv == 0) {
		handle_md5sum_output(&regex1, matches1);
	}

	return;
}





/**
 * \brief Free all memory used by regular expression variables (stdout version)
 *
 * Call regfree() and free() all variables of type regex_t and regmatch_t that
 * were used in processing of data in stdout_pipe_buffer[].
 *
 * See also cdrecord_stderr_regexp_destroy()
 */
void md5sum_stdout_regexp_destroy(void)
{
	regfree(&regex1);
	free(matches1);

	return;
}





/**
   \brief Function stores 32-char digest in md5sum_digestX
*/
void handle_md5sum_output(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing md5sum digest:
                   1
	   "^([a-zA-Z0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);
	unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
	} else {
		cdw_vdm ("INFO: submatch %d / digest for %s: \"%s\"\n",
			 sub,
			 digest_index == CDW_MD5SUM_DIGEST_4_DISC ? "disc" : "image",
			 submatch);

		if (digest_index == CDW_MD5SUM_DIGEST_4_DISC) {
			strncpy(md5sum_digest_disc, submatch, (size_t) len + 1);
		} else if (digest_index == CDW_MD5SUM_DIGEST_4_IMAGE) {
			strncpy(md5sum_digest_image, submatch, (size_t) len + 1);
		} else {
			;
		}
	}
	return;
}

