/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file write_wizard.c
   \brief Window with basic write options, displayed when user selects writing action

   The window displays message "You are writing to X disc", allows user to
   select writing speed, and writing mode. 'Configuration' button is also
   available, so user can jump to configuration window to set more advanced
   options without closing write wizard.

   If current task is writing iso image to blank CD, then 'Verify' checkbox
   is also available. It is coupled with task->verify_write (where \p task
   is argument passed to write_wizard()).

   Writing speed values and modes that can be selected in wizard window are
   calculated every time the window is displayed and are disc-specific.
   They are calculated by functions defined in this file using values stored
   in current_disc variable. Be sure to call run_command_cdrecord_atip(),
   run_command_cdrecord_msinfo() or run_command_dvd_rw_mediainfo() before
   calling write_wizard().

   Code in this file does not check writing preconditions - be sure to call
   cdw_burn_disc_check_preconditions() before calling write_wizard().
*/


#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include <ncursesw/ncurses.h>

#include "cdw_config.h"
#include "cdw_config_ui.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_write_wizard.h"
#include "cdw_main_window.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"


extern cdw_config_t global_config; /* main cdw configuration variable */


enum field_ids {
	f_speed_l = 0,
	f_speed_i,
	f_session_l,
	f_session_i,
	f_disc_l,
	f_disc_i,
	f_verify_l,
	f_verify_i,
	f_config_l,
	f_config_i,
	f_write_i,
	f_cancel_i
};


static cdw_rv_t cdw_write_wizard_init(void);
static cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_write_wizard_destroy(void);
static cdw_rv_t cdw_write_wizard_driver(void);

static CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(cdw_task_t *task);
static CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(cdw_task_t *task);
static CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(cdw_disc_t *disc);

static void cdw_write_wizard_debug_at_exit(cdw_task_t *task);
static void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc);

static bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc);

static void cdw_write_wizard_driver_handle_enter(cdw_form_t *cdw_form, int fi, void *data);

#define CDW_WRITE_WIZARD_N_FIELDS 12
static FIELD *wizard_fields[CDW_WRITE_WIZARD_N_FIELDS + 1];

static struct {
	CDW_DROPDOWN *write_speed_dropdown;
	CDW_DROPDOWN *session_mode_dropdown;
	CDW_DROPDOWN *disc_mode_dropdown;

	CDW_BUTTON *write_button;
	CDW_BUTTON *cancel_button;
	CDW_BUTTON *config_button;

	struct {
		/* used in code related to 'Verify' checkbox, we can do verification
		   only for first track of data on CD*/
		bool visible;
		/* this is temporary var, used instead of task->burn.verify, so that
		   unconfirmed changes aren't saved in task->burn.verify */
		bool checked;

	} verify_checkbox;

	cdw_form_t *cdw_form;
} wizard;



/* *** some layout constraints *** */

#define first_col                  2  /* main message and some labels start in leftmost column */
#define top_label_row              2 /* main message is displayed on top of window */
#define write_speed_label_row      2
#define session_mode_label_row     4
#define disc_mode_label_row        6
#define verify_checkbox_label_row  8
#define config_label_row          10 /* label above of button invoking configuration window */
#define buttons_row               13 /* Write, Cancel and Help buttons are at the bottom of window */

#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 4)
#define wide_label_n_cols          (subwindow_n_cols - 2)
#define narrow_label_n_cols        (subwindow_n_cols - 15)
#define wide_dropdown_n_cols       (subwindow_n_cols - 6)
#define narrow_dropdown_n_cols     8



/**
   \brief Ncurses window showing summary of prepared write process

   A window that shows 'You are writing to X' message and mode and speed of
   writing. Mode and speed can be adjusted in the window. User can press
   'Configuration' button to see and change other options. Two other
   buttons available are 'Write' and 'Cancel'.

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK if user selects 'Write' button
   \return CDW_CANCEL if user presses Escape key or selects 'Cancel' button
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t write_wizard(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %d\n", task->id);
	cdw_assert (disc->type_writable == CDW_TRUE, "ERROR: disc type is not writable\n");

	if (disc->type_writable != CDW_TRUE) {
		cdw_vdm ("ERROR: disc type is not writable\n");
		return CDW_GEN_ERROR;
	}

	cdw_rv_t crv = cdw_write_wizard_init();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_write_wizard_destroy();
		return CDW_GEN_ERROR;
	}
	crv = cdw_write_wizard_build(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build wizard\n");
		cdw_write_wizard_destroy();
		return CDW_GEN_ERROR;
	}

	wrefresh(wizard.cdw_form->subwindow);

	crv = cdw_write_wizard_driver();
	if (crv == CDW_OK) {
		/* save values set in wizard; this works both when uses
		   made any changes in wizard, or when only accepted current
		   settings */
		task->burn.session_mode = cdw_dropdown_get_current_item_id(wizard.session_mode_dropdown);
		task->burn.disc_mode = cdw_dropdown_get_current_item_id(wizard.disc_mode_dropdown);
		task->burn.speed = cdw_dropdown_get_current_item_id(wizard.write_speed_dropdown);
		task->burn.verify = false;
		if (wizard.verify_checkbox.visible) {
			int checked = cdw_form_get_field_bit(wizard.cdw_form->fields[f_verify_i]);
			if (checked == 1) {
				task->burn.verify = true;
			}
		}
#ifndef NDEBUG
		cdw_write_wizard_debug_at_exit(task);
#endif
		cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: write wizard didn't set disc writing mode\n");
		cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: write wizard didn't set session writing mode\n");
	} else {
		cdw_vdm ("INFO: not attempting writing, not displaying settings\n");
	}

	cdw_write_wizard_destroy();

	/* redraw parent */
	cdw_main_ui_main_window_wrefresh();

	return crv;
}





/**
   \brief Set initial values of some variables global in the file

   Function sets values of various fields of "wizard" variable.
*/
cdw_rv_t cdw_write_wizard_init(void)
{
	wizard.write_button = (CDW_BUTTON *) NULL;
	wizard.cancel_button = (CDW_BUTTON *) NULL;
	wizard.config_button = (CDW_BUTTON *) NULL;

	wizard.write_speed_dropdown = (CDW_DROPDOWN *) NULL;
	wizard.session_mode_dropdown = (CDW_DROPDOWN *) NULL;
	wizard.disc_mode_dropdown = (CDW_DROPDOWN *) NULL;

	wizard.verify_checkbox.visible = true;
	wizard.verify_checkbox.checked = false;

	wizard.cdw_form = cdw_form_new(CDW_WRITE_WIZARD_N_FIELDS);
	if (wizard.cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->fields = wizard_fields;

	return CDW_OK;
}





cdw_rv_t cdw_write_wizard_driver(void)
{
	int fi = 0;
	int key = 'a';
	while (key != CDW_KEY_ESCAPE) {
		key = cdw_form_driver(wizard.cdw_form, fi, &global_config);

		/* in this wizard driver we are interested only in
		   ENTER being pressed on either "write" or "cancel"
		   buttons */
		if (key == CDW_KEY_ENTER) {
			fi = field_index(current_field(wizard.cdw_form->form));
			if (fi == f_write_i) {

				/* flush */
				form_driver(wizard.cdw_form->form, REQ_VALIDATION);

				return CDW_OK;
			} else if (fi == f_cancel_i) {
				return CDW_CANCEL;
			} else {
				;
			}
		} else {
			;
		}
	}

	return CDW_CANCEL;
}





/**
   \brief Create dropdown with values showing allowed session modes

   Function returns dropdown with items presenting session modes acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(cdw_task_t *task)
{
	/* labels for _all_ possible session modes allowed for any optical disc */
	char *mode_labels[CDW_SESSION_MODE_N_MAX];

	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_SESSION_MODE_START_MULTI] = _("Start new, appendable disc");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_SESSION_MODE_CREATE_SINGLE] = _("Create non-appendable disc");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_SESSION_MODE_CONTINUE_MULTI] = _("Append data, don\'t close disc");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_SESSION_MODE_WRITE_FINAL] = _("Append data and close disc");
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	mode_labels[CDW_SESSION_MODE_ERROR] = _("(ERROR)");

	int n_items = 0;
	for (n_items = 0; n_items < CDW_SESSION_MODE_N_MAX; n_items++) {
		/* session_modes is non-sparse up until
		   element of value CDW_SESSION_MODE_INIT */
		if (task->burn.session_modes[n_items] == CDW_SESSION_MODE_INIT) {
			break;
		}
	}

	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  session_mode_label_row + 1,
						  first_col + 2,
						  wide_dropdown_n_cols,
						  n_items, /* n_items = n_items_max */
						  CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < n_items; i++) {
		/* this smart line selects only session modes allowed for current disc */
		int id = task->burn.session_modes[i];
		cdw_assert (id != CDW_SESSION_MODE_INIT, "ERROR: found INIT in allowed session modes for i = %d\n", i);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, id, mode_labels[id]);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d with id = %d and label = \"%s\"\n",
				 i, id, mode_labels[id]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		} else {
			cdw_vdm ("INFO: session mode dropdown label #%d = %d /  \"%s\"\n", i, id, cdw_dropdown_get_label_by_ind(dropdown, i));
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		/* 0 - first mode (set by "task" module) is the default one */
		cdw_dropdown_set_current_item_by_ind(dropdown, 0);
		return dropdown;
	}
}





/**
   \brief Create dropdown with values showing allowed disc modes

   Function returns dropdown with items presenting allowed (and supported
   by cdw) modes of writing to current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(cdw_task_t *task)
{
	/* _all_ possible types of writing mode allowed for any optical disc */
	char *mode_labels[CDW_DISC_MODE_N_MAX];

	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_UNSPECIFIED] = _("Unspecified");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_TAO] = _("TAO");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_DAO] = _("DAO");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_SAO] = _("SAO");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_RAW] = _("RAW");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_RAW96P] = _("RAW96R");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_RAW96R] = _("RAW96P");
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	mode_labels[CDW_DISC_MODE_RAW16] = _("RAW16");
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	mode_labels[CDW_DISC_MODE_ERROR] = _("(ERROR)");

	int n_items = 0;
	for (n_items = 0; n_items < CDW_DISC_MODE_N_MAX; n_items++) {
		/* disc_modes is non-sparse up until
		   element of value CDW_DISC_MODE_INIT */
		if (task->burn.disc_modes[n_items] == CDW_DISC_MODE_INIT) {
			break;
		}
	}

	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  disc_mode_label_row + 1,
						  first_col + 2,
						  wide_dropdown_n_cols,
						  n_items, /* n_items = n_items_max */
						  CDW_COLORS_DIALOG);
	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < n_items; i++) {
		/* this smart line selects only disc modes allowed for current disc */
		int id = task->burn.disc_modes[i];
		cdw_assert (id != CDW_DISC_MODE_INIT, "ERROR: found INIT in allowed disc modes for i = %d\n", (int) i);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, id, mode_labels[id]);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add item #%d with id = %d and label = \"%s\"\n",
				 i, id, mode_labels[id]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		} else {
			cdw_vdm ("INFO: disc mode dropdown label #%d = \"%s\"\n", i, cdw_dropdown_get_label_by_ind(dropdown, i));
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		/* 0 - first mode (set by "task" module) is the default one */
		cdw_dropdown_set_current_item_by_ind(dropdown, 0);
		return dropdown;
	}
}





/**
   \brief Create dropdown with values showing allowed write speeds

   Function returns dropdown with items presenting write speeds acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param disc - variable describing current disc

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(cdw_disc_t *disc)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(wizard.cdw_form->subwindow,
						  write_speed_label_row + 1,
						  first_col + 2,
						  narrow_dropdown_n_cols,
						  disc->write_speeds.n_speeds,  /* n_items_max */
						  CDW_COLORS_DIALOG);

	if (dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	char label[5];
	for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
		snprintf(label, 4 + 1, "%d", disc->write_speeds.speeds[i]);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, disc->write_speeds.speeds[i], label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create dropdown label #%d of value %d\n", i, disc->write_speeds.speeds[i]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		/* the function returns index of speed based on wizard.speed_range;
		   cdw_disc module guarantees it to be >= 0 */
		int speed_index = cdw_disc_get_initial_write_speed_index(disc);
		cdw_dropdown_set_current_item_by_ind(dropdown, speed_index);
		return dropdown;
	}
}





/**
   \brief Deallocate all resources allocated by write wizard
*/
void cdw_write_wizard_destroy(void)
{
	/* buttons cleanup code */
	cdw_button_delete(&(wizard.config_button));
	cdw_button_delete(&(wizard.write_button));
	cdw_button_delete(&(wizard.cancel_button));
	cdw_assert (wizard.config_button == (CDW_BUTTON *) NULL, "ERROR: delete() didn't set button to NULL\n");
	cdw_assert (wizard.write_button == (CDW_BUTTON *) NULL, "ERROR: delete() didn't set button to NULL\n");
	cdw_assert (wizard.config_button == (CDW_BUTTON *) NULL, "ERROR: delete() didn't set button to NULL\n");

	cdw_dropdown_delete(&(wizard.write_speed_dropdown));
	cdw_dropdown_delete(&(wizard.session_mode_dropdown));
	cdw_dropdown_delete(&(wizard.disc_mode_dropdown));
	cdw_assert (wizard.write_speed_dropdown == (CDW_DROPDOWN *) NULL, "ERROR: delete() didn't set pointer to NULL\n");
	cdw_assert (wizard.session_mode_dropdown == (CDW_DROPDOWN *) NULL, "ERROR: delete() didn't set pointer to NULL\n");
	cdw_assert (wizard.disc_mode_dropdown == (CDW_DROPDOWN *) NULL, "ERROR: delete() didn't set pointer to NULL\n");

	if (wizard.cdw_form->window != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->window);
		wizard.cdw_form->window = (WINDOW *) NULL;
	}

	if (wizard.cdw_form->subwindow != (WINDOW *) NULL) {
		delwin(wizard.cdw_form->subwindow);
		wizard.cdw_form->subwindow = (WINDOW *) NULL;
	}

	if (wizard.cdw_form != (cdw_form_t *) NULL) {
		cdw_form_delete(&(wizard.cdw_form));
	}

	return;
}





/**
   \brief Print to stderr values of \p task fields set in write wizard

   \param task - task variable, in which some fields were set in write wizard
*/
void cdw_write_wizard_debug_at_exit(cdw_task_t *task)
{
	/* use variables, to which wizard saves result at return,
	   not values from dropdowns/checkboxes */
	cdw_vdm ("INFO: verify burning = \"%s\"\n", task->burn.verify ? "true" : "false");
	cdw_vdm ("INFO:   session mode = \"%s\"\n", cdw_task_get_session_mode_label(task->burn.session_mode));
	cdw_vdm ("INFO:      disc mode = \"%s\"\n", cdw_task_get_disc_mode_label(task->burn.disc_mode));
	cdw_vdm ("INFO:          speed = \"%d\"\n", task->burn.speed);

	return;
}





bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc)
{
	bool disc_blank = disc->state_empty == CDW_TRUE;
	bool disc_overwritable = disc->type == CDW_DVD_RWP
		|| disc->type == CDW_DVD_RW_RES;

	if ((disc_blank || disc_overwritable)
	    && task->id == CDW_TASK_BURN_FROM_IMAGE
	    && cdw_ext_tools_is_tool_available(CDW_TOOL_MD5SUM)
	    && task->verify.tool.id == CDW_TOOL_MD5SUM) {

		return true;
	} else {
		return false;
	}
}





/**
   \brief Print to stderr information why "verify" checkbox is not visible

   If some conditions are met, write wizard may display "verify write"
   checkbox. This function should be called if any of the conditions is
   not met to investigate these conditions and print to stderr information
   which of them was not met. This is a debug function

   \param task - variable describing current task
   \param disc - variable describing current disc
*/
void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->state_empty != CDW_TRUE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not blank (according to cdio)\n");
	}
	if (! (disc->type == CDW_DVD_RWP || disc->type == CDW_DVD_RW_RES)) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not DVD+RW or DVD-RW RES\n");
	}
	if (task->id != CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because task is not CDW_TASK_BURN_FROM_IMAGE\n");
	}
	if (! cdw_ext_tools_is_tool_available(CDW_TOOL_MD5SUM)) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because there is no md5sum tool available\n");
	}
	if (task->verify.tool.id != CDW_TOOL_MD5SUM) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because tool is not set as MD5SUM\n");
	}

	return;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc)
{
	int begin_y = ((LINES - window_n_lines) / 2) - 2;
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window = cdw_ncurses_window_new((WINDOW *) NULL,
							 window_n_lines, window_n_cols,
							 begin_y, begin_x,
							 CDW_COLORS_DIALOG,
							 /* 2TRANS: this is title of wizard window; 'write' as in
							    'writing to optical disc */
							 _("Write wizard"),
							 /* 2TRANS: this is tip at the bottom of window - user can
							    switch between window elements using tab key */
							 _("Use 'Tab' key to move"));
	if (wizard.cdw_form->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->subwindow = cdw_ncurses_window_new(wizard.cdw_form->window,
							    subwindow_n_lines, subwindow_n_cols,
							    3, 1,
							    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (wizard.cdw_form->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_GEN_ERROR;
	}


	/* *** main message in wizard window *** */
	mvwprintw(wizard.cdw_form->window, top_label_row, first_col,
		  /* 2TRANS: this is message to user: current action will
		     be writing to %s, where %s is DVD or CD */
		  _("You are writing to %s disc"), disc->simple_type_label);

	task->burn.verify = false;

	cdw_form_descr_t descr[] = {
		/*     type             begin_y                        begin_x         n_cols                n_lines    field enum      data1                  data2 */

		/* 2TRANS: this is a label in write wizard, after which a writing speed selection list will be displayed */
		{ CDW_WIDGET_LABEL,     write_speed_label_row,       first_col,      wide_label_n_cols,          1,    f_speed_l,   _("Writing speed:"),         0 },
		{ CDW_WIDGET_DROPDOWN,  write_speed_label_row + 1,   first_col + 3,  narrow_dropdown_n_cols - 2, 1,    f_speed_i,   (void *) NULL,               0 },

		/* 2TRANS: this is a label in write wizard, after which a session writing mode selection list will be displayed */
		{ CDW_WIDGET_LABEL,     session_mode_label_row,      first_col,      wide_label_n_cols,          1,    f_session_l, _("Session writing mode:"),  0 },
		{ CDW_WIDGET_DROPDOWN,  session_mode_label_row + 1,  first_col + 3,  wide_dropdown_n_cols - 2,   1,    f_session_i, (void *) NULL,               0 },

		/* 2TRANS: this is a label in write wizard, after which a disc writing mode selection list will be displayed */
		{ CDW_WIDGET_LABEL,     disc_mode_label_row,         first_col,      wide_label_n_cols,          1,    f_disc_l,    _("Disc writing mode:"),     0 },
		{ CDW_WIDGET_DROPDOWN,  disc_mode_label_row + 1,     first_col + 3,  wide_dropdown_n_cols - 2,   1,    f_disc_i,    (void *) NULL,               0 },

		/* 2TRANS: this is label next to checkbox; marked checkbox enables verification of correctness of writing to CD; this feature is experimental */
		{ CDW_WIDGET_LABEL,     verify_checkbox_label_row,   first_col + 4,  narrow_label_n_cols,        1,    f_verify_l,  _("Verify write (experimental)"),        0 },
		{ CDW_WIDGET_CHECKBOX,  verify_checkbox_label_row,   first_col + 1,  1,                          1,    f_verify_i,  (void *) NULL,   task->burn.verify ? 1 : 0 },

		/* 2TRANS: other options/preferences - it refers to button on which user can click to open configuration window */
		{ CDW_WIDGET_LABEL,     config_label_row,            first_col,      wide_label_n_cols,          1,    f_config_l,  _("Other options:"),         0 },
		{ CDW_WIDGET_BUTTON,    config_label_row + 1,        first_col + 2,  2,                          1,    f_config_i,  (void *) NULL,               0 },

		{ CDW_WIDGET_BUTTON,    buttons_row,                 3,              2,                          1,    f_write_i,   (void *) NULL,               0 },
		{ CDW_WIDGET_BUTTON,    buttons_row,                 15,             2,                          1,    f_cancel_i,  (void *) NULL,               0 },

		/* guard */
		{ -1,                   0,                           0,              0,                          0,    0,           (void *) NULL,               0 }};

	/* this function also sets which dropdown item is selected at the beginning */
	wizard.write_speed_dropdown = cdw_write_wizard_make_speed_dropdown(disc);
	if (wizard.write_speed_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create speed dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_speed_i].data1 = (void *) wizard.write_speed_dropdown;
	}

	/* this function also sets which dropdown item is selected at the beginning */
	wizard.session_mode_dropdown = cdw_write_wizard_make_session_mode_dropdown(task);
	if (wizard.session_mode_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create session mode dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_session_i].data1 = (void *) wizard.session_mode_dropdown;
	}
	/* this function also sets which dropdown item is selected at the beginning */
	wizard.disc_mode_dropdown = cdw_write_wizard_make_disc_mode_dropdown(task);
	if (wizard.disc_mode_dropdown == (CDW_DROPDOWN *) NULL) {
		cdw_vdm ("ERROR: failed to create session mode dropdown\n");
		return CDW_GEN_ERROR;
	} else {
		descr[f_disc_i].data1 = (void *) wizard.disc_mode_dropdown;
	}

	wizard.config_button = cdw_button_new(wizard.cdw_form->subwindow, (size_t) config_label_row + 1,
					      (size_t) first_col + 2,
					      /* 2TRANS: button label; this is the
						 same string as label of button in
						 menu of main ui window */
					      _("Configuration"), CDW_COLORS_DIALOG);
	descr[f_config_i].data1 = (void *) wizard.config_button;
	/* *** buttons at the bottom *** */
	wizard.write_button = cdw_button_new(wizard.cdw_form->subwindow, buttons_row, 3,
					     /* 2TRANS: button label, it refers to writing to optical disc */
					     _("Write"), CDW_COLORS_DIALOG);
	descr[f_write_i].data1 = (void *) wizard.write_button;

	wizard.cancel_button = cdw_button_new(wizard.cdw_form->subwindow, buttons_row, 15,
					      /* 2TRANS: button label */
					      _("Cancel"), CDW_COLORS_DIALOG);
	descr[f_cancel_i].data1 = (void *) wizard.cancel_button;

	wizard.cdw_form->n_fields = CDW_WRITE_WIZARD_N_FIELDS;
	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_GEN_ERROR;
	}

	wizard.cdw_form->form = cdw_form_new_form(wizard.cdw_form->window,
						  wizard.cdw_form->subwindow,
						  wizard.cdw_form->fields);
	if (wizard.cdw_form->form == (FORM *) NULL) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_GEN_ERROR;
	}

#if 1 /* 1 - show 'verify' checkbox using program's logic;
	 0 - force showing 'verify' checkbox */
	if (cdw_write_wizard_show_verify_checkbox(task, disc)) { /* this last one is added just in case */

		wizard.verify_checkbox.visible = true;
		/* checkbox should be unchecked by default */
		wizard.verify_checkbox.checked = false;
		cdw_put_ckbox(wizard.cdw_form->subwindow, verify_checkbox_label_row, first_col + 1);

	} else {
		/* invisible AND inactive */
		field_opts_off(wizard.cdw_form->fields[f_verify_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_verify_l], O_VISIBLE);

		wizard.verify_checkbox.visible = false;
#ifndef NDEBUG
		cdw_write_wizard_debug_verify_invisible(task, disc);
#endif
	}
#else   /* force showing checkbox */
	wizard.verify_checkbox.visible = true;
	cdw_put_ckbox(wizard.cdw_form->subwindow, verify_checkbox_label_row, first_col + 1);
#endif

	/* current items are set in functions creating respective
	   dropdowns; now just display them */
	cdw_dropdown_display_current_item(wizard.write_speed_dropdown);
	cdw_dropdown_display_current_item(wizard.session_mode_dropdown);
	cdw_dropdown_display_current_item(wizard.disc_mode_dropdown);

	cdw_button_unfocus(wizard.config_button);
	cdw_button_unfocus(wizard.write_button);
	cdw_button_unfocus(wizard.cancel_button);

	cdw_form_add_return_char(wizard.cdw_form, CDW_KEY_ENTER);
	cdw_form_add_return_char(wizard.cdw_form, CDW_KEY_ESCAPE);

	wizard.cdw_form->handle_enter = cdw_write_wizard_driver_handle_enter;

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





/**
   \brief Check which element has currently keyboard focus, and handle Enter key for this element

   Function checks which element in wizard window (identified by field
   index \p fi) has currently keyboard focus and executes related code that
   performs action appropriate for given element.

   "elements" are buttons and dropdowns. Checkboxes don't "recognize" Enter key.

   \param cdw_form - form in which elements are placed
   \param fi - field index, index of form field that has keyboard focus
   \param data - pointer to additional data that may be used by some elements
*/
void cdw_write_wizard_driver_handle_enter(cdw_form_t *cdw_form, int fi, __attribute__((unused)) void *data)
{
	if (cdw_form->is_button_field_index(cdw_form, fi)) {
		if (fi == f_config_i) {
			cdw_config_ui_window(false);
			redrawwin(cdw_form->window);
			redrawwin(cdw_form->subwindow);
			wrefresh(cdw_form->window);
			wrefresh(cdw_form->subwindow);
		}
	} else if (cdw_form->is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form->get_dropdown_from_form(cdw_form, fi);
		if (dropdown != (CDW_DROPDOWN *) NULL) {
			cdw_rv_t crv = cdw_dropdown_driver(dropdown);
			if (crv == CDW_OK) {
				const char *label = cdw_dropdown_get_current_item_label(dropdown);
				FIELD **fields = form_fields(cdw_form->form);
				/* setting buffer makes field to display
				   string, but slightly moved to left */
				set_field_buffer(*(fields + fi), 0, label);

				/* call to cdw_dropdown_focus() fixes it */
				cdw_dropdown_focus(dropdown);

				form_driver(cdw_form->form, REQ_VALIDATION);
			} /* else field buffer is not updated and nothing changes */
		}
	} else {
		;
	}

	return;
}

