/* 

        Copyright (C) 1995-2000
        Free Software Foundation, Inc.

   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version. 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/

/*****************************************************************************/
/*                                                                           */
/* File: methods.c                                                           */
/*                                                                           */
/*****************************************************************************/

#include "cf.defs.h"
#include "cf.extern.h"

/* Create package with host:ptrr->name as filename
   Includes TTL - should be in filename, since file encrypted
       Filename of module, relative to modulesdir
       Prototypes for checking integrity
       Checksums of attachmeChilnts
       Encrypted

control:

  MethodName = ( "hostwild::name" )
  MethodAccessList = ( regex.. )
  MethodParameters = ()
       
alerts:

  MethodReturn($a,$b...)
  MethodClasses(class2,class5) # return if defined
  
*/

/*****************************************************************************/

void DispatchNewMethod(ptr)

struct Method *ptr;

{ struct Item *ip;
  char label[bufsize]; 
  char ipaddress[64];
  unsigned char digest[EVP_MAX_MD_SIZE+1];
  
for (ip = ptr->servers; ip != NULL; ip=ip->next)
   {
   strncpy(ipaddress,Hostname2IPString(ip->name),63);
   Verbose("Server was (%s)\n",ipaddress);
   
   if (strcmp(ip->name,"localhost") == 0 || strcmp(ip->name,"::1") == 0 || strcmp(ip->name,"127.0.0.1"))
      {
      Verbose("\nDispatch method localhost:%s to %s/rpc_in\n",ptr->name,VLOCKDIR);

      ptr->invitation = 'y';
/*      snprintf(label,bufsize-1,"%s/rpc_in/localhost_localhost_%s",VLOCKDIR,ptr->name);*/

      ChecksumList(ptr->send_args,digest,'m');
      snprintf(label,bufsize-1,"%s/rpc_in/localhost_localhost_%s_%s",VLOCKDIR,ptr->name,ChecksumPrint('m',digest));
      EncapsulateMethod(ptr,label);
      }
   else
      {
      Verbose("\nDispatch method %s:%s to %s/rpc_out\n",ip->name,ptr->name,VLOCKDIR);
      Verbose("No yet implemented\n");

      /* Change callinghost etc into IP addresses for uniqueness , with Hostname2IPString(hostname) */
      
      snprintf(label,bufsize-1,"%s/rpc_out/callinghost_myhostname_%s",VLOCKDIR,ptr->name);
      
      /*Must authenticate ID of remote host and add this info to our local cache*/
      }
   } 
}

/*****************************************************************************/

struct Item *GetPendingMethods(state)

int state;

{ char filename[maxvarsize],path[bufsize],name[bufsize];
  char client[bufsize],server[bufsize],extra[bufsize],digeststring[bufsize];
  DIR *dirh;
  struct dirent *dirp;
  struct Method *mp;
  struct Item *list = NULL, *ip;
  struct stat statbuf;

/* HERE:

   For each host we collaborate with
      { 
      copy any files starting "rhostname_myhostname_" from hostname by secured copy
      }
*/
  
Debug("GetPendingMethods(%d) in (%s/rpc_in)\n",state,VLOCKDIR);
 
snprintf(filename,maxvarsize-1,"%s/rpc_in",VLOCKDIR);
 
if ((dirh = opendir(filename)) == NULL)
   {
   snprintf(OUTPUT,bufsize*2,"Can't open directory %s\n",filename);
   CfLog(cfverbose,OUTPUT,"opendir");
   return NULL;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (!SensibleFile(dirp->d_name,filename,NULL))
      {
      continue;
      }

   snprintf(path,bufsize-1,"%s/%s",filename,dirp->d_name);

   if (stat(path,&statbuf) == -1)
      {
      continue;
      }

   if (statbuf.st_mtime > CFSTARTTIME + (VEXPIREAFTER * 60))
      {
      Verbose("Purging expired method %s\n",path);
      unlink(path);
      continue;
      }
   
   SplitMethodName(dirp->d_name,client,server,name,digeststring,extra);

   Verbose("Looking at method (%s) from (%s) intended for exec on (%s) with arghash %s\n",name,client,server,digeststring);
   if (strlen(name) == 0)
      {
      snprintf(OUTPUT,bufsize,"Unable to extract method name from package (%s)",dirp->d_name);
      CfLog(cfinform,OUTPUT,"");
      continue;
      }

   if (strlen(extra) != 0)
      {
      if (state == cfmethodreply)
	 {
	 Debug("Found an attachment (%s) in reply to %s\n",extra,name);
	 }
      else
	 {
	 Debug("Found an attachment (%s) to incoming method %s\n",extra,name);
	 }
      continue;
      }

   if ((state == cfmethodreply) && (strstr(name,":Reply") == 0))
      {
      Debug("Ignoring bundle (%s) from waiting function call\n",name);
      continue;
      }
   
   if ((state == cfmethodexec) && (strstr(name,":Reply")))
      {
      Debug("Ignoring bundle (%s) from waiting reply\n",name);
      continue;
      }

   ip = SplitStringAsItemList(name,':');
   
   if (mp = IsDefinedMethod(ip->name,digeststring))
      {
      if (IsItemIn(mp->servers,"localhost") || IsItemIn(mp->servers,VIPADDRESS))
	 {
	 if (state == cfmethodreply)
	    {
	    Verbose("Found a local approval to forward reply from local method (%s) to final destination sender %s\n",name,client);
	    }
	 else
	    {
	    Verbose("Found a local approval to execute this method (%s) on behalf of sender %s\n",name,client);
	    }
	 AppendItem(&list,dirp->d_name,"");
	 }
      else
	 {
	 Verbose("No local approval in methods list on this host for received request (%s)\n",dirp->d_name);
	 }
      }

   DeleteItemList(ip);
   }
 
closedir(dirh);

if (list == NULL)
   {
   return NULL;
   }
 
return list;
}


/*****************************************************************************/

void EvaluatePendingMethod(methodname)

char *methodname;

 /* Actually execute any method that is our responsibilty */

{ struct Method *ptr;
  char buffer[bufsize],line[bufsize];
  char options[32];
  char client[bufsize],server[bufsize],name[bufsize],digeststring[bufsize],extra[bufsize];
  char *sp;
  char execstr[bufsize];
  int print;
  mode_t maskval = 0;
  FILE *pp;

SplitMethodName(methodname,client,server,name,digeststring,extra);

/* Could check client for access granted? */

if (strcmp(server,"localhost") == 0 || strcmp(server,VIPADDRESS) == 0)
   {
   }
else
   {
   Debug("Nothing to do for %s\n",methodname);
   return;
   }
 
Debug("EvaluatePendingMethod(%s)\n",name);

options[0] = '\0';

if (INFORM)
   {
   strcat(options,"-I ");
   }

if (VERBOSE)
   {
   strcat(options,"-v ");
   }

if (DEBUG || D2)
   {
   strcat(options,"-d2 ");
   }

ptr = IsDefinedMethod(name,digeststring);

strcat(options,"-Z ");
strcat(options,digeststring);
strcat(options," ");
 
snprintf(execstr,bufsize-1,"%s/bin/cfagent -f %s %s",WORKDIR,GetMethodFilename(ptr),options);

Verbose("Trying %s\n",execstr);
 
if (IsExcluded(ptr->classes))
   {
   return;
   }
 
ResetOutputRoute(ptr->log,ptr->inform);

if (!GetLock(ASUniqueName("method-exec"),CanonifyName(ptr->name),ptr->ifelapsed,ptr->expireafter,VUQNAME,CFSTARTTIME))
   {
   return;
   }
 
snprintf(OUTPUT,bufsize*2,"Executing method %s...(uid=%d,gid=%d)\n",execstr,ptr->uid,ptr->gid);
CfLog(cfinform,OUTPUT,"");
 
if (DONTDO)
   {
   printf("%s: execute method %s\n",VPREFIX,ptr->name);
   }
else
   {    
   switch (ptr->useshell)
      {
      case 'y':  pp = cfpopen_shsetuid(execstr,"r",ptr->uid,ptr->gid,ptr->chdir,ptr->chroot);
	  break;
      default:   pp = cfpopensetuid(execstr,"r",ptr->uid,ptr->gid,ptr->chdir,ptr->chroot);
	  break;	     
      }
   
   if (pp == NULL)
      {
      snprintf(OUTPUT,bufsize*2,"Couldn't open pipe to command %s\n",execstr);
      CfLog(cferror,OUTPUT,"popen");
      ResetOutputRoute('d','d');
      ReleaseCurrentLock();
      return;
      } 
   
   while (!feof(pp))
      {
      if (ferror(pp))  /* abortable */
	 {
	 snprintf(OUTPUT,bufsize*2,"Shell command pipe %s\n",execstr);
	 CfLog(cferror,OUTPUT,"ferror");
	 break;
	 }
      
      ReadLine(line,bufsize,pp);
      
      if (strstr(line,"cfengine-die"))
	 {
	 break;
	 }
      
      if (ferror(pp))  /* abortable */
	 {
	 snprintf(OUTPUT,bufsize*2,"Shell command pipe %s\n",execstr);
	 CfLog(cferror,OUTPUT,"ferror");
	 break;
	 }
      
      
      print = false;
      
      for (sp = line; *sp != '\0'; sp++)
	 {
	 if (! isspace((int)*sp))
	    {
	    print = true;
	    break;
	    }
	 }
      
      if (print)
	 {
	 printf("%s:%s: %s\n",VPREFIX,ptr->name,line);
	 }
      }
   
   cfpclose(pp);  
   }
  
snprintf(OUTPUT,bufsize*2,"Finished local method %s processing\n",execstr);
CfLog(cfinform,OUTPUT,"");
 
ResetOutputRoute('d','d');
ReleaseCurrentLock();
 
return;
}


/*****************************************************************************/
/* Level 2                                                                   */
/*****************************************************************************/

int ParentLoadReplyPackage(methodname)

char *methodname;

{ char client[maxvarsize],server[bufsize],name[bufsize],buf[bufsize];
  char line[bufsize],type[64],arg[bufsize], **methodargv = NULL;
  char basepackage[bufsize],cffilename[bufsize],digeststring[bufsize],extra[maxvarsize];
  unsigned char digest[EVP_MAX_MD_SIZE+1];
  char *sp;
  struct Item *methodargs,*ip;
  struct Method *mp;
  int isreply = false, argnum = 0, i, methodargc = 0;
  FILE *fp,*fin,*fout;
  
Debug("\nParentLoadReplyPackage(%s)\n\nLook for method replies in %s/rpc_in\n",methodname,VLOCKDIR);

SplitMethodName(methodname,client,server,name,digeststring,extra);

for (sp = name; *sp != '\0'; sp++)
   {
   if (*sp == ':') /* Strip off :Reply */
      {
      *sp = '\0';
      break;
      }
   }
 
if ((mp = IsDefinedMethod(name,digeststring)) == NULL)
   {
   return false;
   }

 
i = 1;
   
for (ip = mp->return_vars; ip != NULL; ip=ip->next)
   {
   i++;
   }
 
 methodargv = (char **)  malloc(i*sizeof(char *));
 
 i = 0;
 
 for (ip = mp->return_vars; ip != NULL; ip=ip->next)
    {
    /* Fill this temporarily with the formal parameters */
    methodargv[i++] = ip->name; 
    }
 
methodargc = i;
 
snprintf(basepackage,bufsize-1,"%s/rpc_in/%s",VLOCKDIR,methodname);

ChecksumList(mp->send_args,digest,'m');
  
argnum = CountAttachments(basepackage);
 
if (argnum != methodargc)
   {
   Verbose("Number of return arguments (%d) does not match template (%d)\n",argnum,methodargc);
   Verbose("Discarding method %s\n",name);
   free((char *)methodargv); 
   return false;
   }
 
argnum = 0;
Debug("Opening bundle %s\n",methodname);
 
if ((fp = fopen(basepackage,"r")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could not open a parameter bundle (%s) for method %s",basepackage,name);
   CfLog(cfinform,OUTPUT,"fopen");
   free((char *)methodargv); 
   return false;
   }
 
 while (!feof(fp))
    {
    bzero(line,bufsize);
    fgets(line,bufsize,fp);
    type[0] = '\0';
    arg[0] = '\0';
    
    sscanf(line,"%63s %s",type,arg);
    
    switch (ConvertMethodProto(type))
       {
       case cfmeth_name:
	   if (strncmp(arg,name,strlen(name)) != 0)
	      {
	      snprintf(OUTPUT,bufsize,"Method name %s did not match package name %s",name,arg);
	      CfLog(cfinform,OUTPUT,"");
	      fclose(fp);
	      free((char *)methodargv); 
	      return false;
	      }
	   break;
	   
       case cfmeth_isreply:
	   isreply = true;
	   break;
	   
       case cfmeth_sendclass:
	   
	   if (IsItemIn(mp->return_classes,arg))
	      {
	      AddPrefixedMultipleClasses(name,arg);
	      }
	   else
	      {
	      Verbose("Method returned a class (%s) that was not in the return_classes access list\n",arg);
	      }
	   break;
	   
       case cfmeth_attacharg:
	   
	   if (methodargv[argnum][0] == '/')
	      {
	      int val;
	      struct stat statbuf;
	      FILE *fp,*fin,*fout;
	      
	      if (stat (arg,&statbuf) == -1)
		 {
		 Verbose("Unable to stat file %s\n",arg);
		 return false;		
		 }
	      
	      val = statbuf.st_size;
	      
	      Debug("Install file in %s at %s\n",arg,methodargv[argnum]);
	      
	      if ((fin = fopen(arg,"r")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could open argument bundle %s\n",arg);
		 CfLog(cferror,OUTPUT,"fopen");
		 return false;
		 }
	      
	      if ((fout = fopen(methodargv[argnum],"w")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could not write to local parameter file %s\n",methodargv[argnum]);
		 CfLog(cferror,OUTPUT,"fopen");
		 fclose(fin);
		 return false;
		 }
	      
	      if (val > bufsize)
		 {
		 int count = 0;
		 while(!feof(fin))
		    {
		    fputc(fgetc(fin),fout);
		    count++;
		    if (count == val)
		       {
		       break;
		       }
		    Debug("Wrote #\n");
		    }
		 }
	      else
		 {
		 bzero(buf,bufsize); 
		 fread(buf,val,1,fin);
		 fwrite(buf,val,1,fout);
		 Debug("Wrote #\n");
		 }
	      
	      fclose(fin);
	      fclose(fout);	     
	      }
	   else
	      {
	      char argbuf[bufsize],newname[maxvarsize];
	      bzero(argbuf,bufsize);
	      
	      Debug("Read arg from file %s\n",arg);
	      
	      if ((fin = fopen(arg,"r")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Missing module argument package (%s)\n",arg);
		 FatalError(OUTPUT);
		 }
	      
	      fread(argbuf,bufsize-1,1,fin);
	      fclose(fin);
	      
	      snprintf(newname,maxvarsize-1,"%s.%s",name,methodargv[argnum]);
	      Debug("Setting variable %s to %s\n",newname,argbuf);
	      AddMacroValue(CONTEXTID,newname,argbuf);
	      }
	   
	   unlink(arg);
	   argnum++;
	   
	   break;
	   
       default:
	   break;
       }
    }
 
 fclose(fp); 
 free((char *)methodargv); 
  
 unlink(basepackage);   /* Now remove the invitation */ 
 
 if (!isreply)
    {
    Verbose("Reply package was not marked as a reply\n");
    return false;
    }
 
 return true; 
}


/*****************************************************************************/

int ChildLoadMethodPackage(name,digeststring)

char *name,*digeststring;

{ char line[bufsize],type[64],arg[bufsize];
  char basepackage[bufsize],cffilename[bufsize],buf[bufsize];
  struct stat statbuf;
  FILE *fp,*fin,*fout;
  int argnum = 0, gotmethod = false;

Debug("Look for all incoming methods in %s/rpc_in\n",VLOCKDIR);

METHODFILENAME[0] = '\0'; 
 
if (!CheckForMethodPackage(name))
   {
   Verbose("Could not find anything looking like a current invitation\n");
   return false;
   }

snprintf(basepackage,bufsize-1,"%s/rpc_in/localhost_localhost_%s_%s",VLOCKDIR,name,digeststring);

argnum = CountAttachments(basepackage);
 
if (argnum != METHODARGC)
   {
   Verbose("Number of return arguments (%d) does not match template (%d)\n",argnum,METHODARGC);
   Verbose("Discarding method %s\n",name);
   return false;
   }

argnum = 0; 
 
if ((fp = fopen(basepackage,"r")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could not open a parameter bundle (%s) for method %s",basepackage,name);
   CfLog(cfinform,OUTPUT,"fopen");
   return false;
   }
 
while (!feof(fp))
   {
   bzero(line,bufsize);
   fgets(line,bufsize,fp);
   type[0] = '\0';
   arg[0] = '\0';
   
   sscanf(line,"%63s %s",type,arg);
   
   switch (ConvertMethodProto(type))
      {
      case cfmeth_name:
	  if (strcmp(arg,name) != 0)
	     {
	     snprintf(OUTPUT,bufsize,"Method name %s did not match package name %s",name,arg);
	     CfLog(cfinform,OUTPUT,"");
	     fclose(fp);
	     return false;
	     }
	  break;
	  
      case cfmeth_file:
	  
	  bzero(VBUFF,bufsize);
	  
	  if (GetMacroValue(CONTEXTID,"moduledirectory"))
	     {
	     ExpandVarstring("$(moduledirectory)",VBUFF,NULL);
	     }
	  else
	     {
	     snprintf(VBUFF,bufsize,"%s/modules",VLOCKDIR);
	     }
	  
	  snprintf(cffilename,bufsize-1,"%s/%s",VBUFF,arg);
	  
	  if (lstat(cffilename,&statbuf) != -1)
	     {
	     if (S_ISLNK(statbuf.st_mode))
		{
		snprintf(OUTPUT,bufsize,"SECURITY ALERT. Method (%s) executable is a symbolic link",name);
		CfLog(cferror,OUTPUT,"");
		fclose(fp);
		continue;
		}
	     }
	  
	  if (stat(cffilename,&statbuf) == -1)
	     {
	     snprintf(OUTPUT,bufsize,"Method name %s did not find executable name %s",name,cffilename);
	     CfLog(cfinform,OUTPUT,"stat");
	     fclose(fp);
	     continue;
	     }
	  
	  strncpy(METHODFILENAME,cffilename,maxvarsize);
	  break;
	  
      case cfmeth_replyto:
	  
	  Debug("Found reply to host: %s\n",arg);
	  strncpy(METHODREPLYTO,arg,maxvarsize-1);
	  break;
	  
      case cfmeth_sendclass:
	  Debug("Defining class: %s\n",arg);
	  strncpy(METHODRETURNCLASSES,arg,maxvarsize-1);
	  break;
	  
      case cfmeth_attacharg:

	  Debug("Handling expected local argument (%s)\n",METHODARGV[argnum]);

	  if (METHODARGV[argnum][0] == '/')
	     {
	     int val;
	     struct stat statbuf;

	     if (stat (arg,&statbuf) == -1)
		{
		Verbose("Unable to stat file %s\n",arg);
		return false;		
		}
	     
	     val = statbuf.st_size;
	     
	     Debug("Install file in %s at %s\n",arg,METHODARGV[argnum]);

	      if ((fin = fopen(arg,"r")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could open argument bundle %s\n",arg);
		 CfLog(cferror,OUTPUT,"fopen");
		 return false;
		 }

	     if ((fout = fopen(METHODARGV[argnum],"w")) == NULL)
		{
		snprintf(OUTPUT,bufsize,"Could not write to local parameter file %s\n",METHODARGV[argnum]);
		CfLog(cferror,OUTPUT,"fopen");
		fclose(fin);
		return false;
		}
	     
	     if (val > bufsize)
		{
		int count = 0;
		while(!feof(fin))
		   {
		    fputc(fgetc(fin),fout);
		    count++;
		    if (count == val)
		       {
		       break;
		       }
		    Debug("Wrote #\n");
		   }
		}
	     else
		{
		bzero(buf,bufsize); 
		fread(buf,val,1,fin);
		fwrite(buf,val,1,fout);
		Debug("Wrote #\n");
		}
	     
	     fclose(fin);
	     fclose(fout);	     
	     }
	  else
	     {
	     char argbuf[bufsize];
	     bzero(argbuf,bufsize);

	     Debug("Read arg from file %s\n",arg);
	     
	     if ((fin = fopen(arg,"r")) == NULL)
		{
		snprintf(OUTPUT,bufsize,"Missing module argument package (%s)\n",arg);
		FatalError(OUTPUT);
		}

	     fread(argbuf,bufsize-1,1,fin);
	     fclose(fin);

	     if (strcmp(METHODARGV[argnum],"void") != 0)
		{
		Debug("Setting variable %s to %s\n",METHODARGV[argnum],argbuf);
		AddMacroValue(CONTEXTID,METHODARGV[argnum],argbuf);
		}
	     else
		{
		Verbose("Method had no arguments (void)\n");
		}
	     }

	  unlink(arg);
	  argnum++;
	  break;
	  
      default:
	  break;
      }
   } 

fclose(fp);

if (strlen(METHODFILENAME) == 0)
   {
   snprintf(OUTPUT,bufsize,"Method (%s) package had no filename to execute",name);
   CfLog(cfinform,OUTPUT,"");
   return false;
   }
 
if (strlen(METHODREPLYTO) == 0)
   {
   snprintf(OUTPUT,bufsize,"Method (%s) package had no replyto address",name);
   CfLog(cfinform,OUTPUT,"");
   return false;
   }
 
unlink(basepackage);   /* Now remove the invitation */ 
return true; 
}

/*****************************************************************************/

void DispatchMethodReply()

{ struct Item *ip;
  char label[bufsize]; 
  char ipaddress[64];

Verbose("DispatchMethodReply()\n\n");

/* strncpy(ipaddress,Hostname2IPString(METHODREPLYTO),63);*/
 
 if (strcmp(METHODREPLYTO,"localhost") == 0 || strcmp(METHODREPLYTO,"::1") == 0 || strcmp(METHODREPLYTO,"127.0.0.1"))
    {    
    snprintf(label,bufsize-1,"%s/rpc_in/localhost_localhost_%s:Reply_%s",VLOCKDIR,METHODNAME,METHODMD5);
    EncapsulateReply(label);
    }
 else
    {
    Verbose("\nDispatch method %s:%s to %s/rpc_out\n",METHODREPLYTO,METHODNAME,VLOCKDIR);
    Verbose("No yet implemented\n");
    
    /* Change callinghost etc into IP addresses for uniqueness , with Hostname2IPString(hostname) */
    
    snprintf(label,bufsize-1,"%s/rpc_out/callinghost_myhostname_%s:Reply",VLOCKDIR,METHODNAME);
    
    /*Must authenticate ID of remote host and add this info to our local cache*/
    }
}


/*****************************************************************************/

char *GetMethodFilename(ptr)

struct Method *ptr;

{ char line[bufsize],type[64],arg[bufsize];
  char basepackage[bufsize],cffilename[bufsize];
  static char returnfile[bufsize];
  FILE *fp;
  struct stat statbuf;
  int numargs = 0, argnum = 0;
  unsigned char digest[EVP_MAX_MD_SIZE+1];

numargs = ListLen(ptr->send_args);

ChecksumList(ptr->send_args,digest,'m');
snprintf(basepackage,bufsize-1,"%s/rpc_in/localhost_localhost_%s_%s",VLOCKDIR,ptr->name,ChecksumPrint('m',digest));
   
if ((fp = fopen(basepackage,"r")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could not open a parameter bundle (%s) for method %s",basepackage,ptr->name);
   CfLog(cfinform,OUTPUT,"fopen");
   return returnfile;
   }
 
 argnum = 0;
 
 while (!feof(fp))
    {
    bzero(line,bufsize);
    fgets(line,bufsize,fp);
    type[0] = '\0';
    arg[0] = '\0';
    sscanf(line,"%63s %s",type,arg);
    
    switch (ConvertMethodProto(type))
       {
       case cfmeth_name:
	   if (strcmp(arg,ptr->name) != 0)
	      {
	      snprintf(OUTPUT,bufsize,"Method name %s did not match package name %s",ptr->name,arg);
	      CfLog(cfinform,OUTPUT,"");
	      fclose(fp);
	      return returnfile;
	      }
	   break;
	   
       case cfmeth_file:
	   
	   bzero(VBUFF,bufsize);
	   
	   if (GetMacroValue(CONTEXTID,"moduledirectory"))
	      {
	      ExpandVarstring("$(moduledirectory)",VBUFF,NULL);
	      }
	   else
	      {
	      snprintf(VBUFF,bufsize,"%s/modules",VLOCKDIR);
	      }
	   
	   snprintf(cffilename,bufsize-1,"%s/%s",VBUFF,arg);
	   
	   if (lstat(cffilename,&statbuf) != -1)
	      {
	      if (S_ISLNK(statbuf.st_mode))
		 {
		 snprintf(OUTPUT,bufsize,"SECURITY ALERT. Method (%s) source is a symbolic link",ptr->name);
		 CfLog(cferror,OUTPUT,"");
		 fclose(fp);
		 continue;
		 }
	      }
	   
	   if (stat(cffilename,&statbuf) == -1)
	      {
	      snprintf(OUTPUT,bufsize,"Method name %s did not find package name %s",ptr->name,cffilename);
	      CfLog(cfinform,OUTPUT,"stat");
	      fclose(fp);
	      continue;
	      }
	   
	   strncpy(returnfile,cffilename,maxvarsize);
	   break;
       case cfmeth_attacharg:
	   argnum++;
	   break;
       default:
	   break;
       }
    }
 
 fclose(fp);

if (numargs != argnum)
   {
   Verbose("Method (%s) arguments did not agree with the agreed template in the config file (%d/%d)\n",ptr->name,argnum,numargs);
   return returnfile;
   }
 
return returnfile; 
}

/******************************************************************************/
/* Level 3                                                                    */
/******************************************************************************/

enum methproto ConvertMethodProto(name)

char *name;

{ int i;
for (i = 0; VMETHODPROTO[i] != NULL; i++)
   {
   if (strcmp(name,VMETHODPROTO[i]) == 0)
      {
      break;
      }
   }

return (enum methproto) i;
}

/*****************************************************************************/

int CheckForMethodPackage(methodname)

char *methodname;

{ DIR *dirh;
  struct dirent *dirp;
  struct Item *methodID = NULL;
  char dirname[maxvarsize],path[bufsize];
  char name[bufsize],client[bufsize],server[bufsize],digeststring[bufsize],extra[256];
  struct stat statbuf;
  int gotmethod = false;
  
snprintf(dirname,maxvarsize-1,"%s/rpc_in",VLOCKDIR);
 
if ((dirh = opendir(dirname)) == NULL)
   {
   snprintf(OUTPUT,bufsize*2,"Can't open directory %s\n",dirname);
   CfLog(cfverbose,OUTPUT,"opendir");
   return false;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (!SensibleFile(dirp->d_name,dirname,NULL))
      {
      continue;
      }

   snprintf(path,bufsize-1,"%s/%s",dirname,dirp->d_name);

   if (stat(path,&statbuf) == -1)
      {
      continue;
      }

   if (statbuf.st_mtime > (CFSTARTTIME + (VEXPIREAFTER * 60)))
      {
      Verbose("Purging expired method %s\n",path);
      unlink(path);
      continue;
      }

   Verbose("Examining child's incoming (%s)\n",dirp->d_name);

   SplitMethodName(dirp->d_name,client,server,name,digeststring,extra);

   if (strcmp(methodname,name) == 0)
      {
      gotmethod = true;
      break;
      }
   
   DeleteItemList(methodID);
   }

closedir(dirh); 

if (gotmethod)
   {
   return true;
   }
else
   {
   return false;
   }
}

/*****************************************************************************/

struct Method *IsDefinedMethod(name,digeststring)

char *name, *digeststring;

{ struct Method *mp;
  unsigned char ldigest[EVP_MAX_MD_SIZE+1];

 Debug("Check for defined method (%s/%s)\n",name,digeststring);

for (mp = VMETHODS; mp != NULL; mp=mp->next)
   {
   if (strncmp(mp->name,name,strlen(mp->name)) == 0)
      {
      ChecksumList(mp->send_args,ldigest,'m');
      
      if (strcmp(digeststring,ChecksumPrint('m',ldigest)) == 0)
	 {
	 return mp;
	 }
      }
   }
 
return NULL; 
}

/*****************************************************************************/

int CountAttachments(basepackage)

char *basepackage;

{ FILE *fp;
 int argnum = 0;
 char line[bufsize],type[bufsize],arg[bufsize];

Debug("CountAttachments(%s)\n",basepackage);
 
if ((fp = fopen(basepackage,"r")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could not open a parameter bundle (%s)\n",basepackage);
   CfLog(cfinform,OUTPUT,"fopen");
   return 0;
   }
 
while (!feof(fp))
   {
   bzero(line,bufsize);
   fgets(line,bufsize,fp);
   type[0] = '\0';
   arg[0] = '\0';
   
   sscanf(line,"%63s %s",type,arg);
   
   switch (ConvertMethodProto(type))
      {
      case cfmeth_attacharg:	  	  
	  argnum++;
	  break;
      }
   }
 
fclose(fp);
return argnum;
}

/******************************************************************************/
/* Tool level                                                                 */
/******************************************************************************/

void EncapsulateMethod(ptr,name)

struct Method *ptr;
char *name;

{ struct Item *ip;
  char expbuf[bufsize],filename[bufsize];
  int i = 0,isreply = false;
  FILE *fp;

if (strstr(name,":Reply")) /* TODO - check if remote spoofing this could be dangerous*/
   {
   snprintf(OUTPUT,bufsize,"Method name (%s) should not contain a reply signal",name);
   CfLog(cferror,OUTPUT,"");
   }

if ((fp = fopen(name,"w")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could dispatch method to %s\n",name);
   CfLog(cferror,OUTPUT,"fopen");
   return;
   }

 Debug("EncapsulatingMethod(%s)\n",name);
 
 fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_name],ptr->name);
 fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_file],ptr->file);
 fprintf(fp,"%s %lu\n",VMETHODPROTO[cfmeth_time],(unsigned long)CFSTARTTIME);
 
 if (ptr->servers == NULL || ptr->servers->name)
    {
    fprintf(fp,"%s localhost\n",VMETHODPROTO[cfmeth_replyto]);
    }
 else if (strcmp(ptr->servers->name,"localhost") == 0)
    {
    fprintf(fp,"%s localhost\n",VMETHODPROTO[cfmeth_replyto]);
    }
 else
    {
    fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_replyto],VFQNAME);
    }

 for (ip = ptr->send_classes; ip != NULL; ip=ip->next)
    {
    /* If classX is defined locally, activate class method_classX in method */
    if (IsDefinedClass(ip->name))
       {
       fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_sendclass],ip->name);
       }
    }

 for (ip = ptr->send_args; ip != NULL; ip=ip->next)
    {
    i++;
    fprintf(fp,"%s %s_%d\n",VMETHODPROTO[cfmeth_attacharg],name,i);
    }

 fclose(fp);

 Debug("Packaging done\n");
 
 /* Now open a new file for each argument - this provides more policy control and security
    for remote access, since file sizes can be limited and there is less chance of buffer
    overflows */

 i = 0;
 
 for (ip = ptr->send_args; ip != NULL; ip=ip->next)
    {
    i++;    
    snprintf(filename,bufsize-1,"%s_%d",name,i);
    Debug("Create arg_file %s\n",filename);
    
    if (IsBuiltinFunction(ip->name))
       {
       char name[bufsize],args[bufsize],value[bufsize],sourcefile[bufsize];
       char *sp,maxbytes[bufsize],*nf,*nm;
       int count = 0,val = 0;
       FILE *fin,*fout;

       Debug("Evaluating readfile inclusion...\n");
       sscanf(ip->name,"%255[^(](%255[^)])",name,args);
       ExpandVarstring(name,expbuf,NULL);

       switch (FunctionStringToCode(expbuf))
	  {      
	  case fn_readfile:
	      	      	      
	      sourcefile[0] = '\0';
	      maxbytes[0] = '\0';
	      
	      for (sp = args; *sp != '\0'; sp++)
		 {
		 if (*sp == ',')
		    {
		    count++;
		    }
 		 }
	      
	      if (count != 1)
		 {
		 yyerror("ReadFile(filename,maxbytes): argument error");
		 return;
		 }
	      
	      sscanf(args,"%[^,],%[^)]",sourcefile,maxbytes);
	      Debug("ReadFile [%s] < [%s]\n",sourcefile,maxbytes);
	      
	      if (sourcefile[0]=='\0' || maxbytes[0] == '\0')
		 {
		 yyerror("Argument error in class-function");
		 return;
		 }
	      
	      nf = UnQuote(sourcefile);
	      nm = UnQuote(maxbytes);
	      
	      val = atoi(nm);
	      
	      if ((fin = fopen(nf,"r")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could open ReadFile(%s)\n",nf);
		 CfLog(cferror,OUTPUT,"fopen");
		 return;
		 }

	      Debug("Writing file to %s\n",filename);
	      
	      if ((fout = fopen(filename,"w")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could not open argument attachment %s\n",filename);
		 CfLog(cferror,OUTPUT,"fopen");
		 fclose(fin);
		 return;
		 }
	      
	      if (val > bufsize)
		 {
		 int count = 0;
		 while(!feof(fin))
		    {
		    fputc(fgetc(fin),fout);
		    count++;
		    if (count == val)
		       {
		       break;
		       }
		    Debug("Wrote #\n");
		    }
		 }
	      else
		 {
		 bzero(value,bufsize); 
		 fread(value,val,1,fin);
		 fwrite(value,val,1,fout);
		 Debug("Wrote #\n");
		 }
	      
	      fclose(fin);
	      fclose(fout);
	      Debug("Done with file dispatch\n");
	      break;
	      
	  default:
	      snprintf(OUTPUT,bufsize,"Invalid function (%s) used in function argument processing",expbuf);
	      CfLog(cferror,OUTPUT,"");
	      
	  }
       
       }
    else
       {
       if ((fp = fopen(filename,"w")) == NULL)
	  {
	  snprintf(OUTPUT,bufsize,"Could dispatch method to %s\n",name);
	  CfLog(cferror,OUTPUT,"fopen");
	  return;
	  }

       ExpandVarstring(ip->name,expbuf,NULL);
       fwrite(expbuf,strlen(expbuf),1,fp);       
       fclose(fp);   
       }
    }
}


/******************************************************************************/

void EncapsulateReply(name)

char *name;

{ FILE *fp;
  struct Item *ip;
  char expbuf[bufsize],filename[bufsize];
  int i = 0,isreply = false;;

if ((fp = fopen(name,"w")) == NULL)
   {
   snprintf(OUTPUT,bufsize,"Could dispatch method to %s\n",name);
   CfLog(cferror,OUTPUT,"fopen");
   return;
   }

 Debug("EncapsulatingReply(%s)\n",name);
 
 fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_name],METHODNAME);
 fprintf(fp,"%s %lu\n",VMETHODPROTO[cfmeth_time],(unsigned long)CFSTARTTIME);
 fprintf(fp,"%s true\n",VMETHODPROTO[cfmeth_isreply]);
     
 for (ip = SplitStringAsItemList(METHODRETURNCLASSES,','); ip != NULL; ip=ip->next)
    {
    /* If classX is defined locally, activate class method_classX in method */
    if (IsDefinedClass(ip->name))
       {
       Debug("Packaging return class %s\n",ip->name);
       fprintf(fp,"%s %s\n",VMETHODPROTO[cfmeth_sendclass],ip->name);
       }
    }

 DeleteItemList(ip);
 
 for (ip = ListFromArgs(METHODRETURNVARS); ip != NULL; ip=ip->next)
    {
    i++;
    ExpandVarstring(ip->name,expbuf,NULL);
    
    fprintf(fp,"%s %s_%d (%s)\n",VMETHODPROTO[cfmeth_attacharg],name,i,expbuf);
    }

 DeleteItemList(ip);
 fclose(fp);

 Debug("Packaging done\n");
 
 /* Now open a new file for each argument - this provides more policy control and security
    for remote access, since file sizes can be limited and there is less chance of buffer
    overflows */

 i = 0;
 
 for (ip = ListFromArgs(METHODRETURNVARS); ip != NULL; ip=ip->next)
    {
    i++;    
    snprintf(filename,bufsize-1,"%s_%d",name,i);
    Debug("Create arg_file %s\n",filename);
    
    if (IsBuiltinFunction(ip->name))
       {
       char name[bufsize],args[bufsize],value[bufsize],sourcefile[bufsize];
       char *sp,maxbytes[bufsize],*nf,*nm;
       int count = 0,val = 0;
       FILE *fin,*fout;

       Debug("Evaluating readfile inclusion...\n");
       sscanf(ip->name,"%255[^(](%255[^)])",name,args);
       ExpandVarstring(name,expbuf,NULL);

       switch (FunctionStringToCode(expbuf))
	  {      
	  case fn_readfile:
	      	      	      
	      sourcefile[0] = '\0';
	      maxbytes[0] = '\0';
	      
	      for (sp = args; *sp != '\0'; sp++)
		 {
		 if (*sp == ',')
		    {
		    count++;
		    }
		 }
	      
	      if (count != 1)
		 {
		 yyerror("ReadFile(filename,maxbytes): argument error");
		 return;
		 }
	      
	      sscanf(args,"%[^,],%[^)]",sourcefile,maxbytes);
	      Debug("ReadFile [%s] < [%s]\n",sourcefile,maxbytes);
	      
	      if (sourcefile[0]=='\0' || maxbytes[0] == '\0')
		 {
		 yyerror("Argument error in class-function");
		 return;
		 }
	      
	      nf = UnQuote(sourcefile);
	      nm = UnQuote(maxbytes);
	      
	      val = atoi(nm);
	      
	      if ((fin = fopen(nf,"r")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could open ReadFile(%s)\n",nf);
		 CfLog(cferror,OUTPUT,"fopen");
		 return;
		 }
	      
	      if ((fout = fopen(filename,"w")) == NULL)
		 {
		 snprintf(OUTPUT,bufsize,"Could not open argument attachment %s\n",filename);
		 CfLog(cferror,OUTPUT,"fopen");
		 fclose(fin);
		 return;
		 }
	      
	      if (val > bufsize)
		 {
		 int count = 0;
		 while(!feof(fin))
		    {
		    fputc(fgetc(fin),fout);
		    count++;
		    if (count == val)
		       {
		       break;
		       }		    
		    }
		 }
	      else
		 {
		 bzero(value,bufsize); 
		 fread(value,val,1,fin);
		 fwrite(value,val,1,fout);
		 }
	      fclose(fin);
	      fclose(fout);
	      break;
	      
	  default:
	      snprintf(OUTPUT,bufsize,"Invalid function (%s) used in function argument processing",expbuf);
	      CfLog(cferror,OUTPUT,"");
	      
	  }
       
       }
    else
       {
       if ((fp = fopen(filename,"w")) == NULL)
	  {
	  snprintf(OUTPUT,bufsize,"Could dispatch method to %s\n",name);
	  CfLog(cferror,OUTPUT,"fopen");
	  return;
	  }

       ExpandVarstring(ip->name,expbuf,NULL);
       fwrite(expbuf,strlen(expbuf),1,fp);       
       fclose(fp);   
       }
    }
 
DeleteItemList(ip);
}

/******************************************************************************/

void SplitMethodName(name,client,server,methodname,digeststring,extra)

char *name,*client,*server,*methodname,*digeststring,*extra;

{
methodname[0] = '\0';
client[0] = '\0';
server[0] = '\0';
digeststring[0] = '\0';
extra[0] = '\0';
 
sscanf(name,"%1024[^_]_%1024[^_]_%1024[^_]_%[^_]_%64s",client,server,methodname,digeststring,extra);
}
