/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkColorButton Copyright (C) 2000 Stefan Ondrejicka <ondrej@idata.sk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gtkclrbutton.h"
#include <gtk/gtk.h>
#include <string.h>


#define RED     0
#define GREEN   1
#define BLUE    2
#define OPACITY 3


enum {
  COLOR_CHANGED,
  LAST_SIGNAL
};


static void gtk_color_button_class_init		(GtkColorButtonClass	*klass);
static void gtk_color_button_init		(GtkColorButton		*cbutton);
static void gtk_color_button_destroy		(GtkObject		*object);
static void gtk_color_button_color_changed	(GtkColorButton		*cbutton);
static void gtk_color_button_preview_paint	(GtkColorButton		*cbutton);
static void gtk_color_button_preview_resize	(GtkWidget		*w,
						 GtkAllocation		*allocation,
						 GtkColorButton		*cbutton);
static void gtk_color_button_select_color	(GtkColorButton		*cbutton);
static void gtk_color_button_colorsel_hide	(GtkObject		*object,
						 GtkColorButton         *cbutton);
static void gtk_color_button_colorsel_ok	(GtkObject		*object,
						 GtkColorButton         *cbutton);

static gint color_button_signals[LAST_SIGNAL] = { 0 };
static GtkButtonClass *parent_class = NULL;

GtkType gtk_color_button_get_type (void)
{
	static GtkType color_button_type = 0;

	if (!color_button_type)
	{
		static const GtkTypeInfo color_button_info =
		{
			"GtkColorButton",
			sizeof (GtkColorButton),
			sizeof (GtkColorButtonClass),
			(GtkClassInitFunc) gtk_color_button_class_init,
			(GtkObjectInitFunc) gtk_color_button_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		color_button_type = gtk_type_unique (GTK_TYPE_BUTTON, &color_button_info);
        }

        return color_button_type;;
}

static void gtk_color_button_class_init (GtkColorButtonClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class (GTK_TYPE_BUTTON);
	object_class->destroy = gtk_color_button_destroy;

	color_button_signals[COLOR_CHANGED] = 
		gtk_signal_new ("color_changed",
				GTK_RUN_FIRST, 
#if GTK_CHECK_VERSION(1,3,0)
				GTK_CLASS_TYPE(object_class),
#else
				object_class->type,
#endif
				GTK_SIGNAL_OFFSET (GtkColorButtonClass, color_changed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, color_button_signals, LAST_SIGNAL);

	klass->color_changed = NULL;
}

static void gtk_color_button_init (GtkColorButton *cbutton)
{
	GTK_WIDGET_SET_FLAGS (cbutton, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);
	GTK_WIDGET_UNSET_FLAGS (cbutton, GTK_NO_WINDOW);

	cbutton->color.pixel = 0;
	cbutton->color.red = 0;
	cbutton->color.green = 0;
	cbutton->color.blue = 0;

	cbutton->title = NULL;
	cbutton->colorsel = NULL;
	cbutton->had_colorsel = FALSE;

	cbutton->preview = gtk_preview_new(GTK_PREVIEW_COLOR);
	gtk_preview_set_dither (GTK_PREVIEW (cbutton->preview), GDK_RGB_DITHER_NONE);
	gtk_preview_size (GTK_PREVIEW (cbutton->preview), 30, 20);
	gtk_preview_set_expand (GTK_PREVIEW (cbutton->preview), TRUE);
	gtk_container_add (GTK_CONTAINER (cbutton) , cbutton->preview);
	gtk_widget_show (cbutton->preview);

	gtk_signal_connect_after (GTK_OBJECT (cbutton->preview), "size_allocate",
		GTK_SIGNAL_FUNC (gtk_color_button_preview_resize), cbutton);

	gtk_signal_connect_after (GTK_OBJECT (cbutton), "clicked",
		GTK_SIGNAL_FUNC (gtk_color_button_select_color), NULL);
}

GtkWidget *gtk_color_button_new (void)
{
	return GTK_WIDGET (gtk_type_new (gtk_color_button_get_type ()));
}

void gtk_color_button_set_title (GtkColorButton *cbutton, gchar *title)
{
	g_return_if_fail (cbutton != NULL);
	g_return_if_fail (GTK_IS_COLOR_BUTTON (cbutton));

	g_free(cbutton->title);
	cbutton->title = g_strdup(title);
}

void gtk_color_button_set_color (GtkColorButton *cbutton, GdkColor *color)
{
	g_return_if_fail (cbutton != NULL);
	g_return_if_fail (GTK_IS_COLOR_BUTTON (cbutton));
	g_return_if_fail (color != NULL);

	memcpy(&cbutton->color,  color, sizeof(GdkColor));
	gtk_color_button_preview_paint (cbutton);
	gtk_color_button_color_changed (cbutton);
}

void gtk_color_button_get_color (GtkColorButton *cbutton, GdkColor *color)
{
	g_return_if_fail (cbutton != NULL);
	g_return_if_fail (GTK_IS_COLOR_BUTTON (cbutton));
	g_return_if_fail (color != NULL);

	memcpy(color, &cbutton->color, sizeof(GdkColor));
}

static void gtk_color_button_destroy (GtkObject *object)
{
	g_free(GTK_COLOR_BUTTON(object)->title);

	if (GTK_COLOR_BUTTON(object)->colorsel)
		gtk_widget_destroy(GTK_COLOR_BUTTON(object)->colorsel);
}

static void gtk_color_button_color_changed (GtkColorButton *cbutton)
{
	gtk_signal_emit (GTK_OBJECT (cbutton), color_button_signals[COLOR_CHANGED], &cbutton->color);
}

static void gtk_color_button_preview_paint (GtkColorButton *cbutton)
{
	gint width, height, x, y ,n;
	guchar *sample;
	guchar c[3];

	c[RED] = (guchar)(cbutton->color.red / 256);
	c[GREEN] = (guchar)(cbutton->color.green / 256);
	c[BLUE] = (guchar)(cbutton->color.blue / 256);

	width = cbutton->preview->allocation.width;
	height = cbutton->preview->allocation.height;

	sample = g_new(guchar, 3 * width);

	for (x = 0 ; x < width ; x++)
		for (n = 0; n < 3; n++)
			sample[3*x + n] = c[n];

	for (y = 0 ; y < height ; y++)
	{
		gtk_preview_draw_row (GTK_PREVIEW (cbutton->preview), sample, 0, y, width);
	}

	g_free(sample);
	gtk_widget_queue_draw (cbutton->preview);
}

static void gtk_color_button_preview_resize (GtkWidget *w, GtkAllocation *allocation, GtkColorButton *cbutton)
{
	gtk_color_button_preview_paint (cbutton);
}

static void gtk_color_button_colorsel_hide (GtkObject *object, GtkColorButton *cbutton)
{
	if (cbutton->colorsel)
		gtk_widget_destroy(cbutton->colorsel);
}

static void gtk_color_button_colorsel_ok (GtkObject *object, GtkColorButton *cbutton)
{
	gdouble color[4];

	gtk_color_selection_get_color(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(cbutton->colorsel)->colorsel) , color);

	cbutton->color.red = color[RED] * 65535;
	cbutton->color.green = color[GREEN] * 65535;
	cbutton->color.blue = color[BLUE] * 65535;

	gtk_color_button_preview_paint (cbutton);
	gtk_color_button_color_changed (cbutton);

	gtk_widget_destroy(cbutton->colorsel);
}

static void gtk_color_button_select_color (GtkColorButton *cbutton)
{
	if (!cbutton->colorsel)
	{
		gdouble color[4];

		cbutton->colorsel = gtk_color_selection_dialog_new(cbutton->title);
		gtk_signal_connect (GTK_OBJECT (cbutton->colorsel), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &cbutton->colorsel);

		if (!cbutton->had_colorsel)
		{
			gtk_signal_connect(GTK_OBJECT(gtk_widget_get_toplevel(GTK_WIDGET(cbutton))),
				"hide", GTK_SIGNAL_FUNC(gtk_color_button_colorsel_hide),
				(gpointer) cbutton);

			cbutton->had_colorsel = TRUE;
		}

		color[RED] = ((gdouble)cbutton->color.red) / 65535.0;
		color[GREEN] = ((gdouble)cbutton->color.green) / 65535.0;
		color[BLUE] = ((gdouble)cbutton->color.blue) / 65535.0;
		color[OPACITY] = 1.0;

		gtk_color_selection_set_color(GTK_COLOR_SELECTION (
			GTK_COLOR_SELECTION_DIALOG (cbutton->colorsel)->colorsel) , color);
#if 0
		gtk_color_selection_set_opacity(GTK_COLOR_SELECTION (
			GTK_COLOR_SELECTION_DIALOG (cbutton->colorsel)->colorsel) , FALSE);
#endif

		gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (cbutton->colorsel)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_color_button_colorsel_ok), cbutton);

		gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG (cbutton->colorsel)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_color_button_colorsel_hide), cbutton);

	}

	gtk_widget_show (cbutton->colorsel);
	if (GTK_WIDGET_REALIZED (cbutton->colorsel))
		gdk_window_raise (cbutton->colorsel->window);
}

