// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/drive/file_system/copy_operation.h"

#include <string>

#include "base/file_util.h"
#include "base/task_runner_util.h"
#include "chrome/browser/chromeos/drive/drive.pb.h"
#include "chrome/browser/chromeos/drive/file_cache.h"
#include "chrome/browser/chromeos/drive/file_system/create_file_operation.h"
#include "chrome/browser/chromeos/drive/file_system/download_operation.h"
#include "chrome/browser/chromeos/drive/file_system/move_operation.h"
#include "chrome/browser/chromeos/drive/file_system/operation_observer.h"
#include "chrome/browser/chromeos/drive/file_system_util.h"
#include "chrome/browser/chromeos/drive/job_scheduler.h"
#include "chrome/browser/chromeos/drive/resource_entry_conversion.h"
#include "chrome/browser/drive/drive_api_util.h"
#include "content/public/browser/browser_thread.h"

using content::BrowserThread;

namespace drive {
namespace file_system {

namespace {

// Copies a file from |src_file_path| to |dest_file_path| on the local
// file system using file_util::CopyFile.
// Returns FILE_ERROR_OK on success or FILE_ERROR_FAILED otherwise.
FileError CopyLocalFileOnBlockingPool(
    const base::FilePath& src_file_path,
    const base::FilePath& dest_file_path) {
  return file_util::CopyFile(src_file_path, dest_file_path) ?
      FILE_ERROR_OK : FILE_ERROR_FAILED;
}

// Stores a file to the cache and mark it dirty.
FileError StoreAndMarkDirty(internal::FileCache* cache,
                            const std::string& resource_id,
                            const std::string& md5,
                            const base::FilePath& local_file_path) {
  FileError error = cache->Store(resource_id, md5, local_file_path,
                                 internal::FileCache::FILE_OPERATION_COPY);
  if (error != FILE_ERROR_OK)
    return error;
  return cache->MarkDirty(resource_id, md5);
}

}  // namespace

CopyOperation::CopyOperation(base::SequencedTaskRunner* blocking_task_runner,
                             OperationObserver* observer,
                             JobScheduler* scheduler,
                             internal::ResourceMetadata* metadata,
                             internal::FileCache* cache,
                             DriveServiceInterface* drive_service,
                             const base::FilePath& temporary_file_directory)
  : blocking_task_runner_(blocking_task_runner),
    observer_(observer),
    scheduler_(scheduler),
    metadata_(metadata),
    cache_(cache),
    drive_service_(drive_service),
    create_file_operation_(new CreateFileOperation(blocking_task_runner,
                                                   observer,
                                                   scheduler,
                                                   metadata,
                                                   cache)),
    download_operation_(new DownloadOperation(blocking_task_runner,
                                              observer,
                                              scheduler,
                                              metadata,
                                              cache,
                                              temporary_file_directory)),
    move_operation_(new MoveOperation(observer, scheduler, metadata)),
    weak_ptr_factory_(this) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
}

CopyOperation::~CopyOperation() {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
}

void CopyOperation::Copy(const base::FilePath& src_file_path,
                         const base::FilePath& dest_file_path,
                         const FileOperationCallback& callback) {
  BrowserThread::CurrentlyOn(BrowserThread::UI);
  DCHECK(!callback.is_null());

  metadata_->GetResourceEntryPairByPathsOnUIThread(
      src_file_path,
      dest_file_path.DirName(),
      base::Bind(&CopyOperation::CopyAfterGetResourceEntryPair,
                 weak_ptr_factory_.GetWeakPtr(),
                 dest_file_path,
                 callback));
}

void CopyOperation::TransferFileFromRemoteToLocal(
    const base::FilePath& remote_src_file_path,
    const base::FilePath& local_dest_file_path,
    const FileOperationCallback& callback) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  download_operation_->EnsureFileDownloadedByPath(
      remote_src_file_path,
      ClientContext(USER_INITIATED),
      GetFileContentInitializedCallback(),
      google_apis::GetContentCallback(),
      base::Bind(&CopyOperation::OnGetFileCompleteForTransferFile,
                 weak_ptr_factory_.GetWeakPtr(),
                 local_dest_file_path,
                 callback));
}

void CopyOperation::OnGetFileCompleteForTransferFile(
    const base::FilePath& local_dest_file_path,
    const FileOperationCallback& callback,
    FileError error,
    const base::FilePath& local_file_path,
    scoped_ptr<ResourceEntry> entry) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }

  // GetFileByPath downloads the file from Drive to a local cache, which is then
  // copied to the actual destination path on the local file system using
  // CopyLocalFileOnBlockingPool.
  base::PostTaskAndReplyWithResult(
      blocking_task_runner_.get(),
      FROM_HERE,
      base::Bind(
          &CopyLocalFileOnBlockingPool, local_file_path, local_dest_file_path),
      callback);
}

void CopyOperation::TransferFileFromLocalToRemote(
    const base::FilePath& local_src_file_path,
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  // Make sure the destination directory exists.
  metadata_->GetResourceEntryByPathOnUIThread(
      remote_dest_file_path.DirName(),
      base::Bind(
          &CopyOperation::TransferFileFromLocalToRemoteAfterGetResourceEntry,
          weak_ptr_factory_.GetWeakPtr(),
          local_src_file_path,
          remote_dest_file_path,
          callback));
}

void CopyOperation::ScheduleTransferRegularFile(
    const base::FilePath& local_file_path,
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  const bool fail_if_file_already_exists = true;
  create_file_operation_->CreateFile(
      remote_dest_file_path,
      fail_if_file_already_exists,
      base::Bind(&CopyOperation::ScheduleTransferRegularFileAfterCreate,
                 weak_ptr_factory_.GetWeakPtr(),
                 local_file_path,
                 remote_dest_file_path,
                 callback));
}

void CopyOperation::ScheduleTransferRegularFileAfterCreate(
    const base::FilePath& local_file_path,
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback,
    FileError error) {
  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }

  metadata_->GetResourceEntryByPathOnUIThread(
      remote_dest_file_path,
      base::Bind(
          &CopyOperation::ScheduleTransferRegularFileAfterGetResourceEntry,
          weak_ptr_factory_.GetWeakPtr(),
          local_file_path,
          callback));
}

void CopyOperation::ScheduleTransferRegularFileAfterGetResourceEntry(
    const base::FilePath& local_file_path,
    const FileOperationCallback& callback,
    FileError error,
    scoped_ptr<ResourceEntry> entry) {
  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }

  ResourceEntry* entry_ptr = entry.get();
  base::PostTaskAndReplyWithResult(
      blocking_task_runner_.get(),
      FROM_HERE,
      base::Bind(&StoreAndMarkDirty,
                 cache_,
                 entry_ptr->resource_id(),
                 entry_ptr->file_specific_info().md5(),
                 local_file_path),
      base::Bind(&CopyOperation::ScheduleTransferRegularFileAfterStore,
                 weak_ptr_factory_.GetWeakPtr(),
                 base::Passed(&entry),
                 callback));
}

void CopyOperation::ScheduleTransferRegularFileAfterStore(
    scoped_ptr<ResourceEntry> entry,
    const FileOperationCallback& callback,
    FileError error) {
  if (error == FILE_ERROR_OK)
    observer_->OnCacheFileUploadNeededByOperation(entry->resource_id());
  callback.Run(error);
}

void CopyOperation::CopyHostedDocumentToDirectory(
    const base::FilePath& dir_path,
    const std::string& resource_id,
    const base::FilePath::StringType& new_name,
    const FileOperationCallback& callback) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  scheduler_->CopyHostedDocument(
      resource_id,
      base::FilePath(new_name).AsUTF8Unsafe(),
      base::Bind(&CopyOperation::OnCopyHostedDocumentCompleted,
                 weak_ptr_factory_.GetWeakPtr(),
                 dir_path,
                 callback));
}

void CopyOperation::OnCopyHostedDocumentCompleted(
    const base::FilePath& dir_path,
    const FileOperationCallback& callback,
    google_apis::GDataErrorCode status,
    scoped_ptr<google_apis::ResourceEntry> resource_entry) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  FileError error = util::GDataToFileError(status);
  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }
  DCHECK(resource_entry);

  // The entry was added in the root directory on the server, so we should
  // first add it to the root to mirror the state and then move it to the
  // destination directory by MoveEntryFromRootDirectory().
  metadata_->AddEntryOnUIThread(
      ConvertToResourceEntry(*resource_entry),
      base::Bind(&CopyOperation::MoveEntryFromRootDirectory,
                 weak_ptr_factory_.GetWeakPtr(),
                 dir_path,
                 callback));
}

void CopyOperation::MoveEntryFromRootDirectory(
    const base::FilePath& directory_path,
    const FileOperationCallback& callback,
    FileError error,
    const base::FilePath& file_path) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());
  DCHECK_EQ(util::GetDriveMyDriveRootPath().value(),
            file_path.DirName().value());

  // Return if there is an error or |dir_path| is the root directory.
  if (error != FILE_ERROR_OK ||
      directory_path == util::GetDriveMyDriveRootPath()) {
    callback.Run(error);
    return;
  }

  move_operation_->Move(file_path,
                        directory_path.Append(file_path.BaseName()),
                        callback);
}

void CopyOperation::CopyAfterGetResourceEntryPair(
    const base::FilePath& dest_file_path,
    const FileOperationCallback& callback,
    scoped_ptr<EntryInfoPairResult> result) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());
  DCHECK(result.get());

  if (result->first.error != FILE_ERROR_OK) {
    callback.Run(result->first.error);
    return;
  } else if (result->second.error != FILE_ERROR_OK) {
    callback.Run(result->second.error);
    return;
  }

  scoped_ptr<ResourceEntry> src_file_proto = result->first.entry.Pass();
  scoped_ptr<ResourceEntry> dest_parent_proto = result->second.entry.Pass();

  if (!dest_parent_proto->file_info().is_directory()) {
    callback.Run(FILE_ERROR_NOT_A_DIRECTORY);
    return;
  } else if (src_file_proto->file_info().is_directory()) {
    // TODO(kochi): Implement copy for directories. In the interim,
    // we handle recursive directory copy in the file manager.
    // crbug.com/141596
    callback.Run(FILE_ERROR_INVALID_OPERATION);
    return;
  }

  // If Drive API v2 is enabled, we can copy resources on server side.
  if (util::IsDriveV2ApiEnabled()) {
    base::FilePath new_name = dest_file_path.BaseName();
    if (src_file_proto->file_specific_info().is_hosted_document()) {
      // Drop the document extension, which should not be in the title.
      // TODO(yoshiki): Remove this code with crbug.com/223304.
      new_name = new_name.RemoveExtension();
    }

    scheduler_->CopyResource(
        src_file_proto->resource_id(),
        dest_parent_proto->resource_id(),
        new_name.value(),
        base::Bind(&CopyOperation::OnCopyResourceCompleted,
                   weak_ptr_factory_.GetWeakPtr(), callback));
    return;
  }


  if (src_file_proto->file_specific_info().is_hosted_document()) {
    CopyHostedDocumentToDirectory(
        dest_file_path.DirName(),
        src_file_proto->resource_id(),
        // Drop the document extension, which should not be in the title.
        // TODO(yoshiki): Remove this code with crbug.com/223304.
        dest_file_path.BaseName().RemoveExtension().value(),
        callback);
    return;
  }

  const base::FilePath& src_file_path = result->first.path;
  download_operation_->EnsureFileDownloadedByPath(
      src_file_path,
      ClientContext(USER_INITIATED),
      GetFileContentInitializedCallback(),
      google_apis::GetContentCallback(),
      base::Bind(&CopyOperation::OnGetFileCompleteForCopy,
                 weak_ptr_factory_.GetWeakPtr(),
                 dest_file_path,
                 callback));
}

void CopyOperation::OnCopyResourceCompleted(
    const FileOperationCallback& callback,
    google_apis::GDataErrorCode status,
    scoped_ptr<google_apis::ResourceEntry> resource_entry) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  FileError error = util::GDataToFileError(status);
  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }
  DCHECK(resource_entry);

  // The copy on the server side is completed successfully. Update the local
  // metadata.
  base::PostTaskAndReplyWithResult(
      blocking_task_runner_.get(),
      FROM_HERE,
      base::Bind(&internal::ResourceMetadata::AddEntry,
                 base::Unretained(metadata_),
                 ConvertToResourceEntry(*resource_entry)),
      callback);
}

void CopyOperation::OnGetFileCompleteForCopy(
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback,
    FileError error,
    const base::FilePath& local_file_path,
    scoped_ptr<ResourceEntry> entry) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }

  // This callback is only triggered for a regular file via Copy().
  DCHECK(entry && !entry->file_specific_info().is_hosted_document());
  ScheduleTransferRegularFile(local_file_path, remote_dest_file_path, callback);
}

void CopyOperation::TransferFileFromLocalToRemoteAfterGetResourceEntry(
    const base::FilePath& local_src_file_path,
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback,
    FileError error,
    scoped_ptr<ResourceEntry> entry) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (error != FILE_ERROR_OK) {
    callback.Run(error);
    return;
  }

  DCHECK(entry.get());
  if (!entry->file_info().is_directory()) {
    // The parent of |remote_dest_file_path| is not a directory.
    callback.Run(FILE_ERROR_NOT_A_DIRECTORY);
    return;
  }

  if (util::HasGDocFileExtension(local_src_file_path)) {
    base::PostTaskAndReplyWithResult(
        blocking_task_runner_.get(),
        FROM_HERE,
        base::Bind(&util::ReadResourceIdFromGDocFile, local_src_file_path),
        base::Bind(&CopyOperation::TransferFileForResourceId,
                   weak_ptr_factory_.GetWeakPtr(),
                   local_src_file_path,
                   remote_dest_file_path,
                   callback));
  } else {
    ScheduleTransferRegularFile(local_src_file_path, remote_dest_file_path,
                                callback);
  }
}

void CopyOperation::TransferFileForResourceId(
    const base::FilePath& local_file_path,
    const base::FilePath& remote_dest_file_path,
    const FileOperationCallback& callback,
    const std::string& resource_id) {
  DCHECK(BrowserThread::CurrentlyOn(BrowserThread::UI));
  DCHECK(!callback.is_null());

  if (resource_id.empty()) {
    // If |resource_id| is empty, upload the local file as a regular file.
    ScheduleTransferRegularFile(local_file_path, remote_dest_file_path,
                                callback);
    return;
  }

  // GDoc file may contain a resource ID in the old format.
  const std::string canonicalized_resource_id =
      drive_service_->CanonicalizeResourceId(resource_id);

  // Otherwise, copy the document on the server side and add the new copy
  // to the destination directory (collection).
  CopyHostedDocumentToDirectory(
      remote_dest_file_path.DirName(),
      canonicalized_resource_id,
      // Drop the document extension, which should not be
      // in the document title.
      // TODO(yoshiki): Remove this code with crbug.com/223304.
      remote_dest_file_path.BaseName().RemoveExtension().value(),
      callback);
}

}  // namespace file_system
}  // namespace drive
