// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_EXTENSIONS_API_LOCATION_LOCATION_MANAGER_H_
#define CHROME_BROWSER_EXTENSIONS_API_LOCATION_LOCATION_MANAGER_H_

#include <map>
#include <string>

#include "base/memory/weak_ptr.h"
#include "chrome/browser/extensions/api/profile_keyed_api_factory.h"
#include "content/public/browser/notification_observer.h"
#include "content/public/browser/notification_registrar.h"

class Profile;

namespace content {
struct Geoposition;
}  // namespace content

namespace extensions {
class LocationManager;
class LocationRequest;

namespace api {
namespace location {

struct Coordinates;

}  // namespace location
}  // namespace api

// Profile's manager of all location watch requests created by chrome.location
// API. Lives in the UI thread.
class LocationManager
    : public ProfileKeyedAPI,
      public content::NotificationObserver,
      public base::SupportsWeakPtr<LocationManager> {
 public:
  explicit LocationManager(Profile* profile);
  virtual ~LocationManager();

  // Adds location request for the given extension, and starts the location
  // tracking.
  void AddLocationRequest(const std::string& extension_id,
                          const std::string& request_name);

  // Cancels and removes the request with the given |name| for the given
  // extension.
  void RemoveLocationRequest(const std::string& extension_id,
                             const std::string& name);

  // ProfileKeyedAPI implementation.
  static ProfileKeyedAPIFactory<LocationManager>* GetFactoryInstance();

  // Convenience method to get the LocationManager for a profile.
  static LocationManager* Get(Profile* profile);

 private:
  friend class LocationRequest;
  friend class ProfileKeyedAPIFactory<LocationManager>;

  typedef std::string ExtensionId;
  typedef scoped_refptr<LocationRequest> LocationRequestPointer;
  typedef std::multimap<ExtensionId, LocationRequestPointer> LocationRequestMap;
  typedef LocationRequestMap::iterator LocationRequestIterator;

  // Converts |position| from GeolocationProvider to the location API
  // |coordinates|.
  static void GeopositionToApiCoordinates(
      const content::Geoposition& position,
      api::location::Coordinates* coordinates);

  // Sends a location update to the extension.
  void SendLocationUpdate(const std::string& extension_id,
                          const std::string& request_name,
                          const content::Geoposition& position);

  // NotificationObserver:
  virtual void Observe(int type,
                       const content::NotificationSource& source,
                       const content::NotificationDetails& details) OVERRIDE;

  // ProfileKeyedAPI implementation.
  static const char* service_name() { return "LocationManager"; }

  // Profile for this location manager.
  Profile* const profile_;

  // A map of our pending location requests, per extension.
  // Invariant: None of the LocationRequestLists are empty.
  LocationRequestMap location_requests_;

  // Used for tracking registrations to profile's extensions events.
  content::NotificationRegistrar registrar_;

  DISALLOW_COPY_AND_ASSIGN(LocationManager);
};

}  // namespace extensions

#endif  // CHROME_BROWSER_EXTENSIONS_API_LOCATION_LOCATION_MANAGER_H_
