// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "webkit/browser/chromeos/fileapi/cros_mount_point_provider.h"

#include "base/logging.h"
#include "base/memory/scoped_ptr.h"
#include "base/message_loop.h"
#include "base/path_service.h"
#include "base/strings/stringprintf.h"
#include "base/strings/utf_string_conversions.h"
#include "base/synchronization/lock.h"
#include "chromeos/dbus/cros_disks_client.h"
#include "third_party/WebKit/public/platform/WebCString.h"
#include "third_party/WebKit/public/platform/WebFileSystem.h"
#include "third_party/WebKit/public/platform/WebString.h"
#include "webkit/browser/chromeos/fileapi/file_access_permissions.h"
#include "webkit/browser/chromeos/fileapi/remote_file_stream_writer.h"
#include "webkit/browser/chromeos/fileapi/remote_file_system_operation.h"
#include "webkit/browser/fileapi/async_file_util_adapter.h"
#include "webkit/browser/fileapi/copy_or_move_file_validator.h"
#include "webkit/browser/fileapi/external_mount_points.h"
#include "webkit/browser/fileapi/file_system_context.h"
#include "webkit/browser/fileapi/file_system_file_stream_reader.h"
#include "webkit/browser/fileapi/file_system_operation_context.h"
#include "webkit/browser/fileapi/file_system_task_runners.h"
#include "webkit/browser/fileapi/file_system_url.h"
#include "webkit/browser/fileapi/isolated_context.h"
#include "webkit/browser/fileapi/isolated_file_util.h"
#include "webkit/browser/fileapi/local_file_stream_writer.h"
#include "webkit/browser/fileapi/local_file_system_operation.h"

namespace {

const char kChromeUIScheme[] = "chrome";

}  // namespace

namespace chromeos {

// static
bool CrosMountPointProvider::CanHandleURL(const fileapi::FileSystemURL& url) {
  if (!url.is_valid())
    return false;
  return url.type() == fileapi::kFileSystemTypeNativeLocal ||
         url.type() == fileapi::kFileSystemTypeRestrictedNativeLocal ||
         url.type() == fileapi::kFileSystemTypeDrive;
}

CrosMountPointProvider::CrosMountPointProvider(
    scoped_refptr<quota::SpecialStoragePolicy> special_storage_policy,
    scoped_refptr<fileapi::ExternalMountPoints> mount_points,
    fileapi::ExternalMountPoints* system_mount_points)
    : special_storage_policy_(special_storage_policy),
      file_access_permissions_(new FileAccessPermissions()),
      local_file_util_(new fileapi::AsyncFileUtilAdapter(
          new fileapi::IsolatedFileUtil())),
      mount_points_(mount_points),
      system_mount_points_(system_mount_points) {
}

CrosMountPointProvider::~CrosMountPointProvider() {
}

void CrosMountPointProvider::AddSystemMountPoints() {
  // RegisterFileSystem() is no-op if the mount point with the same name
  // already exists, hence it's safe to call without checking if a mount
  // point already exists or not.

  // TODO(satorux): "Downloads" directory should probably be per-profile. For
  // this to be per-profile, a unique directory path should be chosen per
  // profile, and the mount point should be added to
  // mount_points_. crbug.com/247236
  base::FilePath home_path;
  if (PathService::Get(base::DIR_HOME, &home_path)) {
    system_mount_points_->RegisterFileSystem(
        "Downloads",
        fileapi::kFileSystemTypeNativeLocal,
        home_path.AppendASCII("Downloads"));
  }

  system_mount_points_->RegisterFileSystem(
      "archive",
      fileapi::kFileSystemTypeNativeLocal,
      chromeos::CrosDisksClient::GetArchiveMountPoint());
  system_mount_points_->RegisterFileSystem(
      "removable",
      fileapi::kFileSystemTypeNativeLocal,
      chromeos::CrosDisksClient::GetRemovableDiskMountPoint());
  system_mount_points_->RegisterFileSystem(
      "oem",
      fileapi::kFileSystemTypeRestrictedNativeLocal,
      base::FilePath(FILE_PATH_LITERAL("/usr/share/oem")));
}

bool CrosMountPointProvider::CanHandleType(fileapi::FileSystemType type) const {
  switch (type) {
    case fileapi::kFileSystemTypeExternal:
    case fileapi::kFileSystemTypeDrive:
    case fileapi::kFileSystemTypeRestrictedNativeLocal:
    case fileapi::kFileSystemTypeNativeLocal:
    case fileapi::kFileSystemTypeNativeForPlatformApp:
      return true;
    default:
      return false;
  }
}

void CrosMountPointProvider::OpenFileSystem(
    const GURL& origin_url,
    fileapi::FileSystemType type,
    fileapi::OpenFileSystemMode mode,
    const OpenFileSystemCallback& callback) {
  DCHECK(fileapi::IsolatedContext::IsIsolatedType(type));
  // Nothing to validate for external filesystem.
  callback.Run(base::PLATFORM_FILE_OK);
}

fileapi::FileSystemQuotaUtil* CrosMountPointProvider::GetQuotaUtil() {
  // No quota support.
  return NULL;
}

void CrosMountPointProvider::DeleteFileSystem(
    const GURL& origin_url,
    fileapi::FileSystemType type,
    fileapi::FileSystemContext* context,
    const DeleteFileSystemCallback& callback) {
  NOTREACHED();
  callback.Run(base::PLATFORM_FILE_ERROR_INVALID_OPERATION);
}

bool CrosMountPointProvider::IsAccessAllowed(
    const fileapi::FileSystemURL& url) const {
  if (!url.is_valid())
    return false;

  // Permit access to mount points from internal WebUI.
  const GURL& origin_url = url.origin();
  if (origin_url.SchemeIs(kChromeUIScheme))
    return true;

  // No extra check is needed for isolated file systems.
  if (url.mount_type() == fileapi::kFileSystemTypeIsolated)
    return true;

  if (!CanHandleURL(url))
    return false;

  std::string extension_id = origin_url.host();
  // Check first to make sure this extension has fileBrowserHander permissions.
  if (!special_storage_policy_->IsFileHandler(extension_id))
    return false;

  return file_access_permissions_->HasAccessPermission(extension_id,
                                                       url.virtual_path());
}

void CrosMountPointProvider::GrantFullAccessToExtension(
    const std::string& extension_id) {
  DCHECK(special_storage_policy_->IsFileHandler(extension_id));
  if (!special_storage_policy_->IsFileHandler(extension_id))
    return;

  std::vector<fileapi::MountPoints::MountPointInfo> files;
  mount_points_->AddMountPointInfosTo(&files);
  system_mount_points_->AddMountPointInfosTo(&files);

  for (size_t i = 0; i < files.size(); ++i) {
    file_access_permissions_->GrantAccessPermission(
        extension_id,
        base::FilePath::FromUTF8Unsafe(files[i].name));
  }
}

void CrosMountPointProvider::GrantFileAccessToExtension(
    const std::string& extension_id, const base::FilePath& virtual_path) {
  // All we care about here is access from extensions for now.
  DCHECK(special_storage_policy_->IsFileHandler(extension_id));
  if (!special_storage_policy_->IsFileHandler(extension_id))
    return;

  std::string id;
  fileapi::FileSystemType type;
  base::FilePath path;
  if (!mount_points_->CrackVirtualPath(virtual_path, &id, &type, &path) &&
      !system_mount_points_->CrackVirtualPath(virtual_path,
                                              &id, &type, &path)) {
    return;
  }

  if (type == fileapi::kFileSystemTypeRestrictedNativeLocal) {
    LOG(ERROR) << "Can't grant access for restricted mount point";
    return;
  }

  file_access_permissions_->GrantAccessPermission(extension_id, virtual_path);
}

void CrosMountPointProvider::RevokeAccessForExtension(
      const std::string& extension_id) {
  file_access_permissions_->RevokePermissions(extension_id);
}

std::vector<base::FilePath> CrosMountPointProvider::GetRootDirectories() const {
  std::vector<fileapi::MountPoints::MountPointInfo> mount_points;
  mount_points_->AddMountPointInfosTo(&mount_points);
  system_mount_points_->AddMountPointInfosTo(&mount_points);

  std::vector<base::FilePath> root_dirs;
  for (size_t i = 0; i < mount_points.size(); ++i)
    root_dirs.push_back(mount_points[i].path);
  return root_dirs;
}

fileapi::FileSystemFileUtil* CrosMountPointProvider::GetFileUtil(
    fileapi::FileSystemType type) {
  DCHECK(type == fileapi::kFileSystemTypeNativeLocal ||
         type == fileapi::kFileSystemTypeRestrictedNativeLocal);
  return local_file_util_->sync_file_util();
}

fileapi::AsyncFileUtil* CrosMountPointProvider::GetAsyncFileUtil(
    fileapi::FileSystemType type) {
  DCHECK(type == fileapi::kFileSystemTypeNativeLocal ||
         type == fileapi::kFileSystemTypeRestrictedNativeLocal);
  return local_file_util_.get();
}

fileapi::CopyOrMoveFileValidatorFactory*
CrosMountPointProvider::GetCopyOrMoveFileValidatorFactory(
    fileapi::FileSystemType type, base::PlatformFileError* error_code) {
  DCHECK(error_code);
  *error_code = base::PLATFORM_FILE_OK;
  return NULL;
}

fileapi::FilePermissionPolicy CrosMountPointProvider::GetPermissionPolicy(
    const fileapi::FileSystemURL& url, int permissions) const {
  if (url.type() == fileapi::kFileSystemTypeRestrictedNativeLocal &&
      (permissions & ~fileapi::kReadFilePermissions)) {
    // Restricted file system is read-only.
    return fileapi::FILE_PERMISSION_ALWAYS_DENY;
  }

  if (!IsAccessAllowed(url))
    return fileapi::FILE_PERMISSION_ALWAYS_DENY;

  // Permit access to mount points from internal WebUI.
  const GURL& origin_url = url.origin();
  if (origin_url.SchemeIs(kChromeUIScheme))
    return fileapi::FILE_PERMISSION_ALWAYS_ALLOW;

  if (url.mount_type() == fileapi::kFileSystemTypeIsolated) {
    // Permissions in isolated filesystems should be examined with
    // FileSystem permission.
    return fileapi::FILE_PERMISSION_USE_FILESYSTEM_PERMISSION;
  }

  // Also apply system's file permission by default.
  return fileapi::FILE_PERMISSION_USE_FILE_PERMISSION;
}

fileapi::FileSystemOperation* CrosMountPointProvider::CreateFileSystemOperation(
    const fileapi::FileSystemURL& url,
    fileapi::FileSystemContext* context,
    base::PlatformFileError* error_code) const {
  DCHECK(url.is_valid());

  if (url.type() == fileapi::kFileSystemTypeDrive) {
    fileapi::RemoteFileSystemProxyInterface* remote_proxy =
        GetRemoteProxy(url.filesystem_id());
    if (!remote_proxy) {
      *error_code = base::PLATFORM_FILE_ERROR_NOT_FOUND;
      return NULL;
    }
    return new chromeos::RemoteFileSystemOperation(remote_proxy);
  }

  DCHECK(url.type() == fileapi::kFileSystemTypeNativeLocal ||
         url.type() == fileapi::kFileSystemTypeRestrictedNativeLocal);
  scoped_ptr<fileapi::FileSystemOperationContext> operation_context(
      new fileapi::FileSystemOperationContext(context));
  operation_context->set_root_path(GetFileSystemRootPath(url));
  return new fileapi::LocalFileSystemOperation(url, context,
                                               operation_context.Pass());
}

scoped_ptr<webkit_blob::FileStreamReader>
CrosMountPointProvider::CreateFileStreamReader(
    const fileapi::FileSystemURL& url,
    int64 offset,
    const base::Time& expected_modification_time,
    fileapi::FileSystemContext* context) const {
  DCHECK(url.is_valid());

  if (url.type() == fileapi::kFileSystemTypeDrive) {
    fileapi::RemoteFileSystemProxyInterface* remote_proxy =
        GetRemoteProxy(url.filesystem_id());
    if (!remote_proxy)
      return scoped_ptr<webkit_blob::FileStreamReader>();
    return remote_proxy->CreateFileStreamReader(
        context->task_runners()->file_task_runner(),
        url, offset, expected_modification_time);
  }

  return scoped_ptr<webkit_blob::FileStreamReader>(
      new fileapi::FileSystemFileStreamReader(
          context, url, offset, expected_modification_time));
}

scoped_ptr<fileapi::FileStreamWriter>
CrosMountPointProvider::CreateFileStreamWriter(
    const fileapi::FileSystemURL& url,
    int64 offset,
    fileapi::FileSystemContext* context) const {
  DCHECK(url.is_valid());

  if (url.type() == fileapi::kFileSystemTypeDrive) {
    fileapi::RemoteFileSystemProxyInterface* remote_proxy =
        GetRemoteProxy(url.filesystem_id());
    if (!remote_proxy)
      return scoped_ptr<fileapi::FileStreamWriter>();
    return scoped_ptr<fileapi::FileStreamWriter>(
        new fileapi::RemoteFileStreamWriter(
            remote_proxy, url, offset,
            context->task_runners()->file_task_runner()));
  }

  if (url.type() == fileapi::kFileSystemTypeRestrictedNativeLocal)
    return scoped_ptr<fileapi::FileStreamWriter>();

  DCHECK(url.type() == fileapi::kFileSystemTypeNativeLocal);
  return scoped_ptr<fileapi::FileStreamWriter>(
      new fileapi::LocalFileStreamWriter(
          context->task_runners()->file_task_runner(), url.path(), offset));
}

bool CrosMountPointProvider::GetVirtualPath(
    const base::FilePath& filesystem_path,
    base::FilePath* virtual_path) {
  return mount_points_->GetVirtualPath(filesystem_path, virtual_path) ||
         system_mount_points_->GetVirtualPath(filesystem_path, virtual_path);
}

fileapi::RemoteFileSystemProxyInterface* CrosMountPointProvider::GetRemoteProxy(
    const std::string& mount_name) const {
  fileapi::RemoteFileSystemProxyInterface* proxy =
      mount_points_->GetRemoteFileSystemProxy(mount_name);
  if (proxy)
    return proxy;
  return system_mount_points_->GetRemoteFileSystemProxy(mount_name);
}

base::FilePath CrosMountPointProvider::GetFileSystemRootPath(
    const fileapi::FileSystemURL& url) const {
  DCHECK(fileapi::IsolatedContext::IsIsolatedType(url.mount_type()));
  if (!url.is_valid())
    return base::FilePath();

  base::FilePath root_path;
  std::string mount_name = url.filesystem_id();
  if (!mount_points_->GetRegisteredPath(mount_name, &root_path) &&
      !system_mount_points_->GetRegisteredPath(mount_name, &root_path)) {
    return base::FilePath();
  }

  return root_path.DirName();
}

}  // namespace chromeos
