"""Post-commit hook to submit the commit to CIA (http://cia.navi.cx/)

Requires bzr 0.15 or higher.

Copyright (C) 2005-2008 Jelmer Vernooij <jelmer@samba.org>
Published under the GNU GPLv2 or later

Installation:
Copy this directory to ~/.bazaar/plugins/cia/*

Configuration:
In ~/.bazaar/bazaar.conf, set 'cia_user' to your username on CIA.

Other options that can be set are 'cia_server' and 'cia_quiet', and
'cia_send_revno'

To enable CIA reporting for a particular branch, run:
$ bzr cia-project <name> 
inside that branch
"""

version_info = (1, 0, 0, 'dev', 0)

import socket
import sys
import xmlrpclib
from xml.sax import saxutils

import bzrlib
from bzrlib.branch import Branch
from bzrlib.commands import Command, register_command
from bzrlib.option import Option
import bzrlib.errors as errors
from bzrlib.trace import info, warning


def branch_commit_hook(local, master, old_revno, old_revid, 
                                      new_revno, new_revid):
    if local is None:
        return cia_submit(master, new_revid, new_revno)
    else:
        return cia_submit(local, new_revid, new_revno)


class BacklogList:
    def __init__(self, filename):
        self.filename = filename

    def add_revision(self, revid):
        pass

    def read_revisions(self):
        pass


def generate_cia_xml(repository, revid, project, revname=None, author=None):
    revision = repository.get_revision(revid)
    delta = repository.get_revision_delta(revid)

    if revname is None:
        revname = revid

    if author is None:
        author = revision.committer

    files = []
    [files.append(f) for (f,_,_) in delta.added]
    [files.append(f) for (f,_,_) in delta.removed]
    [files.append(f) for (_,f,_,_,_,_) in delta.renamed]
    [files.append(f) for (f,_,_,_,_) in delta.modified]

    return """
<message>
  <generator> 
    <name>bzr</name> 
    <version>%s</version> 
    <url>http://samba.org/~jelmer/bzr/cia_bzr.py</url>
  </generator>
  <source>
    <project>%s</project>
    <module>%s</module>
  </source>
  <timestamp>%d</timestamp>
  <body>
    <commit>
      <revision>%s</revision>
      <files>%s</files>
      <author>%s</author>
      <log>%s</log>
    </commit>
  </body>
</message>
""" % (bzrlib.version_string, project, revision.properties['branch-nick'], revision.timestamp, revname,
        "\n".join(["<file>%s</file>" % saxutils.escape(f) for f in files]), 
        saxutils.escape(author), 
        saxutils.escape(revision.message))


def cia_submit(branch, revid, revno, dry_run=False):
    config = branch.get_config()

    project = config.get_user_option('cia_project')
    if project is None:
        return

    server = config.get_user_option('cia_server')
    if server is None:
        server = "http://cia.navi.cx"

    author = config.get_user_option('cia_user')
    quiet = config.get_user_option('cia_quiet')

    use_revno = config.get_user_option('cia_send_revno')
    if use_revno is not None and use_revno.lower()[:1] in ('t', '1'):
        revspec = revno
    else:
        revspec = revid

    msg = generate_cia_xml(branch.repository, revid, project, revspec, author)

    if not quiet:
        info("Submitting revision to CIA.")
    if not dry_run:
        try:
            xmlrpclib.ServerProxy(server).hub.deliver(msg)
        except xmlrpclib.ProtocolError, e:
            warning("Unable to submit revision to CIA: %s" % e.errmsg)
        except socket.gaierror, (_, msg):
            warning("Unable to submit revision to CIA: %s" % msg)
    else:
        print msg


class cmd_cia_project(Command):
    """Print or set project to submit changes to on CIA.

    """
    takes_args = ['project?']
    takes_options = []

    def run(self, project=None):
        br = Branch.open('.')
        
        config = br.get_config()
        if project:
            config.set_user_option('cia_project', project)
        else:
            print config.get_user_option('cia_project')


class cmd_cia_submit(Command):
    """Submit a revision to CIA

    """
    takes_args = ['branch?']
    takes_options = ['revision', 
        Option('dry-run', 
            help="Show what would be done, but don't actually do anything.")]

    def run(self, branch='.', revision=None,dry_run=False):
        br = Branch.open(branch)

        if revision is None:
            revs = [br.last_revision()]
        elif len(revision) == 1:
            revs = [revision[0].in_history(br).rev_id]
        elif len(revision) == 2:
            if revision[0].spec is None:
                from_revno = 1
            else:
                from_revno = revision[0].in_history(br).revno

            if revision[1].spec is None:
                to_revno = br.revno()
            else:
                to_revno = revision[1].in_history(br).revno
            revs = br.revision_history()[from_revno:to_revno]
        else:
            raise BzrCommandError('bzr submit-cia --revision takes one or two arguments')

        for rev_id in revs:
            cia_submit(br, rev_id, br.revision_id_to_revno(rev_id), dry_run)
        
        return 0


register_command(cmd_cia_project)
register_command(cmd_cia_submit)

def install_hooks():
    """Install CommitSender to send after commits with bzr >= 0.15 """
    if getattr(Branch.hooks, 'install_named_hook', None):
        Branch.hooks.install_named_hook('post_commit', branch_commit_hook, 
                                        'CIA')
    else:
        Branch.hooks.install_hook('post_commit', branch_commit_hook)
        if getattr(Branch.hooks, 'name_hook', None):
            Branch.hooks.name_hook(branch_commit_hook, "cia-bzr")


install_hooks()

def test_suite():
    from unittest import TestSuite, TestLoader
    import tests
    suite = TestSuite()
    suite.addTest(tests.test_suite())
    return suite
