/* 
 * icc_examin plug-in for cinepaint.
 *
 * Copyright (C) 2004-2005 Kai-Uwe Behrmann <ku.b@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * copies an assigned ICC profil to $TMP_DIR and call icc_examin
 *
 * add writing of image samples out to an profile - name: plug_in_icc_watch
 *  2005-02-28
 * bugfixes
 * query only with found icc_examin executable
 *  2005-04-28
 *  
 */

#define PLUG_IN_NAME          "plug_in_icc_examin"
#define PLUG_IN_NAME2         "plug_in_icc_watch"
#define PLUG_IN_BRIEF         "shows ICC profile of an image"
#define PLUG_IN_BRIEF2        "shows image colours compared to ICC profile"
#define PLUG_IN_DESCRIPTION   "Loads an assigned ICC profil from image to icc_examin."
#define PLUG_IN_DESCRIPTION2  "Shows some colours of the image in a ICC Examin including profile gamut"
#define PLUG_IN_VERSION       "0.2.0 - 28 April 2005"
#define PLUG_IN_AUTHOR        "Kai-Uwe Behrmann <ku.b@gmx.de>"
#define PLUG_IN_COPYRIGHT     "2004/2005 Kai-Uwe Behrmann"

/***   includes   ***/

#include <string>
#include <sstream>
#include <iostream>
#include <fstream>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <vector>

using namespace std;


extern "C" {
#include "lib/plugin_main.h"
#include "lib/wire/libtile.h"
#include "plugin_pdb.h"
#include <lcms.h>
#include <icc34.h>
}


/*** local macros ***/
//#define DEBUG
#ifdef DEBUG
#define DBG_ cout << __FILE__<<":"<<__LINE__ <<" "<< __func__ << "() " ;
#define DBG  {DBG_ cout << endl;}
#define DBG_S(text) { DBG_ cout  << " "<< text << endl; }
#else
#define DBG
#define DBG_S(text)
#endif

#ifdef __cplusplus
extern "C" {
#endif

  // --- plug-in API specific ---
static void   query      (void);
static void   run        (char    *name,
			  int      nparams,
			  GParam  *param,
			  int     *nreturn_vals,
			  GParam **return_vals);

GPlugInInfo PLUG_IN_INFO =
{
  NULL,    /* init_proc */
  NULL,    /* quit_proc */
  query,   /* query_proc */
  run,     /* run_proc */
};
static int n_args_;
static int n_return_vals_;

struct Bvals {
  int ein_Argument;
} bvals;
#ifdef __cplusplus
} /* extern "C" */
#endif

namespace icc_examin_cp {

/*** struct definitions ***/

typedef struct {
  gint32        ID;
  GimpDrawable *drawable;
  GDrawableType/*GimpDrawableType*/ drawable_type;
  GimpPixelRgn  srcRgn;          // Ausgangsbild
  //GimpPixelRgn  dstRgn;          // Schattenbild
  guchar       *pixels,          // raw pixel buffer for colors
               *pixel;           // pointer to actual position in pixels
  int           bytes, samplesperpixel, colours;
  int           width, height;   // Gesamtbreite und -hoehe
  int           offx, offy;      // Ebenenversaetze
} channel;


char data[320] =
  {
    0,0,1,64,108,99,109,115,
    2,48,0,0,110,109,99,108,
    82,71,66,32,76,97,98,32,
    0,0,0,0,0,0,0,0,
    0,0,0,0,97,99,115,112,
    83,71,73,32,0,0,0,0,
    110,111,110,101,110,111,110,101,
    -64,48,11,8,-40,-41,-1,-65,
    0,0,0,0,0,0,-10,-42,
    0,1,0,0,0,0,-45,45,
    67,80,0,0,-32,51,12,8,
    10,0,0,0,40,-40,-1,-65,
    -12,-17,37,64,-12,-17,37,64,
    -64,108,1,64,8,-40,-1,-65,
    -97,-51,13,64,-32,51,12,8,
    -128,48,11,8,40,-40,-1,-65,
    0,0,0,3,100,101,115,99,
    0,0,0,-88,0,0,0,33,
    99,112,114,116,0,0,0,-52,
    0,0,0,29,110,99,108,50,
    0,0,0,-20,0,0,0,84,
    116,101,120,116,0,0,0,0,
    67,105,110,101,80,97,105,110,
    116,32,99,111,108,111,117,114,
    32,115,97,109,112,108,101,115,
    0,0,0,0,116,101,120,116,
    0,0,0,0,110,111,116,32,
    99,111,112,121,114,105,103,104,
    116,101,100,32,100,97,116,97,
    0,0,0,0,110,99,108,50,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,3,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0
  };

struct Ncl2Farbe {
  char name[32];
  icUInt16Number pcsfarbe[3]; // PCS Lab oder XYZ
  icUInt16Number geraetefarbe[16];
};

struct Ncl2 {
  char vendor_flag[4];
  icUInt32Number anzahl;
  icUInt32Number koord;
  char vorname[32];
  char nachname[32];
  Ncl2Farbe *farben;
};



/***   global variables   ***/

cmsHPROFILE hl;                // lcms Strukturen
cmsHPROFILE hp;
cmsHTRANSFORM transf = 0;
long format;
int farb_kanaele;              // Farbkanaele wie im Bildprofil angegeben
double *colour = 0;            // gemessene Farben : 0.0 -> 1.0
double *outbuf = 0;            // nach Lab umgewandelte Farben
char* colour_profile = 0;      // Messfarbprofile (Schmuckfarben)
char *image_profile = NULL;    // Bildprofil
std::vector<double>       pcsfarbe;       // -> ungerechnete Farben: CIE*Lab 
std::vector<double>       geraetefarbe;   // Bildfarben
std::vector<std::string>  name;           // Farbnamen
std::string a, b, tn;          // Dateinamen und Befehlszeile
size_t tag_size;               // ncl2 Abschnittsgroesse
int x_num;                     // Anzahl Messpunkte in x/y Richtung
int y_num;
int l = 30;
int x_diff;                    // Raster Abstaende
int y_diff;
int x_start;                   // Startpunkte des Messrasters
int y_start;
int min_x, min_y, max_x, max_y;// Intensitaets Maxima und Minima
gint32 nlayers;                // beteiligte Ebenen
int    n_points;               // Anzahl Messpunkte
gint32 image_ID;               // CinePaint Bildnummer

bool farben_sind_gleich = true;
int  intent_alt = -12;
static bool erstes_mal = true;
static bool farbe_pruefen_laeuft = false;
}

using namespace icc_examin_cp;


/*** declaration of local functions ***/


  // --- process functions ---
static int      dialog_   (gint32   image_ID);
static int      doExamin  (gint32   image_ID);
static int      doWatch   (gint32   image_ID);
static int      checkEmbeddedProfile( channel *layers );
static void     minMax    (gint32   image_ID, int & min_x, int & min_y,
                                    int & max_x, int & max_y );
static void     getColour  (channel* layers, int n,
                            const unsigned char* data, double *colour,
                            const int & x, const int & y );

static channel* holeLayer      (int image_ID);
static channel* schlieseLayer  (channel* layers);
static void     schreibeProfil (icUInt32Number intent);
static void     schreibeZeiger (icUInt32Number intent);
static void     holeFarbPunkt  (channel* layers, int & x, int & y,
                                unsigned char *buf, int &n,
                                gint32 &image_ID, int &colour_x);
void            setze_renderingIntent (char *header, icUInt32Number intent);
size_t          berechneTagGroesse (int farben_n, int farb_kanaele);


void*           waechter (void* zeiger);
void            pthreatFehler (int fehler);
void            aufraeumen(channel *layers);


/*** functions ***/

MAIN()

static void
query ()
{
  int dependency_error = 0;
  static GParamDef args[] =
  {
    { PARAM_INT32, "run_mode", "Interactive, non-interactive" },
    { PARAM_IMAGE, "image", "Input Image" },
    { PARAM_DRAWABLE, "drawable", "Input Drawable" },
  };
  static GParamDef return_vals[] =
  {
    { PARAM_IMAGE, "image", "Output Image" },
  };
  n_args_ = sizeof (args) / sizeof (args[0]);
  n_return_vals_ = sizeof (return_vals) / sizeof (return_vals[0]);

  #ifndef WIN32
  dependency_error = system("which icc_examin");
  #endif

  if(!dependency_error)
  {
    gimp_install_procedure (PLUG_IN_NAME,
                            PLUG_IN_BRIEF,
                            PLUG_IN_DESCRIPTION,
                            PLUG_IN_AUTHOR,
                            PLUG_IN_COPYRIGHT,
                            PLUG_IN_VERSION,
                            "<Image>/Filters/ICC Color/ICC Examin",
                            "*",
                            PROC_PLUG_IN,
                            n_args_, n_return_vals_,
                            args, return_vals);

    gimp_install_procedure (PLUG_IN_NAME2,
                            PLUG_IN_BRIEF2,
                            PLUG_IN_DESCRIPTION,
                            PLUG_IN_AUTHOR,
                            PLUG_IN_COPYRIGHT,
                            PLUG_IN_VERSION,
                            "<Image>/Filters/ICC Color/ICC Watch",
                            "*",
                            PROC_PLUG_IN,
                            n_args_, n_return_vals_,
                            args, return_vals);
  }
}

static void
run (char    *name,
     int      nparams,
     GParam  *param,
     int     *nreturn_vals,
     GParam **return_vals)
{
  GParam values[n_return_vals_];
  gint32 image_ID;
  GRunModeType run_mode;
  GStatusType status = GIMP_PDB_SUCCESS;

  run_mode = (GRunModeType) param[0].data.d_int32;

  *nreturn_vals = n_return_vals_;
  *return_vals = values;
  values[0].type = PARAM_STATUS;
  values[0].data.d_status = STATUS_CALLING_ERROR;

  if (strcmp (name, PLUG_IN_NAME) == 0)
  {
  switch (run_mode)
    {
    case RUN_INTERACTIVE:
      // gespeicherte Daten abholen
      gimp_get_data (PLUG_IN_NAME, &bvals);

      // einen Dialog oeffnen
      if (! dialog_ (param[1].data.d_image))
	return;
      image_ID = doExamin (param[1].data.d_image);
      break;

    case RUN_NONINTERACTIVE:
      // das Ganze automatisch
      if (nparams != n_args_) 
	status = STATUS_CALLING_ERROR;
      if (status == STATUS_SUCCESS)
	{
	  bvals.ein_Argument = (int)param[2].data.d_drawable;
	}
      image_ID = doExamin (param[1].data.d_image);

      *nreturn_vals = n_return_vals_ + 1;
      values[0].data.d_status = STATUS_SUCCESS;
      values[1].type = PARAM_IMAGE;
      values[1].data.d_image = image_ID;
      break;

    case RUN_WITH_LAST_VALS:
      // gespeicherte Daten abholen
      gimp_get_data (PLUG_IN_NAME, &bvals);
      break;

    default:
      break;
    }
  } else
  if (strcmp (name, PLUG_IN_NAME2) == 0)
  {
  switch (run_mode)
    {
    case RUN_INTERACTIVE:
      // gespeicherte Daten abholen
      gimp_get_data (PLUG_IN_NAME, &bvals);

      // einen Dialog oeffnen
      if (! dialog_ (param[1].data.d_image))
        return;
      image_ID = doWatch (param[1].data.d_image);
      break;

    case RUN_NONINTERACTIVE:
      // das Ganze automatisch
      if (nparams != n_args_) 
	status = STATUS_CALLING_ERROR;
      if (status == STATUS_SUCCESS)
	{
	  bvals.ein_Argument = (int)param[2].data.d_drawable;
	}
      image_ID = doWatch (param[1].data.d_image);

      *nreturn_vals = n_return_vals_ + 1;
      values[0].data.d_status = STATUS_SUCCESS;
      values[1].type = PARAM_IMAGE;
      values[1].data.d_image = image_ID;
      break;

    case RUN_WITH_LAST_VALS:
      // gespeicherte Daten abholen
      gimp_get_data (PLUG_IN_NAME, &bvals);
      break;

    default:
      break;
    }
  }
  values[0].data.d_status = status;
}

static gint32
dialog_ (gint32 image_ID)
{
  return true;
}

static gint32
doExamin (gint32 image_ID)
{
  DBG_S( "Bild: " << image_ID )

  char   *mem_profile=NULL;
  gint32  size;

  if (gimp_image_has_icc_profile(image_ID)) {
    mem_profile = gimp_image_get_icc_profile_by_mem(image_ID, &size);
  } else {
    g_message (_("No profil assigned to image."));
    return -1;
  }


  DBG_S( (int*)mem_profile << " " << size )

  if (size && mem_profile) {
    std::stringstream profil_temp_name;
    if(getenv("TMPDIR"))
      profil_temp_name << getenv("TMPDIR") << "/icc_examin_temp.icc" ;
    else
      profil_temp_name << "/tmp/icc_examin_temp.icc";
    std::ofstream f ( profil_temp_name.str().c_str(),  std::ios::out );
    f.write ( mem_profile, size );
    f.close();
    std::string tn = "icc_examin ";
    tn += profil_temp_name.str();
    system (tn.c_str());
    remove( profil_temp_name.str().c_str() );
  } else
    g_message (_("Profil not written."));

  return image_ID;
}

icUInt32Number
icValue (icUInt32Number val)
{
#if BYTE_ORDER == LITTLE_ENDIAN
  unsigned char        *temp = (unsigned char*) &val;

  static unsigned char  uint32[4];

  uint32[0] = temp[3];
  uint32[1] = temp[2];
  uint32[2] = temp[1];
  uint32[3] = temp[0];

  unsigned int *erg = (unsigned int*) &uint32[0];

  return (int) *erg;
#else
  return (int)val;
#endif
}

icUInt16Number
icValue (icUInt16Number val)
{ 
#if BYTE_ORDER == LITTLE_ENDIAN
  #define BYTES 2
  #define KORB  4
  unsigned char        *temp  = (unsigned char*) &val;
  static unsigned char  korb[KORB];
  for (int i = 0; i < KORB ; i++ )
    korb[i] = (int) 0;  // leeren
    
  int klein = 0,
      gross = BYTES - 1;
  for (; klein < BYTES ; klein++ ) {
    korb[klein] = temp[gross--];
  }

  icUInt16Number *erg = (icUInt16Number*) &korb[0];

  return (icUInt16Number)*erg;
#else
  return (icUInt16Number)val;
#endif
}

void
setze_renderingIntent (char *header, icUInt32Number intent)
{
  icProfile *p = (icProfile *)header;
  p->header.renderingIntent = icValue( intent );
}

size_t
berechneTagGroesse (int farben_n, int farb_kanaele)
{
  size_t groesse = 8 + 76 +
                   (38 + farb_kanaele * sizeof(icUInt16Number)) * farben_n;
  return groesse;
}

char*
schreibeNcl2Tag              ( std::vector<double>       pcsfarbe,
                               std::vector<double>       geraetefarbe,
                               int                       farb_kanaele,
                               const char*               vorname,
                               std::vector<std::string>  name,
                               const char*               nachname)
{
  int    farben_n = pcsfarbe.size() / 3;
  size_t groesse  = berechneTagGroesse( farben_n, farb_kanaele );

  DBG_S( "farb_kanaele: " << farb_kanaele <<" farben_n: "<< farben_n )

  char* tag_block = (char*) new char [groesse];

  DBG_S( "tag_block: " << (int*)tag_block <<" groesse: "<< groesse )

  for(size_t i = 0; i < groesse; ++i)
    tag_block[i] = 0;

  // 0: Anzahl Farben
  // 1...n: CIE*Lab Farbwerte
  // n = 3 * FarbAnzahl

  Ncl2 *ncl2 = (Ncl2*) &tag_block[8];

  ncl2->anzahl = icValue((icUInt32Number)farben_n);
  ncl2->koord  = icValue((icUInt32Number)farb_kanaele);
  if(vorname && strlen(vorname) < 32)
    sprintf(ncl2->vorname, vorname);
  if(nachname && strlen(nachname) < 32)
    sprintf(ncl2->nachname, nachname);

  DBG_S( cout << farben_n <<" "<< pcsfarbe.size() )

  for (int i = 0; i < farben_n; ++i)
  {
    //cout << i << " "; DBG
    Ncl2Farbe *f = (Ncl2Farbe*) ((char*)ncl2 + 76 + // Basisgroesse von Ncl2
                   (i * (38 +                 // Basisgroesse von Ncl2Farbe
                         farb_kanaele         // Anzahl Geraetefarben
                         * sizeof(icUInt16Number))));//Ncl2Farbe::geraetefarbe
    f->pcsfarbe[0] = icValue((icUInt16Number)(pcsfarbe[3*i+0]*65280.0));
    f->pcsfarbe[1] = icValue((icUInt16Number)(pcsfarbe[3*i+1]*65535.0));
    f->pcsfarbe[2] = icValue((icUInt16Number)(pcsfarbe[3*i+2]*65535.0));
    for(int j=0; j < farb_kanaele; ++j)
      f->geraetefarbe[j] = icValue((icUInt16Number)
                                   (geraetefarbe[farb_kanaele*i+j]*65535.0));
    // TODO Zeiger hier markieren
    if (name.size() && name[i].size() < 32)
      sprintf(f->name, name[i].c_str());

    #ifdef DEBUG_
    DBG_S(  icValue(f->pcsfarbe[0]) << "," << pcsfarbe[3*i+0] <<
            icValue(f->pcsfarbe[1]) << "," << pcsfarbe[3*i+1] <<
            f->pcsfarbe[2] << " " << pcsfarbe[3*i+2] <<
            f->geraetefarbe[0] << " " <<
            f->geraetefarbe[1] << " " <<
            f->geraetefarbe[2] )
    #endif

  }

  icTag ic_tag;

  ic_tag.size = icValue ((icUInt32Number)groesse);
  ic_tag.offset = 0;
  memcpy(&ic_tag.sig, "ncl2", 4);

  char sig[] = "ncl2";
  memcpy (&tag_block[0], &sig, 4);

  return tag_block;
}


void
transform_anlegen( int intent )
{
  if(transf)
    cmsDeleteTransform (transf);
  transf =
     cmsCreateTransform (hp, format,
                         hl, TYPE_Lab_DBL,
                         intent,
                         cmsFLAGS_NOTPRECALC);

}

bool
vergleicheFarben(void* zeiger)
{
  farbe_pruefen_laeuft = true;
  DBG
  channel* layers = 0;
  if(zeiger)
    layers = (channel*) zeiger;

  DBG_S( "layers "<< (int*)layers )

  // Farbgedaechtnis - static ist vielleicht gefaehrlich?
  static std::vector<double> vorherige_farben;

  DBG_S( "zeiger " << (int*)zeiger )
  #define WARN_S(text) { cout << text << " "; DBG }

  if(!layers) {
    farbe_pruefen_laeuft = false;
    return true;
  }

  DBG
  pcsfarbe.clear();       // -> ungerechnete Farben: CIE*Lab 
  geraetefarbe.clear();   // Bildfarben
  name.clear();           // Farbnamen
  // Das Bild befragen im Gitterraster
  guchar buf[128]; // Punktspeicher
  int colour_x = 0; // Zaehler
  int x_punkt = 0 , y_punkt = 0;
  int n = 0;
  gint32 drawable_ID = gimp_image_get_active_layer (image_ID);
  while( layers[n].ID != drawable_ID )
    ++n;

  if(nlayers)
    for( int x = 0; x < x_num; ++x )
      for( int y = 0; y < y_num; ++y )
      {
        x_punkt = x_start + x*x_diff;
        y_punkt = y_start + y*y_diff;

        holeFarbPunkt(layers, x_punkt, y_punkt,
                      buf, n, image_ID, colour_x);
        ++colour_x;
      }
DBG
    // Maximalwerte
  holeFarbPunkt(layers, min_x, min_y, buf, n, image_ID, colour_x);
  ++colour_x;
  holeFarbPunkt(layers, max_x, max_y, buf, n, image_ID, colour_x);
DBG
  // Intent anpassen
  icUInt32Number intent = gimp_display_get_cms_intent (image_ID);
  if((int)intent != intent_alt) {
    transform_anlegen( intent );
    intent_alt = intent;
  } else {
    // Vergleich der vorherigen Auslese
    bool gleichviele = true;
    farben_sind_gleich = true;
    if( (int)vorherige_farben.size() != n_points*farb_kanaele ) {
      gleichviele = false;
      farben_sind_gleich = false;

      DBG_S( "n_points: " << n_points << " vorherige_farben.size(): " << vorherige_farben.size() )

    }
DBG
    for(int i = 0; i < n_points*farb_kanaele; ++i)
    {
      //DBG_S( i << " " << n_points*farb_kanaele )
      if(gleichviele && colour[i] != vorherige_farben[i])
      {
        farben_sind_gleich = false;
      }
      if(i < (int)vorherige_farben.size())
        vorherige_farben[i] = colour[i];
      else
        vorherige_farben.push_back(colour[i]);
    }

      // Wir koennen das weitere auslassen
    if(farben_sind_gleich) {
      farbe_pruefen_laeuft = false;
      DBG_S("colours are equal")
      //schreibeZeiger( intent );
      return false;
    }
  }


  DBG_S( colour_x << " " << n_points )
  
  cmsDoTransform( transf, colour, outbuf, n_points);


  // Berechnung Auswerten ...
  for(int i = 0; i < n_points; ++i)
  {
    pcsfarbe.push_back( outbuf[3*i+0]/100.0);
    pcsfarbe.push_back((outbuf[3*i+1]+128.0)/255.0);
    pcsfarbe.push_back((outbuf[3*i+2]+128.0)/255.0);

    //DBG_S( pcsfarbe[farb_kanaele*i+0] << "," << pcsfarbe[farb_kanaele*i+1] << "," << pcsfarbe[farb_kanaele*i+2] )

    for(int j = 0; j < farb_kanaele; ++j)
    {
      geraetefarbe.push_back( colour[farb_kanaele*i+j]/100.0 );

      //DBG_S( colour[farb_kanaele*i+j] )
    }
  }

  // ... und das Profil vervollstaendigen
  memcpy(colour_profile, data, 320);

    // Profilegroesse

  char zahl[4];
  *((icUInt32Number*)zahl) = icValue(236 + (icUInt32Number)tag_size);
  memcpy(colour_profile, zahl, 4);
    // Abschnittsgroesse
  *((icUInt32Number*)zahl) = icValue((icUInt32Number)tag_size);
  memcpy(&colour_profile[164], zahl, 4);

  DBG_S( (int*)image_profile << " " << tag_size )

  schreibeProfil( intent );

  farbe_pruefen_laeuft = false;

  return false;
}


#include <unistd.h>
#include <pthread.h>

void*
waechter (void* zeiger)
{
  // Es gibt zwei threads.
  // Der erste Neben-thread started eine while Schleife zum Beobachten
  // des Bildes. Der Haupthread beobachtet ICC Examin.

  DBG

  bool bin_erste = false;
  if(erstes_mal)
  {
    bin_erste = true;
    erstes_mal = false;
  }

  //ICCkette* obj = (ICCkette*) zeiger;
  channel* layers = 0;
  if(zeiger)
    layers = (channel*) zeiger;

  DBG_S( (int*)layers )
  
  int fehler = false;

  DBG_S( "bin_erste: " << bin_erste )

  static bool freilauf = true;

  // Bild beobachten
  if(bin_erste)
  {
    int sl = 1000000;
    while(!fehler && freilauf)
    {
      double rz = (double)clock()/(double)CLOCKS_PER_SEC;

      fehler = checkEmbeddedProfile( layers );
      if(!fehler)
      fehler = vergleicheFarben( layers );

      rz = (double)clock()/(double)CLOCKS_PER_SEC - rz;
      if(farben_sind_gleich) {
        sl = max(sl,500000);
        usleep(sl*2);
      } else {
        sl = (int)(rz*1000000.0)*4;
        sl = max(sl,50000);
        usleep(sl);
        DBG_S( "rz: " << rz*1000000 << " sl " << sl )
      }
    }
    DBG_S( "bin_erste: " << bin_erste )
    sleep(10);
    DBG_S( "bin_erste: " << bin_erste )
  }

  // ICC Examin starten
  if(!bin_erste)
  {
    tn = "icc_examin ";
    tn += b;  // die Farben
    tn += " ";
    tn += a;  // das Bildprofil

    DBG_S( tn )

    system (tn.c_str());

    DBG_S( "bin_erste: " << bin_erste )
    freilauf = false;
    while(farbe_pruefen_laeuft)
    {
      DBG_S( "bin_erste: " << bin_erste )
      sleep(1);
      DBG_S( "bin_erste: " << bin_erste )
    }
    freilauf = true;
    DBG_S( "bin_erste: " << bin_erste )
  }

  if(freilauf)
  {
    freilauf = false;
    aufraeumen( layers );
    DBG_S( "bin_erste: " << bin_erste )
  } else
    DBG_S( "bin_erste: " << bin_erste );

  return layers;
}

#ifndef EAGAIN
#define EAGAIN 11
#endif
#ifndef PTHREAD_THREADS_MAX
#define PTHREAD_THREADS_MAX 16384
#endif

void
pthreatFehler (int fehler)
{
  if( fehler == EAGAIN)
  {
    WARN_S( _("Waechter Thread nicht gestartet Fehler: ")  << fehler );
  } else
  if( fehler == 64/*PTHREAD_THREADS_MAX -- linux*/ )
  {
    WARN_S( _("zu viele Waechter Threads Fehler: ") << fehler );
  } else
  if( fehler != 0 )
  {
    WARN_S( _("unbekannter Fehler beim Start eines Waechter Threads Fehler: ") << fehler );
  }
}

void
aufraeumen(channel *layers)
{
  {
    while(farbe_pruefen_laeuft) {
      DBG_S( "farbe_pruefen_laeuft " << farbe_pruefen_laeuft )
      sleep(1);
    }
    // Aufraeumen
    DBG
    remove(a.c_str());
    remove(b.c_str());
    if(colour_profile) delete [] colour_profile;
    if(colour) delete [] colour;
    if(outbuf) delete [] outbuf;
    cmsDeleteTransform (transf);
    cmsCloseProfile (hl);
    cmsCloseProfile (hp);
    if(layers) layers = schlieseLayer(layers);
  }
}

static int
checkEmbeddedProfile( channel *layers )
{
  // hat sich der Profilname geaendert?
  if (!gimp_image_has_icc_profile(image_ID)) {
    g_message (_("No profil assigned to image."));
    return 1;
  }

  std::string profil_name = gimp_image_get_icc_profile_description(image_ID);
  static std::string old_profil_name;
  DBG_S( image_ID <<": "<< profil_name <<" "<< old_profil_name )

  if( profil_name.size() &&
      profil_name == old_profil_name )
  {
    old_profil_name = profil_name;
    return 0;
  }
  old_profil_name = profil_name;
  intent_alt = -12;

  gint32   size;
  image_profile = gimp_image_get_icc_profile_by_mem(image_ID, &size);

  // Speichern des eingebetteten Profiles
  if(image_profile && size)
  {
    std::ofstream f;
    f.clear();
    f.open ( a.c_str(),  std::ios::out );
    if(f.good())
    {
      f.write ( image_profile, size );
    }
    f.close();
  } else
    g_message (_("Profile not written."));

  // Berechnung -> CIE*Lab vorbereiten
  hl   = cmsCreateLabProfile( cmsD50_xyY() );
  hp   = cmsOpenProfileFromMem( image_profile, size );
  farb_kanaele = _cmsChannelsOf( cmsGetColorSpace( hp ) );

  format = (COLORSPACE_SH(PT_ANY)|CHANNELS_SH(farb_kanaele)|BYTES_SH(0));
  if(farb_kanaele < layers[0].samplesperpixel)
    format |= EXTRA_SH(layers[0].samplesperpixel - farb_kanaele);

  transform_anlegen(gimp_display_get_cms_intent (image_ID));
  return 0;
}

static int
doWatch (int image_ID_)
{
  image_ID = image_ID_;

  DBG_S( "Bild: " << image_ID )

  std::stringstream profil_temp_name;

  if(getenv("TMPDIR")) {
    profil_temp_name << getenv("TMPDIR") << "/cinepaint_" << time(0) ;
    DBG_S( getenv("TMPDIR") )
  } else {
    profil_temp_name << "/tmp/cinepaint_" << time(0) ;
  }

  DBG_S( profil_temp_name.str() )

  a = profil_temp_name.str(); a.append(".icc");
  b = profil_temp_name.str(); b.append(".icm");
  
  // Daten sammeln
  gimp_image_get_layers (image_ID, &nlayers);
  
  channel *layers = holeLayer( image_ID );
  if(!layers) {
    DBG_S( "no layers found" )
  }
  int w = gimp_image_width(image_ID);
  int h = gimp_image_height(image_ID);

  // Speichern des eingebetteten Profiles -> a
  if (checkEmbeddedProfile( layers ))
    return -1;

  // Min/Max bestimmen
  minMax( image_ID, min_x, min_y, max_x, max_y );

  // Abtastraster festlegen
  if(w > h)
  {
    x_num = l;
    y_num = (int)(x_num * (double)h/w +0.5);
  } else {
    y_num = l;
    x_num = (int)(y_num * (double)w/h +0.5);
  }
  x_diff = w / x_num;
  y_diff = h / y_num;
  x_start = (int)((double)x_num/w / 2);
  y_start = (int)((double)y_num/h / 2);
  #ifdef DEBUG
  cout << x_diff <<","<< y_diff << " "; DBG
  cout << x_start <<","<< y_start << " "; DBG
  cout << x_num <<","<< y_num << " "; DBG
  #endif

  n_points = x_num * y_num + 2;

  DBG_S( n_points )

  outbuf = (double*) new double [n_points*3];

  // weiter Aufbereiten
  tag_size  = berechneTagGroesse( n_points, farb_kanaele );
  colour_profile = (char*) new char [320 + tag_size];

  DBG_S( "320 + tag_size: " << 320 + tag_size )

  for(int i = 0; i < nlayers; ++i)
    layers[i].colours = farb_kanaele;

  colour = (double*) new double [n_points * farb_kanaele];
  for(int i = 0; i < n_points*farb_kanaele; ++i)
    colour[i] = 0.0;


  // Farben Messen und in bestimmten Zeitabstaenden wiederholen
  pthread_t p_t;
  bool fehler = true;
  if(layers)
  {
    fehler = vergleicheFarben( layers );
    if(!fehler)
    {
      fehler = pthread_create(&p_t, NULL, &waechter, (void *)layers);
      if(fehler) pthreatFehler (fehler);
    }
    DBG
  }

  DBG
  // starte icc_examin und warte auf seine Beendigung 
  if(!fehler) {
    waechter(layers);
    DBG
  }

  DBG_S( "end of " <<__func__ )

  return image_ID;
}

static void
holeFarbPunkt (channel* layers, int & x_punkt, int & y_punkt,
               unsigned char *buf, int & n, gint32 & image_ID,
               int &colour_x)
{

        #ifdef DEBUG
        cout << n <<": " << x_punkt <<","<< y_punkt << " "; DBG
        #endif
        gimp_pixel_rgn_get_pixel( &layers[n].srcRgn, buf, x_punkt, y_punkt);

        getColour ( layers, n, buf, &colour[colour_x*farb_kanaele],
                    x_punkt, y_punkt);

        colour[colour_x*farb_kanaele+0] *= 100.0;
        colour[colour_x*farb_kanaele+1] *= 100.0;
        colour[colour_x*farb_kanaele+2] *= 100.0;
        #ifdef DEBUG_
        cout << colour[colour_x*farb_kanaele+0] <<" "<< 
                colour[colour_x*farb_kanaele+1] <<" "<<
                colour[colour_x*farb_kanaele+2] <<" "; DBG
        #endif
}

static void
schreibeProfil (icUInt32Number intent)
{
  char* tag = schreibeNcl2Tag (pcsfarbe, geraetefarbe, farb_kanaele,"",name,"");

  DBG_S( (int*)tag <<" "<< tag_size )

  memcpy (&colour_profile[236], tag, tag_size);
  if(tag)    delete [] tag;

  // Intent setzen
  setze_renderingIntent ( colour_profile, intent );

  // Speichern des Farbprofiles
  
  if(colour_profile && tag_size)
  {
    std::ofstream f;
    f.clear();
    f.open ( b.c_str(),  std::ios::out );
    if(f.good())
    {
      f.write ( colour_profile, 236 + tag_size );
    }
    f.close();
  } else
    g_message (_("Profil not written."));

}

static void
schreibeZeiger (icUInt32Number intent)
{
  char* tag = schreibeNcl2Tag (pcsfarbe, geraetefarbe, farb_kanaele,"",name,"");

  DBG_S( (int*)tag <<" "<< tag_size )

  memcpy (&colour_profile[236], tag, tag_size);
  if(tag)    delete [] tag;

  // Intent setzen
  setze_renderingIntent ( colour_profile, intent );

  // Speichern des Farbprofiles
  
  if(colour_profile && tag_size)
  {
    std::fstream f;
    f.open ( b.c_str(),  std::ios::out | std::ios::out );
    if(f.good())
    {
      f.write ( colour_profile, 236 + tag_size );
    }
    f.close();
  } else
    g_message (_("Profil not written."));

}


static channel*
schlieseLayer (channel* layers)
{
  for ( int i=0 ; i < nlayers; i++)
  {
      gimp_drawable_detach(layers[i].drawable);

      // Zeilenspeicher
      if(layers[i].pixels) free(layers[i].pixels);
      if(layers[i].pixel) free(layers[i].pixel);
  }

  DBG_S( "layers: " << (int*) layers )

  if(layers) delete [] layers;
  layers = 0;
  return layers;
}

static channel*
holeLayer (int image_ID)
{
  
  int      *layer_IDs;
  gint32 nlayers;

  layer_IDs = gimp_image_get_layers (image_ID, &nlayers);
  DBG_S( "image_ID: " << image_ID << " nlayers: " << nlayers )

  channel *layers = 0;

  DBG_S( "layers: " << (int*)layers ) 

  layers = (channel*) new channel [nlayers];

  DBG_S( "layers: " << (int*)layers )
  

  // Durchlauf
  for ( int i=0 ; i < nlayers; ++i)
  {
      // die einzelnen Ebenen vorbereiten
      layers[i].ID = layer_IDs[i];
      layers[i].drawable = gimp_drawable_get (layers[i].ID);
      layers[i].drawable_type = gimp_drawable_type (layers[i].ID);
      layers[i].width  = layers[i].drawable->width;
      layers[i].height = layers[i].drawable->height;

      gimp_pixel_rgn_init (&(layers[i].srcRgn), layers[i].drawable,
                           0,0, layers[i].width,layers[i].height, FALSE,FALSE);
      //gimp_pixel_rgn_init (&(layers[i].dstRgn), layers[i].drawable,
        //                   0,0, layers[i].width,layers[i].height, TRUE,TRUE);

      layers[i].samplesperpixel = gimp_drawable_num_channels(layers[i].ID);
      layers[i].colours = 0; // muss ausserhalb bestimmt werden
      layers[i].bytes = gimp_drawable_bpp(layers[i].ID);

      gimp_drawable_offsets( layers[i].ID, &layers[i].offx, &layers[i].offy);
      #ifdef DEBUG
      cout << layers[i].ID    <<": "<<
              layers[i].offx  <<"," << layers[i].offy   <<
      " "  << layers[i].width <<"x" << layers[i].height << " "; DBG
      #endif
      // Zeilenspeicher
      layers[i].pixels = 0;
      layers[i].pixel = 0; //(char*) calloc (sizeof (double), 16);

  }
  if(layer_IDs) free (layer_IDs);

  return layers;
}

static void
minMax(int image_ID, int & min_x, int & min_y,
                      int & max_x, int & max_y )
{
  DBG_S( "Bild: " << image_ID )

  gint32  *layers;
  gint32   nlayers;

  layers = gimp_image_get_layers (image_ID, &nlayers);

  float max = -100000.0, max_color[4], //max_x, max_y,
        min = 100000.0, min_color[4]; //, min_x, min_y;

  for(int i = 0; i < 4; ++i) {
    max_color[i] = 0.0;
    min_color[i] = 0.0;
  }

  if(nlayers)
  {

    gint32 drawable_ID = gimp_image_get_active_layer (image_ID);

    DBG_S( "drawable: " << drawable_ID )

    int tile_height = gimp_tile_height ();


    GimpPixelRgn pixel_rgn;
    GimpDrawable *drawable = gimp_drawable_get (drawable_ID);
    GDrawableType drawable_type = gimp_drawable_type (drawable_ID);
    int channels = gimp_drawable_num_channels(drawable_ID);
    int colors = channels;
    if (channels == 2 || channels == 4) colors--;

    DBG_S( "colors: " << colors )

    int bpp = gimp_drawable_bpp(drawable_ID);
    unsigned char* data = (unsigned char*) calloc (sizeof (char), drawable->width * channels * bpp * tile_height);

    gimp_pixel_rgn_init (&pixel_rgn, drawable, 0, 0,
                         (gint32)drawable->width, (gint32)drawable->height,
                         FALSE, FALSE);
    DBG

    gimp_tile_cache_size(drawable->width/tile_height+1);

    int yend = 0;
DBG
    int ystart, y,x, c, colori, pos;
    for (ystart = 0; ystart < (int)drawable->height; ystart = yend + 1) {
      yend = ystart + tile_height - 1;
      yend = MIN (yend, (int)drawable->height);

      DBG_S( ": " <<  ystart <<" - " << yend )

      gimp_pixel_rgn_get_rect (&pixel_rgn, data, 0, ystart, drawable->width, yend - ystart);

      DBG_S( "ystart: " << ystart )

      for (y = ystart; y < yend; y++) {
        for (x = 0; x < (int)drawable->width; x++) {
          for (c = 0; c < colors; c++) {
            pos = ( (y-ystart) * drawable->width + x) * channels;
            switch (drawable_type)
            {
              case RGB_IMAGE:
              case RGBA_IMAGE:
              case GRAY_IMAGE:
              case GRAYA_IMAGE:
                     if (max < data[pos + c]
                         ) {
                       for (colori = 0; colori < colors; colori++)
                         max_color[colori] = data[pos + colori];
                       max = data[pos + c];
                       max_x = x;
                       max_y = y;
                     }
                     if (min > data[pos + c]
                        ) {
                       for (colori = 0; colori < colors; colori++)
                         min_color[colori] = data[pos + colori];
                       min = data[pos + c];
                       min_x = x;
                       min_y = y;
                     }
                   break;
              case INDEXED_IMAGE: break;
              case INDEXEDA_IMAGE: break;
              case U16_RGB_IMAGE:
              case U16_RGBA_IMAGE:
              case U16_GRAY_IMAGE:
              case U16_GRAYA_IMAGE:
              case BFP_RGB_IMAGE:
              case BFP_RGBA_IMAGE:
              case BFP_GRAY_IMAGE:
              case BFP_GRAYA_IMAGE:
                     if (max < ((guint16*)data)[pos + c]
                      ) {
                       for (colori = 0; colori < colors; colori++)
                         max_color[colori] = ((guint16*)data)[pos + colori];
                       max = ((guint16*)data)[pos + c];
                       max_x = x;
                       max_y = y;
                     }
                     if (min > ((guint16*)data)[pos + c]
                      ) {
                       for (colori = 0; colori < colors; colori++)
                         min_color[colori] = ((guint16*)data)[pos + colori];
                       min = ((guint16*)data)[pos + c];
                       min_x = x;
                       min_y = y;
                     }
                   break;
              case U16_INDEXED_IMAGE: break;
              case U16_INDEXEDA_IMAGE: break;
              case FLOAT_RGB_IMAGE:
              case FLOAT_RGBA_IMAGE:
              case FLOAT_GRAY_IMAGE:
              case FLOAT_GRAYA_IMAGE:
                     if (max < ((float*)data)[pos + c]) {
                       for (colori = 0; colori < colors; colori++)
                         max_color[colori] = ((float*)data)[pos + colori];
                       max = ((float*)data)[pos + c];
                       max_x = x;
                       max_y = y;
                     }
                     if (min > ((float*)data)[pos + c]) {
                       for (colori = 0; colori < colors; colori++)
                         min_color[colori] = ((float*)data)[pos + colori];
                       min = ((float*)data)[pos + c];
                       min_x = x;
                       min_y = y;
                     }
                   break;
              case FLOAT16_RGB_IMAGE: break;
              case FLOAT16_RGBA_IMAGE: break;
              case FLOAT16_GRAY_IMAGE: break;
              case FLOAT16_GRAYA_IMAGE: break;
              default: g_message (_("What kind of image is this?")); break;
            }
          }
        }
      }
    }

    free (data);
    gimp_drawable_flush(drawable);
    gimp_drawable_detach (drawable);

    DBG
  }
  if(layers) free(layers);

  #ifdef DEBUG
  DBG printf("max(%d,%d) = ",
              (int)max_x,(int)max_y);
  for (int c=0; c < 4; c++)
       printf("%f ", max_color[c]);
  printf ("\n");

  printf("\nmin(%d,%d) = ",
              (int)min_x,(int)min_y); 
  for (int c=0; c < 4; c++)
       printf("%f ", min_color[c]);
  printf ("\n");
  #endif

}


static void
getColour (channel* layers, int i,
           const unsigned char* data, double *colour,
           const int & x, const int & y )
{
  if (x > layers[i].width-1
   || x < 0
   || y > layers[i].height-1
   || y <0)
  {
    colour[0] = 0.0;
    colour[1] = 0.0;
    colour[2] = 0.0;
    if(layers[0].colours == 4)
    colour[3] = 0.0;
    return;
  }

          for (int c = 0; c < layers[i].colours; c++)
          {
            switch (layers[i].drawable_type)
            {
              case RGB_IMAGE:
              case RGBA_IMAGE:
              case GRAY_IMAGE:
              case GRAYA_IMAGE:
                   colour[c] = data[c] / 255.0;
                   break;
              case INDEXED_IMAGE: break;
              case INDEXEDA_IMAGE: break;
              case U16_RGB_IMAGE:
              case U16_RGBA_IMAGE:
              case U16_GRAY_IMAGE:
              case U16_GRAYA_IMAGE:
              case BFP_RGB_IMAGE:
              case BFP_RGBA_IMAGE:
              case BFP_GRAY_IMAGE:
              case BFP_GRAYA_IMAGE:
                   colour[c] = ((guint16*)data)[c] / 65535.0;
                   break;
              case U16_INDEXED_IMAGE: break;
              case U16_INDEXEDA_IMAGE: break;
              case FLOAT_RGB_IMAGE:
              case FLOAT_RGBA_IMAGE:
              case FLOAT_GRAY_IMAGE:
              case FLOAT_GRAYA_IMAGE:
                   colour[c] = ((float*)data)[c];
                   break;
              case FLOAT16_RGB_IMAGE:
              case FLOAT16_RGBA_IMAGE:
              case FLOAT16_GRAY_IMAGE:
              case FLOAT16_GRAYA_IMAGE:
                   ShortsFloat u;
                   colour[c] = (double) FLT( ((guint16*)data)[c],u);
                   break;
              default: std::cout << (_("What kind of image is this?\n")); break;
            }
          }
}


