/*
 * Br.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  Br.hpp
*/
#ifndef Br_hpp
#define Br_hpp


#include <vector>

#include "Br_Image.hpp"         // class Br_ImageVec
#include "Camera.hpp"           // Camera<..>
#include "Distributor.hpp"      // ValueDistributor<..>

// Do we want Br::Event debugging?...
//#define DBG_BR_EVENT

# ifdef DBG_BR_EVENT
   // Activate macro `ynsobs_dbg' from "ys_dbg.hpp"...
#   define YS_DEBUG                
#   include "YS/ys_dbg.hpp"     // macro `ynsobs_dbg'

#   define SPUR_BR_EVENT(args)     ynsobs_dbg args ;
# else
#   define SPUR_BR_EVENT(args)
# endif    


extern const char* br_eventnames[];
  // debug names for the BracketingCore::Event enums.


/**====================================================================
 *
 * BracketingCore  --  class 
 *
 * Collects the GUI-independent globals: dataVec, imgVec, camera; 
 *   distributoren; some numerical parameters. 
 *   Also wrapper functions for stuff, which affect relations between
 *  `imgVec' (the image base) and `camera' (the numerical tool), especially
 *   as via image-de|activation not all images of imgVec have to be used
 *   in camera.
 *
 * `refpic_' and `last_refpic_': 
 *   More than one GUI client present "follow-up curves", which are to 
 *   compute by `camera->z_average(refpic)'. To avoid multiple calls for
 *   the same reason, `last_refpic_' has been introduced. It is set to "-1"
 *   in init_camera() marking "no FollowUpCurves calculation (for any `refpic_')
 *   has been done till now" and set to the just actual `refpic_' value, if 
 *   such calculation has been done. This computation inclusive maintaning of
 *   `last_refpic_' is wrapped by `compute_FollowCurves()'. A GUI client, which
 *   has been asked for drawing follow-up curves has now to do: 
 *       Br.compute_FollowCurves();  // computes only if nessecary
 *       update_my_local_stuff();
 *    
 * Distribution of the enum type `Event' with ValueDistributor<T>.value(..)
 *   requires `value(T)' or `value(const T&)'; not works `value(T&)'.
 *
 *=====================================================================*/
class BracketingCore 
{
public:
    enum Event 
    {  NO_EVENT          = 0,
       CAMERA_INIT       = 1,   // camera-init has done
       CAMERA_OUTDATED   = 2,   // active-imgVec != used images in camera
       CAMERA_REDATED    = 3,   // camera got up-to-date again without init    
       CAMERA_DELETED    = 4,   // camera==0
       IMAGE_LOADED      = 5,   // image loaded, imgVec changed 
       IMAGES_CHANGED    = 6,   // imgVec changed (e.g. cleared)
                                //   (Can '5' subsumes '4'? We will see.)
       CCD_UPDATED       = 7,   // new CCD curves (response fncts) computed
       CCD_OUTDATED      = 8,   // any change happend that could give another
                                //   CCD curves than the existing
       CCD_DELETED       = 9,   // Subsumed by CAMERA_INIT & _DELETED?     
       HDR_UPDATED       = 10   // HDR image created or updated
    };                          
    

private:
    bool camera_uptodate_;
      // True, if the set of active images in `imgVec' is identical with
      //   those used in `camera'; false, if different.
      // Regime for setting: 
      //   Setting true by: init camera
      //   Setting false by: Loading of an image, de/activation of an image,
    
    int refpic_, last_refpic_;
      // For which reference picture the "follow-up curves" shall be computed?

public:
    Br_ImageVec                           imgVec;
    std::vector< Array2D < Rgb<uint8> > > dataVec_u8;
   
    Camera<uint8,double>*  camera8;
    
    // Distributor(en)...
    ValueDistributor<Event> distrib_event;
    ValueDistributor<int>   distrib_refpic;

    // Ctor & Dtor...
    BracketingCore();
    ~BracketingCore() { clear(); }
    
    // (later here the non-template CameraBaseClass)
    Camera<uint8,double>* camera() const  { return camera8; }
    bool camera_initialized()      const  { return (camera() != 0); }
    
    void init_camera();
    void clear_camera();     
    void clear_container();
    void clear()            { clear_camera(); clear_container(); }
    
    size_t size() const     { return imgVec.size(); }
    bool ccd_curves_ready() { return camera() && camera()->ccd_curves_ready(); } 
    
    bool used (int i) const;
      // true if image `imgVec[i]' is used in camera
    int used_as (int i) const;
      // returns camera-index of the image `imgVec[i]'
    int imgVec_index (int k) const;
      // returns imgVec-index to the camera index `k'
    
    bool camera_uptodate () const { return camera_uptodate_; }
    void camera_uptodate (bool b) { camera_uptodate_ = b;    }
    void activate (int pic);
    void deactivate (int pic);

    // Numeric parameters... ('public' until further notice)
    int     n_grid_points_;
    double  smoothing_;
    
    
    // Functions for `refpic_'-handling...
    void refpic (int pic); 
      // Sets refpic_=pic and broadcast the new value if changed.
    int  refpic () const          { return refpic_; }
      // Returns the current value.
    void set_refpic (int pic)     { refpic_ = pic; }
      // Sets `refpic_' without broadcasting
    
    // Update times in `camera' by those in `imgVec'...  
    void update_camera_times();  
      
    // Wrappers for computation calls...
    
    void compute_FollowCurves ();  
      // Computes follow-up curves for the current `refpic_', but only if
      //   it has not been done till now.
    
    void make_CCD_curve ();
      // Computes CCD curves completely (incl. FollowUpCurves)
    
    Array2D<Rgb<float> > calc_HDR ();
      // Computes pure HDR image from existing CCD curves
    Array2D<Rgb<float> > make_HDR ();
      // Computes HDR image completely incl. FollowUpCurves and CCD curves
    Array2D<Rgb<float> > complete_HDR ();
      // Computes pure HDR, if CCD curves already exist, else calc all
    
    void list() const 
      { 
        printf("Br::n_grid_points = %d\n", n_grid_points_);
        printf("Br::smoothing     = %f\n", smoothing_);
        printf("Br::refpic        = %d\n", refpic_);
        printf("Br::last_refpic   = %d\n", last_refpic_);
      }
      
    void cursor_wait();
    void cursor_default();  
};


// the singular global instance we name "Br"...
extern BracketingCore Br;      

#endif  // Br_hpp

// END OF FILE
