/*
 * br_Image.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file  br_Image.hpp
  
  Referenzzaehlende Image-Klasse.
  
  Den vollen Erfolg verhinderte einst noch das `char* name'-Element. Dieses
   entweder bei jeder Copy-Aktion physisch kopieren oder besser auch
   referenzzaehlend. Dazu dient jetzt ein tnt_i_refvec<char>-Element.
   Oder gleich std::string nehmen.
   
  Zu Image passt nur ein ImgVector, der Objekte haelt und nicht ledigl. Zeiger.
   In der Zeigervariante duerften auch hier nie automatische Objekte 
   eingereiht werden, denn einen Zeiger nehmen (kopieren) erhoeht nicht
   den Ref-Zaehler. So wuerden diese Objekte einmal automatisch zerstoert und
   dann ein zweites mal durch ImgVector-Dtor per delete. Nachteil freilich
   der Objekt-Variante, dass ein BrImage schon einige Bytes hat, aber 
   ImgVectoren werden nicht oft kopiert.
   
  ENTSCHEIDUNG: `Image' von Array1D *ableiten* oder (analog `name_') ein 
   Array1D `buffer' als *Element* halten? Ableiten hat Vorteil, dass die 
   Array1D-Funktionalitaet sofort zur Verfuegung steht (z.B. cout gibt lineare
   uchar-Daten aus, ref_count()). Element ist u.U. etwas uebersichtlicher. Zum
   Referenzzaehlen taugt beides, im zweiten Fall wird Objekt zwar per default
   Copy-Ctor kopiert, doch das Element `buffer' bringt dann seinen referenz-
   zaehlenden Copy-Ctor zum Einsatz. Entschieden fuers Ableiten.
*/
#ifndef br_Image_hpp
#define br_Image_hpp

#include <vector>
#include "TNT/tnt_array1d.hpp"
#include "br_types.hpp"         // ImageID
#include "br_enums.hpp"         // DataType, ..., ReportWhat
#include "br_macros.hpp"        // ASSERT(), RELAX, CTOR(), DTOR()



#ifdef BR_CHECK_IMG_TYPE

# define CHECK_IMG_TYPE_U8(img) \
                ASSERT(img.data_type() == DATA_U8, RELAX);\
                ASSERT(img.byte_depth() == 1, RELAX);\
                ASSERT(img.channels() == 1, RELAX);

# define CHECK_IMG_TYPE_RGB_U8(img) \
                ASSERT(img.data_type() == DATA_U8, RELAX);\
                ASSERT(img.byte_depth() == 1, RELAX);\
                ASSERT(img.channels() == 3, RELAX);\
                ASSERT(img.storing() == STORE_INTERLEAVE, RELAX);

# define CHECK_IMG_TYPE_U16(img) \
                ASSERT(img.data_type() == DATA_U16, RELAX);\
                ASSERT(img.byte_depth() == 2, RELAX);\
                ASSERT(img.channels() == 1, RELAX);

# define CHECK_IMG_TYPE_RGB_U16(img) \
                ASSERT(img.data_type() == DATA_U16, RELAX);\
                ASSERT(img.byte_depth() == 2, RELAX);\
                ASSERT(img.channels() == 3, RELAX);\
                ASSERT(img.storing() == STORE_INTERLEAVE, RELAX);

# define CHECK_IMG_TYPE_F32(img) \
                ASSERT(img.data_type() == DATA_F32, RELAX);\
                ASSERT(img.byte_depth() == 4, RELAX);\
                ASSERT(img.channels() == 1, RELAX);

# define CHECK_IMG_TYPE_RGB_F32(img) \
                ASSERT(img.data_type() == DATA_F32, RELAX);\
                ASSERT(img.byte_depth() == 4, RELAX);\
                ASSERT(img.channels() == 3, RELAX);\
                ASSERT(img.storing() == STORE_INTERLEAVE, RELAX);
#else

# define CHECK_IMG_TYPE_U8(img)
# define CHECK_IMG_TYPE_RGB_U8(img)
# define CHECK_IMG_TYPE_U16(img)
# define CHECK_IMG_TYPE_RGB_U16(img)
# define CHECK_IMG_TYPE_F32(img)
# define CHECK_IMG_TYPE_RGB_F32(img)

#endif


/**+*************************************************************************\n
  Macro to give a "not implemented" message
******************************************************************************/
#define NOT_IMPL_IMAGE_CASE(img)    br::not_impl_image_case(__FILE__,__LINE__,__func__,img);


namespace br {

/**===========================================================================

  @class  Image

  Base data of an (input) image; without exposure and statistics stuff. 

  Derived from TNT::Array1D<uchar>. Array1D.dim() gibt die Puffergroesse in
   Bytes. Alle hier hinzugefuegten Elemente werden bei einem `a=b' physisch
   kopiert.
 
  Note: Return-Wert von buffer_size() als `size_t' veranschlagt, auch wenn
   gegenwaertiges Array1D.dim() nur `int' zurueckgibt. Ist kuenftiger.

  Note: Default-Ctor hier gefordert, wenn zB. ein std::vector<Image>(n)
   angelegt werden soll, das braucht Default-Ctor seiner Objekte!

*============================================================================*/
class Image : public TNT::Array1D<unsigned char>
{
    static ImageID id_counter_;         // counting only in Init-Ctor(s)
    
    ImageID   id_;                      // starts with 1, "0" means unset or failure
    int       width_;                   // |
    int       height_;                  // |
    int       channels_;                // | these four define the buffer size
    int       byte_depth_;              // |
    int       bit_depth_;               // could be useful
    DataType  data_type_; 
    StoringScheme  storing_scheme_;     // INTERLEAVE, CHNNLWISE
    ImageType  image_type_;
    TNT::i_refvec<char> name_;          // ref counting string object
  
public:

    Image();
    Image (int W, int H, int channels, DataType, const char* name=0);
    Image (int W, int H, ImageType, const char* name=0);
    virtual ~Image();               // "virt" avoids warnings (Opportunist!)
    
    ImageID   imageID() const           {return id_;}
    int       width() const             {return width_;}    
    int       height() const            {return height_;}    
    int       dim1() const              {return height_;}  // hides Arr.dim1()
    int       dim2() const              {return width_;}
    int       channels() const          {return channels_;}
    int       byte_depth() const        {return byte_depth_;}
    int       bit_depth() const         {return bit_depth_;}
    DataType  data_type() const         {return data_type_;}
    StoringScheme storing() const       {return storing_scheme_;}
    ImageType image_type() const        {return image_type_;}
    size_t    buffer_size() const       {return dim();}    // from Array1D
    int       n_pixel() const           {return width_ * height_;}
    
    const unsigned char* buffer() const {return &(*this)[0];}
    unsigned char*       buffer()       {return &(*this)[0];}
    
    const char* name() const            {return name_.begin();}
    void        name (const char*);
    
    virtual void report(ReportWhat = REPORT_BASE) const;
    
private:
    
    //  Helpers for Constructors
    void _Image (int W, int H, int channels, DataType dtype, 
                 StoringScheme store, const char* name);
                 
    void eval_image_type();             

    //  *Static* helper for usage already in the `:-scope
    static int byte_depth (DataType);    


#ifdef BR_CHECK_IMG_TYPE
    //  Functions to create wrong data for testing...
public:
    void set_data_type (DataType d)           {data_type_ = d;}
    void set_storing (StoringScheme ss)       {storing_scheme_ = ss;}
    void set_channels (int n)                 {channels_ = n;}
#endif

};


/**===========================================================================

  @struct  ExposureData

  Collects photographic exposure data of an input image.

*============================================================================*/
struct ExposureData
{
    float   time;
    float   aperture;   // Blende, not yet used
    float   ISO;        // not yet used
    bool    active;
    
    ExposureData (float time=0.0);
    
    void report() const;
};


/**===========================================================================

  @struct  ImgStatisticsData

  Collects statistics data of an single image.

*============================================================================*/
struct ImgStatisticsData
{
    double  brightness;
    int     n_low;      // number of pixels at the lower bound    
    int     n_high;     // number of pixels at the upper bound
    double  r_wk_range; // ratio of pixels within working range
                        //  == 1.0 - (n_low + n_high) / (width*height)
    ImgStatisticsData() 
      : brightness(0.), n_low(0), n_high(0), r_wk_range(0.)  {}
    
    void report() const;
};


/**===========================================================================

  @class  BrImage  -  sub-class of Image
 
  Image data  +  exposure data  +  statistics data. Prefix "Br" for
   "Bracketing".
  
  OPEN: default value of exposure time = "0" or "-1.0"?

*============================================================================*/
class BrImage : public Image
{
public:
  
    ExposureData  exposure;
    ImgStatisticsData  statistics;
  
    BrImage() {}
    BrImage (int W, int H, int channels, DataType dtype, const char* name=0,
             float tm=0)
      : Image (W, H, channels, dtype, name), 
        exposure (tm)
      {}  
    BrImage (int W, int H, ImageType iType, const char* name=0, float tm=0)
      : Image (W, H, iType, name), 
        exposure (tm)
      {}  
    
    const BrImage & getRef() const      {return *this;}
    BrImage       & getRef()            {return *this;}
    
    bool   active() const               {return exposure.active;}
    void   active (bool val)            {exposure.active = val;}
    float  time() const                 {return exposure.time;}
    double brightness() const           {return statistics.brightness;}
    
    void   set_time (float tm)          {exposure.time = tm;}
    
    void   report(ReportWhat = REPORT_DEFAULT) const;   // virtual
};



/**===========================================================================

  @class  BrImgVector  -  Our input image container.
 
  Vector of (ref counting) BrImage objects (not pointers).

*============================================================================*/
class BrImgVector
{
    std::vector <BrImage>  vec_;
  
public:

    BrImgVector()                             {CTOR("")}
    ~BrImgVector(); 

    DataType    data_type() const;
    ImageType   image_type() const;
    int         size() const                  {return vec_.size();}  
    
    void add (BrImage &);
    void clear()                              {vec_.clear();}
  
    BrImage &       operator[] (int i)        {return vec_[i];}
    const BrImage & operator[] (int i) const  {return vec_[i];}
  
    BrImage &       at (int i)                {return vec_.at(i);}
    const BrImage & at (int i) const          {return vec_.at(i);}
    
    BrImage &       image (int i)             {return vec_[i];}
    const BrImage & image (int i) const       {return vec_[i];}
    
    void activate (int i)                     {vec_.at(i).active( true );}
    void deactivate (int i)                   {vec_.at(i).active( false );} 
    bool active (int i) const                 {return vec_.at(i).active();}
    int  size_active() const;  
    
    void set_TimesByStop (double stop);
    
    void report (ReportWhat = REPORT_DEFAULT) const;
};



/**+*************************************************************************\n
  If we want to emphasize that an `Image' should contain HDR data 
   (type==IMAGE_[RGB]_F32) we use the type identifier "ImageHDR". 
******************************************************************************/
typedef Image  ImageHDR;    



/**+*************************************************************************\n
  Helper for the macro NOT_IMPL_IMAGE_CASE(). Public since used by a macro.
******************************************************************************/
void not_impl_image_case (const char* file, int line, const char* func, Image);


}  // namespace "br"

#endif  // br_Image_hpp

// END OF FILE
