/*
 * br_PackBase.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file  br_PackBase.cpp  -  Base class of the Pack classes
   
  Contents:
   - class `PackBase'  
*/

#include <cmath>                  // log()
#include <cassert>                // assert()
#include <vector>

#include "br_PackBase.hpp"
#include "br_Image.hpp"           // BrImgVector (Ctor argument)
#include "br_enums_strings.hpp"   // DATA_TYPE_STR[] etc.
#include "TNT/tnt_array1d.hpp"
#include "br_macros_varia.hpp"    // SPUR()


namespace br {

/*================================================================
*
*   PackBase  -  class implementation
* 
*=================================================================*/

/**+*************************************************************************\n
  Ctor()

  @param imgVec: vector of `Image's (input images)

  - Alloc some vectors of dimension imgVec.size_active()
  - Take over the identic meta data of the activated images
  - Check consictency between these images
******************************************************************************/
PackBase::PackBase (BrImgVector & imgVec) 
  : 
    dim1_           (0),
    dim2_           (0),
    n_pixel_        (0),
    channels_       (0),
    byte_depth_     (0),
    bit_depth_      (0),
    buffer_size_    (0),
    data_type_      (DATA_NONE),
    storing_scheme_ (STORING_NONE),
    ids_            (imgVec.size_active()),  // allokiert
    times_          (ids_.size()),           // allokiert
    logtimes_       (ids_.size())            // allokiert
{
    CTOR("")
    //  Check (ungallant), besser werfen
    assert (imgVec.size_active() > 1);
    
    //  Take over the image datas...
    bool first = true;
    int k = 0;                          // counts active images
    for (int i=0; i < imgVec.size(); i++)
    {
      BrImage & img = imgVec.image(i);
      if (img.active()) 
      { 
        if (first) {
          //  Take over the identic meta data...
          dim1_           = img.dim1();
          dim2_           = img.dim2();
          channels_       = img.channels();
          byte_depth_     = img.byte_depth();
          bit_depth_      = img.bit_depth();
          buffer_size_    = img.buffer_size();
          data_type_      = img.data_type();
          storing_scheme_ = img.storing();
          n_pixel_        = size_t(dim1_) * dim2_;
        
          first = false;
        } 
        else {
          //  Check consistency with first image... (ungallant)
          assert (img.dim1() == dim1_);
          assert (img.dim2() == dim2_);
          assert (img.channels() == channels_);
          assert (img.byte_depth() == byte_depth_);
          assert (img.bit_depth() == bit_depth_);
          assert (img.data_type() == data_type_);
          assert (img.storing() == storing_scheme_);
          assert (k < (int)size());        // size_active() could be wrong
        }
      
        //  Take over the individual datas....
        ids_     [k] = img.imageID();
        times_   [k] = img.time();
        logtimes_[k] = log( times_[k] );
      
        SPUR(("use image [%d] \"%s\", time=%f\n", 
           i, img.name(), img.exposure.time))
      
        k++;
      }
    }   
}


/**+*************************************************************************\n
  report()
******************************************************************************/
void PackBase::report(ReportWhat what)  const
{
    std::cout << "=== Pack Report === [" << size() << "]:"
              << "\n  [ " << width() << " x " << height() << " ]";
    if (what & REPORT_BASE) {
      std::cout 
        << "\n  [Base Data...]"
        << "\n    channels   : " << channels()
        << "\n    DataType   : " << DATA_TYPE_STR[data_type()]
        << "\n    byte_depth : " << byte_depth()
        << "\n    bit_depth  : " << bit_depth()
        << "\n    buffer_size: " << buffer_size() 
        << "\n    storing scheme: " << STORING_SCHEME_STR[storing()];
    }
    if (what & REPORT_EXPOSE_INFO) {
      std::cout << "\n  [Times...] ";
    
      for (int i=0; i < size(); i++) {
        std::cout << "\n    [" << i << "]  " << time(i) 
                  << ",  " << logtime(i)
                  << "\t\t(ID=" << imageID(i) <<')';
      }
    }
    std::cout << '\n';            
}


}  // namespace "br"

// END OF FILE
