/*
 * ImageTable.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file ImageTabel.cpp

  The image table in the tab "Images" in the main window. 
  
  @todo Verhindern, dass Tabellenspalten bis auf Null zusammengeschoben
    werden koennen.
*/
#include <cstdio>                       // sprintf() 
//#include <cstring>

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/filename.H>                // fl_filename_name()
#include <FL/fl_draw.H>                 // fl_font(), fl_push_clip() etc.
#include "../FL_adds/Fl_Table.H"
#include "../FL_adds/fl_print_event.hpp" // Fltk debug utilities
#include "ImageTable.hpp"               // ImageTable
#include "Br2Hdr.hpp"                   // Br2Hdr
#include "../br_core/br_defs.hpp"       // BR_DEBUG_TABLE, BR_DEBUG_RECEIVER
#include "../br_core/br_macros.hpp"     // CTOR()

#ifdef BR_DEBUG_RECEIVER
#  include <iostream>
   using std::cout;
#endif   


using namespace br;

/*=================================================================
//
//              ImageTable  -  IMPLEMENTATION
//
//================================================================*/
 
/**
*   Constructor...
*   
*   @todo If launched via FLUID, the box decision code comes after the
*      Ctor and we could work here with a wrong frame value (-> unneeded
*      scrollbar). Safe should be a NO_BOX in FLUID.
*/
ImageTable::ImageTable(int X, int Y, int W, int H, const char *la) 
  : 
    Fl_Table (X,Y,W,H,la),
    m_Br2Hdr (Br2Hdr::Instance())     
{ 
    CTOR(la); 
    begin();
    
    selection_color(FL_YELLOW);
    //type(Fl_Table_Row::SELECT_NONE);
    callback((Fl_Callback*)cb_event_, this);
    when(FL_WHEN_RELEASE);  // handle table events on release
    //when(FL_WHEN_CHANGED|FL_WHEN_RELEASE);

#ifdef BR_TIME_INPUT    
    input_ = new Fl_Float_Input(W/2, H/2, 0, 0);
    input_ -> hide();
    input_ -> callback((Fl_Callback*)cb_input_, this);
    input_ -> when(FL_WHEN_ENTER_KEY_ALWAYS);
    input_ -> maximum_size(12);
#endif
      
    int frame     = Fl::box_dw(box()); // *after* box decision!
    int w_name    = 110;
    int w_size    = 100;
    int w_brightn = 80;
    int w_wrange  = 80;
    int w_time;
      
    // Fit the last column into the rest of the table ('25' is the width
    //  of the activation button), but not smaller than...
    w_time = W - (25 + w_name + w_size + w_brightn + w_wrange) - frame;
    w_time = (w_time > 75) ? w_time : 75;
      
    cols(5);
    col_header(1);          // enable col header
    col_header_height(25);
    col_width(0,w_name);    // filename
    col_width(1,w_size);    // image size
    col_width(2,w_brightn); // brightness
    col_width(3,w_wrange);  // working range
    col_width(4,w_time);    // time
    col_resize(4);          // enable col resizing

    rows(m_Br2Hdr.size());  // number of images
    row_header(1);          // enable row header
    row_header_width(25);   // width of activation button
    row_height_all(25);
    row_resize(4);          // enable row resizing
    
    end();    // Fluid auto-generates it if widget launched as 'open group'
}                   


/**
*   draw_cell() -  Handle drawing of all cells of the table
*
*   Wird implizit auch durch einige der im Ctor benutzten Fl_Table-Methoden
*    aufgerufen, dh. wenn Datenreferenz noch nicht initialisiert, anscheinend
*    dort aber nicht in datenreferenzierendem Context (context 64).
*
*   @todo Alignment at decimal points.
*/
void ImageTable::draw_cell(TableContext context, 
                           int R, int C, int X, int Y, int W, int H)
{
#ifdef BR_DEBUG_TABLE
    printf("draw_cell(): %s, R=%d, C=%d, X=%d, Y=%d, W=%d, H=%d\n",
      contextname(context),R,C,X,Y,W,H);
#endif      
    
    char s[128];
    static Fl_Color col_inactive = fl_color_average(FL_WHITE, FL_GRAY, 0.66f);
    int wcell;        // width of *visible* cell (not always == W)
    Fl_Align align;   // alignment inside the visible cell

    switch ( context )
    {
    case CONTEXT_STARTPAGE:
        fl_font(FL_HELVETICA, 14);   // Besser waere globale fontsize policy
        return;

    case CONTEXT_COL_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
          fl_color(FL_BLACK);
          wcell = wix + wiw - X;
          wcell = wcell < W ? wcell : W; // printable width never wider than W
          switch(C) 
          {
          case 0: fl_draw("Name",  X, Y, wcell, H, FL_ALIGN_CENTER); break;
          case 1: fl_draw("Pixel", X, Y, wcell, H, FL_ALIGN_CENTER); break;
          case 2: fl_draw("Brightness", X, Y, wcell, H, FL_ALIGN_CENTER); break;
          case 3: fl_draw("Wk-Range",   X, Y, wcell, H, FL_ALIGN_CENTER); break;
          case 4: fl_draw("Time",  X, Y, wcell, H, FL_ALIGN_CENTER); break;
          default: sprintf(s, "%d/%d", R, C); // something forgotten?
                   fl_draw(s,       X, Y, wcell, H, FL_ALIGN_CENTER);
          }
        }
        fl_pop_clip();
        return;

    case CONTEXT_ROW_HEADER:
        fl_push_clip(X, Y, W, H);
        {
          sprintf(s, "%d", R+1);  // begin with "1", not "0"
          //printf("row=%d: color=%u, GRAY=%u, active()=%d\n", 
          //      R, color(), FL_GRAY, m_Br2Hdr.isActive(R));
          if (m_Br2Hdr.isActive(R)) fl_draw_box(FL_DOWN_BOX, X,Y,W,H, FL_GREEN);
          else                      fl_draw_box(FL_UP_BOX,   X,Y,W,H, color());
          fl_color(FL_WHITE); 
          fl_draw(s, X,Y,W,H, FL_ALIGN_CENTER);
        }
        fl_pop_clip();
        return;
      
    case CONTEXT_CELL:
#ifdef BR_TIME_INPUT
        // don't overwrite input widget...
        if (R == row_edit_ && C == col_edit_ && input_->visible())
          return;
#endif    
        fl_push_clip(X, Y, W, H);
        {
          // BG COLOR
          //Fl_Color c = row_selected(R) ? selection_color() : FL_WHITE;
          Fl_Color c = FL_WHITE;
          if (!m_Br2Hdr.isActive(R)) c = col_inactive;
          draw_box(FL_THIN_UP_BOX, X,Y,W,H, c);

          // TEXT
          m_Br2Hdr.isActive(R) ? fl_color(FL_BLACK) : fl_color(fl_inactive(FL_BLACK));
          wcell = wix + wiw - X;
          wcell = wcell < W ? wcell : W; // printable width never wider than W
          switch (C)
          {
          case 0:
              fl_draw(fl_filename_name(m_Br2Hdr.name(R)), X+3, Y, W, H, FL_ALIGN_LEFT); 
              break;
          
          case 1:
              snprintf(s, 128, "%d x %d", m_Br2Hdr.width(R), m_Br2Hdr.height(R));
              if (wcell > fl_width(s)) align = FL_ALIGN_CENTER; 
              else                     align = FL_ALIGN_LEFT;
              fl_draw(s, X, Y, wcell, H, align);
              break;
          
          case 2:
              snprintf(s, 128, "%.4f ", m_Br2Hdr.brightness(R) / m_Br2Hdr.brightness(0));
              if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
              else                     align = FL_ALIGN_LEFT;
              fl_draw(s, X, Y, wcell, H, align);
              break;
          
          case 3:
              snprintf(s, 128, "%.3f %% ", m_Br2Hdr.relWkRange(R)*100.0);
              if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
              else                     align = FL_ALIGN_LEFT;
              fl_draw(s, X, Y, wcell, H, align);
              break;
          
          case 4:
              snprintf(s, 128, "%.6f ", m_Br2Hdr.time(R));
              if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
              else                     align = FL_ALIGN_LEFT;
              fl_draw(s, X, Y, wcell, H, align);
              break;
          
          default:
              sprintf(s, "%d/%d", R, C);    // something forgotten?
              fl_draw(s, X, Y, wcell, H, FL_ALIGN_CENTER);
          }

          // BORDER (example)
          //fl_color(FL_LIGHT2); 
          //fl_rect(X, Y, W, H);
        }
        fl_pop_clip();
        return;

#ifdef BR_TIME_INPUT         
    case CONTEXT_RC_RESIZE:
        // Input widget mitscrollen...
        if (!input_->visible()) 
          return;
        find_cell(CONTEXT_TABLE, row_edit_, col_edit_, X, Y, W, H);
        if (X==input_->x() && Y==input_->y() && 
            W==input_->w() && H==input_->h())
          return;
        input_->resize(X,Y,W,H);
        return;
#endif
        
    case CONTEXT_ENDPAGE:
        // Draw a box in the "headers corner"; X,Y are in STARTPAGE/ENDPAGE
        //  those of the data table *without* headers, x() and y() are the
        //  outer coords; what we need here are wix, wiy, the (protected) 
        //  inner coords of the widget without border (same as 
        //  X - row_header_width(), Y - col_header_height)!).
        draw_box(FL_EMBOSSED_BOX, wix, wiy, row_header_width(), col_header_height(), color());
        return;
        
    default:
        //printf("ImageTable::%s(): not processed context %d\n",__func__, context);
        return;
    }
}

/**
*   cb_event() -  callback; called whenever someone clicks on different parts
*                  of the table.
*/
void ImageTable::cb_event()
{
#ifdef BR_DEBUG_TABLE
    printf("ImageTable::cb_event(): %s, row=%d col=%d, %s, clicks=%d\n",
        contextname(callback_context()),
        (int)callback_row(),
        (int)callback_col(),
        fl_eventnames[Fl::event()],
        (int)Fl::event_clicks());
#endif
    
    TableContext context = callback_context();
    int R = callback_row();
    int C = callback_col();

    switch (Fl::event())
    {
    case FL_PUSH:
        switch (context)
        {
        case CONTEXT_ROW_HEADER:
            if (m_Br2Hdr.isActive(R)) m_Br2Hdr.deactivate(R);
            else                      m_Br2Hdr.activate(R);
            // Noetig waere eigentlich jetzt nur Neuzeichnen der Zeilenkoepfe,
            //  doch das Untige hat irgendeine Macke und ich weiss nicht warum.
            //  Deshalb alles neu und raus.
            redraw(); 
            return;
            // Table callback by default allready redraws *inner* table, i.e.
            //  nessecary is here only to add the redrawing of row header.
            int XX,YY,WW,HH;
            find_cell(CONTEXT_ROW_HEADER, R,C, XX, YY, WW, HH);
            draw_cell(CONTEXT_ROW_HEADER, R,C, XX, YY, WW, HH);
            return;

#ifdef BR_TIME_INPUT      
        case CONTEXT_CELL:    
            if (C == col_edit_) {   // time column!
              if (input_->visible()) { input_->do_callback(); }
              row_edit_ = R;
              int XX,YY,WW,HH;
              find_cell(CONTEXT_CELL, R,C, XX, YY, WW, HH);
              input_ -> resize(XX,YY,WW,HH);
              char s[30];
              snprintf(s, 30, "%f", m_Br2Hdr.time(R));
              input_ -> value(s);
              input_ -> show();
              input_ -> take_focus();    
            }
            return;
#endif          
        default:           
            return;
        }

    default: ;
    }
}


#ifdef BR_TIME_INPUT
/**
*   cb_input()  -  callback for ImageTable's Float_Input widget
*
*   Translate current input->value() into double and set the data object
*    ("global") variable.
*/
void ImageTable::cb_input ()
{
    printf("cb_input(): [%d] = \"%s\")...\n", row_edit_, input_->value());
    
    char*  endptr;
    double res = strtod (input_->value(), &endptr);

    //  Accept only correct values:
    if (*endptr) {                         
      printf("[%s]: wrong float format: \"%s\"\n", __func__, input_->value());
      //  Data object remains unchanged.
    }
    else if (res <= 0.0) {
      printf("[%s]: times <= 0 not allowed\n", __func__);
      //  Data object remains unchanged.
    } 
    else {                                 // ok, take over
      m_Br2Hdr.set_Time (row_edit_, res);    // Changes data object
    }
    
    input_-> hide();                         // hides input    
}
#endif   


/**
*   EventReceiver'c virtual callback routine:
*
*/
void ImageTable::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    cout << "ImageTable::"; EventReceiver::handle_Event(e);
#endif
    
    switch (e)
    {
    case Br2Hdr::IMAGE_LOADED:          // Update ImageTable!
    case Br2Hdr::IMAGES_CHANGED:  
        BR_EVENT_HANDLED(e);
        rows (m_Br2Hdr.size());         // new row number
        redraw();
        break;
    
    case Br2Hdr::TIMES_CHANGED:
        BR_EVENT_HANDLED(e);
        redraw();
        break;
    
    default: 
        BR_EVENT_NOT_HANDLED(e);
    }
}    



// END OF FILE
