#!/usr/bin/perl -w -T
# +=======================================================================+
# || CipUX::RPC::Server::Daemon                                          ||
# ||                                                                     ||
# || Command line interface for CipUX RPC Server.                        ||
# ||                                                                     ||
# || Copyright (C) 2007 - 2008 by Christian Kuelker                      ||
# ||                                                                     ||
# || License: GNU GPL - GNU General Public License - version 2           ||
# ||          or (at your opinion) any later version.                    ||
# ||                                                                     ||
# +=======================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source

package CipUX::RPC::Server::Daemon;

use warnings;
use strict;
use Carp;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars);
use Getopt::Long;
use Log::Log4perl qw(get_logger :levels);
use Pod::Usage;
use Readonly;
use base qw(CipUX::RPC::Server);

{    # BEGIN INSIDE-OUT CLASS

    use version; our $VERSION = qv('3.4.0.8');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safer

    # +======================================================================+
    # || CONSTANTS                                                          ||
    # +======================================================================+
    Readonly::Scalar my $SCRIPT       => 'CipUX::RPC::Server::Daemon';
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Array my @ACTION        => qw(cipux_rpcd cipux_rpc_list);
    Readonly::Scalar my $LINEWIDTH    => 78;

    # +======================================================================+
    # || INIT ARGS                                                          ||
    # +======================================================================+
    my %name_of : ATTR( init_arg => 'name');

   # +=======================================================================+
   # || GLOBALS                                                             ||
   # +=======================================================================+
    my $L = q{=} x $LINEWIDTH;
    $L .= "\n";
    my %opt    = ();
    my $script = $EMPTY_STRING;
    my %option = (
        'cipux_rpcd' => {
            'must' => [],
            'may'  => [
                qw(c cfg h ? help D debug t test v verbose version)],
            'not' => [qw(l list)],
        },
        'cipux_rpc_list' => {
            'must' => [],
            'may'  => [qw(c cfg h ? help D debug l list v verbose version)],
            'not'  => [qw(t test)],
        },
    );

    sub run {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # Constructor param from CipUX::RPC::Server::Daemon
        my $run_action = $name_of{ ident $self};

        # test right away if we have a valid action
        # is $run_action inside @action?
        if ( scalar( grep { $_ eq $run_action } @ACTION ) < 1 ) {
            $self->exc( { msg => 'unknown action', value => $run_action } );
        }

        # +==================================================================+
        # || ENVIRONMENT                                                    ||
        # +==================================================================+

        # %opt have to be empty!
        # If you have to test of existence, be aware to test it
        # with "exists" or "defined" of the SINGLE letter version.
        # Example:  exists($opt{D}) might be true if --debug or -D is set,
        #           but exists($opt{debug}) is always (!) false!

        Getopt::Long::Configure("bundling");

        GetOptions(
            \%opt,       "c|cfg=s",  "D|debug:s", "h|?|help",
            "l|list",    "p|pretty",  "t|task=s",
            "V|version", "verbose",
            )
            or pod2usage(
            -exitstatus => 2,
            -msg        => "$L Problems parsing command line!\n$L"
            );

        if ( exists $opt{D} and defined $opt{D} ) {
            Log::Log4perl::init_and_watch( '/etc/cipux/log4perl.conf', 60 );
        }

        my $logger = get_logger('__PACKAGE__');
        $logger->debug('BEGIN');
        $logger->debug( '    action: ', $run_action );

        # display help page
        if ( exists( $opt{help} ) or exists( $opt{h} ) ) {
            pod2usage( -exitstatus => 0, -verbose => 1 );
        }

        # display version an exit
        if ( exists( $opt{version} ) ) {
            print "$SCRIPT $VERSION\n";
            exit(0);
        }

        my $ret = $self->test_cli_option(
            {
                script   => $run_action,
                logic_hr => \%option,
                opt_hr   => \%opt,
                debug    => 0,
            }
        );

        # +==================================================================+
        # || MAIN                                                           ||
        # +==================================================================+

        my $action
            = exists $opt{action} ? $opt{action}
            : exists $opt{a}      ? $opt{a}
            :                       undef;

        my $cfg
            = exists $opt{cfg} ? $opt{cfg}
            : exists $opt{c}   ? $opt{c}
            :                    undef;

      # We DO need a configuration
      #my $msg =
      #  $L . 'EXCEPTION: The default configuration file can not be found. ';
      #$msg .= 'Please provide --cfg <FILE>' . "\n$L";
      #pod2usage( -msg => $msg, -exitstatus => 4, -verbose => 0 ) if not $cfg;

        my $pretty
            = exists $opt{pretty} ? 1
            : exists $opt{p}      ? 1
            :                       0;

        if ( defined $cfg ) { $logger->debug( '    cfg: ', $cfg ); }

        $action = $run_action;

        #$action =~ s{^.*_}{}gmx;

        $self->$run_action(
            {
                action => $action,
                pretty => $pretty,
                cfg    => $cfg,
            }
        );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_rpcd                                                           |
    # +----------------------------------------------------------------------+
    sub cipux_rpcd {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $action
            = exists $arg_r->{action}
            ? $self->l( $arg_r->{action} )
            : $self->perr('action');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '> action: ', $action );

        #$logger->debug( '    cfg: ', { filter => \&Dumper, value => $cfg } );

        my $run_action = $action;

        if ( scalar( grep { $_ eq $run_action } @ACTION ) < 1 ) {
            $self->exc( { msg => 'unknown action', value => $run_action } );
        }

        my $server = CipUX::RPC::Server->new();
        $logger->debug('call server start');

        #local $PROGRAM_NAME = "$SCRIPT [accepting connections]";
        my $state = $server->rpc_start();
        $logger->debug( 'server exited with state: ', $state );

        $logger->debug('call server stop');
        $server->rpc_stop;

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $state;

    }

    # +----------------------------------------------------------------------+
    # | cipux_rpc_list                                                       |
    # +----------------------------------------------------------------------+
    sub cipux_rpc_list {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;
        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '> pretty: ', $pretty );
        $logger->debug( '> cfg: ', { filter => \&Dumper, value => $cfg } );

        my $server = CipUX::RPC::Server->new();
        my @list = $server->rpc_list_functions( { scope => 'all' } );

        # mostly this stuff is for pretty print
        my $max_col = 0;
        my $width   = 0;

        if ($pretty) {
            foreach my $line (@list) {

                if ( $max_col < length $line ) {
                    $max_col = length $line;
                }
            }
            $width = 2 + $max_col;
            $self->out( q{+} . q{-} x $width . "+\n" );
            $self->out( sprintf '| %-' . $max_col . "s |\n",
                'xml rpc command' );
            $self->out( q{+} . q{=} x $width . "+\n" );
        } ## end if($pretty)

        foreach my $line (@list) {
            chomp $line;
            if ($pretty) {
                $self->out( sprintf '| %-' . $max_col . "s |\n", $line );
            }
            else {
                $self->out("$line\n");
            }
        } ## end foreach my $line (@$list_aref)
        if ($pretty) {
            $self->out( q{+} . q{-} x $width . "+\n" );
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        exit 0;

        #return;

    }

}    # END INSIDE-OUT CLASS

# Module implementation here

1;    # Magic true value required at end of module

__END__

=head1 NAME

CipUX::RPC::Server::Daemon - Command line interface for CipUX::RPC::Server


=head1 VERSION

This document describes CipUX::RPC::Server::Daemon version 3.4.0.8


=head1 SYNOPSIS

    use CipUX::RPC::Server::Daemon;
    my $daemon = CipUX::RPC::Server::Daemon->new( { name=>'cipux_rpcd' } );
    $daemon->run();

=head1 DESCRIPTION

Command line interface to CipUX::RPC::Server.


=head1 METHODS

=head2 cipux_rpcd

Start the XML-RPC server.

=head2 cipux_rpc_list

List CipUX XML-RPC functions.

=head2 run

Main routine.

=head1 DIAGNOSTICS

    Every single error and warning message that the module can
    generate (even the ones that will "never happen"), with a full
    explanation of each problem, one or more likely causes, and any
    suggested remedies.

=over

=item C<< Problems parsing command line! >>

If Getopt::Long has a non specified problem. See Getopt::Long for details. Most
probable reason is: you provide wrong command line switches or values.

=back

=head1 CONFIGURATION AND ENVIRONMENT

CipUX::RPC::Server::Daemon requires no configuration files or environment
variables.


=head1 DEPENDENCIES

Carp
CipUX::RPC::Server
Class::Std
Data::Dumper
English
Getopt::Long
Log::Log4perl
Pod::Usage
Readonly


=head1 INCOMPATIBILITIES

None reported.


=head1 BUGS AND LIMITATIONS

No bugs have been reported.

=head1 AUTHOR

Christian Kuelker  C<< <christian.kuelker@cipworx.org> >>


=head1 LICENCE AND COPYRIGHT

Copyright (C) 2007 - 2008 by Christian Kuelker

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.
