;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: objects.lisp,v 1.65 2003/03/27 19:45:31 jesse Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

(in-package :odcl)

;; Every class that supports get-property/store-property has an entry
;; in this hashtable of class object-meta.

(defvar *propertied-objects* (make-hash-table)
  "Objects that use the ODCL property interface.")

(defclass base-meta ()
  ((reader :initarg :reader
           :initform nil)
   (writer :initarg :writer
           :initform nil))
  (:documentation "Support behavior common to both objects and slots:
an object can have a generic read/write function, a slot can have a
specialized one."))

(defclass object-meta (base-meta)
  ((class                :initarg :class)
   (members              :initform nil)
   (member-of            :initform nil)
   (properties           :initform (make-hash-table))
   (indexes              :initform (make-hash-table)))
  (:documentation "An object participates in membership (aggregation)
relationships, and has properties and indexes associated with it."))

(defclass object-slot-meta  (base-meta)
  ((type          :initarg :type)
   (display-type  :initarg :type
                  :initform nil)
   (stored        :initarg :stored
                  :reader storedp
                  :initform nil
                  :documentation "Does this property have storage allocated for it.")
   (class-reader  :initarg :class-reader
		  :initform nil
		  :documentation "If class-reader is t, use 3 and 4
arity accessors (object property [value] type); otherwise use 1 and 2 arity (object [value])")
   (class-writer  :initarg :class-writer
		  :initform nil
		  :documentation "If class-writer is t, use 3 and 4
arity accessors (object property [value] type); otherwise use 1 and 2 arity (object [value])")
   (read-auth     :initarg :read-auth)
   (read-daemon   :initarg :read-daemon
                  :initform nil)
   (write-allowed :initarg :write-allowed
                  :initform t)
   (write-auth    :initarg :write-auth
                  :initform nil)
   (write-daemon  :initarg :write-daemon
                  :initform nil)
   (caption       :initarg :caption)
   (default       :initarg :default
     :initform nil)
   (auth-daemon   :initarg :auth-daemon
                  :initform nil)
   (log-daemon    :initarg :log-daemon
                  :initform 'log-property-access)
   (documentation :initarg :documentation
                  :initform nil))
  (:documentation "Specific to single properties of a class."))

;; Return the property information for a class of objects.  Does not
;; support inheritance.

(defmethod get-object-meta (class &key (if-does-not-exist :error) (top nil))
  (if-bind (meta (gethash class *propertied-objects*))
      (if (and top (slot-value meta 'member-of))
          (get-object-meta (slot-value meta 'member-of))
          meta)
      (case if-does-not-exist
        (:create
         (setf (gethash class *propertied-objects*)
               (make-instance 'object-meta :class class)))
        (:error
         (error "No properties are defined for objects of type ~s" class)))))

;; Return the record for a single property of a class.

(defun get-property-meta (class property &key (if-does-not-exist :error))
  (unless (keywordp property)
    (error "properties are named by keywords"))
  (let ((object-meta (get-object-meta class)))
    (with-slots (properties members)
      object-meta
      (or (gethash property properties)
          (case if-does-not-exist
            (:create
             (setf (gethash property properties)
                   (make-instance 'object-slot-meta)))
            (:error
             (error "Class ~s does not define property ~s" class property))
            (:ignore
             nil)
            (:find
             (dolist (member members)
               (when-bind (meta (get-property-meta member property :if-does-not-exist :ignore))
                 (return-from get-property-meta (values meta member))))
             (error "Class ~s does not define property ~s" class property)))))))

(defmacro def-propertied (object &key member-of log-daemon write-daemon write-auth reader writer properties)
  `(let ((object-meta (get-object-meta ',object :if-does-not-exist :create)))
    (setf (slot-value object-meta 'writer) ,writer
     (slot-value object-meta 'reader) ,reader)
    (macrolet ((pushkv (def k v)
                 `(progn
                   (push ,v ,def)
                   (push ,k ,def))))
      ;; attach properties to the primary storage class
      (dolist (property ',properties)
        (destructuring-bind (slotname &rest slotdef)
            property
          (setf slotname (ensure-keyword slotname))
          (pushkv slotdef :class-reader t)
	  (pushkv slotdef :class-writer t)
          (pushkv slotdef :stored t)
          (pushkv slotdef :writer ,writer)
          (pushkv slotdef :reader ,reader)
          (when (and ',write-auth (not (getf slotdef :write-auth)))
            (pushkv slotdef :write-auth ',write-auth))
          (when (and ',write-daemon (not (getf slotdef :write-daemon)))
            (pushkv slotdef :write-daemon ',write-daemon))
          (when (and ',log-daemon (not (getf slotdef :log-daemon)))
            (pushkv slotdef :log-daemon ',log-daemon))
          (unless (getf slotdef :caption)
            (error "No caption supplied for ~s of ~s" property ',object))
          ;; (cmsg "Slotdef: ~s" slotdef)
          (apply-slotdef ',object slotname slotdef)))
      ;; attach properties to the aggregating storage class
      (when ',member-of
        (let ((member-meta (get-object-meta ',member-of :if-does-not-exist :create)))
          (pushnew ',object (slot-value member-meta 'members))
          (setf (slot-value object-meta 'member-of) ',member-of))))))

(defun apply-slotdef (object-name pname pdef)
  (let ((meta (get-property-meta object-name pname :if-does-not-exist :create)))
    (do ((pdef pdef (cddr pdef)))
        ((null pdef))
      (setf (slot-value meta (intern (symbol-name (Car pdef)) :odcl))
            (cadr pdef)))
    meta))

(defun get-class-index (class name &key (if-does-not-exist :error))
  "Look up and return the index for a given class of objects. Index is
named by a keyword."
  (unless (keywordp name)
    (error "Indexes are named by keywords."))
  (let ((object-meta (get-object-meta class)))
    (when-bind (index (car (gethash name (slot-value object-meta 'indexes))))
      (return-from get-class-index index))
    (dolist (member (slot-value object-meta 'members))
      (when-bind (index (get-class-index member name :if-does-not-exist :ignore))
        (return-from get-class-index index)))
    (when (eql if-does-not-exist :error)
      (error "Class ~s does not define index with name ~s." class name))))

(defun get-object-indexes (class)
  "Return all indexes for a class of objects and its aggregated storage."
  (when-bind (object-meta (get-object-meta class))
    (with-slots (indexes members)
      object-meta
      (let ((indexes (hashkeys indexes)))
        (dolist (member members)
          (setf indexes (append indexes (get-object-indexes member))))
        indexes))))

(defun delete-propertied-object (object)
  "Delete an object, all of its member instances, and indexes."
  (let ((props (get-object-meta (type-of object))))
    (dolist (member-class (slot-value props 'members))
      (when-bind (member-instance (get-aggregate-object object member-class
                                                        :if-does-not-exist :ignore))
        (delete-propertied-object member-instance)))
    (maphash (lambda (k v)
               (declare (ignore k))
               (index-clear (car v) (oid object)))
             (slot-value props 'indexes)))
  (ec-delete object *default-editing-context*))

(defgeneric generate-copy (object oid))

(defun duplicate-propertied-object (object)
  (let* ((props (get-object-meta (type-of object)))
         (new-oid (generate-new-oid (get-context-store *default-editing-context* object)
                                    (type-of object)))
         (new-object (generate-copy object new-oid)))
    (mapc (lambda (v)
            (when-bind (member-object (instance-by-key (list v (oid object))))
              (generate-copy member-object new-oid)))
          (slot-value props 'members))
    new-object))

(defmethod print-object ((self object-meta) stream)
  (print-unreadable-object
   (self stream :type t)
   (with-slots (class properties indexes)
     self
     (format stream "~s p/i ~d/~d"
             class
             (hash-table-count properties)
             (hash-table-count indexes)))))

(defgeneric oid (instance)
  (:documentation "Return an integer instance identifier for this instance, unique for all instances of class.")
  (:method (instance)
           (error "No OID method defined for ~A" instance)))

(defun clear-class-properties (class)
  (remhash class *propertied-objects*))

;; Indexes

(defmacro define-index (target-name index-name index-fn index-args &rest index-plist)
  `(let* ((props (get-object-meta ',target-name :if-does-not-exist :error))
          (olddef (gethash (ensure-keyword ',index-name) (slot-value props 'odcl::indexes)))
          (newargs (list ',index-name ',index-fn ',index-args ',index-plist))
          (new-index (when (and (car olddef)
                                (equal (cadr olddef) newargs))
                       (car olddef)))
          (rec (list new-index newargs)))
    (setf (gethash ',index-name (slot-value props 'odcl::indexes)) rec)))

(defgeneric index-type (index)
  )

;; ------------------------------------------------------------
;; Display Attributes
;;

;; defproperties, as opposed to properties defined by def-propertied,
;; do not propogate up to the parents of aggregate objects.

(defmacro defproperties (class props)
  ` (macrolet ((pushkv (def k v)
                 `(progn
                   (push ,v ,def)
                   (push ,k ,def))))
      ;; need to ensure the object-meta instance exists
      (get-object-meta ',class :if-does-not-exist :create)
      (dolist (prop ',props)
        (destructuring-bind (slotname &rest slotdef)
            prop
          (setf slotname (ensure-keyword slotname))
	  (when (getf slotdef :writer)
	    (unless (getf slotdef :class-writer)
	      (pushkv slotdef :class-writer nil)))
	  (when (getf slotdef :reader)
	    (unless (getf slotdef :class-reader)
	      (pushkv slotdef :class-reader nil)))
          (unless (getf slotdef :caption)
            (error "No caption supplied for ~s of ~s" prop ',class))
          (let ((meta (get-property-meta ',class slotname :if-does-not-exist :create)))
            (apply #'reinitialize-instance meta slotdef))))))

(defun property-metadata (class property-name)
  "Returns the metadata for the specified property as a list
containing: reader, writer, type and caption."
  (let ((phash (get-object-meta class)))
    (when-bind (pobj (gethash property-name (slot-value phash 'properties)))
      (with-slots (reader writer type display-type caption)
        pobj
        (list reader writer (or display-type type) caption)))))

(defun get-property-names-for-class (class)
  "Returns the names of all properties of the specified class."
  (with-slots (properties members)
    (get-object-meta class)
    (append (hashkeys properties)
            (mapcan #'get-property-names-for-class members))))

(defun properties-for-class (class &key test &aux results)
  "Returns the metadata for all properties of the specified class, as
list of (name reader writer added-type caption)"
  (with-slots (properties members) (get-object-meta class)
    (maphash (lambda (pname pobj)
	       (unless (and test
			    (not (funcall test pobj)))
		 (with-slots (reader writer type display-type caption)
		     pobj
		   (push (list pname
			       reader
			       writer
			       (or display-type type) caption) results))))
	     properties)
   (append results (mapcan (lambda (x) (properties-for-class x :test test)) members))))

(defun object-properties (self &key test &aux results)
  "Return all known properties for SELF, in format (<name> <value>
<added type> <edit allowed> <caption>)"
  (labels ((get-nonnull-properties (storage-class)
	   (with-slots (properties members)
	       (get-object-meta storage-class)
	     (maphash (lambda (pname pobj)
			(with-slots (reader writer type display-type caption)
			    pobj
			  (unless (and test
				       (not (funcall test pobj)))
			    (when reader
			      (push (list pname
					  (get-property self pname)
					  (or display-type type)
					  writer
					  caption)
				    results)))))
		      properties)
	     (mapcan #'get-nonnull-properties members))))
    (get-nonnull-properties (type-of self)))
  results)

(defmethod get-property ((self standard-object) (property list))
  (case (length property)
    (0
     (error "nil is not a valid property"))
    (1
     (get-property self (car property)))
    (2
     (when-bind (foo (get-property self (car property)))
       (get-property foo (second property))))
    (t
     (when-bind (foo (get-property self (car property)))
       (get-property foo (cdr property))))))

(defmethod get-property ((object standard-object) (property symbol))
  (multiple-value-bind (property-meta forward)
      (get-property-meta (type-of object) property
                         :if-does-not-exist :find)
    (with-slots (reader read-daemon type display-type default class-reader caption)
      property-meta
      (when forward
        (setf object (get-aggregate-object object forward)))
      (let ((this-reader reader))
        (unless this-reader
          (setf this-reader
                (slot-value (get-object-meta (type-of object)) 'reader))
          (setf class-reader t))
        (let ((value (if class-reader
                         (funcall this-reader object property type)
                         (funcall this-reader object))))
          (if value
              (progn
                (when read-daemon
                  (setf value (funcall read-daemon object property value)))
                value)
              (if default
                  (setf value (eval default))))
          (values value (or display-type type) caption))))))

(defun get-aggregated-objects (object &aux result)
  (with-ec-config (:reference)
    (dolist (member-class (slot-value (get-object-meta (type-of object)) 'members))
      (when-bind (aggregrated (instance-by-key (list member-class (oid object))))
        (push aggregrated result)))
    result))

(defmethod authorize-property ((object standard-object) direction authlist)
  (declare (ignore direction authlist))
  t)

(defun get-aggregate-object (obj new-class &key (if-does-not-exist :create))
  (let ((oid (oid obj)))
    (or (instance-by-key (list new-class oid))
        (ecase if-does-not-exist
          (:create
           (make-instance new-class :oid oid))
          (:ignore
           nil)))))

(defmethod store-property ((object standard-object) property value &aux storage-object)
  (multiple-value-bind (property-meta forward)
      (get-property-meta (type-of object) property
                         :if-does-not-exist :find)
    (with-slots (write-auth write-daemon log-daemon writer type class-writer)
      property-meta
      (let ((this-writer writer))
        (unless this-writer
          (setf this-writer
                (slot-value (get-object-meta (type-of object)) 'writer))
          (setf class-writer t))
        (unless this-writer
          (error "No writer for ~s / ~s" object property))
        (setf storage-object
              (if forward
                  (get-aggregate-object object forward)
                  object))
        (when (and write-auth (get-ec-config :authorize))
          (authorize-property object :write write-auth))
        (when (and log-daemon (get-ec-config :log))
          (funcall log-daemon object property :write value))
        (when (and write-daemon (get-ec-config :authorize))
          (funcall write-daemon object property value type))
        (if class-writer
            (funcall this-writer storage-object property value type)
            (funcall this-writer storage-object value))))))

(defvar *show-default-logging* nil)

(defun log-property-access (object property direction &optional value)
  (when *show-default-logging*
    (cmsg "Log: ~s/~s == ~s => ~s" object property direction value)))

(defun validate-properties ()
  (labels ((validate-class (class properties)
             (declare (ignore properties))
             (cmsg "Validate ~s" class)
             (mapc #'validate-class-property (properties-for-class class)))
           (validate-class-property (property-definition)
             (let ((barfed nil))
               (flet ((barf (&rest barf-args)
                        (unless barfed
                          (cmsg "  + ~s" (car property-definition))
                          (setf barfed t))
                        (apply #'cmsg barf-args)))
                 (destructuring-bind (name reader writer added-type caption)
                     property-definition
                   (unless (typep name 'keyword)
                     (barf "    ** Name is not a keyword"))
                   (typecase reader
                     (symbol
                      (when reader
                        (unless (fboundp reader)
                          (barf "    ** Reader function is not bound ~s" reader))))
                     (function
                      t)
                     (t
                      (barf "    ** Reader name is not a symbol or function ~s" reader)))
                   (typecase writer
                     (symbol
                      (when writer
                        (unless (fboundp writer)
                          (barf "    ** Writer function is not bound ~s" writer))))
                     (function t)
                     (t
                      (barf "    ** Writer name is not a symbol: ~s" writer)))
                   (unless (typep added-type 'symbol)
                     (barf "    ** Added-type is not a symbol"))
                   (unless (typep caption 'string)
                     (barf "    ** Caption is not a string")))))))
    (maphash #'validate-class *propertied-objects*)))

(defun describe-properties ()
  (cmsg "~d entities:" (hash-table-count *propertied-objects*))
  (cmsg "")
  (maphash (lambda (class-name class-data)
             (with-slots (properties indexes member-of members)
               class-data
               (cmsg "===========================")
               (cmsg "Entity ~s" class-name)
               (when member-of
                 (cmsg "Member of: ~s" member-of))
               (when members
                 (cmsg "Members: ~s" members))
               (cmsg "")
               (cmsg "Properties:")
               (maphash (lambda (class-name class-data)
                          (declare (ignore class-data))
                          (cmsg "~s" class-name))
                        properties)
               (cmsg "")
               (let (indexes)
                 (maphash (lambda (class-name class-data)
                            (declare (ignore class-data))
                            (push class-name indexes))
                          (slot-value class-data 'indexes))
                 (when indexes
                   (format t ";; Indexes: ~{~s~^ ~}~%" indexes)))
               (cmsg "")))
           *propertied-objects*))
