;;; -*- Mode: LISP; Syntax: ANSI-Common-Lisp; Package: CL-USER; -*-
;;;
;;; Filename:    ssl-ffi-lw.lisp
;;; Authors:     Jochen Schmidt <jsc@dataheaven.de>
;;;              Wade Humeniuk <humeniuw@cadvision.com>
;;;              Kevin Rosenberg <kevin@rosenberg.net>
;;; Description: Definitions of the needed foreign-functions
;;;              and the byte-vector handling needed for
;;;              simple buffering.

(in-package :ssl-internal)

#+lispworks
(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "comm"))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Register Modules ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

;;; This ssl.so has to be created with
;;; ld -shared -o ssl.so -L/usr/lib -lssl -lcrypto

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defparameter *ssl-library*
    (find-foreign-library
     "ssl"
     `("/lib/" "/usr/lib/" "/usr/local/lib/" "/usr/lib/cl-ssl/" "/sw/lib/cl-ssl/")))
  
  (unless *ssl-library*
    (error "Can't find ssl shared library"))
  )

;; OpenSSL dlls compiled by mingw32
#+win32(fli:register-module :libssl32 :real-name "libssl32")
#+win32(fli:register-module :libeay32 :real-name "libeay32")

(eval-when (:compile-toplevel :load-toplevel :execute)
  (load-foreign-library *ssl-library* :module :ssl
			:supporting-libraries '("c"))
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Connection and Error handling  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-foreign-type ssl-method '(* :int))
(def-foreign-type ssl-ctx '(* :int))
(def-foreign-type ssl-ptr '(* :int))
(def-foreign-type x509-stack '(* :int))

(def-function ("SSL_load_error_strings" ssl-load-error-strings)
    ()
  :module :ssl
  :returning :void)

(def-function ("SSL_library_init" ssl-library-init)
    ()
  :module :ssl
  :returning :int)

(def-function ("SSLv23_method" ssl-v23-method)
    ()
  :module :ssl
  :returning ssl-method)

(def-function ("SSL_set_connect_state" ssl-set-connect-state)
    ((ssl ssl-ptr))
  :module :ssl
  :returning :void)
(def-function ("SSL_set_accept_state" ssl-set-accept-state)
    ((ssl ssl-ptr))
  :module :ssl
  :returning :void)

(def-function ("SSL_CTX_new" ssl-ctx-new)
    ((method ssl-method))
  :module :ssl
  :returning ssl-ctx)

(def-function ("SSL_new" ssl-new)
    ((ctx ssl-ctx))
  :module :ssl
  :returning ssl-ptr)

(def-function ("SSL_set_fd" ssl-set-fd) 
  ((ssl ssl-ptr)
   (fd :int))
  :module :ssl
  :returning :int)

(def-function ("SSL_get_error" ssl-get-error) 
    ((ssl ssl-ptr)
     (ret :int))
  :module :ssl
  :returning :int)

(def-function ("SSL_connect" ssl-connect)
    ((ssl ssl-ptr))
  :module :ssl
  :returning :int)

(def-function ("SSL_accept" ssl-accept)
    ((ssl ssl-ptr))
  :module :ssl
  :returning :int)

(def-function ("SSL_write" ssl-write) 
    ((ssl ssl-ptr)
     (buf (* :unsigned-byte))
     (num :int))
  :module :ssl
  :returning :int)

(def-function ("SSL_read" ssl-read)
    ((ssl ssl-ptr)
     (buf (* :unsigned-byte))
     (num :int))
  :module :ssl
  :returning :int)

(def-function ("SSL_pending" ssl-pending)
    ((ssl ssl-ptr))
  :module :ssl
  :returning :int)

(def-function ("SSL_shutdown" ssl-shutdown) 
  ((ssl ssl-ptr))
  :module :ssl
  :returning :void)

(def-function ("SSL_free" ssl-free) 
    ((ssl ssl-ptr))
  :module :ssl
  :returning :void)

(def-function ("SSL_CTX_free" ssl-ctx-free)
    ((ctx ssl-ctx))
  :module :ssl
  :returning :void)

(def-function ("RAND_seed" rand-seed)
  ((buf (* :unsigned-byte))
   (num :int))
  :module :ssl
  :returning :void)

(def-function ("ERR_get_error" err-get-error)
    ()
  :module :ssl
  :returning :unsigned-long)

(def-function ("ERR_error_string" c-err-error-string) 
    ((e :unsigned-long)
     (buf :cstring))
  :module :ssl
  :returning :cstring)

(defun err-error-string (e)
  (with-cstring (buf nil)
    (convert-from-cstring (c-err-error-string e buf))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Certificate and Privatekey handling ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defconstant +ssl-filetype-pem+ 1)
(defconstant +ssl-filetype-asn1+ 2)
(Defconstant +ssl-filetype-default+ 3)

;; Set the set of available ciphers
(def-function ("SSL_set_cipher_list" c-ssl-set-cipher-list) 
    ((ssl ssl-ptr)
     (str :cstring))
  :module :ssl
  :returning :int)

(defun ssl-set-cipher-list (ssl str)
  (with-cstring (cs str)
    (c-ssl-set-cipher-list ssl cs)))

;; Load the RSA privatekey from the specified filename into the connection-handle
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(def-function ("SSL_use_RSAPrivateKey_file" c-ssl-use-rsa-privatekey-file) 
  ((ssl ssl-ptr)
   (str :cstring)
   (type :int))
  :module :ssl
  :returning :int)

(defun ssl-use-rsa-privatekey-file (ssl str type)
  (with-cstring (cs str)
    (c-ssl-use-rsa-privatekey-file ssl cs type)))

;; SSL-CTX-USE-RSA-PRIVATEKEY-FILE
;; Load the RSA privatekey from the specified filename into the CTX
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(def-function ("SSL_CTX_use_RSAPrivateKey_file" c-ssl-ctx-use-rsa-privatekey-file) 
  ((ctx ssl-ctx)
   (str :cstring)
   (type :int))
  :module :ssl
  :returning :int)

(defun ssl-ctx-use-rsa-privatekey-file (ctx str type)
  (with-cstring (cs str)
    (c-ssl-ctx-use-rsa-privatekey-file ctx cs type)))

;; SSL-USE-CERTIFICATE-FILE
;; Load the certificate from the specified filename into the connection-handle
;; type is either +ssl-filetype-pem+ or +ssl-filetype-asn1+
(def-function ("SSL_use_certificate_file" c-ssl-use-certificate-file) 
  ((ssl ssl-ptr)
   (str :cstring)
   (type :int))
  :module :ssl
  :returning :int)

(defun ssl-use-certificate-file (ssl str type)
  (with-cstring (cs str)
    (c-ssl-use-certificate-file ssl cs type)))

;; SSL-CTX-LOAD-VERIFY-LOCATIONS
;; Set the location where further CA certificates can be found to the specified
;; directory and/or file
(def-function ("SSL_CTX_load_verify_locations" c-ssl-ctx-load-verify-locations) 
  ((ctx ssl-ctx)
   (CAfile :cstring)
   (CApath :cstring))
  :module :ssl
  :returning :int)

(defun ssl-ctx-load-verify-locations (ctx cafile capath)
  (with-cstrings ((cs-file cafile)
		  (cs-path capath))
    (c-ssl-ctx-load-verify-locations ctx cs-file cs-path)))

;; SSL-CTX-SET-CLIENT-CA-LIST
;; Set the CTX' list of CAs that are acceptable from the client
(def-function ("SSL_CTX_set_client_CA_list" ssl-ctx-set-client-ca-list) 
    ((ctx ssl-ctx)
     (list x509-stack))
  :module :ssl
  :returning :void)

;; SSL-SET-CLIENT-CA-LIST
;; Set the connection-handle's list of CAs that are acceptable from the client
(def-function ("SSL_set_client_CA_list" ssl-set-client-ca-list)
  ((ssl ssl-ptr)
   (list x509-stack))
  :module :ssl
  :returning :void)

;; SSL-LOAD-CLIENT-CA-FILE
;; Load and return a list of CAs from the specified file
(def-function ("SSL_load_client_CA_file" c-ssl-load-client-ca-file)
  ((file :cstring))
  :module :ssl
  :returning x509-stack)

;; SSL-LOAD-CLIENT-CA-FILE
;; Load and return a list of CAs from the specified file
(defun ssl-load-client-ca-file (file)
  (with-cstring (cs file)
    (c-ssl-load-client-ca-file cs)))

;;;;;;;;;;;;;;;;;;;;
;;; Byte Vectors ;;;
;;;;;;;;;;;;;;;;;;;;


(defun make-byte-vector (size)
  (allocate-foreign-object :unsigned-byte size))

(defun free-byte-vector (vector)
  (free-foreign-object vector))

(defmacro bvref (buffer i)
  `(deref-array ,buffer '(:array :unsigned-byte) ,i))
