#!/usr/bin/env python

########################################################################
#  Chris's Lame File Browser 4  (Gadfly database control functions)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################

import os, string, stat
import gadfly, mimetype


###############################################################
# This function checks for the databases existance and opens the database
###############################################################	
def initialize(database_name):
	'''This creates a database using gadfly with the name
	you pass it.'''

	global connection
	global cursor


	# Check for existance of the database folder
	if not os.path.exists(os.environ['HOME'] + "/.claw/gadfly/"):
		os.makedirs(os.environ['HOME'] + "/.claw/gadfly/")
	
	# Check for existance of the database and open the database
	if not os.path.exists(os.environ['HOME'] + "/.claw/gadfly/" + database_name + ".gfd"):
		connection = gadfly.gadfly()
		connection.startup(database_name, os.environ['HOME'] + "/.claw/gadfly/")
		cursor = connection.cursor() 	
	else:
		connection = gadfly.gadfly(database_name, os.environ['HOME'] + "/.claw/gadfly")	
		cursor = connection.cursor()	


###############################################################
# This function checks if the database table exists already
###############################################################				
def __exists__(location):
	'''Returns 0 if the table name exists in the database.
	Returns 1 if the table does not exist in the database.'''

	table_name = __getTableName__(location)

	cursor.execute("SELECT * FROM __table_names__")
	index = cursor.fetchall()
	results = [table for table in index if string.upper(table_name) in table]
	if  results != []:
		return True
	else:
		return False


###############################################################
# This creates a new table for a directory
###############################################################		
def __create__(table_name):
	'''Creates a table with the name you pass'''

	cursor.execute("CREATE TABLE "  + table_name + " (filename VARCHAR, mimetype VARCHAR, stat VARCHAR)")

		
###############################################################
# This function creates entries for the database (name, mimetype, size)
###############################################################		
def __populate__(table_name, directory):
	'''Populates a table with the file names, mimetypes and sizes
	of the files in the folder you pass to it.'''

	# Get the mimetypes types for the locations
	mimes = mimetype.getTypes(directory)
	
	# Get the latest modification time for the file
	files = os.listdir(directory)
	locations = [os.path.abspath(file) for file in files]
	stats = __getStats__(locations)
	
	# Add the entires to the table
	for location in mimes:
		info_list = (location, mimes[location], stats[location])				
		cursor.execute("INSERT INTO " + string.upper(table_name) + "(filename, mimetype, stat) VALUES (?, ?, ?)", info_list)

	connection.commit()


###############################################################
# This function gets the md5 sums for a list of files and returns a dict
###############################################################	
def __getStats__(locations):
		
	# Weed out any dead links (Thanks to Alberto)
	locations_that_exist = [os.path.abspath(location) for location in locations if os.path.exists(os.path.abspath(location))]

	# Get the last modification times and make a dict
	stats = dict(zip(locations_that_exist, map(lambda location: os.stat(location)[stat.ST_MTIME], locations_that_exist)))

	# Handle dead links
	dead_links = [location for location in locations if not os.path.exists(location)]		
	if dead_links != []:
		for dead_link in dead_links:
			stats[dead_link] = None

	return stats
	
	
###############################################################
# This function reads the information stored in a table and returns it
###############################################################		
def read(directory):
	'''Returns a location indexed dictionary containing the mimetypes
	as listed in the table. i.e. ('/home/me/my_file.py': ['text','x-python'])'''

	# Make sure to get the absolute path (to avoid symlink problems that have
	# plagued the claw for a while)
	directory = os.path.realpath(directory)
	files = os.listdir(directory)

	if directory != "/" and not directory.endswith("/"):
		locations = [directory + "/" + file for file in files]
	else:
		locations = [directory + file for file in files]		

	# First up, check to see if any files are in the directory
	if locations == []:
		return {}

	# Check to see if a table exists for this directory (create a new one if it doesn't)
	table_name = __getTableName__(directory)
	if not __exists__(table_name):
		__create__(table_name)
		__populate__(table_name, directory)
		
	# Get the filenames and mimetypes
	cursor.execute("SELECT filename, mimetype FROM " + string.upper(table_name))
	types = dict(cursor.fetchall())
		
	# Compare the file information to the existing files
	deleted = [filename for filename in types if filename not in locations]
	added = [newfile for newfile in locations if newfile not in types]
		
	# This is supposed to speed things up
	DELETE =  "DELETE FROM " + string.upper(table_name) + " WHERE filename=?"
	INSERT = "INSERT INTO " + string.upper(table_name) + "(filename, mimetype, stat) VALUES (?, ?, ?)"
	SELECT = "SELECT filename, stat FROM " + string.upper(table_name)

	# Remove any deleted files from the database
	if deleted != []:
		for filename in deleted:
			del types[filename]
			cursor.execute(DELETE, (filename,))

	# Add any new files to the database
	if added != []:
		for filename in added:
			file_size, permissions, owner = information(filename)
			mimes = mimetype.getTypes(None, filename)
			stats = __getStats__([filename])
			info_list = (filename, mimes[filename], stats[filename])
			cursor.execute(INSERT, info_list)
			connection.commit()
				
			# Add the file to the mimetype list (this is for icons as well as opening files)				
			types[filename] = mimes[filename]


	# Get the  os.stat() results from the database and do a comparison
	cursor.execute(SELECT)
	stats =  dict(cursor.fetchall()) 
	new_stats = __getStats__(types)
	changed = [location for location in stats if stats[location] != new_stats[location] and not os.path.isdir(location)]

	# Set the new information if needed
	if changed != []:
		for location in changed:
			cursor.execute(DELETE, (location,))
				
			# If the file still exists, then readd it after deleting it
			if os.path.exists(location):
				file_size, permissions, owner = information(location)
				mimes = mimetype.getTypes(None, location)
				stats = __getStats__([location])
				info_list = (filename, mimes[location], stats[location])
				cursor.execute(INSERT, info_list)
				connection.commit()
					
				# Then correct the mime type
				types[location] = mimes[location]

	return types


########################################################################
# This function allows a user to add entries on the fly
#######################################################################
def add_entry(filename):
	'''Use this to add an entry to the database on the fly.
	Simply pass the full path to the file (/home/me/file.txt)
	and it will update the database.'''
	
	# Get the table name
	table_name = __getTableName__(os.path.dirname(filename))
			
	# Get the file information from the mimetypes module
	mimes = mimetype.get_mimetypes([filename])		
	stats = __getStats__([filename])
	info_list = (filename, mimes[filename], stats[filename])	
	cursor.execute("INSERT INTO " + string.upper(table_name) + "(filename, mimetype, stat) VALUES (?, ?, ?)", info_list)
	connection.commit()
	

########################################################################
# This function returns BASH friendly text
########################################################################		
def __getTableName__(location):
	
	# Here are the bad characters
	badChars = [" ","\'","\"","\\","&","(",")", "/", "-",".","[","]","\'", "*","!",":",";","#","+"]
		
	# Here are the replacement characters
	goodChars=["_","_","_","_","_","_","_","___","_","_","_","_","_","_","_","_","_","_","_"]
		
	# Replace the bad with the good
	for x in range(len(badChars)):
		location = string.replace(location, badChars[x], goodChars[x])		
	
	# Return the "cleaned", gadfly friendly text
	return location
	
	
########################################################################
# Get the information for a passed file
########################################################################	
def information(location):
		
	# If this is a non-existant file or dead link, then let's return some bugus values
	if not os.path.exists(location):		
		return "0 bytes", 'Dead Link', "Unknown"	
	
	# Get a list of users on the system
	users = __getUsers__()

	permissions=""

	# Get the file information
	try:
		fileInfo = os.lstat(location)

		# Get the UID and then use the username dictionary to get the user name
		UID = fileInfo[stat.ST_UID]
		owner = users[str(UID)]

		# Check to make sure the user is valid
		if owner == None or owner == "":
			owner = "unknown"

		# Get the file size
		if fileInfo[stat.ST_SIZE] > 1024:
			file_size = str(fileInfo[stat.ST_SIZE]/1024) + " Kbs"
		else:
			file_size = str(fileInfo[stat.ST_SIZE]) + " bytes"

		# Get the permissions in octal form (UNIX numeric [777,755,530] form)
		mode = oct(fileInfo[stat.ST_MODE])

		# This dictionary contains the permission types
		permissionTypes = {7:'rwx', 6:'rw-', 5:'r-x', 4:'r--', 3:'-wx', 2:'-w-', 1:'--x', 0:'---'}
	
		# Determine the permissions
		for number in mode[-3:]:
			permissions += permissionTypes[int(number)]

	except:
		# If we can't get the file information from stat, we'll get it the old fashioned way
		owner = "Unknown"
		file_size = os.path.getsize(location)
			
		# Get the file size
		if file_size > 1024:
			file_size = str(file_size/1024) + " Kbs"
		else:
			file_size = str(file_size) + " bytes"	

		# This is the old way of getting permissions
		if os.access(location, os.R_OK):
			permissions += "r"
		else:
			permissions += "-"
		if os.access(location, os.W_OK):
			permissions += "w"
		else:
			permissions += "-"
		if os.access(location, os.X_OK):
			permissions += "x"
		else:
			permissions += "-"

	# Return whatever we get
	return file_size, permissions, owner
	
	
#################################################
# Get the users on this computer from /etc/passwd
#################################################
def __getUsers__():

	infile = file('/etc/passwd', 'r').readlines()

	users = {}
	for line in infile:
		try:
			line = string.split(line,':')
			users[line[2]] = line[0]
		except:
			print "ERROR: Line ", line
	return users
