package Language::INTERCAL::GenericIO;

# Write/read data

# This file is part of CLC-INTERCAL

# Copyright (c) 2007 Claudio Calvelli, all rights reserved.

# CLC-INTERCAL is copyrighted software. However, permission to use, modify,
# and distribute it is granted provided that the conditions set out in the
# licence agreement are met. See files README and COPYING in the distribution.

use strict;
use vars qw($PERVERSION);
$PERVERSION = "CLC-INTERCAL INTERCAL/GenericIO.pm 1.-94.-3";

my ($PERVNUM) = $PERVERSION =~ /\s(\S+)$/;

use Carp;
use Cwd;
use File::Spec;
use IO::Handle;
use IO::File;

use Language::INTERCAL::Exporter '1.-94.-4';
use Language::INTERCAL::Charset '1.-94.-4', qw(toascii fromascii);
use Language::INTERCAL::Splats '1.-94.-3', qw(:SP);
use Language::INTERCAL::Server '1.-94.-3';

use vars qw(@EXPORT @EXPORT_OK @EXPORT_TAGS
	    $stdread $stdwrite $stdsplat $devnull);

@EXPORT = ();
@EXPORT_OK = qw($stdread $stdwrite $stdsplat $devnull);
@EXPORT_TAGS = (files => [qw($stdread $stdwrite $stdsplat $devnull)]);

$stdread = new Language::INTERCAL::GenericIO('FILE', 'r', '-');
$stdwrite = new Language::INTERCAL::GenericIO('FILE', 'w', '-');
$stdsplat = new Language::INTERCAL::GenericIO('FILE', 'r', '-2');
$devnull = new Language::INTERCAL::GenericIO('TEE', 'r', []);

sub new {
    @_ == 4 || @_ == 5 or croak
	"Usage: new Language::INTERCAL::GenericIO(TYPE, MODE, DATA [, SERVER])";
    my ($class, $type, $mode, $data, $server) = @_;
    if ($mode =~ /^\d+$/) {
	$mode = chr($mode & 0xff) . ($mode & 0x100 ? '+' : '');
    }
    $mode =~ /^[rwau]\+?$/ or faint(SP_IOMODE, $mode);
    $type = uc($type);
    my $filemode = $mode;
    $filemode =~ tr/rw/wr/;
    $filemode = 'r+' if $filemode =~ /u/;
    my %object = (
	type => $type,
	mode => $mode,
	data => $data,
	buffer => '',
	read_convert => sub { shift },
	read_charset => 'ASCII',
	write_convert => sub { shift },
	write_unconvert => sub { shift },
	write_charset => 'ASCII',
	text_newline => "\n",
	exported => 0,
    );
    if ($type eq 'FILE' || $type eq 'UFILE') {
	my $fh;
	if (ref $data && UNIVERSAL::isa($data, 'GLOB')) {
	    $fh = $data;
	} elsif ($data eq '-' || $data eq '-1') {
	    $fh = $mode =~ /r/ ? \*STDOUT : \*STDIN;
	} elsif ($data eq '-2') {
	    $fh = \*STDERR;
	} else {
	    # need absolute paths for use with checkpoint/restart; if
	    # File::Spec is too old and can't do it, tough
	    if (File::Spec->can('rel2abs')) {
		$data = File::Spec->rel2abs($data);
	    }
	    $fh = new IO::File($data, $filemode) or die "$data: $!\n";
	    # $fh->autoflush(1);
	    $object{close_code} = sub { close $fh };
	}
	faint(SP_IOERR, $data, $!) if ! defined $fh;
	if ($type eq 'FILE') {
	    $object{read_code} = sub { print $fh $_[0] };
	    $object{write_code} = sub {
		my ($size) = @_;
		my $b = '';
		read $fh, $b, $size;
		$b;
	    };
	    $object{write_text_code} = sub {
		my ($newline) = @_;
		local $/ = $newline;
		my $data = <$fh>;
		defined $data ? $data : '';
	    };
	    $object{tell_code} = sub { tell $fh };
	    $object{seek_code} = sub { seek $fh, $_[0], $_[1] };
	} else {
	    $object{read_code} = sub {
		my ($line) = @_;
		syswrite $fh, $line, length($line);
	    };
	    $object{write_code} = sub {
		my ($size) = @_;
		my $b = '';
		sysread $fh, $b, $size;
		$b;
	    };
	    $object{tell_code} = sub { sysseek $fh, 0, 1 };
	    $object{seek_code} = sub { sysseek $fh, $_[0], $_[1] };
	}
    } elsif ($type eq 'REMOTE') {
	$data =~ s/:(\w+)$//
	    or croak "DATA must be host:port when TYPE is $type";
	my $port = $1;
	my $host = $data;
	defined $server
	    or croak "SERVER must be provided when TYPE is $type";
	my $id = $server->tcp_socket($data, $port);
	my $line = $server->write_in($id, 1);
	defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	$line =~ /^2/ or faint(SP_INTERNET, $host, $line);
	$object{read_code} = sub {
	    my ($data) = @_;
	    my $len = length $data;
	    $server->read_out($id, "READ $len");
	    my $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^3/ or faint(SP_INTERNET, $host, $line);
	    $server->read_binary($id, $data);
	    $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^2/ or faint(SP_INTERNET, $host, $line);
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    $server->read_out($id, "WRITE $size");
	    my $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^2\d+\s+(\d+)/ or faint(SP_INTERNET, $host, $line);
	    my $len = $1;
	    my $data = $server->write_binary($id, $len);
	    defined $data or faint(SP_INTERNET, $host, 'Lost connection');
	    length($data) == $len or faint(SP_INTERNET, $host, "Invalid data");
	    $data;
	};
	$object{write_text_code} = sub {
	    my ($newline) = @_;
	    $newline =~ s/(\W)/sprintf("!%03d", ord($1))/ge;
	    $server->read_out($id, "WRITE TEXT /$newline/");
	    my $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^2\d+\s+(\d+)/ or faint(SP_INTERNET, $host, $line);
	    my $len = $1;
	    my $data = $server->write_binary($id, $len);
	    defined $data or faint(SP_INTERNET, $host, 'Lost connection');
	    length($data) == $len or faint(SP_INTERNET, $host, "Invalid data");
	    $data;
	};
	$object{tell_code} = sub {
	    $server->read_out($id, 'TELL');
	    my $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^2\d+\s+(\d+)/ or faint(SP_INTERNET, $host, $line);
	    return $1;
	};
	$object{seek_code} = sub {
	    my ($pos, $rel) = @_;
	    if ($rel == SEEK_SET) {
		$rel = 'SET';
	    } elsif ($rel == SEEK_CUR) {
		$rel = 'CUR';
	    } elsif ($rel == SEEK_END) {
		$rel = 'END';
	    } else {
		faint(SP_SEEKERR, "Invalid file position $rel");
	    }
	    $server->read_out($id, "SEEK $pos $rel");
	    my $line = $server->write_in($id, 1);
	    defined $line or faint(SP_INTERNET, $host, 'Lost connection');
	    $line =~ /^2/ or faint(SP_INTERNET, $host, $line);
	};
	$object{close_code} = sub {
	    eval { $server->tcp_socket_close($id); }
	};
    } elsif ($type eq 'TCP' || $type eq 'UTCP') {
	$data =~ s/:(\w+)$//
	    or croak "DATA must be host:port when TYPE is $type";
	my $port = $1;
	my $host = $data;
	defined $server
	    or croak "SERVER must be provided when TYPE is $type";
	my $id = $server->tcp_socket($data, $port);
	my $progress = $type eq 'TCP';
	$object{read_code} = sub {
	    my ($data) = @_;
	    $server->read_binary($id, $data);
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    $server->write_binary($id, $size, $progress);
	};
	$object{close_code} = sub {
	    eval { $server->tcp_socket_close($id); }
	};
	# can't seek a TCP socket
    } elsif ($type eq 'TEE') {
	$mode =~ /[ar]/ or croak "MODE must be \"read\" when TYPE is TEE";
	ref $data && 'ARRAY' eq ref $data or croak "DATA must be a array ref";
	$object{read_code} = sub {
	    my ($line) = @_;
	    for (@$data) { $_->read_binary($line) }
	};
	# always return end-of-file on write
	$object{write_code} = sub {
	    return '';
	};
	# object is not seekable
    } elsif ($type eq 'ARRAY') {
	ref $data && 'ARRAY' eq ref $data or croak "DATA must be a array ref";
	$object{read_code} = sub {
	    my ($line) = @_;
	    push @$data, $line;
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    return '' unless @$data;
	    my $line = shift @$data;
	    while (@$data && length($line) < $size) {
		$line .= shift @$data;
	    }
	    if (length($line) > $size) {
		unshift @$data, substr($line, $size);
		$line = substr($line, 0, $size);
	    }
	    $line;
	};
	# object is (currently) not seekable
    } elsif ($type eq 'STRING') {
	ref $data && 'SCALAR' eq ref $data or croak "DATA must be a scalar ref";
	my $filepos = $mode =~ /w/ ? 0 : length $$data;
	$object{read_code} = sub {
	    my ($line) = @_;
	    if ($filepos > length $$data) {
		$$data .= ' ' x ($filepos - length $$data);
	    }
	    substr($$data, $filepos, 0) = $line;
	    $filepos += length $line;
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    substr($$data, $filepos, $size, '');
	};
	$object{tell_code} = sub { $filepos };
	$object{seek_code} = sub {
	    my ($pos, $rel) = @_;
	    if ($rel == SEEK_SET) {
		$pos < 0 and die "Invalid file position\n";
		$filepos = $pos;
	    } elsif ($rel == SEEK_CUR) {
		$filepos + $pos < 0 and die "Invalid file position\n";
		$filepos += $pos;
	    } elsif ($rel == SEEK_END) {
		length($$data) + $pos < 0 and die "Invalid file position\n";
		$filepos = length($$data) + $pos;
	    } else {
		die "Invalid file position\n";
	    }
	};
    } elsif ($type eq 'OBJECT') {
	ref $data or croak "DATA must be a reference";
	UNIVERSAL::isa($data, 'UNIVERSAL')
	    or croak "DATA must be an object";
	$object{read_code} = sub { $data->read(@_); }
	     if $data->can('read');
	$object{write_code} = sub { $data->write(@_); }
	     if $data->can('write');
	# object is (currently) not seekable
    } elsif ($type eq 'COUNT') {
	$mode =~ /[ar]/ or croak "MODE must be \"read\" when TYPE is COUNT";
	ref $data && 'SCALAR' eq ref $data or croak "DATA must be a scalar ref";
	$object{read_code} = sub { $$data += length($_[0]) };
	# object is (currently) not seekable
    } else {
	# TODO (1.-90) - $type eq 'LECTURE'
	croak "Invalid type \"$type\"";
    }
    bless \%object, $class;
}

sub DESTROY {
    my ($fh) = @_;
    &{$fh->{close_code}} if exists $fh->{close_code};
}

sub can_tell {
    @_ == 1 or croak "Usage: IO->can_tell";
    my ($fh) = @_;
    return exists $fh->{tell_code};
}

sub tell {
    @_ == 1 or croak "Usage: IO->tell";
    my ($fh) = @_;
    exists $fh->{tell_code} or faint(SP_SEEKERR, "Not seekable");
    &{$fh->{tell_code}};
}

sub can_seek {
    @_ == 1 or croak "Usage: IO->can_seek";
    my ($fh) = @_;
    return exists $fh->{seek_code};
}

sub reset {
    @_ == 1 or croak "Usage: IO->reset";
    my ($fh) = @_;
    exists $fh->{seek_code} or faint(SP_SEEKERR, "Not seekable");
    &{$fh->{seek_code}}(0, SEEK_SET);
    $fh->{buffer} = '';
    $fh;
}

sub seek {
    @_ == 2 || @_ == 3
	or croak "Usage: IO->seek(POS [, RELATIVE_TO])";
    my ($fh, $pos, $rel) = @_;
    exists $fh->{seek_code} or faint(SP_SEEKERR, "Not seekable");
    $rel = SEEK_SET if ! defined $rel;
    &{$fh->{seek_code}}($pos, $rel);
    $fh->{buffer} = '';
    $fh;
}

sub data {
    @_ == 1 or croak "Usage: IO->data";
    my ($fh) = @_;
    return $fh->{data};
}

sub can_read {
    @_ == 1 or croak "Usage: IO->can_read";
    my ($fh) = @_;
    return exists $fh->{read_code};
}

sub read_binary {
    @_ == 2 or croak "Usage: IO->read_binary(DATA)";
    my ($fh, $string) = @_;
    faint(SP_MODEERR, "Not readable") if ! exists $fh->{read_code};
    &{$fh->{read_code}}($string);
    $fh;
}

sub read_text {
    @_ == 2 or croak "Usage: IO->read_text(DATA)";
    my ($fh, $string) = @_;
    faint(SP_MODEERR, "Not readable") if ! exists $fh->{read_code};
    faint(SP_MODEERR, "Not set up for text reading")
	if ! exists $fh->{read_convert};
    $string = &{$fh->{read_convert}}($string);
    &{$fh->{read_code}}($string);
    $fh;
}

sub read_charset {
    @_ == 1 || @_ == 2 or croak "Usage: IO->read_charset [(CHARSET)]";
    my $fh = shift;
    my $oc = $fh->{read_charset};
    if (@_) {
	my $charset = shift;
	$fh->{read_charset} = $charset;
	$fh->{read_convert} = fromascii($charset);
    }
    $oc;
}

sub mode {
    @_ == 1 or croak "Usage: IO->mode";
    my ($fh) = @_;
    $fh->{mode};
}

sub export {
    @_ == 2 or croak "Usage: IO->export(SERVER)";
    my ($fh, $server) = @_;
    $fh->{exported} and return $fh->{exported};
    my $port = $server->tcp_listen(\&_open, \&_line, \&_close, $fh);
    $fh->{exported} = $port;
    $port;
}

sub _open {
    my ($id, $sockhost, $peerhost, $close, $fh) = @_;
    $fh->{importers}{$id} = [0, 0, 0, ''];
    return "202 $sockhost ($PERVNUM)";
}

sub _line {
    my ($server, $id, $close, $line, $fh) = @_;
    exists $fh->{importers}{$id}
	or return "580 Internal error in server";
    my $filepos = $fh->{importers}{$id};
    if ($line =~ /^\s*TELL/i) {
	exists $fh->{tell_code} or return "581 Not seekable";
	return "280 $filepos is the current file position";
    }
    if ($line =~ /^\s*SEEK\s+(-?\d+)\s+(SET|CUR|END)/i) {
	exists $fh->{seek_code} or return "581 Not seekable";
	if ($2 eq 'SET') {
	    $1 < 0 and return "582 Invalid file position";
	    $filepos = $1;
	} elsif ($2 eq 'CUR') {
	    $filepos += $1;
	    $filepos < 0 and return "582 Invalid file position";
	} else {
	    my $delta = $1;
	    my $curpos;
	    $@ = '';
	    eval {
		my $oldpos = $fh->tell;
		$fh->seek(0, SEEK_END);
		$curpos = $fh->tell;
		$oldpos = $fh->seek($oldpos, SEEK_SET);
	    };
	    $@ and return "583 Cannot use SEEK_END on this filehandle";
	    $filepos = $curpos + $delta;
	    $filepos < 0 and return "582 Invalid file position";
	}
	$fh->{importers}{$id} = $filepos;
	return "281 $filepos is the new file position";
    }
    if ($line =~ /^\s*WRITE\s+(\d+)/i) {
	my $size = $1;
	exists $fh->{seek_code} and $fh->{seek_code}->($filepos, SEEK_SET);
	$@ = '';
	my $data = eval { $fh->write_binary($size) };
	if ($@) {
	    $@ =~ s/\n+/ /g;
	    return "584 $@";
	}
	eval {
	    exists $fh->{tell_code}
		and $fh->{importers}{$id} = &{$fh->{tell_code}}();
	};
	my $len = length $data;
	$server->read_out($id, "282 $len");
	$server->read_binary($id, $data);
	return ();
    }
    if ($line =~ /^\s*WRITE\s+TEXT\s+\/(\S*)\//i) {
	my $newline = $1;
	$newline =~ s/!(\d{3})/chr($1)/ge;
	$@ = '';
	my $data = eval {
	    exists $fh->{seek_code}
		and $fh->{seek_code}->($filepos, SEEK_SET);
	    $fh->write_text($newline);
	};
	if ($@) {
	    $@ =~ s/\n+/ /g;
	    return "584 $@";
	}
	eval {
	    exists $fh->{tell_code}
		and $fh->{importers}{$id} = &{$fh->{tell_code}}();
	};
	my $len = length $data;
	$server->read_out($id, "282 $len");
	$server->read_binary($id, $data);
	return ();
    }
    if ($line =~ /^\s*READ\s+(\d+)/i) {
	my $len = $1;
	my $code = sub {
	    my $data = shift;
	    defined $data && length($data) == $len
		or return "585 Data size mismatch";
	    $@ = '';
	    eval {
		exists $fh->{seek_code}
		    and $fh->{seek_code}->($filepos, SEEK_SET);
		$fh->read_binary($data);
	    };
	    if ($@) {
		$@ =~ s/\n+/ /g;
		return "586 $@";
	    }
	    eval {
		exists $fh->{tell_code}
		    and $fh->{importers}{$id} = &{$fh->{tell_code}}();
	    };
	    return "283 OK";
	};
	$server->alternate_callback($id, $len, $code);
	return "383 OK, send the data";
    }
    if ($line =~ /^\s*THANKS/i) {
	$$close = 1;
	return "284 You are welcome";
    }
    return "589 Command not understood";
}

sub _close {
    my ($id, $fh) = @_;
    delete $fh->{importers}{$id};
}

sub describe {
    @_ == 1 or croak "Usage: IO->describe";
    my ($fh) = @_;
    my $type = $fh->{type};
    my $mode = $fh->{mode};
    my $data = $fh->{data};
    if ($type eq 'FILE' || $type eq 'UFILE') {
	return "FILE($mode, $data)";
    } elsif ($type eq 'TEE') {
	return "TEE(" . join(',', map { describe($_) } @$data) . ")";
    } elsif ($type eq 'ARRAY' || $type eq 'STRING' || $type eq 'OBJECT') {
	return $type;
    } elsif ($type eq 'COUNT') {
	return "COUNT($$data)";
    } elsif ($type eq 'REMOTE') {
	return "REMOTE($data)";
    } elsif ($type eq 'TCP' || $type eq 'UTCP') {
	return "$type($data)";
    }
    # TODO (1.-90) - $type eq 'LECTURE'
    $fh;
}

sub can_write {
    @_ == 1 or croak "Usage: IO->can_write";
    my ($fh) = @_;
    return exists $fh->{write_code};
}

sub write_binary {
    @_ == 2 or croak "Usage: IO->write_binary(SIZE)";
    my ($fh, $size) = @_;
    faint(SP_MODEERR, "Not writable") if ! exists $fh->{write_code};
confess "size is undef" if ! defined $size;
    if (length($fh->{buffer}) >= $size) {
	return substr($fh->{buffer}, 0, $size, '');
    }
    my $data = '';
    if ($fh->{buffer} ne '') {
	$data = $fh->{buffer};
	$fh->{buffer} = '';
    }
    my $add =  &{$fh->{write_code}}($size - length($data));
    defined $add ? ($data . $add) : $data;
}

sub write_text {
    @_ == 1 or @_ == 2 or croak "Usage: IO->write_text [(NEWLINE)]";
    my ($fh, $newline) = @_;
    faint(SP_MODEERR, "Not writable") if ! exists $fh->{write_code};
    if (defined $newline) {
	if ($newline ne '') {
	    eval { $newline = $fh->{write_unconvert}->($newline) };
	    $newline = "\n" if $@;
	}
    } else {
	$newline = $fh->{text_newline};
    }
    if ($newline eq '') {
	my $line = $fh->{buffer};
	$fh->{buffer} = '';
	while (1) {
	    my $data = &{$fh->{write_code}}(1024);
	    last if ! defined $data || $data eq '';
	    $line .= $data;
	}
	return $line;
    }
    my $nlpos = index $fh->{buffer}, $newline;
    if ($nlpos >= 0) {
	$nlpos += length($newline);
	my $line = substr($fh->{buffer}, 0, $nlpos, '');
	return &{$fh->{write_convert}}($line);
    }
    if (exists $fh->{write_text_code}) {
	my $line = $fh->{buffer} . $fh->{write_text_code}->($newline);
	$fh->{buffer} = '';
	return &{$fh->{write_convert}}($line);
    }
    # we must read one at a time, even though it's painfully slow,
    # otherwise we may have the user typing one line and not knowing
    # why the program is stopped dead.
    while ($nlpos < 0) {
	my $data = &{$fh->{write_code}}(1);
	last if $data eq '';
	$fh->{buffer} .= $data;
	$nlpos = index $fh->{buffer}, $newline;
    }
    if ($nlpos < 0) {
	$nlpos = length($fh->{buffer});
    } else {
	$nlpos += length($newline);
    }
    my $line = substr($fh->{buffer}, 0, $nlpos, '');
    &{$fh->{write_convert}}($line);
}

sub write_charset {
    @_ == 1 || @_ == 2 or croak "Usage: IO->write_charset [(CHARSET)]";
    my $fh = shift;
    my $oc = $fh->{write_charset};
    if (@_) {
	my $charset = shift;
	$fh->{write_charset} = $charset;
	$fh->{write_convert} = toascii($charset);
	$fh->{write_unconvert} = fromascii($charset);
	eval { $fh->{text_newline} = $fh->{write_unconvert}->("\n") };
	$fh->{text_newline} = "\n" if $@;
    }
    $oc;
}

1;
