/*
  Copyright (c) 2005-2009 by Jakob Schroeter <js@camaya.net>
  This file is part of the gloox library. http://camaya.net/gloox

  This software is distributed under a license. The full license
  agreement can be found in the file LICENSE in this distribution.
  This software may not be copied, modified, sold or distributed
  other than expressed in the named license agreement.

  This software is distributed without any warranty.
*/

/*! @mainpage gloox API Documentation
 *
 * @section contents Contents
 * @ref intro_sec <br>
 * @ref handlers_sec <br>
 * @ref comp_sec <br>
 * @ref client_sec <br>
 * @ref block_conn_sec <br>
 * @ref roster_sec <br>
 * @ref privacy_sec <br>
 * @ref auth_sec <br>
 * @ref msg_sec <br>
 * @ref xeps_sec <br>
 * @ref filetransfer_sec <br>
 * @ref proxy_sec <br>
 * @ref upgrading_sec <br>
 * <br>
 *
 * @section intro_sec Introduction
 *
 * The design of gloox follows the so-called observer pattern, which basically means that everything is
 * event-driven. There are two ways you can connect to the Jabber/XMPP network using gloox, either as
 * client or as component. For a C++ XMPP server library see <http://camaya.net/glooxd>.
 *
 * @note Section 11.5 of the XMPP specification (RFC 3290) requires that only UTF-8 is used as encoding
 * for any traffic sent over the wire. Since gloox cannot know which encoding is used in any given input,
 * it is a requirement that any input to gloox is valid UTF-8.
 *
 * @section handlers_sec Event Handlers
 *
 * The most important tools of gloox are the event handlers. Currently, there exist 4 handlers for
 * the basic protocol as defined in the RFCs, as well as numerous handlers for events generated by
 * the included XEP-implementations and for additional functionality. Additionally, a log handler,
 * a generic tag handler and a handler for connection events are available.
 *
 * Basically these handlers are virtual interfaces from which you derive a class and implement a few
 * virtual functions. Then you register such an object with the respective protocol implementation. A
 * short example:
 * @code
 * class MyClass : public PresenceHandler
 * {
 *   public:
 *     // reimplemented from PresenceHandler
 *     virtual void handlePresence( const Presence& presence );
 *
 *   [...]
 * };
 *
 * void MyClass::handlePresence( const Presence& presence )
 * {
 *   // extract further information from the Presence object
 * }
 * @endcode
 *
 * Somewhere else you do something like this:
 * @code
 * OtherClass::doSomething()
 * {
 *   Client* client = new Client( ... );
 *   [...]
 *   MyClass* handler = new MyClass( ... );
 *   client->registerPresenceHandler( handler );
 * }
 * @endcode
 *
 * Now, every time a presence stanza (not subscription stanza) is received, handlePresence() is called
 * with the current stanza as argument. You can then use the extensive getters of the Stanza class to
 * extract stanza data.
 *
 * This works similar for all the other event handlers.
 * Another example, this time using the connection event handler (class @link gloox::ConnectionListener
 * ConnectionListener @endlink):
 * @code
 * class MyClass : public ConnectionListener
 * {
 *   public:
 *     virtual void onConnect();
 *
 *     virtual bool onTLSConnect( ... );
 * };
 *
 * void MyClass::onConnect()
 * {
 *   // do something when the connection is established
 * }
 *
 * bool MyClass::onTLSConnect( const CertInfo& info )
 * {
 *   // decide whether you trust the certificate, examine the CertInfo structure
 *   return true; // if you trust it, otherwise return false
 * }
 * @endcode
 *
 * @note The ConnectionListener interface is a peculiarity. You MUST re-implement
 * @link gloox::ConnectionListener::onTLSConnect() ConnectionListener::onTLSConnect() @endlink if
 * you want to be able to connect successfully to TLS/SSL enabled servers. Even though gloox tries
 * to verify the server's certificate it does not automatically trust a server. The client's programmer
 * and/or user have to decide whether to trust a server or not. This trust is expressed by the return
 * value of onTLSConnect(). @b False means you don't trust the server/certificate and as a consequence
 * the connection is dropped immediately.
 *
 * @section comp_sec Components
 *
 * A component in the Jabber/XMPP network is an add-on to a server which runs externally
 * to the actual server software, but can have similar privileges. Components use a protocol described in
 * XEP-0114 to connect and authenticate to a server.
 *
 * The @link gloox::Component Component @endlink class supports this protocol and can be used to create
 * a new Jabber component. It's as simple as:
 * @code
 * Component* comp = new Component( ... );
 * comp->connect();
 * @endcode
 *
 * @section client_sec Clients
 *
 * A client can be an end-user's chat client, a bot, or a similar entity not tied to a particular
 * server. The @link gloox::Client Client @endlink class implements the necessary functionality to
 * connect to an XMPP server. Usage is, again, pretty simple:
 * @code
 * class MyClass : public ConnectionListener, PresenceHandler
 * {
 *   public:
 *     void doSomething();
 *
 *     virtual void handlePresence( ... );
 *
 *     virtual void onConnect();
 *
 *     virtual bool onTLSConnect( const CertInfo& info );
 * };
 *
 * void MyClass::doSomething()
 * {
 *   JID jid( "jid@server/resource" );
 *   Client* client = new Client( jid, "password" );
 *   client->registerConnectionListener( this );
 *   client->registerPresenceHandler( this );
 *   client->connect();
 * }
 *
 * void MyClass::onConnect()
 * {
 *   // connection established, auth done (see API docs for exceptions)
 * }
 *
 * bool MyClass::onTLSConnect( const CertInfo& info )
 * {
 *   // examine certificate info
 * }
 *
 * void MyClass::handlePresence( Presence* presence )
 * {
 *   // presence info
 * }
 * @endcode
 *
 * @note gloox does not officially support the style of connection which is usually used on port
 * 5223, i.e. SSL encryption before any XML is sent, because it's a legacy method and not standard XMPP.
 * However, gloox includes a ConnectionTLS class that, as a side-effect, allows you to establish such
 * connections.
 *
 * @note @link gloox::Client::connect() Client::connect() @endlink by default blocks until the
 * connection ends (either @link gloox::Client::disconnect() Client::disconnect() @endlink is called
 * or the server closes the connection).
 *
 * @section block_conn_sec Blocking vs. Non-blocking Connections
 *
 * For some kind of bots a blocking connection (the default behaviour) is ideal. All the bot does is
 * react to events coming from the server. However, for end user clients or anything with a GUI this
 * is far from perfect.
 *
 * In these cases non-blocking connections can be used. If
 * @link gloox::ClientBase::connect() ClientBase::connect( false ) @endlink is
 * called, the function returnes immediately after the connection has been established. It is then
 * the resposibility of the programmer to initiate receiving of data from the socket.
 *
 * The easiest way is to call @link gloox::ClientBase::recv() ClientBase::recv() @endlink
 * periodically with the desired timeout (in microseconds) as parameter. The default value of -1
 * means the call blocks until any data was received, which is then parsed automatically.
 *
 * As an alternative to periodic polling you can get a hold of the raw file descriptor used for the
 * connection. You can then use select() on it and use
 * @link gloox::ClientBase::recv() ClientBase::recv() @endlink when select indicates that data is
 * available. You should @b not recv() any data from the file descriptor directly as there is no
 * way to feed that back into the parser.
 *
 * To get the file descriptor you'll need to set a connection class (e.g. an instance of
 * @link gloox::ConnectionTCPClient ConnectionTCPClient @endlink) manually, like so:
 *
 * @code
 * Client* client = new Client( ... );
 * ConnectionTCPClient* conn = new ConnectionTCPClient( client, client->logInstance(), server, port );
 * client->setConnectionImpl( conn );
 *
 * client->connect( false );
 * int sock = conn->socket();
 *
 * [...]
 * @endcode
 *
 * It would also be possible to fetch the fd like this:
 *
 * @code
 * Client* client = new Client( ... );
 * client->connect( false );
 * int sock = static_cast<ConnectionTCPClient*>( client->connectionImpl() )->socket();
 *
 * [...]
 * @endcode
 * Obviously this will only work as long as you haven't set a different type of connection using setConnectionImpl().
 *
 * @note This has changed in 0.9. ClientBase::fileDescriptor() is no longer available.
 *
 * @section roster_sec Roster Management
 *
 * Among others, RFC 3921 defines the protocol to manage one's contact list (roster). In gloox, the
 * @link gloox::RosterManager RosterManager @endlink class implements this functionality. A few
 * easy-to-use functions are available to subscribe to or unsubscribe from the presence of remote
 * entities. It is also possible to add a contact to a roster without actually subscribing to the
 * contacts presence. Additionally, the interface @link gloox::RosterListener RosterListener @endlink
 * offers many callbacks for various roster-related events.
 *
 * If you create a Client object as shown above, you also get a RosterManager for free.
 * @link gloox::Client::rosterManager() Client::rosterManager() @endlink returns a pointer to the
 * object.
 *
 * @section privacy_sec Privacy Lists
 *
 * Also defined in RFC 3921: Privacy Lists. A Privacy List can be used to explicitely block or allow
 * sending of stanzas from and to contacts, respectively. You can define rules based on JID, stanza type,
 * etc. The @link gloox::PrivacyManager PrivacyManager @endlink class and the
 * @link gloox::PrivacyListHandler PrivacyListHandler @endlink virtual interface allow for full
 * flexibility in Privacy List handling.
 *
 * @code
 * PrivacyManager* p = new PrivacyManager( ... );
 * [...]
 * PrivacyListHandler::PrivacyList list;
 * PrivacyItem item( PrivacyItem::TypeJid, PrivacyItem::ActionDeny,
 *                   PrivacyItem::PacketMessage, "me@there.com" );
 * list.push_back( item );
 *
 * PrivacyItem item2( PrivacyItem::TypeJid, PrivacyItem::ActionAllow,
 *                    PrivacyItem::PacketIq, "me@example.org" );
 * list.push_back( item2 );
 *
 * p->store( "myList", list );
 * @endcode
 *
 * @section auth_sec Authentication
 *
 * gloox supports old-style IQ-based authentication defined in XEP-0078 as well as several SASL mechanisms.
 * See the documentation of the @link gloox::Client Client @endlink class for more information.
 *
 * @section msg_sec Sending and Receiving of Chat Messages
 *
 * For Messaging it is recommended to use the MessageSession interface. It handles sending and receiving
 * of messages as well as message events and chat states (such as typing notifications, etc.). See
 * @link gloox::MessageSession MessageSession @endlink for more details.
 *
 * @section xeps_sec Protocol Extensions (XEPs)
 *
 * The XMPP Standards Foundation has published a number of extensions to the core protocols, called
 * XMPP Extension Protocols (XEPs). A couple of these XEPs are implemented in gloox:
 *
 * @li XEP-0004 @link gloox::DataForm Data Forms @endlink
 * @li XEP-0012 @link gloox::LastActivity  Last Activity @endlink
 * @li XEP-0013 @link gloox::FlexibleOffline Flexible Offline Message Retrieval @endlink
 * @li XEP-0022 Message Events (see @link gloox::MessageSession MessageSession @endlink for examples)
 * @li XEP-0027 Current Jabber OpenPGP Usage (see @link gloox::GPGSigned GPGSigned @endlink
 * and @link gloox::GPGEncrypted GPGEncrypted @endlink)
 * @li XEP-0030 @link gloox::Disco Service Discovery @endlink
 * @li XEP-0045 @link gloox::MUCRoom Multi-User Chat @endlink
 * @li XEP-0047 Used with @ref filetransfer_sec
 * @li XEP-0048 @link gloox::BookmarkStorage Bookmark Storage @endlink
 * @li XEP-0049 @link gloox::PrivateXML Private XML Storage @endlink
 * @li XEP-0050 @link gloox::Adhoc Ad-hoc Commands @endlink
 * @li XEP-0054 @link gloox::VCardManager vcard-temp @endlink
 * @li XEP-0060 @link gloox::PubSub::Manager Publish-Subscribe @endlink
 * @li XEP-0065 @link gloox::SOCKS5BytestreamManager SOCKS5 Bytestreams @endlink, used with
 * @ref filetransfer_sec and @ref proxy_sec
 * @li XEP-0066 @link gloox::OOB Out of Band Data @endlink, also used with @ref filetransfer_sec
 * @li XEP-0077 @link gloox::Registration In-Band Registration @endlink
 * @li XEP-0078 Non-SASL Authentication (automatically used if the server does not support SASL)
 * @li XEP-0079 @link gloox::AMP Advanced Message Processing @endlink
 * @li XEP-0083 Nested Roster Groups (automatically used if supported by the server. see
 * @link gloox::RosterManager::delimiter() RosterManager @endlink)
 * @li XEP-0085 Chat State Notifications (see @link gloox::MessageSession MessageSession @endlink for
 * examples)
 * @li XEP-0091 @link gloox::DelayedDelivery Delayed Delivery @endlink (old spec)
 * @li XEP-0092 Software Version (integrated into @link gloox::Disco Service Discovery @endlink)
 * @li XEP-0095 @link gloox::SIManager Stream Initiation @endlink, used with @ref filetransfer_sec
 * @li XEP-0096 @ref filetransfer_sec
 * @li XEP-0106 @link gloox::JID::escapeNode() JID Escaping @endlink
 * @li XEP-0114 @link gloox::Component Jabber Component Protocol @endlink
 * @li XEP-0115 @link gloox::Capabilities Entity Capabilities @endlink (used automatically internally)
 * @li XEP-0124 @link gloox::ConnectionBOSH Bidirectional-streams Over Synchronous HTTP (BOSH) @endlink
 * @li XEP-0131 @link gloox::SHIM Stanza Headers and Internet Metadata @endlink
 * @li XEP-0138 Stream Compression (used automatically if gloox is compiled with zlib and if the server
 * supports it)
 * @li XEP-0145 @link gloox::Annotations Annotations @endlink
 * @li XEP-0153 @link gloox::VCardUpdate vCard-based Avatars @endlink
 * @li XEP-0172 @link gloox::Nickname User Nickname @endlink
 * @li XEP-0184 @link gloox::Receipt Message Receipts @endlink
 * @li XEP-0199 @link gloox::ClientBase::xmppPing() XMPP Ping @endlink
 * @li XEP-0203 @link gloox::DelayedDelivery Delayed Delivery @endlink (new spec)
 * @li XEP-0206 @link gloox::ConnectionBOSH see BOSH @endlink
 * @li XEP-0224 @link gloox::Attention Attention @endlink
 * @li XEP-0256 @link gloox::LastActivity::Query Last Activity in Presence @endlink
 *
 * Further extensions can easily be implemented using
 * @link gloox::StanzaExtension StanzaExtensions @endlink.
 *
 * @section filetransfer_sec File Transfer
 *
 * For file transfer, gloox implements XEP-0095 (Stream Initiation) as well XEP-0096 (File Transfer)
 * for the signalling, and XEP-0065 (SOCKS5 Bytestreams) as well as XEP-0047 (In-Band Bytestreams)
 * for the transport. See @link gloox::SIProfileFT SIProfileFT @endlink.
 *
 * @section proxy_sec HTTP and SOCKS5 Proxy support
 *
 * gloox is capable of traversing HTTP as well as SOCKS5 proxies, even chained. See
 * @link gloox::ConnectionHTTPProxy ConnectionHTTPProxy @endlink and
 * @link gloox::ConnectionSOCKS5Proxy ConnectionSOCKS5Proxy @endlink.
 *
 * @section upgrading_sec Upgrading from earlier versions
 *
 * See <a href='upgrading.html'>Upgrading</a>.
 *
 */

#ifndef GLOOX_H__
#define GLOOX_H__

#include "macros.h"

#include <string>
#include <list>
#include <map>

/**
 * @brief The namespace for the gloox library.
 *
 * @author Jakob Schroeter <js@camaya.net>
 * @since 0.3
 */
namespace gloox
{
  /** Client namespace (RFC 3920)*/
  GLOOX_API extern const std::string XMLNS_CLIENT;

  /** Component Accept namespace (XEP-0114) */
  GLOOX_API extern const std::string XMLNS_COMPONENT_ACCEPT;

  /** Component Connect namespace (XEP-0114) */
  GLOOX_API extern const std::string XMLNS_COMPONENT_CONNECT;

  /** Service Discovery Info namespace (XEP-0030) */
  GLOOX_API extern const std::string XMLNS_DISCO_INFO;

  /** Service Discovery Items namespace (XEP-0030) */
  GLOOX_API extern const std::string XMLNS_DISCO_ITEMS;

  /** Service Discovery Publish namespace (XEP-0030) */
  GLOOX_API extern const std::string XMLNS_DISCO_PUBLISH;

  /** Adhoc Commands namespace (XEP-0050) */
  GLOOX_API extern const std::string XMLNS_ADHOC_COMMANDS;

  /** Stream Compression namespace (XEP-0138) */
  GLOOX_API extern const std::string XMLNS_COMPRESSION;

  /** Flexible Offline Message Retrieval (XEP-0013) */
  GLOOX_API extern const std::string XMLNS_OFFLINE;

  /** Chat State Notifications namespace (XEP-0085) */
  GLOOX_API extern const std::string XMLNS_CHAT_STATES;

  /** Advanced Message Processing (XEP-0079) */
  GLOOX_API extern const std::string XMLNS_AMP;

  /** In-Band Bytestreams namespace (XEP-0047) */
  GLOOX_API extern const std::string XMLNS_IBB;

  /** Feature Negotiation namespace (XEP-0020) */
  GLOOX_API extern const std::string XMLNS_FEATURE_NEG;

  /** Chat Session Negotiation namespace (XEP-0155) */
  GLOOX_API extern const std::string XMLNS_CHATNEG;

  /** XHTML-IM namespace (XEP-0071) */
  GLOOX_API extern const std::string XMLNS_XHTML_IM;

  /** Delayed Delivery namespace (XEP-0203) */
  GLOOX_API extern const std::string XMLNS_DELAY;

  /** Roster namespace (RFC 3921) */
  GLOOX_API extern const std::string XMLNS_ROSTER;

  /** Software Version namespace (XEP-0092) */
  GLOOX_API extern const std::string XMLNS_VERSION;

  /** In-Band Registration namespace (XEP-0077) */
  GLOOX_API extern const std::string XMLNS_REGISTER;

  /** Privacy lists namespace (RFC 3921) */
  GLOOX_API extern const std::string XMLNS_PRIVACY;

  /** Non-SASL Authentication namespace (XEP-0078) */
  GLOOX_API extern const std::string XMLNS_AUTH;

  /** Private XML Storage namespace (XEP-0049) */
  GLOOX_API extern const std::string XMLNS_PRIVATE_XML;

  /** Last Activity namespace (XEP-0012) */
  GLOOX_API extern const std::string XMLNS_LAST;

  /** Jabber Search namespace (XEP-0055) */
  GLOOX_API extern const std::string XMLNS_SEARCH;

  /** Out of Band Data (IQ) namespace (XEP-0066) */
  GLOOX_API extern const std::string XMLNS_IQ_OOB;

  /** Data Forms namespace (XEP-0004) */
  GLOOX_API extern const std::string XMLNS_X_DATA;

  /** Message Events (XEP-0022) */
  GLOOX_API extern const std::string XMLNS_X_EVENT;

  /** Out of Band Data (X) namespace (XEP-0066) */
  GLOOX_API extern const std::string XMLNS_X_OOB;

  /** Delayed Delivery namespace (XEP-0091) */
  GLOOX_API extern const std::string XMLNS_X_DELAY;

  /** Current Jabber OpenPGP Usage (Sign.) (XEP-0027) */
  GLOOX_API extern const std::string XMLNS_X_GPGSIGNED;

  /** Current Jabber OpenPGP Usage (Enc.) (XEP-0027) */
  GLOOX_API extern const std::string XMLNS_X_GPGENCRYPTED;

  /** vcard-temp namespace (XEP-0054) */
  GLOOX_API extern const std::string XMLNS_VCARD_TEMP;

  /** vCard-Based Avatars namespace (XEP-0153) */
  GLOOX_API extern const std::string XMLNS_X_VCARD_UPDATE;

  /** Bookmark Storage namespace (XEP-0048) */
  GLOOX_API extern const std::string XMLNS_BOOKMARKS;

  /** Annotations namespace (XEP-0145) */
  GLOOX_API extern const std::string XMLNS_ANNOTATIONS;

  /** Nested Roster Groups namespace (XEP-0083) */
  GLOOX_API extern const std::string XMLNS_ROSTER_DELIMITER;

  /** XMPP Ping namespace (XEP-0199) */
  GLOOX_API extern const std::string XMLNS_XMPP_PING;

  /** Stream Initiation namespace (XEP-0095) */
  GLOOX_API extern const std::string XMLNS_SI;

  /** File transfer profile of Stream Initiation (XEP-0096) */
  GLOOX_API extern const std::string XMLNS_SI_FT;

  /** SOCKS5 Bytestreams namespace (XEP-0065) */
  GLOOX_API extern const std::string XMLNS_BYTESTREAMS;

  /** Multi-User Chat namespace (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC;

  /** Multi-User Chat namespace (user) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_USER;

  /** Multi-User Chat namespace (admin) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_ADMIN;

  /** Multi-User Chat namespace (unique) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_UNIQUE;

  /** Multi-User Chat namespace (owner) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_OWNER;

  /** Multi-User Chat namespace (roominfo) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_ROOMINFO;

  /** Multi-User Chat namespace (rooms) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_ROOMS;

  /** Multi-User Chat namespace (request) (XEP-0045) */
  GLOOX_API extern const std::string XMLNS_MUC_REQUEST;

  /** PubSub namespace (XEP-0060) */
  GLOOX_API extern const std::string XMLNS_PUBSUB;

  /** PubSub namespace (errors) (XEP-0060) */
  GLOOX_API extern const std::string XMLNS_PUBSUB_ERRORS;

  /** PubSub namespace (event) (XEP-0060) */
  GLOOX_API extern const std::string XMLNS_PUBSUB_EVENT;

  /** PubSub namespace (owner) (XEP-0060) */
  GLOOX_API extern const std::string XMLNS_PUBSUB_OWNER;

  /** Entity Capabilities namespace (XEP-0115) */
  GLOOX_API extern const std::string XMLNS_CAPS;

  /** SOCKS5 Fast Mode namespace */
  GLOOX_API extern const std::string XMLNS_FT_FASTMODE;

  /** XMPP stream namespace (RFC 3920) */
  GLOOX_API extern const std::string XMLNS_STREAM;

  /** XMPP stream namespace (RFC 3920) */
  GLOOX_API extern const std::string XMLNS_XMPP_STREAM;

  /** XMPP stanzas namespace (RFC 3920) */
  GLOOX_API extern const std::string XMLNS_XMPP_STANZAS;

  /** TLS Stream Feature namespace (RFC 3920) */
  GLOOX_API extern const std::string XMLNS_STREAM_TLS;

  /** SASL Stream Feature namespace (RFC 3920) */
  GLOOX_API extern const std::string XMLNS_STREAM_SASL;

  /** Resource Bind Stream Feature (RFC 3921) */
  GLOOX_API extern const std::string XMLNS_STREAM_BIND;

  /** Session Create Stream Feature (RFC 3921) */
  GLOOX_API extern const std::string XMLNS_STREAM_SESSION;

  /** Non-SASL Auth. Stream Feature (XEP-0078) */
  GLOOX_API extern const std::string XMLNS_STREAM_IQAUTH;

  /** In-Band Registration namespace (XEP-0077) */
  GLOOX_API extern const std::string XMLNS_STREAM_IQREGISTER;

  /** Stream Compression Feature namespace (XEP-0138) */
  GLOOX_API extern const std::string XMLNS_STREAM_COMPRESS;

  /** General HTTP binding (BOSH) namespace (XEP-0124) */
  GLOOX_API extern const std::string XMLNS_HTTPBIND;

  /** XMPP-over-BOSH extensions (XEP-0206) */
  GLOOX_API extern const std::string XMLNS_XMPP_BOSH;

  /** Message Receipt namespace (XEP-0184) */
  GLOOX_API extern const std::string XMLNS_RECEIPTS;

  /** Message Receipt namespace (XEP-0172) */
  GLOOX_API extern const std::string XMLNS_NICKNAME;

  /** Jingle namespace (XEP-0166) */
  GLOOX_API extern const std::string XMLNS_JINGLE;

  /** Jingle Audio via RTP namespace (XEP-0167) */
  GLOOX_API extern const std::string XMLNS_JINGLE_AUDIO_RTP;

  /** Jingle ICE-UDP Transport namespace (XEP-0176) */
  GLOOX_API extern const std::string XMLNS_JINGLE_ICE_UDP;

  /** Jingle Raw UDP Transport namespace (XEP-0177) */
  GLOOX_API extern const std::string XMLNS_JINGLE_RAW_UDP;

  /** Jingle Video via RTP namespace (XEP-0180) */
  GLOOX_API extern const std::string XMLNS_JINGLE_VIDEO_RTP;

  /** Stanza Headers and Internet Metadata (SHIM) namespace (XEP-0131) */
  GLOOX_API extern const std::string XMLNS_SHIM;

  /** Attention namespace (XEP-0224) */
  GLOOX_API extern const std::string XMLNS_ATTENTION;


  /** Supported stream version (major). */
  GLOOX_API extern const std::string XMPP_STREAM_VERSION_MAJOR;

  /** Supported stream version (minor). */
  GLOOX_API extern const std::string XMPP_STREAM_VERSION_MINOR;

  /** gloox version */
  GLOOX_API extern const std::string GLOOX_VERSION;

  /** gloox caps node */
  GLOOX_API extern const std::string GLOOX_CAPS_NODE;

  /** A string containing "xmlns". */
  GLOOX_API extern const std::string XMLNS;

  /** A string containing "type". */
  GLOOX_API extern const std::string TYPE;

  /** An empty string. */
  GLOOX_API extern const std::string EmptyString;

  /**
   * This describes the possible states of a stream.
   */
  enum ConnectionState
  {
    StateDisconnected,              /**< The client is in disconnected state. */
    StateConnecting,                /**< The client is currently trying to establish a connection. */
    StateConnected                  /**< The client is connected to the server but authentication is not
                                     * (yet) done. */
  };

  /**
   * Describes stream events that get emitted by means of ConnectionListener::onStreamEvent().
   * @since 0.9
   */
  enum StreamEvent
  {
    StreamEventConnecting,          /**< The Client is about to initaite the connection. */
    StreamEventEncryption,          /**< The Client is about to negotiate encryption. */
    StreamEventCompression,         /**< The Client is about to negotiate compression. */
    StreamEventAuthentication,      /**< The Client is about to authenticate. */
    StreamEventSessionInit,         /**< The Client is about to create a session. */
    StreamEventResourceBinding,     /**< The Client is about to bind a resource to the stream. */
    StreamEventSessionCreation,     /**< The Client is about to create a session.
                                     * @since 0.9.1 */
    StreamEventRoster,              /**< The Client is about to request the roster. */
    StreamEventFinished             /**< The log-in phase is completed. */
  };

  /**
   * This describes connection error conditions.
   */
  enum ConnectionError
  {
    ConnNoError,                    /**< Not really an error. Everything went just fine. */
    ConnStreamError,                /**< A stream error occured. The stream has been closed.
                                     * Use ClientBase::streamError() to find the reason. */
    ConnStreamVersionError,         /**< The incoming stream's version is not supported */
    ConnStreamClosed,               /**< The stream has been closed (by the server). */
    ConnProxyAuthRequired,          /**< The HTTP/SOCKS5 proxy requires authentication.
                                     * @since 0.9 */
    ConnProxyAuthFailed,            /**< HTTP/SOCKS5 proxy authentication failed.
                                     * @since 0.9 */
    ConnProxyNoSupportedAuth,       /**< The HTTP/SOCKS5 proxy requires an unsupported auth mechanism.
                                     * @since 0.9 */
    ConnIoError,                    /**< An I/O error occured. */
    ConnParseError,                 /**< An XML parse error occurred. */
    ConnConnectionRefused,          /**< The connection was refused by the server (on the socket level).
                                     * @since 0.9 */
    ConnDnsError,                   /**< Resolving the server's hostname failed.
                                     * @since 0.9 */
    ConnOutOfMemory,                /**< Out of memory. Uhoh. */
    ConnNoSupportedAuth,            /**< The auth mechanisms the server offers are not supported
                                     * or the server offered no auth mechanisms at all. */
    ConnTlsFailed,                  /**< The server's certificate could not be verified or the TLS
                                     * handshake did not complete successfully. */
    ConnTlsNotAvailable,            /**< The server didn't offer TLS while it was set to be required,
                                     * or TLS was not compiled in.
                                     * @since 0.9.4 */
    ConnCompressionFailed,          /**< Negotiating/initializing compression failed.
                                     * @since 0.9 */
    ConnAuthenticationFailed,       /**< Authentication failed. Username/password wrong or account does
                                     * not exist. Use ClientBase::authError() to find the reason. */
    ConnUserDisconnected,           /**< The user (or higher-level protocol) requested a disconnect. */
    ConnNotConnected                /**< There is no active connection. */
  };

  /**
   * ClientBase's policy regarding TLS usage. Use with ClientBase::setTls().
   */
  enum TLSPolicy
  {
    TLSDisabled,                    /**< Don't use TLS. */
    TLSOptional,                    /**< Use TLS if compiled in and offered by the server. */
    TLSRequired                     /**< Don't attempt to log in if the server didn't offer TLS
                                     * or if TLS was not compiled in. Disconnect error will be
                                     * ConnTlsNotAvailable. */
  };

  /**
   * Supported Stream Features.
   */
  enum StreamFeature
  {
    StreamFeatureBind             =    1, /**< The server supports resource binding. */
    StreamFeatureUnbind           =    2, /**< The server supports binding multiple resources. */
    StreamFeatureSession          =    4, /**< The server supports sessions. */
    StreamFeatureStartTls         =    8, /**< The server supports &lt;starttls&gt;. */
    StreamFeatureIqRegister       =   16, /**< The server supports XEP-0077 (In-Band
                                           * Registration). */
    StreamFeatureIqAuth           =   32, /**< The server supports XEP-0078 (Non-SASL
                                           * Authentication). */
    StreamFeatureCompressZlib     =   64, /**< The server supports XEP-0138 (Stream
                                           * Compression) (Zlib). */
    StreamFeatureCompressDclz     =  128  /**< The server supports XEP-0138 (Stream
                                           * Compression) (LZW/DCLZ). */
    // SASLMechanism below must be adjusted accordingly.
  };

  /**
   * Supported SASL mechanisms.
   */
  // must be adjusted with changes to StreamFeature enum above
  enum SaslMechanism
  {
    SaslMechNone           =     0, /**< Invalid SASL Mechanism. */
    SaslMechDigestMd5      =   256, /**< SASL Digest-MD5 according to RFC 2831. */
    SaslMechPlain          =   512, /**< SASL PLAIN according to RFC 2595 Section 6. */
    SaslMechAnonymous      =  1024, /**< SASL ANONYMOUS according to draft-ietf-sasl-anon-05.txt/
                                     * RFC 2245 Section 6. */
    SaslMechExternal       =  2048, /**< SASL EXTERNAL according to RFC 2222 Section 7.4. */
    SaslMechGssapi         =  4096, /**< SASL GSSAPI (Win32 only). */
    SaslMechNTLM           =  8192, /**< SASL NTLM (Win32 only). */
    SaslMechGoogleToken    = 16384, /**< SASL X-GOOGLE-TOKEN. */
    SaslMechAll            = 65535  /**< Includes all supported SASL mechanisms. */
  };

  /**
   * This decribes stream error conditions as defined in RFC 3920 Sec. 4.7.3.
   */
  enum StreamError
  {
    StreamErrorBadFormat,           /**< The entity has sent XML that cannot be processed;
                                     * this error MAY be used instead of the more specific XML-related
                                     * errors, such as &lt;bad-namespace-prefix/&gt;, &lt;invalid-xml/&gt;,
                                     * &lt;restricted-xml/&gt;, &lt;unsupported-encoding/&gt;, and
                                     * &lt;xml-not-well-formed/&gt;, although the more specific errors are
                                     * preferred. */
    StreamErrorBadNamespacePrefix,  /**< The entity has sent a namespace prefix that is unsupported, or has
                                     * sent no namespace prefix on an element that requires such a prefix
                                     * (see XML Namespace Names and Prefixes (Section 11.2)). */
    StreamErrorConflict,            /**< The server is closing the active stream for this entity because a
                                     * new stream has been initiated that conflicts with the existing
                                     * stream. */
    StreamErrorConnectionTimeout,   /**< The entity has not generated any traffic over the stream for some
                                     * period of time (configurable according to a local service policy).*/
    StreamErrorHostGone,            /**< the value of the 'to' attribute provided by the initiating entity
                                     * in the stream header corresponds to a hostname that is no longer
                                     * hosted by the server.*/
    StreamErrorHostUnknown,         /**< The value of the 'to' attribute provided by the initiating entity
                                     * in the stream header does not correspond to a hostname that is hosted
                                     * by the server. */
    StreamErrorImproperAddressing,  /**< A stanza sent between two servers lacks a 'to' or 'from' attribute
                                     * (or the attribute has no value). */
    StreamErrorInternalServerError, /**< the server has experienced a misconfiguration or an
                                     * otherwise-undefined internal error that prevents it from servicing the
                                     * stream. */
    StreamErrorInvalidFrom,         /**< The JID or hostname provided in a 'from' address does not match an
                                     * authorized JID or validated domain negotiated between servers via SASL
                                     * or dialback, or between a client and a server via authentication and
                                     * resource binding.*/
    StreamErrorInvalidId,           /**< The stream ID or dialback ID is invalid or does not match an ID
                                     * previously provided. */
    StreamErrorInvalidNamespace,    /**< The streams namespace name is something other than
                                     * "http://etherx.jabber.org/streams" or the dialback namespace name is
                                     * something other than "jabber:server:dialback" (see XML Namespace Names
                                     * and Prefixes (Section 11.2)). */
    StreamErrorInvalidXml,          /**< The entity has sent invalid XML over the stream to a server that
                                     * performs validation (see Validation (Section 11.3)). */
    StreamErrorNotAuthorized,       /**< The entity has attempted to send data before the stream has been
                                     * authenticated, or otherwise is not authorized to perform an action
                                     * related to stream negotiation; the receiving entity MUST NOT process
                                     * the offending stanza before sending the stream error. */
    StreamErrorPolicyViolation,     /**< The entity has violated some local service policy; the server MAY
                                     * choose to specify the policy in the &lt;text/&gt;  element or an
                                     * application-specific condition element. */
    StreamErrorRemoteConnectionFailed,/**< The server is unable to properly connect to a remote entity that
                                     * is required for authentication or authorization. */
    StreamErrorResourceConstraint,  /**< the server lacks the system resources necessary to service the
                                     * stream. */
    StreamErrorRestrictedXml,       /**< The entity has attempted to send restricted XML features such as a
                                     * comment, processing instruction, DTD, entity reference, or unescaped
                                     * character (see Restrictions (Section 11.1)). */
    StreamErrorSeeOtherHost,        /**< The server will not provide service to the initiating entity but is
                                     * redirecting traffic to another host; the server SHOULD specify the
                                     * alternate hostname or IP address (which MUST be a valid domain
                                     * identifier) as the XML character data of the &lt;see-other-host/&gt;
                                     * element. */
    StreamErrorSystemShutdown,      /**< The server is being shut down and all active streams are being
                                     * closed. */
    StreamErrorUndefinedCondition,  /**< The error condition is not one of those defined by the other
                                     * conditions in this list; this error condition SHOULD be used only in
                                     * conjunction with an application-specific condition. */
    StreamErrorUnsupportedEncoding, /**< The initiating entity has encoded the stream in an encoding that is
                                     * not supported by the server (see Character Encoding (Section 11.5)).
                                     */
    StreamErrorUnsupportedStanzaType,/**< The initiating entity has sent a first-level child of the stream
                                     * that is not supported by the server. */
    StreamErrorUnsupportedVersion,  /**< The value of the 'version' attribute provided by the initiating
                                     * entity in the stream header specifies a version of XMPP that is not
                                     * supported by the server; the server MAY specify the version(s) it
                                     * supports in the &lt;text/&gt; element. */
    StreamErrorXmlNotWellFormed,    /**< The initiating entity has sent XML that is not well-formed as
                                     * defined by [XML]. */
    StreamErrorUndefined            /**< An undefined/unknown error occured. Also used if a diconnect was
     * user-initiated. Also set before and during a established connection
                                     * (where obviously no error occured). */
  };

  /**
   * Describes types of stanza errors.
   */
  enum StanzaErrorType
  {
    StanzaErrorTypeAuth,            /**< Retry after providing credentials. */
    StanzaErrorTypeCancel,          /**< Do not retry (the error is unrecoverable). */
    StanzaErrorTypeContinue,        /**< Proceed (the condition was only a warning). */
    StanzaErrorTypeModify,          /**< Retry after changing the data sent. */

    StanzaErrorTypeWait,            /**< Retry after waiting (the error is temporary). */
    StanzaErrorTypeUndefined        /**< No error. */
  };

  /**
   * Describes the defined stanza error conditions of RFC 3920.
   * Used by, eg., Stanza::error().
   */
  enum StanzaError
  {

    StanzaErrorBadRequest,          /**< The sender has sent XML that is malformed or that cannot be
                                     * processed (e.g., an IQ stanza that includes an unrecognized value
                                     * of the 'type' attribute); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorConflict,            /**< Access cannot be granted because an existing resource or session
                                     * exists with the same name or address; the associated error type
                                     * SHOULD be "cancel". */
    StanzaErrorFeatureNotImplemented,/**< The feature requested is not implemented by the recipient or
                                      * server and therefore cannot be processed; the associated error
                                      * type SHOULD be "cancel". */
    StanzaErrorForbidden,           /**< The requesting entity does not possess the required permissions to
                                     * perform the action; the associated error type SHOULD be "auth". */
    StanzaErrorGone,                /**< The recipient or server can no longer be contacted at this address
                                     * (the error stanza MAY contain a new address in the XML character data
                                     * of the &lt;gone/&gt; element); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorInternalServerError, /**< The server could not process the stanza because of a
                                     * misconfiguration or an otherwise-undefined internal server error; the
                                     * associated error type SHOULD be "wait". */
    StanzaErrorItemNotFound,        /**< The addressed JID or item requested cannot be found; the associated
                                     * error type SHOULD be "cancel". */
    StanzaErrorJidMalformed,        /**< The sending entity has provided or communicated an XMPP address
                                     * (e.g., a value of the 'to' attribute) or aspect thereof (e.g., a
                                     * resource identifier) that does not adhere to the syntax defined in
                                     * Addressing Scheme (Section 3); the associated error type SHOULD be
                                     * "modify". */
    StanzaErrorNotAcceptable,       /**< The recipient or server understands the request but is refusing to
                                     * process it because it does not meet criteria defined by the recipient
                                     * or server (e.g., a local policy regarding acceptable words in
                                     * messages); the associated error type SHOULD be "modify". */
    StanzaErrorNotAllowed,          /**< The recipient or server does not allow any entity to perform the
                                     * action; the associated error type SHOULD be "cancel". */
    StanzaErrorNotAuthorized,       /**< The sender must provide proper credentials before being allowed to
                                     * perform the action, or has provided impreoper credentials; the
                                     * associated error type should be "auth". */
    StanzaErrorNotModified,         /**< The item requested has not changed since it was last requested;
                                     * the associated error type SHOULD be "continue". */
    StanzaErrorPaymentRequired,     /**< The requesting entity is not authorized to access the requested
                                     * service because payment is required; the associated error type SHOULD
                                     * be "auth". */
    StanzaErrorRecipientUnavailable,/**< The intended recipient is temporarily unavailable; the associated
                                     * error type SHOULD be "wait" (note: an application MUST NOT return
                                     * this error if doing so would provide information about the intended
                                     * recipient's network availability to an entity that is not authorized
                                     * to know such information). */
    StanzaErrorRedirect,            /**< The recipient or server is redirecting requests for this
                                     * information to another entity, usually temporarily (the error
                                     * stanza SHOULD contain the alternate address, which MUST be a valid
                                     * JID, in the XML character data of the &lt;redirect/&gt; element);
                                     * the associated error type SHOULD be "modify". */
    StanzaErrorRegistrationRequired,/**< The requesting entity is not authorized to access the requested
                                     * service because registration is required; the associated error type
                                     * SHOULD be "auth". */
    StanzaErrorRemoteServerNotFound,/**< A remote server or service specified as part or all of the JID of
                                     * the intended recipient does not exist; the associated error type
                                     * SHOULD be "cancel". */
    StanzaErrorRemoteServerTimeout, /**< A remote server or service specified as part or all of the JID of
                                     * the intended recipient (or required to fulfill a request) could not
                                     * be contacted within a reasonable amount of time; the associated error
                                     * type SHOULD be "wait". */
    StanzaErrorResourceConstraint,  /**< The server or recipient lacks the system resources necessary to
                                     * service the request; the associated error type SHOULD be "wait". */
    StanzaErrorServiceUnavailable,  /**< The server or recipient does not currently provide the requested
                                     * service; the associated error type SHOULD be "cancel". */
    StanzaErrorSubscribtionRequired,/**< The requesting entity is not authorized to access the requested
                                     * service because a subscription is required; the associated error type
                                     * SHOULD be "auth". */
    StanzaErrorUndefinedCondition,  /**< The error condition is not one of those defined by the other
                                     * conditions in this list; any error type may be associated with this
                                     * condition, and it SHOULD be used only in conjunction with an
                                     * application-specific condition. */
    StanzaErrorUnexpectedRequest,   /**< The recipient or server understood the request but was not
                                     * expecting it at this time (e.g., the request was out of order);
                                     * the associated error type SHOULD be "wait". */
    StanzaErrorUnknownSender,       /**< The stanza 'from' address specified by a connected client is not
                                     * valid for the stream (e.g., the stanza does not include a 'from'
                                     * address when multiple resources are bound to the stream); the
                                     * associated error type SHOULD be "modify".*/
    StanzaErrorUndefined            /**< No stanza error occured. */
  };

  /**
   * Describes the possible 'available presence' types.
   */
//   enum Presence
//   {
//     PresenceUnknown,                /**< Unknown status. */
//     PresenceAvailable,              /**< The entity or resource is online and available. */
//     PresenceChat,                   /**< The entity or resource is actively interested in chatting. */
//     PresenceAway,                   /**< The entity or resource is temporarily away. */
//     PresenceDnd,                    /**< The entity or resource is busy (dnd = "Do Not Disturb"). */
//     PresenceXa,                     /**< The entity or resource is away for an extended period (xa =
//                                      * "eXtended Away"). */
//     PresenceUnavailable             /**< The entity or resource is offline. */
//   };

  /**
   * Describes the verification results of a certificate.
   */
  enum CertStatus
  {
    CertOk               =  0,      /**< The certificate is valid and trusted. */
    CertInvalid          =  1,      /**< The certificate is not trusted. */
    CertSignerUnknown    =  2,      /**< The certificate hasn't got a known issuer. */
    CertRevoked          =  4,      /**< The certificate has been revoked. */
    CertExpired          =  8,      /**< The certificate has expired. */
    CertNotActive        = 16,      /**< The certifiacte is not yet active. */
    CertWrongPeer        = 32,      /**< The certificate has not been issued for the
                                     * peer we're connected to. */
    CertSignerNotCa      = 64       /**< The signer is not a CA. */
  };

  /**
   * Describes the certificate presented by the peer.
   */
  struct CertInfo
  {
    int status;                     /**< Bitwise or'ed CertStatus or CertOK. */
    bool chain;                     /**< Determines whether the cert chain verified ok. */
    std::string issuer;             /**< The name of the issuing entity.*/
    std::string server;             /**< The server the certificate has been issued for. */
    int date_from;                  /**< The date from which onwards the certificate is valid
                                     * (UNIX timestamp; UTC; not set when using OpenSSL). */
    int date_to;                    /**< The date up to which the certificate is valid
                                     * (UNIX timestamp; UTC; not set when using OpenSSL). */
    std::string protocol;           /**< The encryption protocol used for the connection. */
    std::string cipher;             /**< The cipher used for the connection. */
    std::string mac;                /**< The MAC used for the connection. */
    std::string compression;        /**< The compression used for the connection. */
  };

  /**
   * Describes the defined SASL (and non-SASL) error conditions.
   */
  enum AuthenticationError
  {
    AuthErrorUndefined,             /**< No error occurred, or error condition is unknown. */
    SaslAborted,                    /**< The receiving entity acknowledges an &lt;abort/&gt; element sent
                                     * by the initiating entity; sent in reply to the &lt;abort/&gt;
                                     * element. */
    SaslIncorrectEncoding,          /**< The data provided by the initiating entity could not be processed
                                     * because the [BASE64] encoding is incorrect (e.g., because the encoding
                                     * does not adhere to the definition in Section 3 of [BASE64]); sent in
                                     * reply to a &lt;response/&gt; element or an &lt;auth/&gt; element with
                                     * initial response data. */
    SaslInvalidAuthzid,             /**< The authzid provided by the initiating entity is invalid, either
                                     * because it is incorrectly formatted or because the initiating entity
                                     * does not have permissions to authorize that ID; sent in reply to a
                                     * &lt;response/&gt; element or an &lt;auth/&gt; element with initial
                                     * response data.*/
    SaslInvalidMechanism,           /**< The initiating entity did not provide a mechanism or requested a
                                     * mechanism that is not supported by the receiving entity; sent in reply
                                     * to an &lt;auth/&gt; element. */
    SaslMalformedRequest,           /**< The request is malformed (e.g., the &lt;auth/&gt; element includes
                                     * an initial response but the mechanism does not allow that); sent in
                                     * reply to an &lt;abort/&gt;, &lt;auth/&gt;, &lt;challenge/&gt;, or
                                     * &lt;response/&gt; element. */
    SaslMechanismTooWeak,           /**< The mechanism requested by the initiating entity is weaker than
                                     * server policy permits for that initiating entity; sent in reply to a
                                     * &lt;response/&gt; element or an &lt;auth/&gt; element with initial
                                     * response data. */
    SaslNotAuthorized,              /**< The authentication failed because the initiating entity did not
                                     * provide valid credentials (this includes but is not limited to the
                                     * case of an unknown username); sent in reply to a &lt;response/&gt;
                                     * element or an &lt;auth/&gt; element with initial response data. */
    SaslTemporaryAuthFailure,       /**< The authentication failed because of a temporary error condition
                                     * within the receiving entity; sent in reply to an &lt;auth/&gt; element
                                     * or &lt;response/&gt; element. */
    NonSaslConflict,                /**< XEP-0078: Resource Conflict */
    NonSaslNotAcceptable,           /**< XEP-0078: Required Information Not Provided */
    NonSaslNotAuthorized            /**< XEP-0078: Incorrect Credentials */
  };

  /**
   * Identifies log sources.
   */
  enum LogArea
  {
    LogAreaClassParser                = 0x000001, /**< Log messages from Parser. */
    LogAreaClassConnectionTCPBase     = 0x000002, /**< Log messages from ConnectionTCPBase. */
    LogAreaClassClient                = 0x000004, /**< Log messages from Client. */
    LogAreaClassClientbase            = 0x000008, /**< Log messages from ClientBase. */
    LogAreaClassComponent             = 0x000010, /**< Log messages from Component. */
    LogAreaClassDns                   = 0x000020, /**< Log messages from DNS. */
    LogAreaClassConnectionHTTPProxy   = 0x000040, /**< Log messages from ConnectionHTTPProxy */
    LogAreaClassConnectionSOCKS5Proxy = 0x000080, /**< Log messages from ConnectionSOCKS5Proxy */
    LogAreaClassConnectionTCPClient   = 0x000100, /**< Log messages from ConnectionTCPClient. */
    LogAreaClassConnectionTCPServer   = 0x000200, /**< Log messages from ConnectionTCPServer. */
    LogAreaClassS5BManager            = 0x000400, /**< Log messages from SOCKS5BytestreamManager. */
    LogAreaClassSOCKS5Bytestream      = 0x000800, /**< Log messages from SOCKS5Bytestream. */
    LogAreaClassConnectionBOSH        = 0x001000, /**< Log messages from ConnectionBOSH */
    LogAreaClassConnectionTLS         = 0x002000, /**< Log messages from ConnectionTLS */
    LogAreaAllClasses                 = 0x01FFFF, /**< All log messages from all the classes. */
    LogAreaXmlIncoming                = 0x020000, /**< Incoming XML. */
    LogAreaXmlOutgoing                = 0x040000, /**< Outgoing XML. */
    LogAreaUser                       = 0x800000, /**< User-defined sources. */
    LogAreaAll                        = 0xFFFFFF  /**< All log sources. */
  };

  /**
   * Describes a log message's severity.
   */
  enum LogLevel
  {
    LogLevelDebug,                  /**< Debug messages. */
    LogLevelWarning,                /**< Non-crititcal warning messages. */
    LogLevelError                   /**< Critical, unrecoverable errors. */
  };

  /**
   * The possible Message Events according to XEP-0022.
   */
  enum MessageEventType
  {
    MessageEventOffline   =  1,     /**< Indicates that the message has been stored offline by the
                                     * intended recipient's server. */
    MessageEventDelivered =  2,     /**< Indicates that the message has been delivered to the
                                     * recipient. */
    MessageEventDisplayed =  4,     /**< Indicates that the message has been displayed */
    MessageEventComposing =  8,     /**< Indicates that a reply is being composed. */
    MessageEventInvalid   = 16,     /**< Invalid type. */
    MessageEventCancel    = 32      /**< Cancels the 'Composing' event. */
  };

  /**
   * The possible Chat States according to XEP-0085.
   */
  enum ChatStateType
  {
    ChatStateActive       =  1,     /**< User is actively participating in the chat session. */
    ChatStateComposing    =  2,     /**< User is composing a message. */
    ChatStatePaused       =  4,     /**< User had been composing but now has stopped. */
    ChatStateInactive     =  8,     /**< User has not been actively participating in the chat session. */
    ChatStateGone         = 16,     /**< User has effectively ended their participation in the chat
                                     * session. */
    ChatStateInvalid      = 32      /**< Invalid type. */
  };

  /**
   * Describes the possible error conditions for resource binding.
   */
  enum ResourceBindError
  {
    RbErrorUnknownError,            /**< An unknown error occured. */
    RbErrorBadRequest,              /**< Resource identifier cannot be processed. */
    RbErrorNotAllowed,              /**< Client is not allowed to bind a resource. */
    RbErrorConflict                 /**< Resource identifier is in use. */
  };

  /**
   * Describes the possible error conditions for session establishemnt.
   */
  enum SessionCreateError
  {
    ScErrorUnknownError,            /**< An unknown error occured. */
    ScErrorInternalServerError,     /**< Internal server error. */
    ScErrorForbidden,               /**< Username or resource not allowed to create session. */
    ScErrorConflict                 /**< Server informs newly-requested session of resource
                                     * conflict. */
  };

  /**
   * Currently implemented message session filters.
   */
  enum MessageSessionFilter
  {
    FilterMessageEvents    = 1,     /**< Message Events (XEP-0022) */
    FilterChatStates       = 2      /**< Chat State Notifications (XEP-0085) */
  };

  /**
   * Defined MUC room affiliations. See XEP-0045 for default privileges.
   */
  enum MUCRoomAffiliation
  {
    AffiliationNone,                /**< No affiliation with the room. */
    AffiliationOutcast,             /**< The user has been banned from the room. */
    AffiliationMember,              /**< The user is a member of the room. */
    AffiliationOwner,               /**< The user is a room owner. */
    AffiliationAdmin,               /**< The user is a room admin. */
    AffiliationInvalid              /**< Invalid affiliation. */
  };

  /**
   * Defined MUC room roles. See XEP-0045 for default privileges.
   */
  enum MUCRoomRole
  {
    RoleNone,                       /**< Not present in room. */
    RoleVisitor,                    /**< The user visits a room. */
    RoleParticipant,                /**< The user has voice in a moderatd room. */
    RoleModerator,                  /**< The user is a room moderator. */
    RoleInvalid                     /**< Invalid role. */
  };

  /**
   * Configuration flags for a room.
   */
  enum MUCRoomFlag
  {
    FlagPasswordProtected  = 1<< 1, /**< Password-protected room. */
    FlagPublicLogging      = 1<< 2, /**< Room conversation is logged. Code: 170 */
    FlagPublicLoggingOff   = 1<< 3, /**< Room conversation is not logged. Code: 171 */
    FlagHidden             = 1<< 4, /**< Hidden room. */
    FlagMembersOnly        = 1<< 5, /**< Members-only room. */
    FlagModerated          = 1<< 6, /**< Moderated room. */
    FlagNonAnonymous       = 1<< 7, /**< Non-anonymous room. Code: 100, 172 */
    FlagOpen               = 1<< 8, /**< Open room. */
    FlagPersistent         = 1<< 9, /**< Persistent room .*/
    FlagPublic             = 1<<10, /**< Public room. */
    FlagSemiAnonymous      = 1<<11, /**< Semi-anonymous room. Code: 173 */
    FlagTemporary          = 1<<12, /**< Temporary room. */
    FlagUnmoderated        = 1<<13, /**< Unmoderated room. */
    FlagUnsecured          = 1<<14, /**< Unsecured room. */
    FlagFullyAnonymous     = 1<<15  /**< Fully anonymous room. Code: 174 */
    // keep in sync with MUCUserFlag below
  };

  /**
   * Configuration flags for a user.
   */
  // keep in sync with MUCRoomFlag above
  enum MUCUserFlag
  {
    UserSelf               = 1<<16, /**< Other flags relate to the current user him/herself. Code: 110 */
    UserNickChanged        = 1<<17, /**< The user changed his/her nickname. Code: 303 */
    UserKicked             = 1<<18, /**< The user has been kicked. Code: 307 */
    UserBanned             = 1<<19, /**< The user has been banned. Code: 301 */
    UserAffiliationChanged = 1<<20, /**< The user's affiliation with the room changed and as a result
                                     * he/she has been removed from the room. Code: 321 */
    UserRoomDestroyed      = 1<<21, /**< The room has been destroyed. */
    UserNickAssigned       = 1<<22, /**< Service has assigned or modified occupant's roomnick.
                                     * Code: 210*/
    UserNewRoom            = 1<<23, /**< The room has been newly created. Code: 201*/
    UserMembershipRequired = 1<<24, /**< User is being removed from the room because the room has
                                     * been changed to members-only and the user is not a member.
                                     * Code: 322 */
    UserRoomShutdown       = 1<<25, /**< User is being removed from the room because of a system
                                     * shutdown. Code: 332 */
    UserAffiliationChangedWNR = 1<<26 /**< The user's affiliation changed While Not in the Room.
                                       * Code: 101 */
  };

  /**
   * Describes possible subscription types according to RFC 3921, Section 9.
   */
  enum SubscriptionType
  {
    S10nNone,                       /**< Contact and user are not subscribed to each other, and
                                     * neither has requested a subscription from the other. */
    S10nNoneOut,                    /**< Contact and user are not subscribed to each other, and
                                     * user has sent contact a subscription request but contact
                                     * has not replied yet. */
    S10nNoneIn,                     /**< Contact and user are not subscribed to each other, and
                                     * contact has sent user a subscription request but user has
                                     * not replied yet (note: contact's server SHOULD NOT push or
                                     * deliver roster items in this state, but instead SHOULD wait
                                     * until contact has approved subscription request from user). */
    S10nNoneOutIn,                  /**< Contact and user are not subscribed to each other, contact
                                     * has sent user a subscription request but user has not replied
                                     * yet, and user has sent contact a subscription request but
                                     * contact has not replied yet. */
    S10nTo,                         /**< User is subscribed to contact (one-way). */
    S10nToIn,                       /**< User is subscribed to contact, and contact has sent user a
                                     * subscription request but user has not replied yet. */
    S10nFrom,                       /**< Contact is subscribed to user (one-way). */
    S10nFromOut,                    /**< Contact is subscribed to user, and user has sent contact a
                                     * subscription request but contact has not replied yet. */
    S10nBoth                        /**< User and contact are subscribed to each other (two-way). */
  };

  /**
   * A list of strings.
   */
  typedef std::list<std::string> StringList;

  /**
   * A list of pointers to strings.
   */
  typedef std::list<std::string*> StringPList;

  /**
   * A map of strings.
   */
  typedef std::map<std::string, std::string> StringMap;

  /**
   * A multimap of strings.
   */
  typedef std::multimap<std::string, std::string> StringMultiMap;

  class StanzaExtension;
  /**
   * A list of StanzaExtensions.
   */
  typedef std::list<const StanzaExtension*> StanzaExtensionList;
}

extern "C"
{
  GLOOX_API const char* gloox_version();
}

#endif // GLOOX_H__
