###########################################################################
# clive, video extraction utility
# Copyright (C) 2007 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

import sys
import urllib2
import time
import os
import traceback

from sys import stdout, stderr
from cStringIO import StringIO

from clive import *
from clive.newt import *
from clive.util import *
from clive.login import *


__all__ = ['CliveCLI']


class CliveCLI:
  """Command line interface for clive"""
  def __init__(self):
    sys.excepthook = self._exchook
    self.retval = 0

  def main(self):
    try:
      self.ext = Extract(
        self._onsay,
        self._onscan,
        self._onread,
        self._onprompt
      )

      # Display info on reporting bugs
      if self.ext.opts.bugs:
        return self._reportbugs()

      # Login to youtube.com
      if self.ext.opts.use_youtube_login:
        if self.ext.opts.youtube_user and\
            self.ext.opts.youtube_pass:
          yt = YTLogin(
            self.ext.opts,
            self._onsay,
            self.ext.getcookiefile()
          )
          if not yt.islogged():
            # Log in only if not logged in already
            self._onsay('info : youtube; logging in...')
            yt.login(self.ext.newhttpget(None,hook=None,gzip=True))
            # Do not exit, proceed to extract
        elif (self.ext.opts.youtube_user and not \
              self.ext.opts.youtube_pass) or \
              (not self.ext.opts.youtube_user and \
              self.ext.opts.youtube_pass):
          # Either username or password was not set
          raise CliveError('error : specify both --youtube-user '\
                            'and --youtube-pass')

      # Logout from youtube.com
      if self.ext.opts.youtube_logout:
        yt = YTLogin(None, self._onsay, self.ext.getcookiefile())
        return yt.logout()

      # Display youtube.com log-in status
      if self.ext.opts.youtube_status:
        yt = YTLogin(None, self._onsay, self.ext.getcookiefile())
        return yt.status()

      # Display details, e.g. agent, proxy etc.
      if self.ext.opts.prefix:
        self._onsay('prefix : %s' % self.ext.opts.prefix)

      agent = self.ext.opts.agent
      if not agent:
        agent = '%s/%s' %(urllib2.__name__,urllib2.__version__)

      self._onsay('agent : %s' % agent)
      proxy = self.ext.opts.proxy
      if not proxy:
        proxy = os.environ.get('http_proxy')
        if not proxy:
          proxy = 'no'
      if not self.ext.opts.use_proxy:
        proxy = 'disabled'
      self._onsay('proxy : %s' % proxy)

      yt = YTLogin(None, self._onsay, self.ext.getcookiefile())
      if yt.islogged():
        self._onsay('info : logged-in (youtube.com)')

      # Check opts
      if self.ext.opts.newt:
        self.ext.opts.newt = _newt_installed
        if self.ext.opts.nonewt:
          self.ext.opts.newt = False

      if self.ext.opts.newt:
        self._onsay('newt : yes')
      else:
        self._onsay('newt : no')

      if not self.ext.opts.verbose:
        self.ext.opts.newt = False

      if self.ext.opts.emit:
        self.ext.opts.strip = False

      if self.ext.opts.play_format and \
          not self.ext.opts.noencode and \
          not self.ext.opts.noplay:
        self.ext.opts.play = True
        format = self.ext.opts.play_format.lower()
        setattr(self.ext.opts,format,True)

      if self.ext.opts.noencode:
        self.ext.opts.mpeg = False
        self.ext.opts.avi = False
        self.ext.opts.flv = False

      if self.ext.opts.noplay:
        self.ext.opts.play = False

      if self.ext.opts.noaccept:
        self.ext.opts.accept = False

      if self.ext.opts.noskip:
        self.ext.opts.skip_existing = False

      self._checkencoder()

      # Scan for videos
      (videos,ignored) = self.ext.scan()
      self._onsay(showalways=True) # newline

      if len(videos) == 0:
        self._showignored(ignored)
        raise CliveError('error : nothing to extract')

      # Emit found video URLs to stdout, ignored URLs go unreported
      if self.ext.opts.emit:
        for video in videos:
          size = byteshuman(long(video['size']))
          text = 'video : %s %s %s' % (video['url'],video['file'],size)
          self._onsay(text, showalways=True)
        self._emitignored(ignored)
        return # Exit program

      # Select videos (newt)
      if self.ext.opts.newt and self.ext.opts.verbose:
        # UI: NEWT
        ui = newtUI()
        if not self.ext.opts.accept and len(videos) > 1:
          ui.main(videos, ignored)
          if not ui.result:
            raise CliveError('error : canceled')
          if len(ui.cursel) == 0:
            raise CliveError('error : nothing was selected')
          videos = ui.cursel
        if not self.ext.opts.overwrite and not self.ext.opts.skip_existing:
          if not ui.checkoverwrite(videos, self.ext.opts.prefix):
            raise CliveError('error : canceled')
      else:
        # UI: cmdline only -- print ignored URLs, video selection made later
        self._showignored(ignored)

      # Extract videos
      for video in videos:
        if self._promptvideo(video,len(videos)).lower() == 'y':
          self.ext.extract(video)

    except CliveError, e:
          self._onerror(e.text)

  # Non-public

  def _onsay(self, text='\n', showalways=False):
    if self.ext.opts.verbose or showalways:
      stdout.write(self._cleanuptext(text))

  def _onerror(self, text):
    # Show always
    stderr.write(self._cleanuptext(text))
    self.retval = 1

  def _cleanuptext(self, text):
    if self.ext.opts.strip:
      text = self._striptext(text)
    return text.rstrip('\r\n') + '\n'

  def _striptext(self, text):
    """Strips text to fit in a line"""
    text = text[:70]
    if len(text) >= 70:
      text += ' (...)'
    return text

  def _onscan(self, text):
    self._writebspaced('scan : '+text)

  def _onread(self, read, length):
    text_left = None

    if length == -1:
      text = 'read : %s' % bytesmb(read)
      progress = -1
    else:
      progress = percent(read,length)
      text = 'read : %s / %s (%03d%%)' % (
        bytesmb(read),
        bytesmb(length),
        progress
      )

    if read <= 1024:
      self.timestart = time.time()
    else:
      elapsed = time.time() - self.timestart
      if elapsed == 0: elapsed = 1
      rate = read / elapsed
      left = (length - read) / rate
      rate = byteshuman(long(rate))
      gmtime = time.gmtime(left)
      timeleft = '%02d:%02d:%02d' % (gmtime[3],gmtime[4],gmtime[5])
      text_left = '[%s] %12s/s' % (timeleft,rate)

    if text_left: text += (' '+text_left)
    if progress == 100: text += '\n'

    self._writebspaced(text)

  def _writebspaced(self, text):
    bspaces = '\b' * len(text)
    stdout.write(bspaces + text)
    stdout.flush()

  def _onprompt(self, text):
    text = '>> ' + text
    if self.ext.opts.verbose:
      return raw_input(text)
    else:
      return '(default)'

  def _promptvideo(self, video, maxvideos):
    a = 'y' # Default
    if not self.ext.opts.newt:
      self._onsay('video : %s (%s)' % \
        (video['file'],byteshuman(video['size'])))
      if self.ext.opts.verbose:
        if not self.ext.opts.accept and maxvideos > 1:
          a = self._onprompt('extract video? (y/N):')
    return a

  def _showignored(self, ignored):
    if not self.ext.opts.verbose: return
    if len(ignored) > 0:
      if not self.ext.opts.emit:
        if self.ext.opts.newt and self.ext.opts.verbose:
          # UI: NEWT
          ui = newtUI()
          ui.main(videos=[], ignored=ignored)
        else:
          # cmdline only
          text = 'show ignored urls (%d)? (y/N):' % len(ignored)
          a = self._onprompt(text)
          if a.lower() == 'y':
            for e in ignored:
              self._onsay('(i) : %s' % e['url'])
              self._onsay('(e) : %s' % e['reason'])
      else:
        # --emit
        self._emitignored(ignored)

  def _emitignored(self, ignored):
    for video in ignored:
      reason = video['reason'].replace('error : ','')
      text = 'ignored : %s %s' % (video['url'],reason)
      self._onsay(text, showalways=True)

  def _checkencoder(self):
    if (self.ext.opts.avi or \
      self.ext.opts.mpeg or \
      self.ext.opts.flv) and \
      not self.ext.opts.encoder:
        self._onsay('warn : path to encoder unspecified, use --encoder=PATH')
    else:
      if self.ext.opts.encoder:
        self._onsay('encoder : %s' % self.ext.opts.encoder)
      else:
        self._onsay('encoder : no')

  def _reportbugs(self):
    text = '\nBugs should be reported to the clive bug tracking system.\n' \
      'You will need to create an account with GNA to be able to post\n' \
      'new items to the tracking system.\n\n' \
      'Bug tracker:\n' \
      '\thttps://gna.org/bugs/?func=additem&group=clive'

    self.ext.opts.strip = False
    self._onsay(text)

  def _exchook(self, type, value, tb):
    self.retval = 1 # Set flag

    fmt  = '\n(e) : %s'
    fmt2 = '\n(e) : %s (%s)'

    if type == IOError:
      text = fmt % (value[1][1])
      self._onsay(text)
    elif type == urllib2.URLError or type == urllib2.HTTPError:
      text = fmt % (value)
      self._onsay(text)
    elif type != EOFError and type != KeyboardInterrupt:
      text = fmt2 % (type,value)
      self._onsay(text,showalways=True)
      a = self._onprompt('show traceback? (Y/n):')
      if a.lower() != 'n':
        trace = StringIO()
        traceback.print_exception(type,value,tb,None,trace)
        print trace.getvalue()
    else:
      print # new line, call print instead of .onsay(), --quiet may be used
