###########################################################################
# clive, video extraction utility
# Copyright (C) 2007 Toni Gundogdu
#
# clive is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 0.1.2-1307 USA
###########################################################################

import os.path
import time

from clive.net import *
from clive.util import CliveError

__all__ = [
  'YTLogin'
]


class YTLogin:
  def __init__(self, opts, onsay, cookiefile):
    self.opts = opts
    self.onsay = onsay
    self.cookiefile = cookiefile

  def login(self, cn):
    """
    Login to youtube.com.
      1: The server sets 'LOGIN_INFO' cookie
      2: Accept all inappropriate content by setting the 'is_adult'
         cookie without reconnecting. The user is normally asked
         to click a confirmation button to do this on the website.

    NOTE: Login is done over unsecure HTTP transfers.
          Anyone intercepting the HTTP traffic could steal
          the login info.
    """
    uname = self.opts.youtube_user
    pwd = self.opts.youtube_pass

    self.logout(report=False)

    login_url = \
      'http://www.youtube.com/login?'

    login_path = \
      'current_form=loginform&username=%s&password=%s&action_login=log+in'

    data = login_path % (uname,pwd)
    cn.submit(login_url,data,read=False)

    found = False
    for cookie in cn.cjar:
      if cookie.name == 'LOGIN_INFO':
        self.onsay('info : logged in')
        found = True
        break
    if not found:
      raise CliveError('error : login failed, incorrect login info?')

    # Set is_adult cookie (allow extraction of content that has been
    # flagged as 'mature content'.
    cn.addcookie('is_adult',host='http://youtube.com')
    cn.cjar.save(self.cookiefile)

  def logout(self, report=True):
    """
    Logout from youtube.com. Save bandwidth and just remove the
    local cookie file.
    """
    if os.path.exists(self.cookiefile):
      os.remove(self.cookiefile)
      if report:
        self.onsay('info : logged out')

  def status(self, showinfo=True):
    """
    Display login details
    """
    if not os.path.exists(self.cookiefile):
      raise CliveError('error : not logged in (cookie file not found)')
    cn = HTTPGet(cookies=(True,self.cookiefile))
    found = False
    for cookie in cn.cjar:
      if cookie.name == 'LOGIN_INFO':
        expires = time.asctime(time.localtime(cookie.expires))
        expired = cookie.is_expired()
        found = not expired
        if showinfo:
          self.onsay('domain : %s' % cookie.domain)
          self.onsay('expires : %s' % expires)
          self.onsay('expired : %s' % repr(expired))
        continue
      if cookie.name == 'is_adult':
        if showinfo:
          self.onsay('content : allow inappropriate')
        continue
    if not found:
      raise CliveError('error : not logged in')

  def islogged(self):
    """Return True if logged in"""
    try:
      self.status(showinfo=False)
    except CliveError:
      return False
    return True
