/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Mirco Müller <mirco.mueller@canonical.com>
 *
 */
/** 
 * SECTION:ctk-menu-expandable.c
 * @short_description: Class extending #CtkMenu to allow animating growing and
 * shrinking of the menu
 * @see_also: #CtkMenu
 * @include ctk-menu-expandable.h
 *
 * #CtkMenuExpandable allows to define the background as 5 separate texture
 * pieces (top, upper, anchor, lower and bottom). The parts upper and lower are
 * meant to be tileable and will be the dynamic parts of the menu-background.
 * Thus they will allow to be animated over time and have their height set going
 * from 0.0f to 1.0f. 0.0f means they are not rendered at all and 1.0f means to
 * render them at their max. size, making the menu-background be drawn at its
 * full prefered height.
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif
#include "ctk-menu.h"
#include "ctk-menu-expandable.h"
#include "ctk-menu-seperator.h"
#include "ctk-main.h"
#include <math.h>
#include <glib.h>
#include "ctk-gfx-private.h"
#include "ctk-arb-asm-private.h"

G_DEFINE_TYPE (CtkMenuExpandable, ctk_menu_expandable, CTK_TYPE_MENU);

#define CTK_MENU_EXPANDABLE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  CTK_TYPE_MENU_EXPANDABLE, \
  CtkMenuExpandablePrivate))

static void
ctk_menu_expandable_allocate (ClutterActor           *actor,
                              const ClutterActorBox  *box,
                              ClutterAllocationFlags flags);
                              
struct _CtkMenuExpandableTransitionTextures
{
  guint texture_id;
  guint mask_id;
  guint width;
  guint height;
};

struct _CtkMenuExpandablePrivate
{
  gfloat AnchorPositionX;
  gfloat AnchorPositionY;
  
  gint Padding;
  gint ContentPadding;
  gint ContentPaddingLeftRight;
  
  guint bg_fill_image_id;
  guint bg_fill_image_width;
  guint bg_fill_image_height;
  guint blurred_bkg_id;
  guint blurred_bkg_width;
  guint blurred_bkg_height;
  
  gfloat expansion_size_factor;
  
  gint upper_size;
  gint lower_size;
  
  gfloat menu_rendering_width;
  gfloat menu_rendering_height;
  
  gint center_position_in_menu;
  
  struct _CtkMenuExpandableTransitionTextures transition_array[CTK_MENU_EXPANDABLE_TRANSITION_STEPS];
  guint full_texture_id;
  guint full_mask_id;
  guint full_width;  
  guint full_height;
};

enum
{
  PROP_0,

  PROP_BG_FILL_IMAGE_ID,
  PROP_BG_FILL_IMAGE_WIDTH,
  PROP_BG_FILL_IMAGE_HEIGHT,

  PROP_EXPANSION_SIZE_FACTOR
};

// private API /////////////////////////////////////////////////////////////////

// internal GObject stuff //////////////////////////////////////////////////////
static void
ctk_menu_expandable_finalize (GObject* object)
{
  G_OBJECT_CLASS (ctk_menu_expandable_parent_class)->finalize (object);

  CtkMenuExpandable*        menu = CTK_MENU_EXPANDABLE (object);
  CtkMenuExpandablePrivate* priv = CTK_MENU_EXPANDABLE_GET_PRIVATE (menu);
  g_return_if_fail (priv);


  if (priv->bg_fill_image_id != 0)
    {
      CHECKGL (glDeleteTextures (1, &priv->bg_fill_image_id));
    }

  gint i = 0;
  for (i = 0; i < CTK_MENU_EXPANDABLE_TRANSITION_STEPS; i++)
    {
      CHECKGL (glDeleteTextures (1, &priv->transition_array[i].texture_id));
      CHECKGL (glDeleteTextures (1, &priv->transition_array[i].mask_id));
    }

  CHECKGL (glDeleteTextures (1, &priv->full_texture_id));
  CHECKGL (glDeleteTextures (1, &priv->full_mask_id));
  CHECKGL (glDeleteTextures (1, &priv->blurred_bkg_id));  
}

static void
ctk_menu_expandable_init (CtkMenuExpandable* menu)
{
  CtkMenuExpandablePrivate *priv;
  priv = menu->priv = CTK_MENU_EXPANDABLE_GET_PRIVATE (menu);

  priv->bg_fill_image_id     = 0;
  priv->bg_fill_image_width  = 0;
  priv->bg_fill_image_height = 0;
  priv->expansion_size_factor = 1.0f;
  
  gint i = 0;
  for (i = 0; i < CTK_MENU_EXPANDABLE_TRANSITION_STEPS; i++)
    {
      priv->transition_array[i].texture_id = 0;
      priv->transition_array[i].mask_id = 0;
      priv->transition_array[i].width = 0;
      priv->transition_array[i].height = 0;
    }

  priv->full_texture_id = 0;
  priv->full_mask_id = 0;
  priv->full_width = 0;
  priv->full_height = 0;  
  
  priv->AnchorPositionX = 0;
  priv->AnchorPositionY = 0;
  
  priv->center_position_in_menu = 0;
  
  priv->upper_size = 0;
  priv->lower_size = 0;
  
  priv->Padding = CTK_MENU_EXPANDABLE_PADDING;
  priv->ContentPadding = CTK_MENU_EXPANDABLE_CONTENT_PADDING;
  priv->ContentPaddingLeftRight = CTK_MENU_EXPANDABLE_CONTENT_PADDING;
  
  priv->menu_rendering_width = 0;
  priv->menu_rendering_height = 0;
  
  priv->blurred_bkg_id = 0;
  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  CHECKGL (glGenTextures (1, &priv->blurred_bkg_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, priv->blurred_bkg_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA8, 1, 1, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));
  
  priv->blurred_bkg_width = 0;
  priv->blurred_bkg_height = 0;
  
  ctk_menu_set_is_expandable (CTK_MENU (menu), TRUE);
  
}

static void
ctk_menu_expandable_blurred_bkg (CtkMenuExpandable *menu, gint x, gint y, gint width, gint height)
{
  ClutterActor* stage         = NULL;
  gfloat stage_width          = 0.0f;
  gfloat stage_height         = 0.0f;  
  
  stage = clutter_actor_get_stage (CLUTTER_ACTOR (menu));
  clutter_actor_get_size (CLUTTER_ACTOR (stage), &stage_width, &stage_height);

  CHECKGL( glPixelStorei (GL_PACK_ALIGNMENT, 1));
  CHECKGL( glPixelStorei (GL_PACK_ROW_LENGTH, 0));
  CHECKGL( glPixelStorei (GL_PACK_IMAGE_HEIGHT, 0));
  CHECKGL( glPixelStorei (GL_PACK_SKIP_PIXELS, 0));
  CHECKGL( glPixelStorei (GL_PACK_SKIP_ROWS, 0));

  /* It does not seem rigth to use floating point values to compute a memory allocation size! */
  char* pixels = g_malloc(width * height * 4);
  if (x < 0)
    x = 0;
  if (y < 0)
    y = 0;
  
  glReadPixels(x, stage_height - y - height, width, height, GL_RGBA, GL_UNSIGNED_BYTE, pixels);

  CHECKGL (glPixelStorei (GL_UNPACK_ALIGNMENT, 1));
  CHECKGL (glPixelStorei (GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_SKIP_ROWS, 0));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, menu->priv->blurred_bkg_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA8, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels));
  menu->priv->blurred_bkg_width = width;
  menu->priv->blurred_bkg_height = height;

  CHECKGL (glPixelStorei (GL_UNPACK_ALIGNMENT, 1));
  CHECKGL (glPixelStorei (GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei (GL_UNPACK_SKIP_ROWS, 0));
  
  g_free(pixels);
  
  // Disable Scissoring
  CHECKGL (glDisable (GL_SCISSOR_TEST));

  CtkEffectContext* fxctx = ctk_effect_context_get_default_for_actor (CLUTTER_ACTOR (menu));

  /* Get the current render target */
  CtkRenderTarget* top_rt = ctk_effect_context_peek_render_target (fxctx);

  /* reserve 2 render targets */
  CtkRenderTarget* rt0 = ctk_effect_context_grab_render_target (fxctx);
  CtkRenderTarget* rt1 = ctk_effect_context_grab_render_target (fxctx);

  /* resize the render targets */
  ctk_render_target_resize (rt0, width, height);
  ctk_render_target_resize (rt1, width, height);

  ctk_render_target_bind (rt0);
  CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT) );

  ctk_render_quad_asm(menu->priv->blurred_bkg_id, width, height, g_shTexture_asm,
                      width, height, 0, 0, width, height,
                      1.0f, 1.0f, 1.0f, 1.0f);

  /* Blur the render targets */
  int iPass = 0;
  for (iPass = 0; iPass < 4; iPass++)
    {
      ctk_render_target_bind(rt1);
      CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT) );
      opengl_blur_pass (ctk_render_target_get_color_buffer_ogl_id(rt0),
        ctk_render_target_get_width(rt0),
        ctk_render_target_get_height(rt0),
        0.0f,
        TRUE,
        width,
        height,
        TRUE);

      ctk_render_target_bind(rt0);
      CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT) );
      opengl_blur_pass (ctk_render_target_get_color_buffer_ogl_id(rt1),
        ctk_render_target_get_width(rt1),
        ctk_render_target_get_height(rt1),
        0.0f,
        FALSE,
        width,
        height,
        FALSE);
    }

  /* Copy the blurred texture back to priv->blurred_background_texture */
  ctk_copy_render_target_to_cached_texture_asm(fxctx, rt0, menu->priv->blurred_bkg_id);

  /* Release the render target we use */
  ctk_effect_context_release_render_target (fxctx, rt0);
  ctk_effect_context_release_render_target (fxctx, rt1);

  /* Restore the previous render target */
  if(top_rt)
    {
      ctk_render_target_bind(top_rt);
      CHECKGL( glViewport(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
      //CHECKGL( glScissor(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
    }
  else
    {
      /* There is no render target on the stack. Set back the regular frame buffer */
      ctk_render_target_unbind(); /* unbind whatever render target is binded */
      CHECKGL( glViewport(0, 0, stage_width, stage_height) );
      //CHECKGL( glScissor(0, 0, stage_width, stage_height) );
    }    
}

static void
ctk_menu_expandable_paint (ClutterActor* actor)
{
  CtkActor*                 ctk_actor = CTK_ACTOR (actor);
  CtkMenuExpandable*        menu      = CTK_MENU_EXPANDABLE (actor);
  CtkMenuExpandablePrivate* priv      = menu->priv;
  GSList*                   effects   = NULL;
  ClutterActorBox           box;
 
  cogl_flush ();

  // draw the blurred background
  // to be re-implemented

  ClutterActor* stage        = NULL;
  gfloat        stage_width  = 0.0f;
  gfloat        stage_height = 0.0f;
  gfloat        actor_x;
  gfloat        actor_y;
  gfloat        actor_w;
  gfloat        actor_h;
  gint          w_int;
  gint          h_int;
  ClutterVertex actor_vtx[4];
  gint          new_y;
  gfloat        menu_width;
  gfloat        menu_height;
  
  clutter_actor_get_size (CLUTTER_ACTOR (menu), &menu_width, &menu_height);
  
  if ((menu->priv->menu_rendering_width != menu_width) || (menu->priv->menu_rendering_height != menu_height))
  {
    // recompute the textures to the correct size.
    ctk_menu_expandable_compute_style_textures (menu);
  }
    
  stage = clutter_actor_get_stage (actor);

  clutter_actor_get_size (CLUTTER_ACTOR (stage), &stage_width, &stage_height);
  
  ctk_get_actor_screen_position (CTK_ACTOR (actor),
                                 &actor_x,
                                 &actor_y,
                                 &actor_w,
                                 &actor_h,
                                 actor_vtx);
  actor_y -= 1.0f;
  new_y = (gint) actor_y;

  w_int = actor_w + 1;
  h_int = actor_h + 1;

  gint upper_size = priv->upper_size*priv->expansion_size_factor;
  gint lower_size = priv->lower_size*priv->expansion_size_factor;
  gfloat opacity = 0.8f; //0.5f * (float) clutter_actor_get_paint_opacity (CLUTTER_ACTOR (actor)) / 255.0f;

  guint index = CTK_MENU_EXPANDABLE_TRANSITION_STEPS * priv->expansion_size_factor;
  if(index >= CTK_MENU_EXPANDABLE_TRANSITION_STEPS)
    index = CTK_MENU_EXPANDABLE_TRANSITION_STEPS - 1;

  gint x = priv->AnchorPositionX - ctk_menu_expandable_get_padding (menu);
  gint y = priv->AnchorPositionY - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2 - upper_size - CTK_MENU_EXPANDABLE_CORNER_RADIUS - ctk_menu_expandable_get_padding (menu);

  float v0, v1;

  gint Threshold = CTK_MENU_EXPANDABLE_TRANSITION_STEPS;

  ctk_menu_expandable_blurred_bkg (menu, x, y,
                                   actor_w + 2*ctk_menu_expandable_get_padding (menu),
                                   actor_h + 2*ctk_menu_expandable_get_corner_radius (menu) + 2*ctk_menu_expandable_get_padding (menu));

  //////////////////////////////////////////////////////////////////////////////////////
  // Top Part
  if(upper_size < Threshold)
    {
      index = upper_size;

      ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                              priv->transition_array[index].mask_id,
                              x,
                              y + ctk_menu_expandable_get_padding (menu),
                              priv->transition_array[index].width,
                              priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu),
                              0.0f, 
                              (float)ctk_menu_expandable_get_padding (menu)/(float)priv->transition_array[index].height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu))/(float)priv->transition_array[index].height,
                              0.0f,
                              (float)(ctk_menu_expandable_get_padding (menu))/(float) priv->blurred_bkg_height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu)/2.0f)/(float) priv->blurred_bkg_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              1.0f);

      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                              priv->transition_array[index].mask_id,
                              x,
                              y + ctk_menu_expandable_get_padding (menu),
                              priv->transition_array[index].width,
                              priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu),
                              0.0f, 
                              (float)ctk_menu_expandable_get_padding (menu)/(float)priv->transition_array[index].height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu))/(float)priv->transition_array[index].height,
                              0.0f,
                              0.0f,
                              1.0f,
                              (float)(priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu))/(float) priv->bg_fill_image_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              opacity);
        
      // Render the top half of a full texture. Cut horizontally at the tip of the anchor
      ctk_render_texture_part (priv->transition_array[index].texture_id,
                        x,
                        y,
                        priv->transition_array[index].width,
                        priv->transition_array[index].height/2.0f,
                        0.0f, 
                        0.0f,
                        1.0f,
                        0.5f,
                        (gint) stage_width,
                        (gint) stage_height);

                              
    }
  else
    {
      gint middle_size = upper_size - Threshold;
      
      ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                              priv->full_mask_id,
                              x,
                              y + ctk_menu_expandable_get_padding (menu),
                              priv->full_width,
                              ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f,
                              0.0f, 
                              (float)ctk_menu_expandable_get_padding (menu)/(float)priv->full_height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float)priv->full_height,
                              0.0f,
                              (float)(ctk_menu_expandable_get_padding (menu))/(float) priv->blurred_bkg_height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f)/(float) priv->blurred_bkg_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              1.0);
                              
      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                              priv->full_mask_id,
                              x,
                              y + ctk_menu_expandable_get_padding (menu),
                              priv->full_width,
                              ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f,
                              0.0f, 
                              (float)ctk_menu_expandable_get_padding (menu)/(float)priv->full_height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float)priv->full_height,
                              0.0f,
                              0.0f,
                              1.0f,
                              (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f) / (float) priv->bg_fill_image_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              opacity);

      
      v0 = 0.0f;
      v1 = (ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/priv->full_height;
      // Round corner part + uppersize/2
      ctk_render_texture_part (priv->full_texture_id,
                        x,
                        y,
                        priv->full_width,
                        (ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f),
                        0.0f,
                        v0,
                        1.0f,
                        v1,
                        (gint) stage_width,
                        (gint) stage_height);

      // middle part of first half
      if(middle_size > 0)
        {
          ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                                priv->full_mask_id,
                                x,
                                y + ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                                priv->full_width,
                                middle_size,
                                0.0f, 
                                (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float)priv->full_height,
                                1.0f,
                                (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f + middle_size)/(float)priv->full_height,
                                0.0f,
                                (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f)/(float) priv->blurred_bkg_height,
                                1.0f,
                                (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f  + middle_size)/(float) priv->blurred_bkg_height,
                                (gint) stage_width,
                                (gint) stage_height,
                                1.0);
                                
          ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                                priv->full_mask_id,
                                x,
                                y + ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                                priv->full_width,
                                middle_size,
                                0.0f, 
                                (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float)priv->full_height,
                                1.0f,
                                (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f + middle_size)/(float)priv->full_height,
                                0.0f,
                                (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float) priv->bg_fill_image_height,
                                1.0f,
                                (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f + middle_size)/(float) priv->bg_fill_image_height,
                                (gint) stage_width,
                                (gint) stage_height,
                                opacity);
                                
          y += (ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2);
          v0 = (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2)/(float)priv->full_height;
          v1 = (float)(ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2 + middle_size)/(float)priv->full_height;
          
          ctk_render_texture_part (priv->full_texture_id,
                                  x,
                                  y,
                                  priv->full_width,
                                  middle_size,
                                  0.0f,
                                  v0,
                                  1.0f,
                                  v1,
                                  (gint) stage_width,
                                  (gint) stage_height);
        }
      

      ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                              priv->full_mask_id,
                              x,
                              middle_size ? y + middle_size : y + ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                              priv->full_width,
                              (float)CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + (float)Threshold/2.0f,
                              0.0f, 
                              (float)(priv->full_height / 2.0f - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f - Threshold/2.0f)/(float)priv->full_height,
                              1.0f,
                              (float)(priv->full_height / 2.0f)/priv->full_height,
                              0.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + Threshold/2.0f  + middle_size)/(float) priv->blurred_bkg_height,
                              1.0f,
                              (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + Threshold + middle_size + ctk_menu_expandable_get_anchor_height (menu)/2.0f)/(float) priv->blurred_bkg_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              1.0);
                              
      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                              priv->full_mask_id,
                              x,
                              middle_size ? y + middle_size : y + ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                              priv->full_width,
                              (float)CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + (float)Threshold/2.0f,
                              0.0f, 
                              (float)(priv->full_height / 2.0f - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f - Threshold/2.0f)/(float)priv->full_height,
                              1.0f,
                              (float)(priv->full_height / 2.0f)/priv->full_height,
                              0.0f,
                              (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f + middle_size)/(float) priv->bg_fill_image_height,
                              1.0f,
                              (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f + middle_size + (float)CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + (float)Threshold/2.0f)/(float) priv->bg_fill_image_height,
                              (gint) stage_width,
                              (gint) stage_height,
                              opacity);
                              
                              
      if(middle_size)
        y += middle_size;
      else
        y += (ctk_menu_expandable_get_padding (menu) + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f);

      v0 = (float)(priv->full_height / 2.0f - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f - Threshold/2.0f)/(float)priv->full_height;
      v1 = (float)(0.5f);
      
      // bottom part of first half
      ctk_render_texture_part (priv->full_texture_id,
                                x,
                                y,
                                priv->full_width,
                                (float)CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + (float)Threshold/2.0f,
                                0.0f,
                                v0,
                                1.0f,
                                v1,
                                (gint) stage_width,
                                (gint) stage_height);
      
    }
    
  x = priv->AnchorPositionX - ctk_menu_expandable_get_padding (menu);
  y = priv->AnchorPositionY;

  //////////////////////////////////////////////////////////////////////////////////////
  // Bottom Part
  if(lower_size < Threshold)
    {
      index = lower_size;

      ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                      priv->transition_array[index].mask_id,
                      x,
                      y,
                      priv->transition_array[index].width,
                      priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu),
                      0.0f, 
                      0.5f, 
                      1.0f,
                      (float)((float)priv->transition_array[index].height - ctk_menu_expandable_get_padding (menu))/(float)priv->transition_array[index].height,
                      0.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu)/2.0f)/(float) priv->blurred_bkg_height,
                      1.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + 2*ctk_menu_expandable_get_corner_radius (menu) + upper_size + lower_size + ctk_menu_expandable_get_anchor_height (menu))/(float) priv->blurred_bkg_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      1.0f);
                      
      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                      priv->transition_array[index].mask_id,
                      x,
                      y,
                      priv->transition_array[index].width,
                      priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu),
                      0.0f, 
                      0.5f, 
                      1.0f,
                      (float)((float)priv->transition_array[index].height - ctk_menu_expandable_get_padding (menu))/(float)priv->transition_array[index].height,
                      0.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f)/(float) priv->bg_fill_image_height,
                      1.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + (priv->transition_array[index].height/2.0f - ctk_menu_expandable_get_padding (menu)))/(float) priv->bg_fill_image_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      opacity);
                              
      // Render the bottom half of a full texture. Cut horizontally at the tip of the anchor
      ctk_render_texture_part (priv->transition_array[index].texture_id,
                        x,
                        y,
                        priv->transition_array[index].width,
                        priv->transition_array[index].height/2.0f,
                        0.0f, 
                        0.5f,
                        1.0f,
                        1.0f,
                        (gint) stage_width,
                        (gint) stage_height);
                              
                              
    }
  else
    {
      gint middle_size = lower_size - Threshold;

      ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                      priv->full_mask_id,
                      x,
                      y,
                      priv->full_width,
                      CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f,
                      0.0f, 
                      0.5f, 
                      1.0f,
                      (float)((float)priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/(float)priv->full_height,
                      0.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu)/2.0f)/(float) priv->blurred_bkg_height,
                      1.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu) + Threshold/2.0f)/(float) priv->blurred_bkg_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      1.0f);
                      
      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                      priv->full_mask_id,
                      x,
                      y,
                      priv->full_width,
                      CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f,
                      0.0f, 
                      0.5f, 
                      1.0f,
                      (float)((float)priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/(float)priv->full_height,
                      0.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f)/(float) priv->bg_fill_image_height,
                      1.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT + Threshold/2.0f)/(float) priv->bg_fill_image_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      opacity);
      
      float v0 = 0.5f;
      float v1 = (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/priv->full_height;
      // Anchor part
      ctk_render_texture_part (priv->full_texture_id,
                        x,
                        y,
                        priv->full_width,
                        (CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f),
                        0.0f,
                        v0,
                        1.0f,
                        v1,
                        (gint) stage_width,
                        (gint) stage_height);

      // middle part of second half
      if(middle_size > 0)
        {
          
          y += (CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f);

          ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                          priv->full_mask_id,
                          x,
                          y,
                          priv->full_width,
                          middle_size,
                          0.0f, 
                          (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/(float)priv->full_height, 
                          1.0f,
                          (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f + middle_size)/(float)priv->full_height,
                          0.0f,
                          (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu) + Threshold/2.0f)/(float) priv->blurred_bkg_height,
                          1.0f,
                          (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu) + Threshold/2.0f + middle_size)/(float) priv->blurred_bkg_height,
                          (gint) stage_width,
                          (gint) stage_height,
                          1.0f);
                          
          ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                          priv->full_mask_id,
                          x,
                          y,
                          priv->full_width,
                          middle_size,
                          0.0f, 
                          (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/(float)priv->full_height, 
                          1.0f,
                          (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f + middle_size)/(float)priv->full_height,
                          0.0f,
                          (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT + Threshold/2.0f)/(float) priv->bg_fill_image_height,
                          1.0f,
                          (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT + Threshold/2.0f + middle_size)/(float) priv->bg_fill_image_height,
                          (gint) stage_width,
                          (gint) stage_height,
                          opacity);
                          
          v0 = (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f)/priv->full_height;
          v1 = (float)(priv->full_height/2.0f + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f + middle_size)/(float)priv->full_height;

          ctk_render_texture_part (priv->full_texture_id,
                                  x,
                                  y,
                                  priv->full_width,
                                  middle_size,
                                  0.0f,
                                  v0,
                                  1.0f,
                                  v1,
                                  (gint) stage_width,
                                  (gint) stage_height);
        }
 
       ctk_render_texture_part_with_mask (priv->blurred_bkg_id,
                      priv->full_mask_id,
                      x,
                      middle_size ? y + middle_size : y + (CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f),
                      priv->full_width,
                      CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                      0.0f, 
                      (float)(priv->full_height - ctk_menu_expandable_get_padding (menu) - CTK_MENU_EXPANDABLE_CORNER_RADIUS - Threshold/2.0f)/(float)priv->full_height, 
                      1.0f,
                      (float)(priv->full_height - ctk_menu_expandable_get_padding (menu))/(float)priv->full_height,
                      0.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu) + Threshold/2.0f + middle_size)/(float) priv->blurred_bkg_height,
                      1.0f,
                      (float)(ctk_menu_expandable_get_padding (menu) + ctk_menu_expandable_get_corner_radius (menu) + upper_size + ctk_menu_expandable_get_anchor_height (menu) + lower_size + ctk_menu_expandable_get_corner_radius (menu))/(float) priv->blurred_bkg_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      1.0f);

      ctk_render_texture_part_with_mask (priv->bg_fill_image_id,
                      priv->full_mask_id,
                      x,
                      middle_size ? y + middle_size : y + (CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f),
                      priv->full_width,
                      CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f,
                      0.0f, 
                      (float)(priv->full_height - ctk_menu_expandable_get_padding (menu) - CTK_MENU_EXPANDABLE_CORNER_RADIUS - Threshold/2.0f)/(float)priv->full_height, 
                      1.0f,
                      (float)(priv->full_height - ctk_menu_expandable_get_padding (menu))/(float)priv->full_height,
                      0.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT + Threshold/2.0f + middle_size)/(float) priv->bg_fill_image_height,
                      1.0f,
                      (float)(CTK_MENU_EXPANDABLE_CORNER_RADIUS + upper_size + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT + Threshold/2.0f + middle_size + CTK_MENU_EXPANDABLE_CORNER_RADIUS + Threshold/2.0f)/(float) priv->bg_fill_image_height,
                      (gint) stage_width,
                      (gint) stage_height,
                      opacity);
                      
                      
      if(middle_size)
        y += middle_size;
      else
        y += (CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + Threshold/2.0f);

      v0 = (float)(priv->full_height - ctk_menu_expandable_get_padding (menu) - CTK_MENU_EXPANDABLE_CORNER_RADIUS - Threshold/2.0f)/(float)priv->full_height;
      v1 = (float)(1.0f);
      
      // bottom part of first half
      ctk_render_texture_part (priv->full_texture_id,
                                x,
                                y,
                                priv->full_width,
                                ((float)Threshold/2.0f + CTK_MENU_EXPANDABLE_CORNER_RADIUS + ctk_menu_expandable_get_padding (menu)),
                                0.0f,
                                v0,
                                1.0f,
                                v1,
                                (gint) stage_width,
                                (gint) stage_height);
      
    }

  gint scissor_box [4];
  glGetIntegerv (GL_SCISSOR_BOX, scissor_box);
  
  glEnable (GL_SCISSOR_TEST);
  int sc_x = priv->AnchorPositionX;
  int sc_y = stage_height - (priv->AnchorPositionY + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + lower_size);
  int sc_w = priv->AnchorPositionX + actor_w;
  int sc_h = stage_height - (priv->AnchorPositionY - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f - upper_size);
  
  if (sc_x < 0)
    sc_x = 0;
  if (sc_y < 0)
    sc_y = 0;
  if (sc_w < 0)
    sc_w = 0;
  if (sc_h < 0)
    sc_h = 0;
      

  glScissor (sc_x, sc_y, sc_w, sc_h);

//   g_printf ("scissor box: [%f, %f, %f, %f] \n", 
//           (float)priv->AnchorPositionX,
//           stage_height - (priv->AnchorPositionY + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f + lower_size),
//           priv->AnchorPositionX + actor_w,
//           stage_height - (priv->AnchorPositionY - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT/2.0f - upper_size));

  // draw effects and children
  if (!ctk_actor_get_effects_painting (ctk_actor)
      && (effects = ctk_actor_get_effects (ctk_actor)))
    {
      GSList *e;

      ctk_actor_set_effects_painting (ctk_actor, TRUE);

      for (e = effects; e; e = e->next)
        {
          gboolean last_effect = e->next ? FALSE : TRUE;
          ctk_effect_paint (e->data, ctk_menu_expandable_paint, last_effect);
        }

      ctk_actor_set_effects_painting (ctk_actor, FALSE);
    }
  else
    {
      GList* c = NULL;

      clutter_actor_get_allocation_box (actor, &box);

      /*if (priv->background != NULL)
        clutter_actor_paint (priv->background);*/

      // Now draw the children
      for (c = clutter_container_get_children (CLUTTER_CONTAINER (actor));
           c;
           c = c->next)
        {
          ClutterActor* child = c->data;

          if (CLUTTER_ACTOR_IS_VISIBLE (child))
            clutter_actor_paint (child);
        }
    }
    
  glDisable (GL_SCISSOR_TEST);
  glScissor (scissor_box[0], scissor_box[1], scissor_box[2], scissor_box[3]);
    
}

static void
ctk_menu_expandable_set_property (GObject*      object,
                                  guint         prop_id,
                                  const GValue* value,
                                  GParamSpec*   pspec)
{
  CtkMenuExpandable* menu = CTK_MENU_EXPANDABLE (object);

  switch (prop_id)
    {
      case PROP_BG_FILL_IMAGE_ID:
        ctk_menu_expandable_set_bg_fill_image_id (menu,
                                                  g_value_get_uint (value));
      break;

      case PROP_BG_FILL_IMAGE_WIDTH:
        ctk_menu_expandable_set_bg_fill_image_width (menu,
                                                     g_value_get_uint (value));
      break;

      case PROP_BG_FILL_IMAGE_HEIGHT:
        ctk_menu_expandable_set_bg_fill_image_height (menu,
                                                      g_value_get_uint (value));
      break;

      case PROP_EXPANSION_SIZE_FACTOR:
        ctk_menu_expandable_set_expansion_size_factor (menu, g_value_get_float (value));
      break;
      
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
ctk_menu_expandable_get_property (GObject*    object,
                                  guint       prop_id,
                                  GValue*     value,
                                  GParamSpec* pspec)
{
  CtkMenuExpandable* menu = CTK_MENU_EXPANDABLE (object);

  switch (prop_id)
    {

      case PROP_BG_FILL_IMAGE_ID:
        g_value_set_uint (value,
                          ctk_menu_expandable_get_bg_fill_image_id (menu));
      break;

      case PROP_BG_FILL_IMAGE_WIDTH:
        g_value_set_uint (value,
                          ctk_menu_expandable_get_bg_fill_image_width (menu));
      break;

      case PROP_BG_FILL_IMAGE_HEIGHT:
        g_value_set_uint (value,
                          ctk_menu_expandable_get_bg_fill_image_height (menu));
      break;

      case PROP_EXPANSION_SIZE_FACTOR:
        g_value_set_float (value, ctk_menu_expandable_get_expansion_size_factor (menu));
      break;

      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
ctk_menu_expandable_class_init (CtkMenuExpandableClass* klass)
{
  GObjectClass*      object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass* act_class    = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec*        pspec;

  object_class->finalize = ctk_menu_expandable_finalize;
  object_class->set_property = ctk_menu_expandable_set_property;
  object_class->get_property = ctk_menu_expandable_get_property;
  
  act_class->allocate     = ctk_menu_expandable_allocate;
    
  act_class->paint = ctk_menu_expandable_paint;

  pspec = g_param_spec_uint ("bg-fill-image-id",
                             "bg-fill-image-id",
                             "OpenGL texture-id of bg-fill-image",
                             0,
                             G_MAXUINT,
                             0,
                             G_PARAM_READWRITE);
  g_object_class_install_property (object_class, PROP_BG_FILL_IMAGE_ID, pspec);

  pspec = g_param_spec_uint ("bg-fill-image-width",
                             "bg-fill-image-width",
                             "Native width of OpenGL bg-fill-image",
                             0,
                             G_MAXUINT,
                             0,
                             G_PARAM_READWRITE);
  g_object_class_install_property (object_class,
                                   PROP_BG_FILL_IMAGE_WIDTH, pspec);

  pspec = g_param_spec_uint ("bg-fill-image-height",
                             "bg-fill-image-height",
                             "Native height of OpenGL bg-fill-image",
                             0,
                             G_MAXUINT,
                             0,
                             G_PARAM_READWRITE);
  g_object_class_install_property (object_class,
                                   PROP_BG_FILL_IMAGE_HEIGHT,
                                   pspec);

  pspec = g_param_spec_float ("expansion-size-factor",
                              "expansion-size-factor",
                              "Expansion Factor",
                              0.0f,
                              1.0f,
                              0.0f,
                              G_PARAM_READWRITE);
  g_object_class_install_property (object_class, PROP_EXPANSION_SIZE_FACTOR, pspec);
  
  g_type_class_add_private (klass, sizeof (CtkMenuExpandablePrivate));
}

static void
ctk_menu_expandable_allocate (ClutterActor           *actor,
                              const ClutterActorBox  *box,
                              ClutterAllocationFlags flags)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (actor));
  
  CLUTTER_ACTOR_CLASS (ctk_menu_expandable_parent_class)->allocate (actor,box, flags);
  
  CtkMenuExpandable* menu = CTK_MENU_EXPANDABLE (actor);
    
  gfloat menu_width, menu_height;
  clutter_actor_get_size (CLUTTER_ACTOR (menu), &menu_width, &menu_height);
  
  gint special_item_y_in_menu = ctk_menu_get_special_item_y (CTK_MENU (menu));
  gint special_item_height_in_menu = ctk_menu_get_special_item_height (CTK_MENU (menu));

  if (special_item_y_in_menu < ctk_menu_expandable_get_content_padding (menu))
    special_item_y_in_menu = ctk_menu_expandable_get_content_padding (menu);
  if (special_item_y_in_menu > menu_height - special_item_height_in_menu - ctk_menu_expandable_get_content_padding (menu))
    special_item_y_in_menu = menu_height - special_item_height_in_menu - ctk_menu_expandable_get_content_padding (menu);

  gint menu_x;
  gint menu_y;
  menu_x = menu->priv->AnchorPositionX;
  menu_y = menu->priv->AnchorPositionY - special_item_y_in_menu - special_item_height_in_menu/2.0f;
  
  gfloat stage_height = clutter_actor_get_height (clutter_actor_get_stage (actor));
  
  if (menu_y + menu_height + ctk_menu_expandable_get_corner_radius (menu) + ctk_menu_expandable_get_content_padding (menu) >= stage_height)
  {
    gfloat diff = (menu_y + menu_height + ctk_menu_expandable_get_corner_radius (menu) + ctk_menu_expandable_get_content_padding (menu)) - stage_height;
    menu_y -= diff;
    special_item_y_in_menu += diff;
  }

  if (menu_y - (int)ctk_menu_expandable_get_corner_radius (menu) - (int)ctk_menu_expandable_get_padding (menu) < 0)
  {
    gfloat diff = (menu_y - (int)ctk_menu_expandable_get_corner_radius (menu) - (int)ctk_menu_expandable_get_padding (menu));
    menu_y -= diff;
    special_item_y_in_menu += diff;
  }
  
  menu->priv->upper_size =  special_item_y_in_menu  + special_item_height_in_menu/2.0f - ctk_menu_expandable_get_anchor_height (menu)/2.0f;
  if (menu->priv->upper_size < 0)
  {
    menu->priv->upper_size = 0;
    menu_y = menu->priv->AnchorPositionY - special_item_y_in_menu - special_item_height_in_menu/2.0f;
  }
    
  menu->priv->lower_size = menu_y + menu_height - menu->priv->AnchorPositionY - ctk_menu_expandable_get_anchor_height (menu)/2.0f;
  if (menu->priv->lower_size < 0)
  {
    menu->priv->lower_size = 0;
    if (special_item_y_in_menu + special_item_height_in_menu <= menu_height)
      menu_y = menu->priv->AnchorPositionY - special_item_height_in_menu - special_item_height_in_menu;
    else
      menu_y = menu->priv->AnchorPositionY - menu_height;
  }
  
  menu->priv->center_position_in_menu = special_item_y_in_menu + special_item_height_in_menu/2.0f;
  
  ClutterActorBox *menu_box = clutter_actor_box_copy (box);
  menu_box->x1 = menu_x;
  menu_box->x2 = menu_x + menu_width;
  menu_box->y1 = menu_y;
  menu_box->y2 = menu_y + menu_height;
  
  // Why call allocate a second time? The first call was necessary to get the overall size (width and height)
  // of the menu, and also the location of the special item...
  // These information are required to properply set the position of the menu with regard to the anchor...
  // The second call is necessary because we have to set the position of the menu and of its content.
  // Setting the position of the menu by itself is not enough. That is why we call allocate a second time.
  CLUTTER_ACTOR_CLASS (ctk_menu_expandable_parent_class)->allocate (actor, menu_box, flags);
}

// public API //////////////////////////////////////////////////////////////////

/**
 * ctk_menu_expandable_new:
 *
 * Create a new expandable menu.
 *
 * Returns: a pointer to the created #CtkMenuExpandable
 */
CtkMenuExpandable*
ctk_menu_expandable_new (void)
{
  CtkMenuExpandable* menu;

  menu = g_object_new (CTK_TYPE_MENU_EXPANDABLE,
                       NULL);

  return menu;
}


/**
 * ctk_menu_expandable_get_bg_fill_image_id:
 * @self: a #CtkMenuExpandable
 *
 * Retrieve the OpenGL-texture-id of the bg-fill image.
 *
 * Returns: a #guint, the GL-id of the bg-fill image
 */
guint
ctk_menu_expandable_get_bg_fill_image_id (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0.0f);
  return self->priv->bg_fill_image_id;
}

/**
 * ctk_menu_expandable_get_bg_fill_image_width:
 * @self: a #CtkMenuExpandable
 *
 * Retrieve the native pixel width of the bg-fill image.
 *
 * Returns: a #guint, the native width (in pixels) of the bg-fill OpenGL-image
 */
guint
ctk_menu_expandable_get_bg_fill_image_width (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0.0f);
  return self->priv->bg_fill_image_width;
}

/**
 * ctk_menu_expandable_get_bg_fill_image_height:
 * @self: a #CtkMenuExpandable
 *
 * Retrieve the native pixel height of the bg-fill image.
 *
 * Returns: a #guint, the native height (in pixels) of the bg-fill OpenGL-image
 */
guint
ctk_menu_expandable_get_bg_fill_image_height (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0.0f);
  return self->priv->bg_fill_image_height;
}

/**
 * ctk_menu_expandable_set_bg_fill_image_id:
 * @self: a #CtkMenuExpandable
 * @texture_id: a #guint
 *
 * Set the bg-fill OpenGL-image-id of the menu's background.
 */
void
ctk_menu_expandable_set_bg_fill_image_id (CtkMenuExpandable* self,
                                          guint              image_id)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  self->priv->bg_fill_image_id = image_id;
}

/**
 * ctk_menu_expandable_set_bg_fill_image_width:
 * @self: a #CtkMenuExpandable
 * @width: a #guint
 *
 * Set the native width (in pixels) of bg-fill OpenGL-image.
 */
void
ctk_menu_expandable_set_bg_fill_image_width (CtkMenuExpandable* self,
                                             guint              width)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  self->priv->bg_fill_image_width = width;
}

/**
 * ctk_menu_expandable_set_bg_fill_image_height:
 * @self: a #CtkMenuExpandable
 * @height: a #guint
 *
 * Set the native height (in pixels) of bg-fill OpenGL-image.
 */
void
ctk_menu_expandable_set_bg_fill_image_height (CtkMenuExpandable* self,
                                              guint              height)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  self->priv->bg_fill_image_height = height;
}


/**
 * ctk_menu_expandable_set_bg_fill_image:
 * @self: a #CtkMenuExpandable
 * @surf: a #cairo_surface_t
 *
 * Set the bg-fill image of the menu's background. The passed in surface needs
 * to be an image-surface.
 */
void
ctk_menu_expandable_set_bg_fill_image (CtkMenuExpandable* self,
                                       cairo_surface_t*   surf)
{
  gint unpack;

  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  g_return_if_fail (cairo_surface_get_type (surf) == CAIRO_SURFACE_TYPE_IMAGE);

  self->priv->bg_fill_image_width  = (guint) cairo_image_surface_get_width (surf);
  self->priv->bg_fill_image_height = (guint) cairo_image_surface_get_height (surf);

  CHECKGL (glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack));
  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, 4));
  CHECKGL (glPixelStorei(GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_ROWS, 0));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  if (self->priv->bg_fill_image_id)
    CHECKGL (glDeleteTextures (1, &self->priv->bg_fill_image_id));
  CHECKGL (glGenTextures (1, &self->priv->bg_fill_image_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, self->priv->bg_fill_image_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_RGBA8,
                         (gint) self->priv->bg_fill_image_width,
                         (gint) self->priv->bg_fill_image_height,
                         0,
                         GL_RGBA,
                         GL_UNSIGNED_BYTE,
                         cairo_image_surface_get_data (surf)));

  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, unpack));

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));
}

/**
 * ctk_menu_expandable_set_expansion_size_factor:
 * @self: a #CtkMenuExpandable
 * @size: normalized #gfloat size (height) of the lower-texture
 *
 * Set the normalized size of the upper  and lower texture part. 0.0f means the style 
 * only shows the item that was chosen to be at the same y coord as the anchor.
 *
 */
void
ctk_menu_expandable_set_expansion_size_factor (CtkMenuExpandable* self, gfloat factor)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  self->priv->expansion_size_factor = factor;
  
  gfloat actor_width;
  gfloat actor_height;
  
  clutter_actor_get_size (CLUTTER_ACTOR (self), &actor_width, &actor_height);
  
  if ((self->priv->expansion_size_factor == 1.0f) && clutter_actor_has_clip (CLUTTER_ACTOR (self)))
    {
      clutter_actor_remove_clip (CLUTTER_ACTOR (self));
    }
  else
    {
      clutter_actor_set_clip (CLUTTER_ACTOR (self),
                         0.0f,
                         self->priv->center_position_in_menu - self->priv->upper_size*self->priv->expansion_size_factor - CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT / 2.0f,
                         actor_width,
                         (self->priv->upper_size + self->priv->lower_size)*self->priv->expansion_size_factor + CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT);
    }
}

/**
 * ctk_menu_expandable_get_expansion_size_factor:
 * @self: a #CtkMenuExpandable
 *
 * Retrieve the current normalized size of the upper and lower texture part.
 *
 * Returns: a #gfloat, the factor of the upper and lower texture part
 */
gfloat
ctk_menu_expandable_get_expansion_size_factor (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0.0f);
  return self->priv->expansion_size_factor;
}

/**
 * ctk_menu_expandable_set_transition_textures:
 * @self: a #CtkMenuExpandable
 * @index: an index
 * @surf: a #cairo_surface_t
 * @mask_surf: a #cairo_surface_t
 *
 * Set a transition texture (outline+drop shadow) and mask
 */
void
ctk_menu_expandable_set_transition_textures (CtkMenuExpandable* self,
                                            int index,
                                            cairo_surface_t*   surf,
                                            cairo_surface_t*   mask_surf)
{
  gint unpack;

  
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  g_return_if_fail (cairo_surface_get_type (surf) == CAIRO_SURFACE_TYPE_IMAGE);

  self->priv->transition_array[index].width  = (guint) cairo_image_surface_get_width (surf);
  self->priv->transition_array[index].height = (guint) cairo_image_surface_get_height (surf);

  CHECKGL (glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack));
  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, 4));
  CHECKGL (glPixelStorei(GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_ROWS, 0));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  if(self->priv->transition_array[index].texture_id)
    CHECKGL (glDeleteTextures (1, &self->priv->transition_array[index].texture_id));
  CHECKGL (glGenTextures (1, &self->priv->transition_array[index].texture_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, self->priv->transition_array[index].texture_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_RGBA8,
                         (gint) self->priv->transition_array[index].width,
                         (gint) self->priv->transition_array[index].height,
                         0,
                         GL_RGBA,
                         GL_UNSIGNED_BYTE,
                         cairo_image_surface_get_data (surf)));

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  if(self->priv->transition_array[index].mask_id)
    CHECKGL (glDeleteTextures (1, &self->priv->transition_array[index].mask_id));
  CHECKGL (glGenTextures (1, &self->priv->transition_array[index].mask_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, self->priv->transition_array[index].mask_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_ALPHA8,
                         (gint) self->priv->transition_array[index].width,
                         (gint) self->priv->transition_array[index].height,
                         0,
                         GL_ALPHA,
                         GL_UNSIGNED_BYTE,
                         cairo_image_surface_get_data (mask_surf)));

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  
  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, unpack));

}

/**
 * ctk_menu_expandable_set_full_textures:
 * @self: a #CtkMenuExpandable
 * @surf: a #cairo_surface_t
 * @mask_surf: a #cairo_surface_t
 *
 * Set a transition texture (outline+drop shadow) and mask
 */
void
ctk_menu_expandable_set_full_textures (CtkMenuExpandable* self,
                                       cairo_surface_t*   surf,
                                       cairo_surface_t*   mask_surf)
{
  gint unpack;
  
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  g_return_if_fail (cairo_surface_get_type (surf) == CAIRO_SURFACE_TYPE_IMAGE);

  self->priv->full_width  = (guint) cairo_image_surface_get_width (surf);
  self->priv->full_height = (guint) cairo_image_surface_get_height (surf);

  CHECKGL (glGetIntegerv(GL_UNPACK_ALIGNMENT, &unpack));
  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, 4));
  CHECKGL (glPixelStorei(GL_UNPACK_ROW_LENGTH, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_IMAGE_HEIGHT, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0));
  CHECKGL (glPixelStorei(GL_UNPACK_SKIP_ROWS, 0));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  if(self->priv->full_texture_id)
    CHECKGL (glDeleteTextures (1, &self->priv->full_texture_id));
  CHECKGL (glGenTextures (1, &self->priv->full_texture_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, self->priv->full_texture_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_RGBA8,
                         (gint) self->priv->full_width,
                         (gint) self->priv->full_height,
                         0,
                         GL_RGBA,
                         GL_UNSIGNED_BYTE,
                         cairo_image_surface_get_data (surf)));

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  CHECKGL (glActiveTextureARB (GL_TEXTURE0));
  if(self->priv->full_mask_id)
    CHECKGL (glDeleteTextures (1, &self->priv->full_mask_id));
  CHECKGL (glGenTextures (1, &self->priv->full_mask_id));
  CHECKGL (glBindTexture (GL_TEXTURE_2D, self->priv->full_mask_id));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D,
                         0,
                         GL_ALPHA8,
                         (gint) self->priv->full_width,
                         (gint) self->priv->full_height,
                         0,
                         GL_ALPHA,
                         GL_UNSIGNED_BYTE,
                         cairo_image_surface_get_data (mask_surf)));

  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  CHECKGL (glPixelStorei(GL_UNPACK_ALIGNMENT, unpack));
}

/**
 * ctk_menu_expandable_get_transition_steps:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_transition_steps (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return CTK_MENU_EXPANDABLE_TRANSITION_STEPS;
}

/**
 * ctk_menu_expandable_set_padding:
 * @self: a #CtkMenuExpandable
 * @padding: the size of the padding to apply for the area outside the outline
 *
 */
void
ctk_menu_expandable_set_padding (CtkMenuExpandable* self, gint padding)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  if (padding < 0)
    padding = 0;  
  self->priv->Padding = padding;
}

/**
 * ctk_menu_expandable_get_padding:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_padding (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return self->priv->Padding;
}

/**
 * ctk_menu_expandable_set_content_padding:
 * @self: a #CtkMenuExpandable
 * @padding: the size of the padding to apply to the menu itself
 *
 * Used internally
 */
void
ctk_menu_expandable_set_content_padding (CtkMenuExpandable* self, gint padding)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  if (padding <0)
    padding = 0;
  self->priv->ContentPadding = padding;
}

/**
 * ctk_menu_expandable_set_content_padding_left_right:
 * @self: a #CtkMenuExpandable
 * @padding: set the left and right padding to apply to the menu itself
 *
 * Used internally
 */
void
ctk_menu_expandable_set_content_padding_left_right (CtkMenuExpandable* self, gint padding)
{
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  if (padding < 0)
    padding = 0;
  self->priv->ContentPaddingLeftRight = padding;
}

/**
 * ctk_menu_expandable_get_content_padding_left_right:
 * @self: a #CtkMenuExpandable
 * @padding: get the left and right padding to apply to the menu itself
 *
 * Used internally
 */
gint
ctk_menu_expandable_get_content_padding_left_right (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return self->priv->ContentPaddingLeftRight;
}

/**
 * ctk_menu_expandable_get_content_padding:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
gint
ctk_menu_expandable_get_content_padding (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return self->priv->ContentPadding;  
}

/**
 * ctk_menu_expandable_get_anchor_size_offset:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_anchor_size_offset (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return CTK_MENU_EXPANDABLE_ANCHOR_SIZE_OFFSET;
}

/**
 * ctk_menu_expandable_get_anchor_width:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_anchor_width (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return CTK_MENU_EXPANDABLE_ANCHOR_WIDTH;
}

/**
 * ctk_menu_expandable_get_anchor_height:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_anchor_height (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return CTK_MENU_EXPANDABLE_ANCHOR_HEIGHT;
}

/**
 * ctk_menu_expandable_get_corner_radius:
 * @self: a #CtkMenuExpandable
 *
 * Used internally
 */
guint
ctk_menu_expandable_get_corner_radius (CtkMenuExpandable* self)
{
  g_return_val_if_fail (CTK_IS_MENU_EXPANDABLE (self), 0);
  return CTK_MENU_EXPANDABLE_CORNER_RADIUS;
}

/**
 * ctk_menu_expandable_set_anchor_position:
 * @self: a #CtkMenuExpandable
 * @x: position of the anchor
 * @y: position of the anchor 
 * @tooltip_y_in_menu: position of the tooltive (relative to the menu) that alligns with to position of the anchor
 *
 */
void
ctk_menu_expandable_set_anchor_position (CtkMenuExpandable* self, gfloat x, gfloat y, gint tooltip_y_in_menu)
{  
  g_return_if_fail (CTK_IS_MENU_EXPANDABLE (self));
  self->priv->AnchorPositionX = x;
  self->priv->AnchorPositionY = y;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Rendering

void
_setup (cairo_surface_t** surf,
        cairo_t**         cr,
        gboolean          outline,
        gint              width,
        gint              height,
        gboolean          negative)
{
  // create context
  if (outline)
    *surf = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, width, height);
  else
    *surf = cairo_image_surface_create (CAIRO_FORMAT_A8, width, height);
  *cr = cairo_create (*surf);

  // clear context
  cairo_scale (*cr, 1.0f, 1.0f);
  if (outline)
    {
      cairo_set_source_rgba (*cr, 0.0f, 0.0f, 0.0f, 0.0f);
      cairo_set_operator (*cr, CAIRO_OPERATOR_CLEAR);
    }  
  else
    {
      cairo_set_operator (*cr, CAIRO_OPERATOR_OVER);
      if (negative)
        cairo_set_source_rgba (*cr, 0.0f, 0.0f, 0.0f, 0.0f);
      else
        cairo_set_source_rgba (*cr, 1.0f, 1.0f, 1.0f, 1.0f);
    }
  cairo_paint (*cr);
}

void
_draw (cairo_t* cr,
       gboolean outline,
       gfloat   line_width,
       gfloat*  rgba,
       gboolean negative,
       gboolean stroke)
{
  // prepare drawing
  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

  // actually draw the mask
  if (outline)
    {
      cairo_set_line_width (cr, line_width);
      cairo_set_source_rgba (cr, rgba[0], rgba[1], rgba[2], rgba[3]);
    }
  else
    {
      if (negative)
        cairo_set_source_rgba (cr, 1.0f, 1.0f, 1.0f, 1.0f);
      else
        cairo_set_source_rgba (cr, 0.0f, 0.0f, 0.0f, 0.0f);
    }

  // stroke or fill?
  if (stroke)
    cairo_stroke_preserve (cr);
  else
    cairo_fill_preserve (cr);
}

void
_finalize (cairo_t** cr,
           gboolean  outline,
           gfloat    line_width,
           gfloat*   rgba,
           gboolean  negative,
           gboolean  stroke)
{
  // prepare drawing
  cairo_set_operator (*cr, CAIRO_OPERATOR_SOURCE);

  // actually draw the mask
  if (outline)
    {
      cairo_set_line_width (*cr, line_width);
      cairo_set_source_rgba (*cr, rgba[0], rgba[1], rgba[2], rgba[3]);
    }
  else
    {
      if (negative)
        cairo_set_source_rgba (*cr, 1.0f, 1.0f, 1.0f, 1.0f);
      else
        cairo_set_source_rgba (*cr, 0.0f, 0.0f, 0.0f, 0.0f);
    }

  // stroke or fill?
  if (stroke)
    cairo_stroke (*cr);
  else
    cairo_fill (*cr);

  // clean up
  cairo_destroy (*cr);
}

void
_compute_full_mask_path (cairo_t* cr,
                      gfloat   anchor_width,
                      gfloat   anchor_height,
                      gint     width,
                      gint     height,
                      gint     upper_size,
                      gfloat   radius,
                      guint    pad)
{
//     0  1        2  3
//     +--+--------+--+
//     |              |
//     + 14           + 4
//     |              |
//     |              |
//     |              |
//     + 13           |
//    /               |
//   /                |
//  + 12              |
//   \                |
//    \               |
//  11 +              |
//     |              |
//     |              |
//     |              |
//  10 +              + 5
//     |              |
//     +--+--------+--+ 6
//     9  8        7
    
  
  gfloat padding  = pad;
  int ZEROPOINT5 = 0.0f;
  
  gfloat HeightToAnchor = 0.0f;
  HeightToAnchor = ((gfloat) height - 2.0f * radius - anchor_height -2*padding) / 2.0f;
  if (HeightToAnchor < 0.0f)
  {
    g_warning ("Anchor-height and corner-radius a higher than whole texture!");
    return;
  }

  //gint dynamic_size = height - 2*radius - 2*padding - anchor_height;
  //gint upper_dynamic_size = upper_size;
  //gint lower_dynamic_size = dynamic_size - upper_dynamic_size;
  
  if(upper_size >= 0)
    {
      if(upper_size > height - 2.0f * radius - anchor_height -2 * padding)
        {
          //g_warning ("[_compute_full_mask_path] incorrect upper_size value");
          HeightToAnchor = 0;
        }
      else
        {
          HeightToAnchor = height - 2.0f * radius - anchor_height -2 * padding - upper_size;
        }
    }
  else
    {
      HeightToAnchor = (height - 2.0f * radius - anchor_height -2*padding) / 2.0f;
    }
    
  cairo_translate (cr, -0.5f, -0.5f);
    
  // create path
  cairo_move_to (cr, padding + anchor_width + radius + ZEROPOINT5, padding + ZEROPOINT5); // Point 1
  cairo_line_to (cr, width - padding - radius, padding + ZEROPOINT5);   // Point 2
  cairo_arc (cr,
             width  - padding - radius + ZEROPOINT5,
             padding + radius + ZEROPOINT5,
             radius,
             -90.0f * G_PI / 180.0f,
             0.0f * G_PI / 180.0f);   // Point 4
  cairo_line_to (cr,
                 (gdouble) width - padding + ZEROPOINT5,
                 (gdouble) height - radius - padding + ZEROPOINT5); // Point 5
  cairo_arc (cr,
             (gdouble) width - padding - radius + ZEROPOINT5,
             (gdouble) height - padding - radius + ZEROPOINT5,
             radius,
             0.0f * G_PI / 180.0f,
             90.0f * G_PI / 180.0f);  // Point 7
  cairo_line_to (cr,
                 anchor_width + padding + radius + ZEROPOINT5,
                 (gdouble) height - padding + ZEROPOINT5); // Point 8

  cairo_arc (cr,
             anchor_width + padding + radius + ZEROPOINT5,
             (gdouble) height - padding - radius,
             radius,
             90.0f * G_PI / 180.0f,
             180.0f * G_PI / 180.0f); // Point 10

  cairo_line_to (cr,
                 padding + anchor_width + ZEROPOINT5,
                 (gdouble) height - padding - radius - HeightToAnchor + ZEROPOINT5 );  // Point 11
  cairo_line_to (cr,
                 padding + ZEROPOINT5,
                 (gdouble) height - padding - radius - HeightToAnchor - anchor_height / 2.0f + ZEROPOINT5); // Point 12
  cairo_line_to (cr,
                 padding + anchor_width + ZEROPOINT5,
                 (gdouble) height - padding - radius - HeightToAnchor - anchor_height + ZEROPOINT5);  // Point 13
  
  cairo_line_to (cr, padding + anchor_width + ZEROPOINT5, padding + radius  + ZEROPOINT5);  // Point 14
  cairo_arc (cr,
             padding + anchor_width + radius + ZEROPOINT5,
             padding + radius + ZEROPOINT5,
             radius,
             180.0f * G_PI / 180.0f,
             270.0f * G_PI / 180.0f);

  cairo_close_path (cr);
}

void
compute_mask (cairo_t* cr)
{
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_fill_preserve (cr);
}

void
compute_outline (cairo_t* cr,
          gfloat   line_width,
          gfloat*  rgba_line)
{
  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (cr,
                         rgba_line[0],
                         rgba_line[1],
                         rgba_line[2],
                         rgba_line[3]);
  cairo_set_line_width (cr, line_width);
  cairo_stroke (cr);
}

cairo_surface_t*
tint_dot_hl (gint    width,
             gint    height,
             gfloat  hl_x,
             gfloat  hl_y,
             gfloat  hl_size,
             gfloat* rgba_tint,
             gfloat* rgba_hl)
{
  cairo_surface_t* surf         = NULL;
  cairo_t*         cr           = NULL;
  cairo_surface_t* dots_surf    = NULL;
  cairo_t*         dots_cr      = NULL;
  cairo_pattern_t* dots_pattern = NULL;
  cairo_pattern_t* hl_pattern   = NULL;

  // create normal context
  surf = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, width, height);
  cr = cairo_create (surf);

  // create context for dot-pattern
  dots_surf = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 4, 4);
  dots_cr = cairo_create (dots_surf);

  // clear normal context
  cairo_scale (cr, 1.0f, 1.0f);
  cairo_set_source_rgba (cr, 0.0f, 0.0f, 0.0f, 0.0f);
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (cr);

  // prepare drawing for normal context
  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

  // create path in normal context
  cairo_rectangle (cr, 0.0f, 0.0f, (gdouble) width, (gdouble) height);  

  // fill path of normal context with tint
  cairo_set_source_rgba (cr,
                         rgba_tint[0],
                         rgba_tint[1],
                         rgba_tint[2],
                         rgba_tint[3]);
  cairo_fill_preserve (cr);

  // create pattern in dot-context
  cairo_set_operator (dots_cr, CAIRO_OPERATOR_CLEAR);
  cairo_paint (dots_cr);
  cairo_scale (dots_cr, 1.0f, 1.0f);
  cairo_set_operator (dots_cr, CAIRO_OPERATOR_OVER);
  cairo_set_source_rgba (dots_cr,
                         rgba_hl[0],
                         rgba_hl[1],
                         rgba_hl[2],
                         rgba_hl[3]);
  cairo_rectangle (dots_cr, 0.0f, 0.0f, 1.0f, 1.0f);
  cairo_fill (dots_cr);
  cairo_rectangle (dots_cr, 2.0f, 2.0f, 1.0f, 1.0f);
  cairo_fill (dots_cr);
  dots_pattern = cairo_pattern_create_for_surface (dots_surf);

  // fill path of normal context with dot-pattern
  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
  cairo_set_source (cr, dots_pattern);
  cairo_pattern_set_extend (dots_pattern, CAIRO_EXTEND_REPEAT);
  cairo_fill_preserve (cr);
  cairo_pattern_destroy (dots_pattern);
  cairo_surface_destroy (dots_surf);
  cairo_destroy (dots_cr);

  // draw highlight
  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
  hl_pattern = cairo_pattern_create_radial (hl_x,
                                            hl_y,
                                            0.0f,
                                            hl_x,
                                            hl_y,
                                            hl_size);
  cairo_pattern_add_color_stop_rgba (hl_pattern,
                                     0.0f,
                                     1.0f,
                                     1.0f,
                                     1.0f,
                                     0.65f);
  cairo_pattern_add_color_stop_rgba (hl_pattern, 1.0f, 1.0f, 1.0f, 1.0f, 0.0f);
  cairo_set_source (cr, hl_pattern);
  cairo_fill (cr);
  cairo_pattern_destroy (hl_pattern);

  cairo_destroy (cr);

  return surf;
}

cairo_surface_t*
compute_full_mask (gint     width,
                gint     height,
                gfloat   radius,
                guint    shadow_radius,
                gfloat   anchor_width,
                gfloat   anchor_height,
                gint     upper_size,
                gboolean negative,
                gboolean outline,
                gfloat   line_width,
                gint     padding_size,
                gfloat*  rgba)
{
  cairo_surface_t* surf = NULL;
  cairo_t*         cr   = NULL;

  _setup (&surf, &cr, outline, width, height, negative);
  _compute_full_mask_path (cr,
                        anchor_width,
                        anchor_height,
                        width,
                        height,
                        upper_size,
                        radius,
                        padding_size);
  _finalize (&cr, outline, line_width, rgba, negative, outline);

  return surf;

}

cairo_surface_t*
compute_full_outline_shadow (gint    width,
                          gint    height,
                          gfloat  anchor_width,
                          gfloat  anchor_height,
                          gint   upper_size,
                          gfloat  corner_radius,
                          guint   blur_coeff,
                          gfloat* rgba_shadow,
                          gfloat  line_width,
                          gint    padding_size,
                          gfloat* rgba_line)
{
  cairo_surface_t* surf = NULL;
  cairo_t*         cr   = NULL;

  _setup (&surf, &cr, TRUE, width, height, FALSE);
  _compute_full_mask_path (cr,
                        anchor_width,
                        anchor_height,
                        width,
                        height,
                        upper_size,
                        corner_radius,
                        padding_size);
  _draw (cr, TRUE, line_width, rgba_shadow, FALSE, FALSE);
  ctk_surface_blur (surf, blur_coeff);
  compute_mask (cr);
  compute_outline (cr, line_width, rgba_line);

  cairo_destroy (cr);

  return surf;
}

/**
 * ctk_menu_expandable_compute_cairo_textures:
 * @self: a #CtkMenuExpandable
 *
 * Compute the cairo textures required for the expandable menu 
 */
void
ctk_menu_expandable_compute_style_textures (CtkMenuExpandable* self)
{
  CtkPadding         padding;
  cairo_surface_t*   surf           = NULL;
  gfloat             rgba[4]        = {0.25f, 0.5f, 1.0f, 1.0f};
  gfloat             rgba_tint[4]   = {0.0f, 0.0f, 0.0f, 0.7f};
  gfloat             rgba_hl[4]     = {1.0f, 1.0f, 1.0f, 0.15f};
  gfloat             rgba_line[4]   = {1.0f, 1.0f, 1.0f, 0.75f};
  gfloat             rgba_shadow[4] = {0.0f, 0.0f, 0.0f, 0.48f};
  gfloat             tooltip_width;
  gfloat             tooltip_height;


  gint ANCHOR_WIDTH         = ctk_menu_expandable_get_anchor_width (self);
  gint ANCHOR_HEIGHT        = ctk_menu_expandable_get_anchor_height (self);
  gint RADIUS               = ctk_menu_expandable_get_corner_radius (self);
  gint BLUR_INTENSITY       = 8;
  gfloat LINE_WIDTH         = 1.0f;
  
  padding.top    = (gint) ctk_menu_expandable_get_content_padding (self);
  padding.right  = (gint) ctk_menu_expandable_get_content_padding_left_right (self);
  padding.bottom = (gint) ctk_menu_expandable_get_content_padding (self);
  padding.left   = (gint) (ANCHOR_WIDTH + ctk_menu_expandable_get_content_padding_left_right (self));
  ctk_actor_set_padding (CTK_ACTOR (self), &padding);

  
  clutter_actor_get_size (CLUTTER_ACTOR (self), &self->priv->menu_rendering_width, &self->priv->menu_rendering_height);

  gint PADDING_SIZE = ctk_menu_expandable_get_padding (self);

  tooltip_width = self->priv->menu_rendering_width; // The tooltip width is the full width of the expanded menu.
  tooltip_height = 2*RADIUS + (ANCHOR_HEIGHT) + 2*PADDING_SIZE;
 
  // tinted-dotted-highlight fill image
  surf = tint_dot_hl ((gint)self->priv->menu_rendering_width + 2*PADDING_SIZE,
                      (gint) self->priv->menu_rendering_height + 2*RADIUS + 2*PADDING_SIZE,
                      PADDING_SIZE + padding.left +(self->priv->menu_rendering_width - padding.left - padding.right) / 2.0f,
                      15.0f,
                      (self->priv->menu_rendering_width - padding.left - padding.right) / 2.0f,
                      rgba_tint,
                      rgba_hl);
  ctk_menu_expandable_set_bg_fill_image (self, surf);
  //cairo_surface_write_to_png (surf, "/tmp/tint-dot-hl-surf.png");
  cairo_surface_destroy (surf);

  guint i;
  //guint H = RADIUS + (ANCHOR_HEIGHT) + RADIUS;
  for (i = 0; i < ctk_menu_expandable_get_transition_steps (self); i++)
    {
      cairo_surface_t *texture_surf = compute_full_outline_shadow (2*PADDING_SIZE+tooltip_width,
                                                        tooltip_height + 2*i,
                                                        ANCHOR_WIDTH,
                                                        ANCHOR_HEIGHT,
                                                        -1,
                                                        RADIUS,
                                                        BLUR_INTENSITY,
                                                        rgba_shadow,
                                                        LINE_WIDTH,
                                                        PADDING_SIZE,
                                                        rgba_line);

      cairo_surface_t *mask_surf = compute_full_mask (2*PADDING_SIZE+tooltip_width,
                                              tooltip_height + 2*i,
                                              RADIUS,
                                              BLUR_INTENSITY,
                                              ANCHOR_WIDTH,
                                              ANCHOR_HEIGHT,
                                              -1,
                                              TRUE,
                                              FALSE,
                                              LINE_WIDTH,
                                              PADDING_SIZE,
                                              rgba);

      ctk_menu_expandable_set_transition_textures (self, i, texture_surf, mask_surf);
      cairo_surface_destroy (texture_surf);
      cairo_surface_destroy (mask_surf);
    }
    cairo_surface_t *texture_surf = compute_full_outline_shadow (2 * PADDING_SIZE + tooltip_width,
                                                      2*self->priv->menu_rendering_height + 2 * PADDING_SIZE,
                                                      ANCHOR_WIDTH,
                                                      ANCHOR_HEIGHT,
                                                      -1,
                                                      RADIUS,
                                                      BLUR_INTENSITY,
                                                      rgba_shadow,
                                                      LINE_WIDTH,
                                                      PADDING_SIZE,
                                                      rgba_line);

    cairo_surface_t *mask_surf = compute_full_mask (2 * PADDING_SIZE + tooltip_width,
                                            2*self->priv->menu_rendering_height + 2 * PADDING_SIZE,
                                            RADIUS,
                                            BLUR_INTENSITY,
                                            ANCHOR_WIDTH,
                                            ANCHOR_HEIGHT,
                                            -1,
                                            TRUE,
                                            FALSE,
                                            LINE_WIDTH,
                                            PADDING_SIZE,
                                            rgba);

    ctk_menu_expandable_set_full_textures (self, texture_surf, mask_surf);
    cairo_surface_destroy (texture_surf);
    cairo_surface_destroy (mask_surf);
}
