/*
*
* cobex_serial.c - Functions for serial comm for OBEX-over-cable
*
* Copyright (c) 2005 Fredrik Srensson
*
* History:
* v0.1 - fsn - 05-07-19 - First version
* Source:
*
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "cobex_serial.h"
#include "cobex_tools.h"


//
// Open port
//

v24_port_t* cobex_openPort(char *port) {
	int rc;
	v24_port_t *thePort;

	// Open Port

	thePort=v24OpenPort(port,V24_STANDARD||V24_LOCK);
	if (thePort==NULL) {
		thePort=v24OpenPort(port,V24_STANDARD);
		if (thePort==NULL) {
			fprintf(stderr,"ERROR: Could not open port %s.\n",port);
			exit (1);
		} else {
			fprintf(stderr,"WARNING: Could not create lockfile. Continuing unlocked.\n");
		}
	}

	// Set parameters

	rc=v24SetParameters(thePort,V24_B115200,V24_8BIT,V24_NONE);
	if ( rc!=V24_E_OK ) {
		fputs("ERROR: Setup of the port failed!\n",stderr);
		v24ClosePort(thePort);
		exit(1);
	}

	ctools_set_state_connected();
	
	return thePort;
}

//
// Close port, for sake of completeness
//

int cobex_closePort(v24_port_t *aPort) {
	int rc;
	rc = v24ClosePort(aPort);
	ctools_set_state_not_connected();
	return COBEX_OK;
}

//
// Functions related to send and recieve
//
// Send using libEZv24 

int cobex_packet_send ( obex_packet *inPacket, v24_port_t *port) {
	return (u_int)v24Write( port, inPacket->buffer, inPacket->l ) == inPacket->l ? COBEX_OK : COBEX_ERR ;
}

// Recieve. It should be rather foolproof. (send is SO much easier...)

int cobex_packet_recieve ( obex_packet *outPacket, v24_port_t *port) {
	int result;
	u_int inLgt = 0;

	outPacket->l = 0;

	if ( outPacket->max < 3 ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	while (v24HaveData(port) < 3) {
		usleep(NAPTIME);
	}

	result=v24Read( port, outPacket->buffer, 3);
	if (result == 0) { return COBEX_ERR_TIMEOUT; }
	if (result != 3 ) { return COBEX_ERR; }

	inLgt = (u_int)((outPacket->buffer[1])<<8);
	inLgt = inLgt | (u_int)(outPacket->buffer[2]);
	outPacket->l=inLgt;

	if (outPacket->max < inLgt ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	// Only a 3 byte messages, get out.
	if ((inLgt-3)==0) {return COBEX_OK; }

	while (v24HaveData(port) < ((int)inLgt-3)) {
		usleep(NAPTIME);
	}

	result=v24Read( port, outPacket->buffer+3, (inLgt-3) );
	if (result == 0) { return COBEX_ERR_TIMEOUT; }
	if (result != ((int)inLgt-3) ) { return COBEX_ERR; }

	return COBEX_OK;
}

// The timed version

int cobex_packet_recieve_t ( obex_packet *outPacket, v24_port_t *port, int timeout) {
	int result;
	int c=0;
	u_int inLgt = 0;

	outPacket->l = 0;

	if ( outPacket->max < 3 ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	while (v24HaveData(port) < 3) {
		//printf ("c: %d :",c);
		//fflush(stdout);
		if (c++>timeout) {return COBEX_ERR_TIMEOUT;}
		usleep(NAPTIME);
	}
	c=0;

	result=v24Read( port, outPacket->buffer, 3);
	if (result == 0) { return COBEX_ERR_TIMEOUT; }
	if (result != 3 ) { return COBEX_ERR; }

	inLgt = (u_int)((outPacket->buffer[1])<<8);
	inLgt = inLgt | (u_int)(outPacket->buffer[2]);
	outPacket->l=inLgt;

	if (outPacket->max < inLgt ) { return COBEX_ERR_BUFFER_OVERFLOW; }

	// Only a 3 byte messages, get out.
	if ((inLgt-3)==0) {return COBEX_OK; }

	while (v24HaveData(port) < ((int)inLgt-3)) {
		//printf ("c: %d ",c);
		if (c++>timeout) {return COBEX_ERR_TIMEOUT;}
		usleep(NAPTIME);
	}

	result=v24Read( port, outPacket->buffer+3, (inLgt-3) );
	if (result == 0) { return COBEX_ERR_TIMEOUT; }
	if (result != ((int)inLgt-3) ) { return COBEX_ERR; }

	return COBEX_OK;
}

// Wait for lines when doing AT commands.

int cobex_waitFor(v24_port_t *aPort, char *msg, int lgt, char *bmsg, int blgt) {
	char answer[80];
	u_int nrC;
	u_int i;

	if (lgt > 79) { return COBEX_ERR; } 

	do {
		nrC=v24Gets(aPort,answer,79);
		for (i=0; i<79; i++) {
			if (answer[i]<32) {answer[i]=0; break;}
		}
		if ( blgt > 0) {
			if (strncmp (answer, bmsg, blgt) ==0 ) return COBEX_ERR2;
		}
		if ( nrC == 0 ) return COBEX_ERR_TIMEOUT;
		if ( nrC < 0 ) {
			return COBEX_ERR;
		}
		//  else
			//printf("#%d: '%s'\n",nrC,answer);
	} while (strncmp ( answer, msg , lgt) != 0) ;
	return COBEX_OK;
}

int cobex_testComm( v24_port_t *aPort ) {
	int rc;
	
	rc=v24Puts(aPort,"\r\r\rATZ\r");
	if ( rc < 7 ) {
		return COBEX_ERR;
	} else {
		rc = cobex_waitFor(aPort,"OK",2,"NULL",0);
		if (!rc) { return rc;}
	}

	rc=v24Puts(aPort,"AT+CBC\r");
	if ( rc < 7 ) {
		return COBEX_ERR;
	} else {
		rc = cobex_waitFor(aPort,"OK",2,"NULL",0);
		if (!rc) { return rc;}
	}
	return COBEX_OK;
}

