/*
 * This file is part of Cockpit.
 *
 * Copyright (C) 2017 Red Hat, Inc.
 *
 * Cockpit is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * Cockpit is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Cockpit; If not, see <http://www.gnu.org/licenses/>.
 */
import '../lib/patternfly/patternfly-cockpit.scss';
import 'polyfills'; // once per application

import cockpit from "cockpit";
import React, { useState, useEffect } from "react";
import ReactDOM from 'react-dom';

import moment from "moment";
import {
    Alert, Button, Gallery, Modal, Progress, Popover, Tooltip,
    Card, CardTitle, CardActions, CardHeader, CardBody,
    DescriptionList, DescriptionListTerm, DescriptionListGroup, DescriptionListDescription,
    Flex, FlexItem,
    Page, PageSection, PageSectionVariants,
    Text, TextContent, TextListItem, TextList, TextVariants,
} from '@patternfly/react-core';
import {
    CheckIcon,
    ExclamationCircleIcon,
    ExclamationTriangleIcon,
    RebootingIcon,
    RedoIcon,
    ProcessAutomationIcon
} from "@patternfly/react-icons";
import { Remarkable } from "remarkable";

import { AutoUpdates, AutoUpdatesBody } from "./autoupdates.jsx";
import { History, PackageList } from "./history.jsx";
import { page_status } from "notifications";
import { EmptyStatePanel } from "cockpit-components-empty-state.jsx";
import { ListingTable } from 'cockpit-components-table.jsx';
import { ModalError } from 'cockpit-components-inline-notification.jsx';
import { ShutdownModal } from 'cockpit-components-shutdown.jsx';

import { superuser } from 'superuser';
import * as PK from "packagekit.js";

import * as python from "python.js";
import callTracerScript from 'raw-loader!./callTracer.py';

import "listing.scss";

const _ = cockpit.gettext;

// "available" heading is built dynamically
let STATE_HEADINGS = {};
let PK_STATUS_STRINGS = {};
let PK_STATUS_LOG_STRINGS = {};
const packageSummaries = {};

function init() {
    STATE_HEADINGS = {
        loading: _("Loading available updates, please wait..."),
        locked: _("Some other program is currently using the package manager, please wait..."),
        refreshing: _("Refreshing package information"),
        uptodate: _("System is up to date"),
        applying: _("Applying updates"),
        updateError: _("Applying updates failed"),
        loadError: _("Loading available updates failed"),
    };

    PK_STATUS_STRINGS = {
        [PK.Enum.STATUS_DOWNLOAD]: _("Downloading"),
        [PK.Enum.STATUS_INSTALL]: _("Installing"),
        [PK.Enum.STATUS_UPDATE]: _("Updating"),
        [PK.Enum.STATUS_CLEANUP]: _("Setting up"),
        [PK.Enum.STATUS_SIGCHECK]: _("Verifying"),
    };

    PK_STATUS_LOG_STRINGS = {
        [PK.Enum.STATUS_DOWNLOAD]: _("Downloaded"),
        [PK.Enum.STATUS_INSTALL]: _("Installed"),
        [PK.Enum.STATUS_UPDATE]: _("Updated"),
        [PK.Enum.STATUS_CLEANUP]: _("Set up"),
        [PK.Enum.STATUS_SIGCHECK]: _("Verified"),
    };
}

// parse CVEs from an arbitrary text (changelog) and return URL array
function parseCVEs(text) {
    if (!text)
        return [];

    var cves = text.match(/CVE-\d{4}-\d+/g);
    if (!cves)
        return [];
    return cves.map(n => "https://cve.mitre.org/cgi-bin/cvename.cgi?name=" + n);
}

function deduplicate(list) {
    var d = { };
    list.forEach(i => { if (i) d[i] = true; });
    var result = Object.keys(d);
    result.sort();
    return result;
}

// Insert comma strings in between elements of the list. Unlike list.join(",")
// this does not stringify the elements, which we need to keep as JSX objects.
function insertCommas(list) {
    if (list.length <= 1)
        return list;
    return list.reduce((prev, cur) => [prev, ", ", cur]);
}

// Fedora changelogs are a wild mix of enumerations or not, headings, etc.
// Remove that formatting to avoid an untidy updates overview list
function cleanupChangelogLine(text) {
    if (!text)
        return text;

    // enumerations
    text = text.replace(/^[-* ]*/, "");

    // headings
    text = text.replace(/^=+\s+/, "").replace(/=+\s*$/, "");

    return text.trim();
}

// Replace cockpit-wsinstance-https@[long_id] with a shorter string
function shortenCockpitWsInstance(list) {
    list = [...list];

    list.forEach((item, idx) => {
        if (item.startsWith("cockpit-wsinstance-https"))
            list[idx] = "cockpit-wsinstance-https@.";
    });

    return list;
}

const Expander = ({ title, onExpand, children }) => {
    const [expanded, setExpanded] = useState(false);

    useEffect(() => {
        if (expanded && onExpand)
            onExpand();
    }, [expanded, onExpand]);

    const cls = "expander-caret fa " + (expanded ? "fa-angle-down" : "fa-angle-right");
    return (
        <>
            <div className="expander-title">
                <hr />
                <Button variant="link" onClick={ () => setExpanded(!expanded) }>
                    <i className={cls} />{title}
                </Button>
                <hr />
            </div>
            {expanded ? children : null}
        </>);
};

function count_security_updates(updates) {
    var num_security = 0;
    for (const u in updates)
        if (updates[u].severity === PK.Enum.INFO_SECURITY)
            ++num_security;
    return num_security;
}

function find_highest_severity(updates) {
    var max = PK.Enum.INFO_LOW;
    for (const u in updates)
        if (updates[u].severity > max)
            max = updates[u].severity;
    return max;
}

function getSeverityURL(urls) {
    if (!urls)
        return null;

    // in ascending severity
    const knownLevels = ["low", "moderate", "important", "critical"];
    var highestIndex = -1;
    var highestURL = null;

    // search URLs for highest valid severity; by all means we expect an update to have at most one, but for paranoia..
    urls.map(value => {
        if (value.startsWith("https://access.redhat.com/security/updates/classification/#")) {
            const i = knownLevels.indexOf(value.slice(value.indexOf("#") + 1));
            if (i > highestIndex) {
                highestIndex = i;
                highestURL = value;
            }
        }
    });
    return highestURL;
}

function updateItem(info, pkgNames, key) {
    const remarkable = new Remarkable();
    let bugs = null;
    if (info.bug_urls && info.bug_urls.length) {
        // we assume a bug URL ends with a number; if not, show the complete URL
        bugs = insertCommas(info.bug_urls.map(url => (
            <a key={url} rel="noopener noreferrer" target="_blank" href={url}>
                {url.match(/[0-9]+$/) || url}
            </a>)
        ));
    }

    let cves = null;
    if (info.cve_urls && info.cve_urls.length) {
        cves = insertCommas(info.cve_urls.map(url => (
            <a key={url} href={url} rel="noopener noreferrer" target="_blank">
                {url.match(/[^/=]+$/)}
            </a>)
        ));
    }

    let errata = null;
    if (info.vendor_urls) {
        errata = insertCommas(info.vendor_urls.filter(url => url.indexOf("/errata/") > 0).map(url => (
            <a key={url} href={url} rel="noopener noreferrer" target="_blank">
                {url.match(/[^/=]+$/)}
            </a>)
        ));
        if (!errata.length)
            errata = null; // simpler testing below
    }

    let secSeverityURL = getSeverityURL(info.vendor_urls);
    const secSeverity = secSeverityURL ? secSeverityURL.slice(secSeverityURL.indexOf("#") + 1) : null;
    const iconClasses = PK.getSeverityIcon(info.severity, secSeverity);
    let type;
    if (info.severity === PK.Enum.INFO_SECURITY) {
        if (secSeverityURL)
            secSeverityURL = <a rel="noopener noreferrer" target="_blank" href={secSeverityURL}>{secSeverity}</a>;
        type = (
            <>
                <Tooltip id="tip-severity" content={ secSeverity || _("security") }>
                    <span className={iconClasses} />
                    { (info.cve_urls && info.cve_urls.length > 0) ? info.cve_urls.length : "" }
                </Tooltip>
            </>);
    } else {
        const tip = (info.severity >= PK.Enum.INFO_NORMAL) ? _("bug fix") : _("enhancement");
        type = (
            <>
                <Tooltip id="tip-severity" content={tip}>
                    <span className={iconClasses} />
                    { bugs ? info.bug_urls.length : "" }
                </Tooltip>
            </>);
    }

    const pkgList = pkgNames.map((n, index) => (
        <Tooltip key={n.name + n.arch} id="tip-summary" content={packageSummaries[n.name] + " (" + n.arch + ")"}>
            <span>{n.name + (index !== (pkgNames.length - 1) ? ", " : "")}</span>
        </Tooltip>)
    );
    const pkgs = pkgList;
    let pkgsTruncated = pkgs;
    if (pkgList.length > 4)
        pkgsTruncated = pkgList.slice(0, 4).concat("…");

    let descriptionFirstLine = (info.description || "").trim();
    if (descriptionFirstLine.indexOf("\n") >= 0)
        descriptionFirstLine = descriptionFirstLine.slice(0, descriptionFirstLine.indexOf("\n"));
    descriptionFirstLine = cleanupChangelogLine(descriptionFirstLine);
    let description;
    if (info.markdown) {
        descriptionFirstLine = <span dangerouslySetInnerHTML={{ __html: remarkable.render(descriptionFirstLine) }} />;
        description = <div dangerouslySetInnerHTML={{ __html: remarkable.render(info.description) }} />;
    } else {
        description = <div className="changelog">{info.description}</div>;
    }

    const expandedContent = (
        <>
            <DescriptionList>
                <DescriptionListGroup>
                    <DescriptionListTerm>{_("Packages")}</DescriptionListTerm>
                    <DescriptionListDescription>{pkgs}</DescriptionListDescription>
                </DescriptionListGroup>
                { cves ? <DescriptionListGroup>
                    <DescriptionListTerm>{_("CVE")}</DescriptionListTerm>
                    <DescriptionListDescription>{cves}</DescriptionListDescription>
                </DescriptionListGroup> : null }
                { secSeverityURL ? <DescriptionListGroup>
                    <DescriptionListTerm>{_("Severity")}</DescriptionListTerm>
                    <DescriptionListDescription className="severity">{secSeverityURL}</DescriptionListDescription>
                </DescriptionListGroup> : null }
                { errata ? <DescriptionListGroup>
                    <DescriptionListTerm>{_("Errata")}</DescriptionListTerm>
                    <DescriptionListDescription>{errata}</DescriptionListDescription>
                </DescriptionListGroup> : null }
                { bugs ? <DescriptionListGroup>
                    <DescriptionListTerm>{_("Bugs")}</DescriptionListTerm>
                    <DescriptionListDescription>{bugs}</DescriptionListDescription>
                </DescriptionListGroup> : null }
            </DescriptionList>
            {description}
        </>
    );

    return {
        columns: [
            { title: pkgsTruncated },
            { title: info.version, props: { className: "version truncating" } },
            { title: type, props: { className: "type" } },
            { title: descriptionFirstLine, props: { className: "changelog" } },
        ],
        props: {
            key,
            className: info.severity === PK.Enum.INFO_SECURITY ? ["error"] : [],
        },
        hasPadding: true,
        expandedContent,
    };
}

const UpdatesList = ({ updates }) => {
    const update_ids = [];

    // PackageKit doesn"t expose source package names, so group packages with the same version and changelog
    // create a reverse version+changes → [id] map on iteration
    const sameUpdate = {};
    const packageNames = {};
    Object.keys(updates).forEach(id => {
        const u = updates[id];
        // did we already see the same version and description? then merge
        const hash = u.version + u.description;
        const seenId = sameUpdate[hash];
        if (seenId) {
            packageNames[seenId].push({ name: u.name, arch: u.arch });
        } else {
            // this is a new update
            sameUpdate[hash] = id;
            packageNames[id] = [{ name: u.name, arch: u.arch }];
            update_ids.push(id);
        }
    });

    // sort security first
    update_ids.sort((a, b) => {
        if (updates[a].severity === PK.Enum.INFO_SECURITY && updates[b].severity !== PK.Enum.INFO_SECURITY)
            return -1;
        if (updates[a].severity !== PK.Enum.INFO_SECURITY && updates[b].severity === PK.Enum.INFO_SECURITY)
            return 1;
        return a.localeCompare(b);
    });

    return (
        <ListingTable aria-label={_("Available updates")}
                gridBreakPoint='grid-lg'
                columns={[
                    { title: _("Name") },
                    { title: _("Version") },
                    { title: _("Severity") },
                    { title: _("Details") },
                ]}
                rows={update_ids.map(id => updateItem(updates[id], packageNames[id].sort((a, b) => a.name > b.name), id))} />
    );
};

class RestartServices extends React.Component {
    constructor(props) {
        super(props);
        this.state = {
            dialogError: undefined,
            restartInProgress: false,
        };

        this.dialogErrorSet = this.dialogErrorSet.bind(this);
        this.dialogErrorDismiss = this.dialogErrorDismiss.bind(this);
        this.restart = this.restart.bind(this);
    }

    dialogErrorSet(text, detail) {
        this.setState({ dialogError: text, dialogErrorDetail: detail });
    }

    dialogErrorDismiss() {
        this.setState({ dialogError: undefined });
    }

    restart() {
        // make sure cockpit package is the last to restart
        const daemons = this.props.tracerPackages.daemons.sort((a, b) => {
            if (a.includes("cockpit") && b.includes("cockpit"))
                return 0;
            if (a.includes("cockpit"))
                return 1;
            return a.localeCompare(b);
        });
        const restarts = daemons.map(service => cockpit.spawn(["systemctl", "restart", service + ".service"], { superuser: "required", err: "message" }));
        this.setState({ restartInProgress: true });
        Promise.all(restarts)
                .then(() => {
                    this.props.onValueChanged({ tracerPackages: { reboot: this.props.tracerPackages.reboot, daemons: [], manual: this.props.tracerPackages.manual } });
                    if (this.props.state === "updateSuccess")
                        this.props.loadUpdates();
                    this.setState({ restartInProgress: false });
                    this.props.close();
                })
                .catch(ex => {
                    this.dialogErrorSet(_("Failed to restart service"), ex.message);
                    // call Tracer again to see what services remain
                    this.props.callTracer(null);
                });
    }

    render() {
        let body;
        if (this.props.tracerRunning) {
            body = (<>
                <div className="spinner spinner-xs spinner-inline" />
                <p>{_("Reloading the state of remaining services")}</p>
            </>);
        } else if (this.props.tracerPackages.daemons.length > 0) {
            body = (<>
                {cockpit.ngettext("The following service will be restarted:", "The following services will be restarted:", this.props.tracerPackages.daemons.length)}
                <TwoColumnContent list={this.props.tracerPackages.daemons} flexClassName="restart-services-modal-body" />
            </>);
        }

        return (
            <Modal id="restart-services-modal" isOpen
                   position="top"
                   variant="medium"
                   onClose={this.props.close}
                   title={_("Restart services")}
                   footer={
                       <>
                           {this.state.dialogError && <ModalError dialogError={this.state.dialogError} dialogErrorDetail={this.state.dialogErrorDetail} />}
                           {this.props.tracerPackages.daemons.includes("cockpit") &&
                               <Alert variant="warning"
                                   title={_("Web Console will restart")}
                                   isInline>
                                   <p>
                                       _("When the Web Console is restarted, you will no longer see progress information. However, the update process will continue in the background. Reconnect to continue watching the update process.")
                                   </p>
                               </Alert>}
                           <Button variant='primary'
                               isDisabled={ this.state.restartInProgress }
                               onClick={ this.restart }>
                               {_("Restart services")}
                           </Button>
                           <Button variant='link' className='btn-cancel' onClick={ this.props.close }>
                               {_("Cancel")}
                           </Button>
                       </>
                   }>
                {body}
            </Modal>
        );
    }
}

class ApplyUpdates extends React.Component {
    constructor() {
        super();
        // actions is a chronological list of { status: PK_STATUS_*, package: "name version" } events
        // that happen during applying updates
        this.state = { percentage: 0, timeRemaining: null, actions: [] };
    }

    componentDidMount() {
        var transactionPath = this.props.transaction;

        PK.watchTransaction(transactionPath, {
            Package: (info, packageId) => {
                const pfields = packageId.split(";");

                // small timeout to avoid excessive overlaps from the next PackageKit progress signal
                PK.call(transactionPath, "org.freedesktop.DBus.Properties", "GetAll", [PK.transactionInterface], { timeout: 500 })
                        .done(reply => {
                            const percent = reply[0].Percentage.v;
                            let remain = -1;
                            if ("RemainingTime" in reply[0])
                                remain = reply[0].RemainingTime.v;
                            // info: see PK_STATUS_* at https://github.com/hughsie/PackageKit/blob/master/lib/packagekit-glib2/pk-enum.h
                            const newActions = this.state.actions.slice();
                            newActions.push({ status: info, package: pfields[0] + " " + pfields[1] + " (" + pfields[2] + ")" });

                            const log = document.getElementById("update-log");
                            let atBottom = false;
                            if (log) {
                                if (log.scrollHeight - log.clientHeight <= log.scrollTop + 2)
                                    atBottom = true;
                            }

                            this.setState({
                                actions: newActions,
                                percentage: percent <= 100 ? percent : 0,
                                timeRemaining: remain > 0 ? remain : null
                            });

                            // scroll update log to the bottom, if it already is (almost) at the bottom
                            if (log && atBottom)
                                log.scrollTop = log.scrollHeight;
                        });
            },
        });
    }

    render() {
        const cancelButton = (
            <Button className={this.state.actions.length !== 0 && "progress-cancel"}
                   variant="secondary"
                   onClick={this.props.onCancel}
                   isDisabled={!this.props.allowCancel}>
                {_("Cancel")}
            </Button>
        );
        if (this.state.actions.length === 0) {
            return <EmptyStatePanel title={ _("Initializing...") }
                                    headingLevel="h5"
                                    titleSize="4xl"
                                    secondary={cancelButton}
                                    loading />;
        }

        const lastAction = this.state.actions[this.state.actions.length - 1];
        return (
            <>
                <div className="progress-main-view">
                    <div className="progress-description">
                        <div className="spinner spinner-xs spinner-inline" />
                        <strong>{ PK_STATUS_STRINGS[lastAction.status] || PK_STATUS_STRINGS[PK.Enum.STATUS_UPDATE] }</strong>
                        &nbsp;{lastAction.package}
                    </div>
                    <Progress title={this.state.timeRemaining && moment.duration(this.state.timeRemaining * 1000).humanize()} value={this.state.percentage} />
                    {cancelButton}
                </div>

                <div className="update-log">
                    <Expander title={_("Update log")} onExpand={() => {
                        // always scroll down on expansion
                        const log = document.getElementById("update-log");
                        log.scrollTop = log.scrollHeight;
                    }}>
                        <div id="update-log" className="update-log-content">
                            <table>
                                <tbody>
                                    { this.state.actions.slice(0, -1).map((action, i) => (
                                        <tr key={action.package + i}>
                                            <th>{PK_STATUS_LOG_STRINGS[action.status] || PK_STATUS_LOG_STRINGS[PK.Enum.STATUS_UPDATE]}</th>
                                            <td>{action.package}</td>
                                        </tr>)) }
                                </tbody>
                            </table>
                        </div>
                    </Expander>
                </div>
            </>
        );
    }
}

const TwoColumnContent = ({ list, flexClassName }) => {
    const half = Math.round(list.length / 2);
    const col1 = list.slice(0, half);
    const col2 = list.slice(half);
    return (
        <Flex className={flexClassName}>
            <FlexItem flex={{ default: 'flex_1' }}>
                <TextContent>
                    <TextList>
                        {col1.map(item => (<TextListItem key={item}>{item}</TextListItem>))}
                    </TextList>
                </TextContent>
            </FlexItem>
            {col2.length > 0 && <FlexItem flex={{ default: 'flex_1' }}>
                <TextContent>
                    <TextList>
                        {col2.map(item => (<TextListItem key={item}>{item}</TextListItem>))}
                    </TextList>
                </TextContent>
            </FlexItem>}
        </Flex>
    );
};

const TwoColumnTitle = ({ icon, str }) => {
    return (<>
        {icon}
        <span className="update-success-table-title">
            {str}
        </span>
    </>);
};

const UpdateSuccess = ({ onIgnore, openServiceRestartDialog, openRebootDialog, restart, manual, reboot, tracerAvailable, history }) => {
    if (!tracerAvailable) {
        return (<>
            <EmptyStatePanel icon={RebootingIcon}
                             title={ _("Update was successful") }
                             headingLevel="h5"
                             titleSize="4xl"
                             paragraph={ _("Updated packages may require a reboot to take effect.") }
                             secondary={
                                 <>
                                     <Button id="reboot-system" variant="primary" onClick={openRebootDialog}>{_("Reboot system...")}</Button>
                                     <Button id="ignore" variant="link" onClick={onIgnore}>{_("Ignore")}</Button>
                                 </>
                             } />
            <div className="flow-list-blank-slate">
                <Expander title={_("Package information")}>
                    <PackageList packages={history[0]} />
                </Expander>
            </div>
        </>);
    }

    const entries = [];
    if (reboot.length > 0) {
        entries.push({
            columns: [
                {
                    title: <TwoColumnTitle icon={<RebootingIcon />}
                                           str={cockpit.format(cockpit.ngettext("$0 package needs a system reboot",
                                                                                "$0 packages need a system reboot",
                                                                                reboot.length),
                                                               reboot.length)} />
                },
            ],
            props: { key: "reboot", id: "reboot-row" },
            hasPadding: true,
            expandedContent: <TwoColumnContent list={reboot} />,
        });
    }

    if (restart.length > 0) {
        entries.push({
            columns: [
                {
                    title: <TwoColumnTitle icon={<ProcessAutomationIcon />}
                                           str={cockpit.format(cockpit.ngettext("$0 service needs to be restarted",
                                                                                "$0 services need to be restarted",
                                                                                restart.length),
                                                               restart.length)} />
                },
            ],
            props: { key: "service", id: "service-row" },
            hasPadding: true,
            expandedContent: <TwoColumnContent list={restart} />,
        });
    }

    if (manual.length > 0) {
        entries.push({
            columns: [
                {
                    title: <TwoColumnTitle icon={<ProcessAutomationIcon />}
                                           str={_("Some software needs to be restarted manually")} />
                }
            ],
            props: { key: "manual", id: "manual-row" },
            hasPadding: true,
            expandedContent: <TwoColumnContent list={manual} />,
        });
    }

    return (<>
        <EmptyStatePanel title={ _("Update was successful") }
            headingLevel="h5"
            titleSize="4xl"
            secondary={
                <>
                    { entries.length > 0 && <ListingTable aria-label={_("Update Success Table")}
                        columns={[{ title: _("Info") }]}
                        showHeader={false}
                        className="updates-success-table"
                        rows={entries} /> }
                    <div className="update-success-actions">
                        { (reboot.length > 0 || manual.length > 0) && <Button id="reboot-system" variant="primary" onClick={openRebootDialog}>{_("Reboot system...")}</Button> }
                        { restart.length > 0 && <Button id="choose-service" variant={reboot.length > 0 ? "secondary" : "primary"} onClick={openServiceRestartDialog}>{_("Restart services...")}</Button> }
                        { reboot.length > 0 || restart.length > 0 || manual.length > 0
                            ? <Button id="ignore" variant="link" onClick={onIgnore}>{_("Ignore")}</Button>
                            : <Button id="ignore" variant="primary" onClick={onIgnore}>{_("Continue")}</Button> }
                    </div>
                </>
            } />
        <div className="flow-list-blank-slate">
            <Expander title={_("Package information")}>
                <PackageList packages={history[0]} />
            </Expander>
        </div>
    </>);
};

const StatusCard = ({ updates, highestSeverity, timeSinceRefresh, tracerPackages, onValueChanged }) => {
    const numUpdates = Object.keys(updates).length;
    const numSecurity = count_security_updates(updates);
    const numRestartServices = tracerPackages.daemons.length;
    const numManualSoftware = tracerPackages.manual.length;
    const numRebootPackages = tracerPackages.reboot.length;
    let lastChecked;
    if (timeSinceRefresh !== null)
        lastChecked = cockpit.format(_("Last checked: $0"), moment(moment().valueOf() - timeSinceRefresh * 1000).fromNow());

    const notifications = [];
    if (numUpdates > 0) {
        if (numUpdates == numSecurity) {
            const stateStr = cockpit.ngettext("$0 security fix available", "$0 security fixes available", numSecurity);
            notifications.push({
                id: "security-updates-available",
                stateStr: cockpit.format(stateStr, numSecurity),
                icon: <span id="icon" className={PK.getSeverityIcon(highestSeverity, undefined, "fa-lg")} />,
                secondary: <Text id="last-checked" component={TextVariants.small}>{lastChecked}</Text>
            });
        } else {
            let stateStr = cockpit.ngettext("$0 update available", "$0 updates available", numUpdates);
            if (numSecurity > 0)
                stateStr += cockpit.ngettext(", including $1 security fix", ", including $1 security fixes", numSecurity);
            notifications.push({
                id: "updates-available",
                stateStr: cockpit.format(stateStr, numUpdates, numSecurity),
                icon: <span id="icon" className={PK.getSeverityIcon(highestSeverity, undefined, "fa-lg")} />,
                secondary: <Text id="last-checked" component={TextVariants.small}>{lastChecked}</Text>
            });
        }
    } else if (!numRestartServices && !numRebootPackages && !numManualSoftware) {
        notifications.push({
            id: "system-up-to-date",
            stateStr: STATE_HEADINGS.uptodate,
            icon: <CheckIcon color="green" size="md" />,
            secondary: <Text id="last-checked" component={TextVariants.small}>{lastChecked}</Text>
        });
    }

    if (numRebootPackages > 0) {
        const stateStr = cockpit.ngettext("$0 package needs a system reboot", "$0 packages need a system reboot", numRebootPackages);
        notifications.push({
            id: "packages-need-reboot",
            stateStr: cockpit.format(stateStr, numRebootPackages),
            icon: <RebootingIcon id="icon" size="md" />,
            secondary: <Button variant="danger" onClick={() => onValueChanged("showRebootSystemDialog", true)} isInline>
                {_("Reboot system...")}
            </Button>
        });
    }

    if (numRestartServices > 0) {
        const stateStr = cockpit.ngettext("$0 service needs to be restarted", "$0 services need to be restarted", numRestartServices);
        notifications.push({
            id: "services-need-restart",
            stateStr: cockpit.format(stateStr, numRestartServices),
            icon: <ProcessAutomationIcon id="icon" size="md" />,
            secondary: <Button variant="primary" onClick={() => onValueChanged("showRestartServicesDialog", true)} isInline>
                {_("Restart services...")}
            </Button>
        });
    }

    if (numManualSoftware > 0) {
        notifications.push({
            id: "processes-need-restart",
            stateStr: _("Some software needs to be restarted manually"),
            icon: <ProcessAutomationIcon id="icon" size="md" />,
            secondary: <Text id="last-checked" component={TextVariants.small}>{tracerPackages.manual.join(", ")}</Text>
        });
    }

    return (<>
        { notifications.map(notification => (
            <Flex direction={{ default: 'column', lg: 'row' }} key={notification.id} id={notification.id} className="status-notification">
                <FlexItem>{notification.icon}</FlexItem>
                <Flex flex={{ default: 'flex_1' }} direction={{ default: 'column' }}>
                    <FlexItem>
                        <Text id="state" component={TextVariants.p}>{notification.stateStr}</Text>
                    </FlexItem>
                    <FlexItem>
                        { notification.secondary }
                    </FlexItem>
                </Flex>
            </Flex>
        ))}
    </>);
};

class CardsPage extends React.Component {
    constructor() {
        super();
        this.state = {
            autoUpdatesEnabled: undefined,
            autoUpdatesType: undefined,
            autoUpdatesDay: undefined,
            autoUpdatesTime: undefined,
        };
    }

    render() {
        const cardContents = [
            {
                id: "status",
                className: "ct-card-info",
                title: _("Status"),
                actions: (<Tooltip content={_("Check for updates")}>
                    <Button variant="secondary" onClick={this.props.handleRefresh}><RedoIcon /></Button>
                </Tooltip>),
                body: <StatusCard updates={this.props.updates}
                                  onValueChanged={this.props.onValueChanged}
                                  tracerPackages={this.props.tracerPackages}
                                  highestSeverity={this.props.highestSeverity}
                                  timeSinceRefresh={this.props.timeSinceRefresh} />
            },
            {
                id: "automatic-updates",
                className: "ct-card-info",
                title: _("Automatic updates"),
                actions: (<AutoUpdates onInitialized={newState => this.setState(newState)} privileged={this.props.privileged} />),
                body: (<AutoUpdatesBody enabled={this.state.autoUpdatesEnabled}
                                        type={this.state.autoUpdatesType}
                                        day={this.state.autoUpdatesDay}
                                        time={this.state.autoUpdatesTime} />),
            },
        ];

        if (this.props.state === "available") { // automatic updates are not tracked by PackageKit, hide history when they are enabled
            cardContents.push({
                id: "available-updates",
                title: _("Available updates"),
                actions: (<div className="pk-updates--header--actions">
                    {this.props.cockpitUpdate &&
                        <Flex flex={{ default: 'inlineFlex' }} className="cockpit-update-warning">
                            <FlexItem>
                                <ExclamationTriangleIcon size="ms" color="#f0ab00" className="cockpit-update-warning-icon" />
                                <strong className="cockpit-update-warning-text">
                                    <span className="pf-screen-reader">{_("Danger alert:")}</span>
                                    {_("Web Console will restart")}
                                </strong>
                            </FlexItem>
                            <FlexItem>
                                <Popover aria-label="More information popover"
                                         bodyContent={_("When the Web Console is restarted, you will no longer see progress information. However, the update process will continue in the background. Reconnect to continue watching the update process.")}>
                                    <a href="#">{_("More info...")}</a>
                                </Popover>
                            </FlexItem>
                        </Flex>}
                    {this.props.applySecurity}
                    {this.props.applyAll}
                </div>),
                containsList: true,
                body: <UpdatesList updates={this.props.updates} />
            });
        }

        if (!this.state.autoUpdatesEnabled && this.props.history.length > 0) { // automatic updates are not tracked by PackageKit, hide history when they are enabled
            cardContents.push({
                id: "update-history",
                title: _("Update history"),
                containsList: true,
                body: <History packagekit={this.props.history} />
            });
        }

        return cardContents.map(card => {
            return (
                <Card key={card.id} className={card.className} id={card.id}>
                    <CardHeader>
                        <CardTitle><h2>{card.title}</h2></CardTitle>
                        {card.actions && <CardActions>{card.actions}</CardActions>}
                    </CardHeader>
                    <CardBody className={card.containsList ? "contains-list" : null}>
                        {card.body}
                    </CardBody>
                </Card>
            );
        });
    }
}

class OsUpdates extends React.Component {
    constructor() {
        super();
        this.state = {
            state: "loading",
            errorMessages: [],
            updates: {},
            timeSinceRefresh: null,
            loadPercent: null,
            cockpitUpdate: false,
            allowCancel: null,
            history: [],
            unregistered: false,
            privileged: false,
            autoUpdatesEnabled: undefined,
            tracerPackages: { daemons: [], manual: [], reboot: [] },
            tracerAvailable: false,
            tracerRunning: false,
            showRestartServicesDialog: false,
            showRebootSystemDialog: false,
        };
        this.handleLoadError = this.handleLoadError.bind(this);
        this.handleRefresh = this.handleRefresh.bind(this);
        this.loadUpdates = this.loadUpdates.bind(this);
        this.onValueChanged = this.onValueChanged.bind(this);

        superuser.addEventListener("changed", () => {
            this.setState({ privileged: superuser.allowed });
            // get out of error state when switching from unprivileged to privileged
            if (superuser.allowed && this.state.state.indexOf("Error") >= 0)
                this.loadUpdates();
        });
    }

    onValueChanged(key, value) {
        this.setState({ [key]: value });
    }

    componentDidMount() {
        this.callTracer(null);

        // check if there is an upgrade in progress already; if so, switch to "applying" state right away
        PK.call("/org/freedesktop/PackageKit", "org.freedesktop.PackageKit", "GetTransactionList", [])
                .done(result => {
                    const transactions = result[0];
                    const promises = transactions.map(transactionPath => PK.call(
                        transactionPath, "org.freedesktop.DBus.Properties", "Get", [PK.transactionInterface, "Role"]));

                    Promise.all(promises)
                            .then(roles => {
                                // any transaction with UPDATE_PACKAGES role?
                                for (let idx = 0; idx < roles.length; ++idx) {
                                    if (roles[idx][0].v === PK.Enum.ROLE_UPDATE_PACKAGES) {
                                        this.watchUpdates(transactions[idx]);
                                        return;
                                    }
                                }

                                // no running updates found, proceed to showing available updates
                                this.initialLoadOrRefresh();
                            })
                            .catch(ex => {
                                console.warn("GetTransactionList: failed to read PackageKit transaction roles:", ex.message);
                                // be robust, try to continue with loading updates anyway
                                this.initialLoadOrRefresh();
                            });
                })
                .fail(this.handleLoadError);
    }

    callTracer(state) {
        this.setState({ tracerRunning: true });
        python.spawn(callTracerScript, null, { err: "message", superuser: "require" })
                .then(output => {
                    const tracerPackages = JSON.parse(output);
                    // Filter out duplicates
                    tracerPackages.reboot = [...new Set(shortenCockpitWsInstance(tracerPackages.reboot))];
                    tracerPackages.daemons = [...new Set(shortenCockpitWsInstance(tracerPackages.daemons))];
                    tracerPackages.manual = [...new Set(shortenCockpitWsInstance(tracerPackages.manual))];
                    const nextState = { tracerAvailable: true, tracerRunning: false, tracerPackages: tracerPackages };
                    if (state)
                        nextState.state = state;

                    this.setState(nextState);
                })
                .catch((exception, data) => {
                    console.error(`Tracer failed: "${JSON.stringify(exception)}", data: "${JSON.stringify(data)}"`);
                    // When tracer fails, act like it's not available (demand reboot after every update)
                    const nextState = { tracerAvailable: false, tracerRunning: false, tracerPackages: { reboot: [], daemons: [], manual: [] } };
                    if (state)
                        nextState.state = state;
                    this.setState(nextState);
                });
    }

    handleLoadError(ex) {
        console.warn("loading available updates failed:", JSON.stringify(ex));
        if (ex.problem === "not-found")
            ex = _("PackageKit is not installed");
        this.state.errorMessages.push(ex.detail || ex.message || ex);
        this.setState({ state: "loadError" });
    }

    removeHeading(text) {
        // on Debian the update_text starts with "== version ==" which is
        // redundant; we don't want Markdown headings in the table
        if (text)
            return text.trim().replace(/^== .* ==\n/, "")
                    .trim();
        return text;
    }

    loadUpdateDetails(pkg_ids) {
        PK.cancellableTransaction("GetUpdateDetail", [pkg_ids], null, {
            UpdateDetail: (packageId, updates, obsoletes, vendor_urls, bug_urls, cve_urls, restart,
                update_text, changelog /* state, issued, updated */) => {
                const u = this.state.updates[packageId];
                u.vendor_urls = vendor_urls;
                // HACK: bug_urls and cve_urls also contain titles, in a not-quite-predictable order; ignore them,
                // only pick out http[s] URLs (https://bugs.freedesktop.org/show_bug.cgi?id=104552)
                if (bug_urls)
                    bug_urls = bug_urls.filter(url => url.match(/^https?:\/\//));
                if (cve_urls)
                    cve_urls = cve_urls.filter(url => url.match(/^https?:\/\//));

                u.description = this.removeHeading(update_text) || changelog;
                if (update_text)
                    u.markdown = true;
                u.bug_urls = deduplicate(bug_urls);
                // many backends don't support proper severities; parse CVEs from description as a fallback
                u.cve_urls = deduplicate(cve_urls && cve_urls.length > 0 ? cve_urls : parseCVEs(u.description));
                if (u.cve_urls && u.cve_urls.length > 0)
                    u.severity = PK.Enum.INFO_SECURITY;
                u.vendor_urls = vendor_urls || [];
                // u.restart = restart; // broken (always "1") at least in Fedora

                this.setState({ updates: this.state.updates });
            },
        })
                .then(() => this.setState({ state: "available" }))
                .catch(ex => {
                    console.warn("GetUpdateDetail failed:", JSON.stringify(ex));
                    // still show available updates, with reduced detail
                    this.setState({ state: "available" });
                });
    }

    loadUpdates() {
        var updates = {};
        var cockpitUpdate = false;

        PK.cancellableTransaction("GetUpdates", [0],
                                  data => this.setState({ state: data.waiting ? "locked" : "loading" }),
                                  {
                                      Package: (info, packageId, _summary) => {
                                          const id_fields = packageId.split(";");
                                          packageSummaries[id_fields[0]] = _summary;
                                          // HACK: dnf backend yields wrong severity (https://bugs.freedesktop.org/show_bug.cgi?id=101070)
                                          if (info < PK.Enum.INFO_LOW || info > PK.Enum.INFO_SECURITY)
                                              info = PK.Enum.INFO_NORMAL;
                                          updates[packageId] = { name: id_fields[0], version: id_fields[1], severity: info, arch: id_fields[2] };
                                          if (id_fields[0] == "cockpit-ws")
                                              cockpitUpdate = true;
                                      },
                                  })
                .then(() => {
                    // get the details for all packages
                    const pkg_ids = Object.keys(updates);
                    if (pkg_ids.length) {
                        this.setState({ updates, cockpitUpdate: cockpitUpdate });
                        this.loadUpdateDetails(pkg_ids);
                    } else {
                        this.setState({ updates: {}, state: "uptodate" });
                    }
                    this.loadHistory();
                })
                .catch(this.handleLoadError);
    }

    loadHistory() {
        const history = [];

        // would be nice to filter only for "update-packages" role, but can't here
        PK.transaction("GetOldTransactions", [0], {
            Transaction: (objPath, timeSpec, succeeded, role, duration, data) => {
                if (role !== PK.Enum.ROLE_UPDATE_PACKAGES)
                    return;
                    // data looks like:
                    // downloading\tbash-completion;1:2.6-1.fc26;noarch;updates-testing
                    // updating\tbash-completion;1:2.6-1.fc26;noarch;updates-testing
                const pkgs = { _time: Date.parse(timeSpec) };
                let empty = true;
                data.split("\n").forEach(line => {
                    const fields = line.trim().split("\t");
                    if (fields.length >= 2) {
                        const pkgId = fields[1].split(";");
                        pkgs[pkgId[0]] = pkgId[1];
                        empty = false;
                    }
                });
                if (!empty)
                    history.unshift(pkgs); // PK reports in time-ascending order, but we want the latest first
            },

            // only update the state once to avoid flicker
            Finished: () => {
                if (history.length > 0)
                    this.setState({ history: history });
            }
        })
                .catch(ex => console.warn("Failed to load old transactions:", ex));
    }

    initialLoadOrRefresh() {
        PK.watchRedHatSubscription(registered => this.setState({ unregistered: !registered }));

        cockpit.addEventListener("visibilitychange", () => {
            if (!cockpit.hidden)
                this.loadOrRefresh(false);
        });

        if (!cockpit.hidden)
            this.loadOrRefresh(true);
        else
            this.loadUpdates();
    }

    loadOrRefresh(always_load) {
        PK.call("/org/freedesktop/PackageKit", "org.freedesktop.PackageKit", "GetTimeSinceAction",
                [PK.Enum.ROLE_REFRESH_CACHE])
                .done(results => {
                    const seconds = results[0];

                    this.setState({ timeSinceRefresh: seconds });

                    // automatically trigger refresh for ≥ 1 day or if never refreshed
                    if (seconds >= 24 * 3600 || seconds < 0)
                        this.handleRefresh();
                    else if (always_load)
                        this.loadUpdates();
                })
                .fail(this.handleLoadError);
    }

    watchUpdates(transactionPath) {
        this.setState({ state: "applying", applyTransaction: transactionPath, allowCancel: false });

        PK.call(transactionPath, "DBus.Properties", "Get", [PK.transactionInterface, "AllowCancel"])
                .done(reply => this.setState({ allowCancel: reply[0].v }));

        return PK.watchTransaction(transactionPath,
                                   {
                                       ErrorCode: (code, details) => this.state.errorMessages.push(details),

                                       Finished: exit => {
                                           this.setState({ applyTransaction: null, allowCancel: null });

                                           if (exit === PK.Enum.EXIT_SUCCESS) {
                                               if (this.state.tracerAvailable) {
                                                   this.setState({ state: "loading", loadPercent: null });
                                                   this.callTracer("updateSuccess");
                                               } else {
                                                   this.setState({ state: "updateSuccess", loadPercent: null });
                                               }
                                               this.loadHistory();
                                           } else if (exit === PK.Enum.EXIT_CANCELLED) {
                                               if (this.state.tracerAvailable) {
                                                   this.setState({ state: "loading", loadPercent: null });
                                                   this.callTracer(null);
                                               }
                                               this.loadUpdates();
                                           } else {
                                               // normally we get FAILED here with ErrorCodes; handle unexpected errors to allow for some debugging
                                               if (exit !== PK.Enum.EXIT_FAILED)
                                                   this.state.errorMessages.push(cockpit.format(_("PackageKit reported error code $0"), exit));
                                               this.setState({ state: "updateError" });
                                           }
                                       },

                                       // not working/being used in at least Fedora
                                       RequireRestart: (type, packageId) => console.log("update RequireRestart", type, packageId),
                                   },

                                   notify => {
                                       if ("AllowCancel" in notify)
                                           this.setState({ allowCancel: notify.AllowCancel });
                                   })
                .fail(ex => {
                    this.state.errorMessages.push(ex);
                    this.setState({ state: "updateError" });
                });
    }

    applyUpdates(securityOnly) {
        var ids = Object.keys(this.state.updates);
        if (securityOnly)
            ids = ids.filter(id => this.state.updates[id].severity === PK.Enum.INFO_SECURITY);

        PK.transaction()
                .then(transactionPath => {
                    this.watchUpdates(transactionPath)
                            .then(() => {
                                PK.call(transactionPath, PK.transactionInterface, "UpdatePackages", [0, ids])
                                        .fail(ex => {
                                            // We get more useful error messages through ErrorCode or "PackageKit has crashed", so only
                                            // show this if we don't have anything else
                                            if (this.state.errorMessages.length === 0)
                                                this.state.errorMessages.push(ex.message);
                                            this.setState({ state: "updateError" });
                                        });
                            });
                })
                .catch(ex => {
                    this.state.errorMessages.push(ex.message);
                    this.setState({ state: "updateError" });
                });
    }

    renderContent() {
        var applySecurity, applyAll;

        if (this.state.unregistered) {
            // always show empty state pattern, even if there are some
            // repositories enabled that don't require subscriptions

            page_status.set_own({
                type: "warning",
                title: _("Not registered"),
                details: {
                    link: "subscriptions",
                    icon: "fa fa-exclamation-triangle"
                }
            });

            return <EmptyStatePanel
                title={_("This system is not registered")}
                headingLevel="h5"
                titleSize="4xl"
                paragraph={ _("To get software updates, this system needs to be registered with Red Hat, either using the Red Hat Customer Portal or a local subscription server.") }
                icon={ExclamationCircleIcon}
                action={ _("Register…") }
                onAction={ () => cockpit.jump("/subscriptions", cockpit.transport.host) } />;
        }

        switch (this.state.state) {
        case "loading":
        case "refreshing":
        case "locked":
            page_status.set_own({
                type: null,
                title: _("Checking for package updates..."),
                details: {
                    link: false,
                    icon: "spinner spinner-xs",
                }
            });

            if (this.state.loadPercent)
                return <Progress value={this.state.loadPercent} title={STATE_HEADINGS[this.state.state]} />;
            else
                return <EmptyStatePanel loading title={ _("Checking software status")}
                                        headingLevel="h5"
                                        titleSize="4xl"
                                        paragraph={STATE_HEADINGS[this.state.state]} />;

        case "available":
        {
            const num_updates = Object.keys(this.state.updates).length;
            const num_security_updates = count_security_updates(this.state.updates);
            const highest_severity = find_highest_severity(this.state.updates);
            let text;

            applyAll = (
                <Button id={num_updates == num_security_updates ? "install-security" : "install-all"} variant="primary" onClick={ () => this.applyUpdates(false) }>
                    { num_updates == num_security_updates
                        ? _("Install security updates") : _("Install all updates") }
                </Button>);

            if (num_security_updates > 0 && num_updates > num_security_updates) {
                applySecurity = (
                    <Button id="install-security" variant="secondary" onClick={ () => this.applyUpdates(true) }>
                        {_("Install security updates")}
                    </Button>);
            }

            if (highest_severity == PK.Enum.INFO_SECURITY)
                text = _("Security updates available");
            else if (highest_severity >= PK.Enum.INFO_NORMAL)
                text = _("Bug fix updates available");
            else if (highest_severity >= PK.Enum.INFO_LOW)
                text = _("Enhancement updates available");
            else
                text = _("Updates available");

            page_status.set_own({
                type: num_security_updates > 0 ? "warning" : "info",
                title: text,
                details: {
                    icon: PK.getSeverityIcon(highest_severity)
                }
            });

            return (
                <>
                    <PageSection>
                        <Gallery className='ct-cards-grid' hasGutter>
                            <CardsPage handleRefresh={this.handleRefresh}
                                       applySecurity={applySecurity}
                                       applyAll={applyAll}
                                       highestSeverity={highest_severity}
                                       onValueChanged={this.onValueChanged}
                                       {...this.state} />
                        </Gallery>
                    </PageSection>
                    { this.state.showRestartServicesDialog &&
                        <RestartServices tracerPackages={this.state.tracerPackages}
                            close={() => this.setState({ showRestartServicesDialog: false })}
                            state={this.state.state}
                            callTracer={(state) => this.callTracer(state)}
                            onValueChanged={delta => this.setState(delta)}
                            loadUpdates={this.loadUpdates} />
                    }
                    { this.state.showRebootSystemDialog &&
                        <ShutdownModal onClose={() => this.setState({ showRebootSystemDialog: false })} />
                    }
                </>
            );
        }

        case "loadError":
        case "updateError":
            page_status.set_own({
                type: "error",
                title: STATE_HEADINGS[this.state.state],
                details: {
                    icon: "fa fa-exclamation-circle"
                }
            });
            return this.state.errorMessages.map(m => <pre key={m}>{m}</pre>);

        case "applying":
            page_status.set_own(null);
            return <ApplyUpdates transaction={this.state.applyTransaction}
                                 onCancel={ () => PK.call(this.state.applyTransaction, PK.transactionInterface, "Cancel", []) }
                                 allowCancel={this.state.allowCancel} />;

        case "updateSuccess": {
            let warningTitle;
            if (!this.state.tracerAvailable) {
                warningTitle = _("Reboot recommended");
            } else {
                if (this.state.tracerPackages.reboot.length > 0)
                    warningTitle = cockpit.ngettext("A package needs a system reboot for the updates to take effect:",
                                                    "Some packages need a system reboot for the updates to take effect:",
                                                    this.state.tracerPackages.reboot.length);
                else if (this.state.tracerPackages.daemons.length > 0)
                    warningTitle = cockpit.ngettext("A service needs to be restarted for the updates to take effect:",
                                                    "Some services need to be restarted for the updates to take effect:",
                                                    this.state.tracerPackages.daemons.length);
                else if (this.state.tracerPackages.manual.length > 0)
                    warningTitle = _("Some software needs to be restarted manually");
            }

            if (warningTitle) {
                page_status.set_own({
                    type: "warning",
                    title: warningTitle
                });
            }

            return (
                <>
                    <UpdateSuccess onIgnore={this.loadUpdates}
                        openServiceRestartDialog={() => this.setState({ showRestartServicesDialog: true })}
                        openRebootDialog={() => this.setState({ showRebootSystemDialog: true })}
                        restart={this.state.tracerPackages.daemons}
                        manual={this.state.tracerPackages.manual}
                        reboot={this.state.tracerPackages.reboot}
                        tracerAvailable={this.state.tracerAvailable}
                        history={this.state.history} />
                    { this.state.showRebootSystemDialog &&
                        <ShutdownModal onClose={() => this.setState({ showRebootSystemDialog: false })} />
                    }
                    { this.state.showRestartServicesDialog &&
                        <RestartServices tracerPackages={this.state.tracerPackages}
                            close={() => this.setState({ showRestartServicesDialog: false })}
                            state={this.state.state}
                            callTracer={(state) => this.callTracer(state)}
                            onValueChanged={delta => this.setState(delta)}
                            loadUpdates={this.loadUpdates} />
                    }
                </>);
        }

        case "restart":
            page_status.set_own(null);
            return <EmptyStatePanel loading title={ _("Restarting") }
                                    headingLevel="h5"
                                    titleSize="4xl"
                                    paragraph={ _("Your server will close the connection soon. You can reconnect after it has restarted.") } />;

        case "uptodate":
        {
            page_status.set_own({
                title: STATE_HEADINGS[this.state.state],
                details: {
                    link: false,
                    icon: "fa fa-check-circle-o"
                }
            });

            return (
                <>
                    <PageSection>
                        <Gallery className='ct-cards-grid' hasGutter>
                            <CardsPage onValueChanged={this.onValueChanged} handleRefresh={this.handleRefresh} {...this.state} />
                        </Gallery>
                        { this.state.showRestartServicesDialog &&
                            <RestartServices tracerPackages={this.state.tracerPackages}
                                close={() => this.setState({ showRestartServicesDialog: false })}
                                state={this.state.state}
                                callTracer={(state) => this.callTracer(state)}
                                onValueChanged={delta => this.setState(delta)}
                                loadUpdates={this.loadUpdates} />
                        }
                        { this.state.showRebootSystemDialog &&
                            <ShutdownModal onClose={() => this.setState({ showRebootSystemDialog: false })} />
                        }
                    </PageSection>
                </>
            );
        }

        default:
            page_status.set_own(null);
            return null;
        }
    }

    handleRefresh() {
        this.setState({ state: "refreshing", loadPercent: null });
        PK.cancellableTransaction("RefreshCache", [true], data => this.setState({ loadPercent: data.percentage }))
                .then(() => {
                    this.setState({ timeSinceRefresh: 0 });
                    this.loadUpdates();
                })
                .catch(this.handleLoadError);
    }

    render() {
        let content = this.renderContent();
        if (!["available", "uptodate"].includes(this.state.state))
            content = <PageSection variant={PageSectionVariants.light}>{content}</PageSection>;

        let header;
        if (["updateError", "loadError"].includes(this.state.state))
            header = <PageSection>{STATE_HEADINGS[this.state.state]}</PageSection>;

        return (
            <Page>
                {header}
                {content}
            </Page>
        );
    }
}

document.addEventListener("DOMContentLoaded", () => {
    document.title = cockpit.gettext(document.title);
    moment.locale(cockpit.language);
    init();
    ReactDOM.render(<OsUpdates />, document.getElementById("app"));
});
