!-------------------------------------------------------------------------------

! This file is part of Code_Saturne, a general-purpose CFD tool.
!
! Copyright (C) 1998-2015 EDF S.A.
!
! This program is free software; you can redistribute it and/or modify it under
! the terms of the GNU General Public License as published by the Free Software
! Foundation; either version 2 of the License, or (at your option) any later
! version.
!
! This program is distributed in the hope that it will be useful, but WITHOUT
! ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
! details.
!
! You should have received a copy of the GNU General Public License along with
! this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
! Street, Fifth Floor, Boston, MA 02110-1301, USA.

!-------------------------------------------------------------------------------

!> \file chem_luscheme_siream.f90
!> \brief Routines for atmospheric chemical scheme 3
!> \remarks
!>  These routines are automatically generated by SPACK
!>  See CEREA: http://cerea.enpc.fr/polyphemus

!> kinetic_siream
!> \brief Computation of kinetic rates for atmospheric chemistry
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name              role
!------------------------------------------------------------------------------
!> \param[in]     nr                total number of chemical reactions
!> \param[in]     option_photolysis flag to activate or not photolysis reactions
!> \param[in]     azi               solar zenith angle
!> \param[in]     att               atmospheric attenuation variable
!> \param[in]     temp              temperature
!> \param[in]     press             pressure
!> \param[in]     xlw               water massic fraction
!> \param[out]    rk(nr)            kinetic rates
!______________________________________________________________________________

subroutine kinetic_siream(nr,rk,temp,xlw,press,azi,att,                  &
     option_photolysis)

implicit none

! Arguments

integer nr
double precision rk(nr),temp,xlw,press
double precision azi, att
integer option_photolysis

! Local variables

double precision effko,rapk,facteur,summ
double precision ylh2o

!     Compute third body.
!     Conversion = Avogadro*1d-6/Perfect gas constant.
!     PRESS in Pascal, SUMM in molecules/cm3, TEMP in Kelvin

summ = press * 7.243d16 / temp

!     Number of water molecules computed from the massic fraction
!     (absolute humidity)

ylh2o = 29.d0*summ*xlw/(18.d0+11.d0*xlw)

!     For the zenithal angle at tropics

azi=abs(azi)


if(option_photolysis.eq.2) then
 rk(  1)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  1)=-0.1302720567168795d-07
 rk(  1)=-0.7822279432831311d-06+(azi- 0.00d+00) * rk(  1)
 rk(  1)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  1)
 rk(  1)= 0.9310260000000001d-02+(azi- 0.00d+00) * rk(  1)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  1)= 0.3771617015067078d-08
 rk(  1)=-0.1173044113433769d-05+(azi- 0.10d+02) * rk(  1)
 rk(  1)=-0.1955272056716901d-04+(azi- 0.10d+02) * rk(  1)
 rk(  1)= 0.9219010000000000d-02+(azi- 0.10d+02) * rk(  1)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  1)=-0.5859262388581815d-08
 rk(  1)=-0.1059895602981758d-05+(azi- 0.20d+02) * rk(  1)
 rk(  1)=-0.4188211773132428d-04+(azi- 0.20d+02) * rk(  1)
 rk(  1)= 0.8909950000000000d-02+(azi- 0.20d+02) * rk(  1)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  1)=-0.7024567460738029d-08
 rk(  1)=-0.1235673474639213d-05+(azi- 0.30d+02) * rk(  1)
 rk(  1)=-0.6483780850753392d-04+(azi- 0.30d+02) * rk(  1)
 rk(  1)= 0.8379279999999999d-02+(azi- 0.30d+02) * rk(  1)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  1)=-0.9202467768466835d-08
 rk(  1)=-0.1446410498461367d-05+(azi- 0.40d+02) * rk(  1)
 rk(  1)=-0.9165864823853972d-04+(azi- 0.40d+02) * rk(  1)
 rk(  1)= 0.7600310000000000d-02+(azi- 0.40d+02) * rk(  1)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  1)=-0.1612556146540100d-07
 rk(  1)=-0.1722484531515342d-05+(azi- 0.50d+02) * rk(  1)
 rk(  1)=-0.1233475985383066d-03+(azi- 0.50d+02) * rk(  1)
 rk(  1)= 0.6529880000000000d-02+(azi- 0.50d+02) * rk(  1)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  1)= 0.3226471363007382d-07
 rk(  1)=-0.2206251375477548d-05+(azi- 0.60d+02) * rk(  1)
 rk(  1)=-0.1626349576082332d-03+(azi- 0.60d+02) * rk(  1)
 rk(  1)= 0.5108030000000000d-02+(azi- 0.60d+02) * rk(  1)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  1)= 0.2027078243961372d-06
 rk(  1)=-0.1238309966574737d-05+(azi- 0.70d+02) * rk(  1)
 rk(  1)=-0.1970805710287543d-03+(azi- 0.70d+02) * rk(  1)
 rk(  1)= 0.3293320000000000d-02+(azi- 0.70d+02) * rk(  1)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  1)=-0.7448311471194499d-07
 rk(  1)= 0.3626677818932555d-05+(azi- 0.78d+02) * rk(  1)
 rk(  1)=-0.1779736282099126d-03+(azi- 0.78d+02) * rk(  1)
 rk(  1)= 0.1741210000000000d-02+(azi- 0.78d+02) * rk(  1)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  1)= 0.2490309929270573d-05
 rk(  1)= 0.1839083065842406d-05+(azi- 0.86d+02) * rk(  1)
 rk(  1)=-0.1342475411316713d-03+(azi- 0.86d+02) * rk(  1)
 rk(  1)= 0.5113930000000000d-03+(azi- 0.86d+02) * rk(  1)
elseif(azi.ge.90.d0)then
 rk(  1)= 0.1632080000000000d-03
endif
if(att.lt.0.99999) rk(  1) = rk(  1) * att
endif

rk(  2) = summ * 6.0d-34 * (temp/3.d2) ** (-2.4d0)
rk(  2) = rk(  2) * summ * 0.2d0
rk(  3) =  dexp(-0.2653240882726044d+02                           &
 - (  0.1500000000000000d+04 )/temp)
rk(  4) =  dexp(-0.2590825451818744d+02                           &
 - ( -0.1800000000000000d+03 )/temp)
effko =  0.2500000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.1800000000000000d+01))
rapk =  0.2200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.7000000000000000d+00))
rk(  5) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.9000000000000000d-31* (temp / 3.d2)                    &
           **(- ( 0.1500000000000000d+01))
rapk =  0.3000000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(  6) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(  7) =  dexp(-0.2975128465212864d+02                           &
 - (  0.2450000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk(  8)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  8)=-0.5928286648807996d-09
 rk(  8)=-0.3096171335119280d-07+(azi- 0.00d+00) * rk(  8)
 rk(  8)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  8)
 rk(  8)= 0.4927580000000000d-03+(azi- 0.00d+00) * rk(  8)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  8)= 0.1444859946426876d-09
 rk(  8)=-0.4874657329761677d-07+(azi- 0.10d+02) * rk(  8)
 rk(  8)=-0.7970828664880956d-06+(azi- 0.10d+02) * rk(  8)
 rk(  8)= 0.4890690000000000d-03+(azi- 0.10d+02) * rk(  8)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  8)=-0.5511531369012520d-10
 rk(  8)=-0.4441199345833616d-07+(azi- 0.20d+02) * rk(  8)
 rk(  8)=-0.1728668534047625d-05+(azi- 0.20d+02) * rk(  8)
 rk(  8)= 0.4763680000000000d-03+(azi- 0.20d+02) * rk(  8)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  8)=-0.3000247398821449d-09
 rk(  8)=-0.4606545286904014d-07+(azi- 0.30d+02) * rk(  8)
 rk(  8)=-0.2633442997321385d-05+(azi- 0.30d+02) * rk(  8)
 rk(  8)= 0.4545850000000000d-03+(azi- 0.30d+02) * rk(  8)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  8)=-0.2397857267812366d-09
 rk(  8)=-0.5506619506550444d-07+(azi- 0.40d+02) * rk(  8)
 rk(  8)=-0.3644759476666826d-05+(azi- 0.40d+02) * rk(  8)
 rk(  8)= 0.4233440000000000d-03+(azi- 0.40d+02) * rk(  8)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  8)=-0.1844832352993067d-08
 rk(  8)=-0.6225976686893995d-07+(azi- 0.50d+02) * rk(  8)
 rk(  8)=-0.4818019096011278d-05+(azi- 0.50d+02) * rk(  8)
 rk(  8)= 0.3811500000000000d-03+(azi- 0.50d+02) * rk(  8)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  8)= 0.1101151387530619d-09
 rk(  8)=-0.1176047374587370d-06+(azi- 0.60d+02) * rk(  8)
 rk(  8)=-0.6616664139287950d-05+(azi- 0.60d+02) * rk(  8)
 rk(  8)= 0.3248990000000000d-03+(azi- 0.60d+02) * rk(  8)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  8)=-0.1557211541866474d-07
 rk(  8)=-0.1143012832961418d-06+(azi- 0.70d+02) * rk(  8)
 rk(  8)=-0.8935724346837023d-05+(azi- 0.70d+02) * rk(  8)
 rk(  8)= 0.2470820000000000d-03+(azi- 0.70d+02) * rk(  8)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  8)= 0.4048472604232427d-07
 rk(  8)=-0.4880320533439059d-06+(azi- 0.78d+02) * rk(  8)
 rk(  8)=-0.1375439103995816d-04+(azi- 0.78d+02) * rk(  8)
 rk(  8)= 0.1603080000000000d-03+(azi- 0.78d+02) * rk(  8)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  8)= 0.2066880316654862d-06
 rk(  8)= 0.4836013716715513d-06+(azi- 0.86d+02) * rk(  8)
 rk(  8)=-0.1378983649333310d-04+(azi- 0.86d+02) * rk(  8)
 rk(  8)= 0.3976700000000000d-04+(azi- 0.86d+02) * rk(  8)
elseif(azi.ge.90.d0)then
 rk(  8)= 0.5573310000000000d-05
endif
if(att.lt.0.99999) rk(  8) = rk(  8) * att
endif


if(option_photolysis.eq.2) then
 rk(  9)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  9)=-0.8776629099833464d-10
 rk(  9)=-0.1165033709001661d-07+(azi- 0.00d+00) * rk(  9)
 rk(  9)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  9)
 rk(  9)= 0.3523480000000000d-04+(azi- 0.00d+00) * rk(  9)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  9)= 0.1474988729949909d-09
 rk(  9)=-0.1428332581996665d-07+(azi- 0.10d+02) * rk(  9)
 rk(  9)=-0.2593366290998327d-06+(azi- 0.10d+02) * rk(  9)
 rk(  9)= 0.3398200000000000d-04+(azi- 0.10d+02) * rk(  9)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  9)= 0.1300707990183827d-09
 rk(  9)=-0.9858359630116941d-08+(azi- 0.20d+02) * rk(  9)
 rk(  9)=-0.5007534836006686d-06+(azi- 0.20d+02) * rk(  9)
 rk(  9)= 0.3010780000000000d-04+(azi- 0.20d+02) * rk(  9)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  9)= 0.1988179309314732d-09
 rk(  9)=-0.5956235659565481d-08+(azi- 0.30d+02) * rk(  9)
 rk(  9)=-0.6588994364974921d-06+(azi- 0.30d+02) * rk(  9)
 rk(  9)= 0.2424450000000000d-04+(azi- 0.30d+02) * rk(  9)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  9)= 0.2219574772557277d-09
 rk(  9)= 0.8302268378835231d-11+(azi- 0.40d+02) * rk(  9)
 rk(  9)=-0.7183787704093613d-06+(azi- 0.40d+02) * rk(  9)
 rk(  9)= 0.1725870000000000d-04+(azi- 0.40d+02) * rk(  9)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  9)= 0.1913521600455895d-09
 rk(  9)= 0.6667026586050136d-08+(azi- 0.50d+02) * rk(  9)
 rk(  9)=-0.6516254818650674d-06+(azi- 0.50d+02) * rk(  9)
 rk(  9)= 0.1029770000000000d-04+(azi- 0.50d+02) * rk(  9)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  9)= 0.1602388256152816d-10
 rk(  9)= 0.1240759138741867d-07+(azi- 0.60d+02) * rk(  9)
 rk(  9)=-0.4608793021303539d-06+(azi- 0.60d+02) * rk(  9)
 rk(  9)= 0.4639500000000000d-05+(azi- 0.60d+02) * rk(  9)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  9)=-0.3089359890783960d-09
 rk(  9)= 0.1288830786428400d-07+(azi- 0.70d+02) * rk(  9)
 rk(  9)=-0.2079203096133002d-06+(azi- 0.70d+02) * rk(  9)
 rk(  9)= 0.1287490000000000d-05+(azi- 0.70d+02) * rk(  9)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  9)=-0.2034952628632162d-09
 rk(  9)= 0.5473844126395715d-08+(azi- 0.78d+02) * rk(  9)
 rk(  9)=-0.6102309368797090d-07+(azi- 0.78d+02) * rk(  9)
 rk(  9)= 0.2908040000000000d-06+(azi- 0.78d+02) * rk(  9)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  9)= 0.1623544916128788d-09
 rk(  9)= 0.5899578175158973d-09+(azi- 0.86d+02) * rk(  9)
 rk(  9)=-0.1251267813581064d-07+(azi- 0.86d+02) * rk(  9)
 rk(  9)= 0.4875570000000000d-07+(azi- 0.86d+02) * rk(  9)
elseif(azi.ge.90.d0)then
 rk(  9)= 0.1853500000000000d-07
endif
if(att.lt.0.99999) rk(  9) = rk(  9) * att
endif

rk( 10) =  dexp(-0.2458649867820512d+02                           &
 - ( -0.1020000000000000d+03 )/temp)
rk( 10) = rk( 10) * summ
rk( 11) =  0.2200000000000000d-09
rk( 11) = rk( 11) * ylh2o
rk( 12) =  dexp(-0.2710039286486638d+02                           &
 - (  0.9400000000000000d+03 )/temp)
rk( 13) =  dexp(-0.3223619130191664d+02                           &
 - (  0.4900000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 14)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 14)=-0.1926445199576598d-06
 rk( 14)=-0.1114355480042343d-04+(azi- 0.00d+00) * rk( 14)
 rk( 14)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 14)
 rk( 14)= 0.2129960000000000d+00+(azi- 0.00d+00) * rk( 14)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 14)=-0.3806644012699938d-07
 rk( 14)=-0.1692289039915322d-04+(azi- 0.10d+02) * rk( 14)
 rk( 14)=-0.2806644519957665d-03+(azi- 0.10d+02) * rk( 14)
 rk( 14)= 0.2116890000000000d+00+(azi- 0.10d+02) * rk( 14)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 14)= 0.3591028046562922d-07
 rk( 14)=-0.1806488360296321d-04+(azi- 0.20d+02) * rk( 14)
 rk( 14)=-0.6305421920169308d-03+(azi- 0.20d+02) * rk( 14)
 rk( 14)= 0.2071520000000000d+00+(azi- 0.20d+02) * rk( 14)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 14)=-0.2185746817355116d-06
 rk( 14)=-0.1698757518899435d-04+(azi- 0.30d+02) * rk( 14)
 rk( 14)=-0.9810667799365058d-03+(azi- 0.30d+02) * rk( 14)
 rk( 14)= 0.1990760000000000d+00+(azi- 0.30d+02) * rk( 14)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 14)=-0.2006115535235656d-06
 rk( 14)=-0.2354481564105959d-04+(azi- 0.40d+02) * rk( 14)
 rk( 14)=-0.1386390688237046d-02+(azi- 0.40d+02) * rk( 14)
 rk( 14)= 0.1873480000000000d+00+(azi- 0.40d+02) * rk( 14)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 14)=-0.1055979104170267d-05
 rk( 14)=-0.2956316224676618d-04+(azi- 0.50d+02) * rk( 14)
 rk( 14)=-0.1917470467115309d-02+(azi- 0.50d+02) * rk( 14)
 rk( 14)= 0.1709290000000000d+00+(azi- 0.50d+02) * rk( 14)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 14)=-0.3204720297952934d-06
 rk( 14)=-0.6124253537187832d-04+(azi- 0.60d+02) * rk( 14)
 rk( 14)=-0.2825527443301723d-02+(azi- 0.60d+02) * rk( 14)
 rk( 14)= 0.1477420000000000d+00+(azi- 0.60d+02) * rk( 14)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 14)=-0.6785572971821502d-05
 rk( 14)=-0.7085669626571978d-04+(azi- 0.70d+02) * rk( 14)
 rk( 14)=-0.4146519759677725d-02+(azi- 0.70d+02) * rk( 14)
 rk( 14)= 0.1130420000000000d+00+(azi- 0.70d+02) * rk( 14)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 14)= 0.2349184361304102d-04
 rk( 14)=-0.2337104475894081d-03+(azi- 0.78d+02) * rk( 14)
 rk( 14)=-0.6583056910518970d-02+(azi- 0.78d+02) * rk( 14)
 rk( 14)= 0.7186080000000000d-01+(azi- 0.78d+02) * rk( 14)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 14)= 0.6606749385962285d-04
 rk( 14)= 0.3300937991232988d-03+(azi- 0.86d+02) * rk( 14)
 rk( 14)=-0.5811990098247399d-02+(azi- 0.86d+02) * rk( 14)
 rk( 14)= 0.1626670000000000d-01+(azi- 0.86d+02) * rk( 14)
elseif(azi.ge.90.d0)then
 rk( 14)= 0.2528560000000000d-02
endif
if(att.lt.0.99999) rk( 14) = rk( 14) * att
endif


if(option_photolysis.eq.2) then
 rk( 15)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 15)=-0.3691161034363460d-07
 rk( 15)=-0.1177883896563626d-05+(azi- 0.00d+00) * rk( 15)
 rk( 15)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 15)
 rk( 15)= 0.2656490000000000d-01+(azi- 0.00d+00) * rk( 15)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 15)= 0.1703483103089462d-07
 rk( 15)=-0.2285232206872663d-05+(azi- 0.10d+02) * rk( 15)
 rk( 15)=-0.3463116103436289d-04+(azi- 0.10d+02) * rk( 15)
 rk( 15)= 0.2641020000000000d-01+(azi- 0.10d+02) * rk( 15)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 15)=-0.1532771377994130d-07
 rk( 15)=-0.1774187275945825d-05+(azi- 0.20d+02) * rk( 15)
 rk( 15)=-0.7522535586254776d-04+(azi- 0.20d+02) * rk( 15)
 rk( 15)= 0.2585240000000000d-01+(azi- 0.20d+02) * rk( 15)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 15)=-0.1372397591112858d-07
 rk( 15)=-0.2234018689344064d-05+(azi- 0.30d+02) * rk( 15)
 rk( 15)=-0.1153074155154466d-03+(azi- 0.30d+02) * rk( 15)
 rk( 15)= 0.2490740000000000d-01+(azi- 0.30d+02) * rk( 15)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 15)=-0.2747638257554304d-07
 rk( 15)=-0.2645737966677932d-05+(azi- 0.40d+02) * rk( 15)
 rk( 15)=-0.1641049820756665d-03+(azi- 0.40d+02) * rk( 15)
 rk( 15)= 0.2351720000000000d-01+(azi- 0.40d+02) * rk( 15)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 15)=-0.1322704937867030d-06
 rk( 15)=-0.3470029443944237d-05+(azi- 0.50d+02) * rk( 15)
 rk( 15)=-0.2252626561818883d-03+(azi- 0.50d+02) * rk( 15)
 rk( 15)= 0.2158410000000000d-01+(azi- 0.50d+02) * rk( 15)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 15)=-0.2304164227760796d-07
 rk( 15)=-0.7438144257545598d-05+(azi- 0.60d+02) * rk( 15)
 rk( 15)=-0.3343443931967845d-03+(azi- 0.60d+02) * rk( 15)
 rk( 15)= 0.1885220000000000d-01+(azi- 0.60d+02) * rk( 15)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 15)=-0.1151055949407016d-05
 rk( 15)=-0.8129393525872428d-05+(azi- 0.70d+02) * rk( 15)
 rk( 15)=-0.4900197710309764d-03+(azi- 0.70d+02) * rk( 15)
 rk( 15)= 0.1474190000000000d-01+(azi- 0.70d+02) * rk( 15)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 15)= 0.3051007034753431d-05
 rk( 15)=-0.3575473631163908d-04+(azi- 0.78d+02) * rk( 15)
 rk( 15)=-0.8410928097310894d-03+(azi- 0.78d+02) * rk( 15)
 rk( 15)= 0.9712119999999999d-02+(azi- 0.78d+02) * rk( 15)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 15)= 0.1099207874719760d-04
 rk( 15)= 0.3746943252241897d-04+(azi- 0.86d+02) * rk( 15)
 rk( 15)=-0.8273752400447254d-03+(azi- 0.86d+02) * rk( 15)
 rk( 15)= 0.2257190000000000d-02+(azi- 0.86d+02) * rk( 15)
elseif(azi.ge.90.d0)then
 rk( 15)= 0.2506930000000000d-03
endif
if(att.lt.0.99999) rk( 15) = rk( 15) * att
endif

rk( 16) =  dexp(-0.2492297091482634d+02                           &
 - ( -0.1700000000000000d+03 )/temp)
rk( 17) =  dexp(-0.3073211390514037d+02                           &
 - (  0.1260000000000000d+04 )/temp)
effko =  0.2000000000000000d-29* (temp / 3.d2)                    &
           **(- ( 0.4400000000000000d+01))
rapk =  0.1400000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.7000000000000000d+00))
rk( 18) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk( 19) =  0.1000000000000000d-21
rk( 19) = rk( 19) * ylh2o
effko =  0.1300000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.3500000000000000d+01))*                       &
           dexp(- 0.1100000000000000d+05/temp)
rapk =  0.9700000000000000d+15* (temp / 3.d2)                     &
            **(- (-0.1000000000000000d+00))*                      &
           dexp(- 0.1108000000000000d+05/temp)
rk( 20) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.4500d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))
rk( 21) =  dexp(-0.8860689615829534d+02                           &
 - ( -0.5300000000000000d+03 )/temp)
rk( 21) = rk( 21) * summ * 0.2d0
rk( 22) =  0.5000000000000000d-39
rk( 22) = rk( 22) * ylh2o
effko =  0.7000000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.2600000000000000d+01))
rapk =  0.3600000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.1000000000000000d+00))
rk( 23) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk( 24)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 24)=-0.2887225450832658d-08
 rk( 24)=-0.1810277454916759d-06+(azi- 0.00d+00) * rk( 24)
 rk( 24)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 24)
 rk( 24)= 0.2046710000000000d-02+(azi- 0.00d+00) * rk( 24)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 24)= 0.8216763524985595d-09
 rk( 24)=-0.2676445090166556d-06+(azi- 0.10d+02) * rk( 24)
 rk( 24)=-0.4486722545083316d-05+(azi- 0.10d+02) * rk( 24)
 rk( 24)= 0.2025720000000000d-02+(azi- 0.10d+02) * rk( 24)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 24)=-0.1309479959161308d-08
 rk( 24)=-0.2429942184416991d-06+(azi- 0.20d+02) * rk( 24)
 rk( 24)=-0.9593109819666860d-05+(azi- 0.20d+02) * rk( 24)
 rk( 24)= 0.1954910000000000d-02+(azi- 0.20d+02) * rk( 24)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 24)=-0.1523756515853649d-08
 rk( 24)=-0.2822786172165394d-06+(azi- 0.30d+02) * rk( 24)
 rk( 24)=-0.1484583817624924d-04+(azi- 0.30d+02) * rk( 24)
 rk( 24)= 0.1833370000000000d-02+(azi- 0.30d+02) * rk( 24)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 24)=-0.1745493977424016d-08
 rk( 24)=-0.3279913126921461d-06+(azi- 0.40d+02) * rk( 24)
 rk( 24)=-0.2094853747533609d-04+(azi- 0.40d+02) * rk( 24)
 rk( 24)= 0.1655160000000000d-02+(azi- 0.40d+02) * rk( 24)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 24)=-0.2754267574450560d-08
 rk( 24)=-0.3803561320148624d-06+(azi- 0.50d+02) * rk( 24)
 rk( 24)=-0.2803201192240625d-04+(azi- 0.50d+02) * rk( 24)
 rk( 24)= 0.1411130000000000d-02+(azi- 0.50d+02) * rk( 24)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 24)= 0.1037656427523324d-07
 rk( 24)=-0.4629841592484096d-06+(azi- 0.60d+02) * rk( 24)
 rk( 24)=-0.3646541483503878d-04+(azi- 0.60d+02) * rk( 24)
 rk( 24)= 0.1090020000000000d-02+(azi- 0.60d+02) * rk( 24)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 24)= 0.4335158727139456d-07
 rk( 24)=-0.1516872309913989d-06+(azi- 0.70d+02) * rk( 24)
 rk( 24)=-0.4261212873743828d-04+(azi- 0.70d+02) * rk( 24)
 rk( 24)= 0.6894440000000000d-03+(azi- 0.70d+02) * rk( 24)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 24)=-0.2976581610911796d-07
 rk( 24)= 0.8887508635220705d-06+(azi- 0.78d+02) * rk( 24)
 rk( 24)=-0.3671561967719464d-04+(azi- 0.78d+02) * rk( 24)
 rk( 24)= 0.3610350000000000d-03+(azi- 0.78d+02) * rk( 24)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 24)= 0.5586598403857848d-06
 rk( 24)= 0.1743712769036732d-06+(azi- 0.86d+02) * rk( 24)
 rk( 24)=-0.2821064255377481d-04+(azi- 0.86d+02) * rk( 24)
 rk( 24)= 0.1089500000000000d-03+(azi- 0.86d+02) * rk( 24)
elseif(azi.ge.90.d0)then
 rk( 24)= 0.3465160000000000d-04
endif
if(att.lt.0.99999) rk( 24) = rk( 24) * att
endif

rk( 25) =  dexp(-0.2474064935803238d+02                           &
 - (  0.3900000000000000d+03 )/temp)
rk( 26) =  0.9999999999999999d-20
rk( 27) =  dexp(-0.8322449114623726d+01                           &
 + (-0.3000000000000000d+01 * dlog(temp)) )
effko = 2.4d-14 * dexp(460.0d0 / temp)
rapk = 2.7d-17 * dexp(2199.0d0 / temp)
facteur =6.5d-34 * dexp(1335.0d0 / temp) * summ
rk( 28) = effko + facteur/(1.0d0 + facteur / rapk)
rk( 29) =  dexp(-0.2637825814743318d+02                           &
 - ( -0.2500000000000000d+03 )/temp)
effko =  0.1800000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.3200000000000000d+01))
rapk =  0.4700000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk( 30) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4100000000000000d-04* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1065000000000000d+05/temp)
rapk =  0.4800000000000000d+16* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1117000000000000d+05/temp)
rk( 31) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.6000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))
rk( 32) =  dexp(-0.2736865685146106d+02                           &
 - ( -0.3800000000000000d+03 )/temp)
rk( 33) = 2.3d-13 * dexp(600.0d0 / temp)                          &
            + 1.7d-33* summ * dexp(1000.0d0 / temp)
rk( 34) = 3.22d-34 * dexp(2800.0d0 / temp) +                      &
            2.38d-54 * summ * dexp(3200.0d0 / temp)
rk( 34) = rk( 34) * ylh2o

if(option_photolysis.eq.2) then
 rk( 35)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 35)=-0.1441479345432039d-10
 rk( 35)=-0.1242452065456794d-08+(azi- 0.00d+00) * rk( 35)
 rk( 35)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 35)
 rk( 35)= 0.8394580000000000d-05+(azi- 0.00d+00) * rk( 35)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 35)= 0.8244380362960478d-11
 rk( 35)=-0.1674895869086405d-08+(azi- 0.10d+02) * rk( 35)
 rk( 35)=-0.2917347934543199d-07+(azi- 0.10d+02) * rk( 35)
 rk( 35)= 0.8255920000000000d-05+(azi- 0.10d+02) * rk( 35)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 35)= 0.1172720024787734d-12
 rk( 35)=-0.1427564458197592d-08+(azi- 0.20d+02) * rk( 35)
 rk( 35)=-0.6019808261827194d-07+(azi- 0.20d+02) * rk( 35)
 rk( 35)= 0.7804940000000000d-05+(azi- 0.20d+02) * rk( 35)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 35)= 0.4506531627127201d-11
 rk( 35)=-0.1424046298123240d-08+(azi- 0.30d+02) * rk( 35)
 rk( 35)=-0.8871419018148013d-07+(azi- 0.30d+02) * rk( 35)
 rk( 35)= 0.7060320000000000d-05+(azi- 0.30d+02) * rk( 35)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 35)= 0.1086660148900755d-10
 rk( 35)=-0.1288850349309390d-08+(azi- 0.40d+02) * rk( 35)
 rk( 35)=-0.1158431566558070d-06+(azi- 0.40d+02) * rk( 35)
 rk( 35)= 0.6035280000000000d-05+(azi- 0.40d+02) * rk( 35)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 35)= 0.1803706241686108d-10
 rk( 35)=-0.9628523046392104d-09+(azi- 0.50d+02) * rk( 35)
 rk( 35)=-0.1383601831952927d-06+(azi- 0.50d+02) * rk( 35)
 rk( 35)= 0.4758830000000000d-05+(azi- 0.50d+02) * rk( 35)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 35)= 0.7329514884355585d-10
 rk( 35)=-0.4217404321336693d-09+(azi- 0.60d+02) * rk( 35)
 rk( 35)=-0.1522061105630206d-06+(azi- 0.60d+02) * rk( 35)
 rk( 35)= 0.3296980000000000d-05+(azi- 0.60d+02) * rk( 35)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 35)= 0.6172011386267615d-10
 rk( 35)= 0.1777114033174383d-08+(azi- 0.70d+02) * rk( 35)
 rk( 35)=-0.1386523745526241d-06+(azi- 0.70d+02) * rk( 35)
 rk( 35)= 0.1806040000000000d-05+(azi- 0.70d+02) * rk( 35)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 35)=-0.8279704635596216d-10
 rk( 35)= 0.3258396765877763d-08+(azi- 0.78d+02) * rk( 35)
 rk( 35)=-0.9836828816022045d-07+(azi- 0.78d+02) * rk( 35)
 rk( 35)= 0.8421570000000000d-06+(azi- 0.78d+02) * rk( 35)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 35)= 0.1082517324572734d-08
 rk( 35)= 0.1271267653334671d-08+(azi- 0.86d+02) * rk( 35)
 rk( 35)=-0.6213097280649386d-07+(azi- 0.86d+02) * rk( 35)
 rk( 35)= 0.2213560000000000d-06+(azi- 0.86d+02) * rk( 35)
elseif(azi.ge.90.d0)then
 rk( 35)= 0.6245350000000000d-07
endif
if(att.lt.0.99999) rk( 35) = rk( 35) * att
endif

rk( 36) =  dexp(-0.2656631037893612d+02                           &
 - (  0.1600000000000000d+03 )/temp)
rk( 37) =  0.1100000000000000d-09
rk( 37) = rk( 37) * summ * 5.8d-7
rk( 38) =  dexp(-0.2592627302369012d+02                           &
 - (  0.2000000000000000d+04 )/temp)
rk( 38) = rk( 38) * summ * 5.8d-7
rk( 39) =  dexp(-0.2453997866257023d+02                           &
 - ( -0.1200000000000000d+03 )/temp)
rk( 40) =  dexp(-0.2619593659063922d+02                           &
 - (  0.2400000000000000d+03 )/temp)
effko =  0.6900000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.1000000000000000d+01))
rapk =  0.2600000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk( 41) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk( 42) =  dexp(-0.2375982010502066d+02                           &
 - ( -0.2500000000000000d+03 )/temp)
rk( 43) =  dexp(-0.2422982373426639d+02                           &
 - ( -0.2000000000000000d+03 )/temp)
rk( 44) =  dexp(-0.2729454887930734d+02                           &
 - (  0.2000000000000000d+04 )/temp)
rk( 45) =  0.9999999999999999d-11
rk( 46) =  0.2200000000000000d-10
rk( 47) =  0.3500000000000000d-11
rk( 48) =  0.1000000000000000d-16
rk( 49) =  dexp(-0.2779354004542632d+02                           &
 - (  0.2450000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk( 50)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 50)=-0.1022637521258300d-10
 rk( 50)=-0.1047636247874169d-08+(azi- 0.00d+00) * rk( 50)
 rk( 50)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 50)
 rk( 50)= 0.5038360000000000d-05+(azi- 0.00d+00) * rk( 50)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 50)= 0.9289125637748400d-11
 rk( 50)=-0.1354427504251659d-08+(azi- 0.10d+02) * rk( 50)
 rk( 50)=-0.2402063752125827d-07+(azi- 0.10d+02) * rk( 50)
 rk( 50)= 0.4923370000000000d-05+(azi- 0.10d+02) * rk( 50)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 50)= 0.4979872661590682d-11
 rk( 50)=-0.1075753735119207d-08+(azi- 0.20d+02) * rk( 50)
 rk( 50)=-0.4832244991496692d-07+(azi- 0.20d+02) * rk( 50)
 rk( 50)= 0.4557010000000000d-05+(azi- 0.20d+02) * rk( 50)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 50)= 0.1055138371588688d-10
 rk( 50)=-0.9263575552714849d-09+(azi- 0.30d+02) * rk( 50)
 rk( 50)=-0.6834356281887375d-07+(azi- 0.30d+02) * rk( 50)
 rk( 50)= 0.3971190000000000d-05+(azi- 0.30d+02) * rk( 50)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 50)= 0.1620459247486068d-10
 rk( 50)=-0.6098160437948834d-09+(azi- 0.40d+02) * rk( 50)
 rk( 50)=-0.8370529880953754d-07+(azi- 0.40d+02) * rk( 50)
 rk( 50)= 0.3205670000000000d-05+(azi- 0.40d+02) * rk( 50)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 50)= 0.2215024638467897d-10
 rk( 50)=-0.1236782695490497d-09+(azi- 0.50d+02) * rk( 50)
 rk( 50)=-0.9104024194297703d-07+(azi- 0.50d+02) * rk( 50)
 rk( 50)= 0.2323840000000000d-05+(azi- 0.50d+02) * rk( 50)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 50)= 0.3999042198642357d-10
 rk( 50)= 0.5408291219911870d-09+(azi- 0.60d+02) * rk( 50)
 rk( 50)=-0.8686873341855619d-07+(azi- 0.60d+02) * rk( 50)
 rk( 50)= 0.1423220000000000d-05+(azi- 0.60d+02) * rk( 50)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 50)=-0.2067341717000215d-11
 rk( 50)= 0.1740541781586012d-08+(azi- 0.70d+02) * rk( 50)
 rk( 50)=-0.6405502438280072d-07+(azi- 0.70d+02) * rk( 50)
 rk( 50)= 0.6486060000000000d-06+(azi- 0.70d+02) * rk( 50)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 50)=-0.2916729149837185d-10
 rk( 50)= 0.1690925580378854d-08+(azi- 0.78d+02) * rk( 50)
 rk( 50)=-0.3660328548710213d-07+(azi- 0.78d+02) * rk( 50)
 rk( 50)= 0.2465020000000000d-06+(azi- 0.78d+02) * rk( 50)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 50)= 0.1504439894534147d-09
 rk( 50)= 0.9909105844077646d-09+(azi- 0.86d+02) * rk( 50)
 rk( 50)=-0.1514859616880918d-07+(azi- 0.86d+02) * rk( 50)
 rk( 50)= 0.4696130000000000d-07+(azi- 0.86d+02) * rk( 50)
elseif(azi.ge.90.d0)then
 rk( 50)= 0.1184990000000000d-07
endif
if(att.lt.0.99999) rk( 50) = rk( 50) * att
endif


if(option_photolysis.eq.2) then
 rk( 51)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 51)=-0.1532025548735494d-11
 rk( 51)=-0.1605297445126457d-09+(azi- 0.00d+00) * rk( 51)
 rk( 51)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 51)
 rk( 51)= 0.7475310000000001d-06+(azi- 0.00d+00) * rk( 51)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 51)= 0.1499076646206637d-11
 rk( 51)=-0.2064905109747105d-09+(azi- 0.10d+02) * rk( 51)
 rk( 51)=-0.3670202554873561d-08+(azi- 0.10d+02) * rk( 51)
 rk( 51)= 0.7299460000000000d-06+(azi- 0.10d+02) * rk( 51)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 51)= 0.8827189639089728d-12
 rk( 51)=-0.1615182115885116d-09+(azi- 0.20d+02) * rk( 51)
 rk( 51)=-0.7350289780505779d-08+(azi- 0.20d+02) * rk( 51)
 rk( 51)= 0.6740940000000000d-06+(azi- 0.20d+02) * rk( 51)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 51)= 0.1742047498157359d-11
 rk( 51)=-0.1350366426712420d-09+(azi- 0.30d+02) * rk( 51)
 rk( 51)=-0.1031583832310330d-07+(azi- 0.30d+02) * rk( 51)
 rk( 51)= 0.5853220000000000d-06+(azi- 0.30d+02) * rk( 51)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 51)= 0.2513091043461601d-11
 rk( 51)=-0.8277521772651622d-10+(azi- 0.40d+02) * rk( 51)
 rk( 51)=-0.1249395692708094d-07+(azi- 0.40d+02) * rk( 51)
 rk( 51)= 0.4704020000000000d-06+(azi- 0.40d+02) * rk( 51)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 51)= 0.3194588327996089d-11
 rk( 51)=-0.7382486422669833d-11+(azi- 0.50d+02) * rk( 51)
 rk( 51)=-0.1339553396857280d-07+(azi- 0.50d+02) * rk( 51)
 rk( 51)= 0.3396980000000000d-06+(azi- 0.50d+02) * rk( 51)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 51)= 0.5281855644545232d-11
 rk( 51)= 0.8845516341719630d-10+(azi- 0.60d+02) * rk( 51)
 rk( 51)=-0.1258480719862726d-07+(azi- 0.60d+02) * rk( 51)
 rk( 51)= 0.2081990000000000d-06+(azi- 0.60d+02) * rk( 51)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 51)=-0.2756160173816243d-12
 rk( 51)= 0.2469108327538047d-09+(azi- 0.70d+02) * rk( 51)
 rk( 51)=-0.9231147236917937d-08+(azi- 0.70d+02) * rk( 51)
 rk( 51)= 0.9647830000000000d-07+(azi- 0.70d+02) * rk( 51)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 51)=-0.5270116382809120d-11
 rk( 51)= 0.2402960483368575d-09+(azi- 0.78d+02) * rk( 51)
 rk( 51)=-0.5333492188194333d-08+(azi- 0.78d+02) * rk( 51)
 rk( 51)= 0.3829030000000000d-07+(azi- 0.78d+02) * rk( 51)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 51)= 0.3312732748114380d-10
 rk( 51)= 0.1138132551494386d-09+(azi- 0.86d+02) * rk( 51)
 rk( 51)=-0.2500617760290411d-08+(azi- 0.86d+02) * rk( 51)
 rk( 51)= 0.8303010000000000d-08+(azi- 0.86d+02) * rk( 51)
elseif(azi.ge.90.d0)then
 rk( 51)= 0.2241700000000000d-08
endif
if(att.lt.0.99999) rk( 51) = rk( 51) * att
endif


if(option_photolysis.eq.2) then
 rk( 52)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 52)=-0.1726633761595217d-06
 rk( 52)=-0.3763226238404795d-05+(azi- 0.00d+00) * rk( 52)
 rk( 52)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 52)
 rk( 52)= 0.2758968400000000d-01+(azi- 0.00d+00) * rk( 52)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 52)= 0.8277361284785716d-06
 rk( 52)=-0.8943127523190445d-05+(azi- 0.10d+02) * rk( 52)
 rk( 52)=-0.1270635376159524d-03+(azi- 0.10d+02) * rk( 52)
 rk( 52)= 0.2704069800000000d-01+(azi- 0.10d+02) * rk( 52)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 52)=-0.1363361137754772d-05
 rk( 52)= 0.1588895633116670d-04+(azi- 0.20d+02) * rk( 52)
 rk( 52)=-0.5760524953618990d-04+(azi- 0.20d+02) * rk( 52)
 rk( 52)= 0.2570348600000000d-01+(azi- 0.20d+02) * rk( 52)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 52)= 0.9462274225405240d-06
 rk( 52)=-0.2501187780147645d-04+(azi- 0.30d+02) * rk( 52)
 rk( 52)=-0.1488344642392872d-03+(azi- 0.30d+02) * rk( 52)
 rk( 52)= 0.2535296800000000d-01+(azi- 0.30d+02) * rk( 52)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 52)=-0.2183585524073524d-06
 rk( 52)= 0.3374944874739267d-05+(azi- 0.40d+02) * rk( 52)
 rk( 52)=-0.3652037935066590d-03+(azi- 0.40d+02) * rk( 52)
 rk( 52)= 0.2230966300000000d-01+(azi- 0.40d+02) * rk( 52)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 52)= 0.1378647870888786d-06
 rk( 52)=-0.3175811697481157d-05+(azi- 0.50d+02) * rk( 52)
 rk( 52)=-0.3632124617340762d-03+(azi- 0.50d+02) * rk( 52)
 rk( 52)= 0.1877676100000000d-01+(azi- 0.50d+02) * rk( 52)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 52)=-0.3865635959481289d-06
 rk( 52)= 0.9601319151840079d-06+(azi- 0.60d+02) * rk( 52)
 rk( 52)=-0.3853692595570321d-03+(azi- 0.60d+02) * rk( 52)
 rk( 52)= 0.1496492000000000d-01+(azi- 0.60d+02) * rk( 52)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 52)=-0.1916508555648358d-06
 rk( 52)=-0.1063677596325682d-04+(azi- 0.70d+02) * rk( 52)
 rk( 52)=-0.4821357000377863d-03+(azi- 0.70d+02) * rk( 52)
 rk( 52)= 0.1082067700000000d-01+(azi- 0.70d+02) * rk( 52)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 52)= 0.2540478756918318d-05
 rk( 52)=-0.1523639649680941d-04+(azi- 0.78d+02) * rk( 52)
 rk( 52)=-0.6891210797183578d-03+(azi- 0.78d+02) * rk( 52)
 rk( 52)= 0.6184712500000000d-02+(azi- 0.78d+02) * rk( 52)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 52)= 0.1651057353835306d-05
 rk( 52)= 0.4573509366928574d-04+(azi- 0.86d+02) * rk( 52)
 rk( 52)=-0.4451315023386027d-03+(azi- 0.86d+02) * rk( 52)
 rk( 52)= 0.9973396100000000d-03+(azi- 0.86d+02) * rk( 52)
elseif(azi.ge.90.d0)then
 rk( 52)= 0.5424277000000000d-04
endif
if(att.lt.0.99999) rk( 52) = rk( 52) * att
endif

rk( 53) =  dexp(-0.2667550967090111d+02                           &
 - ( -0.3650000000000000d+03 )/temp)
rk( 54) =  dexp(-0.2667550967090111d+02                           &
 - ( -0.3650000000000000d+03 )/temp)
rk( 55) =  dexp(-0.2791870318838033d+02                           &
 - ( -0.7000000000000000d+03 )/temp)
rk( 56) =  dexp(-0.2791870318838033d+02                           &
 - ( -0.7000000000000000d+03 )/temp)
rk( 57) =  0.6800000000000000d-13
rk( 58) =  0.6800000000000000d-13
rk( 59) =  0.6800000000000000d-13
rk( 60) =  dexp(-0.2815865385801092d+02                           &
 - (  0.3600000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 61)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 61)=-0.6650705923705479d-11
 rk( 61)=-0.6611929407629450d-09+(azi- 0.00d+00) * rk( 61)
 rk( 61)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 61)
 rk( 61)= 0.3379090000000000d-05+(azi- 0.00d+00) * rk( 61)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 61)= 0.5602117771116236d-11
 rk( 61)=-0.8607141184741090d-09+(azi- 0.10d+02) * rk( 61)
 rk( 61)=-0.1521907059237054d-07+(azi- 0.10d+02) * rk( 61)
 rk( 61)= 0.3306320000000000d-05+(azi- 0.10d+02) * rk( 61)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 61)= 0.2622234839240618d-11
 rk( 61)=-0.6926505853406227d-09+(azi- 0.20d+02) * rk( 61)
 rk( 61)=-0.3075271763051784d-07+(azi- 0.20d+02) * rk( 61)
 rk( 61)= 0.3073660000000000d-05+(azi- 0.20d+02) * rk( 61)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 61)= 0.6018942871922055d-11
 rk( 61)=-0.6139835401634017d-09+(azi- 0.30d+02) * rk( 61)
 rk( 61)=-0.4381905888555809d-07+(azi- 0.30d+02) * rk( 61)
 rk( 61)= 0.2699490000000000d-05+(azi- 0.30d+02) * rk( 61)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 61)= 0.9631993673072422d-11
 rk( 61)=-0.4334152540057500d-09+(azi- 0.40d+02) * rk( 61)
 rk( 61)=-0.5429304682724973d-07+(azi- 0.40d+02) * rk( 61)
 rk( 61)= 0.2205920000000000d-05+(azi- 0.40d+02) * rk( 61)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 61)= 0.1346308243578111d-10
 rk( 61)=-0.1444554438135145d-09+(azi- 0.50d+02) * rk( 61)
 rk( 61)=-0.6007175380544298d-07+(azi- 0.50d+02) * rk( 61)
 rk( 61)= 0.1629280000000000d-05+(azi- 0.50d+02) * rk( 61)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 61)= 0.2754267658383055d-10
 rk( 61)= 0.2594370292596410d-09+(azi- 0.60d+02) * rk( 61)
 rk( 61)=-0.5892193795098066d-07+(azi- 0.60d+02) * rk( 61)
 rk( 61)= 0.1027580000000000d-05+(azi- 0.60d+02) * rk( 61)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 61)= 0.3078605881868256d-11
 rk( 61)= 0.1085717326774769d-08+(azi- 0.70d+02) * rk( 61)
 rk( 61)=-0.4547039439064206d-07+(azi- 0.70d+02) * rk( 61)
 rk( 61)= 0.4918470000000000d-06+(azi- 0.70d+02) * rk( 61)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 61)=-0.2153154079143221d-10
 rk( 61)= 0.1159603867942148d-08+(azi- 0.78d+02) * rk( 61)
 rk( 61)=-0.2750782483291689d-07+(azi- 0.78d+02) * rk( 61)
 rk( 61)= 0.1991460000000000d-06+(azi- 0.78d+02) * rk( 61)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 61)= 0.1655300763808969d-09
 rk( 61)= 0.6428468889409992d-09+(azi- 0.86d+02) * rk( 61)
 rk( 61)=-0.1308821877786526d-07+(azi- 0.86d+02) * rk( 61)
 rk( 61)= 0.4227390000000000d-07+(azi- 0.86d+02) * rk( 61)
elseif(azi.ge.90.d0)then
 rk( 61)= 0.1080050000000000d-07
endif
if(att.lt.0.99999) rk( 61) = rk( 61) * att
endif

rk( 62) =  dexp(-0.2652908103716776d+02                           &
 - ( -0.1900000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 63)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 63)=-0.9639086891432908d-11
 rk( 63)=-0.8144091310856724d-09+(azi- 0.00d+00) * rk( 63)
 rk( 63)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 63)
 rk( 63)= 0.5704620000000000d-05+(azi- 0.00d+00) * rk( 63)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 63)= 0.5287260674299008d-11
 rk( 63)=-0.1103581737828659d-08+(azi- 0.10d+02) * rk( 63)
 rk( 63)=-0.1917990868914332d-07+(azi- 0.10d+02) * rk( 63)
 rk( 63)= 0.5613540000000000d-05+(azi- 0.10d+02) * rk( 63)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 63)=-0.2699558057629897d-12
 rk( 63)=-0.9449639175996891d-09+(azi- 0.20d+02) * rk( 63)
 rk( 63)=-0.3966536524342679d-07+(azi- 0.20d+02) * rk( 63)
 rk( 63)= 0.5316670000000000d-05+(azi- 0.20d+02) * rk( 63)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 63)= 0.2382562548753546d-11
 rk( 63)=-0.9530625917725825d-09+(azi- 0.30d+02) * rk( 63)
 rk( 63)=-0.5864563033714949d-07+(azi- 0.30d+02) * rk( 63)
 rk( 63)= 0.4825250000000000d-05+(azi- 0.30d+02) * rk( 63)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 63)= 0.6319705610747424d-11
 rk( 63)=-0.8815857153099513d-09+(azi- 0.40d+02) * rk( 63)
 rk( 63)=-0.7699211340797505d-07+(azi- 0.40d+02) * rk( 63)
 rk( 63)= 0.4145870000000000d-05+(azi- 0.40d+02) * rk( 63)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 63)= 0.1056861500825653d-10
 rk( 63)=-0.6919945469875881d-09+(azi- 0.50d+02) * rk( 63)
 rk( 63)=-0.9272791603094997d-07+(azi- 0.50d+02) * rk( 63)
 rk( 63)= 0.3294110000000000d-05+(azi- 0.50d+02) * rk( 63)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 63)= 0.4850583435620618d-10
 rk( 63)=-0.3749360967397995d-09+(azi- 0.60d+02) * rk( 63)
 rk( 63)=-0.1033972224682254d-06+(azi- 0.60d+02) * rk( 63)
 rk( 63)= 0.2308200000000000d-05+(azi- 0.60d+02) * rk( 63)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 63)= 0.4863918163365898d-10
 rk( 63)= 0.1080238933948239d-08+(azi- 0.70d+02) * rk( 63)
 rk( 63)=-0.9634419409614782d-07+(azi- 0.70d+02) * rk( 63)
 rk( 63)= 0.1285240000000000d-05+(azi- 0.70d+02) * rk( 63)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 63)=-0.6023611040521017d-10
 rk( 63)= 0.2247579293157748d-08+(azi- 0.78d+02) * rk( 63)
 rk( 63)=-0.6972164827933041d-07+(azi- 0.78d+02) * rk( 63)
 rk( 63)= 0.6085250000000000d-06+(azi- 0.78d+02) * rk( 63)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 63)= 0.8106335758155240d-09
 rk( 63)= 0.8019126434123756d-09+(azi- 0.86d+02) * rk( 63)
 rk( 63)=-0.4532571278666100d-07+(azi- 0.86d+02) * rk( 63)
 rk( 63)= 0.1637560000000000d-06+(azi- 0.86d+02) * rk( 63)
elseif(azi.ge.90.d0)then
 rk( 63)= 0.4716430000000000d-07
endif
if(att.lt.0.99999) rk( 63) = rk( 63) * att
endif

rk( 64) = 1.44d-13 * (1.0d0 + 2.381d-20 *                         &
 8.0d-1 * summ)
rk( 65) =  dexp(-0.2673493309137191d+02                           &
 - (  0.1775000000000000d+04 )/temp)
rk( 66) =  dexp(-0.2660140169874739d+02                           &
 - ( -0.3000000000000000d+03 )/temp)
rk( 67) =  dexp(-0.2852261923521233d+02                           &
 - ( -0.7500000000000000d+03 )/temp)
rk( 68) =  dexp(-0.2998489950331015d+02                           &
 - ( -0.3900000000000000d+03 )/temp)
rk( 69) =  dexp(-0.2629602004919621d+02                           &
 - ( -0.2000000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 70)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 70)=-0.9639086891432908d-11
 rk( 70)=-0.8144091310856724d-09+(azi- 0.00d+00) * rk( 70)
 rk( 70)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 70)
 rk( 70)= 0.5704620000000000d-05+(azi- 0.00d+00) * rk( 70)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 70)= 0.5287260674299008d-11
 rk( 70)=-0.1103581737828659d-08+(azi- 0.10d+02) * rk( 70)
 rk( 70)=-0.1917990868914332d-07+(azi- 0.10d+02) * rk( 70)
 rk( 70)= 0.5613540000000000d-05+(azi- 0.10d+02) * rk( 70)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 70)=-0.2699558057629897d-12
 rk( 70)=-0.9449639175996891d-09+(azi- 0.20d+02) * rk( 70)
 rk( 70)=-0.3966536524342679d-07+(azi- 0.20d+02) * rk( 70)
 rk( 70)= 0.5316670000000000d-05+(azi- 0.20d+02) * rk( 70)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 70)= 0.2382562548753546d-11
 rk( 70)=-0.9530625917725825d-09+(azi- 0.30d+02) * rk( 70)
 rk( 70)=-0.5864563033714949d-07+(azi- 0.30d+02) * rk( 70)
 rk( 70)= 0.4825250000000000d-05+(azi- 0.30d+02) * rk( 70)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 70)= 0.6319705610747424d-11
 rk( 70)=-0.8815857153099513d-09+(azi- 0.40d+02) * rk( 70)
 rk( 70)=-0.7699211340797505d-07+(azi- 0.40d+02) * rk( 70)
 rk( 70)= 0.4145870000000000d-05+(azi- 0.40d+02) * rk( 70)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 70)= 0.1056861500825653d-10
 rk( 70)=-0.6919945469875881d-09+(azi- 0.50d+02) * rk( 70)
 rk( 70)=-0.9272791603094997d-07+(azi- 0.50d+02) * rk( 70)
 rk( 70)= 0.3294110000000000d-05+(azi- 0.50d+02) * rk( 70)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 70)= 0.4850583435620618d-10
 rk( 70)=-0.3749360967397995d-09+(azi- 0.60d+02) * rk( 70)
 rk( 70)=-0.1033972224682254d-06+(azi- 0.60d+02) * rk( 70)
 rk( 70)= 0.2308200000000000d-05+(azi- 0.60d+02) * rk( 70)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 70)= 0.4863918163365898d-10
 rk( 70)= 0.1080238933948239d-08+(azi- 0.70d+02) * rk( 70)
 rk( 70)=-0.9634419409614782d-07+(azi- 0.70d+02) * rk( 70)
 rk( 70)= 0.1285240000000000d-05+(azi- 0.70d+02) * rk( 70)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 70)=-0.6023611040521017d-10
 rk( 70)= 0.2247579293157748d-08+(azi- 0.78d+02) * rk( 70)
 rk( 70)=-0.6972164827933041d-07+(azi- 0.78d+02) * rk( 70)
 rk( 70)= 0.6085250000000000d-06+(azi- 0.78d+02) * rk( 70)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 70)= 0.8106335758155240d-09
 rk( 70)= 0.8019126434123756d-09+(azi- 0.86d+02) * rk( 70)
 rk( 70)=-0.4532571278666100d-07+(azi- 0.86d+02) * rk( 70)
 rk( 70)= 0.1637560000000000d-06+(azi- 0.86d+02) * rk( 70)
elseif(azi.ge.90.d0)then
 rk( 70)= 0.4716430000000000d-07
endif
if(att.lt.0.99999) rk( 70) = rk( 70) * att
endif

rk( 71) =  dexp(-0.2564314676777420d+02                           &
 - (  0.6200000000000000d+03 )/temp)
rk( 72) =  0.9000000000000000d-11

if(option_photolysis.eq.2) then
 rk( 73)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 73)=-0.5816837387219519d-10
 rk( 73)=-0.5184316261278087d-08+(azi- 0.00d+00) * rk( 73)
 rk( 73)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 73)
 rk( 73)= 0.3220560000000000d-04+(azi- 0.00d+00) * rk( 73)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 73)= 0.3450512161659949d-10
 rk( 73)=-0.6929367477443941d-08+(azi- 0.10d+02) * rk( 73)
 rk( 73)=-0.1211368373872203d-06+(azi- 0.10d+02) * rk( 73)
 rk( 73)= 0.3162900000000000d-04+(azi- 0.10d+02) * rk( 73)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 73)= 0.3647887405787883d-11
 rk( 73)=-0.5894213828945960d-08+(azi- 0.20d+02) * rk( 73)
 rk( 73)=-0.2493726504511192d-06+(azi- 0.20d+02) * rk( 73)
 rk( 73)= 0.2975920000000000d-04+(azi- 0.20d+02) * rk( 73)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 73)= 0.2530332876025029d-10
 rk( 73)=-0.5784777206772343d-08+(azi- 0.30d+02) * rk( 73)
 rk( 73)=-0.3661625608083018d-06+(azi- 0.30d+02) * rk( 73)
 rk( 73)= 0.2667970000000000d-04+(azi- 0.30d+02) * rk( 73)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 73)= 0.6013879755321277d-10
 rk( 73)=-0.5025677343964861d-08+(azi- 0.40d+02) * rk( 73)
 rk( 73)=-0.4742671063156733d-06+(azi- 0.40d+02) * rk( 73)
 rk( 73)= 0.2246490000000000d-04+(azi- 0.40d+02) * rk( 73)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 73)= 0.1004414810269593d-09
 rk( 73)=-0.3221513417368319d-08+(azi- 0.50d+02) * rk( 73)
 rk( 73)=-0.5567390139290089d-06+(azi- 0.50d+02) * rk( 73)
 rk( 73)= 0.1727980000000000d-04+(azi- 0.50d+02) * rk( 73)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 73)= 0.3107352783389442d-09
 rk( 73)=-0.2082689865616575d-09+(azi- 0.60d+02) * rk( 73)
 rk( 73)=-0.5910368379682989d-06+(azi- 0.60d+02) * rk( 73)
 rk( 73)= 0.1149070000000000d-04+(azi- 0.60d+02) * rk( 73)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 73)= 0.1345128013872392d-09
 rk( 73)= 0.9113789363612173d-08+(azi- 0.70d+02) * rk( 73)
 rk( 73)=-0.5019816341977565d-06+(azi- 0.70d+02) * rk( 73)
 rk( 73)= 0.5870240000000000d-05+(azi- 0.70d+02) * rk( 73)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 73)=-0.1960445509611151d-09
 rk( 73)= 0.1234209659691269d-07+(azi- 0.78d+02) * rk( 73)
 rk( 73)=-0.3303345465135304d-06+(azi- 0.78d+02) * rk( 73)
 rk( 73)= 0.2506540000000000d-05+(azi- 0.78d+02) * rk( 73)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 73)= 0.2279277828298341d-08
 rk( 73)= 0.7637027373778166d-08+(azi- 0.86d+02) * rk( 73)
 rk( 73)=-0.1705015547476783d-06+(azi- 0.86d+02) * rk( 73)
 rk( 73)= 0.5533830000000000d-06+(azi- 0.86d+02) * rk( 73)
elseif(azi.ge.90.d0)then
 rk( 73)= 0.1394430000000000d-06
endif
if(att.lt.0.99999) rk( 73) = rk( 73) * att
endif


if(option_photolysis.eq.2) then
 rk( 74)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 74)=-0.7261045436751545d-10
 rk( 74)=-0.5576895456324842d-08+(azi- 0.00d+00) * rk( 74)
 rk( 74)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 74)
 rk( 74)= 0.4669460000000000d-04+(azi- 0.00d+00) * rk( 74)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 74)= 0.2853136310254372d-10
 rk( 74)=-0.7755209087350302d-08+(azi- 0.10d+02) * rk( 74)
 rk( 74)=-0.1333210454367515d-06+(azi- 0.10d+02) * rk( 74)
 rk( 74)= 0.4606430000000000d-04+(azi- 0.10d+02) * rk( 74)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 74)=-0.1901499804265726d-10
 rk( 74)=-0.6899268194273995d-08+(azi- 0.20d+02) * rk( 74)
 rk( 74)=-0.2798658182529944d-06+(azi- 0.20d+02) * rk( 74)
 rk( 74)= 0.4398410000000000d-04+(azi- 0.20d+02) * rk( 74)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 74)=-0.9471370931908783d-11
 rk( 74)=-0.7469718135553710d-08+(azi- 0.30d+02) * rk( 74)
 rk( 74)=-0.4235556815512711d-06+(azi- 0.30d+02) * rk( 74)
 rk( 74)= 0.4047650000000000d-04+(azi- 0.30d+02) * rk( 74)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 74)= 0.1440048177028887d-10
 rk( 74)=-0.7753859263510966d-08+(azi- 0.40d+02) * rk( 74)
 rk( 74)=-0.5757914555419174d-06+(azi- 0.40d+02) * rk( 74)
 rk( 74)= 0.3548450000000000d-04+(azi- 0.40d+02) * rk( 74)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 74)= 0.3856944385081257d-10
 rk( 74)=-0.7321844810402538d-08+(azi- 0.50d+02) * rk( 74)
 rk( 74)=-0.7265484962810543d-06+(azi- 0.50d+02) * rk( 74)
 rk( 74)= 0.2896560000000000d-04+(azi- 0.50d+02) * rk( 74)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 74)= 0.4136217428262900d-09
 rk( 74)=-0.6164761494878585d-08+(azi- 0.60d+02) * rk( 74)
 rk( 74)=-0.8614145593338596d-06+(azi- 0.60d+02) * rk( 74)
 rk( 74)= 0.2100650000000000d-04+(azi- 0.60d+02) * rk( 74)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 74)= 0.5376701572532502d-09
 rk( 74)= 0.6243890789914774d-08+(azi- 0.70d+02) * rk( 74)
 rk( 74)=-0.8606232663835604d-06+(azi- 0.70d+02) * rk( 74)
 rk( 74)= 0.1218950000000000d-04+(azi- 0.70d+02) * rk( 74)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 74)=-0.5508273899935273d-09
 rk( 74)= 0.1914797456399955d-07+(azi- 0.78d+02) * rk( 74)
 rk( 74)=-0.6574883435524898d-06+(azi- 0.78d+02) * rk( 74)
 rk( 74)= 0.5979410000000000d-05+(azi- 0.78d+02) * rk( 74)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 74)= 0.8530305661922505d-08
 rk( 74)= 0.5928117204100688d-08+(azi- 0.86d+02) * rk( 74)
 rk( 74)=-0.4568796094072541d-06+(azi- 0.86d+02) * rk( 74)
 rk( 74)= 0.1662950000000000d-05+(azi- 0.86d+02) * rk( 74)
elseif(azi.ge.90.d0)then
 rk( 74)= 0.4762210000000000d-06
endif
if(att.lt.0.99999) rk( 74) = rk( 74) * att
endif

rk( 75) =  dexp(-0.2410466059131239d+02                           &
 - (  0.1600000000000000d+04 )/temp)
rk( 76) =  0.5800000000000000d-15
rk( 77) =  dexp(-0.3226665050940135d+02                           &
 - ( -0.6250000000000000d+03 )/temp)
rk( 78) =  dexp( 0.2850648985328245d+02                           &
 - (  0.7000000000000000d+04 )/temp)
rk( 79) =  0.5600000000000000d-11
rk( 80) =  dexp(-0.3281600979716958d+02                           &
 - ( -0.2300000000000000d+04 )/temp)
rk( 81) =  0.4000000000000000d-12
rk( 82) =  dexp(-0.2474064935803238d+02                           &
 - (  0.1100000000000000d+04 )/temp)
rk( 83) =  dexp(-0.2590825451818744d+02                           &
 - ( -0.2700000000000000d+03 )/temp)
rk( 84) =  dexp(-0.2729454887930734d+02                           &
 - (  0.1900000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk( 85)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 85)=-0.1292568102250478d-10
 rk( 85)=-0.1349143189774952d-08+(azi- 0.00d+00) * rk( 85)
 rk( 85)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 85)
 rk( 85)= 0.6105070000000000d-05+(azi- 0.00d+00) * rk( 85)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 85)= 0.1221704306751377d-10
 rk( 85)=-0.1736913620450095d-08+(azi- 0.10d+02) * rk( 85)
 rk( 85)=-0.3086056810225046d-07+(azi- 0.10d+02) * rk( 85)
 rk( 85)= 0.5957230000000000d-05+(azi- 0.10d+02) * rk( 85)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 85)= 0.7227508752451222d-11
 rk( 85)=-0.1370402328424685d-08+(azi- 0.20d+02) * rk( 85)
 rk( 85)=-0.6193372759099823d-07+(azi- 0.20d+02) * rk( 85)
 rk( 85)= 0.5487150000000000d-05+(azi- 0.20d+02) * rk( 85)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 85)= 0.1521292192268060d-10
 rk( 85)=-0.1153577065851153d-08+(azi- 0.30d+02) * rk( 85)
 rk( 85)=-0.8717352153375648d-07+(azi- 0.30d+02) * rk( 85)
 rk( 85)= 0.4738000000000000d-05+(azi- 0.30d+02) * rk( 85)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 85)= 0.2301080355683006d-10
 rk( 85)=-0.6971894081707449d-09+(azi- 0.40d+02) * rk( 85)
 rk( 85)=-0.1056811862739754d-06+(azi- 0.40d+02) * rk( 85)
 rk( 85)= 0.3766120000000000d-05+(azi- 0.40d+02) * rk( 85)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 85)= 0.3106386384999131d-10
 rk( 85)=-0.6865301465823343d-11+(azi- 0.50d+02) * rk( 85)
 rk( 85)=-0.1127217333703412d-06+(azi- 0.50d+02) * rk( 85)
 rk( 85)= 0.2662600000000000d-05+(azi- 0.50d+02) * rk( 85)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 85)= 0.4531874104319860d-10
 rk( 85)= 0.9250506140339690d-09+(azi- 0.60d+02) * rk( 85)
 rk( 85)=-0.1035398802446585d-06+(azi- 0.60d+02) * rk( 85)
 rk( 85)= 0.1565760000000000d-05+(azi- 0.60d+02) * rk( 85)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 85)=-0.1303761111962380d-10
 rk( 85)= 0.2284612845330880d-08+(azi- 0.70d+02) * rk( 85)
 rk( 85)=-0.7144324565100488d-07+(azi- 0.70d+02) * rk( 85)
 rk( 85)= 0.6681850000000000d-06+(azi- 0.70d+02) * rk( 85)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 85)=-0.4077238229824927d-10
 rk( 85)= 0.1971710178462450d-08+(azi- 0.78d+02) * rk( 85)
 rk( 85)=-0.3739266146067857d-07+(azi- 0.78d+02) * rk( 85)
 rk( 85)= 0.2361790000000000d-06+(azi- 0.78d+02) * rk( 85)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 85)= 0.1193377495821847d-09
 rk( 85)= 0.9931730033112436d-09+(azi- 0.86d+02) * rk( 85)
 rk( 85)=-0.1367359600643481d-07+(azi- 0.86d+02) * rk( 85)
 rk( 85)= 0.4235170000000000d-07+(azi- 0.86d+02) * rk( 85)
elseif(azi.ge.90.d0)then
 rk( 85)= 0.1118570000000000d-07
endif
if(att.lt.0.99999) rk( 85) = rk( 85) * att
endif

rk( 86) =  dexp(-0.2553915705425015d+02                           &
 - ( -0.2700000000000000d+03 )/temp)
effko =  0.2700000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.7100000000000000d+01))
rapk =  0.1200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.9000000000000000d+00))
rk( 87) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.3000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4900000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1210000000000000d+05/temp)
rapk =  0.5400000000000000d+17* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1383000000000000d+05/temp)
rk( 88) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.3000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk( 89)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 89)=-0.5245828038559129d-12
 rk( 89)=-0.1048523196144091d-10+(azi- 0.00d+00) * rk( 89)
 rk( 89)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 89)
 rk( 89)= 0.4777587000000000d-07+(azi- 0.00d+00) * rk( 89)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 89)= 0.2522760411567752d-11
 rk( 89)=-0.2622271607711829d-10+(azi- 0.10d+02) * rk( 89)
 rk( 89)=-0.3670794803855920d-09+(azi- 0.10d+02) * rk( 89)
 rk( 89)= 0.4620276400000000d-07+(azi- 0.10d+02) * rk( 89)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 89)=-0.3810250842415100d-11
 rk( 89)= 0.4946009626991426d-10+(azi- 0.20d+02) * rk( 89)
 rk( 89)=-0.1347056784576322d-09+(azi- 0.20d+02) * rk( 89)
 rk( 89)= 0.4243245800000000d-07+(azi- 0.20d+02) * rk( 89)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 89)= 0.2339466958092650d-11
 rk( 89)=-0.6484742900253869d-10+(azi- 0.30d+02) * rk( 89)
 rk( 89)=-0.2885790057838769d-09+(azi- 0.30d+02) * rk( 89)
 rk( 89)= 0.4222116000000000d-07+(azi- 0.30d+02) * rk( 89)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 89)= 0.1867501004448772d-13
 rk( 89)= 0.5336579740240855d-11+(azi- 0.40d+02) * rk( 89)
 rk( 89)=-0.8836874984068571d-09+(azi- 0.40d+02) * rk( 89)
 rk( 89)= 0.3519009400000000d-07+(azi- 0.40d+02) * rk( 89)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 89)= 0.8226700172943887d-13
 rk( 89)= 0.5896830041575641d-11+(azi- 0.50d+02) * rk( 89)
 rk( 89)=-0.7713534005886921d-09+(azi- 0.50d+02) * rk( 89)
 rk( 89)= 0.2690555200000000d-07+(azi- 0.50d+02) * rk( 89)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 89)=-0.9853030169627836d-12
 rk( 89)= 0.8364840093458807d-11+(azi- 0.60d+02) * rk( 89)
 rk( 89)=-0.6287366992383443d-09+(azi- 0.60d+02) * rk( 89)
 rk( 89)= 0.1986396800000000d-07+(azi- 0.60d+02) * rk( 89)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 89)= 0.1136116301266380d-11
 rk( 89)=-0.2119425041541064d-10+(azi- 0.70d+02) * rk( 89)
 rk( 89)=-0.7570308024579321d-09+(azi- 0.70d+02) * rk( 89)
 rk( 89)= 0.1342778200000000d-07+(azi- 0.70d+02) * rk( 89)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 89)= 0.1853495648302295d-11
 rk( 89)= 0.6072540815002347d-11+(azi- 0.78d+02) * rk( 89)
 rk( 89)=-0.8780044792615161d-09+(azi- 0.78d+02) * rk( 89)
 rk( 89)= 0.6596795100000000d-08+(azi- 0.78d+02) * rk( 89)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 89)= 0.4275243906997995d-12
 rk( 89)= 0.5055643637423095d-10+(azi- 0.86d+02) * rk( 89)
 rk( 89)=-0.4249726617465909d-09+(azi- 0.86d+02) * rk( 89)
 rk( 89)= 0.9103916500000000d-09+(azi- 0.86d+02) * rk( 89)
elseif(azi.ge.90.d0)then
 rk( 89)= 0.4676554600000000d-10
endif
if(att.lt.0.99999) rk( 89) = rk( 89) * att
endif

rk( 90) =  dexp(-0.2847499118622308d+02                           &
 - ( -0.1040000000000000d+04 )/temp)
rk( 91) =  dexp(-0.2693787393536860d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk( 92) =  dexp(-0.2845200166799838d+02                           &
 - ( -0.1070000000000000d+04 )/temp)
rk( 93) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk( 94) =  dexp(-0.2854731184780270d+02                           &
 - ( -0.2000000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 95)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 95)= 0.3439467845737435d-11
 rk( 95)=-0.1604640139173746d-09+(azi- 0.00d+00) * rk( 95)
 rk( 95)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 95)
 rk( 95)= 0.6899761847520000d-06+(azi- 0.00d+00) * rk( 95)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 95)=-0.1682071517212295d-11
 rk( 95)=-0.5727997854525157d-10+(azi- 0.10d+02) * rk( 95)
 rk( 95)=-0.2177439924626261d-08+(azi- 0.10d+02) * rk( 95)
 rk( 95)= 0.6773692512060000d-06+(azi- 0.10d+02) * rk( 95)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 95)=-0.1770967004888075d-11
 rk( 95)=-0.1077421240616206d-09+(azi- 0.20d+02) * rk( 95)
 rk( 95)=-0.3827660950694980d-08+(azi- 0.20d+02) * rk( 95)
 rk( 95)= 0.6481847825880000d-06+(azi- 0.20d+02) * rk( 95)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 95)= 0.5197719474764733d-11
 rk( 95)=-0.1608711342082635d-09+(azi- 0.30d+02) * rk( 95)
 rk( 95)=-0.6513793533393824d-08+(azi- 0.30d+02) * rk( 95)
 rk( 95)= 0.5973629936700000d-06+(azi- 0.30d+02) * rk( 95)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 95)=-0.1925844006171251d-11
 rk( 95)=-0.4939549965316779d-11+(azi- 0.40d+02) * rk( 95)
 rk( 95)=-0.8171900375129673d-08+(azi- 0.40d+02) * rk( 95)
 rk( 95)= 0.5213356643900000d-06+(azi- 0.40d+02) * rk( 95)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 95)= 0.3659679272920429d-11
 rk( 95)=-0.6271487015045512d-10+(azi- 0.50d+02) * rk( 95)
 rk( 95)=-0.8848444576287439d-08+(azi- 0.50d+02) * rk( 95)
 rk( 95)= 0.4371968616360000d-06+(azi- 0.50d+02) * rk( 95)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 95)=-0.1519631773950880d-10
 rk( 95)= 0.4707550803713459d-10+(azi- 0.60d+02) * rk( 95)
 rk( 95)=-0.9004838197420313d-08+(azi- 0.60d+02) * rk( 95)
 rk( 95)= 0.3461006081310000d-06+(azi- 0.60d+02) * rk( 95)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 95)= 0.7012506971232375d-10
 rk( 95)=-0.4088140241481955d-09+(azi- 0.70d+02) * rk( 95)
 rk( 95)=-0.1262222335853076d-07+(azi- 0.70d+02) * rk( 95)
 rk( 95)= 0.2455634592210000d-06+(azi- 0.70d+02) * rk( 95)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 95)=-0.1236221759603351d-09
 rk( 95)= 0.1274187648948739d-08+(azi- 0.78d+02) * rk( 95)
 rk( 95)=-0.5699234360131495d-08+(azi- 0.78d+02) * rk( 95)
 rk( 95)= 0.1543256105000000d-06+(azi- 0.78d+02) * rk( 95)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 95)= 0.4706176323768702d-09
 rk( 95)=-0.1692744574097185d-08+(azi- 0.86d+02) * rk( 95)
 rk( 95)=-0.9047689761312286d-08+(azi- 0.86d+02) * rk( 95)
 rk( 95)= 0.1269851910600000d-06+(azi- 0.86d+02) * rk( 95)
elseif(azi.ge.90.d0)then
 rk( 95)= 0.9383004730129999d-07
endif
if(att.lt.0.99999) rk( 95) = rk( 95) * att
endif

rk( 96) =  dexp(-0.2854731184780270d+02                           &
 - ( -0.2000000000000000d+03 )/temp)
rk( 97) =  dexp(-0.2506607175846701d+02                           &
 - (  0.8700000000000000d+03 )/temp)
rk( 98) =  dexp(-0.2600178057619827d+02                           &
 - ( -0.4050000000000000d+03 )/temp)
rk( 99) =  0.6500000000000000d-14

if(option_photolysis.eq.2) then
 rk(100)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(100)=-0.2543209447272390d-06
 rk(100)=-0.4665430552727591d-05+(azi- 0.00d+00) * rk(100)
 rk(100)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(100)
 rk(100)= 0.2672443200000000d-01+(azi- 0.00d+00) * rk(100)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(100)= 0.1160138834181708d-05
 rk(100)=-0.1229505889454476d-04+(azi- 0.10d+02) * rk(100)
 rk(100)=-0.1696048944727235d-03+(azi- 0.10d+02) * rk(100)
 rk(100)= 0.2600356800000000d-01+(azi- 0.10d+02) * rk(100)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(100)=-0.1797915391999585d-05
 rk(100)= 0.2250910613090648d-04+(azi- 0.20d+02) * rk(100)
 rk(100)=-0.6746442210910607d-04+(azi- 0.20d+02) * rk(100)
 rk(100)= 0.2423815200000000d-01+(azi- 0.20d+02) * rk(100)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(100)= 0.1113225733816626d-05
 rk(100)=-0.3142835562908105d-04+(azi- 0.30d+02) * rk(100)
 rk(100)=-0.1566569170908517d-03+(azi- 0.30d+02) * rk(100)
 rk(100)= 0.2401650300000000d-01+(azi- 0.30d+02) * rk(100)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(100)=-0.4892354326693738d-07
 rk(100)= 0.1968416385417769d-05+(azi- 0.40d+02) * rk(100)
 rk(100)=-0.4512563095274845d-03+(azi- 0.40d+02) * rk(100)
 rk(100)= 0.2042032400000000d-01+(azi- 0.40d+02) * rk(100)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(100)= 0.1418524392511286d-06
 rk(100)= 0.5007100874097285d-06+(azi- 0.50d+02) * rk(100)
 rk(100)=-0.4265650447992115d-03+(azi- 0.50d+02) * rk(100)
 rk(100)= 0.1605567900000000d-01+(azi- 0.50d+02) * rk(100)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(100)=-0.6037011137374412d-06
 rk(100)= 0.4756283264942610d-05+(azi- 0.60d+02) * rk(100)
 rk(100)=-0.3739951112756851d-03+(azi- 0.60d+02) * rk(100)
 rk(100)= 0.1198195200000000d-01+(azi- 0.60d+02) * rk(100)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(100)= 0.7324219730526701d-06
 rk(100)=-0.1335475014717716d-04+(azi- 0.70d+02) * rk(100)
 rk(100)=-0.4599797800980132d-03+(azi- 0.70d+02) * rk(100)
 rk(100)= 0.8113928100000001d-02+(azi- 0.70d+02) * rk(100)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(100)= 0.1115677554341943d-05
 rk(100)= 0.4223377206111212d-05+(azi- 0.78d+02) * rk(100)
 rk(100)=-0.5330307636267073d-03+(azi- 0.78d+02) * rk(100)
 rk(100)= 0.3954385900000000d-02+(azi- 0.78d+02) * rk(100)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(100)= 0.6769853777566226d-07
 rk(100)= 0.3099963851027621d-04+(azi- 0.86d+02) * rk(100)
 rk(100)=-0.2512466378952194d-03+(azi- 0.86d+02) * rk(100)
 rk(100)= 0.5316628400000000d-03+(azi- 0.86d+02) * rk(100)
elseif(azi.ge.90.d0)then
 rk(100)= 0.2700321100000000d-04
endif
if(att.lt.0.99999) rk(100) = rk(100) * att
endif

rk(101) =  dexp(-0.2572891358953163d+02                           &
 - ( -0.3400000000000000d+03 )/temp)
effko =  0.2700000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.7100000000000000d+01))
rapk =  0.1200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.9000000000000000d+00))
rk(102) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.3000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4900000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1210000000000000d+05/temp)
rapk =  0.5400000000000000d+17* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1383000000000000d+05/temp)
rk(103) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.3000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk(104)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(104)=-0.5245828038559129d-12
 rk(104)=-0.1048523196144091d-10+(azi- 0.00d+00) * rk(104)
 rk(104)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(104)
 rk(104)= 0.4777587000000000d-07+(azi- 0.00d+00) * rk(104)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(104)= 0.2522760411567752d-11
 rk(104)=-0.2622271607711829d-10+(azi- 0.10d+02) * rk(104)
 rk(104)=-0.3670794803855920d-09+(azi- 0.10d+02) * rk(104)
 rk(104)= 0.4620276400000000d-07+(azi- 0.10d+02) * rk(104)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(104)=-0.3810250842415100d-11
 rk(104)= 0.4946009626991426d-10+(azi- 0.20d+02) * rk(104)
 rk(104)=-0.1347056784576322d-09+(azi- 0.20d+02) * rk(104)
 rk(104)= 0.4243245800000000d-07+(azi- 0.20d+02) * rk(104)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(104)= 0.2339466958092650d-11
 rk(104)=-0.6484742900253869d-10+(azi- 0.30d+02) * rk(104)
 rk(104)=-0.2885790057838769d-09+(azi- 0.30d+02) * rk(104)
 rk(104)= 0.4222116000000000d-07+(azi- 0.30d+02) * rk(104)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(104)= 0.1867501004448772d-13
 rk(104)= 0.5336579740240855d-11+(azi- 0.40d+02) * rk(104)
 rk(104)=-0.8836874984068571d-09+(azi- 0.40d+02) * rk(104)
 rk(104)= 0.3519009400000000d-07+(azi- 0.40d+02) * rk(104)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(104)= 0.8226700172943887d-13
 rk(104)= 0.5896830041575641d-11+(azi- 0.50d+02) * rk(104)
 rk(104)=-0.7713534005886921d-09+(azi- 0.50d+02) * rk(104)
 rk(104)= 0.2690555200000000d-07+(azi- 0.50d+02) * rk(104)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(104)=-0.9853030169627836d-12
 rk(104)= 0.8364840093458807d-11+(azi- 0.60d+02) * rk(104)
 rk(104)=-0.6287366992383443d-09+(azi- 0.60d+02) * rk(104)
 rk(104)= 0.1986396800000000d-07+(azi- 0.60d+02) * rk(104)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(104)= 0.1136116301266380d-11
 rk(104)=-0.2119425041541064d-10+(azi- 0.70d+02) * rk(104)
 rk(104)=-0.7570308024579321d-09+(azi- 0.70d+02) * rk(104)
 rk(104)= 0.1342778200000000d-07+(azi- 0.70d+02) * rk(104)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(104)= 0.1853495648302295d-11
 rk(104)= 0.6072540815002347d-11+(azi- 0.78d+02) * rk(104)
 rk(104)=-0.8780044792615161d-09+(azi- 0.78d+02) * rk(104)
 rk(104)= 0.6596795100000000d-08+(azi- 0.78d+02) * rk(104)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(104)= 0.4275243906997995d-12
 rk(104)= 0.5055643637423095d-10+(azi- 0.86d+02) * rk(104)
 rk(104)=-0.4249726617465909d-09+(azi- 0.86d+02) * rk(104)
 rk(104)= 0.9103916500000000d-09+(azi- 0.86d+02) * rk(104)
elseif(azi.ge.90.d0)then
 rk(104)= 0.4676554600000000d-10
endif
if(att.lt.0.99999) rk(104) = rk(104) * att
endif

rk(105) =  0.3000000000000000d-12
rk(106) =  dexp(-0.2847499118622308d+02                           &
 - ( -0.1040000000000000d+04 )/temp)
rk(107) =  dexp(-0.2693787393536860d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(108) =  dexp(-0.2845200166799838d+02                           &
 - ( -0.1070000000000000d+04 )/temp)
rk(109) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(110) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(111) =  0.8100000000000000d-12
rk(112) =  dexp( 0.3453877639491068d+02                           &
 - (  0.8000000000000000d+04 )/temp)
rk(113) =  0.1600000000000000d+04
rk(114) =  0.1500000000000000d-10
rk(115) =  dexp(-0.2531848569208134d+02                           &
 - (  0.2800000000000000d+03 )/temp)
rk(116) =  0.3200000000000000d-10
rk(117) =  dexp(-0.3266697421800909d+02                           &
 - (  0.1900000000000000d+04 )/temp)
rk(118) =  dexp(-0.2798769605986728d+02                           &
 - (  0.2160000000000000d+04 )/temp)
rk(119) =  dexp(-0.2499196378631329d+02                           &
 - (  0.7920000000000000d+03 )/temp)
effko =  0.1000000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.8000000000000000d+00))
rapk =  0.8800000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(120) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(121) =  dexp(-0.3205386974512268d+02                           &
 - (  0.2630000000000000d+04 )/temp)
rk(122) =  dexp(-0.2643709864745611d+02                           &
 - (  0.2880000000000000d+04 )/temp)
rk(123) =  0.2300000000000000d-10
rk(124) =  dexp(-0.2532843602293450d+02                           &
 - ( -0.5500000000000000d+03 )/temp)
rk(125) =  dexp(-0.3241054468906142d+02                           &
 - (  0.1100000000000000d+04 )/temp)
rk(126) =  dexp(-0.2767184311044880d+02                           &
 - (  0.2700000000000000d+03 )/temp)
rk(127) =  dexp(-0.2704323445102643d+02                           &
 - ( -0.3550000000000000d+03 )/temp)
rk(128) =  0.8100000000000000d-11
rk(129) =  0.4200000000000000d+01
rk(130) =  0.4100000000000000d-10
rk(131) =  0.2200000000000000d-10
rk(132) =  0.1400000000000000d-10
rk(133) =  0.5500000000000000d-11

if(option_photolysis.eq.2) then
 rk(134)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(134)=-0.8584948998100693d-09
 rk(134)=-0.6414505100189953d-07+(azi- 0.00d+00) * rk(134)
 rk(134)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(134)
 rk(134)= 0.5609260000000000d-03+(azi- 0.00d+00) * rk(134)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(134)= 0.3254846994303392d-09
 rk(134)=-0.8989989799620162d-07+(azi- 0.10d+02) * rk(134)
 rk(134)=-0.1540449489981011d-05+(azi- 0.10d+02) * rk(134)
 rk(134)= 0.5536530000000000d-03+(azi- 0.10d+02) * rk(134)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(134)=-0.2454438979115351d-09
 rk(134)=-0.8013535701329145d-07+(azi- 0.20d+02) * rk(134)
 rk(134)=-0.3240802040075941d-05+(azi- 0.20d+02) * rk(134)
 rk(134)= 0.5295840000000000d-03+(azi- 0.20d+02) * rk(134)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(134)=-0.1557091077839579d-09
 rk(134)=-0.8749867395063711d-07+(azi- 0.30d+02) * rk(134)
 rk(134)=-0.4917142349715225d-05+(azi- 0.30d+02) * rk(134)
 rk(134)= 0.4889170000000000d-03+(azi- 0.30d+02) * rk(134)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(134)= 0.5928032904741575d-10
 rk(134)=-0.9216994718415690d-07+(azi- 0.40d+02) * rk(134)
 rk(134)=-0.6713828561063159d-05+(azi- 0.40d+02) * rk(134)
 rk(134)= 0.4308400000000000d-03+(azi- 0.40d+02) * rk(134)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(134)= 0.2595877915944963d-09
 rk(134)=-0.9039153731273486d-07+(azi- 0.50d+02) * rk(134)
 rk(134)=-0.8539443406032087d-05+(azi- 0.50d+02) * rk(134)
 rk(134)= 0.3545440000000000d-03+(azi- 0.50d+02) * rk(134)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(134)= 0.4590368504578084d-08
 rk(134)=-0.8260390356490844d-07+(azi- 0.60d+02) * rk(134)
 rk(134)=-0.1026939781480858d-04+(azi- 0.60d+02) * rk(134)
 rk(134)= 0.2603700000000000d-03+(azi- 0.60d+02) * rk(134)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(134)= 0.7530009721131077d-08
 rk(134)= 0.5510715157249846d-07+(azi- 0.70d+02) * rk(134)
 rk(134)=-0.1054436533473309d-04+(azi- 0.70d+02) * rk(134)
 rk(134)= 0.1540060000000000d-03+(azi- 0.70d+02) * rk(134)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(134)=-0.7917656811272118d-08
 rk(134)= 0.2358273848796985d-06+(azi- 0.78d+02) * rk(134)
 rk(134)=-0.8216889043116161d-05+(azi- 0.78d+02) * rk(134)
 rk(134)= 0.7703330000000000d-04+(azi- 0.78d+02) * rk(134)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(134)= 0.1166127504485630d-06
 rk(134)= 0.4580362140873401d-07+(azi- 0.86d+02) * rk(134)
 rk(134)=-0.5963840992804365d-05+(azi- 0.86d+02) * rk(134)
 rk(134)= 0.2233730000000000d-04+(azi- 0.86d+02) * rk(134)
elseif(azi.ge.90.d0)then
 rk(134)= 0.6678010000000000d-05
endif
if(att.lt.0.99999) rk(134) = rk(134) * att
endif

rk(135) =  0.3000000000000000d-10
rk(136) =  dexp(-0.3745754762732854d+02                           &
 - (  0.5000000000000000d+03 )/temp)
rk(137) =  dexp(-0.2479780777187233d+02                           &
 - ( -0.1160000000000000d+03 )/temp)
rk(138) =  0.1700000000000000d-10

if(option_photolysis.eq.2) then
 rk(139)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(139)=-0.1826021023809213d-09
 rk(139)=-0.1228397897619082d-07+(azi- 0.00d+00) * rk(139)
 rk(139)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(139)
 rk(139)= 0.1204770000000000d-03+(azi- 0.00d+00) * rk(139)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(139)= 0.7680630714277358d-10
 rk(139)=-0.1776204204761847d-07+(azi- 0.10d+02) * rk(139)
 rk(139)=-0.3004602102380929d-06+(azi- 0.10d+02) * rk(139)
 rk(139)= 0.1190660000000000d-03+(azi- 0.10d+02) * rk(139)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(139)=-0.4662312619017368d-10
 rk(139)=-0.1545785283333527d-07+(azi- 0.20d+02) * rk(139)
 rk(139)=-0.6326591590476302d-06+(azi- 0.20d+02) * rk(139)
 rk(139)= 0.1143620000000000d-03+(azi- 0.20d+02) * rk(139)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(139)=-0.3391380238206294d-10
 rk(139)=-0.1685654661904045d-07+(azi- 0.30d+02) * rk(139)
 rk(139)=-0.9558031535713878d-06+(azi- 0.30d+02) * rk(139)
 rk(139)= 0.1064430000000000d-03+(azi- 0.30d+02) * rk(139)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(139)=-0.3482166428159112d-10
 rk(139)=-0.1787396069050238d-07+(azi- 0.40d+02) * rk(139)
 rk(139)=-0.1303108226666818d-05+(azi- 0.40d+02) * rk(139)
 rk(139)= 0.9516540000000000d-04+(azi- 0.40d+02) * rk(139)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(139)=-0.9419954049165213d-10
 rk(139)=-0.1891861061895016d-07+(azi- 0.50d+02) * rk(139)
 rk(139)=-0.1671033939761339d-05+(azi- 0.50d+02) * rk(139)
 rk(139)= 0.8031210000000001d-04+(azi- 0.50d+02) * rk(139)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(139)= 0.4307198262473663d-09
 rk(139)=-0.2174459683369613d-07+(azi- 0.60d+02) * rk(139)
 rk(139)=-0.2077666014287802d-05+(azi- 0.60d+02) * rk(139)
 rk(139)= 0.6161569999999999d-04+(azi- 0.60d+02) * rk(139)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(139)= 0.1932828429018806d-08
 rk(139)=-0.8823002046250573d-08+(azi- 0.70d+02) * rk(139)
 rk(139)=-0.2383342003087316d-05+(azi- 0.70d+02) * rk(139)
 rk(139)= 0.3909530000000000d-04+(azi- 0.70d+02) * rk(139)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(139)= 0.7078805396671869d-10
 rk(139)= 0.3756488025025498d-07+(azi- 0.78d+02) * rk(139)
 rk(139)=-0.2153406977455823d-05+(azi- 0.78d+02) * rk(139)
 rk(139)= 0.2045350000000000d-04+(azi- 0.78d+02) * rk(139)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(139)= 0.2551390080685143d-07
 rk(139)= 0.3926379354523939d-07+(azi- 0.86d+02) * rk(139)
 rk(139)=-0.1538777587090784d-05+(azi- 0.86d+02) * rk(139)
 rk(139)= 0.5666640000000000d-05+(azi- 0.86d+02) * rk(139)
elseif(azi.ge.90.d0)then
 rk(139)= 0.1772640000000000d-05
endif
if(att.lt.0.99999) rk(139) = rk(139) * att
endif

rk(140) =  0.3600000000000000d-10
rk(141) =  dexp(-0.2439627194190406d+02                           &
 - ( -0.4076000000000000d+03 )/temp)
rk(142) =  dexp(-0.3247698978846957d+02                           &
 - (  0.1912000000000000d+04 )/temp)
rk(143) =  dexp(-0.2652245849640727d+02                           &
 - (  0.4480000000000000d+03 )/temp)
rk(144) =  0.3360000000000000d-10
rk(145) =  0.7100000000000001d-17
rk(146) =  0.1000000000000000d-14

if(option_photolysis.eq.2) then
 rk(147)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(147)=-0.2724056225766411d-11
 rk(147)=-0.2068594377423350d-09+(azi- 0.00d+00) * rk(147)
 rk(147)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(147)
 rk(147)= 0.1769770000000000d-05+(azi- 0.00d+00) * rk(147)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(147)= 0.1042168677298912d-11
 rk(147)=-0.2885811245153273d-09+(azi- 0.10d+02) * rk(147)
 rk(147)=-0.4954405622576623d-08+(azi- 0.10d+02) * rk(147)
 rk(147)= 0.1746360000000000d-05+(azi- 0.10d+02) * rk(147)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(147)=-0.7346184834290850d-12
 rk(147)=-0.2573160641963604d-09+(azi- 0.20d+02) * rk(147)
 rk(147)=-0.1041337750969350d-07+(azi- 0.20d+02) * rk(147)
 rk(147)= 0.1669000000000000d-05+(azi- 0.20d+02) * rk(147)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(147)=-0.4136947435822000d-12
 rk(147)=-0.2793546186992334d-09+(azi- 0.30d+02) * rk(147)
 rk(147)=-0.1578008433864942d-07+(azi- 0.30d+02) * rk(147)
 rk(147)= 0.1538400000000000d-05+(azi- 0.30d+02) * rk(147)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(147)= 0.3593974577559286d-12
 rk(147)=-0.2917654610066926d-09+(azi- 0.40d+02) * rk(147)
 rk(147)=-0.2149128513570870d-07+(azi- 0.40d+02) * rk(147)
 rk(147)= 0.1352250000000000d-05+(azi- 0.40d+02) * rk(147)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(147)= 0.1158104912559730d-11
 rk(147)=-0.2809835372740098d-09+(azi- 0.50d+02) * rk(147)
 rk(147)=-0.2721877511851595d-07+(azi- 0.50d+02) * rk(147)
 rk(147)= 0.1108520000000000d-05+(azi- 0.50d+02) * rk(147)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(147)= 0.1481218289200588d-10
 rk(147)=-0.2462403898971914d-09+(azi- 0.60d+02) * rk(147)
 rk(147)=-0.3249101439022863d-07+(azi- 0.60d+02) * rk(147)
 rk(147)= 0.8093920000000000d-06+(azi- 0.60d+02) * rk(147)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(147)= 0.2251236790098139d-10
 rk(147)= 0.1981250968631836d-09+(azi- 0.70d+02) * rk(147)
 rk(147)=-0.3297216732056966d-07+(azi- 0.70d+02) * rk(147)
 rk(147)= 0.4746700000000000d-06+(azi- 0.70d+02) * rk(147)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(147)=-0.2282006684570455d-10
 rk(147)= 0.7384219264865253d-09+(azi- 0.78d+02) * rk(147)
 rk(147)=-0.2547979113377114d-07+(azi- 0.78d+02) * rk(147)
 rk(147)= 0.2350990000000000d-06+(azi- 0.78d+02) * rk(147)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(147)= 0.3441785534773510d-09
 rk(147)= 0.1907403221845338d-09+(azi- 0.86d+02) * rk(147)
 rk(147)=-0.1804649314436201d-07+(azi- 0.86d+02) * rk(147)
 rk(147)= 0.6683580000000001d-07+(azi- 0.86d+02) * rk(147)
elseif(azi.ge.90.d0)then
 rk(147)= 0.1972910000000000d-07
endif
if(att.lt.0.99999) rk(147) = rk(147) * att
endif

rk(148) =  0.3600000000000000d-10
rk(149) =  dexp(-0.2492297091482634d+02                           &
 - ( -0.4490000000000000d+03 )/temp)
rk(150) =  dexp(-0.3435645483811673d+02                           &
 - (  0.8210000000000000d+03 )/temp)
rk(151) =  dexp(-0.2632268829627837d+02                           &
 - ( -0.1750000000000000d+03 )/temp)
effko =  0.3000000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.3300000000000000d+01))
rapk =  0.1500000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(152) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(153) =  dexp(-0.2569949970432533d+02                           &
 - (  0.2300000000000000d+03 )/temp)
rk(154) =  dexp(-0.2546769809026801d+02                           &
 - (  0.1070000000000000d+04 )/temp)
rk(155) =  0.1500000000000000d-18
rk(156) =  dexp(-0.2429169913798448d+02                           &
 - (  0.2730000000000000d+03 )/temp)
rk(157) =  dexp(-0.2513781566332585d+02                           &
 - ( -0.4440000000000000d+03 )/temp)
rk(158) =  dexp(-0.3523192357547063d+02                           &
 - (  0.5300000000000000d+03 )/temp)
rk(159) =  dexp(-0.2389335149764518d+02                           &
 - ( -0.4010000000000000d+03 )/temp)
rk(160) =  dexp(-0.3345697122455896d+02                           &
 - (  0.7830000000000000d+03 )/temp)
rk(161) =  dexp(-0.2745706780880511d+02                           &
 - ( -0.4900000000000000d+03 )/temp)
rk(162) =  0.1220000000000000d-10

return
end subroutine kinetic_siream


!===============================================================================

!> fexchem_siream
!> \brief Computation of the chemical production terms
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name              role
!------------------------------------------------------------------------------
!> \param[in]     nr                total number of chemical reactions
!> \param[in]     ns                total number of chemical species
!> \param[in]     y                 concentrations vector
!> \param[in]     rk                kinetic rates
!> \param[in]     zcsourc           source term
!> \param[in]     convers_factor    conversion factors
!> \param[out]    chem              chemical production terms for every species
!______________________________________________________________________________

subroutine fexchem_siream(ns,nr,y,rk,zcsourc,convers_factor,chem)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns),chem(ns),zcsourc(ns)
double precision convers_factor(ns)

! Local variables

integer i
double precision w(nr)
double precision conc(ns)

do i=1,ns
 chem(i)=0.d0
enddo

!     Conversion mug/m3 to molecules/cm3.

do i = 1, ns
   conc(i) = y(i) * convers_factor(i)
enddo

!     Compute reaction rates.

call rates_siream(ns,nr,rk,conc,w)

!     Chemical production terms.

chem( 57) = chem( 57) + w(  1)
chem( 64) = chem( 64) + w(  1)
chem( 65) = chem( 65) - w(  1)
chem( 57) = chem( 57) - w(  2)
chem( 59) = chem( 59) + w(  2)
chem( 59) = chem( 59) - w(  3)
chem( 64) = chem( 64) - w(  3)
chem( 65) = chem( 65) + w(  3)
chem( 57) = chem( 57) - w(  4)
chem( 64) = chem( 64) + w(  4)
chem( 65) = chem( 65) - w(  4)
chem( 56) = chem( 56) + w(  5)
chem( 57) = chem( 57) - w(  5)
chem( 65) = chem( 65) - w(  5)
chem( 57) = chem( 57) - w(  6)
chem( 64) = chem( 64) - w(  6)
chem( 65) = chem( 65) + w(  6)
chem( 56) = chem( 56) + w(  7)
chem( 59) = chem( 59) - w(  7)
chem( 65) = chem( 65) - w(  7)
chem( 57) = chem( 57) + w(  8)
chem( 59) = chem( 59) - w(  8)
chem(  3) = chem(  3) + w(  9)
chem( 59) = chem( 59) - w(  9)
chem(  3) = chem(  3) - w( 10)
chem( 57) = chem( 57) + w( 10)
chem(  3) = chem(  3) - w( 11)
chem( 55) = chem( 55) +  0.2000000000000000d+01 * w( 11)
chem( 55) = chem( 55) - w( 12)
chem( 59) = chem( 59) - w( 12)
chem( 61) = chem( 61) + w( 12)
chem( 55) = chem( 55) + w( 13)
chem( 59) = chem( 59) - w( 13)
chem( 61) = chem( 61) - w( 13)
chem( 56) = chem( 56) - w( 14)
chem( 57) = chem( 57) + w( 14)
chem( 65) = chem( 65) + w( 14)
chem( 56) = chem( 56) - w( 15)
chem( 64) = chem( 64) + w( 15)
chem( 56) = chem( 56) - w( 16)
chem( 64) = chem( 64) - w( 16)
chem( 65) = chem( 65) +  0.2000000000000000d+01 * w( 16)
chem( 56) = chem( 56) - w( 17)
chem( 64) = chem( 64) + w( 17)
chem( 16) = chem( 16) + w( 18)
chem( 56) = chem( 56) - w( 18)
chem( 65) = chem( 65) - w( 18)
chem( 16) = chem( 16) - w( 19)
chem( 50) = chem( 50) +  0.2000000000000000d+01 * w( 19)
chem( 16) = chem( 16) - w( 20)
chem( 56) = chem( 56) + w( 20)
chem( 65) = chem( 65) + w( 20)
chem( 64) = chem( 64) -  0.2000000000000000d+01 * w( 21)
chem( 65) = chem( 65) +  0.2000000000000000d+01 * w( 21)
chem( 23) = chem( 23) +  0.2000000000000000d+01 * w( 22)
chem( 64) = chem( 64) - w( 22)
chem( 65) = chem( 65) - w( 22)
chem( 23) = chem( 23) + w( 23)
chem( 55) = chem( 55) - w( 23)
chem( 64) = chem( 64) - w( 23)
chem( 23) = chem( 23) - w( 24)
chem( 55) = chem( 55) + w( 24)
chem( 64) = chem( 64) + w( 24)
chem( 23) = chem( 23) - w( 25)
chem( 55) = chem( 55) - w( 25)
chem( 65) = chem( 65) + w( 25)
chem( 23) = chem( 23) -  0.2000000000000000d+01 * w( 26)
chem( 64) = chem( 64) + w( 26)
chem( 65) = chem( 65) + w( 26)
chem( 50) = chem( 50) + w( 27)
chem( 55) = chem( 55) - w( 27)
chem( 65) = chem( 65) - w( 27)
chem( 50) = chem( 50) - w( 28)
chem( 55) = chem( 55) - w( 28)
chem( 56) = chem( 56) + w( 28)
chem( 55) = chem( 55) + w( 29)
chem( 61) = chem( 61) - w( 29)
chem( 64) = chem( 64) - w( 29)
chem( 65) = chem( 65) + w( 29)
chem( 17) = chem( 17) + w( 30)
chem( 61) = chem( 61) - w( 30)
chem( 65) = chem( 65) - w( 30)
chem( 17) = chem( 17) - w( 31)
chem( 61) = chem( 61) + w( 31)
chem( 65) = chem( 65) + w( 31)
chem( 17) = chem( 17) - w( 32)
chem( 55) = chem( 55) - w( 32)
chem( 65) = chem( 65) + w( 32)
chem( 20) = chem( 20) + w( 33)
chem( 61) = chem( 61) -  0.2000000000000000d+01 * w( 33)
chem( 20) = chem( 20) + w( 34)
chem( 61) = chem( 61) -  0.2000000000000000d+01 * w( 34)
chem( 20) = chem( 20) - w( 35)
chem( 55) = chem( 55) +  0.2000000000000000d+01 * w( 35)
chem( 20) = chem( 20) - w( 36)
chem( 55) = chem( 55) - w( 36)
chem( 61) = chem( 61) + w( 36)
chem(  3) = chem(  3) - w( 37)
chem( 55) = chem( 55) + w( 37)
chem( 61) = chem( 61) + w( 37)
chem( 55) = chem( 55) - w( 38)
chem( 61) = chem( 61) + w( 38)
chem( 55) = chem( 55) - w( 39)
chem( 57) = chem( 57) - w( 39)
chem( 61) = chem( 61) + w( 39)
chem( 55) = chem( 55) -  0.2000000000000000d+01 * w( 40)
chem( 57) = chem( 57) + w( 40)
chem( 20) = chem( 20) + w( 41)
chem( 55) = chem( 55) -  0.2000000000000000d+01 * w( 41)
chem( 55) = chem( 55) - w( 42)
chem( 61) = chem( 61) - w( 42)
chem( 55) = chem( 55) + w( 43)
chem( 57) = chem( 57) - w( 43)
chem( 61) = chem( 61) - w( 43)
chem( 20) = chem( 20) - w( 44)
chem( 55) = chem( 55) + w( 44)
chem( 57) = chem( 57) - w( 44)
chem( 61) = chem( 61) + w( 44)
chem( 56) = chem( 56) - w( 45)
chem( 57) = chem( 57) - w( 45)
chem( 65) = chem( 65) + w( 45)
chem( 55) = chem( 55) - w( 46)
chem( 56) = chem( 56) - w( 46)
chem( 61) = chem( 61) + w( 46)
chem( 65) = chem( 65) + w( 46)
chem( 50) = chem( 50) + w( 47)
chem( 56) = chem( 56) - w( 47)
chem( 61) = chem( 61) - w( 47)
chem( 56) = chem( 56) - w( 48)
chem( 59) = chem( 59) - w( 48)
chem( 65) = chem( 65) + w( 48)
chem( 56) = chem( 56) -  0.2000000000000000d+01 * w( 49)
chem( 65) = chem( 65) +  0.2000000000000000d+01 * w( 49)
chem( 17) = chem( 17) - w( 50)
chem( 55) = chem( 55) +  0.3900000000000000d+00 * w( 50)
chem( 56) = chem( 56) +  0.3900000000000000d+00 * w( 50)
chem( 61) = chem( 61) +  0.6100000000000000d+00 * w( 50)
chem( 65) = chem( 65) +  0.6100000000000000d+00 * w( 50)
chem( 50) = chem( 50) - w( 51)
chem( 55) = chem( 55) + w( 51)
chem( 65) = chem( 65) + w( 51)
chem( 16) = chem( 16) - w( 52)
chem( 56) = chem( 56) + w( 52)
chem( 65) = chem( 65) + w( 52)
chem( 63) = chem( 63) - w( 53)
chem( 64) = chem( 64) - w( 53)
chem( 65) = chem( 65) + w( 53)
chem( 46) = chem( 46) + w( 54)
chem( 52) = chem( 52) - w( 54)
chem( 64) = chem( 64) - w( 54)
chem( 32) = chem( 32) + w( 55)
chem( 61) = chem( 61) - w( 55)
chem( 63) = chem( 63) - w( 55)
chem( 32) = chem( 32) + w( 56)
chem( 52) = chem( 52) - w( 56)
chem( 61) = chem( 61) - w( 56)
chem( 63) = chem( 63) -  0.2000000000000000d+01 * w( 57)
chem( 52) = chem( 52) -  0.2000000000000000d+01 * w( 58)
chem( 52) = chem( 52) - w( 59)
chem( 63) = chem( 63) - w( 59)
chem( 46) = chem( 46) - w( 60)
chem( 47) = chem( 47) +  0.6600000000000000d+00 * w( 60)
chem( 48) = chem( 48) +  0.3300000000000000d+00 * w( 60)
chem( 50) = chem( 50) + w( 60)
chem( 54) = chem( 54) +  0.3300000000000000d+00 * w( 60)
chem( 55) = chem( 55) - w( 60)
chem( 60) = chem( 60) +  0.3300000000000000d+00 * w( 60)
chem( 61) = chem( 61) + w( 60)
chem( 46) = chem( 46) - w( 61)
chem( 47) = chem( 47) +  0.6600000000000000d+00 * w( 61)
chem( 48) = chem( 48) +  0.3300000000000000d+00 * w( 61)
chem( 54) = chem( 54) +  0.3300000000000000d+00 * w( 61)
chem( 60) = chem( 60) +  0.3300000000000000d+00 * w( 61)
chem( 61) = chem( 61) + w( 61)
chem( 65) = chem( 65) + w( 61)
chem( 32) = chem( 32) - w( 62)
chem( 48) = chem( 48) +  0.5000000000000000d+00 * w( 62)
chem( 55) = chem( 55) - w( 62)
chem( 60) = chem( 60) +  0.5000000000000000d+00 * w( 62)
chem( 63) = chem( 63) + w( 62)
chem( 32) = chem( 32) - w( 63)
chem( 48) = chem( 48) +  0.5000000000000000d+00 * w( 63)
chem( 55) = chem( 55) + w( 63)
chem( 60) = chem( 60) +  0.5000000000000000d+00 * w( 63)
chem( 61) = chem( 61) + w( 63)
chem( 51) = chem( 51) - w( 64)
chem( 55) = chem( 55) - w( 64)
chem( 61) = chem( 61) + w( 64)
chem(  5) = chem(  5) - w( 65)
chem( 53) = chem( 53) + w( 65)
chem( 55) = chem( 55) - w( 65)
chem( 53) = chem( 53) - w( 66)
chem( 54) = chem( 54) + w( 66)
chem( 61) = chem( 61) + w( 66)
chem( 64) = chem( 64) - w( 66)
chem( 65) = chem( 65) + w( 66)
chem( 22) = chem( 22) + w( 67)
chem( 53) = chem( 53) - w( 67)
chem( 61) = chem( 61) - w( 67)
chem( 14) = chem( 14) +  0.6300000000000000d+00 * w( 68)
chem( 53) = chem( 53) -  0.2000000000000000d+01 * w( 68)
chem( 54) = chem( 54) +  0.1370000000000000d+01 * w( 68)
chem( 61) = chem( 61) +  0.7400000000000000d+00 * w( 68)
chem( 22) = chem( 22) - w( 69)
chem( 53) = chem( 53) +  0.7000000000000000d+00 * w( 69)
chem( 55) = chem( 55) - w( 69)
chem( 61) = chem( 61) +  0.3000000000000000d-01 * w( 69)
chem( 63) = chem( 63) +  0.3000000000000000d+00 * w( 69)
chem( 22) = chem( 22) - w( 70)
chem( 54) = chem( 54) + w( 70)
chem( 55) = chem( 55) + w( 70)
chem( 61) = chem( 61) + w( 70)
chem( 14) = chem( 14) - w( 71)
chem( 54) = chem( 54) + w( 71)
chem( 55) = chem( 55) - w( 71)
chem( 61) = chem( 61) + w( 71)
chem( 51) = chem( 51) + w( 72)
chem( 54) = chem( 54) - w( 72)
chem( 55) = chem( 55) - w( 72)
chem( 61) = chem( 61) + w( 72)
chem( 51) = chem( 51) + w( 73)
chem( 54) = chem( 54) - w( 73)
chem( 61) = chem( 61) +  0.2000000000000000d+01 * w( 73)
chem( 51) = chem( 51) + w( 74)
chem( 54) = chem( 54) - w( 74)
chem( 51) = chem( 51) + w( 75)
chem( 54) = chem( 54) - w( 75)
chem( 55) = chem( 55) + w( 75)
chem( 57) = chem( 57) - w( 75)
chem( 61) = chem( 61) + w( 75)
chem( 50) = chem( 50) + w( 76)
chem( 51) = chem( 51) + w( 76)
chem( 54) = chem( 54) - w( 76)
chem( 56) = chem( 56) - w( 76)
chem( 61) = chem( 61) + w( 76)
chem( 21) = chem( 21) + w( 77)
chem( 54) = chem( 54) - w( 77)
chem( 61) = chem( 61) - w( 77)
chem( 21) = chem( 21) - w( 78)
chem( 54) = chem( 54) + w( 78)
chem( 61) = chem( 61) + w( 78)
chem( 21) = chem( 21) - w( 79)
chem( 35) = chem( 35) + w( 79)
chem( 61) = chem( 61) + w( 79)
chem( 64) = chem( 64) - w( 79)
chem( 65) = chem( 65) + w( 79)
chem( 21) = chem( 21) - w( 80)
chem( 22) = chem( 22) + w( 80)
chem( 61) = chem( 61) - w( 80)
chem( 35) = chem( 35) - w( 81)
chem( 55) = chem( 55) - w( 81)
chem( 61) = chem( 61) + w( 81)
chem( 55) = chem( 55) + w( 82)
chem( 57) = chem( 57) - w( 82)
chem( 60) = chem( 60) - w( 82)
chem( 62) = chem( 62) + w( 82)
chem( 55) = chem( 55) - w( 83)
chem( 60) = chem( 60) - w( 83)
chem( 62) = chem( 62) + w( 83)
chem( 50) = chem( 50) + w( 84)
chem( 56) = chem( 56) - w( 84)
chem( 60) = chem( 60) - w( 84)
chem( 62) = chem( 62) + w( 84)
chem( 51) = chem( 51) + w( 85)
chem( 53) = chem( 53) + w( 85)
chem( 60) = chem( 60) - w( 85)
chem( 61) = chem( 61) + w( 85)
chem( 53) = chem( 53) + w( 86)
chem( 62) = chem( 62) - w( 86)
chem( 64) = chem( 64) - w( 86)
chem( 65) = chem( 65) + w( 86)
chem( 13) = chem( 13) + w( 87)
chem( 62) = chem( 62) - w( 87)
chem( 65) = chem( 65) - w( 87)
chem( 13) = chem( 13) - w( 88)
chem( 62) = chem( 62) + w( 88)
chem( 65) = chem( 65) + w( 88)
chem( 13) = chem( 13) - w( 89)
chem( 62) = chem( 62) + w( 89)
chem( 65) = chem( 65) + w( 89)
chem( 33) = chem( 33) +  0.8000000000000000d+00 * w( 90)
chem( 49) = chem( 49) +  0.2000000000000000d+00 * w( 90)
chem( 59) = chem( 59) +  0.2000000000000000d+00 * w( 90)
chem( 61) = chem( 61) - w( 90)
chem( 62) = chem( 62) - w( 90)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w( 91)
chem( 53) = chem( 53) -  0.9999999999999998d-01 * w( 91)
chem( 54) = chem( 54) + w( 91)
chem( 61) = chem( 61) +  0.9000000000000000d+00 * w( 91)
chem( 62) = chem( 62) - w( 91)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w( 92)
chem( 53) = chem( 53) +  0.9000000000000000d+00 * w( 92)
chem( 62) = chem( 62) - w( 92)
chem( 63) = chem( 63) - w( 92)
chem( 53) = chem( 53) +  0.2000000000000000d+01 * w( 93)
chem( 62) = chem( 62) -  0.2000000000000000d+01 * w( 93)
chem( 33) = chem( 33) - w( 94)
chem( 55) = chem( 55) - w( 94)
chem( 62) = chem( 62) + w( 94)
chem( 33) = chem( 33) - w( 95)
chem( 53) = chem( 53) + w( 95)
chem( 55) = chem( 55) + w( 95)
chem( 49) = chem( 49) - w( 96)
chem( 53) = chem( 53) + w( 96)
chem( 55) = chem( 55) - w( 96)
chem( 48) = chem( 48) - w( 97)
chem( 55) = chem( 55) + w( 97)
chem( 57) = chem( 57) - w( 97)
chem( 58) = chem( 58) + w( 97)
chem( 48) = chem( 48) - w( 98)
chem( 55) = chem( 55) - w( 98)
chem( 58) = chem( 58) + w( 98)
chem( 48) = chem( 48) - w( 99)
chem( 50) = chem( 50) + w( 99)
chem( 56) = chem( 56) - w( 99)
chem( 58) = chem( 58) + w( 99)
chem( 48) = chem( 48) - w(100)
chem( 51) = chem( 51) + w(100)
chem( 53) = chem( 53) + w(100)
chem( 61) = chem( 61) + w(100)
chem( 58) = chem( 58) - w(101)
chem( 60) = chem( 60) + w(101)
chem( 61) = chem( 61) + w(101)
chem( 63) = chem( 63) + w(101)
chem( 64) = chem( 64) - w(101)
chem( 65) = chem( 65) + w(101)
chem( 18) = chem( 18) + w(102)
chem( 58) = chem( 58) - w(102)
chem( 65) = chem( 65) - w(102)
chem( 18) = chem( 18) - w(103)
chem( 58) = chem( 58) + w(103)
chem( 65) = chem( 65) + w(103)
chem( 18) = chem( 18) - w(104)
chem( 58) = chem( 58) + w(104)
chem( 65) = chem( 65) + w(104)
chem( 18) = chem( 18) - w(105)
chem( 55) = chem( 55) - w(105)
chem( 60) = chem( 60) + w(105)
chem( 65) = chem( 65) + w(105)
chem( 33) = chem( 33) +  0.8000000000000000d+00 * w(106)
chem( 49) = chem( 49) +  0.2000000000000000d+00 * w(106)
chem( 58) = chem( 58) - w(106)
chem( 59) = chem( 59) +  0.2000000000000000d+00 * w(106)
chem( 61) = chem( 61) - w(106)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w(107)
chem( 53) = chem( 53) - w(107)
chem( 54) = chem( 54) +  0.1000000000000000d+00 * w(107)
chem( 60) = chem( 60) +  0.9000000000000000d+00 * w(107)
chem( 61) = chem( 61) + w(107)
chem( 62) = chem( 62) - w(107)
chem( 63) = chem( 63) +  0.9000000000000000d+00 * w(107)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w(108)
chem( 60) = chem( 60) +  0.9000000000000000d+00 * w(108)
chem( 62) = chem( 62) - w(108)
chem( 63) = chem( 63) - w(108)
chem( 58) = chem( 58) -  0.2000000000000000d+01 * w(109)
chem( 60) = chem( 60) +  0.2000000000000000d+01 * w(109)
chem( 61) = chem( 61) +  0.2000000000000000d+01 * w(109)
chem( 63) = chem( 63) +  0.2000000000000000d+01 * w(109)
chem( 53) = chem( 53) + w(110)
chem( 58) = chem( 58) - w(110)
chem( 60) = chem( 60) + w(110)
chem( 61) = chem( 61) + w(110)
chem( 62) = chem( 62) - w(110)
chem( 63) = chem( 63) + w(110)
chem( 19) = chem( 19) +  0.7600000000000000d+00 * w(111)
chem( 47) = chem( 47) -  0.8900000000000000d+00 * w(111)
chem( 48) = chem( 48) +  0.5000000000000000d-01 * w(111)
chem( 52) = chem( 52) +  0.1300000000000000d+00 * w(111)
chem( 55) = chem( 55) - w(111)
chem( 60) = chem( 60) +  0.6000000000000000d-01 * w(111)
chem( 61) = chem( 61) +  0.1100000000000000d+00 * w(111)
chem( 63) = chem( 63) +  0.8700000000000000d+00 * w(111)
chem( 19) = chem( 19) -  0.9800000000000000d+00 * w(112)
chem( 47) = chem( 47) +  0.2100000000000000d+01 * w(112)
chem( 48) = chem( 48) +  0.5000000000000000d+00 * w(112)
chem( 52) = chem( 52) +  0.4000000000000000d-01 * w(112)
chem( 60) = chem( 60) +  0.6000000000000000d+00 * w(112)
chem( 61) = chem( 61) +  0.9399999999999999d+00 * w(112)
chem( 63) = chem( 63) +  0.9600000000000000d+00 * w(112)
chem( 19) = chem( 19) - w(113)
chem( 61) = chem( 61) + w(113)
chem( 19) = chem( 19) - w(114)
chem( 46) = chem( 46) + w(114)
chem( 65) = chem( 65) - w(114)
chem( 36) = chem( 36) - w(115)
chem( 47) = chem( 47) +  0.2000000000000000d+00 * w(115)
chem( 48) = chem( 48) +  0.3000000000000000d+00 * w(115)
chem( 51) = chem( 51) +  0.2000000000000000d+00 * w(115)
chem( 52) = chem( 52) +  0.1000000000000000d-01 * w(115)
chem( 54) = chem( 54) +  0.2000000000000000d+00 * w(115)
chem( 55) = chem( 55) +  0.1000000000000000d+00 * w(115)
chem( 57) = chem( 57) - w(115)
chem( 60) = chem( 60) +  0.2000000000000000d+00 * w(115)
chem( 61) = chem( 61) +  0.3000000000000000d+00 * w(115)
chem( 63) = chem( 63) +  0.2000000000000000d+00 * w(115)
chem( 36) = chem( 36) - w(116)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(116)
chem( 47) = chem( 47) +  0.7000000000000000d+00 * w(116)
chem( 48) = chem( 48) +  0.6200000000000000d+00 * w(116)
chem( 54) = chem( 54) +  0.8000000000000000d+00 * w(116)
chem( 55) = chem( 55) - w(116)
chem( 60) = chem( 60) +  0.3300000000000000d+00 * w(116)
chem( 61) = chem( 61) +  0.9500000000000000d+00 * w(116)
chem( 63) = chem( 63) +  0.8000000000000000d+00 * w(116)
chem( 36) = chem( 36) - w(117)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(117)
chem( 47) = chem( 47) + w(117)
chem( 48) = chem( 48) +  0.3200000000000000d+00 * w(117)
chem( 51) = chem( 51) +  0.3300000000000000d+00 * w(117)
chem( 54) = chem( 54) +  0.7400000000000000d+00 * w(117)
chem( 55) = chem( 55) +  0.1000000000000000d+00 * w(117)
chem( 59) = chem( 59) - w(117)
chem( 60) = chem( 60) +  0.1800000000000000d+00 * w(117)
chem( 61) = chem( 61) +  0.4400000000000000d+00 * w(117)
chem( 63) = chem( 63) +  0.2200000000000000d+00 * w(117)
chem( 36) = chem( 36) - w(118)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(118)
chem( 47) = chem( 47) + w(118)
chem( 48) = chem( 48) +  0.5600000000000001d+00 * w(118)
chem( 52) = chem( 52) +  0.9000000000000000d-01 * w(118)
chem( 54) = chem( 54) + w(118)
chem( 56) = chem( 56) - w(118)
chem( 60) = chem( 60) +  0.3500000000000000d+00 * w(118)
chem( 63) = chem( 63) +  0.9100000000000000d+00 * w(118)
chem( 65) = chem( 65) + w(118)
chem( 34) = chem( 34) - w(119)
chem( 51) = chem( 51) + w(119)
chem( 54) = chem( 54) + w(119)
chem( 55) = chem( 55) +  0.3000000000000000d+00 * w(119)
chem( 57) = chem( 57) - w(119)
chem( 61) = chem( 61) +  0.1700000000000000d+01 * w(119)
chem( 63) = chem( 63) +  0.7000000000000000d+00 * w(119)
chem( 34) = chem( 34) - w(120)
chem( 48) = chem( 48) +  0.2200000000000000d+00 * w(120)
chem( 54) = chem( 54) +  0.1560000000000000d+01 * w(120)
chem( 55) = chem( 55) - w(120)
chem( 61) = chem( 61) + w(120)
chem( 63) = chem( 63) + w(120)
chem( 34) = chem( 34) - w(121)
chem( 35) = chem( 35) +  0.3700000000000000d+00 * w(121)
chem( 51) = chem( 51) +  0.6300000000000000d+00 * w(121)
chem( 54) = chem( 54) + w(121)
chem( 55) = chem( 55) +  0.1300000000000000d+00 * w(121)
chem( 59) = chem( 59) - w(121)
chem( 61) = chem( 61) +  0.1300000000000000d+00 * w(121)
chem( 34) = chem( 34) - w(122)
chem( 54) = chem( 54) +  0.2000000000000000d+01 * w(122)
chem( 56) = chem( 56) - w(122)
chem( 63) = chem( 63) + w(122)
chem( 65) = chem( 65) + w(122)
chem( 37) = chem( 37) - w(123)
chem( 47) = chem( 47) +  0.1000000000000000d+00 * w(123)
chem( 48) = chem( 48) +  0.6600000000000000d+00 * w(123)
chem( 51) = chem( 51) +  0.1000000000000000d+00 * w(123)
chem( 57) = chem( 57) - w(123)
chem( 60) = chem( 60) +  0.1240000000000000d+01 * w(123)
chem( 61) = chem( 61) +  0.1000000000000000d+00 * w(123)
chem( 63) = chem( 63) +  0.1000000000000000d+00 * w(123)
chem( 37) = chem( 37) - w(124)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(124)
chem( 48) = chem( 48) +  0.7000000000000000d+00 * w(124)
chem( 55) = chem( 55) - w(124)
chem( 60) = chem( 60) +  0.1300000000000000d+01 * w(124)
chem( 61) = chem( 61) + w(124)
chem( 63) = chem( 63) + w(124)
chem( 37) = chem( 37) - w(125)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(125)
chem( 48) = chem( 48) +  0.3500000000000000d+00 * w(125)
chem( 51) = chem( 51) +  0.2500000000000000d+00 * w(125)
chem( 54) = chem( 54) +  0.2500000000000000d+00 * w(125)
chem( 55) = chem( 55) +  0.5000000000000000d+00 * w(125)
chem( 57) = chem( 57) +  0.5000000000000000d+00 * w(125)
chem( 59) = chem( 59) - w(125)
chem( 60) = chem( 60) +  0.6500000000000000d+00 * w(125)
chem( 61) = chem( 61) +  0.5000000000000000d+00 * w(125)
chem( 37) = chem( 37) - w(126)
chem( 38) = chem( 38) +  0.8000000000000000d-02 * w(126)
chem( 48) = chem( 48) +  0.6400000000000000d+00 * w(126)
chem( 56) = chem( 56) - w(126)
chem( 60) = chem( 60) +  0.1180000000000000d+01 * w(126)
chem( 61) = chem( 61) + w(126)
chem( 65) = chem( 65) + w(126)
chem(  7) = chem(  7) - w(127)
chem( 15) = chem( 15) +  0.5600000000000001d+00 * w(127)
chem( 30) = chem( 30) +  0.3900000000000000d-01 * w(127)
chem( 31) = chem( 31) +  0.1080000000000000d+00 * w(127)
chem( 40) = chem( 40) +  0.3600000000000000d+00 * w(127)
chem( 55) = chem( 55) - w(127)
chem( 61) = chem( 61) +  0.4400000000000000d+00 * w(127)
chem( 63) = chem( 63) +  0.8000000000000000d-01 * w(127)
chem( 15) = chem( 15) - w(128)
chem( 41) = chem( 41) +  0.9000000000000000d+00 * w(128)
chem( 46) = chem( 46) +  0.1000000000000000d+00 * w(128)
chem( 61) = chem( 61) +  0.9000000000000000d+00 * w(128)
chem( 64) = chem( 64) - w(128)
chem( 65) = chem( 65) +  0.9000000000000000d+00 * w(128)
chem( 15) = chem( 15) - w(129)
chem( 40) = chem( 40) + w(129)
chem( 61) = chem( 61) + w(129)
chem( 30) = chem( 30) +  0.3900000000000000d-01 * w(130)
chem( 31) = chem( 31) +  0.1080000000000000d+00 * w(130)
chem( 40) = chem( 40) - w(130)
chem( 41) = chem( 41) +  0.3000000000000000d+00 * w(130)
chem( 43) = chem( 43) +  0.4000000000000000d+00 * w(130)
chem( 55) = chem( 55) - w(130)
chem( 61) = chem( 61) +  0.6000000000000000d+00 * w(130)
chem( 63) = chem( 63) +  0.6000000000000000d+00 * w(130)
chem( 30) = chem( 30) +  0.3900000000000000d-01 * w(131)
chem( 31) = chem( 31) +  0.1080000000000000d+00 * w(131)
chem( 40) = chem( 40) - w(131)
chem( 43) = chem( 43) + w(131)
chem( 50) = chem( 50) + w(131)
chem( 56) = chem( 56) - w(131)
chem( 43) = chem( 43) - w(132)
chem( 46) = chem( 46) + w(132)
chem( 65) = chem( 65) - w(132)
chem( 40) = chem( 40) + w(133)
chem( 43) = chem( 43) - w(133)
chem( 61) = chem( 61) - w(133)
chem( 41) = chem( 41) - w(134)
chem( 51) = chem( 51) + w(134)
chem( 61) = chem( 61) + w(134)
chem( 62) = chem( 62) + w(134)
chem( 41) = chem( 41) - w(135)
chem( 51) = chem( 51) +  0.2000000000000000d+01 * w(135)
chem( 54) = chem( 54) + w(135)
chem( 55) = chem( 55) - w(135)
chem( 61) = chem( 61) +  0.2000000000000000d+01 * w(135)
chem( 62) = chem( 62) + w(135)
chem( 63) = chem( 63) + w(135)
chem( 41) = chem( 41) - w(136)
chem( 42) = chem( 42) +  0.2000000000000000d+00 * w(136)
chem( 48) = chem( 48) +  0.3000000000000000d-01 * w(136)
chem( 51) = chem( 51) +  0.6899999999999999d+00 * w(136)
chem( 54) = chem( 54) +  0.7000000000000000d+00 * w(136)
chem( 55) = chem( 55) +  0.8000000000000000d-01 * w(136)
chem( 59) = chem( 59) - w(136)
chem( 61) = chem( 61) +  0.7600000000000000d+00 * w(136)
chem( 62) = chem( 62) +  0.6200000000000000d+00 * w(136)
chem( 63) = chem( 63) +  0.3000000000000000d-01 * w(136)
chem(  8) = chem(  8) - w(137)
chem( 15) = chem( 15) +  0.3000000000000000d+00 * w(137)
chem( 30) = chem( 30) +  0.3900000000000000d-01 * w(137)
chem( 31) = chem( 31) +  0.1080000000000000d+00 * w(137)
chem( 40) = chem( 40) +  0.2000000000000000d-01 * w(137)
chem( 42) = chem( 42) +  0.8000000000000000d+00 * w(137)
chem( 47) = chem( 47) +  0.1100000000000000d+01 * w(137)
chem( 55) = chem( 55) - w(137)
chem( 61) = chem( 61) +  0.7000000000000000d+00 * w(137)
chem( 63) = chem( 63) +  0.5000000000000000d+00 * w(137)
chem( 42) = chem( 42) - w(138)
chem( 55) = chem( 55) - w(138)
chem( 62) = chem( 62) + w(138)
chem( 63) = chem( 63) + w(138)
chem( 42) = chem( 42) - w(139)
chem( 51) = chem( 51) + w(139)
chem( 61) = chem( 61) + w(139)
chem( 62) = chem( 62) + w(139)
chem( 44) = chem( 44) - w(140)
chem( 45) = chem( 45) +  0.7500000000000000d+00 * w(140)
chem( 47) = chem( 47) +  0.2500000000000000d+00 * w(140)
chem( 54) = chem( 54) +  0.5000000000000000d+00 * w(140)
chem( 57) = chem( 57) - w(140)
chem( 58) = chem( 58) +  0.2500000000000000d+00 * w(140)
chem( 61) = chem( 61) +  0.2500000000000000d+00 * w(140)
chem( 63) = chem( 63) +  0.2500000000000000d+00 * w(140)
chem( 44) = chem( 44) - w(141)
chem( 45) = chem( 45) +  0.9120000000000000d+00 * w(141)
chem( 52) = chem( 52) +  0.8799999999999999d-01 * w(141)
chem( 54) = chem( 54) +  0.6290000000000000d+00 * w(141)
chem( 55) = chem( 55) - w(141)
chem( 61) = chem( 61) +  0.9120000000000000d+00 * w(141)
chem( 63) = chem( 63) +  0.9910000000000000d+00 * w(141)
chem( 44) = chem( 44) - w(142)
chem( 45) = chem( 45) +  0.6500000000000000d+00 * w(142)
chem( 47) = chem( 47) +  0.3500000000000000d+00 * w(142)
chem( 48) = chem( 48) +  0.1500000000000000d+00 * w(142)
chem( 51) = chem( 51) +  0.6600000000000000d-01 * w(142)
chem( 54) = chem( 54) +  0.6000000000000000d+00 * w(142)
chem( 55) = chem( 55) +  0.2660000000000000d+00 * w(142)
chem( 58) = chem( 58) +  0.2000000000000000d+00 * w(142)
chem( 59) = chem( 59) - w(142)
chem( 61) = chem( 61) +  0.6600000000000000d-01 * w(142)
chem( 63) = chem( 63) +  0.2000000000000000d+00 * w(142)
chem( 44) = chem( 44) - w(143)
chem( 45) = chem( 45) +  0.2000000000000000d+00 * w(143)
chem( 46) = chem( 46) +  0.8000000000000000d+00 * w(143)
chem( 47) = chem( 47) +  0.2400000000000000d+01 * w(143)
chem( 48) = chem( 48) +  0.8000000000000000d+00 * w(143)
chem( 56) = chem( 56) - w(143)
chem( 61) = chem( 61) +  0.8000000000000000d+00 * w(143)
chem( 63) = chem( 63) + w(143)
chem( 65) = chem( 65) +  0.2000000000000000d+00 * w(143)
chem( 42) = chem( 42) +  0.1680000000000000d+00 * w(144)
chem( 45) = chem( 45) - w(144)
chem( 47) = chem( 47) +  0.1565000000000000d+01 * w(144)
chem( 48) = chem( 48) +  0.1200000000000000d+00 * w(144)
chem( 51) = chem( 51) +  0.3340000000000000d+00 * w(144)
chem( 54) = chem( 54) +  0.1670000000000000d+00 * w(144)
chem( 55) = chem( 55) - w(144)
chem( 58) = chem( 58) +  0.2500000000000000d+00 * w(144)
chem( 60) = chem( 60) +  0.2520000000000000d+00 * w(144)
chem( 61) = chem( 61) +  0.5030000000000000d+00 * w(144)
chem( 62) = chem( 62) +  0.2100000000000000d+00 * w(144)
chem( 63) = chem( 63) +  0.7130000000000000d+00 * w(144)
chem( 42) = chem( 42) +  0.8500000000000000d+00 * w(145)
chem( 45) = chem( 45) - w(145)
chem( 47) = chem( 47) +  0.3600000000000000d+00 * w(145)
chem( 51) = chem( 51) +  0.2250000000000000d+00 * w(145)
chem( 54) = chem( 54) +  0.1500000000000000d+00 * w(145)
chem( 55) = chem( 55) +  0.2680000000000000d+00 * w(145)
chem( 59) = chem( 59) - w(145)
chem( 60) = chem( 60) +  0.2000000000000000d-01 * w(145)
chem( 61) = chem( 61) +  0.1540000000000000d+00 * w(145)
chem( 62) = chem( 62) +  0.1140000000000000d+00 * w(145)
chem( 63) = chem( 63) +  0.6400000000000000d-01 * w(145)
chem( 45) = chem( 45) - w(146)
chem( 46) = chem( 46) +  0.8500000000000000d+00 * w(146)
chem( 47) = chem( 47) +  0.1282000000000000d+01 * w(146)
chem( 48) = chem( 48) +  0.3570000000000000d+00 * w(146)
chem( 50) = chem( 50) +  0.1500000000000000d+00 * w(146)
chem( 51) = chem( 51) +  0.6430000000000000d+00 * w(146)
chem( 54) = chem( 54) +  0.2820000000000000d+00 * w(146)
chem( 56) = chem( 56) - w(146)
chem( 58) = chem( 58) +  0.7500000000000000d-01 * w(146)
chem( 61) = chem( 61) +  0.9250000000000000d+00 * w(146)
chem( 63) = chem( 63) +  0.7500000000000000d-01 * w(146)
chem( 45) = chem( 45) - w(147)
chem( 47) = chem( 47) +  0.8320000000000000d+00 * w(147)
chem( 51) = chem( 51) +  0.3330000000000000d+00 * w(147)
chem( 54) = chem( 54) +  0.9000000000000000d+00 * w(147)
chem( 60) = chem( 60) +  0.6700000000000000d-01 * w(147)
chem( 61) = chem( 61) +  0.1033000000000000d+01 * w(147)
chem( 62) = chem( 62) +  0.9670000000000000d+00 * w(147)
chem( 63) = chem( 63) +  0.7000000000000000d+00 * w(147)
chem( 39) = chem( 39) - w(148)
chem( 47) = chem( 47) +  0.5120000000000000d+01 * w(148)
chem( 48) = chem( 48) +  0.1500000000000000d+00 * w(148)
chem( 57) = chem( 57) - w(148)
chem( 39) = chem( 39) - w(149)
chem( 47) = chem( 47) +  0.1660000000000000d+01 * w(149)
chem( 48) = chem( 48) +  0.4700000000000000d+00 * w(149)
chem( 52) = chem( 52) +  0.2500000000000000d+00 * w(149)
chem( 54) = chem( 54) +  0.2800000000000000d+00 * w(149)
chem( 55) = chem( 55) - w(149)
chem( 61) = chem( 61) +  0.7500000000000000d+00 * w(149)
chem( 63) = chem( 63) +  0.1250000000000000d+01 * w(149)
chem( 39) = chem( 39) - w(150)
chem( 47) = chem( 47) +  0.7000000000000000d+01 * w(150)
chem( 48) = chem( 48) +  0.2100000000000000d+00 * w(150)
chem( 51) = chem( 51) +  0.1000000000000000d-02 * w(150)
chem( 52) = chem( 52) +  0.1800000000000000d+00 * w(150)
chem( 54) = chem( 54) +  0.2400000000000000d+00 * w(150)
chem( 55) = chem( 55) +  0.5700000000000000d+00 * w(150)
chem( 58) = chem( 58) +  0.3900000000000000d+00 * w(150)
chem( 59) = chem( 59) - w(150)
chem( 61) = chem( 61) +  0.7000000000000001d-01 * w(150)
chem( 63) = chem( 63) +  0.7600000000000000d+00 * w(150)
chem( 39) = chem( 39) - w(151)
chem( 46) = chem( 46) +  0.5300000000000000d+00 * w(151)
chem( 48) = chem( 48) +  0.4700000000000000d+00 * w(151)
chem( 52) = chem( 52) +  0.2500000000000000d+00 * w(151)
chem( 56) = chem( 56) - w(151)
chem( 61) = chem( 61) +  0.2800000000000000d+00 * w(151)
chem( 63) = chem( 63) +  0.1030000000000000d+01 * w(151)
chem( 65) = chem( 65) +  0.4700000000000000d+00 * w(151)
chem(  9) = chem(  9) - w(152)
chem( 10) = chem( 10) + w(152)
chem( 55) = chem( 55) - w(152)
chem( 61) = chem( 61) + w(152)
chem(  4) = chem(  4) - w(153)
chem( 48) = chem( 48) +  0.5000000000000000d-01 * w(153)
chem( 54) = chem( 54) +  0.1000000000000000d+00 * w(153)
chem( 55) = chem( 55) - w(153)
chem( 60) = chem( 60) +  0.9000000000000000d+00 * w(153)
chem( 61) = chem( 61) + w(153)
chem( 63) = chem( 63) +  0.1000000000000000d+00 * w(153)
chem(  6) = chem(  6) - w(154)
chem( 52) = chem( 52) +  0.8999999999999999d-02 * w(154)
chem( 55) = chem( 55) - w(154)
chem( 60) = chem( 60) +  0.9910000000000000d+00 * w(154)
chem( 61) = chem( 61) + w(154)
chem( 63) = chem( 63) +  0.9910000000000000d+00 * w(154)
chem( 44) = chem( 44) - w(155)
chem( 45) = chem( 45) +  0.2000000000000000d+00 * w(155)
chem( 46) = chem( 46) +  0.8000000000000000d+00 * w(155)
chem( 47) = chem( 47) +  0.2400000000000000d+01 * w(155)
chem( 48) = chem( 48) +  0.8000000000000000d+00 * w(155)
chem( 61) = chem( 61) +  0.8000000000000000d+00 * w(155)
chem( 63) = chem( 63) + w(155)
chem( 64) = chem( 64) +  0.2000000000000000d+00 * w(155)
chem( 65) = chem( 65) - w(155)
chem( 11) = chem( 11) - w(156)
chem( 12) = chem( 12) +  0.4800000000000000d-01 * w(156)
chem( 24) = chem( 24) - w(157)
chem( 25) = chem( 25) +  0.6384000000000000d-02 * w(157)
chem( 26) = chem( 26) +  0.5494800000000000d-01 * w(157)
chem( 24) = chem( 24) - w(158)
chem( 25) = chem( 25) +  0.2158800000000000d-01 * w(158)
chem( 26) = chem( 26) +  0.1858110000000000d+00 * w(158)
chem( 27) = chem( 27) - w(159)
chem( 28) = chem( 28) +  0.3716400000000000d-01 * w(159)
chem( 29) = chem( 29) +  0.5631600000000000d-01 * w(159)
chem( 27) = chem( 27) - w(160)
chem( 28) = chem( 28) +  0.1256730000000000d+00 * w(160)
chem( 29) = chem( 29) +  0.1904370000000000d+00 * w(160)
chem( 24) = chem( 24) - w(161)
chem( 27) = chem( 27) - w(162)

!    Conversion molecules/cm3 to mug/m3.

do i = 1, ns
   chem(i) = chem(i) / convers_factor(i)
enddo

!     Volumic source terms.

do i=1,ns
chem(i)=chem(i)+zcsourc(i)
enddo

return
end subroutine fexchem_siream


!===============================================================================

!> jacdchemdc_siream
!> \brief Computation of the Jacobian matrix for atmospheric chemistry
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     ns                 total number of chemical species
!> \param[in]     y                  concentrations vector
!> \param[in]     convers_factor     conversion factors of mug/m3 to molecules/cm3
!> \param[in]     convers_factor_jac conversion factors for the Jacobian matrix (Wmol(i)/Wmol(j))
!> \param[in]     rk                 kinetic rates
!> \param[out]    jacc               Jacobian matrix
!______________________________________________________________________________

subroutine jacdchemdc_siream(ns,nr,y,convers_factor,                     &
                    convers_factor_jac,rk,jacc)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns),jacc(ns,ns)
double precision convers_factor(ns)
double precision convers_factor_jac(ns,ns)

! Local variables

integer i,j
double precision dw(nr,ns)
double precision conc(ns)

do j=1,ns
 do i=1,ns
  jacc(i,j)=0.d0
 enddo
enddo

!     Conversion mug/m3 to molecules/cm3.

do i = 1, ns
   conc(i) = y(i) * convers_factor(i)
enddo

call dratedc_siream(ns,nr,rk,conc,dw)

jacc( 57, 65) = jacc( 57, 65) + dw(  1, 65)
jacc( 64, 65) = jacc( 64, 65) + dw(  1, 65)
jacc( 65, 65) = jacc( 65, 65) - dw(  1, 65)
jacc( 57, 57) = jacc( 57, 57) - dw(  2, 57)
jacc( 59, 57) = jacc( 59, 57) + dw(  2, 57)
jacc( 59, 59) = jacc( 59, 59) - dw(  3, 59)
jacc( 59, 64) = jacc( 59, 64) - dw(  3, 64)
jacc( 64, 59) = jacc( 64, 59) - dw(  3, 59)
jacc( 64, 64) = jacc( 64, 64) - dw(  3, 64)
jacc( 65, 59) = jacc( 65, 59) + dw(  3, 59)
jacc( 65, 64) = jacc( 65, 64) + dw(  3, 64)
jacc( 57, 57) = jacc( 57, 57) - dw(  4, 57)
jacc( 57, 65) = jacc( 57, 65) - dw(  4, 65)
jacc( 64, 57) = jacc( 64, 57) + dw(  4, 57)
jacc( 64, 65) = jacc( 64, 65) + dw(  4, 65)
jacc( 65, 57) = jacc( 65, 57) - dw(  4, 57)
jacc( 65, 65) = jacc( 65, 65) - dw(  4, 65)
jacc( 56, 57) = jacc( 56, 57) + dw(  5, 57)
jacc( 56, 65) = jacc( 56, 65) + dw(  5, 65)
jacc( 57, 57) = jacc( 57, 57) - dw(  5, 57)
jacc( 57, 65) = jacc( 57, 65) - dw(  5, 65)
jacc( 65, 57) = jacc( 65, 57) - dw(  5, 57)
jacc( 65, 65) = jacc( 65, 65) - dw(  5, 65)
jacc( 57, 57) = jacc( 57, 57) - dw(  6, 57)
jacc( 57, 64) = jacc( 57, 64) - dw(  6, 64)
jacc( 64, 57) = jacc( 64, 57) - dw(  6, 57)
jacc( 64, 64) = jacc( 64, 64) - dw(  6, 64)
jacc( 65, 57) = jacc( 65, 57) + dw(  6, 57)
jacc( 65, 64) = jacc( 65, 64) + dw(  6, 64)
jacc( 56, 65) = jacc( 56, 65) + dw(  7, 65)
jacc( 56, 59) = jacc( 56, 59) + dw(  7, 59)
jacc( 59, 65) = jacc( 59, 65) - dw(  7, 65)
jacc( 59, 59) = jacc( 59, 59) - dw(  7, 59)
jacc( 65, 65) = jacc( 65, 65) - dw(  7, 65)
jacc( 65, 59) = jacc( 65, 59) - dw(  7, 59)
jacc( 57, 59) = jacc( 57, 59) + dw(  8, 59)
jacc( 59, 59) = jacc( 59, 59) - dw(  8, 59)
jacc(  3, 59) = jacc(  3, 59) + dw(  9, 59)
jacc( 59, 59) = jacc( 59, 59) - dw(  9, 59)
jacc(  3,  3) = jacc(  3,  3) - dw( 10,  3)
jacc( 57,  3) = jacc( 57,  3) + dw( 10,  3)
jacc(  3,  3) = jacc(  3,  3) - dw( 11,  3)
jacc( 55,  3) = jacc( 55,  3)+ 0.2000000000000000d+01*dw( 11,  3)
jacc( 55, 59) = jacc( 55, 59) - dw( 12, 59)
jacc( 55, 55) = jacc( 55, 55) - dw( 12, 55)
jacc( 59, 59) = jacc( 59, 59) - dw( 12, 59)
jacc( 59, 55) = jacc( 59, 55) - dw( 12, 55)
jacc( 61, 59) = jacc( 61, 59) + dw( 12, 59)
jacc( 61, 55) = jacc( 61, 55) + dw( 12, 55)
jacc( 55, 59) = jacc( 55, 59) + dw( 13, 59)
jacc( 55, 61) = jacc( 55, 61) + dw( 13, 61)
jacc( 59, 59) = jacc( 59, 59) - dw( 13, 59)
jacc( 59, 61) = jacc( 59, 61) - dw( 13, 61)
jacc( 61, 59) = jacc( 61, 59) - dw( 13, 59)
jacc( 61, 61) = jacc( 61, 61) - dw( 13, 61)
jacc( 56, 56) = jacc( 56, 56) - dw( 14, 56)
jacc( 57, 56) = jacc( 57, 56) + dw( 14, 56)
jacc( 65, 56) = jacc( 65, 56) + dw( 14, 56)
jacc( 56, 56) = jacc( 56, 56) - dw( 15, 56)
jacc( 64, 56) = jacc( 64, 56) + dw( 15, 56)
jacc( 56, 56) = jacc( 56, 56) - dw( 16, 56)
jacc( 56, 64) = jacc( 56, 64) - dw( 16, 64)
jacc( 64, 56) = jacc( 64, 56) - dw( 16, 56)
jacc( 64, 64) = jacc( 64, 64) - dw( 16, 64)
jacc( 65, 56) = jacc( 65, 56)+ 0.2000000000000000d+01*dw( 16, 56)
jacc( 65, 64) = jacc( 65, 64)+ 0.2000000000000000d+01*dw( 16, 64)
jacc( 56, 56) = jacc( 56, 56) - dw( 17, 56)
jacc( 56, 65) = jacc( 56, 65) - dw( 17, 65)
jacc( 64, 56) = jacc( 64, 56) + dw( 17, 56)
jacc( 64, 65) = jacc( 64, 65) + dw( 17, 65)
jacc( 16, 56) = jacc( 16, 56) + dw( 18, 56)
jacc( 16, 65) = jacc( 16, 65) + dw( 18, 65)
jacc( 56, 56) = jacc( 56, 56) - dw( 18, 56)
jacc( 56, 65) = jacc( 56, 65) - dw( 18, 65)
jacc( 65, 56) = jacc( 65, 56) - dw( 18, 56)
jacc( 65, 65) = jacc( 65, 65) - dw( 18, 65)
jacc( 16, 16) = jacc( 16, 16) - dw( 19, 16)
jacc( 50, 16) = jacc( 50, 16)+ 0.2000000000000000d+01*dw( 19, 16)
jacc( 16, 16) = jacc( 16, 16) - dw( 20, 16)
jacc( 56, 16) = jacc( 56, 16) + dw( 20, 16)
jacc( 65, 16) = jacc( 65, 16) + dw( 20, 16)
jacc( 64, 64) = jacc( 64, 64)- 0.2000000000000000d+01*dw( 21, 64)
jacc( 64, 64) = jacc( 64, 64)- 0.2000000000000000d+01*dw( 21, 64)
jacc( 65, 64) = jacc( 65, 64)+ 0.2000000000000000d+01*dw( 21, 64)
jacc( 65, 64) = jacc( 65, 64)+ 0.2000000000000000d+01*dw( 21, 64)
jacc( 23, 64) = jacc( 23, 64)+ 0.2000000000000000d+01*dw( 22, 64)
jacc( 23, 65) = jacc( 23, 65)+ 0.2000000000000000d+01*dw( 22, 65)
jacc( 64, 64) = jacc( 64, 64) - dw( 22, 64)
jacc( 64, 65) = jacc( 64, 65) - dw( 22, 65)
jacc( 65, 64) = jacc( 65, 64) - dw( 22, 64)
jacc( 65, 65) = jacc( 65, 65) - dw( 22, 65)
jacc( 23, 55) = jacc( 23, 55) + dw( 23, 55)
jacc( 23, 64) = jacc( 23, 64) + dw( 23, 64)
jacc( 55, 55) = jacc( 55, 55) - dw( 23, 55)
jacc( 55, 64) = jacc( 55, 64) - dw( 23, 64)
jacc( 64, 55) = jacc( 64, 55) - dw( 23, 55)
jacc( 64, 64) = jacc( 64, 64) - dw( 23, 64)
jacc( 23, 23) = jacc( 23, 23) - dw( 24, 23)
jacc( 55, 23) = jacc( 55, 23) + dw( 24, 23)
jacc( 64, 23) = jacc( 64, 23) + dw( 24, 23)
jacc( 23, 55) = jacc( 23, 55) - dw( 25, 55)
jacc( 23, 23) = jacc( 23, 23) - dw( 25, 23)
jacc( 55, 55) = jacc( 55, 55) - dw( 25, 55)
jacc( 55, 23) = jacc( 55, 23) - dw( 25, 23)
jacc( 65, 55) = jacc( 65, 55) + dw( 25, 55)
jacc( 65, 23) = jacc( 65, 23) + dw( 25, 23)
jacc( 23, 23) = jacc( 23, 23)- 0.2000000000000000d+01*dw( 26, 23)
jacc( 23, 23) = jacc( 23, 23)- 0.2000000000000000d+01*dw( 26, 23)
jacc( 64, 23) = jacc( 64, 23) + dw( 26, 23)
jacc( 64, 23) = jacc( 64, 23) + dw( 26, 23)
jacc( 65, 23) = jacc( 65, 23) + dw( 26, 23)
jacc( 65, 23) = jacc( 65, 23) + dw( 26, 23)
jacc( 50, 65) = jacc( 50, 65) + dw( 27, 65)
jacc( 50, 55) = jacc( 50, 55) + dw( 27, 55)
jacc( 55, 65) = jacc( 55, 65) - dw( 27, 65)
jacc( 55, 55) = jacc( 55, 55) - dw( 27, 55)
jacc( 65, 65) = jacc( 65, 65) - dw( 27, 65)
jacc( 65, 55) = jacc( 65, 55) - dw( 27, 55)
jacc( 50, 55) = jacc( 50, 55) - dw( 28, 55)
jacc( 50, 50) = jacc( 50, 50) - dw( 28, 50)
jacc( 55, 55) = jacc( 55, 55) - dw( 28, 55)
jacc( 55, 50) = jacc( 55, 50) - dw( 28, 50)
jacc( 56, 55) = jacc( 56, 55) + dw( 28, 55)
jacc( 56, 50) = jacc( 56, 50) + dw( 28, 50)
jacc( 55, 61) = jacc( 55, 61) + dw( 29, 61)
jacc( 55, 64) = jacc( 55, 64) + dw( 29, 64)
jacc( 61, 61) = jacc( 61, 61) - dw( 29, 61)
jacc( 61, 64) = jacc( 61, 64) - dw( 29, 64)
jacc( 64, 61) = jacc( 64, 61) - dw( 29, 61)
jacc( 64, 64) = jacc( 64, 64) - dw( 29, 64)
jacc( 65, 61) = jacc( 65, 61) + dw( 29, 61)
jacc( 65, 64) = jacc( 65, 64) + dw( 29, 64)
jacc( 17, 61) = jacc( 17, 61) + dw( 30, 61)
jacc( 17, 65) = jacc( 17, 65) + dw( 30, 65)
jacc( 61, 61) = jacc( 61, 61) - dw( 30, 61)
jacc( 61, 65) = jacc( 61, 65) - dw( 30, 65)
jacc( 65, 61) = jacc( 65, 61) - dw( 30, 61)
jacc( 65, 65) = jacc( 65, 65) - dw( 30, 65)
jacc( 17, 17) = jacc( 17, 17) - dw( 31, 17)
jacc( 61, 17) = jacc( 61, 17) + dw( 31, 17)
jacc( 65, 17) = jacc( 65, 17) + dw( 31, 17)
jacc( 17, 55) = jacc( 17, 55) - dw( 32, 55)
jacc( 17, 17) = jacc( 17, 17) - dw( 32, 17)
jacc( 55, 55) = jacc( 55, 55) - dw( 32, 55)
jacc( 55, 17) = jacc( 55, 17) - dw( 32, 17)
jacc( 65, 55) = jacc( 65, 55) + dw( 32, 55)
jacc( 65, 17) = jacc( 65, 17) + dw( 32, 17)
jacc( 20, 61) = jacc( 20, 61) + dw( 33, 61)
jacc( 20, 61) = jacc( 20, 61) + dw( 33, 61)
jacc( 61, 61) = jacc( 61, 61)- 0.2000000000000000d+01*dw( 33, 61)
jacc( 61, 61) = jacc( 61, 61)- 0.2000000000000000d+01*dw( 33, 61)
jacc( 20, 61) = jacc( 20, 61) + dw( 34, 61)
jacc( 20, 61) = jacc( 20, 61) + dw( 34, 61)
jacc( 61, 61) = jacc( 61, 61)- 0.2000000000000000d+01*dw( 34, 61)
jacc( 61, 61) = jacc( 61, 61)- 0.2000000000000000d+01*dw( 34, 61)
jacc( 20, 20) = jacc( 20, 20) - dw( 35, 20)
jacc( 55, 20) = jacc( 55, 20)+ 0.2000000000000000d+01*dw( 35, 20)
jacc( 20, 55) = jacc( 20, 55) - dw( 36, 55)
jacc( 20, 20) = jacc( 20, 20) - dw( 36, 20)
jacc( 55, 55) = jacc( 55, 55) - dw( 36, 55)
jacc( 55, 20) = jacc( 55, 20) - dw( 36, 20)
jacc( 61, 55) = jacc( 61, 55) + dw( 36, 55)
jacc( 61, 20) = jacc( 61, 20) + dw( 36, 20)
jacc(  3,  3) = jacc(  3,  3) - dw( 37,  3)
jacc( 55,  3) = jacc( 55,  3) + dw( 37,  3)
jacc( 61,  3) = jacc( 61,  3) + dw( 37,  3)
jacc( 55, 55) = jacc( 55, 55) - dw( 38, 55)
jacc( 61, 55) = jacc( 61, 55) + dw( 38, 55)
jacc( 55, 55) = jacc( 55, 55) - dw( 39, 55)
jacc( 55, 57) = jacc( 55, 57) - dw( 39, 57)
jacc( 57, 55) = jacc( 57, 55) - dw( 39, 55)
jacc( 57, 57) = jacc( 57, 57) - dw( 39, 57)
jacc( 61, 55) = jacc( 61, 55) + dw( 39, 55)
jacc( 61, 57) = jacc( 61, 57) + dw( 39, 57)
jacc( 55, 55) = jacc( 55, 55)- 0.2000000000000000d+01*dw( 40, 55)
jacc( 55, 55) = jacc( 55, 55)- 0.2000000000000000d+01*dw( 40, 55)
jacc( 57, 55) = jacc( 57, 55) + dw( 40, 55)
jacc( 57, 55) = jacc( 57, 55) + dw( 40, 55)
jacc( 20, 55) = jacc( 20, 55) + dw( 41, 55)
jacc( 20, 55) = jacc( 20, 55) + dw( 41, 55)
jacc( 55, 55) = jacc( 55, 55)- 0.2000000000000000d+01*dw( 41, 55)
jacc( 55, 55) = jacc( 55, 55)- 0.2000000000000000d+01*dw( 41, 55)
jacc( 55, 55) = jacc( 55, 55) - dw( 42, 55)
jacc( 55, 61) = jacc( 55, 61) - dw( 42, 61)
jacc( 61, 55) = jacc( 61, 55) - dw( 42, 55)
jacc( 61, 61) = jacc( 61, 61) - dw( 42, 61)
jacc( 55, 61) = jacc( 55, 61) + dw( 43, 61)
jacc( 55, 57) = jacc( 55, 57) + dw( 43, 57)
jacc( 57, 61) = jacc( 57, 61) - dw( 43, 61)
jacc( 57, 57) = jacc( 57, 57) - dw( 43, 57)
jacc( 61, 61) = jacc( 61, 61) - dw( 43, 61)
jacc( 61, 57) = jacc( 61, 57) - dw( 43, 57)
jacc( 20, 20) = jacc( 20, 20) - dw( 44, 20)
jacc( 20, 57) = jacc( 20, 57) - dw( 44, 57)
jacc( 55, 20) = jacc( 55, 20) + dw( 44, 20)
jacc( 55, 57) = jacc( 55, 57) + dw( 44, 57)
jacc( 57, 20) = jacc( 57, 20) - dw( 44, 20)
jacc( 57, 57) = jacc( 57, 57) - dw( 44, 57)
jacc( 61, 20) = jacc( 61, 20) + dw( 44, 20)
jacc( 61, 57) = jacc( 61, 57) + dw( 44, 57)
jacc( 56, 56) = jacc( 56, 56) - dw( 45, 56)
jacc( 56, 57) = jacc( 56, 57) - dw( 45, 57)
jacc( 57, 56) = jacc( 57, 56) - dw( 45, 56)
jacc( 57, 57) = jacc( 57, 57) - dw( 45, 57)
jacc( 65, 56) = jacc( 65, 56) + dw( 45, 56)
jacc( 65, 57) = jacc( 65, 57) + dw( 45, 57)
jacc( 55, 56) = jacc( 55, 56) - dw( 46, 56)
jacc( 55, 55) = jacc( 55, 55) - dw( 46, 55)
jacc( 56, 56) = jacc( 56, 56) - dw( 46, 56)
jacc( 56, 55) = jacc( 56, 55) - dw( 46, 55)
jacc( 61, 56) = jacc( 61, 56) + dw( 46, 56)
jacc( 61, 55) = jacc( 61, 55) + dw( 46, 55)
jacc( 65, 56) = jacc( 65, 56) + dw( 46, 56)
jacc( 65, 55) = jacc( 65, 55) + dw( 46, 55)
jacc( 50, 56) = jacc( 50, 56) + dw( 47, 56)
jacc( 50, 61) = jacc( 50, 61) + dw( 47, 61)
jacc( 56, 56) = jacc( 56, 56) - dw( 47, 56)
jacc( 56, 61) = jacc( 56, 61) - dw( 47, 61)
jacc( 61, 56) = jacc( 61, 56) - dw( 47, 56)
jacc( 61, 61) = jacc( 61, 61) - dw( 47, 61)
jacc( 56, 56) = jacc( 56, 56) - dw( 48, 56)
jacc( 56, 59) = jacc( 56, 59) - dw( 48, 59)
jacc( 59, 56) = jacc( 59, 56) - dw( 48, 56)
jacc( 59, 59) = jacc( 59, 59) - dw( 48, 59)
jacc( 65, 56) = jacc( 65, 56) + dw( 48, 56)
jacc( 65, 59) = jacc( 65, 59) + dw( 48, 59)
jacc( 56, 56) = jacc( 56, 56)- 0.2000000000000000d+01*dw( 49, 56)
jacc( 56, 56) = jacc( 56, 56)- 0.2000000000000000d+01*dw( 49, 56)
jacc( 65, 56) = jacc( 65, 56)+ 0.2000000000000000d+01*dw( 49, 56)
jacc( 65, 56) = jacc( 65, 56)+ 0.2000000000000000d+01*dw( 49, 56)
jacc( 17, 17) = jacc( 17, 17) - dw( 50, 17)
jacc( 55, 17) = jacc( 55, 17)+ 0.3900000000000000d+00*dw( 50, 17)
jacc( 56, 17) = jacc( 56, 17)+ 0.3900000000000000d+00*dw( 50, 17)
jacc( 61, 17) = jacc( 61, 17)+ 0.6100000000000000d+00*dw( 50, 17)
jacc( 65, 17) = jacc( 65, 17)+ 0.6100000000000000d+00*dw( 50, 17)
jacc( 50, 50) = jacc( 50, 50) - dw( 51, 50)
jacc( 55, 50) = jacc( 55, 50) + dw( 51, 50)
jacc( 65, 50) = jacc( 65, 50) + dw( 51, 50)
jacc( 16, 16) = jacc( 16, 16) - dw( 52, 16)
jacc( 56, 16) = jacc( 56, 16) + dw( 52, 16)
jacc( 65, 16) = jacc( 65, 16) + dw( 52, 16)
jacc( 63, 63) = jacc( 63, 63) - dw( 53, 63)
jacc( 63, 64) = jacc( 63, 64) - dw( 53, 64)
jacc( 64, 63) = jacc( 64, 63) - dw( 53, 63)
jacc( 64, 64) = jacc( 64, 64) - dw( 53, 64)
jacc( 65, 63) = jacc( 65, 63) + dw( 53, 63)
jacc( 65, 64) = jacc( 65, 64) + dw( 53, 64)
jacc( 46, 52) = jacc( 46, 52) + dw( 54, 52)
jacc( 46, 64) = jacc( 46, 64) + dw( 54, 64)
jacc( 52, 52) = jacc( 52, 52) - dw( 54, 52)
jacc( 52, 64) = jacc( 52, 64) - dw( 54, 64)
jacc( 64, 52) = jacc( 64, 52) - dw( 54, 52)
jacc( 64, 64) = jacc( 64, 64) - dw( 54, 64)
jacc( 32, 63) = jacc( 32, 63) + dw( 55, 63)
jacc( 32, 61) = jacc( 32, 61) + dw( 55, 61)
jacc( 61, 63) = jacc( 61, 63) - dw( 55, 63)
jacc( 61, 61) = jacc( 61, 61) - dw( 55, 61)
jacc( 63, 63) = jacc( 63, 63) - dw( 55, 63)
jacc( 63, 61) = jacc( 63, 61) - dw( 55, 61)
jacc( 32, 52) = jacc( 32, 52) + dw( 56, 52)
jacc( 32, 61) = jacc( 32, 61) + dw( 56, 61)
jacc( 52, 52) = jacc( 52, 52) - dw( 56, 52)
jacc( 52, 61) = jacc( 52, 61) - dw( 56, 61)
jacc( 61, 52) = jacc( 61, 52) - dw( 56, 52)
jacc( 61, 61) = jacc( 61, 61) - dw( 56, 61)
jacc( 63, 63) = jacc( 63, 63)- 0.2000000000000000d+01*dw( 57, 63)
jacc( 63, 63) = jacc( 63, 63)- 0.2000000000000000d+01*dw( 57, 63)
jacc( 52, 52) = jacc( 52, 52)- 0.2000000000000000d+01*dw( 58, 52)
jacc( 52, 52) = jacc( 52, 52)- 0.2000000000000000d+01*dw( 58, 52)
jacc( 52, 63) = jacc( 52, 63) - dw( 59, 63)
jacc( 52, 52) = jacc( 52, 52) - dw( 59, 52)
jacc( 63, 63) = jacc( 63, 63) - dw( 59, 63)
jacc( 63, 52) = jacc( 63, 52) - dw( 59, 52)
jacc( 46, 46) = jacc( 46, 46) - dw( 60, 46)
jacc( 46, 55) = jacc( 46, 55) - dw( 60, 55)
jacc( 47, 46) = jacc( 47, 46)+ 0.6600000000000000d+00*dw( 60, 46)
jacc( 47, 55) = jacc( 47, 55)+ 0.6600000000000000d+00*dw( 60, 55)
jacc( 48, 46) = jacc( 48, 46)+ 0.3300000000000000d+00*dw( 60, 46)
jacc( 48, 55) = jacc( 48, 55)+ 0.3300000000000000d+00*dw( 60, 55)
jacc( 50, 46) = jacc( 50, 46) + dw( 60, 46)
jacc( 50, 55) = jacc( 50, 55) + dw( 60, 55)
jacc( 54, 46) = jacc( 54, 46)+ 0.3300000000000000d+00*dw( 60, 46)
jacc( 54, 55) = jacc( 54, 55)+ 0.3300000000000000d+00*dw( 60, 55)
jacc( 55, 46) = jacc( 55, 46) - dw( 60, 46)
jacc( 55, 55) = jacc( 55, 55) - dw( 60, 55)
jacc( 60, 46) = jacc( 60, 46)+ 0.3300000000000000d+00*dw( 60, 46)
jacc( 60, 55) = jacc( 60, 55)+ 0.3300000000000000d+00*dw( 60, 55)
jacc( 61, 46) = jacc( 61, 46) + dw( 60, 46)
jacc( 61, 55) = jacc( 61, 55) + dw( 60, 55)
jacc( 46, 46) = jacc( 46, 46) - dw( 61, 46)
jacc( 47, 46) = jacc( 47, 46)+ 0.6600000000000000d+00*dw( 61, 46)
jacc( 48, 46) = jacc( 48, 46)+ 0.3300000000000000d+00*dw( 61, 46)
jacc( 54, 46) = jacc( 54, 46)+ 0.3300000000000000d+00*dw( 61, 46)
jacc( 60, 46) = jacc( 60, 46)+ 0.3300000000000000d+00*dw( 61, 46)
jacc( 61, 46) = jacc( 61, 46) + dw( 61, 46)
jacc( 65, 46) = jacc( 65, 46) + dw( 61, 46)
jacc( 32, 32) = jacc( 32, 32) - dw( 62, 32)
jacc( 32, 55) = jacc( 32, 55) - dw( 62, 55)
jacc( 48, 32) = jacc( 48, 32)+ 0.5000000000000000d+00*dw( 62, 32)
jacc( 48, 55) = jacc( 48, 55)+ 0.5000000000000000d+00*dw( 62, 55)
jacc( 55, 32) = jacc( 55, 32) - dw( 62, 32)
jacc( 55, 55) = jacc( 55, 55) - dw( 62, 55)
jacc( 60, 32) = jacc( 60, 32)+ 0.5000000000000000d+00*dw( 62, 32)
jacc( 60, 55) = jacc( 60, 55)+ 0.5000000000000000d+00*dw( 62, 55)
jacc( 63, 32) = jacc( 63, 32) + dw( 62, 32)
jacc( 63, 55) = jacc( 63, 55) + dw( 62, 55)
jacc( 32, 32) = jacc( 32, 32) - dw( 63, 32)
jacc( 48, 32) = jacc( 48, 32)+ 0.5000000000000000d+00*dw( 63, 32)
jacc( 55, 32) = jacc( 55, 32) + dw( 63, 32)
jacc( 60, 32) = jacc( 60, 32)+ 0.5000000000000000d+00*dw( 63, 32)
jacc( 61, 32) = jacc( 61, 32) + dw( 63, 32)
jacc( 51, 55) = jacc( 51, 55) - dw( 64, 55)
jacc( 51, 51) = jacc( 51, 51) - dw( 64, 51)
jacc( 55, 55) = jacc( 55, 55) - dw( 64, 55)
jacc( 55, 51) = jacc( 55, 51) - dw( 64, 51)
jacc( 61, 55) = jacc( 61, 55) + dw( 64, 55)
jacc( 61, 51) = jacc( 61, 51) + dw( 64, 51)
jacc(  5, 55) = jacc(  5, 55) - dw( 65, 55)
jacc(  5,  5) = jacc(  5,  5) - dw( 65,  5)
jacc( 53, 55) = jacc( 53, 55) + dw( 65, 55)
jacc( 53,  5) = jacc( 53,  5) + dw( 65,  5)
jacc( 55, 55) = jacc( 55, 55) - dw( 65, 55)
jacc( 55,  5) = jacc( 55,  5) - dw( 65,  5)
jacc( 53, 53) = jacc( 53, 53) - dw( 66, 53)
jacc( 53, 64) = jacc( 53, 64) - dw( 66, 64)
jacc( 54, 53) = jacc( 54, 53) + dw( 66, 53)
jacc( 54, 64) = jacc( 54, 64) + dw( 66, 64)
jacc( 61, 53) = jacc( 61, 53) + dw( 66, 53)
jacc( 61, 64) = jacc( 61, 64) + dw( 66, 64)
jacc( 64, 53) = jacc( 64, 53) - dw( 66, 53)
jacc( 64, 64) = jacc( 64, 64) - dw( 66, 64)
jacc( 65, 53) = jacc( 65, 53) + dw( 66, 53)
jacc( 65, 64) = jacc( 65, 64) + dw( 66, 64)
jacc( 22, 53) = jacc( 22, 53) + dw( 67, 53)
jacc( 22, 61) = jacc( 22, 61) + dw( 67, 61)
jacc( 53, 53) = jacc( 53, 53) - dw( 67, 53)
jacc( 53, 61) = jacc( 53, 61) - dw( 67, 61)
jacc( 61, 53) = jacc( 61, 53) - dw( 67, 53)
jacc( 61, 61) = jacc( 61, 61) - dw( 67, 61)
jacc( 14, 53) = jacc( 14, 53)+ 0.6300000000000000d+00*dw( 68, 53)
jacc( 14, 53) = jacc( 14, 53)+ 0.6300000000000000d+00*dw( 68, 53)
jacc( 53, 53) = jacc( 53, 53)- 0.2000000000000000d+01*dw( 68, 53)
jacc( 53, 53) = jacc( 53, 53)- 0.2000000000000000d+01*dw( 68, 53)
jacc( 54, 53) = jacc( 54, 53)+ 0.1370000000000000d+01*dw( 68, 53)
jacc( 54, 53) = jacc( 54, 53)+ 0.1370000000000000d+01*dw( 68, 53)
jacc( 61, 53) = jacc( 61, 53)+ 0.7400000000000000d+00*dw( 68, 53)
jacc( 61, 53) = jacc( 61, 53)+ 0.7400000000000000d+00*dw( 68, 53)
jacc( 22, 22) = jacc( 22, 22) - dw( 69, 22)
jacc( 22, 55) = jacc( 22, 55) - dw( 69, 55)
jacc( 53, 22) = jacc( 53, 22)+ 0.7000000000000000d+00*dw( 69, 22)
jacc( 53, 55) = jacc( 53, 55)+ 0.7000000000000000d+00*dw( 69, 55)
jacc( 55, 22) = jacc( 55, 22) - dw( 69, 22)
jacc( 55, 55) = jacc( 55, 55) - dw( 69, 55)
jacc( 61, 22) = jacc( 61, 22)+ 0.3000000000000000d-01*dw( 69, 22)
jacc( 61, 55) = jacc( 61, 55)+ 0.3000000000000000d-01*dw( 69, 55)
jacc( 63, 22) = jacc( 63, 22)+ 0.3000000000000000d+00*dw( 69, 22)
jacc( 63, 55) = jacc( 63, 55)+ 0.3000000000000000d+00*dw( 69, 55)
jacc( 22, 22) = jacc( 22, 22) - dw( 70, 22)
jacc( 54, 22) = jacc( 54, 22) + dw( 70, 22)
jacc( 55, 22) = jacc( 55, 22) + dw( 70, 22)
jacc( 61, 22) = jacc( 61, 22) + dw( 70, 22)
jacc( 14, 14) = jacc( 14, 14) - dw( 71, 14)
jacc( 14, 55) = jacc( 14, 55) - dw( 71, 55)
jacc( 54, 14) = jacc( 54, 14) + dw( 71, 14)
jacc( 54, 55) = jacc( 54, 55) + dw( 71, 55)
jacc( 55, 14) = jacc( 55, 14) - dw( 71, 14)
jacc( 55, 55) = jacc( 55, 55) - dw( 71, 55)
jacc( 61, 14) = jacc( 61, 14) + dw( 71, 14)
jacc( 61, 55) = jacc( 61, 55) + dw( 71, 55)
jacc( 51, 54) = jacc( 51, 54) + dw( 72, 54)
jacc( 51, 55) = jacc( 51, 55) + dw( 72, 55)
jacc( 54, 54) = jacc( 54, 54) - dw( 72, 54)
jacc( 54, 55) = jacc( 54, 55) - dw( 72, 55)
jacc( 55, 54) = jacc( 55, 54) - dw( 72, 54)
jacc( 55, 55) = jacc( 55, 55) - dw( 72, 55)
jacc( 61, 54) = jacc( 61, 54) + dw( 72, 54)
jacc( 61, 55) = jacc( 61, 55) + dw( 72, 55)
jacc( 51, 54) = jacc( 51, 54) + dw( 73, 54)
jacc( 54, 54) = jacc( 54, 54) - dw( 73, 54)
jacc( 61, 54) = jacc( 61, 54)+ 0.2000000000000000d+01*dw( 73, 54)
jacc( 51, 54) = jacc( 51, 54) + dw( 74, 54)
jacc( 54, 54) = jacc( 54, 54) - dw( 74, 54)
jacc( 51, 54) = jacc( 51, 54) + dw( 75, 54)
jacc( 51, 57) = jacc( 51, 57) + dw( 75, 57)
jacc( 54, 54) = jacc( 54, 54) - dw( 75, 54)
jacc( 54, 57) = jacc( 54, 57) - dw( 75, 57)
jacc( 55, 54) = jacc( 55, 54) + dw( 75, 54)
jacc( 55, 57) = jacc( 55, 57) + dw( 75, 57)
jacc( 57, 54) = jacc( 57, 54) - dw( 75, 54)
jacc( 57, 57) = jacc( 57, 57) - dw( 75, 57)
jacc( 61, 54) = jacc( 61, 54) + dw( 75, 54)
jacc( 61, 57) = jacc( 61, 57) + dw( 75, 57)
jacc( 50, 54) = jacc( 50, 54) + dw( 76, 54)
jacc( 50, 56) = jacc( 50, 56) + dw( 76, 56)
jacc( 51, 54) = jacc( 51, 54) + dw( 76, 54)
jacc( 51, 56) = jacc( 51, 56) + dw( 76, 56)
jacc( 54, 54) = jacc( 54, 54) - dw( 76, 54)
jacc( 54, 56) = jacc( 54, 56) - dw( 76, 56)
jacc( 56, 54) = jacc( 56, 54) - dw( 76, 54)
jacc( 56, 56) = jacc( 56, 56) - dw( 76, 56)
jacc( 61, 54) = jacc( 61, 54) + dw( 76, 54)
jacc( 61, 56) = jacc( 61, 56) + dw( 76, 56)
jacc( 21, 54) = jacc( 21, 54) + dw( 77, 54)
jacc( 21, 61) = jacc( 21, 61) + dw( 77, 61)
jacc( 54, 54) = jacc( 54, 54) - dw( 77, 54)
jacc( 54, 61) = jacc( 54, 61) - dw( 77, 61)
jacc( 61, 54) = jacc( 61, 54) - dw( 77, 54)
jacc( 61, 61) = jacc( 61, 61) - dw( 77, 61)
jacc( 21, 21) = jacc( 21, 21) - dw( 78, 21)
jacc( 54, 21) = jacc( 54, 21) + dw( 78, 21)
jacc( 61, 21) = jacc( 61, 21) + dw( 78, 21)
jacc( 21, 21) = jacc( 21, 21) - dw( 79, 21)
jacc( 21, 64) = jacc( 21, 64) - dw( 79, 64)
jacc( 35, 21) = jacc( 35, 21) + dw( 79, 21)
jacc( 35, 64) = jacc( 35, 64) + dw( 79, 64)
jacc( 61, 21) = jacc( 61, 21) + dw( 79, 21)
jacc( 61, 64) = jacc( 61, 64) + dw( 79, 64)
jacc( 64, 21) = jacc( 64, 21) - dw( 79, 21)
jacc( 64, 64) = jacc( 64, 64) - dw( 79, 64)
jacc( 65, 21) = jacc( 65, 21) + dw( 79, 21)
jacc( 65, 64) = jacc( 65, 64) + dw( 79, 64)
jacc( 21, 21) = jacc( 21, 21) - dw( 80, 21)
jacc( 21, 61) = jacc( 21, 61) - dw( 80, 61)
jacc( 22, 21) = jacc( 22, 21) + dw( 80, 21)
jacc( 22, 61) = jacc( 22, 61) + dw( 80, 61)
jacc( 61, 21) = jacc( 61, 21) - dw( 80, 21)
jacc( 61, 61) = jacc( 61, 61) - dw( 80, 61)
jacc( 35, 35) = jacc( 35, 35) - dw( 81, 35)
jacc( 35, 55) = jacc( 35, 55) - dw( 81, 55)
jacc( 55, 35) = jacc( 55, 35) - dw( 81, 35)
jacc( 55, 55) = jacc( 55, 55) - dw( 81, 55)
jacc( 61, 35) = jacc( 61, 35) + dw( 81, 35)
jacc( 61, 55) = jacc( 61, 55) + dw( 81, 55)
jacc( 55, 60) = jacc( 55, 60) + dw( 82, 60)
jacc( 55, 57) = jacc( 55, 57) + dw( 82, 57)
jacc( 57, 60) = jacc( 57, 60) - dw( 82, 60)
jacc( 57, 57) = jacc( 57, 57) - dw( 82, 57)
jacc( 60, 60) = jacc( 60, 60) - dw( 82, 60)
jacc( 60, 57) = jacc( 60, 57) - dw( 82, 57)
jacc( 62, 60) = jacc( 62, 60) + dw( 82, 60)
jacc( 62, 57) = jacc( 62, 57) + dw( 82, 57)
jacc( 55, 60) = jacc( 55, 60) - dw( 83, 60)
jacc( 55, 55) = jacc( 55, 55) - dw( 83, 55)
jacc( 60, 60) = jacc( 60, 60) - dw( 83, 60)
jacc( 60, 55) = jacc( 60, 55) - dw( 83, 55)
jacc( 62, 60) = jacc( 62, 60) + dw( 83, 60)
jacc( 62, 55) = jacc( 62, 55) + dw( 83, 55)
jacc( 50, 60) = jacc( 50, 60) + dw( 84, 60)
jacc( 50, 56) = jacc( 50, 56) + dw( 84, 56)
jacc( 56, 60) = jacc( 56, 60) - dw( 84, 60)
jacc( 56, 56) = jacc( 56, 56) - dw( 84, 56)
jacc( 60, 60) = jacc( 60, 60) - dw( 84, 60)
jacc( 60, 56) = jacc( 60, 56) - dw( 84, 56)
jacc( 62, 60) = jacc( 62, 60) + dw( 84, 60)
jacc( 62, 56) = jacc( 62, 56) + dw( 84, 56)
jacc( 51, 60) = jacc( 51, 60) + dw( 85, 60)
jacc( 53, 60) = jacc( 53, 60) + dw( 85, 60)
jacc( 60, 60) = jacc( 60, 60) - dw( 85, 60)
jacc( 61, 60) = jacc( 61, 60) + dw( 85, 60)
jacc( 53, 62) = jacc( 53, 62) + dw( 86, 62)
jacc( 53, 64) = jacc( 53, 64) + dw( 86, 64)
jacc( 62, 62) = jacc( 62, 62) - dw( 86, 62)
jacc( 62, 64) = jacc( 62, 64) - dw( 86, 64)
jacc( 64, 62) = jacc( 64, 62) - dw( 86, 62)
jacc( 64, 64) = jacc( 64, 64) - dw( 86, 64)
jacc( 65, 62) = jacc( 65, 62) + dw( 86, 62)
jacc( 65, 64) = jacc( 65, 64) + dw( 86, 64)
jacc( 13, 62) = jacc( 13, 62) + dw( 87, 62)
jacc( 13, 65) = jacc( 13, 65) + dw( 87, 65)
jacc( 62, 62) = jacc( 62, 62) - dw( 87, 62)
jacc( 62, 65) = jacc( 62, 65) - dw( 87, 65)
jacc( 65, 62) = jacc( 65, 62) - dw( 87, 62)
jacc( 65, 65) = jacc( 65, 65) - dw( 87, 65)
jacc( 13, 13) = jacc( 13, 13) - dw( 88, 13)
jacc( 62, 13) = jacc( 62, 13) + dw( 88, 13)
jacc( 65, 13) = jacc( 65, 13) + dw( 88, 13)
jacc( 13, 13) = jacc( 13, 13) - dw( 89, 13)
jacc( 62, 13) = jacc( 62, 13) + dw( 89, 13)
jacc( 65, 13) = jacc( 65, 13) + dw( 89, 13)
jacc( 33, 62) = jacc( 33, 62)+ 0.8000000000000000d+00*dw( 90, 62)
jacc( 33, 61) = jacc( 33, 61)+ 0.8000000000000000d+00*dw( 90, 61)
jacc( 49, 62) = jacc( 49, 62)+ 0.2000000000000000d+00*dw( 90, 62)
jacc( 49, 61) = jacc( 49, 61)+ 0.2000000000000000d+00*dw( 90, 61)
jacc( 59, 62) = jacc( 59, 62)+ 0.2000000000000000d+00*dw( 90, 62)
jacc( 59, 61) = jacc( 59, 61)+ 0.2000000000000000d+00*dw( 90, 61)
jacc( 61, 62) = jacc( 61, 62) - dw( 90, 62)
jacc( 61, 61) = jacc( 61, 61) - dw( 90, 61)
jacc( 62, 62) = jacc( 62, 62) - dw( 90, 62)
jacc( 62, 61) = jacc( 62, 61) - dw( 90, 61)
jacc( 49, 62) = jacc( 49, 62)+ 0.1000000000000000d+00*dw( 91, 62)
jacc( 49, 53) = jacc( 49, 53)+ 0.1000000000000000d+00*dw( 91, 53)
jacc( 53, 62) = jacc( 53, 62)- 0.9999999999999998d-01*dw( 91, 62)
jacc( 53, 53) = jacc( 53, 53)- 0.9999999999999998d-01*dw( 91, 53)
jacc( 54, 62) = jacc( 54, 62) + dw( 91, 62)
jacc( 54, 53) = jacc( 54, 53) + dw( 91, 53)
jacc( 61, 62) = jacc( 61, 62)+ 0.9000000000000000d+00*dw( 91, 62)
jacc( 61, 53) = jacc( 61, 53)+ 0.9000000000000000d+00*dw( 91, 53)
jacc( 62, 62) = jacc( 62, 62) - dw( 91, 62)
jacc( 62, 53) = jacc( 62, 53) - dw( 91, 53)
jacc( 49, 62) = jacc( 49, 62)+ 0.1000000000000000d+00*dw( 92, 62)
jacc( 49, 63) = jacc( 49, 63)+ 0.1000000000000000d+00*dw( 92, 63)
jacc( 53, 62) = jacc( 53, 62)+ 0.9000000000000000d+00*dw( 92, 62)
jacc( 53, 63) = jacc( 53, 63)+ 0.9000000000000000d+00*dw( 92, 63)
jacc( 62, 62) = jacc( 62, 62) - dw( 92, 62)
jacc( 62, 63) = jacc( 62, 63) - dw( 92, 63)
jacc( 63, 62) = jacc( 63, 62) - dw( 92, 62)
jacc( 63, 63) = jacc( 63, 63) - dw( 92, 63)
jacc( 53, 62) = jacc( 53, 62)+ 0.2000000000000000d+01*dw( 93, 62)
jacc( 53, 62) = jacc( 53, 62)+ 0.2000000000000000d+01*dw( 93, 62)
jacc( 62, 62) = jacc( 62, 62)- 0.2000000000000000d+01*dw( 93, 62)
jacc( 62, 62) = jacc( 62, 62)- 0.2000000000000000d+01*dw( 93, 62)
jacc( 33, 33) = jacc( 33, 33) - dw( 94, 33)
jacc( 33, 55) = jacc( 33, 55) - dw( 94, 55)
jacc( 55, 33) = jacc( 55, 33) - dw( 94, 33)
jacc( 55, 55) = jacc( 55, 55) - dw( 94, 55)
jacc( 62, 33) = jacc( 62, 33) + dw( 94, 33)
jacc( 62, 55) = jacc( 62, 55) + dw( 94, 55)
jacc( 33, 33) = jacc( 33, 33) - dw( 95, 33)
jacc( 53, 33) = jacc( 53, 33) + dw( 95, 33)
jacc( 55, 33) = jacc( 55, 33) + dw( 95, 33)
jacc( 49, 49) = jacc( 49, 49) - dw( 96, 49)
jacc( 49, 55) = jacc( 49, 55) - dw( 96, 55)
jacc( 53, 49) = jacc( 53, 49) + dw( 96, 49)
jacc( 53, 55) = jacc( 53, 55) + dw( 96, 55)
jacc( 55, 49) = jacc( 55, 49) - dw( 96, 49)
jacc( 55, 55) = jacc( 55, 55) - dw( 96, 55)
jacc( 48, 48) = jacc( 48, 48) - dw( 97, 48)
jacc( 48, 57) = jacc( 48, 57) - dw( 97, 57)
jacc( 55, 48) = jacc( 55, 48) + dw( 97, 48)
jacc( 55, 57) = jacc( 55, 57) + dw( 97, 57)
jacc( 57, 48) = jacc( 57, 48) - dw( 97, 48)
jacc( 57, 57) = jacc( 57, 57) - dw( 97, 57)
jacc( 58, 48) = jacc( 58, 48) + dw( 97, 48)
jacc( 58, 57) = jacc( 58, 57) + dw( 97, 57)
jacc( 48, 48) = jacc( 48, 48) - dw( 98, 48)
jacc( 48, 55) = jacc( 48, 55) - dw( 98, 55)
jacc( 55, 48) = jacc( 55, 48) - dw( 98, 48)
jacc( 55, 55) = jacc( 55, 55) - dw( 98, 55)
jacc( 58, 48) = jacc( 58, 48) + dw( 98, 48)
jacc( 58, 55) = jacc( 58, 55) + dw( 98, 55)
jacc( 48, 48) = jacc( 48, 48) - dw( 99, 48)
jacc( 48, 56) = jacc( 48, 56) - dw( 99, 56)
jacc( 50, 48) = jacc( 50, 48) + dw( 99, 48)
jacc( 50, 56) = jacc( 50, 56) + dw( 99, 56)
jacc( 56, 48) = jacc( 56, 48) - dw( 99, 48)
jacc( 56, 56) = jacc( 56, 56) - dw( 99, 56)
jacc( 58, 48) = jacc( 58, 48) + dw( 99, 48)
jacc( 58, 56) = jacc( 58, 56) + dw( 99, 56)
jacc( 48, 48) = jacc( 48, 48) - dw(100, 48)
jacc( 51, 48) = jacc( 51, 48) + dw(100, 48)
jacc( 53, 48) = jacc( 53, 48) + dw(100, 48)
jacc( 61, 48) = jacc( 61, 48) + dw(100, 48)
jacc( 58, 58) = jacc( 58, 58) - dw(101, 58)
jacc( 58, 64) = jacc( 58, 64) - dw(101, 64)
jacc( 60, 58) = jacc( 60, 58) + dw(101, 58)
jacc( 60, 64) = jacc( 60, 64) + dw(101, 64)
jacc( 61, 58) = jacc( 61, 58) + dw(101, 58)
jacc( 61, 64) = jacc( 61, 64) + dw(101, 64)
jacc( 63, 58) = jacc( 63, 58) + dw(101, 58)
jacc( 63, 64) = jacc( 63, 64) + dw(101, 64)
jacc( 64, 58) = jacc( 64, 58) - dw(101, 58)
jacc( 64, 64) = jacc( 64, 64) - dw(101, 64)
jacc( 65, 58) = jacc( 65, 58) + dw(101, 58)
jacc( 65, 64) = jacc( 65, 64) + dw(101, 64)
jacc( 18, 58) = jacc( 18, 58) + dw(102, 58)
jacc( 18, 65) = jacc( 18, 65) + dw(102, 65)
jacc( 58, 58) = jacc( 58, 58) - dw(102, 58)
jacc( 58, 65) = jacc( 58, 65) - dw(102, 65)
jacc( 65, 58) = jacc( 65, 58) - dw(102, 58)
jacc( 65, 65) = jacc( 65, 65) - dw(102, 65)
jacc( 18, 18) = jacc( 18, 18) - dw(103, 18)
jacc( 58, 18) = jacc( 58, 18) + dw(103, 18)
jacc( 65, 18) = jacc( 65, 18) + dw(103, 18)
jacc( 18, 18) = jacc( 18, 18) - dw(104, 18)
jacc( 58, 18) = jacc( 58, 18) + dw(104, 18)
jacc( 65, 18) = jacc( 65, 18) + dw(104, 18)
jacc( 18, 18) = jacc( 18, 18) - dw(105, 18)
jacc( 18, 55) = jacc( 18, 55) - dw(105, 55)
jacc( 55, 18) = jacc( 55, 18) - dw(105, 18)
jacc( 55, 55) = jacc( 55, 55) - dw(105, 55)
jacc( 60, 18) = jacc( 60, 18) + dw(105, 18)
jacc( 60, 55) = jacc( 60, 55) + dw(105, 55)
jacc( 65, 18) = jacc( 65, 18) + dw(105, 18)
jacc( 65, 55) = jacc( 65, 55) + dw(105, 55)
jacc( 33, 58) = jacc( 33, 58)+ 0.8000000000000000d+00*dw(106, 58)
jacc( 33, 61) = jacc( 33, 61)+ 0.8000000000000000d+00*dw(106, 61)
jacc( 49, 58) = jacc( 49, 58)+ 0.2000000000000000d+00*dw(106, 58)
jacc( 49, 61) = jacc( 49, 61)+ 0.2000000000000000d+00*dw(106, 61)
jacc( 58, 58) = jacc( 58, 58) - dw(106, 58)
jacc( 58, 61) = jacc( 58, 61) - dw(106, 61)
jacc( 59, 58) = jacc( 59, 58)+ 0.2000000000000000d+00*dw(106, 58)
jacc( 59, 61) = jacc( 59, 61)+ 0.2000000000000000d+00*dw(106, 61)
jacc( 61, 58) = jacc( 61, 58) - dw(106, 58)
jacc( 61, 61) = jacc( 61, 61) - dw(106, 61)
jacc( 49, 62) = jacc( 49, 62)+ 0.1000000000000000d+00*dw(107, 62)
jacc( 49, 53) = jacc( 49, 53)+ 0.1000000000000000d+00*dw(107, 53)
jacc( 53, 62) = jacc( 53, 62) - dw(107, 62)
jacc( 53, 53) = jacc( 53, 53) - dw(107, 53)
jacc( 54, 62) = jacc( 54, 62)+ 0.1000000000000000d+00*dw(107, 62)
jacc( 54, 53) = jacc( 54, 53)+ 0.1000000000000000d+00*dw(107, 53)
jacc( 60, 62) = jacc( 60, 62)+ 0.9000000000000000d+00*dw(107, 62)
jacc( 60, 53) = jacc( 60, 53)+ 0.9000000000000000d+00*dw(107, 53)
jacc( 61, 62) = jacc( 61, 62) + dw(107, 62)
jacc( 61, 53) = jacc( 61, 53) + dw(107, 53)
jacc( 62, 62) = jacc( 62, 62) - dw(107, 62)
jacc( 62, 53) = jacc( 62, 53) - dw(107, 53)
jacc( 63, 62) = jacc( 63, 62)+ 0.9000000000000000d+00*dw(107, 62)
jacc( 63, 53) = jacc( 63, 53)+ 0.9000000000000000d+00*dw(107, 53)
jacc( 49, 62) = jacc( 49, 62)+ 0.1000000000000000d+00*dw(108, 62)
jacc( 49, 63) = jacc( 49, 63)+ 0.1000000000000000d+00*dw(108, 63)
jacc( 60, 62) = jacc( 60, 62)+ 0.9000000000000000d+00*dw(108, 62)
jacc( 60, 63) = jacc( 60, 63)+ 0.9000000000000000d+00*dw(108, 63)
jacc( 62, 62) = jacc( 62, 62) - dw(108, 62)
jacc( 62, 63) = jacc( 62, 63) - dw(108, 63)
jacc( 63, 62) = jacc( 63, 62) - dw(108, 62)
jacc( 63, 63) = jacc( 63, 63) - dw(108, 63)
jacc( 58, 58) = jacc( 58, 58)- 0.2000000000000000d+01*dw(109, 58)
jacc( 58, 58) = jacc( 58, 58)- 0.2000000000000000d+01*dw(109, 58)
jacc( 60, 58) = jacc( 60, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 60, 58) = jacc( 60, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 61, 58) = jacc( 61, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 61, 58) = jacc( 61, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 63, 58) = jacc( 63, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 63, 58) = jacc( 63, 58)+ 0.2000000000000000d+01*dw(109, 58)
jacc( 53, 58) = jacc( 53, 58) + dw(110, 58)
jacc( 53, 62) = jacc( 53, 62) + dw(110, 62)
jacc( 58, 58) = jacc( 58, 58) - dw(110, 58)
jacc( 58, 62) = jacc( 58, 62) - dw(110, 62)
jacc( 60, 58) = jacc( 60, 58) + dw(110, 58)
jacc( 60, 62) = jacc( 60, 62) + dw(110, 62)
jacc( 61, 58) = jacc( 61, 58) + dw(110, 58)
jacc( 61, 62) = jacc( 61, 62) + dw(110, 62)
jacc( 62, 58) = jacc( 62, 58) - dw(110, 58)
jacc( 62, 62) = jacc( 62, 62) - dw(110, 62)
jacc( 63, 58) = jacc( 63, 58) + dw(110, 58)
jacc( 63, 62) = jacc( 63, 62) + dw(110, 62)
jacc( 19, 47) = jacc( 19, 47)+ 0.7600000000000000d+00*dw(111, 47)
jacc( 19, 55) = jacc( 19, 55)+ 0.7600000000000000d+00*dw(111, 55)
jacc( 47, 47) = jacc( 47, 47)- 0.8900000000000000d+00*dw(111, 47)
jacc( 47, 55) = jacc( 47, 55)- 0.8900000000000000d+00*dw(111, 55)
jacc( 48, 47) = jacc( 48, 47)+ 0.5000000000000000d-01*dw(111, 47)
jacc( 48, 55) = jacc( 48, 55)+ 0.5000000000000000d-01*dw(111, 55)
jacc( 52, 47) = jacc( 52, 47)+ 0.1300000000000000d+00*dw(111, 47)
jacc( 52, 55) = jacc( 52, 55)+ 0.1300000000000000d+00*dw(111, 55)
jacc( 55, 47) = jacc( 55, 47) - dw(111, 47)
jacc( 55, 55) = jacc( 55, 55) - dw(111, 55)
jacc( 60, 47) = jacc( 60, 47)+ 0.6000000000000000d-01*dw(111, 47)
jacc( 60, 55) = jacc( 60, 55)+ 0.6000000000000000d-01*dw(111, 55)
jacc( 61, 47) = jacc( 61, 47)+ 0.1100000000000000d+00*dw(111, 47)
jacc( 61, 55) = jacc( 61, 55)+ 0.1100000000000000d+00*dw(111, 55)
jacc( 63, 47) = jacc( 63, 47)+ 0.8700000000000000d+00*dw(111, 47)
jacc( 63, 55) = jacc( 63, 55)+ 0.8700000000000000d+00*dw(111, 55)
jacc( 19, 19) = jacc( 19, 19)- 0.9800000000000000d+00*dw(112, 19)
jacc( 47, 19) = jacc( 47, 19)+ 0.2100000000000000d+01*dw(112, 19)
jacc( 48, 19) = jacc( 48, 19)+ 0.5000000000000000d+00*dw(112, 19)
jacc( 52, 19) = jacc( 52, 19)+ 0.4000000000000000d-01*dw(112, 19)
jacc( 60, 19) = jacc( 60, 19)+ 0.6000000000000000d+00*dw(112, 19)
jacc( 61, 19) = jacc( 61, 19)+ 0.9399999999999999d+00*dw(112, 19)
jacc( 63, 19) = jacc( 63, 19)+ 0.9600000000000000d+00*dw(112, 19)
jacc( 19, 19) = jacc( 19, 19) - dw(113, 19)
jacc( 61, 19) = jacc( 61, 19) + dw(113, 19)
jacc( 19, 19) = jacc( 19, 19) - dw(114, 19)
jacc( 19, 65) = jacc( 19, 65) - dw(114, 65)
jacc( 46, 19) = jacc( 46, 19) + dw(114, 19)
jacc( 46, 65) = jacc( 46, 65) + dw(114, 65)
jacc( 65, 19) = jacc( 65, 19) - dw(114, 19)
jacc( 65, 65) = jacc( 65, 65) - dw(114, 65)
jacc( 36, 57) = jacc( 36, 57) - dw(115, 57)
jacc( 36, 36) = jacc( 36, 36) - dw(115, 36)
jacc( 47, 57) = jacc( 47, 57)+ 0.2000000000000000d+00*dw(115, 57)
jacc( 47, 36) = jacc( 47, 36)+ 0.2000000000000000d+00*dw(115, 36)
jacc( 48, 57) = jacc( 48, 57)+ 0.3000000000000000d+00*dw(115, 57)
jacc( 48, 36) = jacc( 48, 36)+ 0.3000000000000000d+00*dw(115, 36)
jacc( 51, 57) = jacc( 51, 57)+ 0.2000000000000000d+00*dw(115, 57)
jacc( 51, 36) = jacc( 51, 36)+ 0.2000000000000000d+00*dw(115, 36)
jacc( 52, 57) = jacc( 52, 57)+ 0.1000000000000000d-01*dw(115, 57)
jacc( 52, 36) = jacc( 52, 36)+ 0.1000000000000000d-01*dw(115, 36)
jacc( 54, 57) = jacc( 54, 57)+ 0.2000000000000000d+00*dw(115, 57)
jacc( 54, 36) = jacc( 54, 36)+ 0.2000000000000000d+00*dw(115, 36)
jacc( 55, 57) = jacc( 55, 57)+ 0.1000000000000000d+00*dw(115, 57)
jacc( 55, 36) = jacc( 55, 36)+ 0.1000000000000000d+00*dw(115, 36)
jacc( 57, 57) = jacc( 57, 57) - dw(115, 57)
jacc( 57, 36) = jacc( 57, 36) - dw(115, 36)
jacc( 60, 57) = jacc( 60, 57)+ 0.2000000000000000d+00*dw(115, 57)
jacc( 60, 36) = jacc( 60, 36)+ 0.2000000000000000d+00*dw(115, 36)
jacc( 61, 57) = jacc( 61, 57)+ 0.3000000000000000d+00*dw(115, 57)
jacc( 61, 36) = jacc( 61, 36)+ 0.3000000000000000d+00*dw(115, 36)
jacc( 63, 57) = jacc( 63, 57)+ 0.2000000000000000d+00*dw(115, 57)
jacc( 63, 36) = jacc( 63, 36)+ 0.2000000000000000d+00*dw(115, 36)
jacc( 36, 55) = jacc( 36, 55) - dw(116, 55)
jacc( 36, 36) = jacc( 36, 36) - dw(116, 36)
jacc( 38, 55) = jacc( 38, 55)+ 0.8000000000000000d-02*dw(116, 55)
jacc( 38, 36) = jacc( 38, 36)+ 0.8000000000000000d-02*dw(116, 36)
jacc( 47, 55) = jacc( 47, 55)+ 0.7000000000000000d+00*dw(116, 55)
jacc( 47, 36) = jacc( 47, 36)+ 0.7000000000000000d+00*dw(116, 36)
jacc( 48, 55) = jacc( 48, 55)+ 0.6200000000000000d+00*dw(116, 55)
jacc( 48, 36) = jacc( 48, 36)+ 0.6200000000000000d+00*dw(116, 36)
jacc( 54, 55) = jacc( 54, 55)+ 0.8000000000000000d+00*dw(116, 55)
jacc( 54, 36) = jacc( 54, 36)+ 0.8000000000000000d+00*dw(116, 36)
jacc( 55, 55) = jacc( 55, 55) - dw(116, 55)
jacc( 55, 36) = jacc( 55, 36) - dw(116, 36)
jacc( 60, 55) = jacc( 60, 55)+ 0.3300000000000000d+00*dw(116, 55)
jacc( 60, 36) = jacc( 60, 36)+ 0.3300000000000000d+00*dw(116, 36)
jacc( 61, 55) = jacc( 61, 55)+ 0.9500000000000000d+00*dw(116, 55)
jacc( 61, 36) = jacc( 61, 36)+ 0.9500000000000000d+00*dw(116, 36)
jacc( 63, 55) = jacc( 63, 55)+ 0.8000000000000000d+00*dw(116, 55)
jacc( 63, 36) = jacc( 63, 36)+ 0.8000000000000000d+00*dw(116, 36)
jacc( 36, 59) = jacc( 36, 59) - dw(117, 59)
jacc( 36, 36) = jacc( 36, 36) - dw(117, 36)
jacc( 38, 59) = jacc( 38, 59)+ 0.8000000000000000d-02*dw(117, 59)
jacc( 38, 36) = jacc( 38, 36)+ 0.8000000000000000d-02*dw(117, 36)
jacc( 47, 59) = jacc( 47, 59) + dw(117, 59)
jacc( 47, 36) = jacc( 47, 36) + dw(117, 36)
jacc( 48, 59) = jacc( 48, 59)+ 0.3200000000000000d+00*dw(117, 59)
jacc( 48, 36) = jacc( 48, 36)+ 0.3200000000000000d+00*dw(117, 36)
jacc( 51, 59) = jacc( 51, 59)+ 0.3300000000000000d+00*dw(117, 59)
jacc( 51, 36) = jacc( 51, 36)+ 0.3300000000000000d+00*dw(117, 36)
jacc( 54, 59) = jacc( 54, 59)+ 0.7400000000000000d+00*dw(117, 59)
jacc( 54, 36) = jacc( 54, 36)+ 0.7400000000000000d+00*dw(117, 36)
jacc( 55, 59) = jacc( 55, 59)+ 0.1000000000000000d+00*dw(117, 59)
jacc( 55, 36) = jacc( 55, 36)+ 0.1000000000000000d+00*dw(117, 36)
jacc( 59, 59) = jacc( 59, 59) - dw(117, 59)
jacc( 59, 36) = jacc( 59, 36) - dw(117, 36)
jacc( 60, 59) = jacc( 60, 59)+ 0.1800000000000000d+00*dw(117, 59)
jacc( 60, 36) = jacc( 60, 36)+ 0.1800000000000000d+00*dw(117, 36)
jacc( 61, 59) = jacc( 61, 59)+ 0.4400000000000000d+00*dw(117, 59)
jacc( 61, 36) = jacc( 61, 36)+ 0.4400000000000000d+00*dw(117, 36)
jacc( 63, 59) = jacc( 63, 59)+ 0.2200000000000000d+00*dw(117, 59)
jacc( 63, 36) = jacc( 63, 36)+ 0.2200000000000000d+00*dw(117, 36)
jacc( 36, 56) = jacc( 36, 56) - dw(118, 56)
jacc( 36, 36) = jacc( 36, 36) - dw(118, 36)
jacc( 38, 56) = jacc( 38, 56)+ 0.8000000000000000d-02*dw(118, 56)
jacc( 38, 36) = jacc( 38, 36)+ 0.8000000000000000d-02*dw(118, 36)
jacc( 47, 56) = jacc( 47, 56) + dw(118, 56)
jacc( 47, 36) = jacc( 47, 36) + dw(118, 36)
jacc( 48, 56) = jacc( 48, 56)+ 0.5600000000000001d+00*dw(118, 56)
jacc( 48, 36) = jacc( 48, 36)+ 0.5600000000000001d+00*dw(118, 36)
jacc( 52, 56) = jacc( 52, 56)+ 0.9000000000000000d-01*dw(118, 56)
jacc( 52, 36) = jacc( 52, 36)+ 0.9000000000000000d-01*dw(118, 36)
jacc( 54, 56) = jacc( 54, 56) + dw(118, 56)
jacc( 54, 36) = jacc( 54, 36) + dw(118, 36)
jacc( 56, 56) = jacc( 56, 56) - dw(118, 56)
jacc( 56, 36) = jacc( 56, 36) - dw(118, 36)
jacc( 60, 56) = jacc( 60, 56)+ 0.3500000000000000d+00*dw(118, 56)
jacc( 60, 36) = jacc( 60, 36)+ 0.3500000000000000d+00*dw(118, 36)
jacc( 63, 56) = jacc( 63, 56)+ 0.9100000000000000d+00*dw(118, 56)
jacc( 63, 36) = jacc( 63, 36)+ 0.9100000000000000d+00*dw(118, 36)
jacc( 65, 56) = jacc( 65, 56) + dw(118, 56)
jacc( 65, 36) = jacc( 65, 36) + dw(118, 36)
jacc( 34, 57) = jacc( 34, 57) - dw(119, 57)
jacc( 34, 34) = jacc( 34, 34) - dw(119, 34)
jacc( 51, 57) = jacc( 51, 57) + dw(119, 57)
jacc( 51, 34) = jacc( 51, 34) + dw(119, 34)
jacc( 54, 57) = jacc( 54, 57) + dw(119, 57)
jacc( 54, 34) = jacc( 54, 34) + dw(119, 34)
jacc( 55, 57) = jacc( 55, 57)+ 0.3000000000000000d+00*dw(119, 57)
jacc( 55, 34) = jacc( 55, 34)+ 0.3000000000000000d+00*dw(119, 34)
jacc( 57, 57) = jacc( 57, 57) - dw(119, 57)
jacc( 57, 34) = jacc( 57, 34) - dw(119, 34)
jacc( 61, 57) = jacc( 61, 57)+ 0.1700000000000000d+01*dw(119, 57)
jacc( 61, 34) = jacc( 61, 34)+ 0.1700000000000000d+01*dw(119, 34)
jacc( 63, 57) = jacc( 63, 57)+ 0.7000000000000000d+00*dw(119, 57)
jacc( 63, 34) = jacc( 63, 34)+ 0.7000000000000000d+00*dw(119, 34)
jacc( 34, 55) = jacc( 34, 55) - dw(120, 55)
jacc( 34, 34) = jacc( 34, 34) - dw(120, 34)
jacc( 48, 55) = jacc( 48, 55)+ 0.2200000000000000d+00*dw(120, 55)
jacc( 48, 34) = jacc( 48, 34)+ 0.2200000000000000d+00*dw(120, 34)
jacc( 54, 55) = jacc( 54, 55)+ 0.1560000000000000d+01*dw(120, 55)
jacc( 54, 34) = jacc( 54, 34)+ 0.1560000000000000d+01*dw(120, 34)
jacc( 55, 55) = jacc( 55, 55) - dw(120, 55)
jacc( 55, 34) = jacc( 55, 34) - dw(120, 34)
jacc( 61, 55) = jacc( 61, 55) + dw(120, 55)
jacc( 61, 34) = jacc( 61, 34) + dw(120, 34)
jacc( 63, 55) = jacc( 63, 55) + dw(120, 55)
jacc( 63, 34) = jacc( 63, 34) + dw(120, 34)
jacc( 34, 59) = jacc( 34, 59) - dw(121, 59)
jacc( 34, 34) = jacc( 34, 34) - dw(121, 34)
jacc( 35, 59) = jacc( 35, 59)+ 0.3700000000000000d+00*dw(121, 59)
jacc( 35, 34) = jacc( 35, 34)+ 0.3700000000000000d+00*dw(121, 34)
jacc( 51, 59) = jacc( 51, 59)+ 0.6300000000000000d+00*dw(121, 59)
jacc( 51, 34) = jacc( 51, 34)+ 0.6300000000000000d+00*dw(121, 34)
jacc( 54, 59) = jacc( 54, 59) + dw(121, 59)
jacc( 54, 34) = jacc( 54, 34) + dw(121, 34)
jacc( 55, 59) = jacc( 55, 59)+ 0.1300000000000000d+00*dw(121, 59)
jacc( 55, 34) = jacc( 55, 34)+ 0.1300000000000000d+00*dw(121, 34)
jacc( 59, 59) = jacc( 59, 59) - dw(121, 59)
jacc( 59, 34) = jacc( 59, 34) - dw(121, 34)
jacc( 61, 59) = jacc( 61, 59)+ 0.1300000000000000d+00*dw(121, 59)
jacc( 61, 34) = jacc( 61, 34)+ 0.1300000000000000d+00*dw(121, 34)
jacc( 34, 56) = jacc( 34, 56) - dw(122, 56)
jacc( 34, 34) = jacc( 34, 34) - dw(122, 34)
jacc( 54, 56) = jacc( 54, 56)+ 0.2000000000000000d+01*dw(122, 56)
jacc( 54, 34) = jacc( 54, 34)+ 0.2000000000000000d+01*dw(122, 34)
jacc( 56, 56) = jacc( 56, 56) - dw(122, 56)
jacc( 56, 34) = jacc( 56, 34) - dw(122, 34)
jacc( 63, 56) = jacc( 63, 56) + dw(122, 56)
jacc( 63, 34) = jacc( 63, 34) + dw(122, 34)
jacc( 65, 56) = jacc( 65, 56) + dw(122, 56)
jacc( 65, 34) = jacc( 65, 34) + dw(122, 34)
jacc( 37, 37) = jacc( 37, 37) - dw(123, 37)
jacc( 37, 57) = jacc( 37, 57) - dw(123, 57)
jacc( 47, 37) = jacc( 47, 37)+ 0.1000000000000000d+00*dw(123, 37)
jacc( 47, 57) = jacc( 47, 57)+ 0.1000000000000000d+00*dw(123, 57)
jacc( 48, 37) = jacc( 48, 37)+ 0.6600000000000000d+00*dw(123, 37)
jacc( 48, 57) = jacc( 48, 57)+ 0.6600000000000000d+00*dw(123, 57)
jacc( 51, 37) = jacc( 51, 37)+ 0.1000000000000000d+00*dw(123, 37)
jacc( 51, 57) = jacc( 51, 57)+ 0.1000000000000000d+00*dw(123, 57)
jacc( 57, 37) = jacc( 57, 37) - dw(123, 37)
jacc( 57, 57) = jacc( 57, 57) - dw(123, 57)
jacc( 60, 37) = jacc( 60, 37)+ 0.1240000000000000d+01*dw(123, 37)
jacc( 60, 57) = jacc( 60, 57)+ 0.1240000000000000d+01*dw(123, 57)
jacc( 61, 37) = jacc( 61, 37)+ 0.1000000000000000d+00*dw(123, 37)
jacc( 61, 57) = jacc( 61, 57)+ 0.1000000000000000d+00*dw(123, 57)
jacc( 63, 37) = jacc( 63, 37)+ 0.1000000000000000d+00*dw(123, 37)
jacc( 63, 57) = jacc( 63, 57)+ 0.1000000000000000d+00*dw(123, 57)
jacc( 37, 37) = jacc( 37, 37) - dw(124, 37)
jacc( 37, 55) = jacc( 37, 55) - dw(124, 55)
jacc( 38, 37) = jacc( 38, 37)+ 0.8000000000000000d-02*dw(124, 37)
jacc( 38, 55) = jacc( 38, 55)+ 0.8000000000000000d-02*dw(124, 55)
jacc( 48, 37) = jacc( 48, 37)+ 0.7000000000000000d+00*dw(124, 37)
jacc( 48, 55) = jacc( 48, 55)+ 0.7000000000000000d+00*dw(124, 55)
jacc( 55, 37) = jacc( 55, 37) - dw(124, 37)
jacc( 55, 55) = jacc( 55, 55) - dw(124, 55)
jacc( 60, 37) = jacc( 60, 37)+ 0.1300000000000000d+01*dw(124, 37)
jacc( 60, 55) = jacc( 60, 55)+ 0.1300000000000000d+01*dw(124, 55)
jacc( 61, 37) = jacc( 61, 37) + dw(124, 37)
jacc( 61, 55) = jacc( 61, 55) + dw(124, 55)
jacc( 63, 37) = jacc( 63, 37) + dw(124, 37)
jacc( 63, 55) = jacc( 63, 55) + dw(124, 55)
jacc( 37, 37) = jacc( 37, 37) - dw(125, 37)
jacc( 37, 59) = jacc( 37, 59) - dw(125, 59)
jacc( 38, 37) = jacc( 38, 37)+ 0.8000000000000000d-02*dw(125, 37)
jacc( 38, 59) = jacc( 38, 59)+ 0.8000000000000000d-02*dw(125, 59)
jacc( 48, 37) = jacc( 48, 37)+ 0.3500000000000000d+00*dw(125, 37)
jacc( 48, 59) = jacc( 48, 59)+ 0.3500000000000000d+00*dw(125, 59)
jacc( 51, 37) = jacc( 51, 37)+ 0.2500000000000000d+00*dw(125, 37)
jacc( 51, 59) = jacc( 51, 59)+ 0.2500000000000000d+00*dw(125, 59)
jacc( 54, 37) = jacc( 54, 37)+ 0.2500000000000000d+00*dw(125, 37)
jacc( 54, 59) = jacc( 54, 59)+ 0.2500000000000000d+00*dw(125, 59)
jacc( 55, 37) = jacc( 55, 37)+ 0.5000000000000000d+00*dw(125, 37)
jacc( 55, 59) = jacc( 55, 59)+ 0.5000000000000000d+00*dw(125, 59)
jacc( 57, 37) = jacc( 57, 37)+ 0.5000000000000000d+00*dw(125, 37)
jacc( 57, 59) = jacc( 57, 59)+ 0.5000000000000000d+00*dw(125, 59)
jacc( 59, 37) = jacc( 59, 37) - dw(125, 37)
jacc( 59, 59) = jacc( 59, 59) - dw(125, 59)
jacc( 60, 37) = jacc( 60, 37)+ 0.6500000000000000d+00*dw(125, 37)
jacc( 60, 59) = jacc( 60, 59)+ 0.6500000000000000d+00*dw(125, 59)
jacc( 61, 37) = jacc( 61, 37)+ 0.5000000000000000d+00*dw(125, 37)
jacc( 61, 59) = jacc( 61, 59)+ 0.5000000000000000d+00*dw(125, 59)
jacc( 37, 37) = jacc( 37, 37) - dw(126, 37)
jacc( 37, 56) = jacc( 37, 56) - dw(126, 56)
jacc( 38, 37) = jacc( 38, 37)+ 0.8000000000000000d-02*dw(126, 37)
jacc( 38, 56) = jacc( 38, 56)+ 0.8000000000000000d-02*dw(126, 56)
jacc( 48, 37) = jacc( 48, 37)+ 0.6400000000000000d+00*dw(126, 37)
jacc( 48, 56) = jacc( 48, 56)+ 0.6400000000000000d+00*dw(126, 56)
jacc( 56, 37) = jacc( 56, 37) - dw(126, 37)
jacc( 56, 56) = jacc( 56, 56) - dw(126, 56)
jacc( 60, 37) = jacc( 60, 37)+ 0.1180000000000000d+01*dw(126, 37)
jacc( 60, 56) = jacc( 60, 56)+ 0.1180000000000000d+01*dw(126, 56)
jacc( 61, 37) = jacc( 61, 37) + dw(126, 37)
jacc( 61, 56) = jacc( 61, 56) + dw(126, 56)
jacc( 65, 37) = jacc( 65, 37) + dw(126, 37)
jacc( 65, 56) = jacc( 65, 56) + dw(126, 56)
jacc(  7,  7) = jacc(  7,  7) - dw(127,  7)
jacc(  7, 55) = jacc(  7, 55) - dw(127, 55)
jacc( 15,  7) = jacc( 15,  7)+ 0.5600000000000001d+00*dw(127,  7)
jacc( 15, 55) = jacc( 15, 55)+ 0.5600000000000001d+00*dw(127, 55)
jacc( 30,  7) = jacc( 30,  7)+ 0.3900000000000000d-01*dw(127,  7)
jacc( 30, 55) = jacc( 30, 55)+ 0.3900000000000000d-01*dw(127, 55)
jacc( 31,  7) = jacc( 31,  7)+ 0.1080000000000000d+00*dw(127,  7)
jacc( 31, 55) = jacc( 31, 55)+ 0.1080000000000000d+00*dw(127, 55)
jacc( 40,  7) = jacc( 40,  7)+ 0.3600000000000000d+00*dw(127,  7)
jacc( 40, 55) = jacc( 40, 55)+ 0.3600000000000000d+00*dw(127, 55)
jacc( 55,  7) = jacc( 55,  7) - dw(127,  7)
jacc( 55, 55) = jacc( 55, 55) - dw(127, 55)
jacc( 61,  7) = jacc( 61,  7)+ 0.4400000000000000d+00*dw(127,  7)
jacc( 61, 55) = jacc( 61, 55)+ 0.4400000000000000d+00*dw(127, 55)
jacc( 63,  7) = jacc( 63,  7)+ 0.8000000000000000d-01*dw(127,  7)
jacc( 63, 55) = jacc( 63, 55)+ 0.8000000000000000d-01*dw(127, 55)
jacc( 15, 15) = jacc( 15, 15) - dw(128, 15)
jacc( 15, 64) = jacc( 15, 64) - dw(128, 64)
jacc( 41, 15) = jacc( 41, 15)+ 0.9000000000000000d+00*dw(128, 15)
jacc( 41, 64) = jacc( 41, 64)+ 0.9000000000000000d+00*dw(128, 64)
jacc( 46, 15) = jacc( 46, 15)+ 0.1000000000000000d+00*dw(128, 15)
jacc( 46, 64) = jacc( 46, 64)+ 0.1000000000000000d+00*dw(128, 64)
jacc( 61, 15) = jacc( 61, 15)+ 0.9000000000000000d+00*dw(128, 15)
jacc( 61, 64) = jacc( 61, 64)+ 0.9000000000000000d+00*dw(128, 64)
jacc( 64, 15) = jacc( 64, 15) - dw(128, 15)
jacc( 64, 64) = jacc( 64, 64) - dw(128, 64)
jacc( 65, 15) = jacc( 65, 15)+ 0.9000000000000000d+00*dw(128, 15)
jacc( 65, 64) = jacc( 65, 64)+ 0.9000000000000000d+00*dw(128, 64)
jacc( 15, 15) = jacc( 15, 15) - dw(129, 15)
jacc( 40, 15) = jacc( 40, 15) + dw(129, 15)
jacc( 61, 15) = jacc( 61, 15) + dw(129, 15)
jacc( 30, 55) = jacc( 30, 55)+ 0.3900000000000000d-01*dw(130, 55)
jacc( 30, 40) = jacc( 30, 40)+ 0.3900000000000000d-01*dw(130, 40)
jacc( 31, 55) = jacc( 31, 55)+ 0.1080000000000000d+00*dw(130, 55)
jacc( 31, 40) = jacc( 31, 40)+ 0.1080000000000000d+00*dw(130, 40)
jacc( 40, 55) = jacc( 40, 55) - dw(130, 55)
jacc( 40, 40) = jacc( 40, 40) - dw(130, 40)
jacc( 41, 55) = jacc( 41, 55)+ 0.3000000000000000d+00*dw(130, 55)
jacc( 41, 40) = jacc( 41, 40)+ 0.3000000000000000d+00*dw(130, 40)
jacc( 43, 55) = jacc( 43, 55)+ 0.4000000000000000d+00*dw(130, 55)
jacc( 43, 40) = jacc( 43, 40)+ 0.4000000000000000d+00*dw(130, 40)
jacc( 55, 55) = jacc( 55, 55) - dw(130, 55)
jacc( 55, 40) = jacc( 55, 40) - dw(130, 40)
jacc( 61, 55) = jacc( 61, 55)+ 0.6000000000000000d+00*dw(130, 55)
jacc( 61, 40) = jacc( 61, 40)+ 0.6000000000000000d+00*dw(130, 40)
jacc( 63, 55) = jacc( 63, 55)+ 0.6000000000000000d+00*dw(130, 55)
jacc( 63, 40) = jacc( 63, 40)+ 0.6000000000000000d+00*dw(130, 40)
jacc( 30, 40) = jacc( 30, 40)+ 0.3900000000000000d-01*dw(131, 40)
jacc( 30, 56) = jacc( 30, 56)+ 0.3900000000000000d-01*dw(131, 56)
jacc( 31, 40) = jacc( 31, 40)+ 0.1080000000000000d+00*dw(131, 40)
jacc( 31, 56) = jacc( 31, 56)+ 0.1080000000000000d+00*dw(131, 56)
jacc( 40, 40) = jacc( 40, 40) - dw(131, 40)
jacc( 40, 56) = jacc( 40, 56) - dw(131, 56)
jacc( 43, 40) = jacc( 43, 40) + dw(131, 40)
jacc( 43, 56) = jacc( 43, 56) + dw(131, 56)
jacc( 50, 40) = jacc( 50, 40) + dw(131, 40)
jacc( 50, 56) = jacc( 50, 56) + dw(131, 56)
jacc( 56, 40) = jacc( 56, 40) - dw(131, 40)
jacc( 56, 56) = jacc( 56, 56) - dw(131, 56)
jacc( 43, 43) = jacc( 43, 43) - dw(132, 43)
jacc( 43, 65) = jacc( 43, 65) - dw(132, 65)
jacc( 46, 43) = jacc( 46, 43) + dw(132, 43)
jacc( 46, 65) = jacc( 46, 65) + dw(132, 65)
jacc( 65, 43) = jacc( 65, 43) - dw(132, 43)
jacc( 65, 65) = jacc( 65, 65) - dw(132, 65)
jacc( 40, 43) = jacc( 40, 43) + dw(133, 43)
jacc( 40, 61) = jacc( 40, 61) + dw(133, 61)
jacc( 43, 43) = jacc( 43, 43) - dw(133, 43)
jacc( 43, 61) = jacc( 43, 61) - dw(133, 61)
jacc( 61, 43) = jacc( 61, 43) - dw(133, 43)
jacc( 61, 61) = jacc( 61, 61) - dw(133, 61)
jacc( 41, 41) = jacc( 41, 41) - dw(134, 41)
jacc( 51, 41) = jacc( 51, 41) + dw(134, 41)
jacc( 61, 41) = jacc( 61, 41) + dw(134, 41)
jacc( 62, 41) = jacc( 62, 41) + dw(134, 41)
jacc( 41, 41) = jacc( 41, 41) - dw(135, 41)
jacc( 41, 55) = jacc( 41, 55) - dw(135, 55)
jacc( 51, 41) = jacc( 51, 41)+ 0.2000000000000000d+01*dw(135, 41)
jacc( 51, 55) = jacc( 51, 55)+ 0.2000000000000000d+01*dw(135, 55)
jacc( 54, 41) = jacc( 54, 41) + dw(135, 41)
jacc( 54, 55) = jacc( 54, 55) + dw(135, 55)
jacc( 55, 41) = jacc( 55, 41) - dw(135, 41)
jacc( 55, 55) = jacc( 55, 55) - dw(135, 55)
jacc( 61, 41) = jacc( 61, 41)+ 0.2000000000000000d+01*dw(135, 41)
jacc( 61, 55) = jacc( 61, 55)+ 0.2000000000000000d+01*dw(135, 55)
jacc( 62, 41) = jacc( 62, 41) + dw(135, 41)
jacc( 62, 55) = jacc( 62, 55) + dw(135, 55)
jacc( 63, 41) = jacc( 63, 41) + dw(135, 41)
jacc( 63, 55) = jacc( 63, 55) + dw(135, 55)
jacc( 41, 41) = jacc( 41, 41) - dw(136, 41)
jacc( 41, 59) = jacc( 41, 59) - dw(136, 59)
jacc( 42, 41) = jacc( 42, 41)+ 0.2000000000000000d+00*dw(136, 41)
jacc( 42, 59) = jacc( 42, 59)+ 0.2000000000000000d+00*dw(136, 59)
jacc( 48, 41) = jacc( 48, 41)+ 0.3000000000000000d-01*dw(136, 41)
jacc( 48, 59) = jacc( 48, 59)+ 0.3000000000000000d-01*dw(136, 59)
jacc( 51, 41) = jacc( 51, 41)+ 0.6899999999999999d+00*dw(136, 41)
jacc( 51, 59) = jacc( 51, 59)+ 0.6899999999999999d+00*dw(136, 59)
jacc( 54, 41) = jacc( 54, 41)+ 0.7000000000000000d+00*dw(136, 41)
jacc( 54, 59) = jacc( 54, 59)+ 0.7000000000000000d+00*dw(136, 59)
jacc( 55, 41) = jacc( 55, 41)+ 0.8000000000000000d-01*dw(136, 41)
jacc( 55, 59) = jacc( 55, 59)+ 0.8000000000000000d-01*dw(136, 59)
jacc( 59, 41) = jacc( 59, 41) - dw(136, 41)
jacc( 59, 59) = jacc( 59, 59) - dw(136, 59)
jacc( 61, 41) = jacc( 61, 41)+ 0.7600000000000000d+00*dw(136, 41)
jacc( 61, 59) = jacc( 61, 59)+ 0.7600000000000000d+00*dw(136, 59)
jacc( 62, 41) = jacc( 62, 41)+ 0.6200000000000000d+00*dw(136, 41)
jacc( 62, 59) = jacc( 62, 59)+ 0.6200000000000000d+00*dw(136, 59)
jacc( 63, 41) = jacc( 63, 41)+ 0.3000000000000000d-01*dw(136, 41)
jacc( 63, 59) = jacc( 63, 59)+ 0.3000000000000000d-01*dw(136, 59)
jacc(  8, 55) = jacc(  8, 55) - dw(137, 55)
jacc(  8,  8) = jacc(  8,  8) - dw(137,  8)
jacc( 15, 55) = jacc( 15, 55)+ 0.3000000000000000d+00*dw(137, 55)
jacc( 15,  8) = jacc( 15,  8)+ 0.3000000000000000d+00*dw(137,  8)
jacc( 30, 55) = jacc( 30, 55)+ 0.3900000000000000d-01*dw(137, 55)
jacc( 30,  8) = jacc( 30,  8)+ 0.3900000000000000d-01*dw(137,  8)
jacc( 31, 55) = jacc( 31, 55)+ 0.1080000000000000d+00*dw(137, 55)
jacc( 31,  8) = jacc( 31,  8)+ 0.1080000000000000d+00*dw(137,  8)
jacc( 40, 55) = jacc( 40, 55)+ 0.2000000000000000d-01*dw(137, 55)
jacc( 40,  8) = jacc( 40,  8)+ 0.2000000000000000d-01*dw(137,  8)
jacc( 42, 55) = jacc( 42, 55)+ 0.8000000000000000d+00*dw(137, 55)
jacc( 42,  8) = jacc( 42,  8)+ 0.8000000000000000d+00*dw(137,  8)
jacc( 47, 55) = jacc( 47, 55)+ 0.1100000000000000d+01*dw(137, 55)
jacc( 47,  8) = jacc( 47,  8)+ 0.1100000000000000d+01*dw(137,  8)
jacc( 55, 55) = jacc( 55, 55) - dw(137, 55)
jacc( 55,  8) = jacc( 55,  8) - dw(137,  8)
jacc( 61, 55) = jacc( 61, 55)+ 0.7000000000000000d+00*dw(137, 55)
jacc( 61,  8) = jacc( 61,  8)+ 0.7000000000000000d+00*dw(137,  8)
jacc( 63, 55) = jacc( 63, 55)+ 0.5000000000000000d+00*dw(137, 55)
jacc( 63,  8) = jacc( 63,  8)+ 0.5000000000000000d+00*dw(137,  8)
jacc( 42, 55) = jacc( 42, 55) - dw(138, 55)
jacc( 42, 42) = jacc( 42, 42) - dw(138, 42)
jacc( 55, 55) = jacc( 55, 55) - dw(138, 55)
jacc( 55, 42) = jacc( 55, 42) - dw(138, 42)
jacc( 62, 55) = jacc( 62, 55) + dw(138, 55)
jacc( 62, 42) = jacc( 62, 42) + dw(138, 42)
jacc( 63, 55) = jacc( 63, 55) + dw(138, 55)
jacc( 63, 42) = jacc( 63, 42) + dw(138, 42)
jacc( 42, 42) = jacc( 42, 42) - dw(139, 42)
jacc( 51, 42) = jacc( 51, 42) + dw(139, 42)
jacc( 61, 42) = jacc( 61, 42) + dw(139, 42)
jacc( 62, 42) = jacc( 62, 42) + dw(139, 42)
jacc( 44, 57) = jacc( 44, 57) - dw(140, 57)
jacc( 44, 44) = jacc( 44, 44) - dw(140, 44)
jacc( 45, 57) = jacc( 45, 57)+ 0.7500000000000000d+00*dw(140, 57)
jacc( 45, 44) = jacc( 45, 44)+ 0.7500000000000000d+00*dw(140, 44)
jacc( 47, 57) = jacc( 47, 57)+ 0.2500000000000000d+00*dw(140, 57)
jacc( 47, 44) = jacc( 47, 44)+ 0.2500000000000000d+00*dw(140, 44)
jacc( 54, 57) = jacc( 54, 57)+ 0.5000000000000000d+00*dw(140, 57)
jacc( 54, 44) = jacc( 54, 44)+ 0.5000000000000000d+00*dw(140, 44)
jacc( 57, 57) = jacc( 57, 57) - dw(140, 57)
jacc( 57, 44) = jacc( 57, 44) - dw(140, 44)
jacc( 58, 57) = jacc( 58, 57)+ 0.2500000000000000d+00*dw(140, 57)
jacc( 58, 44) = jacc( 58, 44)+ 0.2500000000000000d+00*dw(140, 44)
jacc( 61, 57) = jacc( 61, 57)+ 0.2500000000000000d+00*dw(140, 57)
jacc( 61, 44) = jacc( 61, 44)+ 0.2500000000000000d+00*dw(140, 44)
jacc( 63, 57) = jacc( 63, 57)+ 0.2500000000000000d+00*dw(140, 57)
jacc( 63, 44) = jacc( 63, 44)+ 0.2500000000000000d+00*dw(140, 44)
jacc( 44, 55) = jacc( 44, 55) - dw(141, 55)
jacc( 44, 44) = jacc( 44, 44) - dw(141, 44)
jacc( 45, 55) = jacc( 45, 55)+ 0.9120000000000000d+00*dw(141, 55)
jacc( 45, 44) = jacc( 45, 44)+ 0.9120000000000000d+00*dw(141, 44)
jacc( 52, 55) = jacc( 52, 55)+ 0.8799999999999999d-01*dw(141, 55)
jacc( 52, 44) = jacc( 52, 44)+ 0.8799999999999999d-01*dw(141, 44)
jacc( 54, 55) = jacc( 54, 55)+ 0.6290000000000000d+00*dw(141, 55)
jacc( 54, 44) = jacc( 54, 44)+ 0.6290000000000000d+00*dw(141, 44)
jacc( 55, 55) = jacc( 55, 55) - dw(141, 55)
jacc( 55, 44) = jacc( 55, 44) - dw(141, 44)
jacc( 61, 55) = jacc( 61, 55)+ 0.9120000000000000d+00*dw(141, 55)
jacc( 61, 44) = jacc( 61, 44)+ 0.9120000000000000d+00*dw(141, 44)
jacc( 63, 55) = jacc( 63, 55)+ 0.9910000000000000d+00*dw(141, 55)
jacc( 63, 44) = jacc( 63, 44)+ 0.9910000000000000d+00*dw(141, 44)
jacc( 44, 59) = jacc( 44, 59) - dw(142, 59)
jacc( 44, 44) = jacc( 44, 44) - dw(142, 44)
jacc( 45, 59) = jacc( 45, 59)+ 0.6500000000000000d+00*dw(142, 59)
jacc( 45, 44) = jacc( 45, 44)+ 0.6500000000000000d+00*dw(142, 44)
jacc( 47, 59) = jacc( 47, 59)+ 0.3500000000000000d+00*dw(142, 59)
jacc( 47, 44) = jacc( 47, 44)+ 0.3500000000000000d+00*dw(142, 44)
jacc( 48, 59) = jacc( 48, 59)+ 0.1500000000000000d+00*dw(142, 59)
jacc( 48, 44) = jacc( 48, 44)+ 0.1500000000000000d+00*dw(142, 44)
jacc( 51, 59) = jacc( 51, 59)+ 0.6600000000000000d-01*dw(142, 59)
jacc( 51, 44) = jacc( 51, 44)+ 0.6600000000000000d-01*dw(142, 44)
jacc( 54, 59) = jacc( 54, 59)+ 0.6000000000000000d+00*dw(142, 59)
jacc( 54, 44) = jacc( 54, 44)+ 0.6000000000000000d+00*dw(142, 44)
jacc( 55, 59) = jacc( 55, 59)+ 0.2660000000000000d+00*dw(142, 59)
jacc( 55, 44) = jacc( 55, 44)+ 0.2660000000000000d+00*dw(142, 44)
jacc( 58, 59) = jacc( 58, 59)+ 0.2000000000000000d+00*dw(142, 59)
jacc( 58, 44) = jacc( 58, 44)+ 0.2000000000000000d+00*dw(142, 44)
jacc( 59, 59) = jacc( 59, 59) - dw(142, 59)
jacc( 59, 44) = jacc( 59, 44) - dw(142, 44)
jacc( 61, 59) = jacc( 61, 59)+ 0.6600000000000000d-01*dw(142, 59)
jacc( 61, 44) = jacc( 61, 44)+ 0.6600000000000000d-01*dw(142, 44)
jacc( 63, 59) = jacc( 63, 59)+ 0.2000000000000000d+00*dw(142, 59)
jacc( 63, 44) = jacc( 63, 44)+ 0.2000000000000000d+00*dw(142, 44)
jacc( 44, 56) = jacc( 44, 56) - dw(143, 56)
jacc( 44, 44) = jacc( 44, 44) - dw(143, 44)
jacc( 45, 56) = jacc( 45, 56)+ 0.2000000000000000d+00*dw(143, 56)
jacc( 45, 44) = jacc( 45, 44)+ 0.2000000000000000d+00*dw(143, 44)
jacc( 46, 56) = jacc( 46, 56)+ 0.8000000000000000d+00*dw(143, 56)
jacc( 46, 44) = jacc( 46, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 47, 56) = jacc( 47, 56)+ 0.2400000000000000d+01*dw(143, 56)
jacc( 47, 44) = jacc( 47, 44)+ 0.2400000000000000d+01*dw(143, 44)
jacc( 48, 56) = jacc( 48, 56)+ 0.8000000000000000d+00*dw(143, 56)
jacc( 48, 44) = jacc( 48, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 56, 56) = jacc( 56, 56) - dw(143, 56)
jacc( 56, 44) = jacc( 56, 44) - dw(143, 44)
jacc( 61, 56) = jacc( 61, 56)+ 0.8000000000000000d+00*dw(143, 56)
jacc( 61, 44) = jacc( 61, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 63, 56) = jacc( 63, 56) + dw(143, 56)
jacc( 63, 44) = jacc( 63, 44) + dw(143, 44)
jacc( 65, 56) = jacc( 65, 56)+ 0.2000000000000000d+00*dw(143, 56)
jacc( 65, 44) = jacc( 65, 44)+ 0.2000000000000000d+00*dw(143, 44)
jacc( 42, 55) = jacc( 42, 55)+ 0.1680000000000000d+00*dw(144, 55)
jacc( 42, 45) = jacc( 42, 45)+ 0.1680000000000000d+00*dw(144, 45)
jacc( 45, 55) = jacc( 45, 55) - dw(144, 55)
jacc( 45, 45) = jacc( 45, 45) - dw(144, 45)
jacc( 47, 55) = jacc( 47, 55)+ 0.1565000000000000d+01*dw(144, 55)
jacc( 47, 45) = jacc( 47, 45)+ 0.1565000000000000d+01*dw(144, 45)
jacc( 48, 55) = jacc( 48, 55)+ 0.1200000000000000d+00*dw(144, 55)
jacc( 48, 45) = jacc( 48, 45)+ 0.1200000000000000d+00*dw(144, 45)
jacc( 51, 55) = jacc( 51, 55)+ 0.3340000000000000d+00*dw(144, 55)
jacc( 51, 45) = jacc( 51, 45)+ 0.3340000000000000d+00*dw(144, 45)
jacc( 54, 55) = jacc( 54, 55)+ 0.1670000000000000d+00*dw(144, 55)
jacc( 54, 45) = jacc( 54, 45)+ 0.1670000000000000d+00*dw(144, 45)
jacc( 55, 55) = jacc( 55, 55) - dw(144, 55)
jacc( 55, 45) = jacc( 55, 45) - dw(144, 45)
jacc( 58, 55) = jacc( 58, 55)+ 0.2500000000000000d+00*dw(144, 55)
jacc( 58, 45) = jacc( 58, 45)+ 0.2500000000000000d+00*dw(144, 45)
jacc( 60, 55) = jacc( 60, 55)+ 0.2520000000000000d+00*dw(144, 55)
jacc( 60, 45) = jacc( 60, 45)+ 0.2520000000000000d+00*dw(144, 45)
jacc( 61, 55) = jacc( 61, 55)+ 0.5030000000000000d+00*dw(144, 55)
jacc( 61, 45) = jacc( 61, 45)+ 0.5030000000000000d+00*dw(144, 45)
jacc( 62, 55) = jacc( 62, 55)+ 0.2100000000000000d+00*dw(144, 55)
jacc( 62, 45) = jacc( 62, 45)+ 0.2100000000000000d+00*dw(144, 45)
jacc( 63, 55) = jacc( 63, 55)+ 0.7130000000000000d+00*dw(144, 55)
jacc( 63, 45) = jacc( 63, 45)+ 0.7130000000000000d+00*dw(144, 45)
jacc( 42, 59) = jacc( 42, 59)+ 0.8500000000000000d+00*dw(145, 59)
jacc( 42, 45) = jacc( 42, 45)+ 0.8500000000000000d+00*dw(145, 45)
jacc( 45, 59) = jacc( 45, 59) - dw(145, 59)
jacc( 45, 45) = jacc( 45, 45) - dw(145, 45)
jacc( 47, 59) = jacc( 47, 59)+ 0.3600000000000000d+00*dw(145, 59)
jacc( 47, 45) = jacc( 47, 45)+ 0.3600000000000000d+00*dw(145, 45)
jacc( 51, 59) = jacc( 51, 59)+ 0.2250000000000000d+00*dw(145, 59)
jacc( 51, 45) = jacc( 51, 45)+ 0.2250000000000000d+00*dw(145, 45)
jacc( 54, 59) = jacc( 54, 59)+ 0.1500000000000000d+00*dw(145, 59)
jacc( 54, 45) = jacc( 54, 45)+ 0.1500000000000000d+00*dw(145, 45)
jacc( 55, 59) = jacc( 55, 59)+ 0.2680000000000000d+00*dw(145, 59)
jacc( 55, 45) = jacc( 55, 45)+ 0.2680000000000000d+00*dw(145, 45)
jacc( 59, 59) = jacc( 59, 59) - dw(145, 59)
jacc( 59, 45) = jacc( 59, 45) - dw(145, 45)
jacc( 60, 59) = jacc( 60, 59)+ 0.2000000000000000d-01*dw(145, 59)
jacc( 60, 45) = jacc( 60, 45)+ 0.2000000000000000d-01*dw(145, 45)
jacc( 61, 59) = jacc( 61, 59)+ 0.1540000000000000d+00*dw(145, 59)
jacc( 61, 45) = jacc( 61, 45)+ 0.1540000000000000d+00*dw(145, 45)
jacc( 62, 59) = jacc( 62, 59)+ 0.1140000000000000d+00*dw(145, 59)
jacc( 62, 45) = jacc( 62, 45)+ 0.1140000000000000d+00*dw(145, 45)
jacc( 63, 59) = jacc( 63, 59)+ 0.6400000000000000d-01*dw(145, 59)
jacc( 63, 45) = jacc( 63, 45)+ 0.6400000000000000d-01*dw(145, 45)
jacc( 45, 56) = jacc( 45, 56) - dw(146, 56)
jacc( 45, 45) = jacc( 45, 45) - dw(146, 45)
jacc( 46, 56) = jacc( 46, 56)+ 0.8500000000000000d+00*dw(146, 56)
jacc( 46, 45) = jacc( 46, 45)+ 0.8500000000000000d+00*dw(146, 45)
jacc( 47, 56) = jacc( 47, 56)+ 0.1282000000000000d+01*dw(146, 56)
jacc( 47, 45) = jacc( 47, 45)+ 0.1282000000000000d+01*dw(146, 45)
jacc( 48, 56) = jacc( 48, 56)+ 0.3570000000000000d+00*dw(146, 56)
jacc( 48, 45) = jacc( 48, 45)+ 0.3570000000000000d+00*dw(146, 45)
jacc( 50, 56) = jacc( 50, 56)+ 0.1500000000000000d+00*dw(146, 56)
jacc( 50, 45) = jacc( 50, 45)+ 0.1500000000000000d+00*dw(146, 45)
jacc( 51, 56) = jacc( 51, 56)+ 0.6430000000000000d+00*dw(146, 56)
jacc( 51, 45) = jacc( 51, 45)+ 0.6430000000000000d+00*dw(146, 45)
jacc( 54, 56) = jacc( 54, 56)+ 0.2820000000000000d+00*dw(146, 56)
jacc( 54, 45) = jacc( 54, 45)+ 0.2820000000000000d+00*dw(146, 45)
jacc( 56, 56) = jacc( 56, 56) - dw(146, 56)
jacc( 56, 45) = jacc( 56, 45) - dw(146, 45)
jacc( 58, 56) = jacc( 58, 56)+ 0.7500000000000000d-01*dw(146, 56)
jacc( 58, 45) = jacc( 58, 45)+ 0.7500000000000000d-01*dw(146, 45)
jacc( 61, 56) = jacc( 61, 56)+ 0.9250000000000000d+00*dw(146, 56)
jacc( 61, 45) = jacc( 61, 45)+ 0.9250000000000000d+00*dw(146, 45)
jacc( 63, 56) = jacc( 63, 56)+ 0.7500000000000000d-01*dw(146, 56)
jacc( 63, 45) = jacc( 63, 45)+ 0.7500000000000000d-01*dw(146, 45)
jacc( 45, 45) = jacc( 45, 45) - dw(147, 45)
jacc( 47, 45) = jacc( 47, 45)+ 0.8320000000000000d+00*dw(147, 45)
jacc( 51, 45) = jacc( 51, 45)+ 0.3330000000000000d+00*dw(147, 45)
jacc( 54, 45) = jacc( 54, 45)+ 0.9000000000000000d+00*dw(147, 45)
jacc( 60, 45) = jacc( 60, 45)+ 0.6700000000000000d-01*dw(147, 45)
jacc( 61, 45) = jacc( 61, 45)+ 0.1033000000000000d+01*dw(147, 45)
jacc( 62, 45) = jacc( 62, 45)+ 0.9670000000000000d+00*dw(147, 45)
jacc( 63, 45) = jacc( 63, 45)+ 0.7000000000000000d+00*dw(147, 45)
jacc( 39, 39) = jacc( 39, 39) - dw(148, 39)
jacc( 39, 57) = jacc( 39, 57) - dw(148, 57)
jacc( 47, 39) = jacc( 47, 39)+ 0.5120000000000000d+01*dw(148, 39)
jacc( 47, 57) = jacc( 47, 57)+ 0.5120000000000000d+01*dw(148, 57)
jacc( 48, 39) = jacc( 48, 39)+ 0.1500000000000000d+00*dw(148, 39)
jacc( 48, 57) = jacc( 48, 57)+ 0.1500000000000000d+00*dw(148, 57)
jacc( 57, 39) = jacc( 57, 39) - dw(148, 39)
jacc( 57, 57) = jacc( 57, 57) - dw(148, 57)
jacc( 39, 39) = jacc( 39, 39) - dw(149, 39)
jacc( 39, 55) = jacc( 39, 55) - dw(149, 55)
jacc( 47, 39) = jacc( 47, 39)+ 0.1660000000000000d+01*dw(149, 39)
jacc( 47, 55) = jacc( 47, 55)+ 0.1660000000000000d+01*dw(149, 55)
jacc( 48, 39) = jacc( 48, 39)+ 0.4700000000000000d+00*dw(149, 39)
jacc( 48, 55) = jacc( 48, 55)+ 0.4700000000000000d+00*dw(149, 55)
jacc( 52, 39) = jacc( 52, 39)+ 0.2500000000000000d+00*dw(149, 39)
jacc( 52, 55) = jacc( 52, 55)+ 0.2500000000000000d+00*dw(149, 55)
jacc( 54, 39) = jacc( 54, 39)+ 0.2800000000000000d+00*dw(149, 39)
jacc( 54, 55) = jacc( 54, 55)+ 0.2800000000000000d+00*dw(149, 55)
jacc( 55, 39) = jacc( 55, 39) - dw(149, 39)
jacc( 55, 55) = jacc( 55, 55) - dw(149, 55)
jacc( 61, 39) = jacc( 61, 39)+ 0.7500000000000000d+00*dw(149, 39)
jacc( 61, 55) = jacc( 61, 55)+ 0.7500000000000000d+00*dw(149, 55)
jacc( 63, 39) = jacc( 63, 39)+ 0.1250000000000000d+01*dw(149, 39)
jacc( 63, 55) = jacc( 63, 55)+ 0.1250000000000000d+01*dw(149, 55)
jacc( 39, 39) = jacc( 39, 39) - dw(150, 39)
jacc( 39, 59) = jacc( 39, 59) - dw(150, 59)
jacc( 47, 39) = jacc( 47, 39)+ 0.7000000000000000d+01*dw(150, 39)
jacc( 47, 59) = jacc( 47, 59)+ 0.7000000000000000d+01*dw(150, 59)
jacc( 48, 39) = jacc( 48, 39)+ 0.2100000000000000d+00*dw(150, 39)
jacc( 48, 59) = jacc( 48, 59)+ 0.2100000000000000d+00*dw(150, 59)
jacc( 51, 39) = jacc( 51, 39)+ 0.1000000000000000d-02*dw(150, 39)
jacc( 51, 59) = jacc( 51, 59)+ 0.1000000000000000d-02*dw(150, 59)
jacc( 52, 39) = jacc( 52, 39)+ 0.1800000000000000d+00*dw(150, 39)
jacc( 52, 59) = jacc( 52, 59)+ 0.1800000000000000d+00*dw(150, 59)
jacc( 54, 39) = jacc( 54, 39)+ 0.2400000000000000d+00*dw(150, 39)
jacc( 54, 59) = jacc( 54, 59)+ 0.2400000000000000d+00*dw(150, 59)
jacc( 55, 39) = jacc( 55, 39)+ 0.5700000000000000d+00*dw(150, 39)
jacc( 55, 59) = jacc( 55, 59)+ 0.5700000000000000d+00*dw(150, 59)
jacc( 58, 39) = jacc( 58, 39)+ 0.3900000000000000d+00*dw(150, 39)
jacc( 58, 59) = jacc( 58, 59)+ 0.3900000000000000d+00*dw(150, 59)
jacc( 59, 39) = jacc( 59, 39) - dw(150, 39)
jacc( 59, 59) = jacc( 59, 59) - dw(150, 59)
jacc( 61, 39) = jacc( 61, 39)+ 0.7000000000000001d-01*dw(150, 39)
jacc( 61, 59) = jacc( 61, 59)+ 0.7000000000000001d-01*dw(150, 59)
jacc( 63, 39) = jacc( 63, 39)+ 0.7600000000000000d+00*dw(150, 39)
jacc( 63, 59) = jacc( 63, 59)+ 0.7600000000000000d+00*dw(150, 59)
jacc( 39, 39) = jacc( 39, 39) - dw(151, 39)
jacc( 39, 56) = jacc( 39, 56) - dw(151, 56)
jacc( 46, 39) = jacc( 46, 39)+ 0.5300000000000000d+00*dw(151, 39)
jacc( 46, 56) = jacc( 46, 56)+ 0.5300000000000000d+00*dw(151, 56)
jacc( 48, 39) = jacc( 48, 39)+ 0.4700000000000000d+00*dw(151, 39)
jacc( 48, 56) = jacc( 48, 56)+ 0.4700000000000000d+00*dw(151, 56)
jacc( 52, 39) = jacc( 52, 39)+ 0.2500000000000000d+00*dw(151, 39)
jacc( 52, 56) = jacc( 52, 56)+ 0.2500000000000000d+00*dw(151, 56)
jacc( 56, 39) = jacc( 56, 39) - dw(151, 39)
jacc( 56, 56) = jacc( 56, 56) - dw(151, 56)
jacc( 61, 39) = jacc( 61, 39)+ 0.2800000000000000d+00*dw(151, 39)
jacc( 61, 56) = jacc( 61, 56)+ 0.2800000000000000d+00*dw(151, 56)
jacc( 63, 39) = jacc( 63, 39)+ 0.1030000000000000d+01*dw(151, 39)
jacc( 63, 56) = jacc( 63, 56)+ 0.1030000000000000d+01*dw(151, 56)
jacc( 65, 39) = jacc( 65, 39)+ 0.4700000000000000d+00*dw(151, 39)
jacc( 65, 56) = jacc( 65, 56)+ 0.4700000000000000d+00*dw(151, 56)
jacc(  9,  9) = jacc(  9,  9) - dw(152,  9)
jacc(  9, 55) = jacc(  9, 55) - dw(152, 55)
jacc( 10,  9) = jacc( 10,  9) + dw(152,  9)
jacc( 10, 55) = jacc( 10, 55) + dw(152, 55)
jacc( 55,  9) = jacc( 55,  9) - dw(152,  9)
jacc( 55, 55) = jacc( 55, 55) - dw(152, 55)
jacc( 61,  9) = jacc( 61,  9) + dw(152,  9)
jacc( 61, 55) = jacc( 61, 55) + dw(152, 55)
jacc(  4, 55) = jacc(  4, 55) - dw(153, 55)
jacc(  4,  4) = jacc(  4,  4) - dw(153,  4)
jacc( 48, 55) = jacc( 48, 55)+ 0.5000000000000000d-01*dw(153, 55)
jacc( 48,  4) = jacc( 48,  4)+ 0.5000000000000000d-01*dw(153,  4)
jacc( 54, 55) = jacc( 54, 55)+ 0.1000000000000000d+00*dw(153, 55)
jacc( 54,  4) = jacc( 54,  4)+ 0.1000000000000000d+00*dw(153,  4)
jacc( 55, 55) = jacc( 55, 55) - dw(153, 55)
jacc( 55,  4) = jacc( 55,  4) - dw(153,  4)
jacc( 60, 55) = jacc( 60, 55)+ 0.9000000000000000d+00*dw(153, 55)
jacc( 60,  4) = jacc( 60,  4)+ 0.9000000000000000d+00*dw(153,  4)
jacc( 61, 55) = jacc( 61, 55) + dw(153, 55)
jacc( 61,  4) = jacc( 61,  4) + dw(153,  4)
jacc( 63, 55) = jacc( 63, 55)+ 0.1000000000000000d+00*dw(153, 55)
jacc( 63,  4) = jacc( 63,  4)+ 0.1000000000000000d+00*dw(153,  4)
jacc(  6, 55) = jacc(  6, 55) - dw(154, 55)
jacc(  6,  6) = jacc(  6,  6) - dw(154,  6)
jacc( 52, 55) = jacc( 52, 55)+ 0.8999999999999999d-02*dw(154, 55)
jacc( 52,  6) = jacc( 52,  6)+ 0.8999999999999999d-02*dw(154,  6)
jacc( 55, 55) = jacc( 55, 55) - dw(154, 55)
jacc( 55,  6) = jacc( 55,  6) - dw(154,  6)
jacc( 60, 55) = jacc( 60, 55)+ 0.9910000000000000d+00*dw(154, 55)
jacc( 60,  6) = jacc( 60,  6)+ 0.9910000000000000d+00*dw(154,  6)
jacc( 61, 55) = jacc( 61, 55) + dw(154, 55)
jacc( 61,  6) = jacc( 61,  6) + dw(154,  6)
jacc( 63, 55) = jacc( 63, 55)+ 0.9910000000000000d+00*dw(154, 55)
jacc( 63,  6) = jacc( 63,  6)+ 0.9910000000000000d+00*dw(154,  6)
jacc( 44, 65) = jacc( 44, 65) - dw(155, 65)
jacc( 44, 44) = jacc( 44, 44) - dw(155, 44)
jacc( 45, 65) = jacc( 45, 65)+ 0.2000000000000000d+00*dw(155, 65)
jacc( 45, 44) = jacc( 45, 44)+ 0.2000000000000000d+00*dw(155, 44)
jacc( 46, 65) = jacc( 46, 65)+ 0.8000000000000000d+00*dw(155, 65)
jacc( 46, 44) = jacc( 46, 44)+ 0.8000000000000000d+00*dw(155, 44)
jacc( 47, 65) = jacc( 47, 65)+ 0.2400000000000000d+01*dw(155, 65)
jacc( 47, 44) = jacc( 47, 44)+ 0.2400000000000000d+01*dw(155, 44)
jacc( 48, 65) = jacc( 48, 65)+ 0.8000000000000000d+00*dw(155, 65)
jacc( 48, 44) = jacc( 48, 44)+ 0.8000000000000000d+00*dw(155, 44)
jacc( 61, 65) = jacc( 61, 65)+ 0.8000000000000000d+00*dw(155, 65)
jacc( 61, 44) = jacc( 61, 44)+ 0.8000000000000000d+00*dw(155, 44)
jacc( 63, 65) = jacc( 63, 65) + dw(155, 65)
jacc( 63, 44) = jacc( 63, 44) + dw(155, 44)
jacc( 64, 65) = jacc( 64, 65)+ 0.2000000000000000d+00*dw(155, 65)
jacc( 64, 44) = jacc( 64, 44)+ 0.2000000000000000d+00*dw(155, 44)
jacc( 65, 65) = jacc( 65, 65) - dw(155, 65)
jacc( 65, 44) = jacc( 65, 44) - dw(155, 44)
jacc( 11, 11) = jacc( 11, 11) - dw(156, 11)
jacc( 11, 55) = jacc( 11, 55) - dw(156, 55)
jacc( 12, 11) = jacc( 12, 11)+ 0.4800000000000000d-01*dw(156, 11)
jacc( 12, 55) = jacc( 12, 55)+ 0.4800000000000000d-01*dw(156, 55)
jacc( 24, 24) = jacc( 24, 24) - dw(157, 24)
jacc( 24, 55) = jacc( 24, 55) - dw(157, 55)
jacc( 25, 24) = jacc( 25, 24)+ 0.6384000000000000d-02*dw(157, 24)
jacc( 25, 55) = jacc( 25, 55)+ 0.6384000000000000d-02*dw(157, 55)
jacc( 26, 24) = jacc( 26, 24)+ 0.5494800000000000d-01*dw(157, 24)
jacc( 26, 55) = jacc( 26, 55)+ 0.5494800000000000d-01*dw(157, 55)
jacc( 24, 24) = jacc( 24, 24) - dw(158, 24)
jacc( 24, 59) = jacc( 24, 59) - dw(158, 59)
jacc( 25, 24) = jacc( 25, 24)+ 0.2158800000000000d-01*dw(158, 24)
jacc( 25, 59) = jacc( 25, 59)+ 0.2158800000000000d-01*dw(158, 59)
jacc( 26, 24) = jacc( 26, 24)+ 0.1858110000000000d+00*dw(158, 24)
jacc( 26, 59) = jacc( 26, 59)+ 0.1858110000000000d+00*dw(158, 59)
jacc( 27, 27) = jacc( 27, 27) - dw(159, 27)
jacc( 27, 55) = jacc( 27, 55) - dw(159, 55)
jacc( 28, 27) = jacc( 28, 27)+ 0.3716400000000000d-01*dw(159, 27)
jacc( 28, 55) = jacc( 28, 55)+ 0.3716400000000000d-01*dw(159, 55)
jacc( 29, 27) = jacc( 29, 27)+ 0.5631600000000000d-01*dw(159, 27)
jacc( 29, 55) = jacc( 29, 55)+ 0.5631600000000000d-01*dw(159, 55)
jacc( 27, 27) = jacc( 27, 27) - dw(160, 27)
jacc( 27, 59) = jacc( 27, 59) - dw(160, 59)
jacc( 28, 27) = jacc( 28, 27)+ 0.1256730000000000d+00*dw(160, 27)
jacc( 28, 59) = jacc( 28, 59)+ 0.1256730000000000d+00*dw(160, 59)
jacc( 29, 27) = jacc( 29, 27)+ 0.1904370000000000d+00*dw(160, 27)
jacc( 29, 59) = jacc( 29, 59)+ 0.1904370000000000d+00*dw(160, 59)
jacc( 24, 24) = jacc( 24, 24) - dw(161, 24)
jacc( 24, 56) = jacc( 24, 56) - dw(161, 56)
jacc( 27, 27) = jacc( 27, 27) - dw(162, 27)
jacc( 27, 56) = jacc( 27, 56) - dw(162, 56)

do j = 1, ns
   do i = 1, ns
      jacc(i,j) = jacc(i,j) * convers_factor_jac(i,j)
   enddo
enddo


return
end subroutine jacdchemdc_siream


!===============================================================================

!> rates_siream
!> \brief Computation of reaction rates
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     ns                 total number of chemical species
!> \param[in]     rk                 kinetic rates
!> \param[in]     y                  concentrations vector
!> \param[out]    w                  reaction rates
!______________________________________________________________________________

subroutine rates_siream(ns,nr,rk,y,w)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns)

! Local variables

double precision w(nr)

w(  1) =  rk(  1) * y( 65)
w(  2) =  rk(  2) * y( 57)
w(  3) =  rk(  3) * y( 59) * y( 64)
w(  4) =  rk(  4) * y( 57) * y( 65)
w(  5) =  rk(  5) * y( 57) * y( 65)
w(  6) =  rk(  6) * y( 57) * y( 64)
w(  7) =  rk(  7) * y( 65) * y( 59)
w(  8) =  rk(  8) * y( 59)
w(  9) =  rk(  9) * y( 59)
w( 10) =  rk( 10) * y(  3)
w( 11) =  rk( 11) * y(  3)
w( 12) =  rk( 12) * y( 59) * y( 55)
w( 13) =  rk( 13) * y( 59) * y( 61)
w( 14) =  rk( 14) * y( 56)
w( 15) =  rk( 15) * y( 56)
w( 16) =  rk( 16) * y( 56) * y( 64)
w( 17) =  rk( 17) * y( 56) * y( 65)
w( 18) =  rk( 18) * y( 56) * y( 65)
w( 19) =  rk( 19) * y( 16)
w( 20) =  rk( 20) * y( 16)
w( 21) =  rk( 21) * y( 64) * y( 64)
w( 22) =  rk( 22) * y( 64) * y( 65)
w( 23) =  rk( 23) * y( 55) * y( 64)
w( 24) =  rk( 24) * y( 23)
w( 25) =  rk( 25) * y( 55) * y( 23)
w( 26) =  rk( 26) * y( 23) * y( 23)
w( 27) =  rk( 27) * y( 65) * y( 55)
w( 28) =  rk( 28) * y( 55) * y( 50)
w( 29) =  rk( 29) * y( 61) * y( 64)
w( 30) =  rk( 30) * y( 61) * y( 65)
w( 31) =  rk( 31) * y( 17)
w( 32) =  rk( 32) * y( 55) * y( 17)
w( 33) =  rk( 33) * y( 61) * y( 61)
w( 34) =  rk( 34) * y( 61) * y( 61)
w( 35) =  rk( 35) * y( 20)
w( 36) =  rk( 36) * y( 55) * y( 20)
w( 37) =  rk( 37) * y(  3)
w( 38) =  rk( 38) * y( 55)
w( 39) =  rk( 39) * y( 55) * y( 57)
w( 40) =  rk( 40) * y( 55) * y( 55)
w( 41) =  rk( 41) * y( 55) * y( 55)
w( 42) =  rk( 42) * y( 55) * y( 61)
w( 43) =  rk( 43) * y( 61) * y( 57)
w( 44) =  rk( 44) * y( 20) * y( 57)
w( 45) =  rk( 45) * y( 56) * y( 57)
w( 46) =  rk( 46) * y( 56) * y( 55)
w( 47) =  rk( 47) * y( 56) * y( 61)
w( 48) =  rk( 48) * y( 56) * y( 59)
w( 49) =  rk( 49) * y( 56) * y( 56)
w( 50) =  rk( 50) * y( 17)
w( 51) =  rk( 51) * y( 50)
w( 52) =  rk( 52) * y( 16)
w( 53) =  rk( 53) * y( 63) * y( 64)
w( 54) =  rk( 54) * y( 52) * y( 64)
w( 55) =  rk( 55) * y( 63) * y( 61)
w( 56) =  rk( 56) * y( 52) * y( 61)
w( 57) =  rk( 57) * y( 63) * y( 63)
w( 58) =  rk( 58) * y( 52) * y( 52)
w( 59) =  rk( 59) * y( 63) * y( 52)
w( 60) =  rk( 60) * y( 46) * y( 55)
w( 61) =  rk( 61) * y( 46)
w( 62) =  rk( 62) * y( 32) * y( 55)
w( 63) =  rk( 63) * y( 32)
w( 64) =  rk( 64) * y( 55) * y( 51)
w( 65) =  rk( 65) * y( 55) * y(  5)
w( 66) =  rk( 66) * y( 53) * y( 64)
w( 67) =  rk( 67) * y( 53) * y( 61)
w( 68) =  rk( 68) * y( 53) * y( 53)
w( 69) =  rk( 69) * y( 22) * y( 55)
w( 70) =  rk( 70) * y( 22)
w( 71) =  rk( 71) * y( 14) * y( 55)
w( 72) =  rk( 72) * y( 54) * y( 55)
w( 73) =  rk( 73) * y( 54)
w( 74) =  rk( 74) * y( 54)
w( 75) =  rk( 75) * y( 54) * y( 57)
w( 76) =  rk( 76) * y( 54) * y( 56)
w( 77) =  rk( 77) * y( 54) * y( 61)
w( 78) =  rk( 78) * y( 21)
w( 79) =  rk( 79) * y( 21) * y( 64)
w( 80) =  rk( 80) * y( 21) * y( 61)
w( 81) =  rk( 81) * y( 35) * y( 55)
w( 82) =  rk( 82) * y( 60) * y( 57)
w( 83) =  rk( 83) * y( 60) * y( 55)
w( 84) =  rk( 84) * y( 60) * y( 56)
w( 85) =  rk( 85) * y( 60)
w( 86) =  rk( 86) * y( 62) * y( 64)
w( 87) =  rk( 87) * y( 62) * y( 65)
w( 88) =  rk( 88) * y( 13)
w( 89) =  rk( 89) * y( 13)
w( 90) =  rk( 90) * y( 62) * y( 61)
w( 91) =  rk( 91) * y( 62) * y( 53)
w( 92) =  rk( 92) * y( 62) * y( 63)
w( 93) =  rk( 93) * y( 62) * y( 62)
w( 94) =  rk( 94) * y( 33) * y( 55)
w( 95) =  rk( 95) * y( 33)
w( 96) =  rk( 96) * y( 49) * y( 55)
w( 97) =  rk( 97) * y( 48) * y( 57)
w( 98) =  rk( 98) * y( 48) * y( 55)
w( 99) =  rk( 99) * y( 48) * y( 56)
w(100) =  rk(100) * y( 48)
w(101) =  rk(101) * y( 58) * y( 64)
w(102) =  rk(102) * y( 58) * y( 65)
w(103) =  rk(103) * y( 18)
w(104) =  rk(104) * y( 18)
w(105) =  rk(105) * y( 18) * y( 55)
w(106) =  rk(106) * y( 58) * y( 61)
w(107) =  rk(107) * y( 62) * y( 53)
w(108) =  rk(108) * y( 62) * y( 63)
w(109) =  rk(109) * y( 58) * y( 58)
w(110) =  rk(110) * y( 58) * y( 62)
w(111) =  rk(111) * y( 47) * y( 55)
w(112) =  rk(112) * y( 19)
w(113) =  rk(113) * y( 19)
w(114) =  rk(114) * y( 19) * y( 65)
w(115) =  rk(115) * y( 57) * y( 36)
w(116) =  rk(116) * y( 55) * y( 36)
w(117) =  rk(117) * y( 59) * y( 36)
w(118) =  rk(118) * y( 56) * y( 36)
w(119) =  rk(119) * y( 57) * y( 34)
w(120) =  rk(120) * y( 55) * y( 34)
w(121) =  rk(121) * y( 59) * y( 34)
w(122) =  rk(122) * y( 56) * y( 34)
w(123) =  rk(123) * y( 37) * y( 57)
w(124) =  rk(124) * y( 37) * y( 55)
w(125) =  rk(125) * y( 37) * y( 59)
w(126) =  rk(126) * y( 37) * y( 56)
w(127) =  rk(127) * y(  7) * y( 55)
w(128) =  rk(128) * y( 15) * y( 64)
w(129) =  rk(129) * y( 15)
w(130) =  rk(130) * y( 55) * y( 40)
w(131) =  rk(131) * y( 40) * y( 56)
w(132) =  rk(132) * y( 43) * y( 65)
w(133) =  rk(133) * y( 43) * y( 61)
w(134) =  rk(134) * y( 41)
w(135) =  rk(135) * y( 41) * y( 55)
w(136) =  rk(136) * y( 41) * y( 59)
w(137) =  rk(137) * y( 55) * y(  8)
w(138) =  rk(138) * y( 55) * y( 42)
w(139) =  rk(139) * y( 42)
w(140) =  rk(140) * y( 57) * y( 44)
w(141) =  rk(141) * y( 55) * y( 44)
w(142) =  rk(142) * y( 59) * y( 44)
w(143) =  rk(143) * y( 56) * y( 44)
w(144) =  rk(144) * y( 55) * y( 45)
w(145) =  rk(145) * y( 59) * y( 45)
w(146) =  rk(146) * y( 56) * y( 45)
w(147) =  rk(147) * y( 45)
w(148) =  rk(148) * y( 39) * y( 57)
w(149) =  rk(149) * y( 39) * y( 55)
w(150) =  rk(150) * y( 39) * y( 59)
w(151) =  rk(151) * y( 39) * y( 56)
w(152) =  rk(152) * y(  9) * y( 55)
w(153) =  rk(153) * y( 55) * y(  4)
w(154) =  rk(154) * y( 55) * y(  6)
w(155) =  rk(155) * y( 65) * y( 44)
w(156) =  rk(156) * y( 11) * y( 55)
w(157) =  rk(157) * y( 24) * y( 55)
w(158) =  rk(158) * y( 24) * y( 59)
w(159) =  rk(159) * y( 27) * y( 55)
w(160) =  rk(160) * y( 27) * y( 59)
w(161) =  rk(161) * y( 24) * y( 56)
w(162) =  rk(162) * y( 27) * y( 56)

return
end subroutine rates_siream


!===============================================================================

!> dratedc_siream
!> \brief Computation of derivatives of reaction rates
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     ns                 total number of chemical species
!> \param[in]     rk                 kinetic rates
!> \param[in]     y                  concentrations vector
!> \param[out]    dw                 derivatives of reaction rates
!______________________________________________________________________________

subroutine dratedc_siream(ns,nr,rk,y,dw)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns)

! Local variables

double precision dw(nr,ns)

dw(  1, 65) =  rk(  1)
dw(  2, 57) =  rk(  2)
dw(  3, 59) =  rk(  3) * y( 64)
dw(  3, 64) =  rk(  3) * y( 59)
dw(  4, 57) =  rk(  4) * y( 65)
dw(  4, 65) =  rk(  4) * y( 57)
dw(  5, 57) =  rk(  5) * y( 65)
dw(  5, 65) =  rk(  5) * y( 57)
dw(  6, 57) =  rk(  6) * y( 64)
dw(  6, 64) =  rk(  6) * y( 57)
dw(  7, 65) =  rk(  7) * y( 59)
dw(  7, 59) =  rk(  7) * y( 65)
dw(  8, 59) =  rk(  8)
dw(  9, 59) =  rk(  9)
dw( 10,  3) =  rk( 10)
dw( 11,  3) =  rk( 11)
dw( 12, 59) =  rk( 12) * y( 55)
dw( 12, 55) =  rk( 12) * y( 59)
dw( 13, 59) =  rk( 13) * y( 61)
dw( 13, 61) =  rk( 13) * y( 59)
dw( 14, 56) =  rk( 14)
dw( 15, 56) =  rk( 15)
dw( 16, 56) =  rk( 16) * y( 64)
dw( 16, 64) =  rk( 16) * y( 56)
dw( 17, 56) =  rk( 17) * y( 65)
dw( 17, 65) =  rk( 17) * y( 56)
dw( 18, 56) =  rk( 18) * y( 65)
dw( 18, 65) =  rk( 18) * y( 56)
dw( 19, 16) =  rk( 19)
dw( 20, 16) =  rk( 20)
dw( 21, 64) =  rk( 21) * y( 64)
dw( 21, 64) =  rk( 21) * y( 64)
dw( 22, 64) =  rk( 22) * y( 65)
dw( 22, 65) =  rk( 22) * y( 64)
dw( 23, 55) =  rk( 23) * y( 64)
dw( 23, 64) =  rk( 23) * y( 55)
dw( 24, 23) =  rk( 24)
dw( 25, 55) =  rk( 25) * y( 23)
dw( 25, 23) =  rk( 25) * y( 55)
dw( 26, 23) =  rk( 26) * y( 23)
dw( 26, 23) =  rk( 26) * y( 23)
dw( 27, 65) =  rk( 27) * y( 55)
dw( 27, 55) =  rk( 27) * y( 65)
dw( 28, 55) =  rk( 28) * y( 50)
dw( 28, 50) =  rk( 28) * y( 55)
dw( 29, 61) =  rk( 29) * y( 64)
dw( 29, 64) =  rk( 29) * y( 61)
dw( 30, 61) =  rk( 30) * y( 65)
dw( 30, 65) =  rk( 30) * y( 61)
dw( 31, 17) =  rk( 31)
dw( 32, 55) =  rk( 32) * y( 17)
dw( 32, 17) =  rk( 32) * y( 55)
dw( 33, 61) =  rk( 33) * y( 61)
dw( 33, 61) =  rk( 33) * y( 61)
dw( 34, 61) =  rk( 34) * y( 61)
dw( 34, 61) =  rk( 34) * y( 61)
dw( 35, 20) =  rk( 35)
dw( 36, 55) =  rk( 36) * y( 20)
dw( 36, 20) =  rk( 36) * y( 55)
dw( 37,  3) =  rk( 37)
dw( 38, 55) =  rk( 38)
dw( 39, 55) =  rk( 39) * y( 57)
dw( 39, 57) =  rk( 39) * y( 55)
dw( 40, 55) =  rk( 40) * y( 55)
dw( 40, 55) =  rk( 40) * y( 55)
dw( 41, 55) =  rk( 41) * y( 55)
dw( 41, 55) =  rk( 41) * y( 55)
dw( 42, 55) =  rk( 42) * y( 61)
dw( 42, 61) =  rk( 42) * y( 55)
dw( 43, 61) =  rk( 43) * y( 57)
dw( 43, 57) =  rk( 43) * y( 61)
dw( 44, 20) =  rk( 44) * y( 57)
dw( 44, 57) =  rk( 44) * y( 20)
dw( 45, 56) =  rk( 45) * y( 57)
dw( 45, 57) =  rk( 45) * y( 56)
dw( 46, 56) =  rk( 46) * y( 55)
dw( 46, 55) =  rk( 46) * y( 56)
dw( 47, 56) =  rk( 47) * y( 61)
dw( 47, 61) =  rk( 47) * y( 56)
dw( 48, 56) =  rk( 48) * y( 59)
dw( 48, 59) =  rk( 48) * y( 56)
dw( 49, 56) =  rk( 49) * y( 56)
dw( 49, 56) =  rk( 49) * y( 56)
dw( 50, 17) =  rk( 50)
dw( 51, 50) =  rk( 51)
dw( 52, 16) =  rk( 52)
dw( 53, 63) =  rk( 53) * y( 64)
dw( 53, 64) =  rk( 53) * y( 63)
dw( 54, 52) =  rk( 54) * y( 64)
dw( 54, 64) =  rk( 54) * y( 52)
dw( 55, 63) =  rk( 55) * y( 61)
dw( 55, 61) =  rk( 55) * y( 63)
dw( 56, 52) =  rk( 56) * y( 61)
dw( 56, 61) =  rk( 56) * y( 52)
dw( 57, 63) =  rk( 57) * y( 63)
dw( 57, 63) =  rk( 57) * y( 63)
dw( 58, 52) =  rk( 58) * y( 52)
dw( 58, 52) =  rk( 58) * y( 52)
dw( 59, 63) =  rk( 59) * y( 52)
dw( 59, 52) =  rk( 59) * y( 63)
dw( 60, 46) =  rk( 60) * y( 55)
dw( 60, 55) =  rk( 60) * y( 46)
dw( 61, 46) =  rk( 61)
dw( 62, 32) =  rk( 62) * y( 55)
dw( 62, 55) =  rk( 62) * y( 32)
dw( 63, 32) =  rk( 63)
dw( 64, 55) =  rk( 64) * y( 51)
dw( 64, 51) =  rk( 64) * y( 55)
dw( 65, 55) =  rk( 65) * y(  5)
dw( 65,  5) =  rk( 65) * y( 55)
dw( 66, 53) =  rk( 66) * y( 64)
dw( 66, 64) =  rk( 66) * y( 53)
dw( 67, 53) =  rk( 67) * y( 61)
dw( 67, 61) =  rk( 67) * y( 53)
dw( 68, 53) =  rk( 68) * y( 53)
dw( 68, 53) =  rk( 68) * y( 53)
dw( 69, 22) =  rk( 69) * y( 55)
dw( 69, 55) =  rk( 69) * y( 22)
dw( 70, 22) =  rk( 70)
dw( 71, 14) =  rk( 71) * y( 55)
dw( 71, 55) =  rk( 71) * y( 14)
dw( 72, 54) =  rk( 72) * y( 55)
dw( 72, 55) =  rk( 72) * y( 54)
dw( 73, 54) =  rk( 73)
dw( 74, 54) =  rk( 74)
dw( 75, 54) =  rk( 75) * y( 57)
dw( 75, 57) =  rk( 75) * y( 54)
dw( 76, 54) =  rk( 76) * y( 56)
dw( 76, 56) =  rk( 76) * y( 54)
dw( 77, 54) =  rk( 77) * y( 61)
dw( 77, 61) =  rk( 77) * y( 54)
dw( 78, 21) =  rk( 78)
dw( 79, 21) =  rk( 79) * y( 64)
dw( 79, 64) =  rk( 79) * y( 21)
dw( 80, 21) =  rk( 80) * y( 61)
dw( 80, 61) =  rk( 80) * y( 21)
dw( 81, 35) =  rk( 81) * y( 55)
dw( 81, 55) =  rk( 81) * y( 35)
dw( 82, 60) =  rk( 82) * y( 57)
dw( 82, 57) =  rk( 82) * y( 60)
dw( 83, 60) =  rk( 83) * y( 55)
dw( 83, 55) =  rk( 83) * y( 60)
dw( 84, 60) =  rk( 84) * y( 56)
dw( 84, 56) =  rk( 84) * y( 60)
dw( 85, 60) =  rk( 85)
dw( 86, 62) =  rk( 86) * y( 64)
dw( 86, 64) =  rk( 86) * y( 62)
dw( 87, 62) =  rk( 87) * y( 65)
dw( 87, 65) =  rk( 87) * y( 62)
dw( 88, 13) =  rk( 88)
dw( 89, 13) =  rk( 89)
dw( 90, 62) =  rk( 90) * y( 61)
dw( 90, 61) =  rk( 90) * y( 62)
dw( 91, 62) =  rk( 91) * y( 53)
dw( 91, 53) =  rk( 91) * y( 62)
dw( 92, 62) =  rk( 92) * y( 63)
dw( 92, 63) =  rk( 92) * y( 62)
dw( 93, 62) =  rk( 93) * y( 62)
dw( 93, 62) =  rk( 93) * y( 62)
dw( 94, 33) =  rk( 94) * y( 55)
dw( 94, 55) =  rk( 94) * y( 33)
dw( 95, 33) =  rk( 95)
dw( 96, 49) =  rk( 96) * y( 55)
dw( 96, 55) =  rk( 96) * y( 49)
dw( 97, 48) =  rk( 97) * y( 57)
dw( 97, 57) =  rk( 97) * y( 48)
dw( 98, 48) =  rk( 98) * y( 55)
dw( 98, 55) =  rk( 98) * y( 48)
dw( 99, 48) =  rk( 99) * y( 56)
dw( 99, 56) =  rk( 99) * y( 48)
dw(100, 48) =  rk(100)
dw(101, 58) =  rk(101) * y( 64)
dw(101, 64) =  rk(101) * y( 58)
dw(102, 58) =  rk(102) * y( 65)
dw(102, 65) =  rk(102) * y( 58)
dw(103, 18) =  rk(103)
dw(104, 18) =  rk(104)
dw(105, 18) =  rk(105) * y( 55)
dw(105, 55) =  rk(105) * y( 18)
dw(106, 58) =  rk(106) * y( 61)
dw(106, 61) =  rk(106) * y( 58)
dw(107, 62) =  rk(107) * y( 53)
dw(107, 53) =  rk(107) * y( 62)
dw(108, 62) =  rk(108) * y( 63)
dw(108, 63) =  rk(108) * y( 62)
dw(109, 58) =  rk(109) * y( 58)
dw(109, 58) =  rk(109) * y( 58)
dw(110, 58) =  rk(110) * y( 62)
dw(110, 62) =  rk(110) * y( 58)
dw(111, 47) =  rk(111) * y( 55)
dw(111, 55) =  rk(111) * y( 47)
dw(112, 19) =  rk(112)
dw(113, 19) =  rk(113)
dw(114, 19) =  rk(114) * y( 65)
dw(114, 65) =  rk(114) * y( 19)
dw(115, 57) =  rk(115) * y( 36)
dw(115, 36) =  rk(115) * y( 57)
dw(116, 55) =  rk(116) * y( 36)
dw(116, 36) =  rk(116) * y( 55)
dw(117, 59) =  rk(117) * y( 36)
dw(117, 36) =  rk(117) * y( 59)
dw(118, 56) =  rk(118) * y( 36)
dw(118, 36) =  rk(118) * y( 56)
dw(119, 57) =  rk(119) * y( 34)
dw(119, 34) =  rk(119) * y( 57)
dw(120, 55) =  rk(120) * y( 34)
dw(120, 34) =  rk(120) * y( 55)
dw(121, 59) =  rk(121) * y( 34)
dw(121, 34) =  rk(121) * y( 59)
dw(122, 56) =  rk(122) * y( 34)
dw(122, 34) =  rk(122) * y( 56)
dw(123, 37) =  rk(123) * y( 57)
dw(123, 57) =  rk(123) * y( 37)
dw(124, 37) =  rk(124) * y( 55)
dw(124, 55) =  rk(124) * y( 37)
dw(125, 37) =  rk(125) * y( 59)
dw(125, 59) =  rk(125) * y( 37)
dw(126, 37) =  rk(126) * y( 56)
dw(126, 56) =  rk(126) * y( 37)
dw(127,  7) =  rk(127) * y( 55)
dw(127, 55) =  rk(127) * y(  7)
dw(128, 15) =  rk(128) * y( 64)
dw(128, 64) =  rk(128) * y( 15)
dw(129, 15) =  rk(129)
dw(130, 55) =  rk(130) * y( 40)
dw(130, 40) =  rk(130) * y( 55)
dw(131, 40) =  rk(131) * y( 56)
dw(131, 56) =  rk(131) * y( 40)
dw(132, 43) =  rk(132) * y( 65)
dw(132, 65) =  rk(132) * y( 43)
dw(133, 43) =  rk(133) * y( 61)
dw(133, 61) =  rk(133) * y( 43)
dw(134, 41) =  rk(134)
dw(135, 41) =  rk(135) * y( 55)
dw(135, 55) =  rk(135) * y( 41)
dw(136, 41) =  rk(136) * y( 59)
dw(136, 59) =  rk(136) * y( 41)
dw(137, 55) =  rk(137) * y(  8)
dw(137,  8) =  rk(137) * y( 55)
dw(138, 55) =  rk(138) * y( 42)
dw(138, 42) =  rk(138) * y( 55)
dw(139, 42) =  rk(139)
dw(140, 57) =  rk(140) * y( 44)
dw(140, 44) =  rk(140) * y( 57)
dw(141, 55) =  rk(141) * y( 44)
dw(141, 44) =  rk(141) * y( 55)
dw(142, 59) =  rk(142) * y( 44)
dw(142, 44) =  rk(142) * y( 59)
dw(143, 56) =  rk(143) * y( 44)
dw(143, 44) =  rk(143) * y( 56)
dw(144, 55) =  rk(144) * y( 45)
dw(144, 45) =  rk(144) * y( 55)
dw(145, 59) =  rk(145) * y( 45)
dw(145, 45) =  rk(145) * y( 59)
dw(146, 56) =  rk(146) * y( 45)
dw(146, 45) =  rk(146) * y( 56)
dw(147, 45) =  rk(147)
dw(148, 39) =  rk(148) * y( 57)
dw(148, 57) =  rk(148) * y( 39)
dw(149, 39) =  rk(149) * y( 55)
dw(149, 55) =  rk(149) * y( 39)
dw(150, 39) =  rk(150) * y( 59)
dw(150, 59) =  rk(150) * y( 39)
dw(151, 39) =  rk(151) * y( 56)
dw(151, 56) =  rk(151) * y( 39)
dw(152,  9) =  rk(152) * y( 55)
dw(152, 55) =  rk(152) * y(  9)
dw(153, 55) =  rk(153) * y(  4)
dw(153,  4) =  rk(153) * y( 55)
dw(154, 55) =  rk(154) * y(  6)
dw(154,  6) =  rk(154) * y( 55)
dw(155, 65) =  rk(155) * y( 44)
dw(155, 44) =  rk(155) * y( 65)
dw(156, 11) =  rk(156) * y( 55)
dw(156, 55) =  rk(156) * y( 11)
dw(157, 24) =  rk(157) * y( 55)
dw(157, 55) =  rk(157) * y( 24)
dw(158, 24) =  rk(158) * y( 59)
dw(158, 59) =  rk(158) * y( 24)
dw(159, 27) =  rk(159) * y( 55)
dw(159, 55) =  rk(159) * y( 27)
dw(160, 27) =  rk(160) * y( 59)
dw(160, 59) =  rk(160) * y( 27)
dw(161, 24) =  rk(161) * y( 56)
dw(161, 56) =  rk(161) * y( 24)
dw(162, 27) =  rk(162) * y( 56)
dw(162, 56) =  rk(162) * y( 27)

return
end subroutine dratedc_siream


!===============================================================================

!> lu_decompose_siream
!> \brief Computation of LU factorization of matrix m
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 matrix row number from the chemical species number
!> \param[in,out] m                  on entry, an invertible matrix. On exit, an LU factorization of m
!______________________________________________________________________________

subroutine lu_decompose_siream (ns,m)

implicit none

! Arguments

integer ns
double precision m(ns,ns)

! Local variables

double precision temp

!     Upper part.
m(3, 59) = m(3, 59) / m(3, 3)

!     Upper part.
m(4, 55) = m(4, 55) / m(4, 4)

!     Upper part.
m(5, 55) = m(5, 55) / m(5, 5)

!     Upper part.
m(6, 55) = m(6, 55) / m(6, 6)

!     Upper part.
m(7, 55) = m(7, 55) / m(7, 7)

!     Upper part.
m(8, 55) = m(8, 55) / m(8, 8)

!     Upper part.
m(9, 55) = m(9, 55) / m(9, 9)

!     Upper part.
temp = m(10, 9) * m(9, 55)
m(10, 55) = ( m(10, 55) - temp ) / m(10, 10)

!     Upper part.
m(11, 55) = m(11, 55) / m(11, 11)

!     Upper part.
temp = m(12, 11) * m(11, 55)
m(12, 55) = ( m(12, 55) - temp ) / m(12, 12)

!     Upper part.
m(13, 62) = m(13, 62) / m(13, 13)
!     Upper part.
m(13, 65) = m(13, 65) / m(13, 13)

!     Upper part.
m(14, 53) = m(14, 53) / m(14, 14)
!     Upper part.
m(14, 55) = m(14, 55) / m(14, 14)

!     Upper part.
temp = m(15, 7) * m(7, 55)
temp = temp + m(15, 8) * m(8, 55)
m(15, 55) = ( m(15, 55) - temp ) / m(15, 15)
!     Upper part.
m(15, 64) = m(15, 64) / m(15, 15)

!     Upper part.
m(16, 56) = m(16, 56) / m(16, 16)
!     Upper part.
m(16, 65) = m(16, 65) / m(16, 16)

!     Upper part.
m(17, 55) = m(17, 55) / m(17, 17)
!     Upper part.
m(17, 61) = m(17, 61) / m(17, 17)
!     Upper part.
m(17, 65) = m(17, 65) / m(17, 17)

!     Upper part.
m(18, 55) = m(18, 55) / m(18, 18)
!     Upper part.
m(18, 58) = m(18, 58) / m(18, 18)
!     Upper part.
m(18, 65) = m(18, 65) / m(18, 18)

!     Upper part.
m(19, 47) = m(19, 47) / m(19, 19)
!     Upper part.
m(19, 55) = m(19, 55) / m(19, 19)
!     Upper part.
m(19, 65) = m(19, 65) / m(19, 19)

!     Upper part.
m(20, 55) = m(20, 55) / m(20, 20)
!     Upper part.
m(20, 57) = m(20, 57) / m(20, 20)
!     Upper part.
m(20, 61) = m(20, 61) / m(20, 20)

!     Upper part.
m(21, 54) = m(21, 54) / m(21, 21)
!     Upper part.
m(21, 61) = m(21, 61) / m(21, 21)
!     Upper part.
m(21, 64) = m(21, 64) / m(21, 21)

!     Upper part.
m(22, 53) = m(22, 53) / m(22, 22)
!     Upper part.
temp = m(22, 21) * m(21, 54)
m(22, 54) = ( m(22, 54) - temp ) / m(22, 22)
!     Upper part.
m(22, 55) = m(22, 55) / m(22, 22)
!     Upper part.
temp = m(22, 21) * m(21, 61)
m(22, 61) = ( m(22, 61) - temp ) / m(22, 22)
!     Upper part.
temp = m(22, 21) * m(21, 64)
m(22, 64) = ( m(22, 64) - temp ) / m(22, 22)

!     Upper part.
m(23, 55) = m(23, 55) / m(23, 23)
!     Upper part.
m(23, 64) = m(23, 64) / m(23, 23)
!     Upper part.
m(23, 65) = m(23, 65) / m(23, 23)

!     Upper part.
m(24, 55) = m(24, 55) / m(24, 24)
!     Upper part.
m(24, 56) = m(24, 56) / m(24, 24)
!     Upper part.
m(24, 59) = m(24, 59) / m(24, 24)

!     Upper part.
temp = m(25, 24) * m(24, 55)
m(25, 55) = ( m(25, 55) - temp ) / m(25, 25)
!     Upper part.
temp = m(25, 24) * m(24, 56)
m(25, 56) = ( m(25, 56) - temp ) / m(25, 25)
!     Upper part.
temp = m(25, 24) * m(24, 59)
m(25, 59) = ( m(25, 59) - temp ) / m(25, 25)

!     Upper part.
temp = m(26, 24) * m(24, 55)
m(26, 55) = ( m(26, 55) - temp ) / m(26, 26)
!     Upper part.
temp = m(26, 24) * m(24, 56)
m(26, 56) = ( m(26, 56) - temp ) / m(26, 26)
!     Upper part.
temp = m(26, 24) * m(24, 59)
m(26, 59) = ( m(26, 59) - temp ) / m(26, 26)

!     Upper part.
m(27, 55) = m(27, 55) / m(27, 27)
!     Upper part.
m(27, 56) = m(27, 56) / m(27, 27)
!     Upper part.
m(27, 59) = m(27, 59) / m(27, 27)

!     Upper part.
temp = m(28, 27) * m(27, 55)
m(28, 55) = ( m(28, 55) - temp ) / m(28, 28)
!     Upper part.
temp = m(28, 27) * m(27, 56)
m(28, 56) = ( m(28, 56) - temp ) / m(28, 28)
!     Upper part.
temp = m(28, 27) * m(27, 59)
m(28, 59) = ( m(28, 59) - temp ) / m(28, 28)

!     Upper part.
temp = m(29, 27) * m(27, 55)
m(29, 55) = ( m(29, 55) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 27) * m(27, 56)
m(29, 56) = ( m(29, 56) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 27) * m(27, 59)
m(29, 59) = ( m(29, 59) - temp ) / m(29, 29)

!     Upper part.
m(30, 40) = m(30, 40) / m(30, 30)
!     Upper part.
temp = m(30, 7) * m(7, 55)
temp = temp + m(30, 8) * m(8, 55)
m(30, 55) = ( m(30, 55) - temp ) / m(30, 30)
!     Upper part.
m(30, 56) = m(30, 56) / m(30, 30)

!     Upper part.
m(31, 40) = m(31, 40) / m(31, 31)
!     Upper part.
temp = m(31, 7) * m(7, 55)
temp = temp + m(31, 8) * m(8, 55)
m(31, 55) = ( m(31, 55) - temp ) / m(31, 31)
!     Upper part.
m(31, 56) = m(31, 56) / m(31, 31)

!     Upper part.
m(32, 52) = m(32, 52) / m(32, 32)
!     Upper part.
m(32, 55) = m(32, 55) / m(32, 32)
!     Upper part.
m(32, 61) = m(32, 61) / m(32, 32)
!     Upper part.
m(32, 63) = m(32, 63) / m(32, 32)

!     Upper part.
m(33, 55) = m(33, 55) / m(33, 33)
!     Upper part.
m(33, 58) = m(33, 58) / m(33, 33)
!     Upper part.
m(33, 61) = m(33, 61) / m(33, 33)
!     Upper part.
m(33, 62) = m(33, 62) / m(33, 33)

!     Upper part.
m(34, 55) = m(34, 55) / m(34, 34)
!     Upper part.
m(34, 56) = m(34, 56) / m(34, 34)
!     Upper part.
m(34, 57) = m(34, 57) / m(34, 34)
!     Upper part.
m(34, 59) = m(34, 59) / m(34, 34)

!     Upper part.
temp = m(35, 21) * m(21, 54)
m(35, 54) = ( m(35, 54) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 34) * m(34, 55)
m(35, 55) = ( m(35, 55) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 34) * m(34, 56)
m(35, 56) = ( m(35, 56) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 34) * m(34, 57)
m(35, 57) = ( m(35, 57) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 34) * m(34, 59)
m(35, 59) = ( m(35, 59) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 21) * m(21, 61)
m(35, 61) = ( m(35, 61) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 21) * m(21, 64)
m(35, 64) = ( m(35, 64) - temp ) / m(35, 35)

!     Upper part.
m(36, 55) = m(36, 55) / m(36, 36)
!     Upper part.
m(36, 56) = m(36, 56) / m(36, 36)
!     Upper part.
m(36, 57) = m(36, 57) / m(36, 36)
!     Upper part.
m(36, 59) = m(36, 59) / m(36, 36)

!     Upper part.
m(37, 55) = m(37, 55) / m(37, 37)
!     Upper part.
m(37, 56) = m(37, 56) / m(37, 37)
!     Upper part.
m(37, 57) = m(37, 57) / m(37, 37)
!     Upper part.
m(37, 59) = m(37, 59) / m(37, 37)

!     Upper part.
temp = m(38, 36) * m(36, 55)
temp = temp + m(38, 37) * m(37, 55)
m(38, 55) = ( m(38, 55) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 36) * m(36, 56)
temp = temp + m(38, 37) * m(37, 56)
m(38, 56) = ( m(38, 56) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 36) * m(36, 57)
temp = temp + m(38, 37) * m(37, 57)
m(38, 57) = ( m(38, 57) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 36) * m(36, 59)
temp = temp + m(38, 37) * m(37, 59)
m(38, 59) = ( m(38, 59) - temp ) / m(38, 38)

!     Upper part.
m(39, 55) = m(39, 55) / m(39, 39)
!     Upper part.
m(39, 56) = m(39, 56) / m(39, 39)
!     Upper part.
m(39, 57) = m(39, 57) / m(39, 39)
!     Upper part.
m(39, 59) = m(39, 59) / m(39, 39)

!     Upper part.
m(40, 43) = m(40, 43) / m(40, 40)
!     Upper part.
temp = m(40, 7) * m(7, 55)
temp = temp + m(40, 8) * m(8, 55)
temp = temp + m(40, 15) * m(15, 55)
m(40, 55) = ( m(40, 55) - temp ) / m(40, 40)
!     Upper part.
m(40, 56) = m(40, 56) / m(40, 40)
!     Upper part.
m(40, 61) = m(40, 61) / m(40, 40)
!     Upper part.
temp = m(40, 15) * m(15, 64)
m(40, 64) = ( m(40, 64) - temp ) / m(40, 40)

!     Upper part.
temp = m(41, 40) * m(40, 43)
m(41, 43) = ( m(41, 43) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 15) * m(15, 55)
temp = temp + m(41, 40) * m(40, 55)
m(41, 55) = ( m(41, 55) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 40) * m(40, 56)
m(41, 56) = ( m(41, 56) - temp ) / m(41, 41)
!     Upper part.
m(41, 59) = m(41, 59) / m(41, 41)
!     Upper part.
temp = m(41, 40) * m(40, 61)
m(41, 61) = ( m(41, 61) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 15) * m(15, 64)
temp = temp + m(41, 40) * m(40, 64)
m(41, 64) = ( m(41, 64) - temp ) / m(41, 41)

!     Upper part.
temp = m(42, 41) * m(41, 43)
m(42, 43) = ( m(42, 43) - temp ) / m(42, 42)
!     Upper part.
m(42, 45) = m(42, 45) / m(42, 42)
!     Upper part.
temp = m(42, 8) * m(8, 55)
temp = temp + m(42, 41) * m(41, 55)
m(42, 55) = ( m(42, 55) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 41) * m(41, 56)
m(42, 56) = ( m(42, 56) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 41) * m(41, 59)
m(42, 59) = ( m(42, 59) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 41) * m(41, 61)
m(42, 61) = ( m(42, 61) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 41) * m(41, 64)
m(42, 64) = ( m(42, 64) - temp ) / m(42, 42)

!     Lower part.
temp = m(43, 40) * m(40, 43)
m(43, 43) = m(43, 43) - temp
!     Lower part.
temp = m(48, 41) * m(41, 43)
m(48, 43) = m(48, 43) - temp
!     Lower part.
temp = m(50, 40) * m(40, 43)
m(50, 43) = m(50, 43) - temp
!     Lower part.
temp = m(51, 41) * m(41, 43)
temp = temp + m(51, 42) * m(42, 43)
m(51, 43) = m(51, 43) - temp
!     Lower part.
temp = m(54, 41) * m(41, 43)
m(54, 43) = m(54, 43) - temp
!     Lower part.
temp = m(55, 40) * m(40, 43)
temp = temp + m(55, 41) * m(41, 43)
temp = temp + m(55, 42) * m(42, 43)
m(55, 43) = m(55, 43) - temp
!     Lower part.
temp = m(56, 40) * m(40, 43)
m(56, 43) = m(56, 43) - temp
!     Lower part.
temp = m(59, 41) * m(41, 43)
m(59, 43) = m(59, 43) - temp
!     Lower part.
temp = m(61, 40) * m(40, 43)
temp = temp + m(61, 41) * m(41, 43)
temp = temp + m(61, 42) * m(42, 43)
m(61, 43) = m(61, 43) - temp
!     Lower part.
temp = m(62, 41) * m(41, 43)
temp = temp + m(62, 42) * m(42, 43)
m(62, 43) = m(62, 43) - temp
!     Lower part.
temp = m(63, 40) * m(40, 43)
temp = temp + m(63, 41) * m(41, 43)
temp = temp + m(63, 42) * m(42, 43)
m(63, 43) = m(63, 43) - temp
!     Upper part.
temp = m(43, 40) * m(40, 55)
m(43, 55) = ( m(43, 55) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 40) * m(40, 56)
m(43, 56) = ( m(43, 56) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 40) * m(40, 61)
m(43, 61) = ( m(43, 61) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 40) * m(40, 64)
m(43, 64) = ( m(43, 64) - temp ) / m(43, 43)
!     Upper part.
m(43, 65) = m(43, 65) / m(43, 43)

!     Upper part.
m(44, 55) = m(44, 55) / m(44, 44)
!     Upper part.
m(44, 56) = m(44, 56) / m(44, 44)
!     Upper part.
m(44, 57) = m(44, 57) / m(44, 44)
!     Upper part.
m(44, 59) = m(44, 59) / m(44, 44)
!     Upper part.
m(44, 65) = m(44, 65) / m(44, 44)

!     Lower part.
temp = m(51, 42) * m(42, 45)
m(51, 45) = m(51, 45) - temp
!     Lower part.
temp = m(55, 42) * m(42, 45)
m(55, 45) = m(55, 45) - temp
!     Lower part.
temp = m(61, 42) * m(42, 45)
m(61, 45) = m(61, 45) - temp
!     Lower part.
temp = m(62, 42) * m(42, 45)
m(62, 45) = m(62, 45) - temp
!     Lower part.
temp = m(63, 42) * m(42, 45)
m(63, 45) = m(63, 45) - temp
!     Upper part.
temp = m(45, 44) * m(44, 55)
m(45, 55) = ( m(45, 55) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 44) * m(44, 56)
m(45, 56) = ( m(45, 56) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 44) * m(44, 57)
m(45, 57) = ( m(45, 57) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 44) * m(44, 59)
m(45, 59) = ( m(45, 59) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 44) * m(44, 65)
m(45, 65) = ( m(45, 65) - temp ) / m(45, 45)

!     Upper part.
temp = m(46, 19) * m(19, 47)
m(46, 47) = ( m(46, 47) - temp ) / m(46, 46)
!     Upper part.
m(46, 52) = m(46, 52) / m(46, 46)
!     Upper part.
temp = m(46, 15) * m(15, 55)
temp = temp + m(46, 19) * m(19, 55)
temp = temp + m(46, 39) * m(39, 55)
temp = temp + m(46, 43) * m(43, 55)
temp = temp + m(46, 44) * m(44, 55)
temp = temp + m(46, 45) * m(45, 55)
m(46, 55) = ( m(46, 55) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 39) * m(39, 56)
temp = temp + m(46, 43) * m(43, 56)
temp = temp + m(46, 44) * m(44, 56)
temp = temp + m(46, 45) * m(45, 56)
m(46, 56) = ( m(46, 56) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 39) * m(39, 57)
temp = temp + m(46, 44) * m(44, 57)
temp = temp + m(46, 45) * m(45, 57)
m(46, 57) = ( m(46, 57) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 39) * m(39, 59)
temp = temp + m(46, 44) * m(44, 59)
temp = temp + m(46, 45) * m(45, 59)
m(46, 59) = ( m(46, 59) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 43) * m(43, 61)
m(46, 61) = ( m(46, 61) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 15) * m(15, 64)
temp = temp + m(46, 43) * m(43, 64)
m(46, 64) = ( m(46, 64) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 19) * m(19, 65)
temp = temp + m(46, 43) * m(43, 65)
temp = temp + m(46, 44) * m(44, 65)
temp = temp + m(46, 45) * m(45, 65)
m(46, 65) = ( m(46, 65) - temp ) / m(46, 46)

!     Lower part.
temp = m(47, 19) * m(19, 47)
temp = temp + m(47, 46) * m(46, 47)
m(47, 47) = m(47, 47) - temp
!     Lower part.
temp = m(48, 19) * m(19, 47)
temp = temp + m(48, 46) * m(46, 47)
m(48, 47) = m(48, 47) - temp
!     Lower part.
temp = m(50, 46) * m(46, 47)
m(50, 47) = m(50, 47) - temp
!     Lower part.
temp = m(52, 19) * m(19, 47)
m(52, 47) = m(52, 47) - temp
!     Lower part.
temp = m(54, 46) * m(46, 47)
m(54, 47) = m(54, 47) - temp
!     Lower part.
temp = m(55, 46) * m(46, 47)
m(55, 47) = m(55, 47) - temp
!     Lower part.
temp = m(60, 19) * m(19, 47)
temp = temp + m(60, 46) * m(46, 47)
m(60, 47) = m(60, 47) - temp
!     Lower part.
temp = m(61, 19) * m(19, 47)
temp = temp + m(61, 46) * m(46, 47)
m(61, 47) = m(61, 47) - temp
!     Lower part.
temp = m(63, 19) * m(19, 47)
m(63, 47) = m(63, 47) - temp
!     Lower part.
temp = m(65, 19) * m(19, 47)
temp = temp + m(65, 46) * m(46, 47)
m(65, 47) = m(65, 47) - temp
!     Upper part.
temp = m(47, 46) * m(46, 52)
m(47, 52) = ( m(47, 52) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 8) * m(8, 55)
temp = temp + m(47, 19) * m(19, 55)
temp = temp + m(47, 36) * m(36, 55)
temp = temp + m(47, 37) * m(37, 55)
temp = temp + m(47, 39) * m(39, 55)
temp = temp + m(47, 44) * m(44, 55)
temp = temp + m(47, 45) * m(45, 55)
temp = temp + m(47, 46) * m(46, 55)
m(47, 55) = ( m(47, 55) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 36) * m(36, 56)
temp = temp + m(47, 37) * m(37, 56)
temp = temp + m(47, 39) * m(39, 56)
temp = temp + m(47, 44) * m(44, 56)
temp = temp + m(47, 45) * m(45, 56)
temp = temp + m(47, 46) * m(46, 56)
m(47, 56) = ( m(47, 56) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 36) * m(36, 57)
temp = temp + m(47, 37) * m(37, 57)
temp = temp + m(47, 39) * m(39, 57)
temp = temp + m(47, 44) * m(44, 57)
temp = temp + m(47, 45) * m(45, 57)
temp = temp + m(47, 46) * m(46, 57)
m(47, 57) = ( m(47, 57) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 36) * m(36, 59)
temp = temp + m(47, 37) * m(37, 59)
temp = temp + m(47, 39) * m(39, 59)
temp = temp + m(47, 44) * m(44, 59)
temp = temp + m(47, 45) * m(45, 59)
temp = temp + m(47, 46) * m(46, 59)
m(47, 59) = ( m(47, 59) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 46) * m(46, 61)
m(47, 61) = ( m(47, 61) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 46) * m(46, 64)
m(47, 64) = ( m(47, 64) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 19) * m(19, 65)
temp = temp + m(47, 44) * m(44, 65)
temp = temp + m(47, 45) * m(45, 65)
temp = temp + m(47, 46) * m(46, 65)
m(47, 65) = ( m(47, 65) - temp ) / m(47, 47)

!     Upper part.
temp = m(48, 32) * m(32, 52)
temp = temp + m(48, 46) * m(46, 52)
temp = temp + m(48, 47) * m(47, 52)
m(48, 52) = ( m(48, 52) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 4) * m(4, 55)
temp = temp + m(48, 19) * m(19, 55)
temp = temp + m(48, 32) * m(32, 55)
temp = temp + m(48, 34) * m(34, 55)
temp = temp + m(48, 36) * m(36, 55)
temp = temp + m(48, 37) * m(37, 55)
temp = temp + m(48, 39) * m(39, 55)
temp = temp + m(48, 41) * m(41, 55)
temp = temp + m(48, 43) * m(43, 55)
temp = temp + m(48, 44) * m(44, 55)
temp = temp + m(48, 45) * m(45, 55)
temp = temp + m(48, 46) * m(46, 55)
temp = temp + m(48, 47) * m(47, 55)
m(48, 55) = ( m(48, 55) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 34) * m(34, 56)
temp = temp + m(48, 36) * m(36, 56)
temp = temp + m(48, 37) * m(37, 56)
temp = temp + m(48, 39) * m(39, 56)
temp = temp + m(48, 41) * m(41, 56)
temp = temp + m(48, 43) * m(43, 56)
temp = temp + m(48, 44) * m(44, 56)
temp = temp + m(48, 45) * m(45, 56)
temp = temp + m(48, 46) * m(46, 56)
temp = temp + m(48, 47) * m(47, 56)
m(48, 56) = ( m(48, 56) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 34) * m(34, 57)
temp = temp + m(48, 36) * m(36, 57)
temp = temp + m(48, 37) * m(37, 57)
temp = temp + m(48, 39) * m(39, 57)
temp = temp + m(48, 44) * m(44, 57)
temp = temp + m(48, 45) * m(45, 57)
temp = temp + m(48, 46) * m(46, 57)
temp = temp + m(48, 47) * m(47, 57)
m(48, 57) = ( m(48, 57) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 34) * m(34, 59)
temp = temp + m(48, 36) * m(36, 59)
temp = temp + m(48, 37) * m(37, 59)
temp = temp + m(48, 39) * m(39, 59)
temp = temp + m(48, 41) * m(41, 59)
temp = temp + m(48, 44) * m(44, 59)
temp = temp + m(48, 45) * m(45, 59)
temp = temp + m(48, 46) * m(46, 59)
temp = temp + m(48, 47) * m(47, 59)
m(48, 59) = ( m(48, 59) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 32) * m(32, 61)
temp = temp + m(48, 41) * m(41, 61)
temp = temp + m(48, 43) * m(43, 61)
temp = temp + m(48, 46) * m(46, 61)
temp = temp + m(48, 47) * m(47, 61)
m(48, 61) = ( m(48, 61) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 32) * m(32, 63)
m(48, 63) = ( m(48, 63) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 41) * m(41, 64)
temp = temp + m(48, 43) * m(43, 64)
temp = temp + m(48, 46) * m(46, 64)
temp = temp + m(48, 47) * m(47, 64)
m(48, 64) = ( m(48, 64) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 19) * m(19, 65)
temp = temp + m(48, 43) * m(43, 65)
temp = temp + m(48, 44) * m(44, 65)
temp = temp + m(48, 45) * m(45, 65)
temp = temp + m(48, 46) * m(46, 65)
temp = temp + m(48, 47) * m(47, 65)
m(48, 65) = ( m(48, 65) - temp ) / m(48, 48)

!     Upper part.
m(49, 53) = m(49, 53) / m(49, 49)
!     Upper part.
m(49, 55) = m(49, 55) / m(49, 49)
!     Upper part.
m(49, 58) = m(49, 58) / m(49, 49)
!     Upper part.
m(49, 61) = m(49, 61) / m(49, 49)
!     Upper part.
m(49, 62) = m(49, 62) / m(49, 49)
!     Upper part.
m(49, 63) = m(49, 63) / m(49, 49)

!     Upper part.
temp = m(50, 46) * m(46, 52)
temp = temp + m(50, 47) * m(47, 52)
temp = temp + m(50, 48) * m(48, 52)
m(50, 52) = ( m(50, 52) - temp ) / m(50, 50)
!     Upper part.
m(50, 54) = m(50, 54) / m(50, 50)
!     Upper part.
temp = m(50, 40) * m(40, 55)
temp = temp + m(50, 43) * m(43, 55)
temp = temp + m(50, 45) * m(45, 55)
temp = temp + m(50, 46) * m(46, 55)
temp = temp + m(50, 47) * m(47, 55)
temp = temp + m(50, 48) * m(48, 55)
m(50, 55) = ( m(50, 55) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 16) * m(16, 56)
temp = temp + m(50, 40) * m(40, 56)
temp = temp + m(50, 43) * m(43, 56)
temp = temp + m(50, 45) * m(45, 56)
temp = temp + m(50, 46) * m(46, 56)
temp = temp + m(50, 47) * m(47, 56)
temp = temp + m(50, 48) * m(48, 56)
m(50, 56) = ( m(50, 56) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 45) * m(45, 57)
temp = temp + m(50, 46) * m(46, 57)
temp = temp + m(50, 47) * m(47, 57)
temp = temp + m(50, 48) * m(48, 57)
m(50, 57) = ( m(50, 57) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 45) * m(45, 59)
temp = temp + m(50, 46) * m(46, 59)
temp = temp + m(50, 47) * m(47, 59)
temp = temp + m(50, 48) * m(48, 59)
m(50, 59) = ( m(50, 59) - temp ) / m(50, 50)
!     Upper part.
m(50, 60) = m(50, 60) / m(50, 50)
!     Upper part.
temp = m(50, 40) * m(40, 61)
temp = temp + m(50, 43) * m(43, 61)
temp = temp + m(50, 46) * m(46, 61)
temp = temp + m(50, 47) * m(47, 61)
temp = temp + m(50, 48) * m(48, 61)
m(50, 61) = ( m(50, 61) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 48) * m(48, 63)
m(50, 63) = ( m(50, 63) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 40) * m(40, 64)
temp = temp + m(50, 43) * m(43, 64)
temp = temp + m(50, 46) * m(46, 64)
temp = temp + m(50, 47) * m(47, 64)
temp = temp + m(50, 48) * m(48, 64)
m(50, 64) = ( m(50, 64) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 16) * m(16, 65)
temp = temp + m(50, 43) * m(43, 65)
temp = temp + m(50, 45) * m(45, 65)
temp = temp + m(50, 46) * m(46, 65)
temp = temp + m(50, 47) * m(47, 65)
temp = temp + m(50, 48) * m(48, 65)
m(50, 65) = ( m(50, 65) - temp ) / m(50, 50)

!     Upper part.
temp = m(51, 48) * m(48, 52)
m(51, 52) = ( m(51, 52) - temp ) / m(51, 51)
!     Upper part.
m(51, 54) = m(51, 54) / m(51, 51)
!     Upper part.
temp = m(51, 34) * m(34, 55)
temp = temp + m(51, 36) * m(36, 55)
temp = temp + m(51, 37) * m(37, 55)
temp = temp + m(51, 39) * m(39, 55)
temp = temp + m(51, 41) * m(41, 55)
temp = temp + m(51, 42) * m(42, 55)
temp = temp + m(51, 43) * m(43, 55)
temp = temp + m(51, 44) * m(44, 55)
temp = temp + m(51, 45) * m(45, 55)
temp = temp + m(51, 48) * m(48, 55)
m(51, 55) = ( m(51, 55) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 34) * m(34, 56)
temp = temp + m(51, 36) * m(36, 56)
temp = temp + m(51, 37) * m(37, 56)
temp = temp + m(51, 39) * m(39, 56)
temp = temp + m(51, 41) * m(41, 56)
temp = temp + m(51, 42) * m(42, 56)
temp = temp + m(51, 43) * m(43, 56)
temp = temp + m(51, 44) * m(44, 56)
temp = temp + m(51, 45) * m(45, 56)
temp = temp + m(51, 48) * m(48, 56)
m(51, 56) = ( m(51, 56) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 34) * m(34, 57)
temp = temp + m(51, 36) * m(36, 57)
temp = temp + m(51, 37) * m(37, 57)
temp = temp + m(51, 39) * m(39, 57)
temp = temp + m(51, 44) * m(44, 57)
temp = temp + m(51, 45) * m(45, 57)
temp = temp + m(51, 48) * m(48, 57)
m(51, 57) = ( m(51, 57) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 34) * m(34, 59)
temp = temp + m(51, 36) * m(36, 59)
temp = temp + m(51, 37) * m(37, 59)
temp = temp + m(51, 39) * m(39, 59)
temp = temp + m(51, 41) * m(41, 59)
temp = temp + m(51, 42) * m(42, 59)
temp = temp + m(51, 44) * m(44, 59)
temp = temp + m(51, 45) * m(45, 59)
temp = temp + m(51, 48) * m(48, 59)
m(51, 59) = ( m(51, 59) - temp ) / m(51, 51)
!     Upper part.
m(51, 60) = m(51, 60) / m(51, 51)
!     Upper part.
temp = m(51, 41) * m(41, 61)
temp = temp + m(51, 42) * m(42, 61)
temp = temp + m(51, 43) * m(43, 61)
temp = temp + m(51, 48) * m(48, 61)
m(51, 61) = ( m(51, 61) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 48) * m(48, 63)
m(51, 63) = ( m(51, 63) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 41) * m(41, 64)
temp = temp + m(51, 42) * m(42, 64)
temp = temp + m(51, 43) * m(43, 64)
temp = temp + m(51, 48) * m(48, 64)
m(51, 64) = ( m(51, 64) - temp ) / m(51, 51)
!     Upper part.
temp = m(51, 43) * m(43, 65)
temp = temp + m(51, 44) * m(44, 65)
temp = temp + m(51, 45) * m(45, 65)
temp = temp + m(51, 48) * m(48, 65)
m(51, 65) = ( m(51, 65) - temp ) / m(51, 51)

!     Lower part.
temp = m(52, 47) * m(47, 52)
m(52, 52) = m(52, 52) - temp
!     Lower part.
temp = m(53, 48) * m(48, 52)
m(53, 52) = m(53, 52) - temp
!     Lower part.
temp = m(54, 46) * m(46, 52)
temp = temp + m(54, 47) * m(47, 52)
m(54, 52) = m(54, 52) - temp
!     Lower part.
temp = m(55, 32) * m(32, 52)
temp = temp + m(55, 46) * m(46, 52)
temp = temp + m(55, 47) * m(47, 52)
temp = temp + m(55, 48) * m(48, 52)
temp = temp + m(55, 50) * m(50, 52)
temp = temp + m(55, 51) * m(51, 52)
m(55, 52) = m(55, 52) - temp
!     Lower part.
temp = m(56, 48) * m(48, 52)
temp = temp + m(56, 50) * m(50, 52)
m(56, 52) = m(56, 52) - temp
!     Lower part.
temp = m(57, 48) * m(48, 52)
m(57, 52) = m(57, 52) - temp
!     Lower part.
temp = m(58, 48) * m(48, 52)
m(58, 52) = m(58, 52) - temp
!     Lower part.
temp = m(60, 32) * m(32, 52)
temp = temp + m(60, 46) * m(46, 52)
temp = temp + m(60, 47) * m(47, 52)
m(60, 52) = m(60, 52) - temp
!     Lower part.
temp = m(61, 32) * m(32, 52)
temp = temp + m(61, 46) * m(46, 52)
temp = temp + m(61, 47) * m(47, 52)
temp = temp + m(61, 48) * m(48, 52)
temp = temp + m(61, 51) * m(51, 52)
m(61, 52) = m(61, 52) - temp
!     Lower part.
temp = m(63, 32) * m(32, 52)
temp = temp + m(63, 47) * m(47, 52)
m(63, 52) = m(63, 52) - temp
!     Lower part.
temp = m(65, 46) * m(46, 52)
temp = temp + m(65, 47) * m(47, 52)
temp = temp + m(65, 50) * m(50, 52)
m(65, 52) = m(65, 52) - temp
!     Upper part.
temp = m(52, 6) * m(6, 55)
temp = temp + m(52, 19) * m(19, 55)
temp = temp + m(52, 36) * m(36, 55)
temp = temp + m(52, 39) * m(39, 55)
temp = temp + m(52, 44) * m(44, 55)
temp = temp + m(52, 47) * m(47, 55)
m(52, 55) = ( m(52, 55) - temp ) / m(52, 52)
!     Upper part.
temp = m(52, 36) * m(36, 56)
temp = temp + m(52, 39) * m(39, 56)
temp = temp + m(52, 44) * m(44, 56)
temp = temp + m(52, 47) * m(47, 56)
m(52, 56) = ( m(52, 56) - temp ) / m(52, 52)
!     Upper part.
temp = m(52, 36) * m(36, 57)
temp = temp + m(52, 39) * m(39, 57)
temp = temp + m(52, 44) * m(44, 57)
temp = temp + m(52, 47) * m(47, 57)
m(52, 57) = ( m(52, 57) - temp ) / m(52, 52)
!     Upper part.
temp = m(52, 36) * m(36, 59)
temp = temp + m(52, 39) * m(39, 59)
temp = temp + m(52, 44) * m(44, 59)
temp = temp + m(52, 47) * m(47, 59)
m(52, 59) = ( m(52, 59) - temp ) / m(52, 52)
!     Upper part.
temp = m(52, 47) * m(47, 61)
m(52, 61) = ( m(52, 61) - temp ) / m(52, 52)
!     Upper part.
m(52, 63) = m(52, 63) / m(52, 52)
!     Upper part.
temp = m(52, 47) * m(47, 64)
m(52, 64) = ( m(52, 64) - temp ) / m(52, 52)
!     Upper part.
temp = m(52, 19) * m(19, 65)
temp = temp + m(52, 44) * m(44, 65)
temp = temp + m(52, 47) * m(47, 65)
m(52, 65) = ( m(52, 65) - temp ) / m(52, 52)

!     Lower part.
temp = m(53, 22) * m(22, 53)
temp = temp + m(53, 49) * m(49, 53)
m(53, 53) = m(53, 53) - temp
!     Lower part.
temp = m(54, 14) * m(14, 53)
temp = temp + m(54, 22) * m(22, 53)
m(54, 53) = m(54, 53) - temp
!     Lower part.
temp = m(55, 14) * m(14, 53)
temp = temp + m(55, 22) * m(22, 53)
temp = temp + m(55, 49) * m(49, 53)
m(55, 53) = m(55, 53) - temp
!     Lower part.
temp = m(61, 14) * m(14, 53)
temp = temp + m(61, 22) * m(22, 53)
m(61, 53) = m(61, 53) - temp
!     Lower part.
temp = m(63, 22) * m(22, 53)
m(63, 53) = m(63, 53) - temp
!     Upper part.
temp = m(53, 22) * m(22, 54)
m(53, 54) = ( m(53, 54) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 5) * m(5, 55)
temp = temp + m(53, 22) * m(22, 55)
temp = temp + m(53, 33) * m(33, 55)
temp = temp + m(53, 48) * m(48, 55)
temp = temp + m(53, 49) * m(49, 55)
temp = temp + m(53, 52) * m(52, 55)
m(53, 55) = ( m(53, 55) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 48) * m(48, 56)
temp = temp + m(53, 52) * m(52, 56)
m(53, 56) = ( m(53, 56) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 48) * m(48, 57)
temp = temp + m(53, 52) * m(52, 57)
m(53, 57) = ( m(53, 57) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 33) * m(33, 58)
temp = temp + m(53, 49) * m(49, 58)
m(53, 58) = ( m(53, 58) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 48) * m(48, 59)
temp = temp + m(53, 52) * m(52, 59)
m(53, 59) = ( m(53, 59) - temp ) / m(53, 53)
!     Upper part.
m(53, 60) = m(53, 60) / m(53, 53)
!     Upper part.
temp = m(53, 22) * m(22, 61)
temp = temp + m(53, 33) * m(33, 61)
temp = temp + m(53, 48) * m(48, 61)
temp = temp + m(53, 49) * m(49, 61)
temp = temp + m(53, 52) * m(52, 61)
m(53, 61) = ( m(53, 61) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 33) * m(33, 62)
temp = temp + m(53, 49) * m(49, 62)
m(53, 62) = ( m(53, 62) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 48) * m(48, 63)
temp = temp + m(53, 49) * m(49, 63)
temp = temp + m(53, 52) * m(52, 63)
m(53, 63) = ( m(53, 63) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 22) * m(22, 64)
temp = temp + m(53, 48) * m(48, 64)
temp = temp + m(53, 52) * m(52, 64)
m(53, 64) = ( m(53, 64) - temp ) / m(53, 53)
!     Upper part.
temp = m(53, 48) * m(48, 65)
temp = temp + m(53, 52) * m(52, 65)
m(53, 65) = ( m(53, 65) - temp ) / m(53, 53)

!     Lower part.
temp = m(54, 21) * m(21, 54)
temp = temp + m(54, 22) * m(22, 54)
temp = temp + m(54, 53) * m(53, 54)
m(54, 54) = m(54, 54) - temp
!     Lower part.
temp = m(55, 22) * m(22, 54)
temp = temp + m(55, 35) * m(35, 54)
temp = temp + m(55, 50) * m(50, 54)
temp = temp + m(55, 51) * m(51, 54)
temp = temp + m(55, 53) * m(53, 54)
m(55, 54) = m(55, 54) - temp
!     Lower part.
temp = m(56, 50) * m(50, 54)
m(56, 54) = m(56, 54) - temp
!     Lower part.
temp = m(60, 53) * m(53, 54)
m(60, 54) = m(60, 54) - temp
!     Lower part.
temp = m(61, 21) * m(21, 54)
temp = temp + m(61, 22) * m(22, 54)
temp = temp + m(61, 35) * m(35, 54)
temp = temp + m(61, 51) * m(51, 54)
temp = temp + m(61, 53) * m(53, 54)
m(61, 54) = m(61, 54) - temp
!     Lower part.
temp = m(62, 53) * m(53, 54)
m(62, 54) = m(62, 54) - temp
!     Lower part.
temp = m(63, 22) * m(22, 54)
temp = temp + m(63, 53) * m(53, 54)
m(63, 54) = m(63, 54) - temp
!     Lower part.
temp = m(64, 21) * m(21, 54)
temp = temp + m(64, 53) * m(53, 54)
m(64, 54) = m(64, 54) - temp
!     Lower part.
temp = m(65, 21) * m(21, 54)
temp = temp + m(65, 50) * m(50, 54)
temp = temp + m(65, 53) * m(53, 54)
m(65, 54) = m(65, 54) - temp
!     Upper part.
temp = m(54, 4) * m(4, 55)
temp = temp + m(54, 14) * m(14, 55)
temp = temp + m(54, 22) * m(22, 55)
temp = temp + m(54, 34) * m(34, 55)
temp = temp + m(54, 36) * m(36, 55)
temp = temp + m(54, 37) * m(37, 55)
temp = temp + m(54, 39) * m(39, 55)
temp = temp + m(54, 41) * m(41, 55)
temp = temp + m(54, 43) * m(43, 55)
temp = temp + m(54, 44) * m(44, 55)
temp = temp + m(54, 45) * m(45, 55)
temp = temp + m(54, 46) * m(46, 55)
temp = temp + m(54, 47) * m(47, 55)
temp = temp + m(54, 52) * m(52, 55)
temp = temp + m(54, 53) * m(53, 55)
m(54, 55) = ( m(54, 55) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 34) * m(34, 56)
temp = temp + m(54, 36) * m(36, 56)
temp = temp + m(54, 37) * m(37, 56)
temp = temp + m(54, 39) * m(39, 56)
temp = temp + m(54, 41) * m(41, 56)
temp = temp + m(54, 43) * m(43, 56)
temp = temp + m(54, 44) * m(44, 56)
temp = temp + m(54, 45) * m(45, 56)
temp = temp + m(54, 46) * m(46, 56)
temp = temp + m(54, 47) * m(47, 56)
temp = temp + m(54, 52) * m(52, 56)
temp = temp + m(54, 53) * m(53, 56)
m(54, 56) = ( m(54, 56) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 34) * m(34, 57)
temp = temp + m(54, 36) * m(36, 57)
temp = temp + m(54, 37) * m(37, 57)
temp = temp + m(54, 39) * m(39, 57)
temp = temp + m(54, 44) * m(44, 57)
temp = temp + m(54, 45) * m(45, 57)
temp = temp + m(54, 46) * m(46, 57)
temp = temp + m(54, 47) * m(47, 57)
temp = temp + m(54, 52) * m(52, 57)
temp = temp + m(54, 53) * m(53, 57)
m(54, 57) = ( m(54, 57) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 53) * m(53, 58)
m(54, 58) = ( m(54, 58) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 34) * m(34, 59)
temp = temp + m(54, 36) * m(36, 59)
temp = temp + m(54, 37) * m(37, 59)
temp = temp + m(54, 39) * m(39, 59)
temp = temp + m(54, 41) * m(41, 59)
temp = temp + m(54, 44) * m(44, 59)
temp = temp + m(54, 45) * m(45, 59)
temp = temp + m(54, 46) * m(46, 59)
temp = temp + m(54, 47) * m(47, 59)
temp = temp + m(54, 52) * m(52, 59)
temp = temp + m(54, 53) * m(53, 59)
m(54, 59) = ( m(54, 59) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 53) * m(53, 60)
m(54, 60) = ( m(54, 60) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 21) * m(21, 61)
temp = temp + m(54, 22) * m(22, 61)
temp = temp + m(54, 41) * m(41, 61)
temp = temp + m(54, 43) * m(43, 61)
temp = temp + m(54, 46) * m(46, 61)
temp = temp + m(54, 47) * m(47, 61)
temp = temp + m(54, 52) * m(52, 61)
temp = temp + m(54, 53) * m(53, 61)
m(54, 61) = ( m(54, 61) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 53) * m(53, 62)
m(54, 62) = ( m(54, 62) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 52) * m(52, 63)
temp = temp + m(54, 53) * m(53, 63)
m(54, 63) = ( m(54, 63) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 21) * m(21, 64)
temp = temp + m(54, 22) * m(22, 64)
temp = temp + m(54, 41) * m(41, 64)
temp = temp + m(54, 43) * m(43, 64)
temp = temp + m(54, 46) * m(46, 64)
temp = temp + m(54, 47) * m(47, 64)
temp = temp + m(54, 52) * m(52, 64)
temp = temp + m(54, 53) * m(53, 64)
m(54, 64) = ( m(54, 64) - temp ) / m(54, 54)
!     Upper part.
temp = m(54, 43) * m(43, 65)
temp = temp + m(54, 44) * m(44, 65)
temp = temp + m(54, 45) * m(45, 65)
temp = temp + m(54, 46) * m(46, 65)
temp = temp + m(54, 47) * m(47, 65)
temp = temp + m(54, 52) * m(52, 65)
temp = temp + m(54, 53) * m(53, 65)
m(54, 65) = ( m(54, 65) - temp ) / m(54, 54)

!     Lower part.
temp = m(55, 4) * m(4, 55)
temp = temp + m(55, 5) * m(5, 55)
temp = temp + m(55, 6) * m(6, 55)
temp = temp + m(55, 7) * m(7, 55)
temp = temp + m(55, 8) * m(8, 55)
temp = temp + m(55, 9) * m(9, 55)
temp = temp + m(55, 14) * m(14, 55)
temp = temp + m(55, 17) * m(17, 55)
temp = temp + m(55, 18) * m(18, 55)
temp = temp + m(55, 20) * m(20, 55)
temp = temp + m(55, 22) * m(22, 55)
temp = temp + m(55, 23) * m(23, 55)
temp = temp + m(55, 32) * m(32, 55)
temp = temp + m(55, 33) * m(33, 55)
temp = temp + m(55, 34) * m(34, 55)
temp = temp + m(55, 35) * m(35, 55)
temp = temp + m(55, 36) * m(36, 55)
temp = temp + m(55, 37) * m(37, 55)
temp = temp + m(55, 39) * m(39, 55)
temp = temp + m(55, 40) * m(40, 55)
temp = temp + m(55, 41) * m(41, 55)
temp = temp + m(55, 42) * m(42, 55)
temp = temp + m(55, 43) * m(43, 55)
temp = temp + m(55, 44) * m(44, 55)
temp = temp + m(55, 45) * m(45, 55)
temp = temp + m(55, 46) * m(46, 55)
temp = temp + m(55, 47) * m(47, 55)
temp = temp + m(55, 48) * m(48, 55)
temp = temp + m(55, 49) * m(49, 55)
temp = temp + m(55, 50) * m(50, 55)
temp = temp + m(55, 51) * m(51, 55)
temp = temp + m(55, 52) * m(52, 55)
temp = temp + m(55, 53) * m(53, 55)
temp = temp + m(55, 54) * m(54, 55)
m(55, 55) = m(55, 55) - temp
!     Lower part.
temp = m(56, 17) * m(17, 55)
temp = temp + m(56, 34) * m(34, 55)
temp = temp + m(56, 36) * m(36, 55)
temp = temp + m(56, 37) * m(37, 55)
temp = temp + m(56, 39) * m(39, 55)
temp = temp + m(56, 40) * m(40, 55)
temp = temp + m(56, 43) * m(43, 55)
temp = temp + m(56, 44) * m(44, 55)
temp = temp + m(56, 45) * m(45, 55)
temp = temp + m(56, 48) * m(48, 55)
temp = temp + m(56, 50) * m(50, 55)
temp = temp + m(56, 52) * m(52, 55)
temp = temp + m(56, 54) * m(54, 55)
m(56, 55) = m(56, 55) - temp
!     Lower part.
temp = m(57, 20) * m(20, 55)
temp = temp + m(57, 34) * m(34, 55)
temp = temp + m(57, 36) * m(36, 55)
temp = temp + m(57, 37) * m(37, 55)
temp = temp + m(57, 39) * m(39, 55)
temp = temp + m(57, 44) * m(44, 55)
temp = temp + m(57, 48) * m(48, 55)
temp = temp + m(57, 52) * m(52, 55)
temp = temp + m(57, 54) * m(54, 55)
m(57, 55) = m(57, 55) - temp
!     Lower part.
temp = m(58, 18) * m(18, 55)
temp = temp + m(58, 39) * m(39, 55)
temp = temp + m(58, 44) * m(44, 55)
temp = temp + m(58, 45) * m(45, 55)
temp = temp + m(58, 48) * m(48, 55)
temp = temp + m(58, 52) * m(52, 55)
m(58, 55) = m(58, 55) - temp
!     Lower part.
temp = m(59, 34) * m(34, 55)
temp = temp + m(59, 36) * m(36, 55)
temp = temp + m(59, 37) * m(37, 55)
temp = temp + m(59, 39) * m(39, 55)
temp = temp + m(59, 41) * m(41, 55)
temp = temp + m(59, 43) * m(43, 55)
temp = temp + m(59, 44) * m(44, 55)
temp = temp + m(59, 45) * m(45, 55)
m(59, 55) = m(59, 55) - temp
!     Lower part.
temp = m(60, 4) * m(4, 55)
temp = temp + m(60, 6) * m(6, 55)
temp = temp + m(60, 18) * m(18, 55)
temp = temp + m(60, 19) * m(19, 55)
temp = temp + m(60, 32) * m(32, 55)
temp = temp + m(60, 36) * m(36, 55)
temp = temp + m(60, 37) * m(37, 55)
temp = temp + m(60, 45) * m(45, 55)
temp = temp + m(60, 46) * m(46, 55)
temp = temp + m(60, 47) * m(47, 55)
temp = temp + m(60, 52) * m(52, 55)
temp = temp + m(60, 53) * m(53, 55)
temp = temp + m(60, 54) * m(54, 55)
m(60, 55) = m(60, 55) - temp
!     Lower part.
temp = m(61, 4) * m(4, 55)
temp = temp + m(61, 6) * m(6, 55)
temp = temp + m(61, 7) * m(7, 55)
temp = temp + m(61, 8) * m(8, 55)
temp = temp + m(61, 9) * m(9, 55)
temp = temp + m(61, 14) * m(14, 55)
temp = temp + m(61, 15) * m(15, 55)
temp = temp + m(61, 17) * m(17, 55)
temp = temp + m(61, 19) * m(19, 55)
temp = temp + m(61, 20) * m(20, 55)
temp = temp + m(61, 22) * m(22, 55)
temp = temp + m(61, 32) * m(32, 55)
temp = temp + m(61, 34) * m(34, 55)
temp = temp + m(61, 35) * m(35, 55)
temp = temp + m(61, 36) * m(36, 55)
temp = temp + m(61, 37) * m(37, 55)
temp = temp + m(61, 39) * m(39, 55)
temp = temp + m(61, 40) * m(40, 55)
temp = temp + m(61, 41) * m(41, 55)
temp = temp + m(61, 42) * m(42, 55)
temp = temp + m(61, 43) * m(43, 55)
temp = temp + m(61, 44) * m(44, 55)
temp = temp + m(61, 45) * m(45, 55)
temp = temp + m(61, 46) * m(46, 55)
temp = temp + m(61, 47) * m(47, 55)
temp = temp + m(61, 48) * m(48, 55)
temp = temp + m(61, 51) * m(51, 55)
temp = temp + m(61, 52) * m(52, 55)
temp = temp + m(61, 53) * m(53, 55)
temp = temp + m(61, 54) * m(54, 55)
m(61, 55) = m(61, 55) - temp
!     Lower part.
temp = m(62, 33) * m(33, 55)
temp = temp + m(62, 41) * m(41, 55)
temp = temp + m(62, 42) * m(42, 55)
temp = temp + m(62, 43) * m(43, 55)
temp = temp + m(62, 45) * m(45, 55)
temp = temp + m(62, 53) * m(53, 55)
temp = temp + m(62, 54) * m(54, 55)
m(62, 55) = m(62, 55) - temp
!     Lower part.
temp = m(63, 4) * m(4, 55)
temp = temp + m(63, 6) * m(6, 55)
temp = temp + m(63, 7) * m(7, 55)
temp = temp + m(63, 8) * m(8, 55)
temp = temp + m(63, 19) * m(19, 55)
temp = temp + m(63, 22) * m(22, 55)
temp = temp + m(63, 32) * m(32, 55)
temp = temp + m(63, 34) * m(34, 55)
temp = temp + m(63, 36) * m(36, 55)
temp = temp + m(63, 37) * m(37, 55)
temp = temp + m(63, 39) * m(39, 55)
temp = temp + m(63, 40) * m(40, 55)
temp = temp + m(63, 41) * m(41, 55)
temp = temp + m(63, 42) * m(42, 55)
temp = temp + m(63, 43) * m(43, 55)
temp = temp + m(63, 44) * m(44, 55)
temp = temp + m(63, 45) * m(45, 55)
temp = temp + m(63, 47) * m(47, 55)
temp = temp + m(63, 52) * m(52, 55)
temp = temp + m(63, 53) * m(53, 55)
temp = temp + m(63, 54) * m(54, 55)
m(63, 55) = m(63, 55) - temp
!     Lower part.
temp = m(64, 15) * m(15, 55)
temp = temp + m(64, 23) * m(23, 55)
temp = temp + m(64, 44) * m(44, 55)
temp = temp + m(64, 52) * m(52, 55)
temp = temp + m(64, 53) * m(53, 55)
temp = temp + m(64, 54) * m(54, 55)
m(64, 55) = m(64, 55) - temp
!     Lower part.
temp = m(65, 15) * m(15, 55)
temp = temp + m(65, 17) * m(17, 55)
temp = temp + m(65, 18) * m(18, 55)
temp = temp + m(65, 19) * m(19, 55)
temp = temp + m(65, 23) * m(23, 55)
temp = temp + m(65, 34) * m(34, 55)
temp = temp + m(65, 36) * m(36, 55)
temp = temp + m(65, 37) * m(37, 55)
temp = temp + m(65, 39) * m(39, 55)
temp = temp + m(65, 43) * m(43, 55)
temp = temp + m(65, 44) * m(44, 55)
temp = temp + m(65, 46) * m(46, 55)
temp = temp + m(65, 47) * m(47, 55)
temp = temp + m(65, 50) * m(50, 55)
temp = temp + m(65, 52) * m(52, 55)
temp = temp + m(65, 53) * m(53, 55)
temp = temp + m(65, 54) * m(54, 55)
m(65, 55) = m(65, 55) - temp
!     Upper part.
temp = m(55, 34) * m(34, 56)
temp = temp + m(55, 35) * m(35, 56)
temp = temp + m(55, 36) * m(36, 56)
temp = temp + m(55, 37) * m(37, 56)
temp = temp + m(55, 39) * m(39, 56)
temp = temp + m(55, 40) * m(40, 56)
temp = temp + m(55, 41) * m(41, 56)
temp = temp + m(55, 42) * m(42, 56)
temp = temp + m(55, 43) * m(43, 56)
temp = temp + m(55, 44) * m(44, 56)
temp = temp + m(55, 45) * m(45, 56)
temp = temp + m(55, 46) * m(46, 56)
temp = temp + m(55, 47) * m(47, 56)
temp = temp + m(55, 48) * m(48, 56)
temp = temp + m(55, 50) * m(50, 56)
temp = temp + m(55, 51) * m(51, 56)
temp = temp + m(55, 52) * m(52, 56)
temp = temp + m(55, 53) * m(53, 56)
temp = temp + m(55, 54) * m(54, 56)
m(55, 56) = ( m(55, 56) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 20) * m(20, 57)
temp = temp + m(55, 34) * m(34, 57)
temp = temp + m(55, 35) * m(35, 57)
temp = temp + m(55, 36) * m(36, 57)
temp = temp + m(55, 37) * m(37, 57)
temp = temp + m(55, 39) * m(39, 57)
temp = temp + m(55, 44) * m(44, 57)
temp = temp + m(55, 45) * m(45, 57)
temp = temp + m(55, 46) * m(46, 57)
temp = temp + m(55, 47) * m(47, 57)
temp = temp + m(55, 48) * m(48, 57)
temp = temp + m(55, 50) * m(50, 57)
temp = temp + m(55, 51) * m(51, 57)
temp = temp + m(55, 52) * m(52, 57)
temp = temp + m(55, 53) * m(53, 57)
temp = temp + m(55, 54) * m(54, 57)
m(55, 57) = ( m(55, 57) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 18) * m(18, 58)
temp = temp + m(55, 33) * m(33, 58)
temp = temp + m(55, 49) * m(49, 58)
temp = temp + m(55, 53) * m(53, 58)
temp = temp + m(55, 54) * m(54, 58)
m(55, 58) = ( m(55, 58) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 3) * m(3, 59)
temp = temp + m(55, 34) * m(34, 59)
temp = temp + m(55, 35) * m(35, 59)
temp = temp + m(55, 36) * m(36, 59)
temp = temp + m(55, 37) * m(37, 59)
temp = temp + m(55, 39) * m(39, 59)
temp = temp + m(55, 41) * m(41, 59)
temp = temp + m(55, 42) * m(42, 59)
temp = temp + m(55, 44) * m(44, 59)
temp = temp + m(55, 45) * m(45, 59)
temp = temp + m(55, 46) * m(46, 59)
temp = temp + m(55, 47) * m(47, 59)
temp = temp + m(55, 48) * m(48, 59)
temp = temp + m(55, 50) * m(50, 59)
temp = temp + m(55, 51) * m(51, 59)
temp = temp + m(55, 52) * m(52, 59)
temp = temp + m(55, 53) * m(53, 59)
temp = temp + m(55, 54) * m(54, 59)
m(55, 59) = ( m(55, 59) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 50) * m(50, 60)
temp = temp + m(55, 51) * m(51, 60)
temp = temp + m(55, 53) * m(53, 60)
temp = temp + m(55, 54) * m(54, 60)
m(55, 60) = ( m(55, 60) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 17) * m(17, 61)
temp = temp + m(55, 20) * m(20, 61)
temp = temp + m(55, 22) * m(22, 61)
temp = temp + m(55, 32) * m(32, 61)
temp = temp + m(55, 33) * m(33, 61)
temp = temp + m(55, 35) * m(35, 61)
temp = temp + m(55, 40) * m(40, 61)
temp = temp + m(55, 41) * m(41, 61)
temp = temp + m(55, 42) * m(42, 61)
temp = temp + m(55, 43) * m(43, 61)
temp = temp + m(55, 46) * m(46, 61)
temp = temp + m(55, 47) * m(47, 61)
temp = temp + m(55, 48) * m(48, 61)
temp = temp + m(55, 49) * m(49, 61)
temp = temp + m(55, 50) * m(50, 61)
temp = temp + m(55, 51) * m(51, 61)
temp = temp + m(55, 52) * m(52, 61)
temp = temp + m(55, 53) * m(53, 61)
temp = temp + m(55, 54) * m(54, 61)
m(55, 61) = ( m(55, 61) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 33) * m(33, 62)
temp = temp + m(55, 49) * m(49, 62)
temp = temp + m(55, 53) * m(53, 62)
temp = temp + m(55, 54) * m(54, 62)
m(55, 62) = ( m(55, 62) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 32) * m(32, 63)
temp = temp + m(55, 48) * m(48, 63)
temp = temp + m(55, 49) * m(49, 63)
temp = temp + m(55, 50) * m(50, 63)
temp = temp + m(55, 51) * m(51, 63)
temp = temp + m(55, 52) * m(52, 63)
temp = temp + m(55, 53) * m(53, 63)
temp = temp + m(55, 54) * m(54, 63)
m(55, 63) = ( m(55, 63) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 22) * m(22, 64)
temp = temp + m(55, 23) * m(23, 64)
temp = temp + m(55, 35) * m(35, 64)
temp = temp + m(55, 40) * m(40, 64)
temp = temp + m(55, 41) * m(41, 64)
temp = temp + m(55, 42) * m(42, 64)
temp = temp + m(55, 43) * m(43, 64)
temp = temp + m(55, 46) * m(46, 64)
temp = temp + m(55, 47) * m(47, 64)
temp = temp + m(55, 48) * m(48, 64)
temp = temp + m(55, 50) * m(50, 64)
temp = temp + m(55, 51) * m(51, 64)
temp = temp + m(55, 52) * m(52, 64)
temp = temp + m(55, 53) * m(53, 64)
temp = temp + m(55, 54) * m(54, 64)
m(55, 64) = ( m(55, 64) - temp ) / m(55, 55)
!     Upper part.
temp = m(55, 17) * m(17, 65)
temp = temp + m(55, 18) * m(18, 65)
temp = temp + m(55, 23) * m(23, 65)
temp = temp + m(55, 43) * m(43, 65)
temp = temp + m(55, 44) * m(44, 65)
temp = temp + m(55, 45) * m(45, 65)
temp = temp + m(55, 46) * m(46, 65)
temp = temp + m(55, 47) * m(47, 65)
temp = temp + m(55, 48) * m(48, 65)
temp = temp + m(55, 50) * m(50, 65)
temp = temp + m(55, 51) * m(51, 65)
temp = temp + m(55, 52) * m(52, 65)
temp = temp + m(55, 53) * m(53, 65)
temp = temp + m(55, 54) * m(54, 65)
m(55, 65) = ( m(55, 65) - temp ) / m(55, 55)

!     Lower part.
temp = m(56, 16) * m(16, 56)
temp = temp + m(56, 34) * m(34, 56)
temp = temp + m(56, 36) * m(36, 56)
temp = temp + m(56, 37) * m(37, 56)
temp = temp + m(56, 39) * m(39, 56)
temp = temp + m(56, 40) * m(40, 56)
temp = temp + m(56, 43) * m(43, 56)
temp = temp + m(56, 44) * m(44, 56)
temp = temp + m(56, 45) * m(45, 56)
temp = temp + m(56, 48) * m(48, 56)
temp = temp + m(56, 50) * m(50, 56)
temp = temp + m(56, 52) * m(52, 56)
temp = temp + m(56, 54) * m(54, 56)
temp = temp + m(56, 55) * m(55, 56)
m(56, 56) = m(56, 56) - temp
!     Lower part.
temp = m(57, 34) * m(34, 56)
temp = temp + m(57, 36) * m(36, 56)
temp = temp + m(57, 37) * m(37, 56)
temp = temp + m(57, 39) * m(39, 56)
temp = temp + m(57, 44) * m(44, 56)
temp = temp + m(57, 48) * m(48, 56)
temp = temp + m(57, 52) * m(52, 56)
temp = temp + m(57, 54) * m(54, 56)
temp = temp + m(57, 55) * m(55, 56)
m(57, 56) = m(57, 56) - temp
!     Lower part.
temp = m(58, 39) * m(39, 56)
temp = temp + m(58, 44) * m(44, 56)
temp = temp + m(58, 45) * m(45, 56)
temp = temp + m(58, 48) * m(48, 56)
temp = temp + m(58, 52) * m(52, 56)
temp = temp + m(58, 55) * m(55, 56)
m(58, 56) = m(58, 56) - temp
!     Lower part.
temp = m(59, 34) * m(34, 56)
temp = temp + m(59, 36) * m(36, 56)
temp = temp + m(59, 37) * m(37, 56)
temp = temp + m(59, 39) * m(39, 56)
temp = temp + m(59, 41) * m(41, 56)
temp = temp + m(59, 43) * m(43, 56)
temp = temp + m(59, 44) * m(44, 56)
temp = temp + m(59, 45) * m(45, 56)
temp = temp + m(59, 55) * m(55, 56)
m(59, 56) = m(59, 56) - temp
!     Lower part.
temp = m(60, 36) * m(36, 56)
temp = temp + m(60, 37) * m(37, 56)
temp = temp + m(60, 45) * m(45, 56)
temp = temp + m(60, 46) * m(46, 56)
temp = temp + m(60, 47) * m(47, 56)
temp = temp + m(60, 52) * m(52, 56)
temp = temp + m(60, 53) * m(53, 56)
temp = temp + m(60, 54) * m(54, 56)
temp = temp + m(60, 55) * m(55, 56)
m(60, 56) = m(60, 56) - temp
!     Lower part.
temp = m(61, 34) * m(34, 56)
temp = temp + m(61, 35) * m(35, 56)
temp = temp + m(61, 36) * m(36, 56)
temp = temp + m(61, 37) * m(37, 56)
temp = temp + m(61, 39) * m(39, 56)
temp = temp + m(61, 40) * m(40, 56)
temp = temp + m(61, 41) * m(41, 56)
temp = temp + m(61, 42) * m(42, 56)
temp = temp + m(61, 43) * m(43, 56)
temp = temp + m(61, 44) * m(44, 56)
temp = temp + m(61, 45) * m(45, 56)
temp = temp + m(61, 46) * m(46, 56)
temp = temp + m(61, 47) * m(47, 56)
temp = temp + m(61, 48) * m(48, 56)
temp = temp + m(61, 51) * m(51, 56)
temp = temp + m(61, 52) * m(52, 56)
temp = temp + m(61, 53) * m(53, 56)
temp = temp + m(61, 54) * m(54, 56)
temp = temp + m(61, 55) * m(55, 56)
m(61, 56) = m(61, 56) - temp
!     Lower part.
temp = m(62, 41) * m(41, 56)
temp = temp + m(62, 42) * m(42, 56)
temp = temp + m(62, 43) * m(43, 56)
temp = temp + m(62, 45) * m(45, 56)
temp = temp + m(62, 53) * m(53, 56)
temp = temp + m(62, 54) * m(54, 56)
temp = temp + m(62, 55) * m(55, 56)
m(62, 56) = m(62, 56) - temp
!     Lower part.
temp = m(63, 34) * m(34, 56)
temp = temp + m(63, 36) * m(36, 56)
temp = temp + m(63, 37) * m(37, 56)
temp = temp + m(63, 39) * m(39, 56)
temp = temp + m(63, 40) * m(40, 56)
temp = temp + m(63, 41) * m(41, 56)
temp = temp + m(63, 42) * m(42, 56)
temp = temp + m(63, 43) * m(43, 56)
temp = temp + m(63, 44) * m(44, 56)
temp = temp + m(63, 45) * m(45, 56)
temp = temp + m(63, 47) * m(47, 56)
temp = temp + m(63, 52) * m(52, 56)
temp = temp + m(63, 53) * m(53, 56)
temp = temp + m(63, 54) * m(54, 56)
temp = temp + m(63, 55) * m(55, 56)
m(63, 56) = m(63, 56) - temp
!     Lower part.
temp = m(64, 44) * m(44, 56)
temp = temp + m(64, 52) * m(52, 56)
temp = temp + m(64, 53) * m(53, 56)
temp = temp + m(64, 54) * m(54, 56)
temp = temp + m(64, 55) * m(55, 56)
m(64, 56) = m(64, 56) - temp
!     Lower part.
temp = m(65, 16) * m(16, 56)
temp = temp + m(65, 34) * m(34, 56)
temp = temp + m(65, 36) * m(36, 56)
temp = temp + m(65, 37) * m(37, 56)
temp = temp + m(65, 39) * m(39, 56)
temp = temp + m(65, 43) * m(43, 56)
temp = temp + m(65, 44) * m(44, 56)
temp = temp + m(65, 46) * m(46, 56)
temp = temp + m(65, 47) * m(47, 56)
temp = temp + m(65, 50) * m(50, 56)
temp = temp + m(65, 52) * m(52, 56)
temp = temp + m(65, 53) * m(53, 56)
temp = temp + m(65, 54) * m(54, 56)
temp = temp + m(65, 55) * m(55, 56)
m(65, 56) = m(65, 56) - temp
!     Upper part.
temp = m(56, 34) * m(34, 57)
temp = temp + m(56, 36) * m(36, 57)
temp = temp + m(56, 37) * m(37, 57)
temp = temp + m(56, 39) * m(39, 57)
temp = temp + m(56, 44) * m(44, 57)
temp = temp + m(56, 45) * m(45, 57)
temp = temp + m(56, 48) * m(48, 57)
temp = temp + m(56, 50) * m(50, 57)
temp = temp + m(56, 52) * m(52, 57)
temp = temp + m(56, 54) * m(54, 57)
temp = temp + m(56, 55) * m(55, 57)
m(56, 57) = ( m(56, 57) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 54) * m(54, 58)
temp = temp + m(56, 55) * m(55, 58)
m(56, 58) = ( m(56, 58) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 34) * m(34, 59)
temp = temp + m(56, 36) * m(36, 59)
temp = temp + m(56, 37) * m(37, 59)
temp = temp + m(56, 39) * m(39, 59)
temp = temp + m(56, 44) * m(44, 59)
temp = temp + m(56, 45) * m(45, 59)
temp = temp + m(56, 48) * m(48, 59)
temp = temp + m(56, 50) * m(50, 59)
temp = temp + m(56, 52) * m(52, 59)
temp = temp + m(56, 54) * m(54, 59)
temp = temp + m(56, 55) * m(55, 59)
m(56, 59) = ( m(56, 59) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 50) * m(50, 60)
temp = temp + m(56, 54) * m(54, 60)
temp = temp + m(56, 55) * m(55, 60)
m(56, 60) = ( m(56, 60) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 17) * m(17, 61)
temp = temp + m(56, 40) * m(40, 61)
temp = temp + m(56, 43) * m(43, 61)
temp = temp + m(56, 48) * m(48, 61)
temp = temp + m(56, 50) * m(50, 61)
temp = temp + m(56, 52) * m(52, 61)
temp = temp + m(56, 54) * m(54, 61)
temp = temp + m(56, 55) * m(55, 61)
m(56, 61) = ( m(56, 61) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 54) * m(54, 62)
temp = temp + m(56, 55) * m(55, 62)
m(56, 62) = ( m(56, 62) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 48) * m(48, 63)
temp = temp + m(56, 50) * m(50, 63)
temp = temp + m(56, 52) * m(52, 63)
temp = temp + m(56, 54) * m(54, 63)
temp = temp + m(56, 55) * m(55, 63)
m(56, 63) = ( m(56, 63) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 40) * m(40, 64)
temp = temp + m(56, 43) * m(43, 64)
temp = temp + m(56, 48) * m(48, 64)
temp = temp + m(56, 50) * m(50, 64)
temp = temp + m(56, 52) * m(52, 64)
temp = temp + m(56, 54) * m(54, 64)
temp = temp + m(56, 55) * m(55, 64)
m(56, 64) = ( m(56, 64) - temp ) / m(56, 56)
!     Upper part.
temp = m(56, 16) * m(16, 65)
temp = temp + m(56, 17) * m(17, 65)
temp = temp + m(56, 43) * m(43, 65)
temp = temp + m(56, 44) * m(44, 65)
temp = temp + m(56, 45) * m(45, 65)
temp = temp + m(56, 48) * m(48, 65)
temp = temp + m(56, 50) * m(50, 65)
temp = temp + m(56, 52) * m(52, 65)
temp = temp + m(56, 54) * m(54, 65)
temp = temp + m(56, 55) * m(55, 65)
m(56, 65) = ( m(56, 65) - temp ) / m(56, 56)

!     Lower part.
temp = m(57, 20) * m(20, 57)
temp = temp + m(57, 34) * m(34, 57)
temp = temp + m(57, 36) * m(36, 57)
temp = temp + m(57, 37) * m(37, 57)
temp = temp + m(57, 39) * m(39, 57)
temp = temp + m(57, 44) * m(44, 57)
temp = temp + m(57, 48) * m(48, 57)
temp = temp + m(57, 52) * m(52, 57)
temp = temp + m(57, 54) * m(54, 57)
temp = temp + m(57, 55) * m(55, 57)
temp = temp + m(57, 56) * m(56, 57)
m(57, 57) = m(57, 57) - temp
!     Lower part.
temp = m(58, 39) * m(39, 57)
temp = temp + m(58, 44) * m(44, 57)
temp = temp + m(58, 45) * m(45, 57)
temp = temp + m(58, 48) * m(48, 57)
temp = temp + m(58, 52) * m(52, 57)
temp = temp + m(58, 55) * m(55, 57)
temp = temp + m(58, 56) * m(56, 57)
m(58, 57) = m(58, 57) - temp
!     Lower part.
temp = m(59, 34) * m(34, 57)
temp = temp + m(59, 36) * m(36, 57)
temp = temp + m(59, 37) * m(37, 57)
temp = temp + m(59, 39) * m(39, 57)
temp = temp + m(59, 44) * m(44, 57)
temp = temp + m(59, 45) * m(45, 57)
temp = temp + m(59, 55) * m(55, 57)
temp = temp + m(59, 56) * m(56, 57)
m(59, 57) = m(59, 57) - temp
!     Lower part.
temp = m(60, 36) * m(36, 57)
temp = temp + m(60, 37) * m(37, 57)
temp = temp + m(60, 45) * m(45, 57)
temp = temp + m(60, 46) * m(46, 57)
temp = temp + m(60, 47) * m(47, 57)
temp = temp + m(60, 52) * m(52, 57)
temp = temp + m(60, 53) * m(53, 57)
temp = temp + m(60, 54) * m(54, 57)
temp = temp + m(60, 55) * m(55, 57)
temp = temp + m(60, 56) * m(56, 57)
m(60, 57) = m(60, 57) - temp
!     Lower part.
temp = m(61, 20) * m(20, 57)
temp = temp + m(61, 34) * m(34, 57)
temp = temp + m(61, 35) * m(35, 57)
temp = temp + m(61, 36) * m(36, 57)
temp = temp + m(61, 37) * m(37, 57)
temp = temp + m(61, 39) * m(39, 57)
temp = temp + m(61, 44) * m(44, 57)
temp = temp + m(61, 45) * m(45, 57)
temp = temp + m(61, 46) * m(46, 57)
temp = temp + m(61, 47) * m(47, 57)
temp = temp + m(61, 48) * m(48, 57)
temp = temp + m(61, 51) * m(51, 57)
temp = temp + m(61, 52) * m(52, 57)
temp = temp + m(61, 53) * m(53, 57)
temp = temp + m(61, 54) * m(54, 57)
temp = temp + m(61, 55) * m(55, 57)
temp = temp + m(61, 56) * m(56, 57)
m(61, 57) = m(61, 57) - temp
!     Lower part.
temp = m(62, 45) * m(45, 57)
temp = temp + m(62, 53) * m(53, 57)
temp = temp + m(62, 54) * m(54, 57)
temp = temp + m(62, 55) * m(55, 57)
temp = temp + m(62, 56) * m(56, 57)
m(62, 57) = m(62, 57) - temp
!     Lower part.
temp = m(63, 34) * m(34, 57)
temp = temp + m(63, 36) * m(36, 57)
temp = temp + m(63, 37) * m(37, 57)
temp = temp + m(63, 39) * m(39, 57)
temp = temp + m(63, 44) * m(44, 57)
temp = temp + m(63, 45) * m(45, 57)
temp = temp + m(63, 47) * m(47, 57)
temp = temp + m(63, 52) * m(52, 57)
temp = temp + m(63, 53) * m(53, 57)
temp = temp + m(63, 54) * m(54, 57)
temp = temp + m(63, 55) * m(55, 57)
temp = temp + m(63, 56) * m(56, 57)
m(63, 57) = m(63, 57) - temp
!     Lower part.
temp = m(64, 44) * m(44, 57)
temp = temp + m(64, 52) * m(52, 57)
temp = temp + m(64, 53) * m(53, 57)
temp = temp + m(64, 54) * m(54, 57)
temp = temp + m(64, 55) * m(55, 57)
temp = temp + m(64, 56) * m(56, 57)
m(64, 57) = m(64, 57) - temp
!     Lower part.
temp = m(65, 34) * m(34, 57)
temp = temp + m(65, 36) * m(36, 57)
temp = temp + m(65, 37) * m(37, 57)
temp = temp + m(65, 39) * m(39, 57)
temp = temp + m(65, 44) * m(44, 57)
temp = temp + m(65, 46) * m(46, 57)
temp = temp + m(65, 47) * m(47, 57)
temp = temp + m(65, 50) * m(50, 57)
temp = temp + m(65, 52) * m(52, 57)
temp = temp + m(65, 53) * m(53, 57)
temp = temp + m(65, 54) * m(54, 57)
temp = temp + m(65, 55) * m(55, 57)
temp = temp + m(65, 56) * m(56, 57)
m(65, 57) = m(65, 57) - temp
!     Upper part.
temp = m(57, 54) * m(54, 58)
temp = temp + m(57, 55) * m(55, 58)
temp = temp + m(57, 56) * m(56, 58)
m(57, 58) = ( m(57, 58) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 3) * m(3, 59)
temp = temp + m(57, 34) * m(34, 59)
temp = temp + m(57, 36) * m(36, 59)
temp = temp + m(57, 37) * m(37, 59)
temp = temp + m(57, 39) * m(39, 59)
temp = temp + m(57, 44) * m(44, 59)
temp = temp + m(57, 48) * m(48, 59)
temp = temp + m(57, 52) * m(52, 59)
temp = temp + m(57, 54) * m(54, 59)
temp = temp + m(57, 55) * m(55, 59)
temp = temp + m(57, 56) * m(56, 59)
m(57, 59) = ( m(57, 59) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 54) * m(54, 60)
temp = temp + m(57, 55) * m(55, 60)
temp = temp + m(57, 56) * m(56, 60)
m(57, 60) = ( m(57, 60) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 20) * m(20, 61)
temp = temp + m(57, 48) * m(48, 61)
temp = temp + m(57, 52) * m(52, 61)
temp = temp + m(57, 54) * m(54, 61)
temp = temp + m(57, 55) * m(55, 61)
temp = temp + m(57, 56) * m(56, 61)
m(57, 61) = ( m(57, 61) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 54) * m(54, 62)
temp = temp + m(57, 55) * m(55, 62)
temp = temp + m(57, 56) * m(56, 62)
m(57, 62) = ( m(57, 62) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 48) * m(48, 63)
temp = temp + m(57, 52) * m(52, 63)
temp = temp + m(57, 54) * m(54, 63)
temp = temp + m(57, 55) * m(55, 63)
temp = temp + m(57, 56) * m(56, 63)
m(57, 63) = ( m(57, 63) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 48) * m(48, 64)
temp = temp + m(57, 52) * m(52, 64)
temp = temp + m(57, 54) * m(54, 64)
temp = temp + m(57, 55) * m(55, 64)
temp = temp + m(57, 56) * m(56, 64)
m(57, 64) = ( m(57, 64) - temp ) / m(57, 57)
!     Upper part.
temp = m(57, 44) * m(44, 65)
temp = temp + m(57, 48) * m(48, 65)
temp = temp + m(57, 52) * m(52, 65)
temp = temp + m(57, 54) * m(54, 65)
temp = temp + m(57, 55) * m(55, 65)
temp = temp + m(57, 56) * m(56, 65)
m(57, 65) = ( m(57, 65) - temp ) / m(57, 57)

!     Lower part.
temp = m(58, 18) * m(18, 58)
temp = temp + m(58, 55) * m(55, 58)
temp = temp + m(58, 56) * m(56, 58)
temp = temp + m(58, 57) * m(57, 58)
m(58, 58) = m(58, 58) - temp
!     Lower part.
temp = m(59, 55) * m(55, 58)
temp = temp + m(59, 56) * m(56, 58)
temp = temp + m(59, 57) * m(57, 58)
m(59, 58) = m(59, 58) - temp
!     Lower part.
temp = m(60, 18) * m(18, 58)
temp = temp + m(60, 53) * m(53, 58)
temp = temp + m(60, 54) * m(54, 58)
temp = temp + m(60, 55) * m(55, 58)
temp = temp + m(60, 56) * m(56, 58)
temp = temp + m(60, 57) * m(57, 58)
m(60, 58) = m(60, 58) - temp
!     Lower part.
temp = m(61, 53) * m(53, 58)
temp = temp + m(61, 54) * m(54, 58)
temp = temp + m(61, 55) * m(55, 58)
temp = temp + m(61, 56) * m(56, 58)
temp = temp + m(61, 57) * m(57, 58)
m(61, 58) = m(61, 58) - temp
!     Lower part.
temp = m(62, 33) * m(33, 58)
temp = temp + m(62, 53) * m(53, 58)
temp = temp + m(62, 54) * m(54, 58)
temp = temp + m(62, 55) * m(55, 58)
temp = temp + m(62, 56) * m(56, 58)
temp = temp + m(62, 57) * m(57, 58)
m(62, 58) = m(62, 58) - temp
!     Lower part.
temp = m(63, 53) * m(53, 58)
temp = temp + m(63, 54) * m(54, 58)
temp = temp + m(63, 55) * m(55, 58)
temp = temp + m(63, 56) * m(56, 58)
temp = temp + m(63, 57) * m(57, 58)
m(63, 58) = m(63, 58) - temp
!     Lower part.
temp = m(64, 53) * m(53, 58)
temp = temp + m(64, 54) * m(54, 58)
temp = temp + m(64, 55) * m(55, 58)
temp = temp + m(64, 56) * m(56, 58)
temp = temp + m(64, 57) * m(57, 58)
m(64, 58) = m(64, 58) - temp
!     Lower part.
temp = m(65, 18) * m(18, 58)
temp = temp + m(65, 53) * m(53, 58)
temp = temp + m(65, 54) * m(54, 58)
temp = temp + m(65, 55) * m(55, 58)
temp = temp + m(65, 56) * m(56, 58)
temp = temp + m(65, 57) * m(57, 58)
m(65, 58) = m(65, 58) - temp
!     Upper part.
temp = m(58, 39) * m(39, 59)
temp = temp + m(58, 44) * m(44, 59)
temp = temp + m(58, 45) * m(45, 59)
temp = temp + m(58, 48) * m(48, 59)
temp = temp + m(58, 52) * m(52, 59)
temp = temp + m(58, 55) * m(55, 59)
temp = temp + m(58, 56) * m(56, 59)
temp = temp + m(58, 57) * m(57, 59)
m(58, 59) = ( m(58, 59) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 55) * m(55, 60)
temp = temp + m(58, 56) * m(56, 60)
temp = temp + m(58, 57) * m(57, 60)
m(58, 60) = ( m(58, 60) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 48) * m(48, 61)
temp = temp + m(58, 52) * m(52, 61)
temp = temp + m(58, 55) * m(55, 61)
temp = temp + m(58, 56) * m(56, 61)
temp = temp + m(58, 57) * m(57, 61)
m(58, 61) = ( m(58, 61) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 55) * m(55, 62)
temp = temp + m(58, 56) * m(56, 62)
temp = temp + m(58, 57) * m(57, 62)
m(58, 62) = ( m(58, 62) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 48) * m(48, 63)
temp = temp + m(58, 52) * m(52, 63)
temp = temp + m(58, 55) * m(55, 63)
temp = temp + m(58, 56) * m(56, 63)
temp = temp + m(58, 57) * m(57, 63)
m(58, 63) = ( m(58, 63) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 48) * m(48, 64)
temp = temp + m(58, 52) * m(52, 64)
temp = temp + m(58, 55) * m(55, 64)
temp = temp + m(58, 56) * m(56, 64)
temp = temp + m(58, 57) * m(57, 64)
m(58, 64) = ( m(58, 64) - temp ) / m(58, 58)
!     Upper part.
temp = m(58, 18) * m(18, 65)
temp = temp + m(58, 44) * m(44, 65)
temp = temp + m(58, 45) * m(45, 65)
temp = temp + m(58, 48) * m(48, 65)
temp = temp + m(58, 52) * m(52, 65)
temp = temp + m(58, 55) * m(55, 65)
temp = temp + m(58, 56) * m(56, 65)
temp = temp + m(58, 57) * m(57, 65)
m(58, 65) = ( m(58, 65) - temp ) / m(58, 58)

!     Lower part.
temp = m(59, 34) * m(34, 59)
temp = temp + m(59, 36) * m(36, 59)
temp = temp + m(59, 37) * m(37, 59)
temp = temp + m(59, 39) * m(39, 59)
temp = temp + m(59, 41) * m(41, 59)
temp = temp + m(59, 44) * m(44, 59)
temp = temp + m(59, 45) * m(45, 59)
temp = temp + m(59, 55) * m(55, 59)
temp = temp + m(59, 56) * m(56, 59)
temp = temp + m(59, 57) * m(57, 59)
temp = temp + m(59, 58) * m(58, 59)
m(59, 59) = m(59, 59) - temp
!     Lower part.
temp = m(60, 36) * m(36, 59)
temp = temp + m(60, 37) * m(37, 59)
temp = temp + m(60, 45) * m(45, 59)
temp = temp + m(60, 46) * m(46, 59)
temp = temp + m(60, 47) * m(47, 59)
temp = temp + m(60, 52) * m(52, 59)
temp = temp + m(60, 53) * m(53, 59)
temp = temp + m(60, 54) * m(54, 59)
temp = temp + m(60, 55) * m(55, 59)
temp = temp + m(60, 56) * m(56, 59)
temp = temp + m(60, 57) * m(57, 59)
temp = temp + m(60, 58) * m(58, 59)
m(60, 59) = m(60, 59) - temp
!     Lower part.
temp = m(61, 3) * m(3, 59)
temp = temp + m(61, 34) * m(34, 59)
temp = temp + m(61, 35) * m(35, 59)
temp = temp + m(61, 36) * m(36, 59)
temp = temp + m(61, 37) * m(37, 59)
temp = temp + m(61, 39) * m(39, 59)
temp = temp + m(61, 41) * m(41, 59)
temp = temp + m(61, 42) * m(42, 59)
temp = temp + m(61, 44) * m(44, 59)
temp = temp + m(61, 45) * m(45, 59)
temp = temp + m(61, 46) * m(46, 59)
temp = temp + m(61, 47) * m(47, 59)
temp = temp + m(61, 48) * m(48, 59)
temp = temp + m(61, 51) * m(51, 59)
temp = temp + m(61, 52) * m(52, 59)
temp = temp + m(61, 53) * m(53, 59)
temp = temp + m(61, 54) * m(54, 59)
temp = temp + m(61, 55) * m(55, 59)
temp = temp + m(61, 56) * m(56, 59)
temp = temp + m(61, 57) * m(57, 59)
temp = temp + m(61, 58) * m(58, 59)
m(61, 59) = m(61, 59) - temp
!     Lower part.
temp = m(62, 41) * m(41, 59)
temp = temp + m(62, 42) * m(42, 59)
temp = temp + m(62, 45) * m(45, 59)
temp = temp + m(62, 53) * m(53, 59)
temp = temp + m(62, 54) * m(54, 59)
temp = temp + m(62, 55) * m(55, 59)
temp = temp + m(62, 56) * m(56, 59)
temp = temp + m(62, 57) * m(57, 59)
temp = temp + m(62, 58) * m(58, 59)
m(62, 59) = m(62, 59) - temp
!     Lower part.
temp = m(63, 34) * m(34, 59)
temp = temp + m(63, 36) * m(36, 59)
temp = temp + m(63, 37) * m(37, 59)
temp = temp + m(63, 39) * m(39, 59)
temp = temp + m(63, 41) * m(41, 59)
temp = temp + m(63, 42) * m(42, 59)
temp = temp + m(63, 44) * m(44, 59)
temp = temp + m(63, 45) * m(45, 59)
temp = temp + m(63, 47) * m(47, 59)
temp = temp + m(63, 52) * m(52, 59)
temp = temp + m(63, 53) * m(53, 59)
temp = temp + m(63, 54) * m(54, 59)
temp = temp + m(63, 55) * m(55, 59)
temp = temp + m(63, 56) * m(56, 59)
temp = temp + m(63, 57) * m(57, 59)
temp = temp + m(63, 58) * m(58, 59)
m(63, 59) = m(63, 59) - temp
!     Lower part.
temp = m(64, 44) * m(44, 59)
temp = temp + m(64, 52) * m(52, 59)
temp = temp + m(64, 53) * m(53, 59)
temp = temp + m(64, 54) * m(54, 59)
temp = temp + m(64, 55) * m(55, 59)
temp = temp + m(64, 56) * m(56, 59)
temp = temp + m(64, 57) * m(57, 59)
temp = temp + m(64, 58) * m(58, 59)
m(64, 59) = m(64, 59) - temp
!     Lower part.
temp = m(65, 34) * m(34, 59)
temp = temp + m(65, 36) * m(36, 59)
temp = temp + m(65, 37) * m(37, 59)
temp = temp + m(65, 39) * m(39, 59)
temp = temp + m(65, 44) * m(44, 59)
temp = temp + m(65, 46) * m(46, 59)
temp = temp + m(65, 47) * m(47, 59)
temp = temp + m(65, 50) * m(50, 59)
temp = temp + m(65, 52) * m(52, 59)
temp = temp + m(65, 53) * m(53, 59)
temp = temp + m(65, 54) * m(54, 59)
temp = temp + m(65, 55) * m(55, 59)
temp = temp + m(65, 56) * m(56, 59)
temp = temp + m(65, 57) * m(57, 59)
temp = temp + m(65, 58) * m(58, 59)
m(65, 59) = m(65, 59) - temp
!     Upper part.
temp = m(59, 55) * m(55, 60)
temp = temp + m(59, 56) * m(56, 60)
temp = temp + m(59, 57) * m(57, 60)
temp = temp + m(59, 58) * m(58, 60)
m(59, 60) = ( m(59, 60) - temp ) / m(59, 59)
!     Upper part.
temp = m(59, 41) * m(41, 61)
temp = temp + m(59, 43) * m(43, 61)
temp = temp + m(59, 55) * m(55, 61)
temp = temp + m(59, 56) * m(56, 61)
temp = temp + m(59, 57) * m(57, 61)
temp = temp + m(59, 58) * m(58, 61)
m(59, 61) = ( m(59, 61) - temp ) / m(59, 59)
!     Upper part.
temp = m(59, 55) * m(55, 62)
temp = temp + m(59, 56) * m(56, 62)
temp = temp + m(59, 57) * m(57, 62)
temp = temp + m(59, 58) * m(58, 62)
m(59, 62) = ( m(59, 62) - temp ) / m(59, 59)
!     Upper part.
temp = m(59, 55) * m(55, 63)
temp = temp + m(59, 56) * m(56, 63)
temp = temp + m(59, 57) * m(57, 63)
temp = temp + m(59, 58) * m(58, 63)
m(59, 63) = ( m(59, 63) - temp ) / m(59, 59)
!     Upper part.
temp = m(59, 41) * m(41, 64)
temp = temp + m(59, 43) * m(43, 64)
temp = temp + m(59, 55) * m(55, 64)
temp = temp + m(59, 56) * m(56, 64)
temp = temp + m(59, 57) * m(57, 64)
temp = temp + m(59, 58) * m(58, 64)
m(59, 64) = ( m(59, 64) - temp ) / m(59, 59)
!     Upper part.
temp = m(59, 43) * m(43, 65)
temp = temp + m(59, 44) * m(44, 65)
temp = temp + m(59, 45) * m(45, 65)
temp = temp + m(59, 55) * m(55, 65)
temp = temp + m(59, 56) * m(56, 65)
temp = temp + m(59, 57) * m(57, 65)
temp = temp + m(59, 58) * m(58, 65)
m(59, 65) = ( m(59, 65) - temp ) / m(59, 59)

!     Lower part.
temp = m(60, 53) * m(53, 60)
temp = temp + m(60, 54) * m(54, 60)
temp = temp + m(60, 55) * m(55, 60)
temp = temp + m(60, 56) * m(56, 60)
temp = temp + m(60, 57) * m(57, 60)
temp = temp + m(60, 58) * m(58, 60)
temp = temp + m(60, 59) * m(59, 60)
m(60, 60) = m(60, 60) - temp
!     Lower part.
temp = m(61, 51) * m(51, 60)
temp = temp + m(61, 53) * m(53, 60)
temp = temp + m(61, 54) * m(54, 60)
temp = temp + m(61, 55) * m(55, 60)
temp = temp + m(61, 56) * m(56, 60)
temp = temp + m(61, 57) * m(57, 60)
temp = temp + m(61, 58) * m(58, 60)
temp = temp + m(61, 59) * m(59, 60)
m(61, 60) = m(61, 60) - temp
!     Lower part.
temp = m(62, 53) * m(53, 60)
temp = temp + m(62, 54) * m(54, 60)
temp = temp + m(62, 55) * m(55, 60)
temp = temp + m(62, 56) * m(56, 60)
temp = temp + m(62, 57) * m(57, 60)
temp = temp + m(62, 58) * m(58, 60)
temp = temp + m(62, 59) * m(59, 60)
m(62, 60) = m(62, 60) - temp
!     Lower part.
temp = m(63, 53) * m(53, 60)
temp = temp + m(63, 54) * m(54, 60)
temp = temp + m(63, 55) * m(55, 60)
temp = temp + m(63, 56) * m(56, 60)
temp = temp + m(63, 57) * m(57, 60)
temp = temp + m(63, 58) * m(58, 60)
temp = temp + m(63, 59) * m(59, 60)
m(63, 60) = m(63, 60) - temp
!     Lower part.
temp = m(64, 53) * m(53, 60)
temp = temp + m(64, 54) * m(54, 60)
temp = temp + m(64, 55) * m(55, 60)
temp = temp + m(64, 56) * m(56, 60)
temp = temp + m(64, 57) * m(57, 60)
temp = temp + m(64, 58) * m(58, 60)
temp = temp + m(64, 59) * m(59, 60)
m(64, 60) = m(64, 60) - temp
!     Lower part.
temp = m(65, 50) * m(50, 60)
temp = temp + m(65, 53) * m(53, 60)
temp = temp + m(65, 54) * m(54, 60)
temp = temp + m(65, 55) * m(55, 60)
temp = temp + m(65, 56) * m(56, 60)
temp = temp + m(65, 57) * m(57, 60)
temp = temp + m(65, 58) * m(58, 60)
temp = temp + m(65, 59) * m(59, 60)
m(65, 60) = m(65, 60) - temp
!     Upper part.
temp = m(60, 32) * m(32, 61)
temp = temp + m(60, 46) * m(46, 61)
temp = temp + m(60, 47) * m(47, 61)
temp = temp + m(60, 52) * m(52, 61)
temp = temp + m(60, 53) * m(53, 61)
temp = temp + m(60, 54) * m(54, 61)
temp = temp + m(60, 55) * m(55, 61)
temp = temp + m(60, 56) * m(56, 61)
temp = temp + m(60, 57) * m(57, 61)
temp = temp + m(60, 58) * m(58, 61)
temp = temp + m(60, 59) * m(59, 61)
m(60, 61) = ( m(60, 61) - temp ) / m(60, 60)
!     Upper part.
temp = m(60, 53) * m(53, 62)
temp = temp + m(60, 54) * m(54, 62)
temp = temp + m(60, 55) * m(55, 62)
temp = temp + m(60, 56) * m(56, 62)
temp = temp + m(60, 57) * m(57, 62)
temp = temp + m(60, 58) * m(58, 62)
temp = temp + m(60, 59) * m(59, 62)
m(60, 62) = ( m(60, 62) - temp ) / m(60, 60)
!     Upper part.
temp = m(60, 32) * m(32, 63)
temp = temp + m(60, 52) * m(52, 63)
temp = temp + m(60, 53) * m(53, 63)
temp = temp + m(60, 54) * m(54, 63)
temp = temp + m(60, 55) * m(55, 63)
temp = temp + m(60, 56) * m(56, 63)
temp = temp + m(60, 57) * m(57, 63)
temp = temp + m(60, 58) * m(58, 63)
temp = temp + m(60, 59) * m(59, 63)
m(60, 63) = ( m(60, 63) - temp ) / m(60, 60)
!     Upper part.
temp = m(60, 46) * m(46, 64)
temp = temp + m(60, 47) * m(47, 64)
temp = temp + m(60, 52) * m(52, 64)
temp = temp + m(60, 53) * m(53, 64)
temp = temp + m(60, 54) * m(54, 64)
temp = temp + m(60, 55) * m(55, 64)
temp = temp + m(60, 56) * m(56, 64)
temp = temp + m(60, 57) * m(57, 64)
temp = temp + m(60, 58) * m(58, 64)
temp = temp + m(60, 59) * m(59, 64)
m(60, 64) = ( m(60, 64) - temp ) / m(60, 60)
!     Upper part.
temp = m(60, 18) * m(18, 65)
temp = temp + m(60, 19) * m(19, 65)
temp = temp + m(60, 45) * m(45, 65)
temp = temp + m(60, 46) * m(46, 65)
temp = temp + m(60, 47) * m(47, 65)
temp = temp + m(60, 52) * m(52, 65)
temp = temp + m(60, 53) * m(53, 65)
temp = temp + m(60, 54) * m(54, 65)
temp = temp + m(60, 55) * m(55, 65)
temp = temp + m(60, 56) * m(56, 65)
temp = temp + m(60, 57) * m(57, 65)
temp = temp + m(60, 58) * m(58, 65)
temp = temp + m(60, 59) * m(59, 65)
m(60, 65) = ( m(60, 65) - temp ) / m(60, 60)

!     Lower part.
temp = m(61, 17) * m(17, 61)
temp = temp + m(61, 20) * m(20, 61)
temp = temp + m(61, 21) * m(21, 61)
temp = temp + m(61, 22) * m(22, 61)
temp = temp + m(61, 32) * m(32, 61)
temp = temp + m(61, 35) * m(35, 61)
temp = temp + m(61, 40) * m(40, 61)
temp = temp + m(61, 41) * m(41, 61)
temp = temp + m(61, 42) * m(42, 61)
temp = temp + m(61, 43) * m(43, 61)
temp = temp + m(61, 46) * m(46, 61)
temp = temp + m(61, 47) * m(47, 61)
temp = temp + m(61, 48) * m(48, 61)
temp = temp + m(61, 51) * m(51, 61)
temp = temp + m(61, 52) * m(52, 61)
temp = temp + m(61, 53) * m(53, 61)
temp = temp + m(61, 54) * m(54, 61)
temp = temp + m(61, 55) * m(55, 61)
temp = temp + m(61, 56) * m(56, 61)
temp = temp + m(61, 57) * m(57, 61)
temp = temp + m(61, 58) * m(58, 61)
temp = temp + m(61, 59) * m(59, 61)
temp = temp + m(61, 60) * m(60, 61)
m(61, 61) = m(61, 61) - temp
!     Lower part.
temp = m(62, 33) * m(33, 61)
temp = temp + m(62, 41) * m(41, 61)
temp = temp + m(62, 42) * m(42, 61)
temp = temp + m(62, 43) * m(43, 61)
temp = temp + m(62, 53) * m(53, 61)
temp = temp + m(62, 54) * m(54, 61)
temp = temp + m(62, 55) * m(55, 61)
temp = temp + m(62, 56) * m(56, 61)
temp = temp + m(62, 57) * m(57, 61)
temp = temp + m(62, 58) * m(58, 61)
temp = temp + m(62, 59) * m(59, 61)
temp = temp + m(62, 60) * m(60, 61)
m(62, 61) = m(62, 61) - temp
!     Lower part.
temp = m(63, 22) * m(22, 61)
temp = temp + m(63, 32) * m(32, 61)
temp = temp + m(63, 40) * m(40, 61)
temp = temp + m(63, 41) * m(41, 61)
temp = temp + m(63, 42) * m(42, 61)
temp = temp + m(63, 43) * m(43, 61)
temp = temp + m(63, 47) * m(47, 61)
temp = temp + m(63, 52) * m(52, 61)
temp = temp + m(63, 53) * m(53, 61)
temp = temp + m(63, 54) * m(54, 61)
temp = temp + m(63, 55) * m(55, 61)
temp = temp + m(63, 56) * m(56, 61)
temp = temp + m(63, 57) * m(57, 61)
temp = temp + m(63, 58) * m(58, 61)
temp = temp + m(63, 59) * m(59, 61)
temp = temp + m(63, 60) * m(60, 61)
m(63, 61) = m(63, 61) - temp
!     Lower part.
temp = m(64, 21) * m(21, 61)
temp = temp + m(64, 52) * m(52, 61)
temp = temp + m(64, 53) * m(53, 61)
temp = temp + m(64, 54) * m(54, 61)
temp = temp + m(64, 55) * m(55, 61)
temp = temp + m(64, 56) * m(56, 61)
temp = temp + m(64, 57) * m(57, 61)
temp = temp + m(64, 58) * m(58, 61)
temp = temp + m(64, 59) * m(59, 61)
temp = temp + m(64, 60) * m(60, 61)
m(64, 61) = m(64, 61) - temp
!     Lower part.
temp = m(65, 17) * m(17, 61)
temp = temp + m(65, 21) * m(21, 61)
temp = temp + m(65, 43) * m(43, 61)
temp = temp + m(65, 46) * m(46, 61)
temp = temp + m(65, 47) * m(47, 61)
temp = temp + m(65, 50) * m(50, 61)
temp = temp + m(65, 52) * m(52, 61)
temp = temp + m(65, 53) * m(53, 61)
temp = temp + m(65, 54) * m(54, 61)
temp = temp + m(65, 55) * m(55, 61)
temp = temp + m(65, 56) * m(56, 61)
temp = temp + m(65, 57) * m(57, 61)
temp = temp + m(65, 58) * m(58, 61)
temp = temp + m(65, 59) * m(59, 61)
temp = temp + m(65, 60) * m(60, 61)
m(65, 61) = m(65, 61) - temp
!     Upper part.
temp = m(61, 53) * m(53, 62)
temp = temp + m(61, 54) * m(54, 62)
temp = temp + m(61, 55) * m(55, 62)
temp = temp + m(61, 56) * m(56, 62)
temp = temp + m(61, 57) * m(57, 62)
temp = temp + m(61, 58) * m(58, 62)
temp = temp + m(61, 59) * m(59, 62)
temp = temp + m(61, 60) * m(60, 62)
m(61, 62) = ( m(61, 62) - temp ) / m(61, 61)
!     Upper part.
temp = m(61, 32) * m(32, 63)
temp = temp + m(61, 48) * m(48, 63)
temp = temp + m(61, 51) * m(51, 63)
temp = temp + m(61, 52) * m(52, 63)
temp = temp + m(61, 53) * m(53, 63)
temp = temp + m(61, 54) * m(54, 63)
temp = temp + m(61, 55) * m(55, 63)
temp = temp + m(61, 56) * m(56, 63)
temp = temp + m(61, 57) * m(57, 63)
temp = temp + m(61, 58) * m(58, 63)
temp = temp + m(61, 59) * m(59, 63)
temp = temp + m(61, 60) * m(60, 63)
m(61, 63) = ( m(61, 63) - temp ) / m(61, 61)
!     Upper part.
temp = m(61, 15) * m(15, 64)
temp = temp + m(61, 21) * m(21, 64)
temp = temp + m(61, 22) * m(22, 64)
temp = temp + m(61, 35) * m(35, 64)
temp = temp + m(61, 40) * m(40, 64)
temp = temp + m(61, 41) * m(41, 64)
temp = temp + m(61, 42) * m(42, 64)
temp = temp + m(61, 43) * m(43, 64)
temp = temp + m(61, 46) * m(46, 64)
temp = temp + m(61, 47) * m(47, 64)
temp = temp + m(61, 48) * m(48, 64)
temp = temp + m(61, 51) * m(51, 64)
temp = temp + m(61, 52) * m(52, 64)
temp = temp + m(61, 53) * m(53, 64)
temp = temp + m(61, 54) * m(54, 64)
temp = temp + m(61, 55) * m(55, 64)
temp = temp + m(61, 56) * m(56, 64)
temp = temp + m(61, 57) * m(57, 64)
temp = temp + m(61, 58) * m(58, 64)
temp = temp + m(61, 59) * m(59, 64)
temp = temp + m(61, 60) * m(60, 64)
m(61, 64) = ( m(61, 64) - temp ) / m(61, 61)
!     Upper part.
temp = m(61, 17) * m(17, 65)
temp = temp + m(61, 19) * m(19, 65)
temp = temp + m(61, 43) * m(43, 65)
temp = temp + m(61, 44) * m(44, 65)
temp = temp + m(61, 45) * m(45, 65)
temp = temp + m(61, 46) * m(46, 65)
temp = temp + m(61, 47) * m(47, 65)
temp = temp + m(61, 48) * m(48, 65)
temp = temp + m(61, 51) * m(51, 65)
temp = temp + m(61, 52) * m(52, 65)
temp = temp + m(61, 53) * m(53, 65)
temp = temp + m(61, 54) * m(54, 65)
temp = temp + m(61, 55) * m(55, 65)
temp = temp + m(61, 56) * m(56, 65)
temp = temp + m(61, 57) * m(57, 65)
temp = temp + m(61, 58) * m(58, 65)
temp = temp + m(61, 59) * m(59, 65)
temp = temp + m(61, 60) * m(60, 65)
m(61, 65) = ( m(61, 65) - temp ) / m(61, 61)

!     Lower part.
temp = m(62, 13) * m(13, 62)
temp = temp + m(62, 33) * m(33, 62)
temp = temp + m(62, 53) * m(53, 62)
temp = temp + m(62, 54) * m(54, 62)
temp = temp + m(62, 55) * m(55, 62)
temp = temp + m(62, 56) * m(56, 62)
temp = temp + m(62, 57) * m(57, 62)
temp = temp + m(62, 58) * m(58, 62)
temp = temp + m(62, 59) * m(59, 62)
temp = temp + m(62, 60) * m(60, 62)
temp = temp + m(62, 61) * m(61, 62)
m(62, 62) = m(62, 62) - temp
!     Lower part.
temp = m(63, 53) * m(53, 62)
temp = temp + m(63, 54) * m(54, 62)
temp = temp + m(63, 55) * m(55, 62)
temp = temp + m(63, 56) * m(56, 62)
temp = temp + m(63, 57) * m(57, 62)
temp = temp + m(63, 58) * m(58, 62)
temp = temp + m(63, 59) * m(59, 62)
temp = temp + m(63, 60) * m(60, 62)
temp = temp + m(63, 61) * m(61, 62)
m(63, 62) = m(63, 62) - temp
!     Lower part.
temp = m(64, 53) * m(53, 62)
temp = temp + m(64, 54) * m(54, 62)
temp = temp + m(64, 55) * m(55, 62)
temp = temp + m(64, 56) * m(56, 62)
temp = temp + m(64, 57) * m(57, 62)
temp = temp + m(64, 58) * m(58, 62)
temp = temp + m(64, 59) * m(59, 62)
temp = temp + m(64, 60) * m(60, 62)
temp = temp + m(64, 61) * m(61, 62)
m(64, 62) = m(64, 62) - temp
!     Lower part.
temp = m(65, 13) * m(13, 62)
temp = temp + m(65, 53) * m(53, 62)
temp = temp + m(65, 54) * m(54, 62)
temp = temp + m(65, 55) * m(55, 62)
temp = temp + m(65, 56) * m(56, 62)
temp = temp + m(65, 57) * m(57, 62)
temp = temp + m(65, 58) * m(58, 62)
temp = temp + m(65, 59) * m(59, 62)
temp = temp + m(65, 60) * m(60, 62)
temp = temp + m(65, 61) * m(61, 62)
m(65, 62) = m(65, 62) - temp
!     Upper part.
temp = m(62, 53) * m(53, 63)
temp = temp + m(62, 54) * m(54, 63)
temp = temp + m(62, 55) * m(55, 63)
temp = temp + m(62, 56) * m(56, 63)
temp = temp + m(62, 57) * m(57, 63)
temp = temp + m(62, 58) * m(58, 63)
temp = temp + m(62, 59) * m(59, 63)
temp = temp + m(62, 60) * m(60, 63)
temp = temp + m(62, 61) * m(61, 63)
m(62, 63) = ( m(62, 63) - temp ) / m(62, 62)
!     Upper part.
temp = m(62, 41) * m(41, 64)
temp = temp + m(62, 42) * m(42, 64)
temp = temp + m(62, 43) * m(43, 64)
temp = temp + m(62, 53) * m(53, 64)
temp = temp + m(62, 54) * m(54, 64)
temp = temp + m(62, 55) * m(55, 64)
temp = temp + m(62, 56) * m(56, 64)
temp = temp + m(62, 57) * m(57, 64)
temp = temp + m(62, 58) * m(58, 64)
temp = temp + m(62, 59) * m(59, 64)
temp = temp + m(62, 60) * m(60, 64)
temp = temp + m(62, 61) * m(61, 64)
m(62, 64) = ( m(62, 64) - temp ) / m(62, 62)
!     Upper part.
temp = m(62, 13) * m(13, 65)
temp = temp + m(62, 43) * m(43, 65)
temp = temp + m(62, 45) * m(45, 65)
temp = temp + m(62, 53) * m(53, 65)
temp = temp + m(62, 54) * m(54, 65)
temp = temp + m(62, 55) * m(55, 65)
temp = temp + m(62, 56) * m(56, 65)
temp = temp + m(62, 57) * m(57, 65)
temp = temp + m(62, 58) * m(58, 65)
temp = temp + m(62, 59) * m(59, 65)
temp = temp + m(62, 60) * m(60, 65)
temp = temp + m(62, 61) * m(61, 65)
m(62, 65) = ( m(62, 65) - temp ) / m(62, 62)

!     Lower part.
temp = m(63, 32) * m(32, 63)
temp = temp + m(63, 52) * m(52, 63)
temp = temp + m(63, 53) * m(53, 63)
temp = temp + m(63, 54) * m(54, 63)
temp = temp + m(63, 55) * m(55, 63)
temp = temp + m(63, 56) * m(56, 63)
temp = temp + m(63, 57) * m(57, 63)
temp = temp + m(63, 58) * m(58, 63)
temp = temp + m(63, 59) * m(59, 63)
temp = temp + m(63, 60) * m(60, 63)
temp = temp + m(63, 61) * m(61, 63)
temp = temp + m(63, 62) * m(62, 63)
m(63, 63) = m(63, 63) - temp
!     Lower part.
temp = m(64, 52) * m(52, 63)
temp = temp + m(64, 53) * m(53, 63)
temp = temp + m(64, 54) * m(54, 63)
temp = temp + m(64, 55) * m(55, 63)
temp = temp + m(64, 56) * m(56, 63)
temp = temp + m(64, 57) * m(57, 63)
temp = temp + m(64, 58) * m(58, 63)
temp = temp + m(64, 59) * m(59, 63)
temp = temp + m(64, 60) * m(60, 63)
temp = temp + m(64, 61) * m(61, 63)
temp = temp + m(64, 62) * m(62, 63)
m(64, 63) = m(64, 63) - temp
!     Lower part.
temp = m(65, 50) * m(50, 63)
temp = temp + m(65, 52) * m(52, 63)
temp = temp + m(65, 53) * m(53, 63)
temp = temp + m(65, 54) * m(54, 63)
temp = temp + m(65, 55) * m(55, 63)
temp = temp + m(65, 56) * m(56, 63)
temp = temp + m(65, 57) * m(57, 63)
temp = temp + m(65, 58) * m(58, 63)
temp = temp + m(65, 59) * m(59, 63)
temp = temp + m(65, 60) * m(60, 63)
temp = temp + m(65, 61) * m(61, 63)
temp = temp + m(65, 62) * m(62, 63)
m(65, 63) = m(65, 63) - temp
!     Upper part.
temp = m(63, 22) * m(22, 64)
temp = temp + m(63, 40) * m(40, 64)
temp = temp + m(63, 41) * m(41, 64)
temp = temp + m(63, 42) * m(42, 64)
temp = temp + m(63, 43) * m(43, 64)
temp = temp + m(63, 47) * m(47, 64)
temp = temp + m(63, 52) * m(52, 64)
temp = temp + m(63, 53) * m(53, 64)
temp = temp + m(63, 54) * m(54, 64)
temp = temp + m(63, 55) * m(55, 64)
temp = temp + m(63, 56) * m(56, 64)
temp = temp + m(63, 57) * m(57, 64)
temp = temp + m(63, 58) * m(58, 64)
temp = temp + m(63, 59) * m(59, 64)
temp = temp + m(63, 60) * m(60, 64)
temp = temp + m(63, 61) * m(61, 64)
temp = temp + m(63, 62) * m(62, 64)
m(63, 64) = ( m(63, 64) - temp ) / m(63, 63)
!     Upper part.
temp = m(63, 19) * m(19, 65)
temp = temp + m(63, 43) * m(43, 65)
temp = temp + m(63, 44) * m(44, 65)
temp = temp + m(63, 45) * m(45, 65)
temp = temp + m(63, 47) * m(47, 65)
temp = temp + m(63, 52) * m(52, 65)
temp = temp + m(63, 53) * m(53, 65)
temp = temp + m(63, 54) * m(54, 65)
temp = temp + m(63, 55) * m(55, 65)
temp = temp + m(63, 56) * m(56, 65)
temp = temp + m(63, 57) * m(57, 65)
temp = temp + m(63, 58) * m(58, 65)
temp = temp + m(63, 59) * m(59, 65)
temp = temp + m(63, 60) * m(60, 65)
temp = temp + m(63, 61) * m(61, 65)
temp = temp + m(63, 62) * m(62, 65)
m(63, 65) = ( m(63, 65) - temp ) / m(63, 63)

!     Lower part.
temp = m(64, 15) * m(15, 64)
temp = temp + m(64, 21) * m(21, 64)
temp = temp + m(64, 23) * m(23, 64)
temp = temp + m(64, 52) * m(52, 64)
temp = temp + m(64, 53) * m(53, 64)
temp = temp + m(64, 54) * m(54, 64)
temp = temp + m(64, 55) * m(55, 64)
temp = temp + m(64, 56) * m(56, 64)
temp = temp + m(64, 57) * m(57, 64)
temp = temp + m(64, 58) * m(58, 64)
temp = temp + m(64, 59) * m(59, 64)
temp = temp + m(64, 60) * m(60, 64)
temp = temp + m(64, 61) * m(61, 64)
temp = temp + m(64, 62) * m(62, 64)
temp = temp + m(64, 63) * m(63, 64)
m(64, 64) = m(64, 64) - temp
!     Lower part.
temp = m(65, 15) * m(15, 64)
temp = temp + m(65, 21) * m(21, 64)
temp = temp + m(65, 23) * m(23, 64)
temp = temp + m(65, 43) * m(43, 64)
temp = temp + m(65, 46) * m(46, 64)
temp = temp + m(65, 47) * m(47, 64)
temp = temp + m(65, 50) * m(50, 64)
temp = temp + m(65, 52) * m(52, 64)
temp = temp + m(65, 53) * m(53, 64)
temp = temp + m(65, 54) * m(54, 64)
temp = temp + m(65, 55) * m(55, 64)
temp = temp + m(65, 56) * m(56, 64)
temp = temp + m(65, 57) * m(57, 64)
temp = temp + m(65, 58) * m(58, 64)
temp = temp + m(65, 59) * m(59, 64)
temp = temp + m(65, 60) * m(60, 64)
temp = temp + m(65, 61) * m(61, 64)
temp = temp + m(65, 62) * m(62, 64)
temp = temp + m(65, 63) * m(63, 64)
m(65, 64) = m(65, 64) - temp
!     Upper part.
temp = m(64, 23) * m(23, 65)
temp = temp + m(64, 44) * m(44, 65)
temp = temp + m(64, 52) * m(52, 65)
temp = temp + m(64, 53) * m(53, 65)
temp = temp + m(64, 54) * m(54, 65)
temp = temp + m(64, 55) * m(55, 65)
temp = temp + m(64, 56) * m(56, 65)
temp = temp + m(64, 57) * m(57, 65)
temp = temp + m(64, 58) * m(58, 65)
temp = temp + m(64, 59) * m(59, 65)
temp = temp + m(64, 60) * m(60, 65)
temp = temp + m(64, 61) * m(61, 65)
temp = temp + m(64, 62) * m(62, 65)
temp = temp + m(64, 63) * m(63, 65)
m(64, 65) = ( m(64, 65) - temp ) / m(64, 64)

!     Lower part.
temp = m(65, 13) * m(13, 65)
temp = temp + m(65, 16) * m(16, 65)
temp = temp + m(65, 17) * m(17, 65)
temp = temp + m(65, 18) * m(18, 65)
temp = temp + m(65, 19) * m(19, 65)
temp = temp + m(65, 23) * m(23, 65)
temp = temp + m(65, 43) * m(43, 65)
temp = temp + m(65, 44) * m(44, 65)
temp = temp + m(65, 46) * m(46, 65)
temp = temp + m(65, 47) * m(47, 65)
temp = temp + m(65, 50) * m(50, 65)
temp = temp + m(65, 52) * m(52, 65)
temp = temp + m(65, 53) * m(53, 65)
temp = temp + m(65, 54) * m(54, 65)
temp = temp + m(65, 55) * m(55, 65)
temp = temp + m(65, 56) * m(56, 65)
temp = temp + m(65, 57) * m(57, 65)
temp = temp + m(65, 58) * m(58, 65)
temp = temp + m(65, 59) * m(59, 65)
temp = temp + m(65, 60) * m(60, 65)
temp = temp + m(65, 61) * m(61, 65)
temp = temp + m(65, 62) * m(62, 65)
temp = temp + m(65, 63) * m(63, 65)
temp = temp + m(65, 64) * m(64, 65)
m(65, 65) = m(65, 65) - temp


return
end subroutine lu_decompose_siream


!===============================================================================

!> lu_solve_siream
!> \brief Resolution of MY=X where M is an LU factorization computed by lu_decompose_3
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 matrix row number from the chemical species number
!> \param[in]     m                  an LU factorization computed by lu_decompose_3
!> \param[in,out] x                  on entry, the right-hand side of the equation.
!                                    on exit, the solution of the equation
!______________________________________________________________________________

subroutine lu_solve_siream (ns, m, x)

implicit none

! Arguments

integer ns
double precision m(ns,ns)
double precision x(ns)

! Local variables

double precision temp

!     Forward substitution.

x(1) = x(1) / m(1, 1)

x(2) = x(2) / m(2, 2)

x(3) = x(3) / m(3, 3)

x(4) = x(4) / m(4, 4)

x(5) = x(5) / m(5, 5)

x(6) = x(6) / m(6, 6)

x(7) = x(7) / m(7, 7)

x(8) = x(8) / m(8, 8)

x(9) = x(9) / m(9, 9)

temp = m(10, 9) * x(9)
x(10) = ( x(10) - temp ) / m(10, 10)

x(11) = x(11) / m(11, 11)

temp = m(12, 11) * x(11)
x(12) = ( x(12) - temp ) / m(12, 12)

x(13) = x(13) / m(13, 13)

x(14) = x(14) / m(14, 14)

temp = m(15, 7) * x(7)
temp = temp + m(15, 8) * x(8)
x(15) = ( x(15) - temp ) / m(15, 15)

x(16) = x(16) / m(16, 16)

x(17) = x(17) / m(17, 17)

x(18) = x(18) / m(18, 18)

x(19) = x(19) / m(19, 19)

x(20) = x(20) / m(20, 20)

x(21) = x(21) / m(21, 21)

temp = m(22, 21) * x(21)
x(22) = ( x(22) - temp ) / m(22, 22)

x(23) = x(23) / m(23, 23)

x(24) = x(24) / m(24, 24)

temp = m(25, 24) * x(24)
x(25) = ( x(25) - temp ) / m(25, 25)

temp = m(26, 24) * x(24)
x(26) = ( x(26) - temp ) / m(26, 26)

x(27) = x(27) / m(27, 27)

temp = m(28, 27) * x(27)
x(28) = ( x(28) - temp ) / m(28, 28)

temp = m(29, 27) * x(27)
x(29) = ( x(29) - temp ) / m(29, 29)

temp = m(30, 7) * x(7)
temp = temp + m(30, 8) * x(8)
x(30) = ( x(30) - temp ) / m(30, 30)

temp = m(31, 7) * x(7)
temp = temp + m(31, 8) * x(8)
x(31) = ( x(31) - temp ) / m(31, 31)

x(32) = x(32) / m(32, 32)

x(33) = x(33) / m(33, 33)

x(34) = x(34) / m(34, 34)

temp = m(35, 21) * x(21)
temp = temp + m(35, 34) * x(34)
x(35) = ( x(35) - temp ) / m(35, 35)

x(36) = x(36) / m(36, 36)

x(37) = x(37) / m(37, 37)

temp = m(38, 36) * x(36)
temp = temp + m(38, 37) * x(37)
x(38) = ( x(38) - temp ) / m(38, 38)

x(39) = x(39) / m(39, 39)

temp = m(40, 7) * x(7)
temp = temp + m(40, 8) * x(8)
temp = temp + m(40, 15) * x(15)
x(40) = ( x(40) - temp ) / m(40, 40)

temp = m(41, 15) * x(15)
temp = temp + m(41, 40) * x(40)
x(41) = ( x(41) - temp ) / m(41, 41)

temp = m(42, 8) * x(8)
temp = temp + m(42, 41) * x(41)
x(42) = ( x(42) - temp ) / m(42, 42)

temp = m(43, 40) * x(40)
x(43) = ( x(43) - temp ) / m(43, 43)

x(44) = x(44) / m(44, 44)

temp = m(45, 44) * x(44)
x(45) = ( x(45) - temp ) / m(45, 45)

temp = m(46, 15) * x(15)
temp = temp + m(46, 19) * x(19)
temp = temp + m(46, 39) * x(39)
temp = temp + m(46, 43) * x(43)
temp = temp + m(46, 44) * x(44)
temp = temp + m(46, 45) * x(45)
x(46) = ( x(46) - temp ) / m(46, 46)

temp = m(47, 8) * x(8)
temp = temp + m(47, 19) * x(19)
temp = temp + m(47, 36) * x(36)
temp = temp + m(47, 37) * x(37)
temp = temp + m(47, 39) * x(39)
temp = temp + m(47, 44) * x(44)
temp = temp + m(47, 45) * x(45)
temp = temp + m(47, 46) * x(46)
x(47) = ( x(47) - temp ) / m(47, 47)

temp = m(48, 4) * x(4)
temp = temp + m(48, 19) * x(19)
temp = temp + m(48, 32) * x(32)
temp = temp + m(48, 34) * x(34)
temp = temp + m(48, 36) * x(36)
temp = temp + m(48, 37) * x(37)
temp = temp + m(48, 39) * x(39)
temp = temp + m(48, 41) * x(41)
temp = temp + m(48, 43) * x(43)
temp = temp + m(48, 44) * x(44)
temp = temp + m(48, 45) * x(45)
temp = temp + m(48, 46) * x(46)
temp = temp + m(48, 47) * x(47)
x(48) = ( x(48) - temp ) / m(48, 48)

x(49) = x(49) / m(49, 49)

temp = m(50, 16) * x(16)
temp = temp + m(50, 40) * x(40)
temp = temp + m(50, 43) * x(43)
temp = temp + m(50, 45) * x(45)
temp = temp + m(50, 46) * x(46)
temp = temp + m(50, 47) * x(47)
temp = temp + m(50, 48) * x(48)
x(50) = ( x(50) - temp ) / m(50, 50)

temp = m(51, 34) * x(34)
temp = temp + m(51, 36) * x(36)
temp = temp + m(51, 37) * x(37)
temp = temp + m(51, 39) * x(39)
temp = temp + m(51, 41) * x(41)
temp = temp + m(51, 42) * x(42)
temp = temp + m(51, 43) * x(43)
temp = temp + m(51, 44) * x(44)
temp = temp + m(51, 45) * x(45)
temp = temp + m(51, 48) * x(48)
x(51) = ( x(51) - temp ) / m(51, 51)

temp = m(52, 6) * x(6)
temp = temp + m(52, 19) * x(19)
temp = temp + m(52, 36) * x(36)
temp = temp + m(52, 39) * x(39)
temp = temp + m(52, 44) * x(44)
temp = temp + m(52, 47) * x(47)
x(52) = ( x(52) - temp ) / m(52, 52)

temp = m(53, 5) * x(5)
temp = temp + m(53, 22) * x(22)
temp = temp + m(53, 33) * x(33)
temp = temp + m(53, 48) * x(48)
temp = temp + m(53, 49) * x(49)
temp = temp + m(53, 52) * x(52)
x(53) = ( x(53) - temp ) / m(53, 53)

temp = m(54, 4) * x(4)
temp = temp + m(54, 14) * x(14)
temp = temp + m(54, 21) * x(21)
temp = temp + m(54, 22) * x(22)
temp = temp + m(54, 34) * x(34)
temp = temp + m(54, 36) * x(36)
temp = temp + m(54, 37) * x(37)
temp = temp + m(54, 39) * x(39)
temp = temp + m(54, 41) * x(41)
temp = temp + m(54, 43) * x(43)
temp = temp + m(54, 44) * x(44)
temp = temp + m(54, 45) * x(45)
temp = temp + m(54, 46) * x(46)
temp = temp + m(54, 47) * x(47)
temp = temp + m(54, 52) * x(52)
temp = temp + m(54, 53) * x(53)
x(54) = ( x(54) - temp ) / m(54, 54)

temp = m(55, 3) * x(3)
temp = temp + m(55, 4) * x(4)
temp = temp + m(55, 5) * x(5)
temp = temp + m(55, 6) * x(6)
temp = temp + m(55, 7) * x(7)
temp = temp + m(55, 8) * x(8)
temp = temp + m(55, 9) * x(9)
temp = temp + m(55, 14) * x(14)
temp = temp + m(55, 17) * x(17)
temp = temp + m(55, 18) * x(18)
temp = temp + m(55, 20) * x(20)
temp = temp + m(55, 22) * x(22)
temp = temp + m(55, 23) * x(23)
temp = temp + m(55, 32) * x(32)
temp = temp + m(55, 33) * x(33)
temp = temp + m(55, 34) * x(34)
temp = temp + m(55, 35) * x(35)
temp = temp + m(55, 36) * x(36)
temp = temp + m(55, 37) * x(37)
temp = temp + m(55, 39) * x(39)
temp = temp + m(55, 40) * x(40)
temp = temp + m(55, 41) * x(41)
temp = temp + m(55, 42) * x(42)
temp = temp + m(55, 43) * x(43)
temp = temp + m(55, 44) * x(44)
temp = temp + m(55, 45) * x(45)
temp = temp + m(55, 46) * x(46)
temp = temp + m(55, 47) * x(47)
temp = temp + m(55, 48) * x(48)
temp = temp + m(55, 49) * x(49)
temp = temp + m(55, 50) * x(50)
temp = temp + m(55, 51) * x(51)
temp = temp + m(55, 52) * x(52)
temp = temp + m(55, 53) * x(53)
temp = temp + m(55, 54) * x(54)
x(55) = ( x(55) - temp ) / m(55, 55)

temp = m(56, 16) * x(16)
temp = temp + m(56, 17) * x(17)
temp = temp + m(56, 34) * x(34)
temp = temp + m(56, 36) * x(36)
temp = temp + m(56, 37) * x(37)
temp = temp + m(56, 39) * x(39)
temp = temp + m(56, 40) * x(40)
temp = temp + m(56, 43) * x(43)
temp = temp + m(56, 44) * x(44)
temp = temp + m(56, 45) * x(45)
temp = temp + m(56, 48) * x(48)
temp = temp + m(56, 50) * x(50)
temp = temp + m(56, 52) * x(52)
temp = temp + m(56, 54) * x(54)
temp = temp + m(56, 55) * x(55)
x(56) = ( x(56) - temp ) / m(56, 56)

temp = m(57, 3) * x(3)
temp = temp + m(57, 20) * x(20)
temp = temp + m(57, 34) * x(34)
temp = temp + m(57, 36) * x(36)
temp = temp + m(57, 37) * x(37)
temp = temp + m(57, 39) * x(39)
temp = temp + m(57, 44) * x(44)
temp = temp + m(57, 48) * x(48)
temp = temp + m(57, 52) * x(52)
temp = temp + m(57, 54) * x(54)
temp = temp + m(57, 55) * x(55)
temp = temp + m(57, 56) * x(56)
x(57) = ( x(57) - temp ) / m(57, 57)

temp = m(58, 18) * x(18)
temp = temp + m(58, 39) * x(39)
temp = temp + m(58, 44) * x(44)
temp = temp + m(58, 45) * x(45)
temp = temp + m(58, 48) * x(48)
temp = temp + m(58, 52) * x(52)
temp = temp + m(58, 55) * x(55)
temp = temp + m(58, 56) * x(56)
temp = temp + m(58, 57) * x(57)
x(58) = ( x(58) - temp ) / m(58, 58)

temp = m(59, 34) * x(34)
temp = temp + m(59, 36) * x(36)
temp = temp + m(59, 37) * x(37)
temp = temp + m(59, 39) * x(39)
temp = temp + m(59, 41) * x(41)
temp = temp + m(59, 43) * x(43)
temp = temp + m(59, 44) * x(44)
temp = temp + m(59, 45) * x(45)
temp = temp + m(59, 55) * x(55)
temp = temp + m(59, 56) * x(56)
temp = temp + m(59, 57) * x(57)
temp = temp + m(59, 58) * x(58)
x(59) = ( x(59) - temp ) / m(59, 59)

temp = m(60, 4) * x(4)
temp = temp + m(60, 6) * x(6)
temp = temp + m(60, 18) * x(18)
temp = temp + m(60, 19) * x(19)
temp = temp + m(60, 32) * x(32)
temp = temp + m(60, 36) * x(36)
temp = temp + m(60, 37) * x(37)
temp = temp + m(60, 45) * x(45)
temp = temp + m(60, 46) * x(46)
temp = temp + m(60, 47) * x(47)
temp = temp + m(60, 52) * x(52)
temp = temp + m(60, 53) * x(53)
temp = temp + m(60, 54) * x(54)
temp = temp + m(60, 55) * x(55)
temp = temp + m(60, 56) * x(56)
temp = temp + m(60, 57) * x(57)
temp = temp + m(60, 58) * x(58)
temp = temp + m(60, 59) * x(59)
x(60) = ( x(60) - temp ) / m(60, 60)

temp = m(61, 3) * x(3)
temp = temp + m(61, 4) * x(4)
temp = temp + m(61, 6) * x(6)
temp = temp + m(61, 7) * x(7)
temp = temp + m(61, 8) * x(8)
temp = temp + m(61, 9) * x(9)
temp = temp + m(61, 14) * x(14)
temp = temp + m(61, 15) * x(15)
temp = temp + m(61, 17) * x(17)
temp = temp + m(61, 19) * x(19)
temp = temp + m(61, 20) * x(20)
temp = temp + m(61, 21) * x(21)
temp = temp + m(61, 22) * x(22)
temp = temp + m(61, 32) * x(32)
temp = temp + m(61, 34) * x(34)
temp = temp + m(61, 35) * x(35)
temp = temp + m(61, 36) * x(36)
temp = temp + m(61, 37) * x(37)
temp = temp + m(61, 39) * x(39)
temp = temp + m(61, 40) * x(40)
temp = temp + m(61, 41) * x(41)
temp = temp + m(61, 42) * x(42)
temp = temp + m(61, 43) * x(43)
temp = temp + m(61, 44) * x(44)
temp = temp + m(61, 45) * x(45)
temp = temp + m(61, 46) * x(46)
temp = temp + m(61, 47) * x(47)
temp = temp + m(61, 48) * x(48)
temp = temp + m(61, 51) * x(51)
temp = temp + m(61, 52) * x(52)
temp = temp + m(61, 53) * x(53)
temp = temp + m(61, 54) * x(54)
temp = temp + m(61, 55) * x(55)
temp = temp + m(61, 56) * x(56)
temp = temp + m(61, 57) * x(57)
temp = temp + m(61, 58) * x(58)
temp = temp + m(61, 59) * x(59)
temp = temp + m(61, 60) * x(60)
x(61) = ( x(61) - temp ) / m(61, 61)

temp = m(62, 13) * x(13)
temp = temp + m(62, 33) * x(33)
temp = temp + m(62, 41) * x(41)
temp = temp + m(62, 42) * x(42)
temp = temp + m(62, 43) * x(43)
temp = temp + m(62, 45) * x(45)
temp = temp + m(62, 53) * x(53)
temp = temp + m(62, 54) * x(54)
temp = temp + m(62, 55) * x(55)
temp = temp + m(62, 56) * x(56)
temp = temp + m(62, 57) * x(57)
temp = temp + m(62, 58) * x(58)
temp = temp + m(62, 59) * x(59)
temp = temp + m(62, 60) * x(60)
temp = temp + m(62, 61) * x(61)
x(62) = ( x(62) - temp ) / m(62, 62)

temp = m(63, 4) * x(4)
temp = temp + m(63, 6) * x(6)
temp = temp + m(63, 7) * x(7)
temp = temp + m(63, 8) * x(8)
temp = temp + m(63, 19) * x(19)
temp = temp + m(63, 22) * x(22)
temp = temp + m(63, 32) * x(32)
temp = temp + m(63, 34) * x(34)
temp = temp + m(63, 36) * x(36)
temp = temp + m(63, 37) * x(37)
temp = temp + m(63, 39) * x(39)
temp = temp + m(63, 40) * x(40)
temp = temp + m(63, 41) * x(41)
temp = temp + m(63, 42) * x(42)
temp = temp + m(63, 43) * x(43)
temp = temp + m(63, 44) * x(44)
temp = temp + m(63, 45) * x(45)
temp = temp + m(63, 47) * x(47)
temp = temp + m(63, 52) * x(52)
temp = temp + m(63, 53) * x(53)
temp = temp + m(63, 54) * x(54)
temp = temp + m(63, 55) * x(55)
temp = temp + m(63, 56) * x(56)
temp = temp + m(63, 57) * x(57)
temp = temp + m(63, 58) * x(58)
temp = temp + m(63, 59) * x(59)
temp = temp + m(63, 60) * x(60)
temp = temp + m(63, 61) * x(61)
temp = temp + m(63, 62) * x(62)
x(63) = ( x(63) - temp ) / m(63, 63)

temp = m(64, 15) * x(15)
temp = temp + m(64, 21) * x(21)
temp = temp + m(64, 23) * x(23)
temp = temp + m(64, 44) * x(44)
temp = temp + m(64, 52) * x(52)
temp = temp + m(64, 53) * x(53)
temp = temp + m(64, 54) * x(54)
temp = temp + m(64, 55) * x(55)
temp = temp + m(64, 56) * x(56)
temp = temp + m(64, 57) * x(57)
temp = temp + m(64, 58) * x(58)
temp = temp + m(64, 59) * x(59)
temp = temp + m(64, 60) * x(60)
temp = temp + m(64, 61) * x(61)
temp = temp + m(64, 62) * x(62)
temp = temp + m(64, 63) * x(63)
x(64) = ( x(64) - temp ) / m(64, 64)

temp = m(65, 13) * x(13)
temp = temp + m(65, 15) * x(15)
temp = temp + m(65, 16) * x(16)
temp = temp + m(65, 17) * x(17)
temp = temp + m(65, 18) * x(18)
temp = temp + m(65, 19) * x(19)
temp = temp + m(65, 21) * x(21)
temp = temp + m(65, 23) * x(23)
temp = temp + m(65, 34) * x(34)
temp = temp + m(65, 36) * x(36)
temp = temp + m(65, 37) * x(37)
temp = temp + m(65, 39) * x(39)
temp = temp + m(65, 43) * x(43)
temp = temp + m(65, 44) * x(44)
temp = temp + m(65, 46) * x(46)
temp = temp + m(65, 47) * x(47)
temp = temp + m(65, 50) * x(50)
temp = temp + m(65, 52) * x(52)
temp = temp + m(65, 53) * x(53)
temp = temp + m(65, 54) * x(54)
temp = temp + m(65, 55) * x(55)
temp = temp + m(65, 56) * x(56)
temp = temp + m(65, 57) * x(57)
temp = temp + m(65, 58) * x(58)
temp = temp + m(65, 59) * x(59)
temp = temp + m(65, 60) * x(60)
temp = temp + m(65, 61) * x(61)
temp = temp + m(65, 62) * x(62)
temp = temp + m(65, 63) * x(63)
temp = temp + m(65, 64) * x(64)
x(65) = ( x(65) - temp ) / m(65, 65)


!     Backward substitution.

temp = m(64, 65) * x(65)
x(64) = x(64) - temp

temp = m(63, 64) * x(64)
temp = temp + m(63, 65) * x(65)
x(63) = x(63) - temp

temp = m(62, 63) * x(63)
temp = temp + m(62, 64) * x(64)
temp = temp + m(62, 65) * x(65)
x(62) = x(62) - temp

temp = m(61, 62) * x(62)
temp = temp + m(61, 63) * x(63)
temp = temp + m(61, 64) * x(64)
temp = temp + m(61, 65) * x(65)
x(61) = x(61) - temp

temp = m(60, 61) * x(61)
temp = temp + m(60, 62) * x(62)
temp = temp + m(60, 63) * x(63)
temp = temp + m(60, 64) * x(64)
temp = temp + m(60, 65) * x(65)
x(60) = x(60) - temp

temp = m(59, 60) * x(60)
temp = temp + m(59, 61) * x(61)
temp = temp + m(59, 62) * x(62)
temp = temp + m(59, 63) * x(63)
temp = temp + m(59, 64) * x(64)
temp = temp + m(59, 65) * x(65)
x(59) = x(59) - temp

temp = m(58, 59) * x(59)
temp = temp + m(58, 60) * x(60)
temp = temp + m(58, 61) * x(61)
temp = temp + m(58, 62) * x(62)
temp = temp + m(58, 63) * x(63)
temp = temp + m(58, 64) * x(64)
temp = temp + m(58, 65) * x(65)
x(58) = x(58) - temp

temp = m(57, 58) * x(58)
temp = temp + m(57, 59) * x(59)
temp = temp + m(57, 60) * x(60)
temp = temp + m(57, 61) * x(61)
temp = temp + m(57, 62) * x(62)
temp = temp + m(57, 63) * x(63)
temp = temp + m(57, 64) * x(64)
temp = temp + m(57, 65) * x(65)
x(57) = x(57) - temp

temp = m(56, 57) * x(57)
temp = temp + m(56, 58) * x(58)
temp = temp + m(56, 59) * x(59)
temp = temp + m(56, 60) * x(60)
temp = temp + m(56, 61) * x(61)
temp = temp + m(56, 62) * x(62)
temp = temp + m(56, 63) * x(63)
temp = temp + m(56, 64) * x(64)
temp = temp + m(56, 65) * x(65)
x(56) = x(56) - temp

temp = m(55, 56) * x(56)
temp = temp + m(55, 57) * x(57)
temp = temp + m(55, 58) * x(58)
temp = temp + m(55, 59) * x(59)
temp = temp + m(55, 60) * x(60)
temp = temp + m(55, 61) * x(61)
temp = temp + m(55, 62) * x(62)
temp = temp + m(55, 63) * x(63)
temp = temp + m(55, 64) * x(64)
temp = temp + m(55, 65) * x(65)
x(55) = x(55) - temp

temp = m(54, 55) * x(55)
temp = temp + m(54, 56) * x(56)
temp = temp + m(54, 57) * x(57)
temp = temp + m(54, 58) * x(58)
temp = temp + m(54, 59) * x(59)
temp = temp + m(54, 60) * x(60)
temp = temp + m(54, 61) * x(61)
temp = temp + m(54, 62) * x(62)
temp = temp + m(54, 63) * x(63)
temp = temp + m(54, 64) * x(64)
temp = temp + m(54, 65) * x(65)
x(54) = x(54) - temp

temp = m(53, 54) * x(54)
temp = temp + m(53, 55) * x(55)
temp = temp + m(53, 56) * x(56)
temp = temp + m(53, 57) * x(57)
temp = temp + m(53, 58) * x(58)
temp = temp + m(53, 59) * x(59)
temp = temp + m(53, 60) * x(60)
temp = temp + m(53, 61) * x(61)
temp = temp + m(53, 62) * x(62)
temp = temp + m(53, 63) * x(63)
temp = temp + m(53, 64) * x(64)
temp = temp + m(53, 65) * x(65)
x(53) = x(53) - temp

temp = m(52, 55) * x(55)
temp = temp + m(52, 56) * x(56)
temp = temp + m(52, 57) * x(57)
temp = temp + m(52, 59) * x(59)
temp = temp + m(52, 61) * x(61)
temp = temp + m(52, 63) * x(63)
temp = temp + m(52, 64) * x(64)
temp = temp + m(52, 65) * x(65)
x(52) = x(52) - temp

temp = m(51, 52) * x(52)
temp = temp + m(51, 54) * x(54)
temp = temp + m(51, 55) * x(55)
temp = temp + m(51, 56) * x(56)
temp = temp + m(51, 57) * x(57)
temp = temp + m(51, 59) * x(59)
temp = temp + m(51, 60) * x(60)
temp = temp + m(51, 61) * x(61)
temp = temp + m(51, 63) * x(63)
temp = temp + m(51, 64) * x(64)
temp = temp + m(51, 65) * x(65)
x(51) = x(51) - temp

temp = m(50, 52) * x(52)
temp = temp + m(50, 54) * x(54)
temp = temp + m(50, 55) * x(55)
temp = temp + m(50, 56) * x(56)
temp = temp + m(50, 57) * x(57)
temp = temp + m(50, 59) * x(59)
temp = temp + m(50, 60) * x(60)
temp = temp + m(50, 61) * x(61)
temp = temp + m(50, 63) * x(63)
temp = temp + m(50, 64) * x(64)
temp = temp + m(50, 65) * x(65)
x(50) = x(50) - temp

temp = m(49, 53) * x(53)
temp = temp + m(49, 55) * x(55)
temp = temp + m(49, 58) * x(58)
temp = temp + m(49, 61) * x(61)
temp = temp + m(49, 62) * x(62)
temp = temp + m(49, 63) * x(63)
x(49) = x(49) - temp

temp = m(48, 52) * x(52)
temp = temp + m(48, 55) * x(55)
temp = temp + m(48, 56) * x(56)
temp = temp + m(48, 57) * x(57)
temp = temp + m(48, 59) * x(59)
temp = temp + m(48, 61) * x(61)
temp = temp + m(48, 63) * x(63)
temp = temp + m(48, 64) * x(64)
temp = temp + m(48, 65) * x(65)
x(48) = x(48) - temp

temp = m(47, 52) * x(52)
temp = temp + m(47, 55) * x(55)
temp = temp + m(47, 56) * x(56)
temp = temp + m(47, 57) * x(57)
temp = temp + m(47, 59) * x(59)
temp = temp + m(47, 61) * x(61)
temp = temp + m(47, 64) * x(64)
temp = temp + m(47, 65) * x(65)
x(47) = x(47) - temp

temp = m(46, 47) * x(47)
temp = temp + m(46, 52) * x(52)
temp = temp + m(46, 55) * x(55)
temp = temp + m(46, 56) * x(56)
temp = temp + m(46, 57) * x(57)
temp = temp + m(46, 59) * x(59)
temp = temp + m(46, 61) * x(61)
temp = temp + m(46, 64) * x(64)
temp = temp + m(46, 65) * x(65)
x(46) = x(46) - temp

temp = m(45, 55) * x(55)
temp = temp + m(45, 56) * x(56)
temp = temp + m(45, 57) * x(57)
temp = temp + m(45, 59) * x(59)
temp = temp + m(45, 65) * x(65)
x(45) = x(45) - temp

temp = m(44, 55) * x(55)
temp = temp + m(44, 56) * x(56)
temp = temp + m(44, 57) * x(57)
temp = temp + m(44, 59) * x(59)
temp = temp + m(44, 65) * x(65)
x(44) = x(44) - temp

temp = m(43, 55) * x(55)
temp = temp + m(43, 56) * x(56)
temp = temp + m(43, 61) * x(61)
temp = temp + m(43, 64) * x(64)
temp = temp + m(43, 65) * x(65)
x(43) = x(43) - temp

temp = m(42, 43) * x(43)
temp = temp + m(42, 45) * x(45)
temp = temp + m(42, 55) * x(55)
temp = temp + m(42, 56) * x(56)
temp = temp + m(42, 59) * x(59)
temp = temp + m(42, 61) * x(61)
temp = temp + m(42, 64) * x(64)
x(42) = x(42) - temp

temp = m(41, 43) * x(43)
temp = temp + m(41, 55) * x(55)
temp = temp + m(41, 56) * x(56)
temp = temp + m(41, 59) * x(59)
temp = temp + m(41, 61) * x(61)
temp = temp + m(41, 64) * x(64)
x(41) = x(41) - temp

temp = m(40, 43) * x(43)
temp = temp + m(40, 55) * x(55)
temp = temp + m(40, 56) * x(56)
temp = temp + m(40, 61) * x(61)
temp = temp + m(40, 64) * x(64)
x(40) = x(40) - temp

temp = m(39, 55) * x(55)
temp = temp + m(39, 56) * x(56)
temp = temp + m(39, 57) * x(57)
temp = temp + m(39, 59) * x(59)
x(39) = x(39) - temp

temp = m(38, 55) * x(55)
temp = temp + m(38, 56) * x(56)
temp = temp + m(38, 57) * x(57)
temp = temp + m(38, 59) * x(59)
x(38) = x(38) - temp

temp = m(37, 55) * x(55)
temp = temp + m(37, 56) * x(56)
temp = temp + m(37, 57) * x(57)
temp = temp + m(37, 59) * x(59)
x(37) = x(37) - temp

temp = m(36, 55) * x(55)
temp = temp + m(36, 56) * x(56)
temp = temp + m(36, 57) * x(57)
temp = temp + m(36, 59) * x(59)
x(36) = x(36) - temp

temp = m(35, 54) * x(54)
temp = temp + m(35, 55) * x(55)
temp = temp + m(35, 56) * x(56)
temp = temp + m(35, 57) * x(57)
temp = temp + m(35, 59) * x(59)
temp = temp + m(35, 61) * x(61)
temp = temp + m(35, 64) * x(64)
x(35) = x(35) - temp

temp = m(34, 55) * x(55)
temp = temp + m(34, 56) * x(56)
temp = temp + m(34, 57) * x(57)
temp = temp + m(34, 59) * x(59)
x(34) = x(34) - temp

temp = m(33, 55) * x(55)
temp = temp + m(33, 58) * x(58)
temp = temp + m(33, 61) * x(61)
temp = temp + m(33, 62) * x(62)
x(33) = x(33) - temp

temp = m(32, 52) * x(52)
temp = temp + m(32, 55) * x(55)
temp = temp + m(32, 61) * x(61)
temp = temp + m(32, 63) * x(63)
x(32) = x(32) - temp

temp = m(31, 40) * x(40)
temp = temp + m(31, 55) * x(55)
temp = temp + m(31, 56) * x(56)
x(31) = x(31) - temp

temp = m(30, 40) * x(40)
temp = temp + m(30, 55) * x(55)
temp = temp + m(30, 56) * x(56)
x(30) = x(30) - temp

temp = m(29, 55) * x(55)
temp = temp + m(29, 56) * x(56)
temp = temp + m(29, 59) * x(59)
x(29) = x(29) - temp

temp = m(28, 55) * x(55)
temp = temp + m(28, 56) * x(56)
temp = temp + m(28, 59) * x(59)
x(28) = x(28) - temp

temp = m(27, 55) * x(55)
temp = temp + m(27, 56) * x(56)
temp = temp + m(27, 59) * x(59)
x(27) = x(27) - temp

temp = m(26, 55) * x(55)
temp = temp + m(26, 56) * x(56)
temp = temp + m(26, 59) * x(59)
x(26) = x(26) - temp

temp = m(25, 55) * x(55)
temp = temp + m(25, 56) * x(56)
temp = temp + m(25, 59) * x(59)
x(25) = x(25) - temp

temp = m(24, 55) * x(55)
temp = temp + m(24, 56) * x(56)
temp = temp + m(24, 59) * x(59)
x(24) = x(24) - temp

temp = m(23, 55) * x(55)
temp = temp + m(23, 64) * x(64)
temp = temp + m(23, 65) * x(65)
x(23) = x(23) - temp

temp = m(22, 53) * x(53)
temp = temp + m(22, 54) * x(54)
temp = temp + m(22, 55) * x(55)
temp = temp + m(22, 61) * x(61)
temp = temp + m(22, 64) * x(64)
x(22) = x(22) - temp

temp = m(21, 54) * x(54)
temp = temp + m(21, 61) * x(61)
temp = temp + m(21, 64) * x(64)
x(21) = x(21) - temp

temp = m(20, 55) * x(55)
temp = temp + m(20, 57) * x(57)
temp = temp + m(20, 61) * x(61)
x(20) = x(20) - temp

temp = m(19, 47) * x(47)
temp = temp + m(19, 55) * x(55)
temp = temp + m(19, 65) * x(65)
x(19) = x(19) - temp

temp = m(18, 55) * x(55)
temp = temp + m(18, 58) * x(58)
temp = temp + m(18, 65) * x(65)
x(18) = x(18) - temp

temp = m(17, 55) * x(55)
temp = temp + m(17, 61) * x(61)
temp = temp + m(17, 65) * x(65)
x(17) = x(17) - temp

temp = m(16, 56) * x(56)
temp = temp + m(16, 65) * x(65)
x(16) = x(16) - temp

temp = m(15, 55) * x(55)
temp = temp + m(15, 64) * x(64)
x(15) = x(15) - temp

temp = m(14, 53) * x(53)
temp = temp + m(14, 55) * x(55)
x(14) = x(14) - temp

temp = m(13, 62) * x(62)
temp = temp + m(13, 65) * x(65)
x(13) = x(13) - temp

temp = m(12, 55) * x(55)
x(12) = x(12) - temp

temp = m(11, 55) * x(55)
x(11) = x(11) - temp

temp = m(10, 55) * x(55)
x(10) = x(10) - temp

temp = m(9, 55) * x(55)
x(9) = x(9) - temp

temp = m(8, 55) * x(55)
x(8) = x(8) - temp

temp = m(7, 55) * x(55)
x(7) = x(7) - temp

temp = m(6, 55) * x(55)
x(6) = x(6) - temp

temp = m(5, 55) * x(55)
x(5) = x(5) - temp

temp = m(4, 55) * x(55)
x(4) = x(4) - temp

temp = m(3, 59) * x(59)
x(3) = x(3) - temp


return
end subroutine lu_solve_siream

