# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gobject, dbus

from telepathy.constants import CONNECTION_HANDLE_TYPE_CONTACT
from telepathy.interfaces import CONN_INTERFACE, CHANNEL_INTERFACE, CHANNEL_INTERFACE_GROUP

from cohoba.common.Utils import global_error_handler, SignalConnection
from cohoba.common.TelepathyGroupMember import TelepathyGroupMember, GROUP_MEMBER_FLAG_LOCAL, GROUP_MEMBER_FLAG_REMOTE, GROUP_MEMBER_FLAG_MEMBER
from cohoba.common.DBusProxyProvider import get_proxy_provider

class TelepathyGroup:
	"""Wrapper for a Telepathy Channel implementing the Group interface."""

	def __init__(self, conn_path, obj_path, handle):
		"""Constructor.

		:Parameters:
			conn_path : string
				The DBus object path of the connection
			obj_path : string
				The DBus object path of the Channel
				represented by this TelepathyGroup
			handle : integer
				Handle representing the user. FIXME:
				why is this passed to the constructor?
				We auto-discover it anyway.
		"""
		self.handle = handle
		self.flags = 0

		self.members = {}
		"""Keys are integer handles; values are arbitrary data
		of a type determined by subclasses. In this implementation,
		values are the same integer handle."""

		self.member_list = {}
		"""Keys are integer handles; values are
		TelepathyGroupMember instances. Subclasses don't
		necessarily use this dict."""

		self.conn_path = conn_path
		self.conn_iface = get_proxy_provider().get_iface(conn_path, CONN_INTERFACE)
		
		get_proxy_provider().create_proxy(get_proxy_provider().get_service(conn_path), obj_path)
		
		self.chan_iface = get_proxy_provider().get_iface(obj_path, CHANNEL_INTERFACE)
		self.__Closed_conn = SignalConnection(self.chan_iface, 'Closed', self.on_channel_closed)
		
		self.member_list_iface = get_proxy_provider().get_iface(obj_path, CHANNEL_INTERFACE_GROUP)
		self.__MembersChanged_conn = SignalConnection(self.member_list_iface, 'MembersChanged', self.on_members_changed)
		self.__GroupFlagsChanged_conn = SignalConnection(self.member_list_iface, 'GroupFlagsChanged', self.on_flags_changed)
		
		self.member_list_iface.GetAllMembers(reply_handler=self.on_got_members, error_handler=global_error_handler)
		self.member_list_iface.GetGroupFlags(reply_handler=self.on_got_flags, error_handler=global_error_handler)
		self.member_list_iface.GetSelfHandle(reply_handler=self.on_got_self_handle, error_handler=global_error_handler)
		
	
	# --------External methods----------------
	def add_member(self, handle_or_name, reason="No Reason."):
		"""Request that the given contact is added to this Group.

		:Parameters:
			handle_or_name : string or integer
				The name (e.g. JID) or handle of the contact
				to be added
			reason : string
				Reason to be passed to Telepathy
				(e.g. for a subscription request)
		"""
		handle = handle_or_name
		if type(handle_or_name) in (str, unicode):
			handles = self.conn_iface.RequestHandles(CONNECTION_HANDLE_TYPE_CONTACT, [handle_or_name])
			assert len(handles) == 1
			handle = handles[0]

		print 'Got new handle:', handle
		self.member_list_iface.AddMembers(dbus.Array([handle],signature='u'), reason, reply_handler=lambda: None, error_handler=global_error_handler)
		return handle
	
	def remove_member(self, handle, reason="No Reason."):
		"""Request that the given contact handle is removed from
		this Group.

		:Parameters:
			handle : integer
				The handle of a contact currently in this
				Group
			reason : string
				Reason to be passed to Telepathy
		"""
		print 'Remove member sent to dbus'
		self.member_list_iface.RemoveMembers(dbus.Array([handle],signature='u'), reason, reply_handler=lambda: None, error_handler=global_error_handler)
	
	def get_members(self):
		"""Return a list of TelepathyGroupMember instances
		corresponding to the contact handles which are
		members of this Group.
		"""
		return self.__get_members_with_flag("is_member")
		
	def get_local_members(self):
		"""Return a list of TelepathyGroupMember instances
		corresponding to the contact handles which are
		locally-pending in this Group.
		"""
		return self.__get_members_with_flag("is_local")
		
	def get_remote_members(self):
		"""Return a list of TelepathyGroupMember instances
		corresponding to the contact handles which are
		remotely-pending in this Group.
		"""
		return self.__get_members_with_flag("is_remote")
	#--------------------------------------------------
	
	def __get_members_with_flag(self, flag):
		"""Return a list of TelepathyGroupMember instances
		corresponding to contact handles in this Group.
		Filter the returned objects to only include those for which
		the method named `flag` returns a true value.
		"""
		contacts = [self._member_for_data(self.members[handle]) for handle in self.members.keys()]
		return [contact for contact in contacts if getattr(contact, flag)()]
		
	def __add_members(self, members, flag=GROUP_MEMBER_FLAG_MEMBER):
		"""Update internal data structures to note that the given
		contact handles have been added to the members,
		local-pending or remote-pending lists, as appropriate for
		the value of flag.

		:Parameters:
			members : list of integers
				List of handles to be added
			flag : integer
				One of GROUP_MEMBER_FLAG_MEMBER etc.
		"""
		for handle in members:
			if handle in self.members:
				member = self._member_for_data(self.members[handle])
				member.set_flag(flag)
			else:
				self.members[handle] = self._add_member(handle, flag)
			
	def __remove_members(self, members):
		"""Remove the handles listed in `members` from internal
		data structures.
		"""
		for handle in members:
			if handle not in self.members:
				continue
			
			self._remove_member(self.members[handle])
			del self.members[handle]
			
	def on_members_changed(self, reason, added, removed, local_pending, remote_pending, actor, reason_code):
		"""Callback for MembersChanged signal on Group interface,
		connected by c'tor.
		"""
		print 'Members changed:', reason, added, removed, local_pending, remote_pending, actor, reason_code
		self.__add_members(local_pending, GROUP_MEMBER_FLAG_LOCAL)		
		self.__add_members(remote_pending, GROUP_MEMBER_FLAG_REMOTE)
		self.__add_members(added)
		self.__remove_members(removed)
		self._members_changed()
		
	def on_got_members(self, members, local_pending, remote_pending):
		"""Callback for result of initial GetAllMembers call
		on Group interface of Channel, made in c'tor.
		"""
		print 'Got Members:', members, local_pending, remote_pending
		self.__add_members(local_pending, GROUP_MEMBER_FLAG_LOCAL)
		self.__add_members(remote_pending, GROUP_MEMBER_FLAG_REMOTE)
		self.__add_members(members)
		self._members_changed()
	
	def on_got_flags(self, flags):
		"""Callback for result of initial GetGroupFlags call
		on Group interface of Channel, made in c'tor.
		"""
		print 'Got new group flags:', flags
		self.flags = flags
		self._flags_changed()
		
	def on_flags_changed(self, added, removed):
		"""Callback for FlagsChanged signal on Group interface of
		Channel, set up by c'tor.

		Updates self.flags and calls _flags_changed.
		"""
		print 'Got new group flags: now (0x%x | 0x%x) & ~0x%x = 0x%x', self.flags, added, removed, (self.flags|added) & ~removed
		self.flags = (self.flags|added) & ~removed
		self._flags_changed()
	
	def on_got_self_handle(self, handle):
		"""Callback for return from GetSelfHandle() on Group
		interface of Channel.
		
		Set self.handle to the returned handle (overriding the one
		set by the c'tor) and call _members_changed().
		"""
		self.handle = handle
		self._members_changed()
		
	def on_channel_closed(self):
		"""Callback for Closed signal on channel, set up by c'tor.

		Remove all members from internal data structure, then call
		_channel_closed().
		"""
		print 'Channel Closed'
		self.__remove_members(self.members.keys())
		self._channel_closed()
	
	def _member_for_data(self, handle):
		"""Given an object of whatever type was returned from
		_add_member, return a corresponding TelepathyGroupMember.

		This implementation returns an item from the `member_list`.
		"""
		return self.member_list[handle]
				
	def _add_member(self, handle, flag):
		"""Return some arbitrary piece of data to be stored in
		the `members` dict as a result of adding the given handle
		as a member.

		This implementation creates a TelepathyGroupMember for
		the given handle, sets flag on it, and stores it in
		the `member_list` dict, returning the handle (again) to be
		stored in `members`.
		"""
		member = TelepathyGroupMember(handle)
		member.set_flag(flag)
		self.member_list[handle] = member
		return handle
		
	def _remove_member(self, handle):
		"""Undo the effect of `_add_member`. The parameter is of
		whatever type was returned from _add_member.

		This implementation deletes the TelepathyGroupMember from
		the `member_list`.
		"""
		del self.member_list[handle]
	
	def _channel_closed(self):
		"""Respond to the channel closing. Subclasses may override
		this method.
		"""
		pass
	
	def _members_changed(self):
		"""Respond to a change of members. Subclasses may override
		this method.
		"""
		pass
		
	def _flags_changed(self):
		"""Respond to a change of flags. Subclasses may override
		this method.
		"""
		pass
