# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk, gobject, cohoba, pango
from os.path import join
from gettext import gettext as _

from telepathy.constants import CONNECTION_HANDLE_TYPE_CONTACT

from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.contacts.MergedContactList import get_contact_list, MergedContactList
from cohoba.contacts.Contact import Contact
from cohoba.contacts.ContactGroup import (ContactGroup, get_group_filter,
		FavoriteContactGroup)
from cohoba.common.ui.AccountComboBox import AccountComboBox

class ContactListView(gtk.TreeView):
	def __init__(self, group=None):
		if group == None:
			group = FavoriteContactGroup()
		
		gtk.TreeView.__init__(self, get_group_filter(group, get_contact_list()))
		
		props = {
			'headers-visible': False,
			"enable-search": False,
			"rules-hint": True,
		}
		for prop, val in props.items():
			self.set_property(prop, val)
			
		cell = gtk.CellRendererPixbuf()
		col = gtk.TreeViewColumn(_("Avatar"), cell)
		col.set_cell_data_func(cell, self.get_contact_icon)
		self.append_column(col)
		
		cell = gtk.CellRendererText ()
		col = gtk.TreeViewColumn(_("Description"), cell)
		cell.set_property("ellipsize", pango.ELLIPSIZE_END)
		cell.set_property("width-chars", 40)
		col.set_cell_data_func(cell, self.get_contact_description)
		self.append_column(col)
		
		self.connect("row-activated", self.on_contact_activated)
		self.connect('button-press-event', self.on_button_press)
	
	def get_contact_for_iter(self, iter):
		child_iter = self.get_model().convert_iter_to_child_iter()
		return get_contact_list()[child_iter]
	
	def get_contact_for_path(self, path):
		child_path = self.get_model().convert_path_to_child_path(path)
		return get_contact_list()[child_path]
		
	def on_button_press(self, view, event):
		if event.button != 3:
			return
			
		path_at_pos = self.get_path_at_pos(int(event.x), int(event.y))
		menu = gtk.Menu()

		if path_at_pos is not None:
			path, column, x, y = path_at_pos
			contact = self.get_contact_for_path(path)
			if not contact.is_group:
				item = gtk.MenuItem(_("Chat"))
				item.connect("activate", lambda item: self.on_contact_activated(view, path))
				menu.append(item)
				
				item = gtk.MenuItem(_("Remove"))
				item.connect("activate", self.on_remove_contact, contact)
				menu.append(item)
				
				item = gtk.MenuItem(_("Groups..."))
				item.connect("activate", self.on_contact_groups, contact)
				menu.append(item)
			else:
				item = gtk.MenuItem(_("Edit..."))
				item.connect("activate", self.on_add_contact_group, contact)
				menu.append(item)
				
				item = gtk.MenuItem(_("Remove"))
				item.connect("activate", self.on_remove_contact_group, path)
				menu.append(item)
			menu.append(gtk.SeparatorMenuItem())

		# FIXME: put these in a top-level menu instead
		item = gtk.MenuItem(_("Add..."))
		item.connect("activate", self.on_add_contact)
		menu.append(item)

		item = gtk.MenuItem(_("Add Group..."))
		item.connect("activate", self.on_add_contact_group)
		menu.append(item)
				
		menu.show_all()
		menu.popup(None, None, None, event.button, event.time)
	
	def on_remove_contact(self, item, contact):
		get_contact_list().remove_member(contact.conn_path, contact.handle)
	
	def on_add_contact(self, item):
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "main.glade"), "add_contact_dialog")
		dialog = glade.get_widget("add_contact_dialog")
		entry = glade.get_widget("name")
		
		accounts = AccountComboBox()
		#accounts.set_active(0)
		accounts.show()
		
		glade.get_widget("account_table").attach(accounts, 1, 2, 0, 1, xoptions=gtk.FILL, yoptions=gtk.FILL)
		
		response = dialog.run()
		dialog.destroy()
		if response == gtk.RESPONSE_OK and entry.get_text() != "":
			account = accounts.get_active_account()
			if account != None:
				get_contact_list().add_member(account.conn_obj, entry.get_text())
	
	def on_contact_groups(self, item, contact):
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "main.glade"), "edit_contact_groups_dialog")
		dialog = glade.get_widget("edit_contact_groups_dialog")
		entry = glade.get_widget("groups")
		entry.set_text(' '.join(contact.groups))
		
		favorite = glade.get_widget("favorite")
		favorite.set_active(contact.favorite)
		
		response = dialog.run()
		dialog.destroy()
		if response == gtk.RESPONSE_OK:
			contact.set_groups([group for group in entry.get_text().split(" ") if group.strip() != ""])
			contact.set_favorite(favorite.get_active())
	
	def on_add_contact_group(self, item, existing_group=None):
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "main.glade"), "edit_group_dialog")
		dialog = glade.get_widget("edit_group_dialog")
		groups_entry = glade.get_widget("groups")
		name_entry = glade.get_widget("name")
		
		if existing_group != None:
			groups_entry.set_text(' '.join(existing_group.groups))
			name_entry.set_text(existing_group.name)
			
		response = dialog.run()
		dialog.destroy()
		if response == gtk.RESPONSE_OK and name_entry.get_text() != "":
			groups = [group for group in groups_entry.get_text().split(" ") if group.strip() != ""]
			name = name_entry.get_text()
			if existing_group != None:
				existing_group.set_name(name)
				existing_group.set_groups(groups)
			else:
				get_contact_list().add_contact_group(ContactGroup(name_entry.get_text(), groups))
	
	def on_remove_contact_group(self, item, path):
		path = self.get_model().convert_path_to_child_path(path)
		iter = get_contact_list().get_iter(path)
		get_contact_list().remove(iter)
				
	def on_contact_activated(self, view, path, col=None):
		self.get_contact_for_path(path).action()
	
	def get_contact_description(self, column, cell, model, iter):
		cell.set_property('markup', model[iter][0].get_description())
				
	def get_contact_icon(self, column, cell, model, iter):
		cell.set_property('pixbuf', model[iter][0].get_icon())
