# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

from telepathy.constants import CONNECTION_STATUS_DISCONNECTED
from telepathy.interfaces import CONN_INTERFACE, CHANNEL_HANDLER_INTERFACE

from cohoba.houston.AccountList import AccountList
from cohoba.common.Utils import ignore_error_handler, SignalConnection, get_logger
from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.common.ChandlersRegistry import ChandlersRegistry

LOG = get_logger("Houston")

class ChannelHandler:
	def __init__(self, accounts):
		self.accounts = accounts
		self.chandlers = ChandlersRegistry()
		self.chandlers.LoadChandlers()
		
		# Store the connections which we listen to
		self.connections = {}
		
	def handle_connection(self, conn_service, conn_obj):
		# Check if we already listen to this connection
		if conn_service in self.connections:
			return
		
		get_proxy_provider().create_proxy(conn_service, conn_obj)
		conn_iface = get_proxy_provider().get_iface(conn_obj, CONN_INTERFACE)
		
		LOG.info('Listening for connection events: %s', conn_obj)
		StatusChanged_conn = SignalConnection(conn_iface, 'StatusChanged', lambda status, reason: self.on_status_changed(status, reason, conn_service))
		NewChannel_conn = SignalConnection(conn_iface, 'NewChannel', lambda *channel: self.handle_channel(conn_service, conn_obj, *channel))
		# We want to retreive and handle all existing channels on the connection
		conn_iface.ListChannels(reply_handler=lambda channels: self.on_channels_listed(conn_service, conn_obj, channels), error_handler=ignore_error_handler)
		
		# Store the signals for the connection, so we can unsubscribe when the connection disappears
		self.connections[conn_service] = (StatusChanged_conn, NewChannel_conn)
	
	def on_status_changed(self, status, reason, conn_service):
		if status == CONNECTION_STATUS_DISCONNECTED:
			for signal in self.connections[conn_service]:
				signal.disconnect()
			del self.connections[conn_service]
	
	def on_channels_listed(self, conn_service, conn_obj, channels):
		for channel in channels:
			chan_obj, chan_type, handle_type, handle = channel
			self.handle_channel(conn_service, conn_obj, chan_obj, chan_type, handle_type, handle)
	
	def handle_channel(self, conn_service, conn_obj, chan_obj, chan_type, handle_type, handle, suppress_handler=False):
		LOG.info('New Channel: obj %r (type %r), handle (type %r) %r, suppress %r', chan_obj, chan_type, handle_type, handle, suppress_handler)
		
		# Check if this channel requested to be ignored by MC
		if suppress_handler:
			return
		
		# Read the available channel handlers for the channel type, and launch them one by one
		chandlers = self.chandlers.GetChandlers(chan_type)
		LOG.debug('--Available channel handler for type "%s": %s', chan_type, chandlers)
		for chandler in chandlers:
			get_proxy_provider().create_proxy(
				self.chandlers.GetBusName(chandler),
				self.chandlers.GetObjectPath(chandler))

			chandler_iface = get_proxy_provider().get_iface(
				self.chandlers.GetObjectPath(chandler),
				CHANNEL_HANDLER_INTERFACE)
			
			LOG.info('---Handling channel with: %s %s', self.chandlers.GetBusName(chandler), self.chandlers.GetObjectPath(chandler))
			chandler_iface.HandleChannel(conn_service, conn_obj, chan_type, chan_obj, handle_type, handle)
