# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk
import gtk.glade
from os.path import join
import cohoba
from gettext import gettext as _

from cohoba.technobabble.ui.ConversationTextView import ConversationUpTextView
from cohoba.common.ui.PresenceIcons import get_presence_icons
from cohoba.common.Utils import presence_status_to_string, global_error_handler
from cohoba.common.TelepathyConstants import (MISSION_CONTROL_INTERFACE,
	MISSION_CONTROL_OBJECT_PATH, MISSION_CONTROL_BUS_NAME)
from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.technobabble.TelepathyVoipGroup import (
	TELEPATHY_VOIP_GROUP_TYPE_INCOMING,
	TELEPATHY_VOIP_GROUP_TYPE_OUTGOING,
	TELEPATHY_VOIP_GROUP_STATE_CALLING,
	TELEPATHY_VOIP_GROUP_STATE_WAITING_ACCEPT,
	TELEPATHY_VOIP_GROUP_STATE_TALKING,
	TELEPATHY_VOIP_GROUP_STATE_CLOSING,
	TELEPATHY_VOIP_GROUP_STATE_CLOSED)

from telepathy.constants import CONNECTION_HANDLE_TYPE_CONTACT
from telepathy.interfaces import CHANNEL_TYPE_STREAMED_MEDIA

class ConversationWindow(gtk.Window):
	NOTIFICATION_TYPING = 0
	
	def __init__(self, contact):
		gtk.Window.__init__(self)
		self.set_property("border_width", 12)
		self.set_property("window_position", gtk.WIN_POS_CENTER)
		self.set_property("default-height", 400)
		self.set_property("default-width", 300)

		self.contact = contact		
		glade = gtk.glade.XML(join(cohoba.SHARED_DATA_DIR, "conversation.glade"), "contents")
		
		# Glade Widgets
		self.add(glade.get_widget("contents"))
		self.conv_text_view = None
		self.conv_text_view_container = glade.get_widget("conversationscroll")
		self.status = glade.get_widget("status")
		self.avatar = glade.get_widget("avatar")
		self.alias = glade.get_widget("alias")
		self.notificationicon = glade.get_widget("notificationicon") 
		self.notification = glade.get_widget("notification")
		self.talkbutton = glade.get_widget("talkbutton")
		self.voice_label = glade.get_widget("voice_label")
		# FIXME; detect teh crap
		self.talkbutton.show()
		
		# Widgets signals
		self.connect('destroy', self.on_window_destroy)
		self.talktoggled_id = self.talkbutton.connect('toggled', self.on_talk_toggled)
		#conversation.connect('closed', self.on_conversation_closed)
		
		
		# Default Values
		self.set_alias(None, _("Unknown"))
		self.set_status(None)
		self.set_avatar(None, None)
		self.set_notification(None)
		
		self.contact.connect('updated', self.on_contact_updated)
		self.on_contact_updated(contact)
		
		#self.conversation = conversation
		#self.conversation.connect('message-received', self.on_message_received)
		
		self.voip_call = None
		self.conversation = None
	
	def start_text(self, conversation):
		if self.conversation != None:
			return
		
		self.conversation = conversation
		if self.conv_text_view != None:
			self.conv_text_view_container.remove(self.conv_text_view)
			
		self.conv_text_view = ConversationUpTextView(conversation)
		self.conv_text_view.show()
		
		self.conv_text_view_container.add(self.conv_text_view)
		conversation.connect('closed', self.on_conversation_closed)
		
		self.conversation.connect('message-received', self.on_message_received)
	
	def on_conversation_closed(self, conversation):
		self.conversation = None
		
	def on_contact_updated(self, contact):
		name = contact.name
		if name == None:
			name = _("New Conversation")
		self.set_title(name)
			
		self.set_alias(contact.alias, contact.name)
		self.set_avatar(contact.status[0], contact.avatar[0])
		self.set_status(presence_status_to_string(contact.status[0]), contact.status[1])
			
	def on_message_received(self, conversation, id, timestamp, sender, message_type, flags, message):
		self.present_with_time(gtk.get_current_event_time())
		
	def on_window_destroy(self, win):
		if self.conversation != None:
			self.conversation.close()
		if self.voip_call != None:
			self.voip_call.stop()
	
	def on_talk_toggled(self, button):
		if button.get_active() and self.voip_call == None:
			get_proxy_provider().create_proxy(
				MISSION_CONTROL_BUS_NAME,
				MISSION_CONTROL_OBJECT_PATH)
			mission_control_iface = get_proxy_provider().get_iface(
				MISSION_CONTROL_OBJECT_PATH,
				MISSION_CONTROL_INTERFACE)
			mission_control_iface.RequestChannel(
				get_proxy_provider().get_service(self.contact.conn_path),
				self.contact.conn_path,
				CHANNEL_TYPE_STREAMED_MEDIA,
				CONNECTION_HANDLE_TYPE_CONTACT,
				self.contact.handle,
				reply_handler=lambda: None,
				error_handler=global_error_handler)
		elif button.get_active() and self.voip_call != None:
			self.voip_call.accept()
		elif self.voip_call != None:
			self.voip_call.stop()
			
	def start_media(self, voip_call):
		print 'Media Started', voip_call
		self.voip_call = voip_call
		voip_call.connect('updated', self.on_voip_updated)
		self.on_voip_updated(voip_call)
	
	def on_voip_updated(self, voip_group):
		print 'Voip Updated:', voip_group
		if voip_group.call_type == TELEPATHY_VOIP_GROUP_TYPE_INCOMING:
			call_type = _("Incoming Call")
		elif voip_group.call_type == TELEPATHY_VOIP_GROUP_TYPE_OUTGOING:
			call_type = _("Outgoing Call")
		else:
			call_type = _("Not Talking")
			
		if voip_group.state == TELEPATHY_VOIP_GROUP_STATE_CALLING:
			status = _("Calling")
		elif voip_group.state == TELEPATHY_VOIP_GROUP_STATE_WAITING_ACCEPT:
			status = _("Waiting for authorization")
		elif voip_group.state == TELEPATHY_VOIP_GROUP_STATE_TALKING:
			status = _("Talking")
		elif voip_group.state == TELEPATHY_VOIP_GROUP_STATE_CLOSING:
			status = _("Ending")
		elif voip_group.state == TELEPATHY_VOIP_GROUP_STATE_CLOSED:
			status = _("Ended")
		else:
			status = _("Starting")
		
		self.voice_label.set_markup("%s\n<span size='smaller' foreground='#555'>%s</span>" % (call_type, status))
		
		if voip_group.state == TELEPATHY_VOIP_GROUP_STATE_CLOSED:
			self.voip_call = None
			self.talkbutton.handler_block(self.talktoggled_id)
			self.talkbutton.set_active(False)
			self.talkbutton.handler_unblock(self.talktoggled_id)
		
	def set_status(self, status, msg=None):
		if status == None and msg == None:
			self.status.hide()
			return

		template = "<span size='small'>%s</span>"			
		if status != None and msg != None:
			template = template % ("%s\n%s" % (status, msg))
		elif status != None:
			template = template % status
		elif msg != None:
			template = template % msg
			
		self.status.set_markup(template)
		self.status.show()
	
	def set_notification(self, notification):
		if notification == self.NOTIFICATION_TYPING:
			self.notificationicon.set_from_stock(gtk.STOCK_EDIT, gtk.ICON_SIZE_MENU)
			self.notification.set_markup("<span size='small' style='italic'>%s</span>" % _("Composing a Message"))
		else:
			self.notificationicon.hide()
			self.notification.hide()
			return
			
		self.notificationicon.show()
		self.notification.show()
		
	def set_avatar(self, presence, avatar):
		if presence == None:
			self.avatar.set_property("icon-name", "stock_person")
			self.avatar.set_property("icon-size", gtk.ICON_SIZE_DIALOG)
		else:
			self.avatar.set_from_pixbuf(get_presence_icons(presence, avatar)[1])
		
	def set_alias(self, alias, name):
		if alias == None:
			alias = name
			name = ""
			
		self.alias.set_markup("<span size='large' weight='ultrabold'>%s</span><span weight='ultrabold'>\n%s</span>" % (alias, name))
