/*
 * Copyright (c) 2006 Darryll Truchan 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this 
 * software and associated documentation files (the "Software"), to deal in the Software 
 * without restriction, including without limitation the rights to use, copy, modify, 
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to the following 
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies 
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR 
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <compiz.h>
#include <X11/Xatom.h>


#define PUT_SPEED_DEFAULT       1.5f
#define PUT_SPEED_MIN           0.1f
#define PUT_SPEED_MAX           50.0f
#define PUT_SPEED_PRECISION     0.1f

#define PUT_TIMESTEP_DEFAULT    0.5f
#define PUT_TIMESTEP_MIN        0.1f
#define PUT_TIMESTEP_MAX        50.0f
#define PUT_TIMESTEP_PRECISION  0.1f

#define PUT_LEFT_PAD_DEFAULT    0
#define PUT_LEFT_PAD_MIN        0
#define PUT_LEFT_PAD_MAX        500

#define PUT_TOP_PAD_DEFAULT     0
#define PUT_TOP_PAD_MIN         0
#define PUT_TOP_PAD_MAX         500

#define PUT_RIGHT_PAD_DEFAULT   0
#define PUT_RIGHT_PAD_MIN       0
#define PUT_RIGHT_PAD_MAX       500

#define PUT_BOTTOM_PAD_DEFAULT  0
#define PUT_BOTTOM_PAD_MIN      0
#define PUT_BOTTOM_PAD_MAX      500

#define PUT_RESTORE_MODIFIERS_DEFAULT       CompSuperMask
#define PUT_RESTORE_KEY_DEFAULT             "KP_Insert"

#define PUT_CENTER_MODIFIERS_DEFAULT        CompSuperMask
#define PUT_CENTER_KEY_DEFAULT              "KP_Begin"

#define PUT_LEFT_MODIFIERS_DEFAULT          CompSuperMask
#define PUT_LEFT_KEY_DEFAULT                "KP_Left"

#define PUT_TOPLEFT_MODIFIERS_DEFAULT       CompSuperMask
#define PUT_TOPLEFT_KEY_DEFAULT             "KP_Home"

#define PUT_TOP_MODIFIERS_DEFAULT           CompSuperMask
#define PUT_TOP_KEY_DEFAULT                 "KP_Up"

#define PUT_TOPRIGHT_MODIFIERS_DEFAULT      CompSuperMask
#define PUT_TOPRIGHT_KEY_DEFAULT            "KP_Prior"

#define PUT_RIGHT_MODIFIERS_DEFAULT         CompSuperMask
#define PUT_RIGHT_KEY_DEFAULT               "KP_Right"

#define PUT_BOTTOMRIGHT_MODIFIERS_DEFAULT   CompSuperMask
#define PUT_BOTTOMRIGHT_KEY_DEFAULT         "KP_Next"

#define PUT_BOTTOM_MODIFIERS_DEFAULT        CompSuperMask
#define PUT_BOTTOM_KEY_DEFAULT              "KP_Down"

#define PUT_BOTTOMLEFT_MODIFIERS_DEFAULT    CompSuperMask
#define PUT_BOTTOMLEFT_KEY_DEFAULT          "KP_End"

#define PUT_POINTER_MODIFIERS_DEFAULT       CompSuperMask
#define PUT_POINTER_KEY_DEFAULT             "z"

#define PUT_SELF_ANIMATE_DEFAULT            TRUE
#define PUT_CENTER_INITIAL_DEFAULT          FALSE
#define PUT_UNFOCUS_WINDOW_DEFAULT          FALSE

#define PUT_DISPLAY_OPTION_CENTER           0
#define PUT_DISPLAY_OPTION_LEFT             1
#define PUT_DISPLAY_OPTION_TOPLEFT          2
#define PUT_DISPLAY_OPTION_TOP              3
#define PUT_DISPLAY_OPTION_TOPRIGHT         4
#define PUT_DISPLAY_OPTION_RIGHT            5
#define PUT_DISPLAY_OPTION_BOTTOMRIGHT      6
#define PUT_DISPLAY_OPTION_BOTTOM           7
#define PUT_DISPLAY_OPTION_BOTTOMLEFT       8
#define PUT_DISPLAY_OPTION_RESTORE          9
#define PUT_DISPLAY_OPTION_VIEWPORT_LEFT    10
#define PUT_DISPLAY_OPTION_VIEWPORT_RIGHT   11
#define PUT_DISPLAY_OPTION_VIEWPORT         12
#define PUT_DISPLAY_OPTION_VIEWPORT_1       13
#define PUT_DISPLAY_OPTION_VIEWPORT_2       14
#define PUT_DISPLAY_OPTION_VIEWPORT_3       15
#define PUT_DISPLAY_OPTION_VIEWPORT_4       16
#define PUT_DISPLAY_OPTION_VIEWPORT_5       17
#define PUT_DISPLAY_OPTION_VIEWPORT_6       18
#define PUT_DISPLAY_OPTION_VIEWPORT_7       19
#define PUT_DISPLAY_OPTION_VIEWPORT_8       20
#define PUT_DISPLAY_OPTION_VIEWPORT_9       21
#define PUT_DISPLAY_OPTION_VIEWPORT_10      22
#define PUT_DISPLAY_OPTION_VIEWPORT_11      23
#define PUT_DISPLAY_OPTION_VIEWPORT_12      24
#define PUT_DISPLAY_OPTION_EXACT            25
#define PUT_DISPLAY_OPTION_POINTER          26
#define PUT_DISPLAY_OPTION_VIEWPORT_UP      27
#define PUT_DISPLAY_OPTION_VIEWPORT_DOWN    28
#define PUT_DISPLAY_OPTION_NUM              29

#define PUT_SCREEN_OPTION_SPEED             0
#define PUT_SCREEN_OPTION_TIMESTEP          1
#define PUT_SCREEN_OPTION_OWN_ANIM          2
#define PUT_SCREEN_OPTION_LEFT_PAD          3
#define PUT_SCREEN_OPTION_TOP_PAD           4
#define PUT_SCREEN_OPTION_RIGHT_PAD         5
#define PUT_SCREEN_OPTION_BOTTOM_PAD        6
#define PUT_SCREEN_OPTION_UNFOCUS_WINDOW    7
#define PUT_SCREEN_OPTION_NUM               8

#define GET_PUT_DISPLAY(d) ((PutDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define PUT_DISPLAY(d) PutDisplay *pd = GET_PUT_DISPLAY (d)
#define GET_PUT_SCREEN(s, pd) ((PutScreen *) (s)->privates[(pd)->screenPrivateIndex].ptr)
#define PUT_SCREEN(s) PutScreen *ps = GET_PUT_SCREEN (s, GET_PUT_DISPLAY (s->display))
#define GET_PUT_WINDOW(w, ps) ((PutWindow *) (w)->privates[(ps)->windowPrivateIndex].ptr)
#define PUT_WINDOW(w) PutWindow *pw = GET_PUT_WINDOW  (w, GET_PUT_SCREEN  (w->screen, GET_PUT_DISPLAY (w->screen->display)))

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

static Bool usePlane = FALSE;

static int displayPrivateIndex;

typedef enum {
    PutUnknown       = 0,
    PutBottomLeft    = 1,
    PutBottom        = 2,
    PutBottomRight   = 3,
    PutLeft          = 4,
    PutCenter        = 5,
    PutRight         = 6,
    PutTopLeft       = 7,
    PutTop           = 8,
    PutTopRight      = 9,
    PutRestore       = 10,
    PutViewport      = 11,
    PutViewportLeft  = 12,
    PutViewportRight = 13,
    PutExact         = 14,
    PutPointer       = 15,
    PutViewportUp    = 16,
    PutViewportDown  = 17
} PutType;

typedef struct _PutDisplay {
    int             screenPrivateIndex;
    CompOption      opt[PUT_DISPLAY_OPTION_NUM];
    HandleEventProc handleEvent;
    Atom            compizPutWindowAtom;
} PutDisplay;

typedef struct _PutScreen {
    int windowPrivateIndex;
    CompOption              opt[PUT_SCREEN_OPTION_NUM];
    PreparePaintScreenProc  preparePaintScreen;
    DonePaintScreenProc     donePaintScreen;
    PaintScreenProc         paintScreen;
    PaintWindowProc         paintWindow;
    float                   speed;
    float                   timestep;
    int                     moreAdjust;
    int                     grabIndex;
    Bool                    ownAnim;
    int                     padLeft, padTop, padRight, padBottom;
    unsigned int            wMask;
    Bool                    vpMoving;
    Bool                    unfocusWindow;
} PutScreen;

typedef struct _PutWindow {
    GLfloat     xVelocity, yVelocity;
    GLfloat     tx, ty;
    int         dx, dy, x, y;
    int         lastX, lastY;
    Bool        adjust;
} PutWindow;

static CompOption *
putGetScreenOptions (CompScreen *screen, int *count)
{
    PUT_SCREEN (screen);
    *count = NUM_OPTIONS (ps);
    return ps->opt;
}

static Bool
putSetScreenOption (CompScreen *screen, char *name, CompOptionValue *value)
{
    CompOption *o;
    int index;

    PUT_SCREEN (screen);

    o = compFindOption (ps->opt, NUM_OPTIONS (ps), name, &index);
    if (!o)
        return FALSE;

    switch (index) 
    {
        case PUT_SCREEN_OPTION_SPEED:
            if (compSetFloatOption (o, value))
            {
                ps->speed = o->value.f;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_TIMESTEP:
            if (compSetFloatOption (o, value))
            {
                ps->timestep = o->value.f;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_OWN_ANIM:
            if (compSetBoolOption (o, value))
            {
                ps->ownAnim = o->value.b;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_LEFT_PAD:
            if (compSetIntOption (o, value))
            {
                ps->padLeft = o->value.i;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_TOP_PAD:
            if (compSetIntOption (o, value))
            {
                ps->padTop = o->value.i;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_RIGHT_PAD:
            if (compSetIntOption (o, value))
            {
                ps->padRight = o->value.i;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_BOTTOM_PAD:
            if (compSetIntOption (o, value))
            {
                ps->padBottom = o->value.i;
                return TRUE;
            }
            break;

        case PUT_SCREEN_OPTION_UNFOCUS_WINDOW:
            if (compSetBoolOption (o, value))
            {
                ps->unfocusWindow = o->value.b;
                return TRUE;
            }
            break;

        default:
            break;
    }

    return FALSE;
}

static void
putScreenInitOptions (PutScreen *ps)
{
    CompOption *o;

    o = &ps->opt[PUT_SCREEN_OPTION_SPEED];
    o->name             = "speed";
    o->shortDesc        = N_("Speed");
    o->longDesc         = N_("Minimize speed");
    o->type             = CompOptionTypeFloat;
    o->value.f          = PUT_SPEED_DEFAULT;
    o->rest.f.min       = PUT_SPEED_MIN;
    o->rest.f.max       = PUT_SPEED_MAX;
    o->rest.f.precision = PUT_SPEED_PRECISION;

    o = &ps->opt[PUT_SCREEN_OPTION_TIMESTEP];
    o->name             = "timestep";
    o->shortDesc        = N_("Timestep");
    o->longDesc         = N_("Minimize timestep");
    o->type             = CompOptionTypeFloat;
    o->value.f          = PUT_TIMESTEP_DEFAULT;
    o->rest.f.min       = PUT_TIMESTEP_MIN;
    o->rest.f.max       = PUT_TIMESTEP_MAX;
    o->rest.f.precision = PUT_TIMESTEP_PRECISION;

    o = &ps->opt[PUT_SCREEN_OPTION_LEFT_PAD];
    o->name             = "pad_left";
    o->shortDesc        = N_("Pad Left");
    o->longDesc         = N_("Left side padding");
    o->type             = CompOptionTypeInt;
    o->value.i          = PUT_LEFT_PAD_DEFAULT;
    o->rest.i.min       = PUT_LEFT_PAD_MIN;
    o->rest.i.max       = PUT_LEFT_PAD_MAX;

    o = &ps->opt[PUT_SCREEN_OPTION_TOP_PAD];
    o->name             = "pad_top";
    o->shortDesc        = N_("Pad Top");
    o->longDesc         = N_("Top side padding");
    o->type             = CompOptionTypeInt;
    o->value.i          = PUT_TOP_PAD_DEFAULT;
    o->rest.i.min       = PUT_TOP_PAD_MIN;
    o->rest.i.max       = PUT_TOP_PAD_MAX;

    o = &ps->opt[PUT_SCREEN_OPTION_RIGHT_PAD];
    o->name             = "pad_right";
    o->shortDesc        = N_("Pad Right");
    o->longDesc         = N_("Right side padding");
    o->type             = CompOptionTypeInt;
    o->value.i          = PUT_RIGHT_PAD_DEFAULT;
    o->rest.i.min       = PUT_RIGHT_PAD_MIN;
    o->rest.i.max       = PUT_RIGHT_PAD_MAX;

    o = &ps->opt[PUT_SCREEN_OPTION_BOTTOM_PAD];
    o->name             = "pad_bottom";
    o->shortDesc        = N_("Pad Bottom");
    o->longDesc         = N_("Bottom side padding");
    o->type             = CompOptionTypeInt;
    o->value.i          = PUT_BOTTOM_PAD_DEFAULT;
    o->rest.i.min       = PUT_BOTTOM_PAD_MIN;
    o->rest.i.max       = PUT_BOTTOM_PAD_MAX;

    o = &ps->opt[PUT_SCREEN_OPTION_OWN_ANIM];
    o->name             = "self_animate";
    o->shortDesc        = N_("Self Animate");
    o->longDesc         = N_("Enable built-in animation");
    o->type             = CompOptionTypeBool;
    o->value.b          = PUT_SELF_ANIMATE_DEFAULT;    

    o = &ps->opt[PUT_SCREEN_OPTION_UNFOCUS_WINDOW];
    o->name             = "unfocus_window";
    o->shortDesc        = N_("Unfocus Window");
    o->longDesc         = N_("Unfocus windows that are moved off the viewport");
    o->type             = CompOptionTypeBool;
    o->value.b          = PUT_UNFOCUS_WINDOW_DEFAULT;
}

static int
adjustPutVelocity (CompWindow *w)
{
    float dx, dy, adjust, amount;
    float x1, y1;

    PUT_WINDOW (w);

    x1 = pw->lastX + pw->dx;
    y1 = pw->lastY + pw->dy;

    dx = x1 - (pw->lastX + pw->tx);
    dy = y1 - (pw->lastY + pw->ty);

    adjust = dx * 0.15f;
    amount = fabs (dx) * 1.5;
    if (amount < 0.5f)
        amount = 0.5f;
    else if (amount > 5.0f)
        amount = 5.0f;

    pw->xVelocity = (amount * pw->xVelocity + adjust) / (amount +1.0f);

    adjust = dy * 0.15f;
    amount = fabs (dy) * 1.5f;
    if (amount < 0.5f)
	amount = 0.5f;
    else if (amount > 5.0f)
	amount = 5.0f;

    pw->yVelocity = (amount * pw->yVelocity + adjust) / (amount + 1.0f);

    if (fabs (dx) < 0.1f && fabs (pw->xVelocity) < 0.2f &&
	fabs (dy) < 0.1f && fabs (pw->yVelocity) < 0.2f)
    {
        pw->xVelocity = pw->yVelocity = 0.0f;

        pw->tx = x1 - pw->lastX;
        pw->ty = y1 - pw->lastY;
        
        pw->dx = pw->dy = 0;

        syncWindowPosition (w);
        return 0;
    }
    return 1;
}

static void
putPreparePaintScreen (CompScreen *s, int msSinceLastPaint)
{
    PUT_SCREEN (s);

    if (ps->moreAdjust)    
    {
        CompWindow *w;
        int steps, dx, dy;
        float amount, chunk;

        amount = msSinceLastPaint * 0.025f * ps->speed;
        steps  = amount / (0.5f * ps->timestep);
        if (!steps) steps = 1;
        chunk  = amount / (float) steps;

        while (steps--)
        {
            ps->moreAdjust = 0;
            for (w = s->windows; w; w = w->next)
            {
                PUT_WINDOW (w);

                if (pw->adjust)
                {
                    pw->adjust = adjustPutVelocity (w);
                    ps->moreAdjust |= pw->adjust;
                    
                    pw->tx += pw->xVelocity * chunk;
                    pw->ty += pw->yVelocity * chunk;

                    dx = (pw->lastX + pw->tx) - pw->x;
                    dy = (pw->lastY + pw->ty) - pw->y;

                    if (ps->vpMoving && !usePlane)
                        moveWindowToViewportPosition (w, pw->x + dx, TRUE);
                    else
                        moveWindow (w, dx, dy, TRUE, TRUE);
                   
                    pw->x += dx;
                    pw->y += dy;
                }
            }
            if (!ps->moreAdjust)
            {
                if (ps->unfocusWindow)
                    focusDefaultWindow (s->display);
	        break;
            }
        }
    }

    UNWRAP (ps, s, preparePaintScreen);
    (*s->preparePaintScreen) (s, msSinceLastPaint);
    WRAP (ps, s, preparePaintScreen, putPreparePaintScreen);
}

static void
putDonePaintScreen (CompScreen *s)
{
    PUT_SCREEN (s);
    if (ps->moreAdjust)
    {
        CompWindow *w;

        for (w = s->windows; w; w = w->next)
        {
            PUT_WINDOW (w);

            if (pw->adjust)
            {
                addWindowDamage (w);
            }
        }
    }
    else
    {
        if (ps->grabIndex)
        {
            removeScreenGrab (s, ps->grabIndex, NULL);
            ps->grabIndex = 0;
        }
    }

    UNWRAP (ps, s, donePaintScreen);
    (*s->donePaintScreen) (s);
    WRAP (ps, s, donePaintScreen, putDonePaintScreen);
}

static Bool
putPaintScreen (CompScreen *s, const ScreenPaintAttrib *sAttrib, Region region, int output, unsigned int mask)
{
    Bool status;

    PUT_SCREEN (s);

    if (ps->moreAdjust)
        mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;

    UNWRAP (ps, s, paintScreen);
    status = (*s->paintScreen) (s, sAttrib, region, output, mask);
    WRAP (ps, s, paintScreen, putPaintScreen);

    return status;
}

static Bool
putPaintWindow (CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask)
{
    CompScreen *s = w->screen;
    Bool       status;

    PUT_SCREEN (s);
    PUT_WINDOW (w);

    if (pw->adjust)
        mask |= PAINT_WINDOW_TRANSFORMED_MASK;

    UNWRAP (ps, s, paintWindow);
    status = (*s->paintWindow) (w, attrib, region, mask);
    WRAP (ps, s, paintWindow, putPaintWindow);

    return status;
}

static Bool
putInitiate (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompWindow *w;
    Window xid;
    int px, py, x, y, dx, dy, head, width, height, hx, hy, face, face_x, face_y, hdirection, vdirection;
    PutType type;
    CompPlugin *p;

    xid  = getIntOptionNamed (option, nOption, "window", 0);
    type = getIntOptionNamed (option, nOption, "type", PutCenter);
    px    = getIntOptionNamed (option, nOption, "x", 0);
    py    = getIntOptionNamed (option, nOption, "y", 0);

    if (!xid)
        xid = d->activeWindow;

    w = findWindowAtDisplay (d, xid);

    if (w)
    {
        PUT_SCREEN (w->screen);
        PUT_WINDOW (w);

        if (!ps->grabIndex)
        {
            if (otherScreenGrabExist (w->screen, "put", 0))
                return FALSE;

            ps->grabIndex = pushScreenGrab (w->screen, w->screen->invisibleCursor, "put");
        }

        if (ps->grabIndex)
        {
            ps->vpMoving = FALSE;

            if (w->attrib.override_redirect)
                return FALSE;

            if (w->type & CompWindowTypeDesktopMask ||
                w->type & CompWindowTypeDockMask    ||
                w->type & CompWindowTypeFullscreenMask)
            {    
                return FALSE;
            }

            head = 0;
            width  = w->screen->workArea.width;
            height = w->screen->workArea.height;
            hx = w->screen->workArea.x;        
            hy = w->screen->workArea.y;

            x = w->attrib.x;
            y = w->attrib.y;

            for (p = getPlugins (); p; p = p->next)
            {
                if (strcmp (p->vTable->name, "plane") == 0)
                {
                    usePlane = TRUE;
                    break;
                }
            }

            switch (type)
            {
                case PutCenter:
                    dx = (width  / 2) - (w->width  / 2) - (x - hx);
                    dy = (height / 2) - (w->height / 2) - (y - hy);
                    break;

                case PutLeft:
                    dx = - (x - hx) + w->input.left + ps->padLeft;
                    dy = (height / 2) - (w->height / 2) - (y - hy);
                    break;

                case PutTopLeft:
                    dx = - (x - hx) + w->input.left + ps->padLeft;
                    dy = - (y - hy) + w->input.top + ps->padTop;
                    break;

                case PutTop:
                    dx = (width / 2)  - (w->width / 2)  - (x - hx);
                    dy = - (y - hy) + w->input.top + ps->padTop;
                    break;

                case PutTopRight:
                    dx = width - w->width - (x - hx) - w->input.right - ps->padRight;
                    dy = - (y - hy) + w->input.top + ps->padTop;
                    break;

                case PutRight:
                    dx = width - w->width - (x - hx) - w->input.right - ps->padRight;
                    dy = (height / 2) - (w->height / 2) - (y - hy);
                    break;

                case PutBottomRight:
                    dx = width  - w->width  - (x - hx) - w->input.right - ps->padRight;
                    dy = height - w->height - (y - hy) - w->input.bottom - ps->padBottom;
                    break;

                case PutBottom:
                    dx = (width / 2)  - (w->width / 2)  - (x - hx);
                    dy = height - w->height - (y - hy) - w->input.bottom - ps->padBottom;
                    break;

                case PutBottomLeft:
                    dx = - (x - hx) + w->input.left + ps->padLeft;
                    dy = height - w->height - (y - hy) - w->input.bottom - ps->padBottom;
                    break;

                case PutRestore:
                    dx = pw->lastX - x;
                    dy = pw->lastY - y;
                    break;

                case PutViewport:
                    face = getIntOptionNamed (option, nOption, "face", 0);
                    if (face < 1)
                        return FALSE;

                    if (usePlane)
                    {
                        face_x = face % w->screen->hsize;
                        face_y = face / w->screen->hsize;
                    }
                    else
                    {
                        face_x = face;
                        face_y = 0;
                    }

                    hdirection = (face - (w->screen->x + 1));
                    if (hdirection > w->screen->hsize / 2)
                        hdirection = (hdirection - w->screen->hsize);
                    else if (hdirection < -w->screen->hsize / 2)
                        hdirection = (hdirection + w->screen->hsize);

                    dx = w->screen->width  * hdirection;
                    dy = 0;

                    if (usePlane)
                    {
                        vdirection = (face_y - w->screen->y);
                        if (vdirection > w->screen->vsize / 2)
                            vdirection = (vdirection - w->screen->vsize);
                        else if (hdirection < -w->screen->hsize / 2)
                            vdirection = (vdirection + w->screen->vsize);

                        dx = w->screen->width * hdirection;
                        dy = w->screen->height * vdirection;
                    }
                    ps->vpMoving = TRUE;
                    break;

                case PutViewportLeft:
                    dx = - w->screen->workArea.width;
                    dy = 0;
                    ps->vpMoving = TRUE;
                    break;

                case PutViewportRight:
                    dx = w->screen->workArea.width;
                    dy = 0;
                    ps->vpMoving = TRUE;
                    break;

                case PutViewportUp:
                    dx = 0;
                    dy = 0;
                    if (usePlane)
                        dy = - w->screen->workArea.height;
                    ps->vpMoving = TRUE;
                    break;

                case PutViewportDown:
                    dx = 0;
                    dy = 0;
                    if (usePlane)
                        dy = w->screen->workArea.height;
                    ps->vpMoving = TRUE;
                    break;

                case PutExact:
                    if (px < 0)
                        dx = px + w->screen->workArea.width - w->width - x - w->input.right;
                    else
                        dx = px - x + w->input.left;

                    if (py < 0)
                        dy = py + w->screen->workArea.height - w->height - y - w->input.bottom;
                    else 
                        dy = py - y + w->input.top;

                    break;

                case PutPointer:
                    {
                        int rx, ry;
                        Window root, child;
                        int winx, winy;
                        unsigned int pmask;

                        XQueryPointer (d->display, w->id, &root, &child, &rx, &ry, &winx, &winy, &pmask);

                        if (rx < w->screen->workArea.width / 2 && ry < w->screen->workArea.height / 2)
                        {
                            // top left quad
                            dx = rx - x + w->input.left;
                            dy = ry - y + w->input.top;
                        }
                        else if (rx < w->screen->workArea.width / 2 && ry >= w->screen->workArea.height / 2)
                        {
                            // bottom left quad
                            dx = rx - x + w->input.left;
                            dy = ry - w->height - y - w->input.bottom;
                        }
                        else if (rx >= w->screen->workArea.width / 2 && ry < w->screen->workArea.height / 2)
                        {
                            // top right quad
                            dx = rx - w->width - x - w->input.right;
                            dy = ry - y + w->input.top;
                        }
                        else
                        {
                            // bottom right quad
                            dx = rx - w->width - x - w->input.right;
                            dy = ry - w->height - y - w->input.bottom;
                        }
                    }
                    break;

                default:
                    dx = dy = 0;
                    break;
            }

            if (dx != 0 || dy != 0)
            {

                if (w->saveMask & CWX)
                    w->saveWc.x += dx;

                if (w->saveMask & CWY)
                    w->saveWc.y += dy;

                if (ps->ownAnim)
                {
                    /* Make sure everyting starts out at the windows current position */
                    pw->lastX = pw->x = w->serverX = x;
                    pw->lastY = pw->y = w->serverY = y; 

                    pw->dx = dx;
                    pw->dy = dy;

                    pw->adjust = TRUE;
                    ps->moreAdjust = TRUE;

                    pw->tx = pw->ty = 0;

                    addWindowDamage (w);
                }
                else
                {
                    (w->screen->windowGrabNotify) (w, x + (w->attrib.width / 2), y + (w->attrib.height / 2), 0, CompWindowGrabMoveMask | CompWindowGrabButtonMask);
                    moveWindow (w, dx, dy, TRUE, FALSE);
                    syncWindowPosition (w);
                }
            }
        }
    }
    return FALSE;
}

static Bool
putToViewport (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    int face, i;

    PUT_DISPLAY (d);

    face = getIntOptionNamed (option, nOption, "face", -1);
    if (face < 0)
    {
        i = PUT_DISPLAY_OPTION_VIEWPORT_1;

        while (i <= PUT_DISPLAY_OPTION_VIEWPORT_12)
        {
            if (action == &pd->opt[i].value.action)
            {
                face = i - PUT_DISPLAY_OPTION_VIEWPORT_1 + 1;
                break;
            }
            i++;
        }
    }

    CompOption o [5];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "face";
    o[2].value.i = face;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "type";
    o[3].value.i = PutViewport;

    o[4].type    = CompOptionTypeInt;
    o[4].name    = "window";
    o[4].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 5);

    return FALSE;
}

static Bool
putViewportLeft (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutViewportLeft;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putViewportRight (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutViewportRight;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putViewportUp (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutViewportUp;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putViewportDown (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutViewportDown;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
restore (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{

    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutRestore;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putPointer (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o[4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutPointer;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putExact (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutExact;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putCenter (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutCenter;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putLeft (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutLeft;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putTopLeft (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutTopLeft;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putTop (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutTop;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putTopRight (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutTopRight;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putRight (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutRight;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putBottomRight (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutBottomRight;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putBottom (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutBottom;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static Bool
putBottomLeft (CompDisplay *d, CompAction *action, CompActionState state, CompOption *option, int nOption)
{
    CompOption o [4];

    o[0].type    = CompOptionTypeInt;
    o[0].name    = "x";
    o[0].value.i = getIntOptionNamed (option, nOption, "x", 0);

    o[1].type    = CompOptionTypeInt;
    o[1].name    = "y";
    o[1].value.i = getIntOptionNamed (option, nOption, "y", 0);

    o[2].type    = CompOptionTypeInt;
    o[2].name    = "type";
    o[2].value.i = PutBottomLeft;

    o[3].type    = CompOptionTypeInt;
    o[3].name    = "window";
    o[3].value.i = getIntOptionNamed (option, nOption, "window", 0);

    putInitiate (d, NULL, 0, o, 4);

    return FALSE;
}

static void
putHandleEvent (CompDisplay *d, XEvent *event)
{
    PUT_DISPLAY (d);

    switch (event->type)
    {
        case ClientMessage:
            if (event->xclient.message_type == pd->compizPutWindowAtom)            
            {
                CompWindow *w;
                w = findWindowAtDisplay (d, event->xclient.window);
                if (w)
                {
                    CompOption opt[6];

                    opt[0].type    = CompOptionTypeInt;
                    opt[0].name    = "window";
                    opt[0].value.i = event->xclient.window;

                    opt[1].type    = CompOptionTypeInt;
                    opt[1].name    = "x";
                    opt[1].value.i = event->xclient.data.l[0];

                    opt[2].type    = CompOptionTypeInt;
                    opt[2].name    = "y";
                    opt[2].value.i = event->xclient.data.l[1];

                    opt[3].type    = CompOptionTypeInt;
                    opt[3].name    = "face";
                    opt[3].value.i = event->xclient.data.l[2];

                    opt[4].type    = CompOptionTypeInt;
                    opt[4].name    = "type";
                    opt[4].value.i = event->xclient.data.l[3];

                    opt[5].type    = CompOptionTypeInt;
                    opt[5].name    = "head";
                    opt[5].value.i = event->xclient.data.l[4];

                    putInitiate (w->screen->display, NULL, 0, opt, 6);

                }
            }
            break;

        default:
            break;
    }

    UNWRAP (pd, d, handleEvent);
    (*d->handleEvent) (d, event);
    WRAP (pd, d, handleEvent, putHandleEvent);
}

static CompOption *
putGetDisplayOptions (CompDisplay *display, int *count)
{
    PUT_DISPLAY (display);
    *count = NUM_OPTIONS (pd);
    return pd->opt;
}

static Bool
putSetDisplayOption (CompDisplay *display, char *name, CompOptionValue *value)
{
    CompOption *o;
    int index;

    PUT_DISPLAY (display);

    o = compFindOption (pd->opt, NUM_OPTIONS (pd), name, &index);

    if (!o)
        return FALSE;

    switch (index) 
    {
        case PUT_DISPLAY_OPTION_RESTORE:
        case PUT_DISPLAY_OPTION_CENTER:
        case PUT_DISPLAY_OPTION_LEFT:
        case PUT_DISPLAY_OPTION_TOPLEFT:
        case PUT_DISPLAY_OPTION_TOP:
        case PUT_DISPLAY_OPTION_TOPRIGHT:
        case PUT_DISPLAY_OPTION_RIGHT:
        case PUT_DISPLAY_OPTION_BOTTOMRIGHT:
        case PUT_DISPLAY_OPTION_BOTTOM:
        case PUT_DISPLAY_OPTION_BOTTOMLEFT:
        case PUT_DISPLAY_OPTION_VIEWPORT_LEFT:
        case PUT_DISPLAY_OPTION_VIEWPORT_RIGHT: 
        case PUT_DISPLAY_OPTION_VIEWPORT_UP:
        case PUT_DISPLAY_OPTION_VIEWPORT_DOWN: 
        case PUT_DISPLAY_OPTION_VIEWPORT:
        case PUT_DISPLAY_OPTION_VIEWPORT_1:
        case PUT_DISPLAY_OPTION_VIEWPORT_2:
        case PUT_DISPLAY_OPTION_VIEWPORT_3:
        case PUT_DISPLAY_OPTION_VIEWPORT_4:
        case PUT_DISPLAY_OPTION_VIEWPORT_5:
        case PUT_DISPLAY_OPTION_VIEWPORT_6:
        case PUT_DISPLAY_OPTION_VIEWPORT_7:
        case PUT_DISPLAY_OPTION_VIEWPORT_8:
        case PUT_DISPLAY_OPTION_VIEWPORT_9:
        case PUT_DISPLAY_OPTION_VIEWPORT_10:
        case PUT_DISPLAY_OPTION_VIEWPORT_11:
        case PUT_DISPLAY_OPTION_VIEWPORT_12:
        case PUT_DISPLAY_OPTION_EXACT:
        case PUT_DISPLAY_OPTION_POINTER:
            if (setDisplayAction (display, o, value))
                return TRUE;
        default:
            break;
    }
    return FALSE;
}

static void
putDisplayInitOptions (PutDisplay *pd, Display *display)
{
    CompOption *o;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT];
    o->name                     = "put_viewport";
    o->shortDesc                = N_("Put on Face");             
    o->longDesc                 = N_("Move window to face");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = 0;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_1];
    o->name                     = "put_viewport_1";
    o->shortDesc                = N_("Put on Face 1");             
    o->longDesc                 = N_("Move window to face 1");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_2];
    o->name                     = "put_viewport_2";
    o->shortDesc                = N_("Put on Face 2");             
    o->longDesc                 = N_("Move window to face 2");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_3];
    o->name                     = "put_viewport_3";
    o->shortDesc                = N_("Put on Face 3");             
    o->longDesc                 = N_("Move window to face 3");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_4];
    o->name                     = "put_viewport_4";
    o->shortDesc                = N_("Put on Face 4");             
    o->longDesc                 = N_("Move window to face 4");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_5];
    o->name                     = "put_viewport_5";
    o->shortDesc                = N_("Put on Face 5");             
    o->longDesc                 = N_("Move window to face 5");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_6];
    o->name                     = "put_viewport_6";
    o->shortDesc                = N_("Put on Face 6");             
    o->longDesc                 = N_("Move window to face 6");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_7];
    o->name                     = "put_viewport_7";
    o->shortDesc                = N_("Put on Face 7");             
    o->longDesc                 = N_("Move window to face 7");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_8];
    o->name                     = "put_viewport_8";
    o->shortDesc                = N_("Put on Face 8");             
    o->longDesc                 = N_("Move window to face 8");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_9];
    o->name                     = "put_viewport_9";
    o->shortDesc                = N_("Put on Face 9");             
    o->longDesc                 = N_("Move window to face 9");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_10];
    o->name                     = "put_viewport_10";
    o->shortDesc                = N_("Put on Face 10");             
    o->longDesc                 = N_("Move window to face 10");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_11];
    o->name                     = "put_viewport_11";
    o->shortDesc                = N_("Put on Face 11");             
    o->longDesc                 = N_("Move window to face 11");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_12];
    o->name                     = "put_viewport_12";
    o->shortDesc                = N_("Put on Face 12");             
    o->longDesc                 = N_("Move window to face 12");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putToViewport;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_LEFT];
    o->name                     = "put_viewport_left";
    o->shortDesc                = N_("Viewport Left");
    o->longDesc                 = N_("Move window to the viewport on left");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putViewportLeft;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_RIGHT];
    o->name                     = "put_viewport_right";
    o->shortDesc                = N_("Viewport Right");
    o->longDesc                 = N_("Move window to the viewport on right");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putViewportRight;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_UP];
    o->name                     = "put_viewport_up";
    o->shortDesc                = N_("Viewport Up");
    o->longDesc                 = N_("Move window to the viewport on top");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putViewportUp;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_VIEWPORT_DOWN];
    o->name                     = "put_viewport_down";
    o->shortDesc                = N_("Viewport Down");
    o->longDesc                 = N_("Move window to the viewport on bottom");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putViewportDown;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_RESTORE];
    o->name                     = "put_restore";
    o->shortDesc                = N_("Restore Position");
    o->longDesc                 = N_("Move window to the last position");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = restore;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_RESTORE_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_RESTORE_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_POINTER];
    o->name                     = "put_pointer";
    o->shortDesc                = N_("Put Pointer");             
    o->longDesc                 = N_("Move window to the pointer position using screen quadrant");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putPointer;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_POINTER_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_POINTER_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_EXACT];
    o->name                     = "put_exact";
    o->shortDesc                = N_("Put Exact");             
    o->longDesc                 = N_("Move window to x, y");           
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putExact;      
    o->value.action.terminate   = 0;                 
    o->value.action.bell        = FALSE;            
    o->value.action.edgeMask    = 0;               
    o->value.action.state       = 0;
    o->value.action.type        = CompBindingTypeNone;

    o = &pd->opt[PUT_DISPLAY_OPTION_CENTER];
    o->name                     = "put_center";
    o->shortDesc                = N_("Put Center");
    o->longDesc                 = N_("Move window to the center");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putCenter;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_CENTER_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_CENTER_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_LEFT];
    o->name                     = "put_left";
    o->shortDesc                = N_("Put Left");
    o->longDesc                 = N_("Move window to the center of the left edge");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putLeft;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_LEFT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_LEFT_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_TOPLEFT];
    o->name                     = "put_top_left";
    o->shortDesc                = N_("Put Top Left");
    o->longDesc                 = N_("Move Window to the top left corner");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putTopLeft;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_TOPLEFT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_TOPLEFT_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_TOP];
    o->name                     = "put_top";
    o->shortDesc                = N_("Put Top");
    o->longDesc                 = N_("Move window to the center of the top edge");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putTop;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_TOP_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_TOP_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_TOPRIGHT];
    o->name                     = "put_top_right";
    o->shortDesc                = N_("Put Top Right");
    o->longDesc                 = N_("Move window to the top right corner");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putTopRight;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_TOPRIGHT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_TOPRIGHT_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_RIGHT];
    o->name                     = "put_right";
    o->shortDesc                = N_("Put Right");
    o->longDesc                 = N_("Move window to the center of the right edge");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putRight;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_RIGHT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_RIGHT_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_BOTTOMRIGHT];
    o->name                     = "put_bottom_right";
    o->shortDesc                = N_("Put Bottom Right");
    o->longDesc                 = N_("Move window to the bottom right corner");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putBottomRight;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_BOTTOMRIGHT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_BOTTOMRIGHT_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_BOTTOM];
    o->name                     = "put_bottom";
    o->shortDesc                = N_("Put Bottom");
    o->longDesc                 = N_("Move window to the center of the bottom edge");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putBottom;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_BOTTOM_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_BOTTOM_KEY_DEFAULT));

    o = &pd->opt[PUT_DISPLAY_OPTION_BOTTOMLEFT];
    o->name                     = "put_bottom_left";
    o->shortDesc                = N_("Put Bottom Left");
    o->longDesc                 = N_("Move window to the bottom left corner");
    o->type                     = CompOptionTypeAction;
    o->value.action.initiate    = putBottomLeft;
    o->value.action.terminate   = 0;
    o->value.action.bell        = FALSE;
    o->value.action.edgeMask    = 0;
    o->value.action.state       = CompActionStateInitKey;
    o->value.action.state      |= CompActionStateInitButton;
    o->value.action.type        = CompBindingTypeKey;
    o->value.action.key.modifiers = PUT_BOTTOMLEFT_MODIFIERS_DEFAULT;
    o->value.action.key.keycode   = XKeysymToKeycode (display, XStringToKeysym (PUT_BOTTOMLEFT_KEY_DEFAULT));
}

static Bool
putInitDisplay (CompPlugin *p, CompDisplay *d)
{
    PutDisplay *pd;

    pd = malloc (sizeof (PutDisplay));
    if (!pd)
        return FALSE;

    pd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (pd->screenPrivateIndex < 0)
    {
        free (pd);
        return FALSE;
    }

    pd->compizPutWindowAtom = XInternAtom (d->display, "_COMPIZ_PUT_WINDOW", 0);

    putDisplayInitOptions (pd, d->display);
    WRAP (pd, d, handleEvent, putHandleEvent);
    d->privates[displayPrivateIndex].ptr = pd;

    return TRUE;
}

static void
putFiniDisplay (CompPlugin *p, CompDisplay *d)
{
    PUT_DISPLAY (d);
    freeScreenPrivateIndex (d, pd->screenPrivateIndex);
    UNWRAP (pd, d, handleEvent);
    free (pd);
}

static Bool
putInitScreen (CompPlugin *p, CompScreen *s)
{
    PutScreen *ps;

    PUT_DISPLAY (s->display);

    ps = malloc (sizeof (PutScreen));
    if (!ps)
        return FALSE;

    ps->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (ps->windowPrivateIndex < 0)
    {
        free (ps);
        return FALSE;
    }

    ps->grabIndex = 0;
    ps->speed = PUT_SPEED_DEFAULT;
    ps->timestep = PUT_TIMESTEP_DEFAULT;
    ps->moreAdjust = FALSE;
    ps->ownAnim = TRUE;
    ps->padLeft = PUT_LEFT_PAD_DEFAULT;
    ps->padTop = PUT_TOP_PAD_DEFAULT;
    ps->padRight = PUT_RIGHT_PAD_DEFAULT;
    ps->padBottom = PUT_BOTTOM_PAD_DEFAULT;
    ps->vpMoving = FALSE;
    ps->unfocusWindow = PUT_UNFOCUS_WINDOW_DEFAULT;

    putScreenInitOptions (ps);

    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_POINTER].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_RESTORE].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_CENTER].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_LEFT].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_TOPLEFT].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_TOP].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_TOPRIGHT].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_RIGHT].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_BOTTOMRIGHT].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_BOTTOM].value.action);
    addScreenAction (s, &pd->opt[PUT_DISPLAY_OPTION_BOTTOMLEFT].value.action);

    WRAP (ps, s, preparePaintScreen, putPreparePaintScreen);
    WRAP (ps, s, donePaintScreen, putDonePaintScreen);
    WRAP (ps, s, paintScreen, putPaintScreen);
    WRAP (ps, s, paintWindow, putPaintWindow);

    s->privates[pd->screenPrivateIndex].ptr = ps;
    return TRUE;
}

static void
putFiniScreen (CompPlugin *p, CompScreen *s)
{
    PUT_SCREEN (s);
    freeWindowPrivateIndex (s, ps->windowPrivateIndex);
    UNWRAP (ps, s, preparePaintScreen);
    UNWRAP (ps, s, donePaintScreen);
    UNWRAP (ps, s, paintScreen);
    UNWRAP (ps, s, paintWindow);
    free (ps);
}

static Bool
putInitWindow (CompPlugin *p, CompWindow *w)
{
    PutWindow *pw;

    PUT_SCREEN (w->screen);

    pw = malloc (sizeof (PutWindow));
    if (!pw)
        return FALSE;

    pw->tx = pw->ty = pw->xVelocity = pw->yVelocity = 0.0f;
    pw->dx = pw->dy = 0;
    pw->lastX = pw->x = w->serverX;
    pw->lastY = pw->y = w->serverY;
    pw->adjust = FALSE;

    w->privates[ps->windowPrivateIndex].ptr = pw;

    return TRUE;
}

static void
putFiniWindow (CompPlugin *p, CompWindow *w)
{
    PUT_WINDOW (w);
    free (pw);
}

static Bool
putInit (CompPlugin *p)
{
    displayPrivateIndex = allocateDisplayPrivateIndex ();
    if (displayPrivateIndex < 0)
        return FALSE;

    return TRUE;
}

static void
putFini (CompPlugin *p)
{
    if (displayPrivateIndex >= 0)
        freeDisplayPrivateIndex (displayPrivateIndex);
}

static int
putGetVersion (CompPlugin *plugin, int version)
{
     return ABIVERSION;
}

CompPluginVTable putVTable = {
    "put",
    N_("Put"),
    N_("Put window"),
    putGetVersion,
    putInit,
    putFini,
    putInitDisplay,
    putFiniDisplay,
    putInitScreen,
    putFiniScreen,
    putInitWindow,
    putFiniWindow,
    putGetDisplayOptions,
    putSetDisplayOption,
    putGetScreenOptions,
    putSetScreenOption,
    0,
    0,
};

CompPluginVTable *
getCompPluginInfo (void)
{
    return &putVTable;
}


