/*
 * Copyright (C) 2006 Fran\xc3\xa7ois Ingelrest <Athropos@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
 *
 * trailfocus.c: rewrite
 *
 * Author: Dennis Kasprzyk
 * E-Mail: onestone@beryl-project.org
 *
 * ---------------------------------------------
 *
 * trailfocus.c:
 * author: casey langen, 2006
 * version: 0.1.2
 * purpose: leave a "trail" of focused windows
 *
 * ---------------------------------------------
 *
 * based on...
 *
 * trailFocus.c v0.03
 *
 * Author: Fran\xc3\xa7ois Ingelrest
 * E-Mail: Athropos@gmail.com
 *
 **/

#include <compiz.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>



/* option identifiers ---------------------------------------------------- */


#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL     0
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL     1
#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL  2
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL  3
#define TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL  4
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL  5
#define TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT       6
#define TRAILFOCUS_SCREEN_OPTION_WINDOW_TYPE               7
#define TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR        8
#define TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER          9
#define TRAILFOCUS_SCREEN_OPTION_DISABLE_ON_SCREENGRAB_OF  10
#define TRAILFOCUS_SCREEN_OPTION_NUM                       11

#define TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR_DEFAULT        TRUE
#define TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER_DEFAULT          TRUE

static char *DefaultWinType[] = {
    N_("Normal"),
};

static char *DisableOnScreengrabDefault[] = { "switcher", "scale" };

/* macros --------------------------------------------------------------- */


#define GET_TRAILFOCUS_DISPLAY(display)    \
    ((TrailFocusDisplay*)display->privates[mDisplayPrivateIndex].ptr)

#define GET_TRAILFOCUS_SCREEN(screen)      \
    ((TrailFocusScreen*)screen->privates[GET_TRAILFOCUS_DISPLAY(screen->display)->screenPrivateIndex].ptr)

#define GET_TRAILFOCUS_WINDOW(w, tfs)				         \
    ((TrailFocusWindow *) (w)->privates[(tfs)->windowPrivateIndex].ptr)

#define NUM_OPTIONS(screen) (sizeof((screen)->opt) / sizeof(CompOption))

#define LIST_SIZE(l) (sizeof (l) / sizeof (l[0]))

/* structs -------------------------------------------------------------- */


typedef struct _TrailFocusDisplay
{
    int screenPrivateIndex;
} TrailFocusDisplay;

typedef struct _TrailFocusScreen
{
    int windowPrivateIndex;
    int wmask;                  // Which kind of windows are we looking at?

    int minSaturationLevel;
    int minBrightnessLevel;
    int minOpacityLevel;
    int maxTrailCount;

    int maxSaturationLevel;
    int maxBrightnessLevel;
    int maxOpacityLevel;

    PaintWindowProc paintWindow;
    PreparePaintScreenProc preparePaintScreen;

    CompOption opt[TRAILFOCUS_SCREEN_OPTION_NUM];
} TrailFocusScreen;

typedef struct _TrailFocusWindow
{
    Bool isActive;
    Bool isUpdated;
    int position;
    GLushort saturation;
    GLushort brightness;
    GLushort opacity;
} TrailFocusWindow;


/* locals ---------------------------------------------------------------- */

static int mDisplayPrivateIndex;

/* Beryl prototypes ----------------------------------------------------- */


static Bool trailFocusInit (CompPlugin * p);
static void trailFocusFini (CompPlugin * p);
static Bool trailFocusInitDisplay (CompPlugin * p, CompDisplay * d);
static void trailFocusFiniDisplay (CompPlugin * p, CompDisplay * d);
static Bool trailFocusInitScreen (CompPlugin * p, CompScreen * s);
static void trailFocusFiniScreen (CompPlugin * p, CompScreen * s);
static Bool trailFocusInitWindow (CompPlugin * p, CompWindow * w);
static void trailFocusFiniWindow (CompPlugin * p, CompWindow * w);
static Bool trailFocusSetScreenOption (CompScreen * screen, char *name,
                                       CompOptionValue * value);
static CompOption *trailFocusGetScreenOptions (CompScreen * screen,
                                               int *count);
static Bool trailFocusPaintWindow (CompWindow * w,
                                   const WindowPaintAttrib * attrib,
                                   Region region, unsigned int mask);
static void trailFocusPreparePaintScreen (CompScreen * s, int ms);

static Bool screenGrabExist (CompScreen * s, const char *name);


static int
trailfocusGetVersion (CompPlugin *plugin,
		int	   version)
{
    return ABIVERSION;
}
/* Beryl requests a nice cup of tea */
static CompPluginVTable trailFocusVTable = {
    "trailfocus",
    "Trailfocus",
    "Window focus leaves a trail",
    trailfocusGetVersion,
    trailFocusInit,
    trailFocusFini,
    trailFocusInitDisplay,
    trailFocusFiniDisplay,
    trailFocusInitScreen,
    trailFocusFiniScreen,
    trailFocusInitWindow,
    trailFocusFiniWindow,
    0,                          /* GetDisplayOptions */
    0,                          /* SetDisplayOption */
    trailFocusGetScreenOptions,
    trailFocusSetScreenOption,
    NULL,
    0,
    0,
    0
};


/* entry point */
CompPluginVTable *
getCompPluginInfo (void)
{
    return &trailFocusVTable;
}

/* plugin init */
static Bool
trailFocusInit (CompPlugin * p)
{
    mDisplayPrivateIndex = allocateDisplayPrivateIndex ();
    if (mDisplayPrivateIndex < 0)
    {
        return FALSE;
    }

    return TRUE;
}

/* plugin exit */
static void
trailFocusFini (CompPlugin * p)
{
    if (mDisplayPrivateIndex >= 0)
    {
        freeDisplayPrivateIndex (mDisplayPrivateIndex);
    }
}

/* display init */
static Bool
trailFocusInitDisplay (CompPlugin * p, CompDisplay * d)
{
    TrailFocusDisplay *tfd;

    tfd = malloc (sizeof (TrailFocusDisplay));
    if (!tfd)
    {
        return FALSE;
    }

    tfd->screenPrivateIndex = allocateScreenPrivateIndex (d);
    if (tfd->screenPrivateIndex < 0)
    {
        free (tfd);
        return FALSE;
    }

    d->privates[mDisplayPrivateIndex].ptr = tfd;

    return TRUE;
}

/* display exit */
static void
trailFocusFiniDisplay (CompPlugin * p, CompDisplay * d)
{
    TrailFocusDisplay *tfd = GET_TRAILFOCUS_DISPLAY (d);

    freeScreenPrivateIndex (d, tfd->screenPrivateIndex);

    free (tfd);
}
static void
trailFocusScreenInitOptions (TrailFocusScreen * tfs)
{
    CompOption *o;
    int i;

    /* Options for this plugin */
    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL];
    o->name = "minimum_window_opacity_level";
    o->shortDesc = "Opacity level of completely unfocused windows";
    o->longDesc = "Opacity level of completely unfocused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 90;
    o->rest.i.min = 25;
    o->rest.i.max = 100;
    tfs->minOpacityLevel = 90;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL];
    o->name = "maximum_window_opacity_level";
    o->shortDesc = "Opacity level of completely focused windows";
    o->longDesc = "Opacity level of completely focused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 100;
    o->rest.i.min = 25;
    o->rest.i.max = 100;
    tfs->maxOpacityLevel = 100;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL];
    o->name = "minimum_window_saturation_level";
    o->shortDesc = "Saturation level of completely unfocused windows";
    o->longDesc = "Saturation level of completely unfocused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 100;
    o->rest.i.min = 0;
    o->rest.i.max = 100;
    tfs->minSaturationLevel = 100;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL];
    o->name = "maximum_window_saturation_level";
    o->shortDesc = "Saturation level of completely focused windows";
    o->longDesc = "Saturation level of completely focused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 100;
    o->rest.i.min = 0;
    o->rest.i.max = 100;
    tfs->maxSaturationLevel = 100;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL];
    o->name = "minimum_window_brightness_level";
    o->shortDesc = "Brightness level of unfocused windows";
    o->longDesc = "Brightness level of unfocused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 90;
    o->rest.i.min = 0;
    o->rest.i.max = 100;
    tfs->minBrightnessLevel = 90;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL];
    o->name = "maximum_window_brightness_level";
    o->shortDesc = "Brightness level of focused windows";
    o->longDesc = "Brightness level of focused windows";
    o->type = CompOptionTypeInt;
    o->value.i = 100;
    o->rest.i.min = 0;
    o->rest.i.max = 100;
    tfs->maxBrightnessLevel = 100;


    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT];
    o->name = "maximum_trail_count";
    o->shortDesc = "Number of recently used windows to keep track of";
    o->longDesc = "Number of recently used windows to keep track of";
    o->type = CompOptionTypeInt;
    o->value.i = 6;
    o->rest.i.min = 1;
    o->rest.i.max = 64;
    tfs->maxTrailCount = 6;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_WINDOW_TYPE];
    o->name = "window_types";
    o->shortDesc = "Trailfocus window types";
    o->longDesc = "Trailfocus window types";
    o->type = CompOptionTypeList;
    o->value.list.type = CompOptionTypeString;
    o->value.list.nValue = LIST_SIZE (DefaultWinType);
    o->value.list.value =
        malloc (sizeof (CompOptionValue) * LIST_SIZE (DefaultWinType));
    for (i = 0; i < LIST_SIZE (DefaultWinType); i++)
        o->value.list.value[i].s = strdup (DefaultWinType[i]);
    o->rest.s.string = windowTypeString;
    o->rest.s.nString = nWindowTypeString;
    tfs->wmask = compWindowTypeMaskFromStringList (&o->value);

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR];
    o->name = "ignore_skiptaskbar";
    o->shortDesc = N_("Ignore \"SkipTaskbar\" Windows");
    o->longDesc = N_("Ignore \"SkipTaskbar\" Windows");
    o->type = CompOptionTypeBool;
    o->value.b = TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR_DEFAULT;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER];
    o->name = "ignore_skippager";
    o->shortDesc = N_("Ignore \"SkipPager\" Windows");
    o->longDesc = N_("Ignore \"SkipPager\" Windows");
    o->type = CompOptionTypeBool;
    o->value.b = TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER_DEFAULT;

    o = &tfs->opt[TRAILFOCUS_SCREEN_OPTION_DISABLE_ON_SCREENGRAB_OF];
    o->name = "disable_on_screengrab";
    o->shortDesc = "Disable on screengrab of";
    o->longDesc = "Disable Trailfocus on screengrab of";
    o->type = CompOptionTypeList;
    o->value.list.type = CompOptionTypeString;
    o->value.list.nValue = LIST_SIZE (DisableOnScreengrabDefault);
    o->value.list.value =
        malloc (sizeof (CompOptionValue) *
                LIST_SIZE (DisableOnScreengrabDefault));
    o->rest.s.nString=0;
    o->rest.s.string=NULL;
    for (i = 0; i < LIST_SIZE (DisableOnScreengrabDefault); i++)
        o->value.list.value[i].s = strdup (DisableOnScreengrabDefault[i]);

}


/* screen init */
static Bool
trailFocusInitScreen (CompPlugin * p, CompScreen * s)
{
    TrailFocusScreen *tfs;
    TrailFocusDisplay *tfd;

    tfd = GET_TRAILFOCUS_DISPLAY (s->display);

    tfs = malloc (sizeof (TrailFocusScreen));
    if (!tfs)
    {
        return FALSE;
    }

    tfs->windowPrivateIndex = allocateWindowPrivateIndex (s);
    if (tfs->windowPrivateIndex < 0)
    {
        free (tfs);
        return FALSE;
    }

    trailFocusScreenInitOptions (tfs);

    s->privates[tfd->screenPrivateIndex].ptr = tfs;

    WRAP (tfs, s, paintWindow, trailFocusPaintWindow);
    WRAP (tfs, s, preparePaintScreen, trailFocusPreparePaintScreen);

    return TRUE;
}

/* screen exit */
static void
trailFocusFiniScreen (CompPlugin * p, CompScreen * s)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (s);

    freeWindowPrivateIndex (s, tfs->windowPrivateIndex);

    UNWRAP (tfs, s, paintWindow);
    UNWRAP (tfs, s, preparePaintScreen);

    free (tfs);
}

/* a screen option has changed */
static Bool
trailFocusSetScreenOption (CompScreen * s, char *name,
                           CompOptionValue * value)
{
    int index;
    CompOption *o;
    TrailFocusScreen *tfs;

    tfs = GET_TRAILFOCUS_SCREEN (s);
    o = compFindOption (tfs->opt, NUM_OPTIONS (tfs), name, &index);

    if (!o)
    {
        return FALSE;
    }

    /* todo: optimize with a dictionary (hashmap) */
    switch (index)
    {
    case TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER:
    case TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR:
        if (compSetBoolOption (o, value))
        {
            damageScreen (s);
            return TRUE;
        }
        break;
    case TRAILFOCUS_SCREEN_OPTION_MINIMUM_OPACITY_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->minOpacityLevel = o->value.i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_MINIMUM_SATURATION_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->minSaturationLevel = value->i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_MINIMUM_BRIGHTNESS_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->minBrightnessLevel = value->i;
            damageScreen (s);
            return TRUE;
        }
        break;
    case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_OPACITY_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->maxOpacityLevel = o->value.i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_SATURATION_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->maxSaturationLevel = value->i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_BRIGHTNESS_LEVEL:
        if (compSetIntOption (o, value))
        {
            tfs->maxBrightnessLevel = value->i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_MAXIMUM_TRAIL_COUNT:
        if (compSetIntOption (o, value))
        {
            tfs->maxTrailCount = value->i;
            damageScreen (s);
            return TRUE;
        }
        break;

    case TRAILFOCUS_SCREEN_OPTION_WINDOW_TYPE:
        if (compSetOptionList (o, value))
        {
            tfs->wmask = compWindowTypeMaskFromStringList (&o->value);
            damageScreen (s);
            return TRUE;
        }
        break;

    default:
        break;
    }

    return FALSE;
}


/* return our options */
static CompOption *
trailFocusGetScreenOptions (CompScreen * screen, int *count)
{
    if (screen)
    {
        TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (screen);

        *count = NUM_OPTIONS (tfs);
        return tfs->opt;
    }
    else
    {
        TrailFocusScreen *tfs = malloc (sizeof (TrailFocusScreen));
        trailFocusScreenInitOptions (tfs);
        *count = NUM_OPTIONS (tfs);
        return tfs->opt;
    }
}

/* window begin */
static Bool
trailFocusInitWindow (CompPlugin * p, CompWindow * w)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (w->screen);

    TrailFocusWindow *tfw = malloc (sizeof (TrailFocusWindow));
    if (!tfw)
    {
        return FALSE;
    }

    tfw->saturation = 0;
    tfw->brightness = 0;
    tfw->opacity = 0;
    tfw->isUpdated = FALSE;
    tfw->position = 0;

    w->privates[tfs->windowPrivateIndex].ptr = tfw;

    return TRUE;
}

/* window end */
static void
trailFocusFiniWindow (CompPlugin * p, CompWindow * w)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (w->screen);
    TrailFocusWindow *tfw = GET_TRAILFOCUS_WINDOW (w, tfs);

    free (tfw);
}

/* trailfocus definitions ------------------------------------------------ */


static Bool
trailFocusPaintWindow (CompWindow * w,
                       const WindowPaintAttrib * attrib,
                       Region region, unsigned int mask)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (w->screen);
    TrailFocusWindow *tfw = GET_TRAILFOCUS_WINDOW (w, tfs);

    Bool status;
    WindowPaintAttrib tfAttrib = *attrib;

    int i = 0;
    Bool active = TRUE;
    for (i = 0;
         i <
         tfs->opt[TRAILFOCUS_SCREEN_OPTION_DISABLE_ON_SCREENGRAB_OF].value.
         list.nValue; i++)
    {
        active &=
            !screenGrabExist (w->screen,
                              tfs->
                              opt
                              [TRAILFOCUS_SCREEN_OPTION_DISABLE_ON_SCREENGRAB_OF].
                              value.list.value[i].s);
    }

    if (tfw->isActive & active)
    {
        tfAttrib.opacity = MIN (tfw->opacity, tfAttrib.opacity);
        tfAttrib.brightness = MIN (tfw->brightness, tfAttrib.brightness);
        tfAttrib.saturation = MIN (tfw->saturation, tfAttrib.saturation);
    }

    UNWRAP (tfs, w->screen, paintWindow);
    status = (*w->screen->paintWindow) (w, &tfAttrib, region, mask);
    WRAP (tfs, w->screen, paintWindow, trailFocusPaintWindow);
    return status;
}

static Bool
isTrailFocusWindow (CompScreen * s, CompWindow * w)
{

    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (w->screen);

    if (WINDOW_INVISIBLE (w))
        return FALSE;
    if (w->state & CompWindowStateHiddenMask)
        return FALSE;

    if (tfs->opt[TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPTASKBAR].value.b
        && w->state & CompWindowStateSkipTaskbarMask)
        return FALSE;
    if (tfs->opt[TRAILFOCUS_SCREEN_OPTION_IGNORE_SKIPPAGER].value.b
        && w->state & CompWindowStateSkipPagerMask)
        return FALSE;
    if (!(tfs->wmask & w->type))
        return FALSE;

    return TRUE;
}

static void
trailFocusPreparePaintScreen (CompScreen * s, int ms)
{
    TrailFocusScreen *tfs = GET_TRAILFOCUS_SCREEN (s);
    TrailFocusWindow *tfw;

    float sat, bright, opacity;
    float sa_step, br_step, op_step;

    int count = 0;

    CompWindow *w, *activeWindow;

    activeWindow = findWindowAtDisplay (s->display, s->display->activeWindow);
    if (!activeWindow)
    {
        UNWRAP (tfs, s, preparePaintScreen);
        (*s->preparePaintScreen) (s, ms);
        WRAP (tfs, s, preparePaintScreen, trailFocusPreparePaintScreen);
        return;
    }

    for (w = s->reverseWindows; w; w = w->prev)
    {
        if (isTrailFocusWindow (s, w))
        {
            tfw = GET_TRAILFOCUS_WINDOW (w, tfs);
            tfw->isUpdated = FALSE;
            count++;
        }
    }

    count = MAX (1, MIN (tfs->maxTrailCount, count - 1));
    tfw = GET_TRAILFOCUS_WINDOW (activeWindow, tfs);
    tfw->position = count;
    tfw->isUpdated = TRUE;

    int i;
    int max_pos;
    TrailFocusWindow *max_win, *tmp_win;

    for (i = count - 1; i >= 0; i--)
    {
        max_win = NULL;
        max_pos = -1;
        for (w = s->reverseWindows; w; w = w->prev)
        {
            tmp_win = GET_TRAILFOCUS_WINDOW (w, tfs);
            if (isTrailFocusWindow (s, w))
            {
                if (tmp_win->position > max_pos && !tmp_win->isUpdated)
                {
                    max_pos = tmp_win->position;
                    max_win = tmp_win;
                }
            }
            else
            {
                tmp_win->position = 0;
            }
        }
        if (max_win)
        {
            max_win->isUpdated = TRUE;
            max_win->position = i;
        }
    }

    sa_step = (tfs->maxSaturationLevel - tfs->minSaturationLevel) / (float) count;
    br_step = (tfs->maxBrightnessLevel - tfs->minBrightnessLevel) / (float) count;
    op_step = (tfs->maxOpacityLevel - tfs->minOpacityLevel) / (float) count;
    for (w = s->reverseWindows; w; w = w->prev)
    {
        tfw = GET_TRAILFOCUS_WINDOW (w, tfs);
        tfw->isActive = isTrailFocusWindow (s, w);
        if (tfw->isActive)
        {
            sat = tfs->minSaturationLevel;
            bright = tfs->minBrightnessLevel;
            opacity = tfs->minOpacityLevel;

            if (tfw->isUpdated)
            {
                sat += sa_step * tfw->position;
                bright += br_step * tfw->position;
                opacity += op_step * tfw->position;
            }

            sat = sat * 0xffff / 100.0;
            bright = bright * 0xffff / 100.0;
            opacity = opacity * 0xffff / 100.0;

            if (sat != tfw->saturation || bright != tfw->brightness
                || opacity != tfw->opacity)
                addWindowDamage (w);
            tfw->saturation = sat;
            tfw->brightness = bright;
            tfw->opacity = opacity;
            count = MAX (0, count - 1);
        }
    }
    UNWRAP (tfs, s, preparePaintScreen);
    (*s->preparePaintScreen) (s, ms);
    WRAP (tfs, s, preparePaintScreen, trailFocusPreparePaintScreen);
}

static Bool
screenGrabExist (CompScreen * s, const char *name)
{
    int i;

    for (i = 0; i < s->maxGrab; i++)
    {
        if (s->grabs[i].active)
        {
            if (strcmp (name, s->grabs[i].name) == 0)
                return TRUE;
        }
    }
    return FALSE;
}
